/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/**
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute it 
 *and/or modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will be useful, 
 *but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. If not, write to the Free Software Foundation, Inc., 
 *59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>
#include <libxml/uri.h>
#include <libxml/tree.h>
#include <glade/glade.h>
#include <gnome.h>
#include "mlview-editor.h"
#include "mlview-tree-view.h"
#ifdef MLVIEW_WITH_STYLE
#include "mlview-styled-view.h"
#endif
#include "mlview-source-view.h"
#include "mlview-file-descriptor.h"
#include "mlview-xml-document.h"
#include "mlview-utils.h"
#include "mlview-xslt-utils.h"
#include "mlview-schemas-window.h"
#include "mlview-validator-window.h"

#include "recent-files/egg-recent.h"

/**
 *@file
 *The definition of the methods of #MlViewEditor class.
 */

enum {
        DOCUMENT_CHANGED,
        LAST_VIEW_REMOVED,
        FIRST_VIEW_ADDED,
        SIGNAL_NUM
};

struct MlViewViewDesc gv_view_types[] = {
        {
                (gchar*)"tree-view", 
		(gchar*)N_ ("Tree view") ,
                (gchar*)N_("An editing view well suited for the tree oriented editing paradigm"),
                mlview_tree_view_new
        },
#ifdef MLVIEW_WITH_STYLE
        {
                (gchar*)"styled-view",
		N_(Styled view),
                (gchar*)N_("A highly experimental view to render/edit XML using a CSS"),
                mlview_styled_view_new
        },
#endif        
        {
                (gchar*)"source-view", 
		(gchar*)N_ ("Source view"),
                (gchar*)N_("An editing view well suited for the source oriented editing paradigm"),
                mlview_source_view_new
        },
        {0}
} ;

/*======================================================
 *Some private (static) data structures used by
 *MlViewEditor widget.
 *=====================================================*/

/**
 *Data from the New Document dialog
 */
typedef struct _MlViewNewDocumentDialogData {
        gchar *root_node_name;
        gchar *xml_version;
        gchar *encoding;
        MlViewSchema *schema;
} MlViewNewDocumentDialogData;
        
/**
 *The #MlViewEditor private data member structure.
 *The fields of this structure must be accessed only by 
 *the mean of the available accessors.
 */
struct _MlViewEditorPrivate {
        /**
	 *The set of instances of #MlViewXMLDocument
	 *opened in the editor. The key is the pointer to 
	 *the instance of #MlViewXMLDocument opened in the editor
	 *The value is a an hash table that contains the 
	 *instances of #MlViewIView that holds 
	 *the key #MlViewXMLDocument.
	 */
        GHashTable *mlview_xml_docs;

        /**
	 *A hash table where the keys are the instances of
	 *MlViewXMLDocumentViews and the values are their
	 *associated document.
	 */
        GHashTable *mlview_xml_doc_views;

        GHashTable *mlview_xml_doc_schemas_windows;

        GHashTable *mlview_xml_doc_validation_windows;

        /**
	 *The notebook that holds all the notebook pages 
	 *Each notebook page contains one editing view.
	 */
        GtkNotebook *notebook;

        /**
	 *The current document being viewed by the user
	 */
        MlViewIView *cur_view ;

        /**
	 *a hash table which keys are the base name of 
	 *the files already opened. 
	 *The associated data is the number 
	 *of times the file name
	 *has been opened. When destroying this hash 
	 *table, do not destroy the referenced data (base names) because
	 *the base names are substring of file_paths. 
	 *And file names are hold by the instances of MlViewXMLDocument.
	 */
        GHashTable *opened_file_base_names;

        /**
	 *a hash table wich keys are the file paths of 
	 *the file files already opened.
	 *the associated MlViewXMLDocument. 
	 *When destroying this hashtable do not 
	 *destroy the referenced data (file path)
	 *because file paths are hold by instances 
	 *of MlViewXMLDocument and destroyed by them.
	 */
        GHashTable *opened_file_paths;

        /**
	 *An hash table that associates the opened document label names to
	 *the matching opened view.
	 */
        GHashTable *opened_document_label_names;

        /**
         *Number of untitled document opened.
         */
        guint untitled_docs_num;

        /**
         *total number of docs opened
         */
        guint opened_docs_num;

        /**
         *The context of the application
         */
        MlViewAppContext *app_context;

        /*
         *the editor contextual menu 
         */
        GtkWidget *contextual_menu;

        gboolean dispose_has_run ;
};

#define PRIVATE(editor) (editor->priv)

static void mlview_editor_class_init (MlViewEditorClass * a_klass);

static void mlview_editor_init (MlViewEditor * a_this);

/*signal callbacks and default handlers*/
static void mlview_editor_switch_notebook_page_cb (GtkNotebook * a_notebook,
                                                   GtkNotebookPage * a_page,
                                                   gint a_page_num,
                                                           MlViewEditor * a_this);

static GtkWidget *create_tab_title (MlViewEditor *a_this, MlViewIView *a_view, guchar * title);

static void close_tab_button_clicked_cb (GtkButton *button, gpointer a_this);

static void view_name_changed_cb (MlViewIView * a_view,
                                  gpointer a_this);

static void mlview_editor_connect_to_app_context (MlViewEditor *a_this,
                                                  MlViewAppContext *a_context) ;

static void mlview_editor_disconnect_from_app_context (MlViewEditor *a_this,
                                                       MlViewAppContext *a_context) ;

static GList *build_view_list_from_hashtable (GHashTable *a_views);

static GList *build_doc_list_from_hashtable (GHashTable * a_docs);

static void add_hash_key_to_list (gpointer a_key,
                                  gpointer a_value, GList ** a_list);

static GtkWidget * build_reload_file_confirmation_dialog (void) ;

static void mlview_editor_finalize (GObject *a_this) ;

static void mlview_editor_dispose (GObject *a_this) ;

static enum MlViewStatus get_current_view_from_notebook (GtkNotebook *a_notebook,
							 MlViewIView **a_view)  ;

/* NOT USED YET
static enum MlViewStatus get_nth_view_from_notebook (GtkNotebook *a_notebook,
		                                     gint a_page_num,
						     MlViewIView **a_view) ;
						     */

static enum MlViewStatus get_view_tab_label_from_notebook (GtkNotebook *a_notebook,
		                                           MlViewIView *a_view,
							   GtkWidget **a_label) ;
/* NOT USED YET
static enum MlViewStatus get_view_page_number_from_notebook (GtkNotebook *a_notebook,
							     MlViewIView *a_view,
							     gint *a_page_num) ;
							     */

static enum MlViewStatus remove_view_page_from_notebook (GtkNotebook *a_notebook,
		                                         MlViewIView *a_view) ;

static GtkVBoxClass *gv_parent_class = NULL;
static guint gv_signals[SIGNAL_NUM] = { 0 };

struct DocumentWindowData {
        MlViewEditor *editor;
        MlViewXMLDocument *document;
        GtkWidget *window;
};

static void schemas_window_destroy_cb (GtkWidget *a_widget,
                                       struct DocumentWindowData *a_win);

static void validation_window_destroy_cb (GtkWidget *a_widget,
                                          struct DocumentWindowData *a_win);

/*============================================================
 *private method required by the GTK typing system
 *=============================================================*/


/**
 *The vtable initialyzer.
 *@param a_klass the vtable structure.
 *
 */
static void
mlview_editor_class_init (MlViewEditorClass * a_klass)
{
        GObjectClass *object_class = NULL ;

        gv_parent_class = g_type_class_peek_parent (a_klass);

        object_class = G_OBJECT_CLASS (a_klass) ;
        g_return_if_fail (object_class) ;

        /*overload the destroy method of GtkObject */
        object_class->dispose = mlview_editor_dispose ;
        object_class->finalize = mlview_editor_finalize ;

        /*define the signals */
        gv_signals[DOCUMENT_CHANGED] =
                g_signal_new ("document-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewEditorClass,
                               document_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        gv_signals[LAST_VIEW_REMOVED] =
                g_signal_new ("last_view_removed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewEditorClass,
                               last_view_removed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL) ;

        gv_signals[FIRST_VIEW_ADDED] =
                g_signal_new ("first_view_added",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewEditorClass,
                               first_view_added), NULL, NULL,
                              gtk_marshal_NONE__POINTER,
                              GTK_TYPE_NONE, 1, G_TYPE_POINTER) ;

        /*gtk_object_class_add_signals (object_class, mlview_editor_signals, 
           SIGNAL_NUM) ; */
        a_klass->document_changed = NULL;
}



/**
 *The #MlViewEditor instance allocator.
 *Allocates the space necessary to the members of the #MlViewEditor structure
 *and does very basic initialyzation.
 *
 *@param a_this the instance of #MlViewEditor to allocate.
 */
static void
mlview_editor_init (MlViewEditor * a_this)
{
        g_assert (a_this != NULL);

        PRIVATE (a_this) =
                g_malloc0 (sizeof (MlViewEditorPrivate));

        PRIVATE (a_this)->notebook =
                GTK_NOTEBOOK (gtk_notebook_new ());
	
        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->notebook),
                          "switch-page",
                          G_CALLBACK
                          (mlview_editor_switch_notebook_page_cb),
                          a_this);

        gtk_box_pack_start (GTK_BOX (a_this),
                            GTK_WIDGET (PRIVATE (a_this)->
                                        notebook), TRUE, TRUE,
                            0);

        PRIVATE (a_this)->opened_file_base_names =
                g_hash_table_new (g_str_hash, g_str_equal);

        PRIVATE (a_this)->opened_file_paths =
                g_hash_table_new (g_str_hash, g_str_equal);

        PRIVATE (a_this)->opened_document_label_names =
                g_hash_table_new (g_str_hash, g_str_equal);

        PRIVATE (a_this)->mlview_xml_docs =
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (a_this)->mlview_xml_doc_views =
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (a_this)->mlview_xml_doc_schemas_windows = 
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (a_this)->mlview_xml_doc_validation_windows = 
                g_hash_table_new (g_direct_hash, g_direct_equal);

        PRIVATE (a_this)->untitled_docs_num = 0;

        PRIVATE (a_this)->opened_docs_num = 0;
}



static gboolean
validation_windows_foreach_func (MlViewXMLDocument *a_doc,
                                 struct DocumentWindowData *a_data,
                                 gpointer a_user_data)
{
        g_return_val_if_fail (a_data, TRUE);
        g_return_val_if_fail (a_data->window, TRUE);
        g_return_val_if_fail (GTK_IS_WIDGET (a_data->window), TRUE);

        g_signal_handlers_disconnect_by_func (G_OBJECT (a_data->window),
                                              G_CALLBACK (validation_window_destroy_cb),
                                              a_data);

        gtk_widget_destroy (a_data->window);
        a_data->window = NULL;

        g_free (a_data);
        a_data = NULL;

        return TRUE;
}

static gboolean
schemas_windows_foreach_func (MlViewXMLDocument *a_doc,
                              struct DocumentWindowData *a_data,
                              gpointer a_user_data)
{     
        g_return_val_if_fail (a_data, TRUE);
        g_return_val_if_fail (a_data->window, TRUE);
        g_return_val_if_fail (GTK_IS_WIDGET (a_data->window), TRUE);
        
        g_signal_handlers_disconnect_by_func (G_OBJECT (a_data->window),
                                              G_CALLBACK (schemas_window_destroy_cb),
                                              a_data);

        gtk_widget_destroy (a_data->window);
        a_data->window = NULL;

        g_free (a_data);
        a_data = NULL;

        return TRUE;
}

static void
mlview_editor_dispose (GObject *a_this)
{
        MlViewEditor *editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)) ;
        editor = MLVIEW_EDITOR (a_this) ;
        g_return_if_fail (PRIVATE (editor))  ;

        if (PRIVATE (editor)->dispose_has_run == TRUE)
                return ;

        /*
         *destroy each remaining view.
         **/
        if (PRIVATE (editor)->mlview_xml_doc_views) {
                GList *list_item = NULL ;
                GList *view_list = build_view_list_from_hashtable 
                        (PRIVATE (editor)->mlview_xml_doc_views) ;

                for (list_item = view_list ;
                     list_item ;list_item = list_item->next) {
                        if (list_item->data 
                            && MLVIEW_IS_IVIEW (list_item->data)) {
                                mlview_editor_remove_view 
					(editor, 
					 MLVIEW_IVIEW (list_item->data)) ;
                        }
                }
                g_hash_table_destroy
                        (PRIVATE (editor)->mlview_xml_doc_views);
                PRIVATE (editor)->mlview_xml_doc_views = NULL;
        }
        /*
         *do not free this element because 
         *it has been freed as an element of the link list above
         */
        PRIVATE (editor)->cur_view = NULL;

        if (PRIVATE (editor)->mlview_xml_docs) {
                g_hash_table_destroy (PRIVATE (editor)->mlview_xml_docs) ;
                PRIVATE (editor)->mlview_xml_docs = NULL ;
        }
        if (PRIVATE (editor)->opened_file_base_names) {
                g_hash_table_destroy
                        (PRIVATE (editor)->
                         opened_file_base_names);

                PRIVATE (editor)->opened_file_base_names = NULL;
        }
        if (PRIVATE (editor)->opened_file_paths) {
                g_hash_table_destroy
                        (PRIVATE (editor)->opened_file_paths);

                PRIVATE (editor)->opened_file_paths = NULL;
        }

        if (PRIVATE (editor)->opened_document_label_names) {
                g_hash_table_destroy
                        (PRIVATE (editor)->
                         opened_document_label_names);

                PRIVATE (editor)->opened_document_label_names = NULL;
        }

        if (PRIVATE (editor)->mlview_xml_doc_schemas_windows) {
                g_hash_table_foreach_remove 
                        (PRIVATE (editor)->mlview_xml_doc_schemas_windows,
                         (GHRFunc) schemas_windows_foreach_func, NULL);

                g_hash_table_destroy 
                        (PRIVATE (editor)->mlview_xml_doc_schemas_windows);
                PRIVATE (editor)->mlview_xml_doc_schemas_windows = NULL;
        }

        if (PRIVATE (editor)->mlview_xml_doc_validation_windows) {
                g_hash_table_foreach_remove
                        (PRIVATE (editor)->mlview_xml_doc_validation_windows,
                         (GHRFunc) validation_windows_foreach_func, NULL);

                g_hash_table_destroy 
                        (PRIVATE (editor)->mlview_xml_doc_validation_windows);
                PRIVATE (editor)->mlview_xml_doc_validation_windows = NULL;
        }

        if (PRIVATE (editor)->app_context) {
                mlview_editor_disconnect_from_app_context (editor,
                                                           PRIVATE (editor)->app_context) ;
                PRIVATE (editor)->app_context = NULL ;
        }

        if (G_OBJECT_CLASS (gv_parent_class)->dispose) {
                G_OBJECT_CLASS (gv_parent_class)->dispose (a_this) ;
        }
}


static void
mlview_editor_finalize (GObject *a_this)
{
        MlViewEditor *editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)) ;
        
        editor = MLVIEW_EDITOR (a_this) ;

        if (PRIVATE (editor)) {
                g_free (PRIVATE (editor)) ;
                PRIVATE (editor) = NULL ;
        }
}

/*****************************************
 *signal callbacks and default handlers
 *
 *****************************************/
/**
 *this callback is called when the editor
 *switch from one view to another.
 *
 *@param a_notebook the notebook editor that holds the view notebook pages.
 *@param a_page the notebook page that holds the view 
 *(the view is an instance of #MlViewXMLDocumentView)
 *@param a_page_num the number of the new notebook page.
 *@param a_this the current instance of #MlViewEditor.
 */
static void
mlview_editor_switch_notebook_page_cb (GtkNotebook * a_notebook,
                                       GtkNotebookPage * a_page,
                                       gint a_page_num,
                                       MlViewEditor * a_this)
{
	gboolean must_rebuild_view =  TRUE ;
        MlViewIView *doc_view = NULL, *prev_view = NULL ;
        GtkWidget *cur_child_widget = NULL, *main_menubar = NULL;
	GList *children_widgets = NULL ;
        MlViewAppContext *ctxt = NULL ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        cur_child_widget = gtk_notebook_get_nth_page (a_notebook,
                                                      a_page_num);
        g_return_if_fail (cur_child_widget);
	children_widgets = gtk_container_get_children 
		(GTK_CONTAINER (cur_child_widget)) ;

        doc_view = MLVIEW_IVIEW (children_widgets->data) ;
        g_return_if_fail (doc_view != NULL);

        prev_view = PRIVATE (a_this)->cur_view ;
        PRIVATE (a_this)->cur_view = doc_view;

        /*
         *Detach the content of the edit menu 
         *(when we can do menu merging)so that
         *the newly selected view can attach it's "edit"
         *menu content there.
         */
        ctxt = mlview_editor_get_app_context (a_this) ;
        g_return_if_fail (ctxt) ;

        main_menubar =
                mlview_app_context_get_element
                (ctxt, "MlViewAppMainMenuBar") ;
        if (main_menubar) {
                /*
                 *put here the code of menu merging when we
                 *have it.
                 */
        }

        /*
         *emit the "is-swapped-out" signal.
         *on the previous view and the
         *"is-swapped-in" signal on the current view.
         */
        if (prev_view) {
		mlview_iview_notify_swapped_out (prev_view) ;

		mlview_iview_get_must_rebuild_upon_document_reload 
			(doc_view, &must_rebuild_view) ;

		if (must_rebuild_view == TRUE) {
			MlViewIView *new_doc_view = NULL ;

			g_object_ref (G_OBJECT (doc_view)) ;
			/*g_signal_handlers_block_by_func 
				(G_OBJECT (a_notebook),
				 mlview_editor_switch_notebook_page_cb,
				 a_this) ; */
			mlview_editor_rebuild_view (a_this, doc_view, 
					            &new_doc_view) ;
			/*g_signal_handlers_unblock_by_func 
				(G_OBJECT (a_notebook),
				 mlview_editor_switch_notebook_page_cb,
				 a_this) ;*/
			g_object_unref (doc_view) ;
			if (new_doc_view) {
				/*
				 *unrefing doc_view 
				 * should have freed it by now
				 */
				doc_view = new_doc_view ; 
			}
		}
	}
        if (doc_view)
                mlview_iview_notify_swapped_in (doc_view) ;

        /*
         *notify the application that a new view became the
         *current view.
         */        
        mlview_app_context_notify_view_swapped (ctxt,
                                                prev_view, 
                                                doc_view) ;
	PRIVATE (a_this)->cur_view = doc_view ;

	/*
	 * notify  the app that the undo state changed
	 * this way, the app can update the can undo/can redo arrows
	 * in the menu bar, for example
	 */
	 mlview_app_context_notify_view_undo_state_changed (ctxt) ;
}


/**
 *This callback is called when the name of the current
 *instance of #MlViewXMLDocumentView changes.
 *
 *@param a_view the current instance of #MlViewXMLDocumentView
 *which name has changed.
 *@param a_this the current instance of #MlViewEditor.
 */
static void
view_name_changed_cb (MlViewIView * a_view,
                      gpointer a_this)
{
        MlViewEditor *editor = NULL;
        guchar *new_view_name = NULL;
	GtkWidget *view_container = NULL, *view_impl = NULL ;

        g_return_if_fail (a_view 
                          && MLVIEW_IS_IVIEW (a_view)
                          && a_this
                          && MLVIEW_IS_EDITOR (a_this));

        editor = MLVIEW_EDITOR (a_this);
        g_return_if_fail (PRIVATE (editor) != NULL);
        g_return_if_fail (PRIVATE (editor)->notebook != NULL);

        mlview_iview_get_name (a_view, &new_view_name) ;
        g_return_if_fail (new_view_name) ;

	mlview_iview_get_impl (a_view, &view_impl) ;
	g_return_if_fail (view_impl) ;

	view_container = gtk_widget_get_parent (view_impl) ;
	g_return_if_fail (view_container) ;

        gtk_notebook_set_tab_label (PRIVATE (editor)->notebook,
				    view_container, 
				    create_tab_title ((MlViewEditor*)a_this, 
				    (MlViewIView*)a_view, new_view_name));

}

static gchar *
replace_slashes (gchar *str)
{
	int i ;
	
	g_return_val_if_fail (str != NULL, NULL);
	
	for (i = 0; str[i] != '\0'; i++) {
		if (str[i] == '/')
			str[i] = '-';
	}

	return str;
}
/*
 * Create an hbox for tab titles which contains:
 * a file icon representing the mime type
 * the label representing the current uri
 * a close button
 */
static GtkWidget *
create_tab_title (MlViewEditor *a_this, MlViewIView *a_view, guchar *title)
{
	MlViewXMLDocument *mlview_xml_document = NULL;
	MlViewFileDescriptor *file_desc = NULL;
	gchar *filename = NULL, *mime_type = NULL;
	gint w, h;
	GtkWidget *tab_title_box = NULL,
		  *tab_title_ficon = NULL,
		  *tab_title_label = NULL,
		  *tab_title_cbutton = NULL,
		  *tab_title_cicon = NULL;
	GdkPixbuf *ficon_pixbuf = NULL;

	/* get icon file name from mime type */
	mlview_iview_get_document (a_view, &mlview_xml_document) ;
	if (mlview_xml_document != NULL) {
		file_desc = mlview_xml_document_get_file_descriptor (mlview_xml_document) ;
		if (file_desc != NULL) {
			mime_type = mlview_file_descriptor_get_mime_type (file_desc);
			filename = g_strconcat ("gnome-mime-", replace_slashes (mime_type), NULL);
			g_free (mime_type);
		}
	} 
	
	if (filename == NULL) {
		filename = g_strdup ("gnome-mime-text-xml");
	}
	
	gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &w, &h);
	
	tab_title_box = gtk_hbox_new (FALSE, 2);
	
	ficon_pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
			filename,
			w,
			0,
			NULL);
	tab_title_ficon = gtk_image_new_from_pixbuf (ficon_pixbuf);
	gtk_widget_show (tab_title_ficon);
	gtk_box_pack_start (GTK_BOX (tab_title_box), tab_title_ficon, FALSE, FALSE, 0);
	
	tab_title_label = gtk_label_new (title);
	gtk_widget_show (tab_title_label);
	gtk_box_pack_start (GTK_BOX (tab_title_box), tab_title_label, FALSE, TRUE, 2);
	
	tab_title_cicon = gtk_image_new_from_stock ("gtk-close", GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (tab_title_cicon);
	
	tab_title_cbutton = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (tab_title_cbutton), GTK_RELIEF_NONE);
	
	gtk_widget_set_size_request (tab_title_cbutton, w + 4, h + 4);
	
	gtk_container_add (GTK_CONTAINER (tab_title_cbutton), tab_title_cicon);
	gtk_widget_show (tab_title_cbutton);
	gtk_box_pack_start (GTK_BOX (tab_title_box), tab_title_cbutton, FALSE, FALSE, 0);
	
	gtk_widget_show_all (tab_title_box);
	
	g_signal_connect (GTK_OBJECT (tab_title_cbutton),
			"clicked",
			GTK_SIGNAL_FUNC (close_tab_button_clicked_cb),
			a_this);

	if (ficon_pixbuf)
		gdk_pixbuf_unref (ficon_pixbuf);
	
	return tab_title_box;
}
/* Callbak for the tab title close button */
static void
close_tab_button_clicked_cb (GtkButton* button, gpointer a_this)
{
	mlview_editor_close_xml_document ((MlViewEditor*)a_this, TRUE);
}

static void
mlview_editor_connect_to_app_context (MlViewEditor *a_this,
                                      MlViewAppContext *a_context)
{
        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                          && a_context && MLVIEW_IS_APP_CONTEXT (a_context)) ;

}

static void
mlview_editor_disconnect_from_app_context (MlViewEditor *a_this,
                                           MlViewAppContext *a_context)
{
        g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                          && a_context && MLVIEW_IS_APP_CONTEXT (a_context)) ;
}

static GList *
build_view_list_from_hashtable (GHashTable * a_views)
{
        GList *result = NULL;

        g_hash_table_foreach (a_views,
                              (GHFunc) add_hash_key_to_list,
                              &result);
        return result;
}

static GList *
build_doc_list_from_hashtable (GHashTable * a_docs)
{
        GList *result = NULL;

        g_hash_table_foreach (a_docs,
                              (GHFunc) add_hash_key_to_list,
                              &result);
        return result;
}

static void
add_hash_key_to_list (gpointer a_key,
                      gpointer a_value, GList ** a_list)
{

        g_return_if_fail (a_list != NULL);
        *a_list = g_list_append (*a_list, a_key);
}

static GtkWidget *
build_reload_file_confirmation_dialog (void)
{
        GtkWidget *dialog=NULL, *hbox=NULL, *image=NULL,
                *label=NULL;

        dialog = gtk_dialog_new_with_buttons 
                (_("File already opened"), NULL,
                 GTK_DIALOG_MODAL,
                 GTK_STOCK_NO,GTK_RESPONSE_CANCEL,
                 GTK_STOCK_YES, GTK_RESPONSE_OK,
                 NULL) ;

        gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
        gtk_dialog_set_default_response (GTK_DIALOG (dialog), 
                                         GTK_RESPONSE_OK);

        hbox = gtk_hbox_new (FALSE, 6);
        image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION,
                                          GTK_ICON_SIZE_DIALOG);
        gtk_misc_set_alignment (GTK_MISC (image), 
			        (gdouble)0.5, (gdouble)0.0);
        label = gtk_label_new 
                (_("This document is already open in the editor.\n"
                   " Do you want to reload it ?"));
        gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                            hbox, FALSE, FALSE, 0);

        gtk_widget_show_all (dialog);
        return dialog ;
}

static struct MlViewViewDesc *
mlview_editor_select_view_to_open (void)
{
        GtkWidget *dialog = NULL ;
        GtkWidget *dialog_vbox = NULL ;
        GtkWidget *hbox1 = NULL ;
        GtkWidget *label1 = NULL ;
        GtkWidget *option_menu = NULL ;
        GtkWidget *menu = NULL ;
        GtkWidget *menu_item = NULL ;
        GtkWidget *dialog_action_area1 = NULL ;
        GtkWidget *button5 = NULL ;
        GtkWidget *button6 = NULL ;
        GtkWidget *sel_menu_item = NULL ;

        enum MLVIEW_SELECTED_BUTTON button;
        struct MlViewViewDesc *result = NULL;
        struct MlViewViewDesc *view_desc_ptr = NULL ;
        gchar *base_name = NULL;
        guint nr_view_desc = 0 ;

        nr_view_desc = mlview_editor_get_number_of_view_desc () ;
        g_return_val_if_fail (nr_view_desc, NULL) ;        
        
        /*
         *If there is only one type of view registered
         *in the system, get its view descriptor and use that
         *one.
         */
        if (nr_view_desc == 1) {
                result = mlview_editor_get_view_descriptor_at (0) ;
                g_return_val_if_fail (result, NULL) ;
                return result ;
        }

        dialog = gtk_dialog_new ();
        gtk_window_set_title (GTK_WINDOW (dialog), _("Select View"));
        
        dialog_vbox = GTK_DIALOG (dialog)->vbox;
        gtk_widget_show (dialog_vbox);
        
        hbox1 = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox1);
        gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox1, TRUE, TRUE, 0);
        label1 = gtk_label_new (_("Select view to open"));
        gtk_widget_show (label1);
        gtk_box_pack_start (GTK_BOX (hbox1), label1, FALSE, FALSE, 10);
        
        /* build select view menu */
        option_menu = gtk_option_menu_new();
        menu = gtk_menu_new();
        gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
        
        gtk_widget_show (menu);
        gtk_widget_show (option_menu);
        
        gtk_box_pack_start (GTK_BOX (hbox1), option_menu, TRUE, TRUE, 0);
        for (view_desc_ptr = gv_view_types; 
             view_desc_ptr && view_desc_ptr->view_type_name;
             view_desc_ptr ++) {
                base_name = view_desc_ptr->view_type_name; 
                menu_item = gtk_menu_item_new_with_label(base_name);
                gtk_menu_shell_append (GTK_MENU_SHELL(menu),  menu_item);
                gtk_widget_show(menu_item);
                g_object_set_data (G_OBJECT(menu_item), "mlview_view_desc", (gpointer)view_desc_ptr);
        }
        gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu), 0);

        /* dialog box buttons */
        dialog_action_area1 = GTK_DIALOG (dialog)->action_area;
        gtk_widget_show (dialog_action_area1);
        gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);

        button5 = gtk_button_new_from_stock ("gtk-cancel");
        gtk_widget_show (button5);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button5, GTK_RESPONSE_CANCEL);
        GTK_WIDGET_SET_FLAGS (button5, GTK_CAN_DEFAULT);

        button6 = gtk_button_new_from_stock ("gtk-ok");
        gtk_widget_show (button6);
        gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button6, GTK_RESPONSE_OK);  
        GTK_WIDGET_SET_FLAGS (button6, GTK_CAN_DEFAULT);

        /* show dialog */
        button = gtk_dialog_run (GTK_DIALOG (dialog));


        switch (button) {
        case GTK_RESPONSE_OK:
                sel_menu_item = gtk_menu_get_active(GTK_MENU(menu));  
             
                result = (struct MlViewViewDesc *)g_object_get_data (G_OBJECT(sel_menu_item), "mlview_view_desc");
                break;
                
        default:
                result = NULL ;
                break;
        }
        gtk_widget_destroy (dialog);
        return result; 
}

static enum MlViewStatus 
get_current_view_from_notebook (GtkNotebook *a_notebook,
		                MlViewIView **a_view) 
{
	GList *children_widgets = NULL ;
	gint page_num = 0 ;
	GtkWidget *view_container = NULL ;
	MlViewIView *result= NULL ;

	g_return_val_if_fail (a_notebook && GTK_IS_NOTEBOOK (a_notebook)
			      && a_view,
			      MLVIEW_BAD_PARAM_ERROR); 

	page_num = gtk_notebook_get_current_page (a_notebook) ;
	if (page_num == -1) {
		gint nb_pages = 0 ;
		nb_pages = gtk_notebook_get_n_pages (a_notebook) ;
		if (nb_pages  == 0) {
			*a_view  = NULL ;
			return MLVIEW_OK ;
		} else {
			mlview_utils_trace_debug ("It seems notebook is broken") ;
			return MLVIEW_ERROR ;
		}
	}
	view_container = gtk_notebook_get_nth_page (a_notebook, page_num) ;
	g_return_val_if_fail (view_container && GTK_IS_CONTAINER (view_container), 
			      MLVIEW_ERROR) ;
	children_widgets = gtk_container_get_children 
		(GTK_CONTAINER (view_container)) ;
	g_return_val_if_fail (children_widgets && children_widgets->data, 
			      MLVIEW_ERROR) ;
	result = children_widgets->data ;
	g_return_val_if_fail (result && MLVIEW_IS_IVIEW (result), MLVIEW_ERROR) ;
	*a_view = result ;

	return MLVIEW_OK ;
}

/* NOT USED YET
static enum MlViewStatus 
get_nth_view_from_notebook (GtkNotebook *a_notebook,
		            gint a_page_num,
		            MlViewIView **a_view)
{
	GList *children_widgets = NULL ;
	GtkWidget *view_container = NULL ;
	MlViewIView *result= NULL ;

	g_return_val_if_fail (a_notebook && GTK_IS_NOTEBOOK (a_notebook)
			      && a_view,
			      MLVIEW_BAD_PARAM_ERROR); 

	view_container = gtk_notebook_get_nth_page (a_notebook, a_page_num) ;
	g_return_val_if_fail (view_container && GTK_IS_CONTAINER (view_container), 
			      MLVIEW_ERROR) ;
	children_widgets = gtk_container_get_children 
		(GTK_CONTAINER (view_container)) ;
	g_return_val_if_fail (children_widgets && children_widgets->data, 
			      MLVIEW_ERROR) ;
	result = children_widgets->data ;
	g_return_val_if_fail (result && MLVIEW_IS_IVIEW (result), MLVIEW_ERROR) ;
	*a_view = result ;

	return MLVIEW_OK ;
}
*/

static enum MlViewStatus 
get_view_tab_label_from_notebook (GtkNotebook *a_notebook,
			          MlViewIView *a_view,
				  GtkWidget **a_label)
{
	GtkWidget *view_container = NULL ; 
	GtkWidget *view_impl = NULL ;

	g_return_val_if_fail (a_notebook && GTK_IS_NOTEBOOK (a_notebook)
			      && a_view && MLVIEW_IS_IVIEW (a_view) && a_label,
			      MLVIEW_BAD_PARAM_ERROR) ;

	mlview_iview_get_impl (a_view, &view_impl) ;
	g_return_val_if_fail (view_impl && GTK_IS_WIDGET (view_impl), 
			      MLVIEW_ERROR) ;
	view_container = gtk_widget_get_parent (view_impl) ;
	g_return_val_if_fail (view_container 
			      && GTK_IS_CONTAINER (view_container), MLVIEW_ERROR) ;
	*a_label = gtk_notebook_get_tab_label (a_notebook, view_container) ;
	return MLVIEW_OK ;
}

/* NOT USED YET
static enum MlViewStatus
get_view_page_number_from_notebook (GtkNotebook *a_notebook,
		                    MlViewIView *a_view,
				    gint *a_page_num)
{
	GtkWidget *view_container = NULL, *view_impl = NULL ; 

	g_return_val_if_fail (a_notebook && GTK_IS_NOTEBOOK (a_notebook)
			      && a_view && MLVIEW_IS_IVIEW (a_view) && a_page_num,
			      MLVIEW_BAD_PARAM_ERROR) ;

	mlview_iview_get_impl (a_view, &view_impl) ;
	g_return_val_if_fail (view_impl && GTK_IS_WIDGET (view_impl), 
			      MLVIEW_ERROR) ;
	view_container = gtk_widget_get_parent (view_impl) ;
	g_return_val_if_fail (view_container, MLVIEW_ERROR) ;
	*a_page_num = gtk_notebook_page_num (a_notebook, view_container) ;

	return MLVIEW_OK ;
}
*/

static enum MlViewStatus 
remove_view_page_from_notebook (GtkNotebook *a_notebook,
			        MlViewIView *a_view)
{
	GtkWidget *view_impl = NULL, *view_container = NULL ;
	gint page_num = 0 ;

	g_return_val_if_fail (a_notebook && GTK_IS_NOTEBOOK (a_notebook)
			      && a_view && MLVIEW_IS_IVIEW (a_view),
			      MLVIEW_BAD_PARAM_ERROR) ;

	mlview_iview_get_impl (a_view, &view_impl) ;
	g_return_val_if_fail (view_impl, MLVIEW_ERROR) ;
	view_container = gtk_widget_get_parent (view_impl) ;
	g_return_val_if_fail (view_container 
			      && GTK_IS_CONTAINER (view_container),
			      MLVIEW_ERROR) ;
	page_num = gtk_notebook_page_num (a_notebook, view_container) ;
	g_return_val_if_fail (page_num != -1, MLVIEW_ERROR) ;
	gtk_notebook_remove_page (a_notebook, page_num) ;

	return MLVIEW_OK ;
}

/*=================================================
 *public methods
 *=================================================*/

/**
 *the standard type builder of #MlViewEditor 
 *@return the type id of #MlViewEditor.
 */
GType
mlview_editor_get_type (void)
{
        static GType type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewEditorClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_editor_class_init,
                        NULL, NULL,
                        sizeof (MlViewEditor),
                        0,
                        (GInstanceInitFunc) mlview_editor_init
                };

                type = g_type_register_static (GTK_TYPE_VBOX,
                                               "MlViewEditor",
                                               &type_info, 0);
        }
        return type;
}


/**
 *Creates and returns a new instance of #MlViewEditor.
 *
 *@param a_title the title of the editor (not used yet).
 *@param a_context the current instance of MlViewApplicationContext, or NULL
 *if you want this function to create a new context.
 */
GtkWidget *
mlview_editor_new (const guchar * a_title,
                   MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;
        MlViewAppContext *context = a_context ;

        if (!context) {
               context =  MLVIEW_APP_CONTEXT (mlview_app_context_get_instance ()) ;
               g_return_val_if_fail (context, NULL) ;
        }

        /*
         *if this app context is brand new 
         *(i.e: this instance of #MlViewEditor created it), make sure
         *the current instance of #MlViewEditor is "set" in the 
         *app context
         *If the app context is not "new", don't set the current instance
         *of #MlViewEditor in it. It would be the duty of the caller to
         *set the current instance of #MlViewEditor in the app context,
         *under a name she chooses.
         */
        if (!a_context) {
                mlview_app_context_set_element (context,
                                                "MlViewEditor", 
                                                editor);
        }

        editor = g_object_new (MLVIEW_TYPE_EDITOR, NULL);

        PRIVATE (editor)->app_context = context ;

        if (context) {
                mlview_editor_connect_to_app_context (editor, context) ;
        }

        return GTK_WIDGET (editor);
}


/**
 *The setter of the application context associated to the 
 *current mlview editor.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@param a_context the new application context.
 */
void
mlview_editor_set_app_context (MlViewEditor * a_this,
                               MlViewAppContext * a_context)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (a_context));

        PRIVATE (a_this)->app_context = a_context;
}


/**
 *The getter of the application context associated
 *to the current mlview editor.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@return the application context associated to the editor.
 */
MlViewAppContext *
mlview_editor_get_app_context (MlViewEditor * a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        return PRIVATE (a_this)->app_context;
}

/**
 *Opens the xml file a_file_path, loads it, build a view to edit it and
 *adds this new view to the editor.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@param a_file_path the path of the file to open.
 */
void
mlview_editor_load_xml_file (MlViewEditor * a_this,
                             const gchar * a_file_path,
			     gboolean a_interactive)
{
        mlview_editor_load_xml_file_with_dtd (a_this, a_file_path, NULL,
			                      a_interactive);
        
}

/**
 *Opens the xml file a_file_path, loads it, build a view to edit it and
 *adds this new view to the editor.
 *The file is validated with the DTD if provided or specified in the 
 *document / by the user at the prompt.
 *If a DTD is specified the user won't be prompted for DTD informations.
 *
 *@param a_this the current instance of #MlViewEditor.
 *@param a_file_path the path of the file to open.
 *@param a_dtd_path the path of the dtd to validate against.
 */
void
mlview_editor_load_xml_file_with_dtd (MlViewEditor * a_this,
                                      const gchar * a_file_path,
                                      const gchar * a_dtd_path,
				      gboolean a_interactive)
{
        MlViewXMLDocument *mlview_xml_document = NULL;
        enum MlViewStatus status = MLVIEW_OK;
        gboolean is_relative = FALSE;
        gchar *absolute_path = NULL, *cur_dir = NULL;
        gchar *vfs_uri = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this));

        if (a_file_path == NULL)
                return;

        status = mlview_utils_uri_is_relative (a_file_path, &is_relative);
        
        if (status != MLVIEW_OK) {
                mlview_app_context_error (PRIVATE (a_this)->app_context, 
                                          _("The following URI "
					    "is not well formed: %s"),
                                          a_file_path) ;
                
                return;
        }

        if (is_relative == TRUE) {
                cur_dir = g_get_current_dir ();

                g_return_if_fail (cur_dir);

                mlview_utils_relative_uri_to_absolute_uri (a_file_path,
                                                           cur_dir,
                                                           &absolute_path);

                g_free (cur_dir);
                cur_dir = NULL;
        } else 
                absolute_path = g_strdup (a_file_path);

        g_return_if_fail (absolute_path);

        if (strstr(absolute_path, "://"))
                vfs_uri = g_strdup (absolute_path);
        else 
                vfs_uri = gnome_vfs_get_uri_from_local_path (absolute_path);
        g_return_if_fail (vfs_uri);

        mlview_app_context_sbar_push_message (PRIVATE (a_this)->app_context,
                                              _("Opening file %s..."),
                                              absolute_path);
        
        if ((a_dtd_path && strcmp (a_dtd_path, ""))
	    || !a_interactive) 
                mlview_xml_document =
                        mlview_xml_document_open_with_dtd 
			(vfs_uri,
			 a_dtd_path,
                         PRIVATE (a_this)->app_context);
        else
                mlview_xml_document = 
                        mlview_xml_document_open_with_dtd_interactive 
			(vfs_uri,
                         PRIVATE (a_this)->app_context);
        
        if (mlview_xml_document) {
		struct MlViewViewDesc *view_descriptor = NULL ;
                GtkWidget *parent_window = NULL;
                MlViewIView *new_view = NULL ;

		/* get the descriptor of the default editing view
		 * so that we can instanciate a view of the type
		 * of the default editing view defined in gconf
		 */
		mlview_editor_get_default_view_descriptor 
			(a_this, &view_descriptor) ;

		if (!view_descriptor) {
			/* if we could not get the descriptor 
			 * of the default editing view, use the
			 * source editing view as the default view
			 */
			 new_view = MLVIEW_IVIEW 
				 (mlview_source_view_new 
					 (mlview_xml_document,
				          vfs_uri,
				          PRIVATE (a_this)->app_context)) ;
			 g_return_if_fail (new_view != NULL) ;
			mlview_iview_set_desc_type_name 
				(new_view, 
				 "source-view") ;
		} else {
			new_view = MLVIEW_IVIEW 
				(view_descriptor->view_constructor 
					(mlview_xml_document,
					 vfs_uri,
					 PRIVATE (a_this)->app_context)) ;
			g_return_if_fail (new_view != NULL) ;
			mlview_iview_set_desc_type_name 
				(new_view, 
				 view_descriptor->view_type_name) ;
		}

                g_return_if_fail (new_view != NULL);

                parent_window = gtk_widget_get_toplevel
                        (GTK_WIDGET (a_this));

                mlview_editor_add_view (a_this, new_view) ;
        }

        g_free (absolute_path);
        g_free (vfs_uri);
        absolute_path = NULL;
        vfs_uri = NULL;

        /*
         *FIXME: add support for exception thrown by 
         *mlview_xml_document_open in case 
         *of error and display that error => BIG DEAL
         */

        mlview_app_context_sbar_pop_message (PRIVATE (a_this)->
                                             app_context);
}

/**
 * interactively open a xml file from a remote location
 *
 * @param a_this the current mlview editor
 */
void
mlview_editor_open_xml_document_interactive (MlViewEditor *a_this)
{
        GladeXML *glade_xml=NULL;
        gchar *path=NULL, *uri=NULL;
        gint response = 0;
        GtkDialog *uri_location_dialog = NULL;
        GnomeEntry *uri_location_entry = NULL;

        path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                                          "mlview/mlview-uri-dialog.glade",
                                          TRUE, NULL) ;

        if (!path) {
                mlview_utils_trace_debug ("Couldn't find mlview-uri-dialog.glade");
                return;
        }

        glade_xml = glade_xml_new (path, "URIDialog", NULL);
        if (!glade_xml) {
                mlview_utils_trace_debug ("Couldn't extract dialog from glade file");
                goto cleanup;
        }

        uri_location_dialog =  GTK_DIALOG (glade_xml_get_widget (glade_xml, "URIDialog")) ;
        if (!uri_location_dialog) {
                mlview_utils_trace_debug ("Couldn't get dialog widget from glade file");
                goto cleanup;
        }
        gtk_dialog_set_has_separator (GTK_DIALOG (uri_location_dialog), FALSE);

        uri_location_entry = GNOME_ENTRY (glade_xml_get_widget (glade_xml, "URILocationEntry"));
        if (!uri_location_entry) {
                mlview_utils_trace_debug ("Couldn't get the location entry widget from glade file");
                goto cleanup;
        }

        /* run dialog */
        response = gtk_dialog_run (uri_location_dialog);

        switch (response) {
        case GTK_RESPONSE_OK:
                uri = g_strdup (gtk_entry_get_text (GTK_ENTRY (gnome_entry_gtk_entry (uri_location_entry))));  

                /*
                 *if the document is already opened in the editor
                 *ask the user if she wants to reload it.
                 */
                if (uri && strcmp (uri, "")) {
                        gboolean is_already_opened = FALSE;
                        mlview_editor_is_document_opened_in_editor (a_this,
                                                                    uri,
                                                                    &is_already_opened) ;
                        if (is_already_opened == TRUE) {
                                GtkWidget *dialog = NULL ;
                                gint res =  0 ;
                                dialog = build_reload_file_confirmation_dialog () ;
                                if (dialog) {
                                        res = gtk_dialog_run (GTK_DIALOG (dialog)) ;
                                        switch (res) {
                                        case GTK_RESPONSE_OK:
                                                gnome_entry_prepend_history 
							(uri_location_entry, 
							 TRUE, uri);
                                                mlview_editor_load_xml_file 
							(a_this, uri, TRUE);
                                                break ;
                                        case GTK_RESPONSE_CANCEL:
                                                break;
                                        default:
                                                g_assert_not_reached () ;
                                        }
                                        gtk_widget_destroy (dialog) ;
                                        dialog = NULL ;
                                }
                        } else {
                                gnome_entry_prepend_history (uri_location_entry, TRUE, uri);
                                mlview_editor_load_xml_file  (a_this, uri, TRUE);
                        }
                }
                break;
        case GTK_RESPONSE_CANCEL:
        default:
                break;
        }
        gtk_widget_destroy (GTK_WIDGET (uri_location_dialog));

	if (uri) {
		EggRecentModel *model = NULL;

		model = (EggRecentModel*)
		mlview_app_context_get_element (
				PRIVATE (a_this)->app_context,
				"MlViewRecentModel");
		g_return_if_fail (model && EGG_IS_RECENT_MODEL (model)) ;
		egg_recent_model_add (model, uri);
	}

        if (uri) {
                g_free (uri);
                uri = NULL;
        }
        
 cleanup:        
        if (path) {
                g_free (path) ;
                path = NULL ;
        }
        if (glade_xml) {
                g_object_unref (glade_xml) ;
                glade_xml = NULL ;
        }

        return;                
}

/**
 *interactively open/edit a local xml file name.
 *@param a_this the current mlview editor.
 */
void
mlview_editor_open_local_xml_document_interactive (MlViewEditor *a_this)
{
        gchar *file_name = NULL;
#ifndef MLVIEW_WITH_GTK_FILE_CHOOSER
        MlViewFileSelection *file_selector;
        enum MLVIEW_SELECTED_BUTTON button;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->app_context !=
                          NULL);

        file_selector =
                mlview_app_context_get_file_selector
                (PRIVATE (a_this)->app_context,
                 _("Open xml document"));

        g_return_if_fail (file_selector != NULL);

        mlview_app_context_sbar_push_message
                (PRIVATE (a_this)->app_context,
                 _("Choose the xml file to open"));

        button = mlview_file_selection_run
                (MLVIEW_FILE_SELECTION (file_selector), TRUE);

        switch (button) {
        case OK_BUTTON:
        {
		EggRecentModel *model = NULL;
                gboolean is_already_opened = FALSE ;
		gchar *file_uri = NULL;
                file_name =
                        g_strdup (gtk_file_selection_get_filename
                                  (GTK_FILE_SELECTION
                                   (file_selector)));
		file_uri = gnome_vfs_get_uri_from_local_path (file_name);

                /*
                 *if the document is already opened in the editor
                 *ask the user if she wants to reload it.
                 */
                if (file_name && strcmp (file_name, "")) {
                        mlview_editor_is_document_opened_in_editor (a_this,
                                                                    file_uri, /*file_name*/
                                                                    &is_already_opened) ;
                        if (is_already_opened == TRUE) {
                                GtkWidget *dialog = NULL ;
                                gint res =  0 ;
                                dialog = build_reload_file_confirmation_dialog () ;
                                if (dialog) {
                                        res = gtk_dialog_run (GTK_DIALOG (dialog)) ;
                                        switch (res) {
                                        case GTK_RESPONSE_OK:
                                                mlview_editor_load_xml_file 
                                                        (a_this, file_name);
                                                break ;
                                        case GTK_RESPONSE_CANCEL:
                                                break;
                                        default:
                                                g_assert_not_reached () ;
                                        }
                                        gtk_widget_destroy (dialog) ;
                                        dialog = NULL ;
                                }
                        } else {
                                mlview_editor_load_xml_file 
                                        (a_this, file_name);
                        }
                }

		/* Add entry to list of recent items */
		model = (EggRecentModel*)
			mlview_app_context_get_element (
				PRIVATE (a_this)->app_context,
				"MlViewRecentModel");

                if (file_name) {
                        g_free (file_name);
                        file_name = NULL;
                }
        }
        break ;
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                break;
        }

        mlview_app_context_sbar_pop_message
                (PRIVATE (a_this)->app_context);
#else
	gint response = 0;

	GtkWidget *file_dialog = GTK_WIDGET (mlview_app_context_get_file_chooser
                                             (PRIVATE (a_this)->app_context, 
                                              _("Open xml document"), MlViewFileChooserOpenMode)) ;
        g_return_if_fail (file_dialog != NULL);

        mlview_app_context_sbar_push_message
                (PRIVATE (a_this)->app_context,
                 _("Choose the xml file to open"));

	response = gtk_dialog_run (GTK_DIALOG(file_dialog));
	gtk_widget_hide (file_dialog);

	if (response == GTK_RESPONSE_OK) {
		gchar *file_uri = NULL;
                file_name = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_dialog));
		file_uri = gnome_vfs_get_uri_from_local_path (file_name);

                if (file_name && strcmp (file_name, "")) {
                        gboolean is_already_opened = FALSE ;
                        mlview_editor_is_document_opened_in_editor (a_this,
                                                                    file_uri,
                                                                    &is_already_opened) ;
                        if (is_already_opened == TRUE) {
                                GtkWidget *dialog = NULL ;
                                gint res =  0 ;
                                dialog = build_reload_file_confirmation_dialog () ;
                                if (dialog) {
                                        res = gtk_dialog_run (GTK_DIALOG (dialog)) ;
                                        switch (res) {
                                        case GTK_RESPONSE_OK:
                                                mlview_editor_load_xml_file 
                                                        (a_this, file_uri, TRUE);
                                                break ;
                                        case GTK_RESPONSE_CANCEL:
                                                break;
                                        default:
                                                g_assert_not_reached () ;
                                        }
                                        gtk_widget_destroy (dialog) ;
                                        dialog = NULL ;
                                }
                        } else {
                                mlview_editor_load_xml_file (a_this, file_uri,
						             TRUE) ;
                        }
                }

		if (file_uri) {
			EggRecentModel *model = NULL;

			model = (EggRecentModel*)
				mlview_app_context_get_element (
						PRIVATE (a_this)->app_context,
						"MlViewRecentModel");
			egg_recent_model_add (model, file_uri);
		}

		if (file_uri) {
			g_free (file_uri);
			file_uri = NULL;
		}
		
		if (file_name) {
			g_free (file_name);
			file_name = NULL;
		}
	}

        mlview_app_context_sbar_pop_message
                (PRIVATE (a_this)->app_context);
#endif
}

/**
 *Creates a new view on an existing document.
 *
 *@param a_this the current mlview editor.
 *@param a_xml_doc the xml document on which the new
 *view must be created.
 *@param a_view_type the type of view.
 *@return the newly created view.
 */
MlViewIView *
mlview_editor_create_new_view_on_document (MlViewEditor *a_this,
                                           MlViewXMLDocument *a_xml_doc)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);
        g_return_val_if_fail (a_xml_doc != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this)->app_context,
                              NULL) ;

	return mlview_editor_create_new_view_on_document2 
		(a_this, a_xml_doc, NULL) ;
}

MlViewIView *
mlview_editor_create_new_view_on_document2 (MlViewEditor *a_this,
		                            MlViewXMLDocument *a_xml_doc,
					    const gchar *a_view_desc_type_name)
{
        MlViewIView *result = NULL;
	char *view_desc_type_name = NULL ;
        struct MlViewAppSettings *settings = NULL ;
        struct MlViewViewDesc *view_descriptor = NULL ;
        
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);
        g_return_val_if_fail (a_xml_doc != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this)->app_context,
                              NULL) ;

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_val_if_fail (settings, NULL) ;

	view_desc_type_name = (gchar*)a_view_desc_type_name ;
	if (view_desc_type_name) {
		view_descriptor = mlview_editor_peek_editing_view_descriptor 
			(view_desc_type_name) ;
	} 
	if (!view_descriptor) {
		view_descriptor = mlview_editor_select_view_to_open ();
	}

        if (!view_descriptor
            || !view_descriptor->view_constructor) {
                mlview_utils_trace_debug
                        ("Unknown view type name: ") ;
                mlview_utils_trace_debug (settings->general.default_editing_view_type) ;
                mlview_utils_trace_debug ("This may be caused by a gconfd "
                                         "problem or a bad mlview default "
                                         "view type name gconf key\n"
                                         "First, try to killall gconfd and restart it\n"
                                         "If you still have the problem, send a mail to"
                                         "mlview-list@gnome.org to ask for help\n") ;
                return NULL ;
        }
        result = MLVIEW_IVIEW (view_descriptor->view_constructor
                               (a_xml_doc, NULL,
                                PRIVATE (a_this)->app_context)) ;
	g_return_val_if_fail (result, NULL) ;

	mlview_iview_set_desc_type_name (result, 
			                 view_descriptor->view_type_name) ;
        /*mlview_editor_add_view (a_this, result) ;*/
        return result;
}

MlViewIView *
mlview_editor_create_new_view_on_document3 (MlViewEditor *a_this,
		                            MlViewXMLDocument *a_doc,
					    struct MlViewViewDesc *a_desc)
{
        MlViewIView *result = NULL ;	
	g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
			      && a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)
			      && a_desc, NULL) ;

	result = mlview_editor_create_new_view_on_document2 (a_this, a_doc, 
			                                     a_desc->view_type_name) ;
	return result ;
}

MlViewIView *
mlview_editor_create_new_view_on_current_document (MlViewEditor *a_this,
						   struct MlViewViewDesc *a_desc)
{
	MlViewXMLDocument *doc = NULL ;
	MlViewIView *result = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this) && a_desc, NULL) ;
        doc = mlview_editor_get_current_document (a_this) ;	
	if (!doc) {
		mlview_utils_trace_debug ("Could not get the current doc being edited") ;
		return NULL ; 
	}
	result = mlview_editor_create_new_view_on_document3 (a_this, doc, a_desc) ;
	return result ;
}

/**
 *Interactively creates a new view on an existing
 *xml document.
 *The view will be created if and only if an non empty document is
 *currently selected in the editor.
 *
 *@param a_this the current mlview editor.
 *@return the newly created view or NULL if it could not be created.
 */
MlViewIView*
mlview_editor_create_new_view_on_current_document_interactive (MlViewEditor * a_this) 
{
        MlViewXMLDocument *xml_document = NULL;
	MlViewIView *result = NULL ; 

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);


        if (PRIVATE (a_this)->cur_view == NULL)
                return NULL;

        mlview_iview_get_document
                (PRIVATE (a_this)->cur_view, &xml_document);

        result = mlview_editor_create_new_view_on_document
                (a_this, xml_document);
	mlview_editor_add_view (a_this, result) ;
	return result ;
}

/**
 *Getter of the current selected edition
 *view.
 *@param a_this the current instance of #MlViewEditor.
 *@return the current MlViewXMLDocumentView or NULL.
 */

MlViewIView *
mlview_editor_get_current_document_view (MlViewEditor * a_this)
{
        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL);

        return PRIVATE (a_this)->cur_view;
}


/**
 *Getter of the current selected instance of
 *#MlViewXMLDocument *
 *
 *@param a_this the current instance of #MlViewEditor.
 *@return the current selected instance of #MlViewXMLDocument,
 *or NULL.
 */
MlViewXMLDocument *
mlview_editor_get_current_document (MlViewEditor * a_this)
{
        MlViewXMLDocument *doc = NULL ;

        g_return_val_if_fail (a_this && PRIVATE (a_this), NULL);

        if (PRIVATE (a_this)->cur_view) {
                mlview_iview_get_document
                        (PRIVATE (a_this)->cur_view, &doc);
        }

        return doc ;
}

/**
 *Returns the number of views opened with the document, or
 *-1 if an error occurs.
 *@param a_this the current instance.
 *@param a_doc the XML document.
 *@return the corresponding number of views.
 */
gint
mlview_editor_get_number_of_views_opened_with_doc (MlViewEditor *a_this,
                                                   MlViewXMLDocument *a_doc)
{
        GHashTable *views_related_to_document = NULL;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this) &&
                              PRIVATE (a_this) && PRIVATE (a_this)->mlview_xml_docs, -1);
        g_return_val_if_fail (a_doc, -1);

        views_related_to_document =
                g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_docs,
                 a_doc);

        if (!views_related_to_document)
                return 0;
        else
                return g_hash_table_size (views_related_to_document);
}

/**
 *Getter of the glist of opened docs
 *#MlViewXMLDocument *
 *
 *@param a_this the current instance of #MlViewEditor.
 *@return a GList* of #MlViewXMLDocument, taht the caller 
 * is responsble to free
 */
GList *
mlview_editor_get_list_of_open_documents (MlViewEditor * a_this)
{
   GList *docs = NULL;

   docs = build_doc_list_from_hashtable
                (PRIVATE (a_this)->mlview_xml_docs);

   return docs;
}

/**
 *Setter of the name of the currently selected view.
 *
 *@param a_this the current mlview editor.
 *@param a_name the new name of the view.
 */
void
mlview_editor_set_current_view_name (MlViewEditor * a_this, gchar * a_name)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->cur_view == NULL)
                return;
        mlview_iview_set_name
                (PRIVATE (a_this)->cur_view, a_name);
}


/**
 *Interactively sets the name of the view currently
 *selected in the mlview editor.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_set_current_view_name_interactive (MlViewEditor *a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->cur_view == NULL)
                return;

        mlview_iview_set_name_interactive
                (PRIVATE (a_this)->cur_view);
}


/**
 *Adds a document view to the editor. If a_xml_doc_view is a new 
 *document view, adds a new notebook page to the #MlViewEditor notebook.
 *The base name of the document is shown in the notebook page tab. 
 *If the underlying document of a_xml_doc_view has the 
 *same base name as a document already opened,
 *the notebook page tab string will be "basename<nb>" where 'nb' is 
 *the number of docs that have the same name in the editor (like in emacs).
 *If a_xml_doc_view is already loaded in the current instance of #MlViewEditor, 
 *the old instance of a_xml_doc is destroyed 
 *(as well as the notebook page that contains it) and the new 
 *instance of a_xml_doc is added to the editor. 
 *
 *@param a_this the current mlview editor.
 *@param a_view the document view to be added to the editor.
 */
void
mlview_editor_add_view (MlViewEditor * a_this,
                        MlViewIView *a_view)
{
	g_return_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
		          && a_view && MLVIEW_IS_IVIEW (a_view)) ;

	mlview_editor_add_view_at_index (a_this, a_view, -1) ;
}


/**
 *Adds an editing view at a certain index in the editor's notebook.
 * If a_view is a new 
 * If a_view is a new document view, adds a new notebook page to 
 * the #MlViewEditor notebook.
 * The base name of the document is shown in the notebook page tab. 
 * If the underlying document of a_xml_doc_view has the 
 * same base name as a document already opened,
 * the notebook page tab string will be "basename<nb>" where 'nb' is 
 * the number of docs that have the same name in the editor (like in emacs).
 * If a_xml_doc_view is already loaded in the current instance of #MlViewEditor, 
 * the old instance of a_xml_doc is destroyed 
 * (as well as the notebook page that contains it) and the new 
 * instance of a_xml_doc is added to the editor. 
 * @param a_this the current instance of #MlViewEditor.
 * @param a_view the view to add to the editor
 * @param a_index the index at which the view is to be added into the editor.
 * If set to -1, then the view as to be added after all the view tabs.
 * @return MLVIEW_OK upon sucessful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_editor_add_view_at_index (MlViewEditor *a_this,
		                 MlViewIView *a_view,
				 glong a_index)
{

        MlViewFileDescriptor *file_desc = NULL;
        guchar *file_path = NULL,
	       *file_uri = NULL,
                *base_name = NULL,
                *label_str = NULL;
        MlViewXMLDocument *mlview_xml_document = NULL;
        GHashTable *views_associated_to_document = NULL;
        MlViewIView *iview = NULL;
        gpointer ptr = NULL ;
        gboolean is_new_doc_tree = TRUE;
	gboolean is_local = FALSE;
	GtkWidget *label = NULL,
		  *view_impl = NULL,
		  *view_impl_tmp = NULL,
		  *view_container =  NULL;

        g_return_val_if_fail (a_this != NULL, MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (PRIVATE (a_this)->notebook != NULL,
		              MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (a_view != NULL, MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (MLVIEW_IS_IVIEW (a_view), MLVIEW_BAD_PARAM_ERROR) ;
        mlview_iview_get_impl (a_view, &view_impl) ;
        g_return_val_if_fail (view_impl, MLVIEW_BAD_PARAM_ERROR) ;
        g_return_val_if_fail (PRIVATE (a_this)->mlview_xml_docs, 
			      MLVIEW_BAD_PARAM_ERROR);
        g_return_val_if_fail (PRIVATE (a_this)->mlview_xml_doc_views, 
			      MLVIEW_BAD_PARAM_ERROR);

        mlview_iview_get_document (a_view, &mlview_xml_document) ;
        g_return_val_if_fail (mlview_xml_document != NULL,
			      MLVIEW_BAD_PARAM_ERROR);

        file_desc = mlview_xml_document_get_file_descriptor 
                (mlview_xml_document) ;
        if (file_desc) {
		file_uri =
			mlview_file_descriptor_get_uri
			(file_desc); /* used for storing name of opened documents */
                file_path =
                        mlview_file_descriptor_get_file_path
                        (file_desc) ;

		/* use filename for local files */
		if (!mlview_file_descriptor_is_local (file_desc, &is_local) ) {
			if (is_local)
				file_path =
					g_path_get_basename (file_path);
		}
	}

        /*
         *check if a view on the same document has been added to
         *the editor already.
         */
        views_associated_to_document =
                g_hash_table_lookup
                (PRIVATE (a_this)->mlview_xml_docs,
                 mlview_xml_document);

        if (views_associated_to_document) {
                is_new_doc_tree = FALSE;
        }


        if (file_path == NULL) {
                gchar *tmp_str = NULL,
                        *label_str = NULL;

                if (is_new_doc_tree == TRUE)
                        PRIVATE (a_this)->untitled_docs_num++;
                tmp_str =
                        g_strdup_printf
                        ("%d",
                         PRIVATE (a_this)->untitled_docs_num);

                label_str =
                        g_strconcat ("Untitled Document ", tmp_str, NULL);
                label = gtk_label_new (label_str);
                g_free (label_str);
                g_free (tmp_str);

        } else {

                gint base_name_nb = 0;

                gboolean file_is_already_opened = FALSE;

                base_name = g_strdup (file_path);
		
		/* crop name for long uris */
		if (strlen (base_name) > 23) {
			base_name[20] = '.';
			base_name[21] = '.';
			base_name[22] = '.';
			base_name[23] = '\0';
		}
		
                if (is_new_doc_tree
                    && (iview = g_hash_table_lookup
                        (PRIVATE (a_this)->opened_file_paths,
                         file_uri)) != NULL) {
                        /*
                         *There is an xml document coming from the
                         *same url that is opened already. 
                         *So, reopen it.
                         *That is, remove it previous 
                         *instance and add
                         *the new one.
                         */
                        GtkWidget *old_label = NULL ;
                        gchar *old_label_str_tmp = NULL,
                                *old_label_str = NULL;
			GList *children = NULL, *p = NULL;

                        /*
                         *get the old label string because 
                         *this document
                         *must have the same label as 
                         *the one alreay opened
                         */
                        mlview_iview_get_impl (iview, 
                                               &view_impl_tmp) ;
			/* <ugly hack> */
			children = gtk_container_get_children (
					GTK_CONTAINER (gtk_notebook_get_tab_label (
							PRIVATE (a_this)->notebook,
							view_impl_tmp)));
			p = g_list_next (children);
			old_label = (GtkWidget*)p->data;
			/* </ugly hack> */
                        g_assert (old_label != NULL);

                        gtk_label_get (GTK_LABEL (old_label),
                                       &old_label_str_tmp);

                        /*
                         *make a copy of the label string because
                         *mlview_editor_remove_xml_document ()
                         *will destroy this label.
                         */

                        /*old_label_str_tmp belongs to label,
                         *and will be freed by him
                         */
                        old_label_str =
                                g_strdup (old_label_str_tmp);

                        mlview_editor_remove_view
                                (a_this, iview);

                        /*create the label of this document notebook page */
                        label = gtk_label_new (old_label_str);
                        /*old_label_str has been strduped */
                        g_free (old_label_str);
                        gtk_label_get (GTK_LABEL (label),
                                       &old_label_str);
                        /*
                         *old_label_str belongs to 
                         *label and will be freed by him
                         *so we can use it as a key in a hash 
                         *table without any need of
                         *freeing it.
                         */
                        g_hash_table_insert
                                (PRIVATE
                                 (a_this)->
                                 opened_document_label_names,
                                 g_strdup (old_label_str),
                                 a_view);

                        file_is_already_opened = TRUE;

                } else if ((ptr = g_hash_table_lookup
                            (PRIVATE (a_this)->
                             opened_file_base_names,
                             base_name)) == NULL) {
                        /*
                         *It is the first time a 
                         *document with this basename is opened
                         */
                        base_name_nb = 1;

                } else if (ptr != NULL) {
                        /*
                         *some documents with the this basename 
                         *are already opened
                         */
                        base_name_nb = GPOINTER_TO_INT (ptr);
                        if (!is_new_doc_tree)
                                base_name_nb++;
                }

                g_hash_table_insert
                        (PRIVATE (a_this)->opened_file_base_names, 
                         base_name, GINT_TO_POINTER (base_name_nb));

                g_hash_table_insert
                        (PRIVATE (a_this)->opened_file_paths,
                         file_uri, a_view);
                if (base_name_nb > 1) {
                        gchar *tmp_str = NULL ;
                        while (1) {
                                tmp_str =
                                        g_strdup_printf 
                                        ("%d", base_name_nb);
                                label_str =
                                        g_strconcat (base_name,
                                                     "<", tmp_str,
                                                     ">", NULL);

                                if (g_hash_table_lookup
                                    (PRIVATE (a_this)->
                                     opened_document_label_names,
                                     label_str)) {
                                        base_name_nb++;
                                        g_free (tmp_str);
                                        continue;
                                }
                                break;
                        }                        
                        label = gtk_label_new (label_str);
                        g_hash_table_insert
                                (PRIVATE
                                 (a_this)->
                                 opened_document_label_names,
                                 g_strdup (label_str),
                                 a_view) ;
                        g_free (tmp_str);
                        g_free (label_str);
                } else if (file_is_already_opened == FALSE) {
                        label = gtk_label_new (base_name);
                        g_hash_table_insert
                                (PRIVATE
                                 (a_this)->
                                 opened_document_label_names,
                                 g_strdup (base_name),
                                 a_view);
                }
        }

        /*update the view->document index */
        g_hash_table_insert 
                (PRIVATE (a_this)->mlview_xml_doc_views,
                 a_view, mlview_xml_document) ;

        /*update the document->views index */
        views_associated_to_document =
                g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_docs,
                 mlview_xml_document);
        if (!views_associated_to_document) {
                views_associated_to_document =
                        g_hash_table_new (g_direct_hash,
                                          g_direct_equal);
                g_assert (views_associated_to_document != NULL);
                g_hash_table_insert 
                        (PRIVATE (a_this)->mlview_xml_docs,
                         mlview_xml_document,
                         views_associated_to_document);
        }
        g_hash_table_insert (views_associated_to_document,
                             a_view,
                             mlview_xml_document);
        if (is_new_doc_tree == TRUE)
                PRIVATE (a_this)->opened_docs_num++;

	/*now, visually add the view */
	g_signal_handlers_block_by_func (G_OBJECT (PRIVATE (a_this)->notebook),
			                 mlview_editor_switch_notebook_page_cb,
					 a_this) ;
	view_container = gtk_vbox_new (TRUE, 0) ;
	gtk_box_pack_start_defaults (GTK_BOX (view_container), view_impl) ;
        gtk_notebook_insert_page (PRIVATE (a_this)->notebook,
                                  view_container, NULL, a_index);
	g_signal_handlers_unblock_by_func (G_OBJECT (PRIVATE (a_this)->notebook),
			                   mlview_editor_switch_notebook_page_cb,
					   a_this) ;
        g_signal_connect (G_OBJECT (a_view),
                          "name-changed",
                          G_CALLBACK (view_name_changed_cb),
                          a_this);
        label_str = (guchar*)gtk_label_get_text (GTK_LABEL (label)) ;
        if (label_str) {
                mlview_iview_set_name (a_view, label_str) ;
                label_str = NULL ;
        }

        /*notify the view that it is "swapped in"*/
        mlview_iview_notify_swapped_in (MLVIEW_IVIEW (view_impl)) ;

        /*make sure the newly added view comes on "front"*/
	g_signal_handlers_block_by_func (G_OBJECT (PRIVATE (a_this)->notebook),
			                 mlview_editor_switch_notebook_page_cb,
					 a_this) ;
        gtk_notebook_set_current_page
                (PRIVATE (a_this)->notebook, a_index) ;
	g_signal_handlers_unblock_by_func (G_OBJECT (PRIVATE (a_this)->notebook),
			                   mlview_editor_switch_notebook_page_cb,
					   a_this) ;

        if (g_hash_table_size 
            (PRIVATE (a_this)->mlview_xml_doc_views) == 1) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[FIRST_VIEW_ADDED], 0,
                               a_view) ;
        }
        gtk_widget_show_all (GTK_WIDGET (a_this));

	return MLVIEW_OK ;
}


/**
 * Replaces an editing view with a new one built on the same document.
 * The former view is unrefed. Depending on its refcount, it is destroyed
 * or not.
 * @param a_this the current instance of #MlViewEditor.
 * @param a_view the view to rebuild.
 * @return MLVIEW_OK upon succesful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_editor_rebuild_view (MlViewEditor *a_this,
		            MlViewIView *a_view,
			    MlViewIView **a_new_view)
{
	enum MlViewStatus status = MLVIEW_OK ;
	gchar *view_type_name = NULL ;
	glong tab_num = 0 ;
	GtkWidget *view_impl = NULL, 
		  *new_view_impl = NULL,
		  *view_container = NULL ;
	MlViewIView *new_view = NULL ;
	MlViewXMLDocument *cur_doc = NULL ;
	GHashTable* views_related_to_doc = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
			      && PRIVATE (a_this) && MLVIEW_IS_IVIEW (a_view), 
			      MLVIEW_BAD_PARAM_ERROR) ;

	g_return_val_if_fail (PRIVATE (a_this)->notebook 
			      && GTK_IS_NOTEBOOK (PRIVATE (a_this)->notebook),
			      MLVIEW_BAD_PARAM_ERROR) ;

	mlview_iview_get_impl (a_view, &view_impl) ;
	if (!view_impl) {
		mlview_utils_trace_debug ("view_impl is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
	mlview_iview_get_document (a_view, &cur_doc) ;
	if (!cur_doc) {
		mlview_utils_trace_debug ("cur_doc is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;

	}
	mlview_iview_get_desc_type_name (a_view, &view_type_name) ;
	if (!view_type_name) {
		mlview_utils_trace_debug ("view type name is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
	view_container = gtk_widget_get_parent (view_impl) ;
	if (!view_container) {
		mlview_utils_trace_debug ("view not added to the editor yet") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
	tab_num = gtk_notebook_page_num (PRIVATE (a_this)->notebook, 
			                 view_container) ;
	if (tab_num < 0) {
		mlview_utils_trace_debug ("the view not part of the editor") ;
		status = MLVIEW_ERROR ;
		goto  out ;
	}

	new_view = mlview_editor_create_new_view_on_document2 (a_this, 
			                                       cur_doc,
							       view_type_name) ;
	if (!new_view) {
		mlview_utils_trace_debug ("new_view is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
	mlview_iview_get_impl (new_view, &new_view_impl) ;
	if (!new_view) {
		mlview_utils_trace_debug ("new_view_impl is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
	g_signal_handlers_block_by_func 
				(G_OBJECT (PRIVATE (a_this)->notebook),
				 mlview_editor_switch_notebook_page_cb,
				 a_this) ;

	view_container = gtk_notebook_get_nth_page 
		(PRIVATE (a_this)->notebook,
	         tab_num) ;
        if (!view_container) {
		mlview_utils_trace_debug ("view container is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
        gtk_container_remove (GTK_CONTAINER (view_container),
			      view_impl) ;
	gtk_widget_show_all (new_view_impl) ;
	gtk_box_pack_start_defaults (GTK_BOX (view_container), new_view_impl) ;
	gtk_widget_show_all (view_container) ;

	views_related_to_doc = g_hash_table_lookup 
		(PRIVATE (a_this)->mlview_xml_docs, cur_doc) ;
	if (!views_related_to_doc) {
		mlview_utils_trace_debug ("view_related_to_doc is NULL") ;
		status = MLVIEW_ERROR ;
		goto out ;
	}
        g_signal_connect (G_OBJECT (new_view_impl),
                          "name-changed",
                          G_CALLBACK (view_name_changed_cb),
                          a_this);
	g_hash_table_remove (views_related_to_doc, a_view) ;
	g_hash_table_remove (PRIVATE (a_this)->mlview_xml_doc_views,
			     a_view) ;
	
	/*mlview_editor_add_view_at_index (a_this, new_view, tab_num) ;*/

	g_signal_handlers_unblock_by_func 
				(G_OBJECT (PRIVATE (a_this)->notebook),
				 mlview_editor_switch_notebook_page_cb,
				 a_this) ;

	g_hash_table_insert (views_related_to_doc, new_view, cur_doc) ;

	g_hash_table_insert (PRIVATE (a_this)->mlview_xml_doc_views,
			     new_view, cur_doc) ;

	*a_new_view = new_view ;
	new_view = NULL ;

out:
	if (new_view) {
		g_object_unref (G_OBJECT (new_view)) ;
		new_view = NULL ;
	}

	return status ;
}

/**
 *removes the document view a_view from the editor.
 *This method does not save the document.
 *
 *@param a_this the current mlview editor.
 *@param a_view the document view to remove from the editor.
 */
void
mlview_editor_remove_view (MlViewEditor * a_this,
                           MlViewIView *a_view)
{
        GtkWidget *label = NULL, *view_impl = NULL, *tab_label = NULL ;
        gpointer *ptr = NULL ;
        MlViewFileDescriptor *file_desc = NULL;
        MlViewXMLDocument *mlview_xml_doc = NULL;
        GHashTable *views_related_to_document = NULL;
        gboolean doc_to_be_closed = FALSE;
        guchar *file_path = NULL,
	       *file_uri = NULL,
                *base_name = NULL,
                *label_str = NULL,
		**str_ptr = NULL;
	GList *children = NULL, *p = NULL;
	MlViewIView *cur_view = NULL ;
        
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail 
                (PRIVATE (a_this)->mlview_xml_doc_views != NULL);
        g_return_if_fail (a_view != NULL);
        g_return_if_fail (MLVIEW_IS_IVIEW (a_view)) ;
        mlview_iview_get_impl (a_view, &view_impl) ;
        g_return_if_fail (view_impl) ;

        mlview_iview_get_document (a_view, &mlview_xml_doc) ;
        g_return_if_fail (mlview_xml_doc);

        file_desc = mlview_xml_document_get_file_descriptor
                (mlview_xml_doc) ;
        if (file_desc) {
		file_uri =
			mlview_file_descriptor_get_uri
			(file_desc);
                file_path =
                        mlview_file_descriptor_get_file_path
                        (file_desc);
	}
        if (file_path != NULL)
                base_name = (guchar *) g_basename (file_path);
	/* <ugly hack> */
	get_view_tab_label_from_notebook (PRIVATE (a_this)->notebook,
					  a_view,
			                  &tab_label) ; 
	g_return_if_fail (tab_label) ;

	children = gtk_container_get_children (GTK_CONTAINER (tab_label)) ;
	g_return_if_fail (children) ;

	p = g_list_next (children);
	label = (GtkWidget*)p->data;
	/* </ugly hack> */

	str_ptr = &label_str ;
        gtk_label_get (GTK_LABEL (label), (char **) str_ptr);
        label_str = g_strdup (label_str);
        g_return_if_fail (label != NULL);

        /*
         *check if the document view a_view
         *is opened in this instance of MlViewEditor.
         */
        ptr = g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_doc_views,
                 a_view);
        g_return_if_fail (ptr != NULL);
        /*
         *removes a_view from the hashtable of 
         *the opened document views.
         *and from the opened_file_paths hash_table.
         */
        g_hash_table_remove (PRIVATE (a_this)->mlview_xml_doc_views,
                             a_view);
        views_related_to_document =
                g_hash_table_lookup 
                (PRIVATE (a_this)->mlview_xml_docs,
                 mlview_xml_doc);
        g_return_if_fail (views_related_to_document != NULL);

        ptr = g_hash_table_lookup (views_related_to_document, 
                                   a_view) ;
        g_return_if_fail (ptr != NULL) ;
        g_hash_table_remove (views_related_to_document,
                             a_view) ;

        /*
         *removes the notebook page that contains the view 
         */

        /*ref the view_impl so the the removal of the notebook page
         *that contain it doesn't destroy it. This way, signal
         *sent to notify the view removal won't happen *after* the view_impl
         *is destroyed, but rather before.
         */
        g_object_ref (G_OBJECT (view_impl)) ;

        /*
         *notify the view that it is going to be "swapp out"
         */
        mlview_iview_notify_swapped_out (MLVIEW_IVIEW (view_impl)) ;

	remove_view_page_from_notebook (PRIVATE (a_this)->notebook, a_view) ;

	/*Set the new current view because the "switch-page"
	 * signal we connected to for this doesn't get called
	 * upon page removal ... weird.
	 */
	get_current_view_from_notebook (PRIVATE (a_this)->notebook, &cur_view) ;
	PRIVATE (a_this)->cur_view = cur_view ;

        if (g_hash_table_size (views_related_to_document) == 0) {
                /*
                 *no views are opened on the current 
                 *doc anymore=>close the doc
                 */
                g_hash_table_remove (PRIVATE (a_this)->
                                     mlview_xml_docs,
                                     mlview_xml_doc);

                if (file_uri != NULL) {
                        g_hash_table_remove
                                (PRIVATE (a_this)->
                                 opened_file_paths, file_uri);
                }

                doc_to_be_closed = TRUE;

                PRIVATE (a_this)->opened_docs_num--;
        }

        if (doc_to_be_closed && label_str) {
                /*
                 *remove the entry in the 
                 *opened_document_label_names hash table
                 */
                g_hash_table_remove
                        (PRIVATE (a_this)->
                         opened_document_label_names, label_str);

                g_free (label_str);
                label_str = NULL;
        }

        /*if there are several docs that have the save base name as this one,
         *decrement their number, and if the number reaches 0, 
         *remove the entry matching this base name
         *from the hash table.
         */
        if (doc_to_be_closed == TRUE && file_path != NULL) {
                gint tmp_int;

                ptr = g_hash_table_lookup
                        (PRIVATE (a_this)->
                         opened_file_base_names, base_name);

                tmp_int = GPOINTER_TO_INT (ptr);
                tmp_int--;

                if (tmp_int == 0) {
                        g_hash_table_remove
                                (PRIVATE (a_this)->
                                 opened_file_base_names,
                                 base_name);
                } else {
                        ptr = GINT_TO_POINTER (tmp_int);
                        g_hash_table_insert
                                (PRIVATE (a_this)->
                                 opened_file_base_names,
                                 base_name, ptr);
                }
        } else if (doc_to_be_closed == TRUE && !file_path) {
                PRIVATE (a_this)->untitled_docs_num--;
        }

        if (doc_to_be_closed && mlview_xml_doc) {
                mlview_xml_document_unref (mlview_xml_doc);
                mlview_xml_doc = NULL;
        }
        if (g_hash_table_size
            (PRIVATE (a_this)->mlview_xml_doc_views) == 0) {
                g_signal_emit (G_OBJECT (a_this),
                               gv_signals[LAST_VIEW_REMOVED], 0) ;
        }
        g_object_unref (G_OBJECT (view_impl)) ;
}

static gboolean
mlview_editor_show_new_document_dialog (MlViewEditor *a_this, 
		                        MlViewNewDocumentDialogData **data)
{
        gint result = -1;
        gboolean status = FALSE;
        GladeXML *gxml   = NULL;
        gchar *gfile  = NULL;
        GtkWidget *docentry = NULL;
        GtkWidget *dialog = NULL;
        GtkWidget *schema_file_entry = NULL;
        GtkWidget *schema_uri_entry = NULL;
        GtkWidget *schema_type_combo = NULL;
        GtkWidget *encoding_combo = NULL;
        GtkWidget *xmlversion_entry = NULL;
        GList *available_encodings = NULL, *p = NULL;
        guint curencoding = -1;
        GtkListStore *text = NULL;
        GtkTreeIter iter = { 0 };
	GtkCellRenderer *renderer  = NULL;
        gchar *schema_uri = NULL;
        gint schema_type_index = -1;
        enum MlViewSchemaType schema_type = -1;
        
        g_return_val_if_fail (*data != NULL, FALSE);
        
        gfile = gnome_program_locate_file 
                (NULL, 
                 GNOME_FILE_DOMAIN_APP_DATADIR,
                 PACKAGE "/mlview-new-document.glade", 
                 TRUE, NULL);
        
        if (!gfile)
                return FALSE;
        
        gxml = glade_xml_new (gfile, NULL, "NewDocumentDialog");
        
        g_free (gfile);
        gfile = NULL;
        
        if (!gxml)
                goto cleanup;

        dialog = glade_xml_get_widget (gxml, "NewDocumentDialog");

        if (!(dialog && GTK_IS_DIALOG (dialog)))
                goto cleanup;

        docentry = glade_xml_get_widget (gxml, "RootNodeNameEntry");

        if (!(docentry && GTK_IS_ENTRY (docentry)))
                goto cleanup;

        schema_file_entry = glade_xml_get_widget (gxml, "SchemaFileEntry");

        if (!(schema_file_entry && GNOME_IS_FILE_ENTRY (schema_file_entry)))
                goto cleanup;
        
        schema_uri_entry = glade_xml_get_widget (gxml, "SchemaUriEntry");

        if (!(schema_uri_entry && GTK_IS_ENTRY (schema_uri_entry)))
                goto cleanup;
        
        schema_type_combo = glade_xml_get_widget (gxml, "SchemaTypeCombo");

        if (!(schema_type_combo && GTK_IS_COMBO_BOX (schema_type_combo)))
                goto cleanup;

        text = gtk_list_store_new (1, G_TYPE_STRING);
        
        if (!text)
                goto cleanup;
        
        gtk_list_store_append (text, &iter);
        gtk_list_store_set (text, &iter, 0, "Document Type Definition (DTD)", -1);
        gtk_list_store_append (text, &iter);
        gtk_list_store_set (text, &iter, 0, "Relax-NG Schema (RNG)", -1);
        gtk_list_store_append (text, &iter);
        gtk_list_store_set (text, &iter, 0, "XML Schema Definition (XSD)", -1);
        
        renderer = gtk_cell_renderer_text_new ();
        
        gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (schema_type_combo), 
                                    renderer, 
                                    TRUE);
        
        gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (schema_type_combo),
                                        renderer, 
                                        "text",
                                        0,
                                        NULL);
        
        gtk_combo_box_set_model (GTK_COMBO_BOX (schema_type_combo),
                                 GTK_TREE_MODEL (text));
        
        gtk_combo_box_set_active (GTK_COMBO_BOX (schema_type_combo),
                                  0);
     
        g_object_unref (G_OBJECT (text));
        text = NULL;

        xmlversion_entry = glade_xml_get_widget (gxml, "XMLVersionEntry");        

        if (!(xmlversion_entry && GTK_IS_ENTRY (xmlversion_entry)))
                goto cleanup;

        encoding_combo = glade_xml_get_widget (gxml, "EncodingCombo");
        
        if (!(encoding_combo && GTK_IS_COMBO_BOX (encoding_combo)))
                goto cleanup;

        /* fill encoding combo data */
        available_encodings = mlview_utils_get_available_encodings ();

        g_return_val_if_fail (available_encodings, FALSE);

        p = available_encodings;
        while (p != NULL) {
                gtk_combo_box_insert_text (
                        GTK_COMBO_BOX (encoding_combo),
                        ++curencoding,
                        (const gchar*)p->data);

                p = g_list_next (p);
        }

        gtk_combo_box_set_active (GTK_COMBO_BOX (encoding_combo),
                              0);
        
        
        result = gtk_dialog_run (GTK_DIALOG (dialog));
        switch (result) {
        case GTK_RESPONSE_ACCEPT:
                /* fill data */
                (*data)->root_node_name = (gchar*)g_strdup (
                        gtk_entry_get_text (GTK_ENTRY (docentry)));
                (*data)->xml_version = (gchar*)g_strdup (
                        gtk_entry_get_text (GTK_ENTRY (xmlversion_entry)));
                
#ifdef GTK_2_6_SERIE_OR_ABOVE	
                (*data)->encoding = (gchar*)g_strdup (
                        gtk_combo_box_get_active_text 
			           (GTK_COMBO_BOX (encoding_combo)));
#else
                (*data)->encoding = (gchar*)g_strdup (
                        mlview_utils_combo_box_get_active_text 
			(GTK_COMBO_BOX (encoding_combo)));
#endif	

                
                schema_type_index = gtk_combo_box_get_active (GTK_COMBO_BOX (schema_type_combo));
		schema_uri = (gchar*) gtk_entry_get_text (GTK_ENTRY (schema_uri_entry));
                        
                if (schema_uri && (strcmp (schema_uri,"") != 0)) {
			g_warning ("SCHEMA SPECIFIED");
                        if (schema_type_index == -1)
                                return FALSE;
	
                        switch (schema_type_index) {
                        case 0:
                                schema_type = SCHEMA_TYPE_DTD;
                                break;
                        case 1:
                                schema_type = SCHEMA_TYPE_RNG;
                                break;
                        case 2:
                                schema_type = SCHEMA_TYPE_XSD;
                                break;
                        }
                        
                        (*data)->schema = mlview_schema_load_from_file (
                                schema_uri,
                                schema_type,
                                PRIVATE (a_this)->app_context);
                        

                        /* add schema to combo history */
                        if ((*data)->schema)
                                gnome_entry_prepend_history (
                                        GNOME_ENTRY (gnome_file_entry_gnome_entry (
                                                             GNOME_FILE_ENTRY (schema_file_entry))),
                                        TRUE,
                                        schema_uri);
                }
                
                status = TRUE;
                break;
        default:
                status = FALSE;
                break;
    }
        
        
 cleanup:
        if (gxml)
                g_object_unref (gxml);
        
        if (dialog)
                gtk_widget_destroy (GTK_WIDGET (dialog));
        
        return status;
}

/**
 *Interactively create a new document.
 *Asks the user for info like 
 *root element, dtd (if validation is switched on) ...etc.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_create_new_xml_document (MlViewEditor * a_this)
{
        struct MlViewAppSettings *settings = NULL;
        MlViewXMLDocument *mlview_doc = NULL;
        xmlDocPtr xml_doc = NULL;
        xmlNodePtr xml_node = NULL;
        MlViewIView *view = NULL ;
        struct MlViewViewDesc *view_desc_ptr = NULL ;

        gboolean loop = TRUE;
        gchar *utf8_elname = NULL;
        MlViewNewDocumentDialogData *newdoc_dialog_data = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->app_context != NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context);
        g_return_if_fail (settings);

        view_desc_ptr = 
                mlview_editor_peek_editing_view_descriptor 
                (settings->general.default_editing_view_type);
        if (!view_desc_ptr) {
                mlview_utils_trace_debug
                        ("Unknown view type name: ") ;
                mlview_utils_trace_debug (settings->general.default_editing_view_type) ;
                mlview_utils_trace_debug ("This may be caused by a gconfd "
                                         "problem or a bad mlview default "
                                         "view type name gconf key\n"
                                         "First, try to killall gconfd and restart it\n"
                                         "If you still have the problem, send a mail to "
                                         "mlview-list@gnome.org to ask for help\n") ;
        }
        g_return_if_fail (view_desc_ptr) ;


        newdoc_dialog_data = g_try_malloc (sizeof(MlViewNewDocumentDialogData));
	newdoc_dialog_data = (MlViewNewDocumentDialogData*)memset(
			(void*)newdoc_dialog_data,
			0,
			sizeof(MlViewNewDocumentDialogData));

        g_return_if_fail (newdoc_dialog_data);

        while (loop) {
                gboolean res = FALSE;
                enum MlViewStatus status = MLVIEW_OK;
                guchar *name_end = NULL, *parsed_name = NULL;
                gulong len = 0 ;
                
                res = mlview_editor_show_new_document_dialog (
                        a_this,
                        &newdoc_dialog_data);

                utf8_elname = newdoc_dialog_data->root_node_name;
                
                /*the user hit cancel */
                if (!res)
                        break;
                /*empty root name: show the dialog again */
                if (!newdoc_dialog_data->root_node_name)
                        continue;

                if (mlview_utils_is_white_string (utf8_elname) == TRUE)
                        continue;

                /*check if is a valid element name */
                status = mlview_utils_parse_element_name (utf8_elname,
                                                          &name_end);
                if (status != MLVIEW_OK || !name_end) {
                        mlview_app_context_error (PRIVATE (a_this)->app_context,
                                _("The string entered is not a well formed element name!"));
                        continue;
                }
                len = (gulong)name_end - (gulong)utf8_elname + 1;
                parsed_name = g_strndup (utf8_elname, len) ;
                if (!parsed_name) {
                        mlview_app_context_error (PRIVATE (a_this)->app_context,
                                _("The string entered is not a well formed element name!"));
                        continue ;
                }
                xml_node = xmlNewNode (NULL, parsed_name);
                if (parsed_name) {
                        g_free (parsed_name) ;
                        parsed_name = NULL ;
                }
                xml_doc = xmlNewDoc ("1.0");
                xml_doc->name = g_strdup ("Untitled Document");
                xmlDocSetRootElement (xml_doc, xml_node);
                xml_doc->version  = xmlCharStrdup ((const char*)newdoc_dialog_data->xml_version);
                xml_doc->encoding = xmlCharStrdup ((const char*)newdoc_dialog_data->encoding);

                mlview_doc = mlview_xml_document_new (xml_doc,
                                                      PRIVATE (a_this)->app_context);
                g_return_if_fail (mlview_doc != NULL);

                /* add the schema (if specified) to the document's schema list */
                if (newdoc_dialog_data->schema) {
                        mlview_schema_list_add_schema (
                                mlview_xml_document_get_schema_list (mlview_doc),
                                newdoc_dialog_data->schema);
                }

                /*validation */
                if (settings->general.validation_is_on == TRUE) {
                        if (xml_node->type == XML_ELEMENT_NODE) {
                                mlview_parsing_utils_build_required_attributes_list
                                                (PRIVATE (a_this)->app_context,
                                                 xml_node);

                                mlview_parsing_utils_build_required_children_tree
                                                (PRIVATE
                                                 (a_this)->app_context,
                                                 &xml_node);
                        }
                }
                view = MLVIEW_IVIEW 
                        (view_desc_ptr->view_constructor
                         (mlview_doc, (gchar*)"",
                          PRIVATE (a_this)->app_context)) ;
                if (!view) {
                        mlview_utils_trace_debug ("view instanciation failed") ;
                        return ;
                }
		mlview_iview_set_desc_type_name (view, 
				                 view_desc_ptr->view_type_name) ;
                mlview_editor_add_view (a_this, view);
                /*everything went fine */
                break;
        }
}


/**
 *Edits the xml document given in argument.
 *Actually, this method creates a view on this document and
 *adds the view to the editor.
 *@param a_this the current mlview editor.
 *@param a_doc the xml document to edit. This is an
 *object of the libxml2
 *@param a_doc_name the name of the xml document to edit.
 */
void
mlview_editor_edit_xml_document (MlViewEditor * a_this,
                                 xmlDocPtr a_doc,
                                 gchar * a_doc_name)
{
        MlViewIView *doc_view = NULL ;
        MlViewXMLDocument *mlview_xml_doc = NULL;
        struct MlViewViewDesc *view_desc_ptr = NULL ;
        struct MlViewAppSettings *settings = NULL ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (a_doc != NULL);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context) ;
        g_return_if_fail (settings) ;
        view_desc_ptr = 
                mlview_editor_peek_editing_view_descriptor 
                (settings->general.default_editing_view_type);
        g_return_if_fail (view_desc_ptr) ;

        mlview_xml_doc =
                mlview_xml_document_new
                (a_doc, PRIVATE (a_this)->app_context) ;

        doc_view = MLVIEW_IVIEW 
                        (view_desc_ptr->view_constructor
                        (mlview_xml_doc, a_doc_name,
                         PRIVATE (a_this)->app_context)) ;
	mlview_iview_set_desc_type_name (doc_view, 
			                 view_desc_ptr->view_type_name) ;
        mlview_editor_add_view (a_this, doc_view) ;
}

/**
 *Execute an editing editing action.
 *@param a_this the current instance of #MlViewEditor
 *@param a_action the action to execute.
 *@return MLVIEW_OK upon successful completion an error code
 *otherwise.
 */
enum MlViewStatus
mlview_editor_execute_action (MlViewEditor *a_this,
                              MlViewAction *a_action)
{
        g_return_val_if_fail (a_this
                              && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (PRIVATE (a_this)->cur_view) {
                 mlview_iview_execute_action
                        (PRIVATE (a_this)->cur_view,
                         a_action) ;
		 return MLVIEW_OK ;
        }
        return MLVIEW_ERROR ;
}


/**
 *Getter of the file path of the document 
 *associated to the currently selected view.
 *@param a_this the current mlview editor.
 *@return the file path.
 */
gchar *
mlview_editor_get_current_xml_doc_file_path (MlViewEditor *a_this)
{
        MlViewXMLDocument *doc = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        mlview_iview_get_document
                (PRIVATE (a_this)->cur_view, &doc);
        g_return_val_if_fail (doc != NULL, NULL);

        return mlview_xml_document_get_file_path (doc);
}


/**
 *Interactively saves the underlying xml document of the 
 *currently selected document view.
 *If the document has an associated file name, 
 *save it in that file or else, asks the user where to
 *save it.
 *@param a_this the current mlview editor.
 */
void
mlview_editor_save_xml_document (MlViewEditor * a_this)
{
        MlViewXMLDocument *xml_doc = NULL;
        gchar *file_path = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        if (!PRIVATE (a_this)->cur_view)
                return;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view, &xml_doc) ;
        if (xml_doc == NULL)
                return;

        file_path =
                mlview_editor_get_current_xml_doc_file_path
                (a_this);

        if (file_path == NULL)
                mlview_editor_save_xml_document_as_interactive
                        (a_this);
        else
                mlview_editor_save_xml_document_as (a_this,
                                                    file_path);

}


/**
 *Classical "save as" functionnality.
 *Saves the underlying document of the currently selected view
 *into the file denoted by the file path a_file_path.
 *
 *@param a_this the current mlview editor.
 *@param a_file_path file path where to save the document.
 */
void
mlview_editor_save_xml_document_as (MlViewEditor * a_this,
                                    gchar * a_file_path)
{
        MlViewXMLDocument *mlview_xml_document = NULL;
        gboolean file_was_untitled = FALSE;
        guchar *prev_file_path = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        if (!PRIVATE (a_this)->cur_view)
                return;
        g_return_if_fail (PRIVATE (a_this)->opened_file_paths);
        g_return_if_fail (a_file_path != NULL);

	mlview_xml_document = mlview_editor_get_current_document (a_this) ;
	g_return_if_fail (mlview_xml_document) ;

        mlview_app_context_sbar_push_message
                (PRIVATE (a_this)->app_context,
                 _("Saving xml document as file %s..."),
                 a_file_path);

        file_was_untitled =
                (mlview_xml_document_get_file_descriptor (mlview_xml_document)
                 == NULL);

        prev_file_path =
                mlview_xml_document_get_file_path
                (mlview_xml_document);

        /*really save the document now */
        if (mlview_xml_document_save (mlview_xml_document,
                                      a_file_path, TRUE) > 0
            && (!prev_file_path
                || strcmp (prev_file_path, a_file_path))) {
                guchar *new_file_path = NULL;

                /*
                 *The save was OK and the new file path of this doc
                 * is different from the previous file path of this doc.
                 */

                /*
                 *remove the reference to the previous file path 
                 *of this document.
                 */
                if (prev_file_path) {
                        g_hash_table_remove
                                (PRIVATE (a_this)->
                                 opened_file_paths,
                                 prev_file_path);
                }

                /*
                 *Now, reference the new path of this doc.
                 *We must make sure that the string we put in the
                 *hash table belongs to mlview_xml_document.
                 *That way, one can destroy the hash table without
                 *any fear to leak the memory hold by the string.
                 */
                new_file_path = mlview_xml_document_get_file_path
                        (mlview_xml_document);

                if (new_file_path)
                        g_hash_table_insert (PRIVATE (a_this)->
                                             opened_file_paths,
                                             new_file_path,
                                             PRIVATE (a_this)->
                                             cur_view);
        }

        mlview_app_context_sbar_pop_message (PRIVATE (a_this)->
                                             app_context);
}


/**
 *Interactively saves the underlying document of the currently
 *selected view. Graphically asks the user where to save the
 *document and saves it.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_save_xml_document_as_interactive (MlViewEditor *a_this)
{
        gchar *file_name = NULL;
#ifndef MLVIEW_WITH_GTK_FILE_CHOOSER
        MlViewFileSelection *file_sel;
        enum MLVIEW_SELECTED_BUTTON button;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->app_context !=
                          NULL);

        file_sel =
                mlview_app_context_get_file_selector
                (PRIVATE (a_this)->app_context,
                 _("Choose a xml document"));

        g_return_if_fail (file_sel);

        button = mlview_file_selection_run (file_sel, TRUE);

        switch (button) {
        case OK_BUTTON:
                file_name =
                        g_strdup (gtk_file_selection_get_filename
                                  (GTK_FILE_SELECTION
                                   (file_sel)));

                if (file_name && strcmp (file_name, "")) {
                        mlview_editor_save_xml_document_as
                                (a_this, file_name);
                }

                if (file_name) {
                        g_free (file_name);
                        file_name = NULL;
                }
                break;
        case CANCEL_BUTTON:
        case WINDOW_CLOSED:
        default:
                break;
        }
#else
	gint response = 0;
	GtkWidget *file_dialog = NULL ;;

	file_dialog = GTK_WIDGET (mlview_app_context_get_file_chooser 
                                  (PRIVATE (a_this)->app_context,
                                   _("Save xml document"), MlViewFileChooserSaveMode)) ;
        g_return_if_fail (file_dialog != NULL);

        mlview_app_context_sbar_push_message
                (PRIVATE (a_this)->app_context,
		 _("Choose where to save the xml file"));

	response = gtk_dialog_run (GTK_DIALOG(file_dialog));
	gtk_window_set_modal (GTK_WINDOW (file_dialog), FALSE);
	gtk_widget_hide (GTK_WIDGET (file_dialog));

	if (response == GTK_RESPONSE_OK) {
                file_name = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_dialog));
		
		if (file_name && strcmp (file_name, "")) {
                        mlview_editor_save_xml_document_as
                                (a_this, file_name);
                }

                if (file_name) {
                        g_free (file_name);
                        file_name = NULL;
                }
	}

        mlview_app_context_sbar_pop_message
                (PRIVATE (a_this)->app_context);
#endif
}

/**
 *closes the current view without saving the underlying document.
 *@param a_this the current mlview editor.
 */
void
mlview_editor_close_xml_document_without_saving (MlViewEditor *a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        mlview_editor_remove_view
                (a_this,
                 PRIVATE (a_this)->cur_view);

        if (g_hash_table_size
            (PRIVATE (a_this)->mlview_xml_docs) == 0)
                PRIVATE (a_this)->cur_view = NULL;
}


/**
 *Saves the underlying document of the currently selected view
 *and closes the view.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_save_and_close_xml_document (MlViewEditor *a_this)
{
        MlViewFileDescriptor *file_desc = NULL ;
#ifndef MLVIEW_WITH_GTK_FILE_CHOOSER
        MlViewFileSelection *file_sel = NULL;
#else
	GtkWidget *file_dialog ;
	gint response = 0 ;
#endif
        MlViewXMLDocument *mlview_xml_document = NULL;
        gchar *file_name = NULL, *tmp_str = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        if (!PRIVATE (a_this)->cur_view)
                return;
        g_return_if_fail (PRIVATE (a_this)->app_context !=
                          NULL);
        
        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &mlview_xml_document) ;
        g_return_if_fail (mlview_xml_document != NULL);

        file_desc = mlview_xml_document_get_file_descriptor 
                (mlview_xml_document) ;
        if (!file_desc) {
#ifndef MLVIEW_WITH_GTK_FILE_CHOOSER
                gint button = 0 ;
                /*
                 *No file is associated to the document.
                 *=>Ask the user which file she wants to save
                 *the document in.
                 */
                file_sel = mlview_app_context_get_file_selector
                        (PRIVATE (a_this)->app_context,
                         _("Choose a xml document"));
                g_return_if_fail (file_sel);

                button = mlview_file_selection_run (file_sel, TRUE);

                switch (button) {
                case OK_BUTTON:
                        file_name =
                                g_strdup (gtk_file_selection_get_filename
                                          (GTK_FILE_SELECTION
                                           (file_sel))) ;
                        break;
                case CANCEL_BUTTON:
                case WINDOW_CLOSED:
                        break;
                }
#else
		file_dialog = GTK_WIDGET (mlview_app_context_get_file_chooser 
                                          (PRIVATE (a_this)->app_context, 
                                           _("Save xml document"), MlViewFileChooserSaveMode)) ;
		g_return_if_fail (file_dialog != NULL);

		mlview_app_context_sbar_push_message
		  (PRIVATE (a_this)->app_context,
		   _("Choose where to save the xml file"));

		response = gtk_dialog_run (GTK_DIALOG(file_dialog));

		gtk_window_set_modal (GTK_WINDOW (file_dialog), FALSE);
		gtk_widget_hide (GTK_WIDGET (file_dialog));

		if (response == GTK_RESPONSE_OK) {
			file_name = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_dialog));
		}
		mlview_app_context_sbar_pop_message
		  (PRIVATE (a_this)->app_context);
#endif
        } else {
                /*The document already has*/
                tmp_str = mlview_file_descriptor_get_file_path (file_desc);
                g_return_if_fail (tmp_str);
                file_name = g_strdup (tmp_str);
        }
        if (file_name && strcmp (file_name, "")) {
                mlview_xml_document_save
                        (mlview_xml_document, file_name,
                         TRUE);
                mlview_editor_close_xml_document_without_saving
                        (a_this);
        }
        if (file_name) {
                g_free (file_name);
                file_name = NULL;
        }
}


/**
 *Closes all the views (and their underlying documents).
 *For document that needs to be saved, this function 
 *popups a dialog asking the user if she wants
 *to save the document before closing, closing without saving
 *or cancel. If she cancels, then the function just returns FALSE
 *and the processing is stopped.
 *
 *@param a_this the current mlview editor.
 *@return TRUE if all the document views have been closed, FALSE otherwise.
 */
gboolean
mlview_editor_close_all_xml_documents (MlViewEditor *a_this,
		                       gboolean a_interactive)
{
        GList *views = NULL,
                *mobile_view_ptr = NULL;

        g_return_val_if_fail (a_this != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_EDITOR (a_this), FALSE);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, FALSE);
        g_return_val_if_fail (PRIVATE (a_this)->
                          mlview_xml_doc_views != NULL, FALSE);

        views = build_view_list_from_hashtable
                (PRIVATE (a_this)->mlview_xml_doc_views);

        if (views == NULL)
                return TRUE ;

        for (mobile_view_ptr = views;
             mobile_view_ptr;
             mobile_view_ptr = mobile_view_ptr->next) {
                PRIVATE (a_this)->cur_view =
                        (MlViewIView *) mobile_view_ptr->data;
                mlview_editor_close_xml_document (a_this, a_interactive);
        }

        if (g_list_length (mlview_editor_get_list_of_open_documents (a_this)) == 0)
                return TRUE ;
        else
                return FALSE ;
}


/**
 *Shows a confirmation dialog before closing a modified document.
 *
 *@param a_this the current mlview editor.
 */
static void
mlview_editor_confirm_close (MlViewEditor *a_this)
{
        enum MlViewStatus status = MLVIEW_OK ;
        GtkWidget *dialog;
        gint ret;
        guchar *a_name = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this));

        status = mlview_iview_get_name(PRIVATE (a_this)->cur_view, &a_name) ;

        g_return_if_fail (status == MLVIEW_OK && a_name) ;

        dialog = gtk_message_dialog_new (NULL,
                                         GTK_DIALOG_MODAL,
                                         GTK_MESSAGE_QUESTION,
                                         GTK_BUTTONS_NONE,
                                         _("The document \"%s\" has been modifed.\n"
					   "Should I save it before closing it?"), 
                                         a_name);

        gtk_dialog_add_buttons (GTK_DIALOG (dialog),
                                _("_Close without Saving"), GTK_RESPONSE_NO,
                                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                GTK_STOCK_SAVE, GTK_RESPONSE_YES, NULL);

        gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_YES);

        ret = gtk_dialog_run (GTK_DIALOG (dialog));
        switch (ret) {
        case GTK_RESPONSE_YES:
                mlview_editor_save_and_close_xml_document (a_this);
                break;
        case GTK_RESPONSE_NO:
                mlview_editor_close_xml_document_without_saving (a_this);
                break;
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
                break ;
        default:
                g_assert_not_reached ();
        }

        gtk_widget_destroy (dialog);
}


/**
 *Interactively closes the current document view.
 *Saves the underlying document if needed.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_close_xml_document (MlViewEditor *a_this,
		                  gboolean a_interactive)
{
        MlViewXMLDocument *doc = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this));

        if (!PRIVATE (a_this)->cur_view)
                return ;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &doc) ;
        if (!doc) {
                mlview_utils_trace_debug
                        ("The current view has no associated document. "
                         "This is truly weird, something bad is happening.") ;
                return ;
        }

        if (mlview_editor_get_number_of_views_opened_with_doc (a_this, doc) > 1 
	    || mlview_xml_document_needs_saving (doc) == FALSE) {
                mlview_editor_close_xml_document_without_saving (a_this);
	} else {
		if (a_interactive == TRUE) {
			mlview_editor_confirm_close (a_this);
		} else {
			mlview_editor_close_xml_document_without_saving (a_this) ;
		}
	}
}

/**
 *Interactively edits the editor settings.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_edit_settings_interactive (MlViewEditor * a_this)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

}


/**
 *Shows the tool used to manage the schemas associated with an
 *XML document.
 *
 *@param a_this the current mlview editor.
 *@return 0 if association worked, 1 if not xml document is opened, 
 */
gint
mlview_editor_manage_associated_schemas (MlViewEditor * a_this)
{
        MlViewXMLDocument *mlview_xml_doc = NULL;

        g_return_val_if_fail (a_this != NULL, -1);
        g_return_val_if_fail (MLVIEW_IS_EDITOR (a_this), -1);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, -1);

        if (!PRIVATE (a_this)->cur_view)
                return 1;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &mlview_xml_doc) ;
        if (mlview_xml_doc == NULL)
                return 1;
        
        mlview_editor_show_schemas_window_for_doc (a_this, 
                                                   mlview_xml_doc);

        return 0;
}

/**
 *validate the current selected document against the dtd it is associated to.
 *
 *@param a_this the current mlview editor.
 */
void
mlview_editor_validate (MlViewEditor * a_this)
{
        MlViewXMLDocument *doc = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (!PRIVATE (a_this)->cur_view)
                return;

        mlview_iview_get_document (PRIVATE (a_this)->cur_view,
                                   &doc) ;
        
        if (doc == NULL)
                return;
        
        mlview_editor_show_validation_window_for_doc (a_this, 
                                                      doc);
}


/**
 *Applies a XSLT stylesheet to the current document
 *displaying a dialog for xslt selection and opens
 *the transformation resulting document in the editor
 *@param a_this the current instance of #MlViewEditor.
 *
 */
void
mlview_editor_xslt_transform_document_interactive (MlViewEditor *a_this)
{
        MlViewXMLDocument *src_doc = NULL;
        MlViewXMLDocument *xsl_doc = NULL;
        MlViewXMLDocument *res_doc = NULL;
	MlViewIView *view = NULL ; 

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_EDITOR (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        
        src_doc = mlview_editor_get_current_document (a_this);
        xsl_doc = mlview_xslt_utils_select_xsl_doc (a_this);
        if (xsl_doc != NULL) {
                res_doc = mlview_xslt_utils_transform_document
                        (src_doc, xsl_doc);
                mlview_xml_document_unref(xsl_doc);
                if (res_doc != NULL)  {
                        view = mlview_editor_create_new_view_on_document 
                                (a_this, res_doc);
			if (view) {
				mlview_editor_add_view (a_this, view) ;
			}
		}
        }
}

/**
 * Gets the descriptor of the default editing view to be used.
 * the type default editing view is set in gconf.
 * @param a_this the current instance of #MlViewEditor
 * @param a_view_desc out parameter. The descriptor asked for
 * @return MLVIEW_OK upon succesful completion, an error code otherwise.
 */
enum MlViewStatus 
mlview_editor_get_default_view_descriptor (MlViewEditor *a_this,
                                           struct MlViewViewDesc **a_view_desc)
{
	struct MlViewAppSettings *settings = NULL ;
	struct MlViewViewDesc *view_desc = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
	                      && a_view_desc, MLVIEW_BAD_PARAM_ERROR) ;

	settings = mlview_app_context_get_settings 
		(PRIVATE (a_this)->app_context) ;
	g_return_val_if_fail (settings, MLVIEW_ERROR) ;

	view_desc = mlview_editor_peek_editing_view_descriptor 
		(settings->general.default_editing_view_type) ;
	if (!view_desc) {
		mlview_utils_trace_debug ("Unknown view type name:") ;
		mlview_utils_trace_debug 
		(settings->general.default_editing_view_type) ;

		mlview_utils_trace_debug ("This may be caused by a gconfd "
                                         "problem or a bad mlview default "
                                         "view type name gconf key\n"
                                         "First, try to killall "
					 "gconfd and restart it\n"
                                         "If you still have the problem, "
					 "send a email to "
                                         "mlview-list@gnome.org "
					 "asking for help\n") ;
		return MLVIEW_ERROR ;
	}
	*a_view_desc = view_desc ;

	return MLVIEW_OK ;
}

/**
 *Returns the descriptor of an editing view
 *denoted by it's view type name.
 *@param a_view_name the view name to test
 *@return a pointer to the view descriptor.
 *This pointer must *NOT* be freed by the caller.
 */
struct MlViewViewDesc *
mlview_editor_peek_editing_view_descriptor (gchar *a_view_type_name)
{
        struct MlViewViewDesc *view_desc_ptr = NULL ;

        for (view_desc_ptr = gv_view_types; 
             view_desc_ptr && view_desc_ptr->view_type_name;
             view_desc_ptr ++) {
                if (view_desc_ptr->view_type_name
                    && a_view_type_name
                    && !strcmp (view_desc_ptr->view_type_name,
                                a_view_type_name)) {
                        return view_desc_ptr ;
                }
        }
        return NULL ;
}

/**
 *Getter of the number of view descriptors available
 *in the system.
 *@return the number of view descriptor available in the system.
 */
guint
mlview_editor_get_number_of_view_desc (void)
{
        gint result = 0 ;
        struct MlViewViewDesc *cur_view_desc = NULL ;

        for (cur_view_desc = gv_view_types ; 
             cur_view_desc && cur_view_desc->view_type_name;
             cur_view_desc ++)
        {
                result ++ ;
        }
        return result ;
}

/**
 *Get a view descriptor from the table of view descriptors.
 *@param a_offset the offset of the view descriptor to get.
 *The first view descriptor in the table has an offset of zero.
 *@param the offset of the view descriptor to get
 *@return the found view descriptor, or NULL if no view descriptor is
 *found.
 */
struct MlViewViewDesc *
mlview_editor_get_view_descriptor_at (guint a_offset)
{
        guint nr_views = 0 ;

        nr_views = mlview_editor_get_number_of_view_desc () ;
        if (a_offset >= nr_views)
                return NULL ;
        return &gv_view_types[a_offset] ;
}

static void
schemas_window_destroy_cb (GtkWidget *a_widget,
                           struct DocumentWindowData *a_win)
{
        g_return_if_fail (a_win);
        g_return_if_fail (a_win->editor && MLVIEW_IS_EDITOR (a_win->editor));
        g_return_if_fail (a_win->document);
        g_return_if_fail (PRIVATE (a_win->editor));
        g_return_if_fail (PRIVATE (a_win->editor)->mlview_xml_doc_schemas_windows);
        
        g_hash_table_remove (PRIVATE (a_win->editor)->mlview_xml_doc_schemas_windows,
                             a_win->document);

        g_free (a_win);
        a_win = NULL;
}

GtkWidget *
mlview_editor_show_schemas_window_for_doc (MlViewEditor *a_this,
                                           MlViewXMLDocument *a_doc)
{
        GtkWidget *win = NULL;
        struct DocumentWindowData *data = NULL;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this),
                              NULL);
        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this),
                              NULL);
        g_return_val_if_fail (PRIVATE (a_this)->mlview_xml_doc_schemas_windows,
                              NULL);

        data = g_hash_table_lookup (PRIVATE (a_this)->mlview_xml_doc_schemas_windows,
                                    a_doc);
        
        if (data) {
                g_return_val_if_fail (data->window, NULL);
                g_return_val_if_fail (GTK_IS_WIDGET (data->window), NULL);

                gtk_widget_hide (data->window);
                gtk_widget_show (data->window);
                
                return data->window;
        }

        win = mlview_schemas_window_new_with_document
                (a_doc, PRIVATE (a_this)->app_context);

        data = g_try_malloc (sizeof (struct DocumentWindowData));

        if (!data) {
                gtk_widget_destroy (win);

                return NULL;
        }

        memset (data, 0, sizeof (struct DocumentWindowData));

        data->document = a_doc;
        data->editor = a_this;
        data->window = win;

        g_signal_connect (G_OBJECT (win), "destroy",
                          G_CALLBACK (schemas_window_destroy_cb),
                          data);

        g_hash_table_insert (PRIVATE (a_this)->mlview_xml_doc_schemas_windows,
                             a_doc, data);

        gtk_widget_show_all (win);

        return win;
}

static void
validation_window_destroy_cb (GtkWidget *a_widget, 
                              struct DocumentWindowData *a_win)
{    
        g_return_if_fail (a_win);
        g_return_if_fail (a_win->editor && MLVIEW_IS_EDITOR (a_win->editor));
        g_return_if_fail (a_win->document);
        g_return_if_fail (PRIVATE (a_win->editor));
        g_return_if_fail (PRIVATE (a_win->editor)->mlview_xml_doc_validation_windows);
        
        g_hash_table_remove (PRIVATE (a_win->editor)->mlview_xml_doc_validation_windows,
                             a_win->document);
        
        g_free (a_win);
        a_win = NULL;
}

GtkWidget *
mlview_editor_show_validation_window_for_doc (MlViewEditor *a_this,
                                              MlViewXMLDocument *a_doc)
{        
        GtkWidget *win = NULL;
        struct DocumentWindowData *data = NULL;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this), NULL);
        g_return_val_if_fail (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc), NULL);
        g_return_val_if_fail (PRIVATE (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this)->mlview_xml_doc_validation_windows, NULL);

        data = g_hash_table_lookup (PRIVATE (a_this)->mlview_xml_doc_validation_windows,
                                    a_doc);
        
        if (data) {
                g_return_val_if_fail (data->window, NULL);
                g_return_val_if_fail (GTK_IS_WIDGET (data->window), NULL);
                
                gtk_widget_hide (data->window);
                gtk_widget_show (data->window);
                
                return data->window;
        }

        win = mlview_validator_window_new (a_doc);
        
        data = g_try_malloc (sizeof (struct DocumentWindowData));

        if (!data) {
                gtk_widget_destroy (win);

                return NULL;
        }

        memset (data, 0, sizeof (struct DocumentWindowData));

        data->document = a_doc;
        data->editor = a_this;
        data->window = win;
        
        g_signal_connect (G_OBJECT (win), "destroy", 
                          G_CALLBACK (validation_window_destroy_cb), data);

        g_hash_table_insert (PRIVATE (a_this)->mlview_xml_doc_validation_windows,
                             a_doc, data);

        gtk_widget_show_all (win);

        return win;
        
}

/**
 *Tests whether a document referenced by its absolute path or uri
 *is already loaded in the editor or not.
 */
enum MlViewStatus 
mlview_editor_is_document_opened_in_editor (MlViewEditor *a_this,
                                            gchar *a_doc_absolute_path,
                                            gboolean *a_result)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->opened_file_paths,
                              MLVIEW_BAD_PARAM_ERROR) ;
        g_return_val_if_fail (a_doc_absolute_path && a_result,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (g_hash_table_lookup (PRIVATE (a_this)->opened_file_paths,
                                 a_doc_absolute_path)) {
                *a_result = TRUE ;
        } else {
                *a_result = FALSE ;
        }
        return MLVIEW_OK ;
}


enum MlViewStatus
mlview_editor_make_current_view_populate_application_edit_menu (MlViewEditor *a_this)
{
        MlViewIView *cur_view = NULL ;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        cur_view = mlview_editor_get_current_document_view (a_this) ;
        if (!cur_view)
                return MLVIEW_OK ;
        mlview_iview_request_application_menu_populating (cur_view) ;

        return MLVIEW_OK ;
}

/**
 * Reloads the document being edited, that is, loads the file from the disk
 * or from where it has been loaded from. All the changes that have not been
 * saved before calling this method will then be lost.
 * @param a_this the current instance of #MlViewEditor
 * @return MLVIEW_OK upon successful completion, an error code otherwise
 */
enum MlViewStatus
mlview_editor_reload_document (MlViewEditor *a_this,
		               gboolean a_interactive)
{
	MlViewXMLDocument *doc = NULL ;
	gchar *file_path = NULL ;

	g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

        doc = mlview_editor_get_current_document (a_this) ; 
	if (!doc) {
		/*Hugh ? there is no current doc ? So how did the UI
		 *let the user hit a button that lets her end up here then ?
		 *man, go fix the UI, so that when there is no document
		 *opened, grey out the buttons that let the user endup here.
		 */
	         mlview_utils_trace_debug ("You asked for the current doc, but"
			    "but there is no current doc.\n"
			    "Maybe the UI shouldn't have let the user "
			    "Hit a button that make her ask for the current "
			    "doc when there is no current doc loaded ?\n"
			    "Homeboy, go fix the damn UI.") ;	
	}
	
	file_path = mlview_xml_document_get_file_path (doc) ;
	if (!file_path) {
		/*seems like this doc is new and has never been saved yet.
		 *We can then not reload it.
		 */
		return MLVIEW_CANT_RELOAD_ERROR ;
	}
        mlview_editor_load_xml_file (a_this, file_path, a_interactive) ; 

	if (file_path) {
		g_free (file_path) ;
		file_path = NULL ;
	}
	return MLVIEW_OK ;
}

/**
 * Undo the last editing action performed on the current
 * editing view
 * @param a_this the current instance of #MLViewEditor
 * @return MLVIEW_OK upon successful completion, an error code otherwise
 */
enum MlViewStatus
mlview_editor_undo (MlViewEditor *a_this)
{
        MlViewIView *cur_view = NULL ;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        cur_view = mlview_editor_get_current_document_view (a_this) ;

        if (!cur_view) {
                mlview_utils_trace_debug ("No current selected view found") ;
                return MLVIEW_ERROR;
        }
	return mlview_iview_undo (cur_view) ;

}

/**
 * Redo the last undo'ed editing action performed in the current
 * editing view
 * @param a_this the current instance of #MlViewEditor
 * @return MLVIEW_OK upon successful completion, an error code otherwise
 */
enum MlViewStatus
mlview_editor_redo (MlViewEditor *a_this)
{
        MlViewIView *cur_view = NULL ;  

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        cur_view = mlview_editor_get_current_document_view (a_this) ;

        if (!cur_view) {
                mlview_utils_trace_debug ("No current selected view found") ;
                return MLVIEW_ERROR;
        }

        return mlview_iview_redo (cur_view);
}

/**
 * Checks wether the last editing action performed on the current editing
 * view can be undo'ed
 * @param a_this the current instance of #MlViewEditor
 * @return TRUE if the action can be undo'ed, FALSE otherwise
 */
gboolean
mlview_editor_can_undo (MlViewEditor *a_this)
{
        MlViewIView *cur_view = NULL ;
	gboolean can_undo =  FALSE ;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              FALSE) ;

        cur_view = mlview_editor_get_current_document_view (a_this) ;

        if (!cur_view) {
                return FALSE;
        }
	mlview_iview_can_undo (cur_view, &can_undo) ;
	return can_undo ;
}


/**
 * Checks if the last editing action performed on the current editing
 * view can be undo'ed or not.
 * @param a_this the current instance of #MlViewEditor
 * @return TRUE if the action can be undo'ed, FALSE otherwise
 */
gboolean
mlview_editor_can_redo (MlViewEditor *a_this)
{
        MlViewIView *cur_view = NULL ;
	gboolean can_redo = FALSE ;

        g_return_val_if_fail (a_this && MLVIEW_IS_EDITOR (a_this)
                              && PRIVATE (a_this),
                              FALSE) ;

        cur_view = mlview_editor_get_current_document_view (a_this) ;

        if (!cur_view) {
                return FALSE;
        }
	mlview_iview_can_redo (cur_view, &can_redo) ;
	return can_redo ;
}
