/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h> 
#include "mlview-iview.h"
#include "mlview-marshal.h"

/**
 *@file
 *The definition of the #MlViewIView interface.
 */
enum MlViewXMLDocViewSignals {
        IS_SWAPPED_OUT,
	IS_SWAPPED_IN,
        NAME_CHANGED,
        APPLICATION_MENU_POPULATING_REQUESTED,
	NB_SIGNALS
} ;

struct _MlViewIViewPriv {
	gboolean dispose_has_run ;
} ;

#define PRIVATE(obj) (obj)->priv

static guint gv_signals[NB_SIGNALS] = {0} ;

static void mlview_iview_base_init (gpointer a_klass) ;

static void
mlview_iview_base_init (gpointer a_iface)
{

	static gboolean initialized = FALSE ;

	if (initialized)
		return ;

	gv_signals[NAME_CHANGED] =
		g_signal_new ("name-changed",
			      MLVIEW_TYPE_IVIEW,
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MlViewIView,
					       name_changed),
			      NULL, NULL,
			      mlview_marshal_VOID__VOID,
			      G_TYPE_NONE, 0, NULL) ;

        gv_signals[IS_SWAPPED_OUT] =
		g_signal_new ("is-swapped-out",
			      MLVIEW_TYPE_IVIEW,
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MlViewIView,
					       is_swapped_out),
			      NULL, NULL,
			      mlview_marshal_VOID__VOID,
			      G_TYPE_NONE, 0, NULL) ;

        gv_signals[IS_SWAPPED_IN] =
		g_signal_new ("is-swapped-in",
			      MLVIEW_TYPE_IVIEW,
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MlViewIView,
					       is_swapped_in),
			      NULL, NULL,
			      mlview_marshal_VOID__VOID,
			      G_TYPE_NONE, 0, NULL) ;

        gv_signals[APPLICATION_MENU_POPULATING_REQUESTED] =
		g_signal_new ("application-menu-populating-requested",
			      MLVIEW_TYPE_IVIEW,
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MlViewIView,
					       application_menu_populating_requested),
			      NULL, NULL,
			      mlview_marshal_VOID__VOID,
			      G_TYPE_NONE, 0, NULL) ;

		initialized = TRUE ;
}


/**
 *Getter/builder of the type identifier of the
 *the #MlViewIView class.
 *@return the new type identifier.
 */
GType
mlview_iview_get_type (void)
{
	static GType type = 0 ;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MlViewIView), /*class size*/
			mlview_iview_base_init, /*base_init*/
			NULL, /*base_finalize*/
			NULL, /*class init*/
			NULL, /*class finalize*/
			NULL, /*class data*/
			0, /*instance size*/
			0, /*n_prealloc*/
			NULL,/*instance_init*/
		} ;
		type = g_type_register_static (G_TYPE_INTERFACE,
					       "MlViewIView",
					       &info, 0) ;
		g_type_interface_add_prerequisite (type, G_TYPE_OBJECT) ;
	}
	return type ;
}

/**
 *Connects the view to the document object model.
 *@param a_this the current instance of #MlViewIView.
 *@param a_doc the current document object model.
 */
enum MlViewStatus
mlview_iview_connect_to_doc (MlViewIView *a_this,
                             MlViewXMLDocument *a_doc)
{
	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	g_return_val_if_fail (a_this, MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->connect_to_doc) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->connect_to_doc 
		(a_this, a_doc) ;
	} else {
		mlview_utils_trace_debug ("virtual method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;

}

/**
 *Disconnects from the document object model.
 *@param a_this the current instance of #MlViewIView
 *@param a_doc the document object model to disconnect from
 */
enum MlViewStatus
mlview_iview_disconnect_from_doc (MlViewIView *a_this,
                                  MlViewXMLDocument *a_doc)
{
	g_return_val_if_fail (a_this 
		              && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->disconnect_from_doc) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->disconnect_from_doc 
		(a_this, a_doc) ;
	} else {
		mlview_utils_trace_debug ("virtual method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;

}


/**
 *Let's this view execute an editing action.
 *@param a_this the current instance of #MlViewIView.
 *@param a_action the action to execute.
 */
enum MlViewStatus
mlview_iview_execute_action (MlViewIView *a_this,
                             MlViewAction *a_action)
{
	g_return_val_if_fail (a_this 
		              && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->execute_action) {
		return a_this->execute_action (a_this, a_action) ;
	} else {
		mlview_utils_trace_debug ("virtual method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;

}

/**
 *Sets the name of the view.
 *@param a_this the current instance of #MlViewIView
 *@param a_name the new name of the view.
 *@return MLIVEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_iview_set_name (MlViewIView *a_this,
                       guchar *a_name)
{
	enum MlViewStatus status = MLVIEW_OK ;
	g_return_val_if_fail (a_this
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_name
			      && a_name, 
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->set_name) {
		status = MLVIEW_IVIEW_GET_IFACE (a_this)->set_name (a_this, a_name) ;
		if (status == MLVIEW_OK) {
			g_signal_emit (G_OBJECT (a_this),
				       gv_signals[NAME_CHANGED],0) ;
		}
	} else {
		mlview_utils_trace_debug ("virutal method not implemented") ;
		status = MLVIEW_IFACE_NOT_DEFINED_ERROR ;
	}
	return status ; 
}

/**
 *Gets the name of the current editing view.
 *@param a_this the current instance of #MlViewIView.
 *@param a_name out parameter. Points to where to store the name
 *of the view. Caller must not free *a_name.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_get_name (MlViewIView *a_this,
                       guchar **a_name)
{
	g_return_val_if_fail (a_this
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_name
			      && a_name, 
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->get_name) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->get_name (a_this, a_name) ;
	} else {
		mlview_utils_trace_debug ("virtual method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

/**
 *Interactively sets the name of the current view.
 *@param a_this the current instance of #MlViewIView.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus 
mlview_iview_set_name_interactive (MlViewIView *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (MLVIEW_IVIEW_GET_IFACE (a_this)->set_name_interactive) {
                return MLVIEW_IVIEW_GET_IFACE (a_this)->set_name_interactive 
					(a_this) ;
        } else {
		mlview_utils_trace_debug ("virtual method not implemented") ;
	}
        return MLVIEW_IFACE_NOT_DEFINED_ERROR;
}


/**
 *Gets the document being edited by the current
 *view.
 *@param a_this the current instance of #MlViewIView .
 *@param a_doc out parameter.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_get_document (MlViewIView *a_this,
                           MlViewXMLDocument **a_doc)
{	
	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_doc, MLVIEW_BAD_PARAM_ERROR) ;


	if (MLVIEW_IVIEW_GET_IFACE (a_this)->get_document) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->get_document 
			(a_this, a_doc) ;
	} else {
		mlview_utils_trace_debug ("virtual method not implemented") ;
	}
	return  MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}


/**
 *Sets associates a new document to the current view.
 *@param a_this the current instance of #MlViewIView.
 *@param a_doc the new document.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_set_document (MlViewIView *a_this,
                           MlViewXMLDocument *a_doc)
{
	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_doc, MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->set_document) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->set_document 
			(a_this, a_doc) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

enum MlViewStatus 
mlview_iview_set_desc_type_name (MlViewIView *a_this,
                                 const gchar *a_desc_type_name)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			      && a_desc_type_name,
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->set_desc_type_name) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->set_desc_type_name 
			(a_this, a_desc_type_name) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

enum MlViewStatus 
mlview_iview_get_desc_type_name (MlViewIView *a_this,
                                 gchar **a_desc_type_name)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
			      && a_desc_type_name,
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->get_desc_type_name) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->get_desc_type_name 
			(a_this, a_desc_type_name) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

/**
 * Tests if the view has to be rebuild when the document gets reloaded or not.
 * This function actually emits the "get-must-rebuild-upon-document-reload" signal
 * on the view.
 * View implementations then have to connect to the 
 * "get-must-rebuild-upon-document-reload" signal to actually set the result of
 * the test.
 *
 * @param a_this the current instance of #MlViewIView .
 * @param a_result the result of the test.
 * @return MLVIEW_OK upon successful completion, and error code otherwise.
 */
enum MlViewStatus 
mlview_iview_get_must_rebuild_upon_document_reload (MlViewIView *a_this, 
		                                    gboolean *a_result)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;


	if (MLVIEW_IVIEW_GET_IFACE (a_this)->get_must_rebuild_upon_document_reload) {
		return MLVIEW_IVIEW_GET_IFACE 
		(a_this)->get_must_rebuild_upon_document_reload (a_this,
	                                                         a_result) ;
	} else {
		mlview_utils_trace_debug ("abstract method not defined") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

/**
 *Edit the contextual menu according to the
 *services proposed by this view.
 *@param a_this the current instance of #MlViewIView.
 *@param a_menu out parameter.
 */
enum MlViewStatus
mlview_iview_update_contextual_menu (MlViewIView *a_this,
                                     GtkWidget **a_menu)
{
	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->update_contextual_menu) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->update_contextual_menu 
			(a_this, a_menu) ;
	} else {
		mlview_utils_trace_debug ("abstract method not defined") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}


/**
 *Gets the implementation of the current view interface.
 *@param a_this the current instance of #MlViewIView.
 *@param a_impl out parameter. 
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus 
mlview_iview_get_impl (MlViewIView *a_this, GtkWidget **a_impl)
{
	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->get_impl) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->get_impl 
			(a_this, a_impl) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

enum MlViewStatus 
mlview_iview_undo (MlViewIView *a_this)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
	                      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->undo) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->undo (a_this) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

enum MlViewStatus 
mlview_iview_redo (MlViewIView *a_this)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
	                      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->redo) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->redo (a_this) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

enum MlViewStatus 
mlview_iview_can_undo (MlViewIView *a_this,
                       gboolean *a_can_undo)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
	                      MLVIEW_BAD_PARAM_ERROR) ; 

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->can_redo) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->can_undo (a_this, a_can_undo) ;
	} else {
		mlview_utils_trace_debug ("abstract method not implemented") ;
	}
	return MLVIEW_IFACE_NOT_DEFINED_ERROR ;
}

enum MlViewStatus 
mlview_iview_can_redo (MlViewIView *a_this,
                       gboolean *a_can_redo)
{
	g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
	                      MLVIEW_BAD_PARAM_ERROR) ;

	if (MLVIEW_IVIEW_GET_IFACE (a_this)->can_redo) {
		return MLVIEW_IVIEW_GET_IFACE (a_this)->can_redo 
							(a_this, a_can_redo) ;
	} else {
		mlview_utils_trace_debug ("abstrace method not implemented") ;
	}
	return  MLVIEW_IFACE_NOT_DEFINED_ERROR  ;
}

enum MlViewStatus
mlview_iview_notify_swapped_out (MlViewIView *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[IS_SWAPPED_OUT], 0) ;
        return MLVIEW_OK ;
}


enum MlViewStatus
mlview_iview_notify_swapped_in (MlViewIView *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[IS_SWAPPED_IN], 0) ;
        return MLVIEW_OK ;
}

enum MlViewStatus 
mlview_iview_request_application_menu_populating (MlViewIView *a_this)
{
        g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_signal_emit (G_OBJECT (a_this),
                       gv_signals[APPLICATION_MENU_POPULATING_REQUESTED], 0) ;
        return MLVIEW_OK ;
}


