/*
 * ===================================================================
 *  TS 26.104
 *  R99   V3.3.0 2001-09
 *  REL-4 V4.2.0 2001-09
 *  3GPP AMR Floating-point Speech Codec
 * ===================================================================
 *
 */

/*
 * rom_enc.h
 *
 *
 * Project:
 *    AMR Floating-Point Codec
 *
 * Contains:
 *    This file contains all the tables needed by AMR encoder functions.
 *
 */
#ifndef _ROM_ENC_H_
#define _ROM_ENC_H_


#include "typedef.h"
#include "sp_enc.h"

#define M               10    /* Order of LP filter */
#define MP1             (M+1) /* Order of LP filter + 1 */
#define L_WINDOW        240   /* Window size in LP analysis */
#define L_NEXT          40    /* Overhead in LP analysis */
#define LTPG_MEM_SIZE   5     /* number of stored past LTP coding gains + 1 */
#define N_FRAME         7     /* old pitch gains in average calculation */
#define DTX_HIST_SIZE   8     /* DTX history size */
#define L_TOTAL         320   /* Total size of speech buffer. */
#define L_FRAME         160   /* Frame size */
#define L_FRAME_BY2     80    /* Frame size divided by 2 */
#define L_SUBFR         40    /* Subframe size */
#define L_CODE          40    /* codevector length */
#define PIT_MAX         143   /* Maximum pitch lag */
#define PIT_MIN         20    /* Minimum pitch lag */
#define PIT_MIN_MR122   18    /* Minimum pitch lag (MR122 mode) */
#define L_INTERPOL      (10+1)/* Length of filter for interpolation */
#define NPRED           4     /* number of prediction taps */
#define SHARPMIN        0     /* Minimum value of pitch sharpening */
#define MAX_PRM_SIZE    57    /* max. num. of params */
#define L_INTER_SRCH    4     /* Length of filter for CL LTP search interpolation */
#define GP_CLIP         0.95F /* Pitch gain clipping */
#define UP_SAMP_MAX     6
#define NB_TRACK        5     /* number of tracks */
#define NB_TRACK_MR102  4     /* number of tracks mode mr102 */
#define STEP            5     /* codebook step size */
#define STEP_MR102      4     /* codebook step size mode mr102 */
#define NC              M/2   /* Order of LP filter divided by 2 */

#define SCALE_LSP_FREQ  (Float32)(4000.0/3.141592654)
#define SCALE_FREQ_LSP  (Float32)(3.141592654/4000.0)
#define SLOPE1_WGHT_LSF (Float32)((3.347-1.8)/(450.0-0.0))
#define SLOPE2_WGHT_LSF (Float32)((1.8-1.0)/(1500.0-450.0))

#define FRAME_LEN 160    /* Length (samples) of the input frame          */
#define COMPLEN 9        /* Number of sub-bands used by VAD              */
#define INV_COMPLEN 3641 /* 1.0/COMPLEN*2^15                             */
#define LOOKAHEAD 40     /* length of the lookahead used by speech coder */

#define UNITY 512        /* Scaling used with SNR calculation            */
#define UNIRSHFT 6       /* = log2(MAX_16/UNITY)                         */

#define TONE_THR 0.65F   /* Threshold for tone detection   */

/* Constants for background spectrum update */
#define ALPHA_UP1   (Float32)(1.0 - 0.95)  /* Normal update, upwards:   */
#define ALPHA_DOWN1 (Float32)(1.0 - 0.936) /* Normal update, downwards  */
#define ALPHA_UP2   (Float32)(1.0 - 0.985) /* Forced update, upwards    */
#define ALPHA_DOWN2 (Float32)(1.0 - 0.943) /* Forced update, downwards  */
#define ALPHA3      (Float32)(1.0 - 0.95)  /* Update downwards          */
#define ALPHA4      (Float32)(1.0 - 0.9)   /* For stationary estimation */
#define ALPHA5      (Float32)(1.0 - 0.5)   /* For stationary estimation */

/* Constants for VAD threshold */
#define VAD_THR_HIGH 1260 /* Highest threshold                 */
#define VAD_THR_LOW  720  /* Lowest threshold                  */
#define VAD_P1 0          /* Noise level for highest threshold */
#define VAD_P2 6300       /* Noise level for lowest threshold  */
#define VAD_SLOPE (float)(VAD_THR_LOW-VAD_THR_HIGH)/(float)(VAD_P2-VAD_P1)

/* Parameters for background spectrum recovery function */
#define STAT_COUNT 20         /* threshold of stationary detection counter         */
#define STAT_COUNT_BY_2 10    /* threshold of stationary detection counter         */
#define CAD_MIN_STAT_COUNT 5  /* threshold of stationary detection counter         */

#define STAT_THR_LEVEL 184    /* Threshold level for stationarity detection        */
#define STAT_THR 1000         /* Threshold for stationarity detection              */

/* Limits for background noise estimate */
#define NOISE_MIN 40          /* minimum */
#define NOISE_MAX 16000       /* maximum */
#define NOISE_INIT 150        /* initial */

/* Constants for VAD hangover addition */
#define HANG_NOISE_THR 100
#define BURST_LEN_HIGH_NOISE 4
#define HANG_LEN_HIGH_NOISE 7
#define BURST_LEN_LOW_NOISE 5
#define HANG_LEN_LOW_NOISE 4

/* Thresholds for signal power */
#define VAD_POW_LOW (Word32)15000/2    /* If input power is lower,                    */
                                       /*     VAD is set to 0                         */
#define POW_PITCH_THR (Word32)343040/2 /* If input power is lower, pitch              */
                                       /*     detection is ignored                    */

#define POW_COMPLEX_THR (Word32)15000/2/* If input power is lower, complex            */
                                       /* flags  value for previous frame  is un-set  */
/*
 * VAD Constants
 */

/* Constants for the filter bank */
#define LEVEL_SHIFT 0                     /* scaling                                  */
#define COEFF3   (Float32)13363/32768     /* coefficient for the 3rd order filter     */
#define COEFF5_1 (Float32)21955/32768     /* 1st coefficient the for 5th order filter */
#define COEFF5_2 (Float32)6390/32768      /* 2nd coefficient the for 5th order filter */

/* Constants for pitch detection */
#define LTHRESH 4
#define NTHRESH 4

/* Constants for complex signal VAD  */
#define CVAD_THRESH_ADAPT_HIGH  0.6F            /* threshold for adapt stopping high */
#define CVAD_THRESH_ADAPT_LOW  0.5F             /* threshold for adapt stopping low */
#define CVAD_THRESH_IN_NOISE  0.65F * 32768.0F  /* threshold going into speech on
                                                   a short term basis */
#define CVAD_THRESH_HANG  0.70F           /* threshold */
#define CVAD_HANG_LIMIT  (Word16)(100)    /* 2 second estimation time */
#define CVAD_HANG_LENGTH  (Word16)(250)   /* 5 second hangover */

#define CVAD_LOWPOW_RESET 0.40F  /* init in low power segment */
#define CVAD_MIN_CORR 0.40F      /* lowest adaptation value */

#define CVAD_BURST 20                        /* speech burst length for speech reset */
#define CVAD_ADAPT_SLOW 1.0F - 0.98F         /* threshold for slow adaption */
#define CVAD_ADAPT_FAST 1.0F - 0.92F         /* threshold for fast adaption */
#define CVAD_ADAPT_REALLY_FAST 1.0F - 0.80F  /* threshold for really fast adaption */

/* track table for algebraic code book search (MR475, MR515) */
static Word8 trackTable[4 * 5] =
   {
      /* subframe 1; track to code; -1 do not code this position */ 0,
      1,
      0,
      1,
      - 1,
      /* subframe 2 */ 0, 
      - 1,
      1,
      0,
      1,
      /* subframe 3 */ 0,
      1,
      0,
      - 1,
      1,
      /* subframe 4 */ 0,
      1,
      - 1,
      0,
      1
   };
static const Float32 gamma1[M] =
{
   0.9400024414063F,
   0.8836059570313F,
   0.8305969238281F,
   0.78076171875F,
   0.7339172363281F,
   0.6898803710938F,
   0.6484985351563F,
   0.6095886230469F,
   0.5730285644531F,
   0.5386352539063F
};

/*
 *  gamma1 for the 12k2 coder
 */
static const Float32 gamma1_12k2[M] =
{
   0.8999938964844F,
   0.8099975585938F,
   0.72900390625F,
   0.6560974121094F,
   0.5904846191406F,
   0.5314331054688F,
   0.478271484375F,
   0.4304504394531F,
   0.3873901367188F,
   0.3486633300781F
};
static const Float32 gamma2[M] =
{
   0.6000061035156F,
   0.3600158691406F,
   0.2160034179688F,
   0.1296081542969F,
   0.0777587890625F,
   0.04666137695313F,
   0.02798461914063F,
   0.01678466796875F,
   0.01007080078125F,
   0.00604248046875F
};

/* 1/6 resolution interpolation filter  (-3 dB at 3600 Hz)
 * Note: the 1/3 resolution filter is simply a subsampled
 *       version of the 1/6 resolution filter, i.e. it uses
 *       every second coefficient:
 *
 *       inter_6(1/3)[k] = inter_6(1/3)[2*k], 0 <= k <= 3*L_INTER10
 */
static Float32 b60[UP_SAMP_MAX*(L_INTERPOL-1)+1] =
{
   0.898529F,
   0.865051F,
   0.769257F,
   0.624054F,
   0.448639F,
   0.265289F,
   0.0959167F,
   - 0.0412598F,
   - 0.134338F,
   - 0.178986F,
   - 0.178528F,
   - 0.142609F,
   - 0.0849304F,
   - 0.0205078F,
   0.0369568F,
   0.0773926F,
   0.0955200F,
   0.0912781F,
   0.0689392F,
   0.0357056F,
   0.000000F,
   - 0.0305481F,
   - 0.0504150F,
   - 0.0570068F,
   - 0.0508423F,
   - 0.0350037F,
   - 0.0141602F,
   0.00665283F,
   0.0230713F,
   0.0323486F,
   0.0335388F,
   0.0275879F,
   0.0167847F,
   0.00411987F,
   - 0.00747681F,
   - 0.0156860F,
   - 0.0193481F,
   - 0.0183716F,
   - 0.0137634F,
   - 0.00704956F,
   0.000000F,
   0.00582886F,
   0.00939941F,
   0.0103760F,
   0.00903320F,
   0.00604248F,
   0.00238037F,
   - 0.00109863F,
   - 0.00366211F,
   - 0.00497437F,
   - 0.00503540F,
   - 0.00402832F,
   - 0.00241089F,
   - 0.000579834F,
   0.00103760F,
   0.00222778F,
   0.00277710F,
   0.00271606F,
   0.00213623F,
   0.00115967F,
   0.000000F
};

/* same in fixed-point */
static const Word32 inter6[61] =
{
   29443,
   28346,
   25207,
   20449,
   14701,
   8693,
   3143,
   - 1352,
   - 4402,
   - 5865,
   - 5850,
   - 4673,
   - 2783,
   - 672,
   1211,
   2536,
   3130,
   2991,
   2259,
   1170,
   0,
   - 1001,
   - 1652,
   - 1868,
   - 1666,
   - 1147,
   - 464,
   218,
   756,
   1060,
   1099,
   904,
   550,
   135,
   - 245,
   - 514,
   - 634,
   - 602,
   - 451,
   - 231,
   0,
   191,
   308,
   340,
   296,
   198,
   78,
   - 36,
   - 120,
   - 163,
   - 165,
   - 132,
   - 79,
   - 19,
   34,
   73,
   91,
   89,
   70,
   38,
   0
};

static const Word16 startPos1[2] =
   {
      1,
      3
   };

static const Word16 startPos2[4] =
   {
      0,
      1,
      2,
      4
   };

static const Word16 startPos[2 * 4 * 2] =
   {
      0,
      2,
      0,
      3,
      0,
      2,
      0,
      3,
      1,
      3,
      2,
      4,
      1,
      4,
      1,
      4
   };

/* Scalar quantization tables of the pitch gain and the codebook gain. */
#define NB_QUA_PITCH 16

static const Float32 qua_gain_pitch[NB_QUA_PITCH] =
{
0.0F,
0.20001220703125F,
0.400146484375F,
0.5F,
0.5999755859375F,
0.70001220703125F,
0.75F,
0.79998779296875F,
0.8499755859375F,
0.9000244140625F,
0.95001220703125F,
1.0F,
1.04998779296875F,
1.0999755859375F,
1.1500244140625F,
1.20001220703125F
};

static const Float32 qua_gain_pitch_MR122[NB_QUA_PITCH] =
{
0.0F,
0.199951171875F,
0.400146484375F,
0.5F,
0.599853515625F,
0.699951171875F,
0.75F,
0.7998046875F,
0.849853515625F,
0.89990234375F,
0.949951171875F,
1.0F,
1.0498046875F,
1.099853515625F,
1.14990234375F,
1.199951171875F
};

#define NB_QUA_CODE 32
static const Float32 gain_factor[NB_QUA_CODE] =
{
/* gain factor (g_fac) */
   0.0776367F,
   0.100586F,
   0.130859F,
   0.170410F,
   0.204590F,
   0.235352F,
   0.270508F,
   0.311035F,
   0.357910F,
   0.411133F,
   0.473145F,
   0.543945F,
   0.625488F,
   0.719238F,
   0.827148F,
   0.951172F,
   1.09424F,
   1.25830F,
   1.44678F,
   1.66406F,
   1.91357F,
   2.20068F,
   2.53076F,
   2.91016F,
   3.34717F,
   3.84912F,
   4.42627F,
   5.09033F,
   6.10840F,
   7.94092F,
   10.3232F,
   13.4204F
};






static const Word8 gray[8] =
   {
      0,
      1,
      3,
      2,
      6,
      4,
      5,
      7
   };

/*
 * grid[0] = 1.0;
 * grid[61] = -1.0;
 * for (i = 1; i < 61; i++)
 *    grid[i] = (Float32)cos((6.283185307*i)/(2.0*60));
 */
static const Float32 grid[61] =
{
   1.00000F,
   0.998630F,
   0.994522F,
   0.987688F,
   0.978148F,
   0.965926F,
   0.951057F,
   0.933580F,
   0.913545F,
   0.891007F,
   0.866025F,
   0.838671F,
   0.809017F,
   0.777146F,
   0.743145F,
   0.707107F,
   0.669131F,
   0.629320F,
   0.587785F,
   0.544639F,
   0.500000F,
   0.453990F,
   0.406737F,
   0.358368F,
   0.309017F,
   0.258819F,
   0.207912F,
   0.156434F,
   0.104528F,
   0.0523360F,
   4.48966e-011F,
   - 0.0523360F,
   - 0.104528F,
   - 0.156434F,
   - 0.207912F,
   - 0.258819F,
   - 0.309017F,
   - 0.358368F,
   - 0.406737F,
   - 0.453990F,
   - 0.500000F,
   - 0.544639F,
   - 0.587785F,
   - 0.629320F,
   - 0.669131F,
   - 0.707107F,
   - 0.743145F,
   - 0.777146F,
   - 0.809017F,
   - 0.838671F,
   - 0.866025F,
   - 0.891007F,
   - 0.913545F,
   - 0.933580F,
   - 0.951057F,
   - 0.965926F,
   - 0.978148F,
   - 0.987688F,
   - 0.994522F,
   - 0.998630F,
   - 1.00000F
};

/*
 * 1/6 resolution interpolation filter  (-3 dB at 3600 Hz)
 * Note: The IS641 (7.4) 1/3 resolution filter is simply a subsampled
 *       version of the 1/6 resolution filter, i.e. it uses
 *       every second coefficient:
 *
 *       b24[k](1/3) = b24[2*k](1/6), 0 <= k <= 3*L_INTER_SRCH
 */
static const Float32 b24[UP_SAMP_MAX*L_INTER_SRCH+1] =
{
   0.900848F,
   0.864136F,
   0.760071F,
   0.605408F,
   0.424072F,
   0.242462F,
   0.0840759F,
   - 0.0343933F,
   - 0.105560F,
   - 0.131348F,
   - 0.121124F,
   - 0.0884705F,
   - 0.0476379F,
   - 0.0102539F,
   0.0162964F,
   0.0296021F,
   0.0312195F,
   0.0251160F,
   0.0157471F,
   0.00671387F,
   0.000000F,
   - 0.00399780F,
   - 0.00592041F,
   - 0.00656128F,
   0.000000F
};

/*
 * Exponential Window coefficients used to weight the autocorrelation
 * coefficients for 60 Hz bandwidth expansion of high pitched voice
 * before Levinson-Durbin recursion to compute the LPC coefficients.
 *
 * lagwindow[i] =  exp( -0.5*(2*pi*F0*(i+1)/Fs)^2 ); i = 0,...,9
 * F0 = 60 Hz, Fs = 8000 Hz
 */
static Float32 lag_wind[M] =
{
   0.99889028F,
   0.99556851F,
   0.99005681F,
   0.98239160F,
   0.97262347F,
   0.96081644F,
   0.94704735F,
   0.93140495F,
   0.91398895F,
   0.89490914F
};

/* initialization table for lsp history in DTX */
static const Float32 lsp_init_data[M] =
   {
      0.9595F,
      0.8413F,
      0.6549F,
      0.4154F,
      0.1423F,
      - 0.1423F,
      - 0.4154F,
      - 0.6549F,
      - 0.8413F,
      - 0.9595F
   };

#define PAST_RQ_INIT_SIZE 8

/* initalization table for MA predictor in dtx mode */
static const Float32 past_rq_init[80] =
{
-62.9883F, -77.6367F, -107.178F, -154.785F, -160.156F, -188.721F, -173.584F, -122.559F,
-65.4297F, -47.1191F, -0.488281F, 30.5176F, 29.7852F, -9.52148F, -2.19727F, 25.6348F,
31.4941F, 69.0918F, 90.8203F, 140.381F, -67.627F, -79.1016F, -48.0957F, -118.896F,
-108.643F, -88.3789F, -71.2891F, -6.5918F, 43.2129F, 132.568F, 83.4961F, 126.221F,
125.977F, 31.7383F, 6.5918F, -25.3906F, -29.2969F, -34.1797F, -18.0664F, -13.6719F,
-137.695F, -230.225F, -371.094F, -235.596F, -198.73F, -128.418F, -78.6133F, -0.488281F,
38.8184F, 160.4F, -76.1719F, -69.3359F, -94.2383F, -145.752F, -120.361F, -128.418F,
-102.051F, -55.9082F, 25.6348F, 109.619F, -135.986F, -212.402F, -262.451F, -224.365F,
-231.934F, -183.594F, -173.096F, -77.1484F, 15.1367F, 118.652F, -76.6602F, -46.6309F,
-49.5605F, -80.5664F, -39.0625F, -25.1465F, -12.4512F, 31.9824F, 82.5195F, 125.732F
};

/* LSF means ->normalize frequency domain */

static const Float32 mean_lsf_3[10] =
{
   377.441F,
	554.688F,
	922.363F,
	1339.84F,
	1702.15F,
	2046.39F,
	2452.88F,
	2741.46F,
	3116.70F,
	3348.14F,
};

static const Float32 mean_lsf_5[10] =
{
   337.891F,
	507.080F,
	834.961F,
	1247.07F,
	1646.00F,
	1982.91F,
	2407.96F,
	2708.01F,
	3104.00F,
	3344.97F,
};


/* LSF prediction factors (not in MR122) */
static const Float32 pred_fac[10] =
{
   0.291626F,
   0.328644F,
   0.383636F,
   0.405640F,
   0.438873F,
   0.355560F,
   0.323120F,
   0.298065F,
   0.262238F,
   0.197876F
};

/* codebooks from IS641 */
#define DICO1_SIZE_3  256
#define DICO2_SIZE_3  512
#define DICO3_SIZE_3  512
static const Float32 dico1_lsf_3[] =
{
   	1.46484F,
		20.0195F,
		-31.9824F,
		37.5977F,
		-13.6719F,
		-179.443F,
		44.6777F,
		-15.8691F,
		-64.6973F,
		2.19727F,
		-51.2695F,
		-88.1348F,
		27.5879F,
		175.293F,
		443.604F,
		246.582F,
		296.387F,
		384.033F,
		209.229F,
		325.439F,
		555.664F,
		201.904F,
		382.813F,
		471.924F,
		175.049F,
		485.596F,
		538.574F,
		204.590F,
		286.133F,
		445.068F,
		176.025F,
		244.141F,
		525.879F,
		69.8242F,
		116.211F,
		368.408F,
		-60.3027F,
		-129.639F,
		56.1523F,
		35.8887F,
		-20.0195F,
		138.916F,
		6.34766F,
		-43.2129F,
		-230.469F,
		-6.59180F,
		-66.6504F,
		168.945F,
		-40.0391F,
		-64.4531F,
		-44.6777F,
		54.6875F,
		192.871F,
		253.662F,
		219.482F,
		230.957F,
		146.729F,
		118.408F,
		188.232F,
		280.762F,
		127.930F,
		165.283F,
		220.459F,
		-34.1797F,
		91.5527F,
		189.941F,
		100.098F,
		165.039F,
		104.736F,
		73.4863F,
		129.395F,
		246.338F,
		175.537F,
		157.715F,
		9.27734F,
		55.1758F,
		89.5996F,
		9.76563F,
		35.4004F,
		-10.9863F,
		-123.291F,
		70.8008F,
		29.5410F,
		-29.5410F,
		73.7305F,
		31.0059F,
		40.5273F,
		-30.2734F,
		-93.5059F,
		-233.398F,
		-87.4023F,
		-111.084F,
		-238.525F,
		174.561F,
		214.355F,
		218.262F,
		238.770F,
		225.342F,
		51.5137F,
		116.455F,
		66.4063F,
		15.6250F,
		45.8984F,
		-19.0430F,
		4.15039F,
		-34.9121F,
		-15.8691F,
		9.27734F,
		156.982F,
		143.066F,
		151.611F,
		-32.7148F,
		-104.004F,
		-158.936F,
		84.7168F,
		133.057F,
		688.477F,
		290.039F,
		665.527F,
		596.191F,
		34.6680F,
		-19.5313F,
		423.584F,
		69.0918F,
		31.7383F,
		112.549F,
		-63.9648F,
		-97.4121F,
		-279.541F,
		-100.342F,
		37.8418F,
		104.980F,
		80.3223F,
		91.5527F,
		190.186F,
		12.9395F,
		-55.1758F,
		-33.9355F,
		-31.4941F,
		-57.6172F,
		410.645F,
		69.5801F,
		181.641F,
		323.975F,
		180.176F,
		170.166F,
		406.250F,
		76.1719F,
		99.8535F,
		64.9414F,
		79.3457F,
		175.781F,
		32.9590F,
		0.244141F,
		53.9551F,
		110.596F,
		1.95313F,
		49.5605F,
		35.4004F,
		72.9980F,
		156.250F,
		185.547F,
		7.08008F,
		114.258F,
		155.762F,
		25.1465F,
		104.736F,
		92.5293F,
		102.539F,
		232.910F,
		227.539F,
		323.730F,
		295.410F,
		307.129F,
		171.875F,
		247.070F,
		281.250F,
		-40.5273F,
		-108.398F,
		-64.9414F,
		-77.1484F,
		-31.7383F,
		-91.7969F,
		46.6309F,
		281.006F,
		464.844F,
		-58.5938F,
		-132.568F,
		-307.617F,
		-27.3438F,
		65.4297F,
		294.678F,
		17.0898F,
		259.277F,
		386.475F,
		67.8711F,
		332.031F,
		384.277F,
		-62.9883F,
		-66.4063F,
		-187.500F,
		4.63867F,
		137.451F,
		546.875F,
		-0.732422F,
		-64.6973F,
		32.9590F,
		-72.0215F,
		-144.287F,
		-94.7266F,
		34.1797F,
		86.4258F,
		-50.2930F,
		-63.4766F,
		-123.047F,
		-194.092F,
		-105.713F,
		-175.293F,
		-322.021F,
		26.6113F,
		80.8105F,
		234.863F,
		-104.736F,
		-21.2402F,
		159.180F,
		-72.2656F,
		104.004F,
		248.779F,
		-58.3496F,
		189.209F,
		207.764F,
		119.385F,
		325.684F,
		261.963F,
		-81.5430F,
		-81.0547F,
		6.10352F,
		132.568F,
		294.434F,
		441.162F,
		79.5898F,
		14.8926F,
		177.490F,
		141.113F,
		207.275F,
		343.018F,
		-50.7813F,
		-67.6270F,
		80.3223F,
		-37.1094F,
		15.6250F,
		163.330F,
		-105.957F,
		-165.527F,
		-177.490F,
		-110.840F,
		-17.3340F,
		61.2793F,
		147.705F,
		117.188F,
		62.0117F,
		-117.676F,
		2.68555F,
		243.164F,
		-70.5566F,
		96.4355F,
		118.652F,
		176.270F,
		256.104F,
		351.563F,
		-7.32422F,
		-77.1484F,
		-191.895F,
		-25.8789F,
		-28.0762F,
		-151.123F,
		210.205F,
		359.863F,
		344.727F,
		257.568F,
		333.496F,
		289.063F,
		198.242F,
		302.002F,
		225.830F,
		10.2539F,
		-61.2793F,
		-140.625F,
		83.4961F,
		34.4238F,
		-110.840F,
		-41.0156F,
		-19.5313F,
		331.787F,
		-83.4961F,
		-160.156F,
		-430.420F,
		24.4141F,
		200.439F,
		177.002F,
		241.699F,
		182.373F,
		195.313F,
		81.0547F,
		107.422F,
		138.672F,
		161.865F,
		92.5293F,
		208.008F,
		27.3438F,
		40.2832F,
		-90.0879F,
		145.752F,
		222.168F,
		68.8477F,
		-1.95313F,
		203.613F,
		312.744F,
		-85.9375F,
		139.648F,
		169.678F,
		112.793F,
		548.340F,
		440.918F,
		84.2285F,
		46.3867F,
		335.449F,
		101.563F,
		223.389F,
		528.809F,
		41.0156F,
		-20.0195F,
		68.3594F,
		-125.977F,
		-108.887F,
		205.078F,
		11.4746F,
		130.127F,
		10.7422F,
		-88.3789F,
		-173.584F,
		-279.053F,
		5.37109F,
		47.1191F,
		359.375F,
		-20.7520F,
		56.8848F,
		442.627F,
		-15.1367F,
		141.357F,
		367.188F,
		134.277F,
		230.469F,
		427.002F,
		176.514F,
		158.691F,
		280.273F,
		237.305F,
		215.820F,
		340.576F,
		-103.760F,
		156.982F,
		0.000000F,
		244.141F,
		232.422F,
		268.066F,
		60.7910F,
		353.027F,
		164.063F,
		-81.5430F,
		-21.2402F,
		530.273F,
		-135.254F,
		459.473F,
		652.344F,
		34.1797F,
		445.801F,
		452.393F,
		224.609F,
		427.002F,
		632.324F,
		262.695F,
		471.924F,
		497.559F,
		-33.4473F,
		-108.154F,
		-379.639F,
		309.814F,
		286.621F,
		114.258F,
		-120.361F,
		-29.7852F,
		371.338F,
		-110.107F,
		252.197F,
		296.387F,
		117.676F,
		413.818F,
		272.949F,
		198.975F,
		158.447F,
		93.7500F,
		-108.887F,
		-168.945F,
		26.1230F,
		-77.8809F,
		-147.705F,
		-28.8086F,
		-50.5371F,
		-123.291F,
		128.174F,
		-114.258F,
		-2.92969F,
		667.969F,
		18.3105F,
		472.168F,
		318.604F,
		214.844F,
		575.684F,
		553.467F,
		313.721F,
		384.521F,
		489.258F,
		-11.7188F,
		-74.2188F,
		-289.551F,
		-106.201F,
		-112.549F,
		-61.2793F,
		-89.3555F,
		-98.6328F,
		-133.545F,
		-70.5566F,
		-147.705F,
		-145.752F,
		-131.348F,
		-197.754F,
		-40.2832F,
		-29.2969F,
		0.732422F,
		86.9141F,
		156.006F,
		302.979F,
		366.699F,
		23.4375F,
		43.2129F,
		183.105F,
		-106.201F,
		-142.822F,
		-286.621F,
		-86.9141F,
		26.6113F,
		-19.2871F,
		-118.408F,
		70.3125F,
		489.502F,
		2.19727F,
		272.461F,
		178.467F,
		214.844F,
		520.996F,
		230.957F,
		-64.6973F,
		386.963F,
		260.010F,
		282.471F,
		295.410F,
		205.811F,
		-121.582F,
		-163.086F,
		105.225F,
		91.3086F,
		78.3691F,
		-55.9082F,
		351.563F,
		512.939F,
		337.158F,
		109.619F,
		112.549F,
		281.982F,
		-25.6348F,
		9.52148F,
		-93.7500F,
		-64.2090F,
		89.5996F,
		44.4336F,
		-90.5762F,
		-161.133F,
		188.721F,
		-45.8984F,
		281.006F,
		237.061F,
		325.439F,
		398.438F,
		350.342F,
		188.965F,
		309.326F,
		298.096F,
		-117.676F,
		-203.125F,
		-363.525F,
		-57.8613F,
		-51.2695F,
		209.961F,
		217.285F,
		394.287F,
		259.766F,
		115.234F,
		259.277F,
		291.016F,
		45.1660F,
		262.939F,
		241.455F,
		-138.672F,
		-242.188F,
		-416.016F,
		-109.619F,
		-220.215F,
		-498.779F,
		-34.6680F,
		-92.0410F,
		-111.816F,
		-51.2695F,
		-135.254F,
		-251.221F,
		-2.68555F,
		276.611F,
		552.979F,
		-80.3223F,
		-164.795F,
		-218.018F,
		-61.0352F,
		160.400F,
		289.795F,
		126.709F,
		368.652F,
		434.326F,
		126.953F,
		131.592F,
		342.529F,
		128.662F,
		346.924F,
		317.871F,
		-137.451F,
		-212.646F,
		-304.688F,
		-35.8887F,
		-113.037F,
		214.600F,
		-18.5547F,
		569.824F,
		693.359F,
		137.451F,
		628.174F,
		582.275F,
		154.297F,
		470.215F,
		712.891F,
		175.537F,
		493.896F,
		449.219F,
		-133.057F,
		-176.514F,
		270.508F,
		31.4941F,
		-30.5176F,
		215.820F,
		345.947F,
		398.438F,
		225.830F,
		-22.9492F,
		382.324F,
		427.490F,
		-83.2520F,
		374.268F,
		378.662F,
		144.287F,
		96.4355F,
		-66.8945F,
		-18.5547F,
		239.502F,
		691.162F,
		37.3535F,
		728.760F,
		450.195F,
		251.953F,
		626.221F,
		671.143F,
		368.164F,
		691.406F,
		458.740F,
		193.115F,
		292.725F,
		131.348F,
		-46.3867F,
		-110.596F,
		363.525F,
		-67.8711F,
		-133.789F,
		282.715F,
		-59.8145F,
		473.877F,
		499.023F,
		250.000F,
		380.859F,
		402.832F,
		125.000F,
		61.7676F,
		113.770F,
		-15.1367F,
		-78.8574F,
		281.006F,
		-115.479F,
		-91.7969F,
		123.779F,
		-105.713F,
		336.914F,
		527.832F,
		219.482F,
		474.365F,
		352.783F,
		32.7148F,
		171.875F,
		107.422F,
		112.305F,
		128.174F,
		-6.83594F,
		-109.863F,
		68.1152F,
		326.660F,
		0.000000F,
		237.061F,
		61.5234F,
		-108.643F,
		-153.076F,
		-241.943F,
		-84.9609F,
		-146.973F,
		-347.656F,
		97.1680F,
		173.828F,
		404.297F,
		-26.1230F,
		76.6602F,
		-43.4570F,
		22.7051F,
		543.457F,
		546.387F,
		126.465F,
		207.275F,
		160.156F,
		-112.793F,
		-173.584F,
		-109.131F,
		42.4805F,
		-8.30078F,
		290.771F,
		-29.0527F,
		10.2539F,
		245.361F,
		-90.8203F,
		66.8945F,
		185.059F,
		252.930F,
		574.219F,
		448.730F,
		164.795F,
		420.898F,
		365.723F,
		104.980F,
		313.965F,
		520.752F,
		-31.4941F,
		-107.178F,
		0.000000F,
		-91.0645F,
		195.313F,
		523.438F,
		1.46484F,
		387.451F,
		604.980F,
		116.699F,
		145.508F,
		519.531F,
		-104.492F,
		-179.688F,
		367.432F,
		93.9941F,
		43.4570F,
		239.258F,
		33.9355F,
		109.619F,
		299.072F,
		-128.418F,
		-205.566F,
		-239.746F,
		35.4004F,
		379.395F,
		303.223F,
		152.100F,
		353.516F,
		160.156F,
		85.2051F,
		248.047F,
		361.816F,
		7.56836F,
		-68.3594F,
		101.318F,
		-77.1484F,
		176.758F,
		400.635F,
		87.8906F,
		258.301F,
		135.742F,
		-106.445F,
		-87.4023F,
		293.213F,
		-86.6699F,
		274.170F,
		473.389F,
		97.9004F,
		386.719F,
		548.828F,
		-128.662F,
		-247.070F,
		86.6699F,
		56.8848F,
		58.1055F,
		545.166F,
		-134.277F,
		-218.994F,
		-156.006F,
		-89.1113F,
		-122.314F,
		477.783F,
		94.9707F,
		454.102F,
		395.752F,
		39.5508F,
		276.367F,
		308.594F,
		-57.8613F,
		286.621F,
		339.355F,
		-156.250F,
		-100.342F,
		28.3203F,
		-55.6641F,
		413.574F,
		561.035F,
		400.146F,
		533.691F,
		553.467F,
		137.207F,
		310.791F,
		648.926F,
		78.8574F,
		82.5195F,
		433.105F,
		141.113F,
		270.264F,
		208.008F,
		5.37109F,
		145.020F,
		228.027F,
		-34.9121F,
		175.293F,
		108.887F

};
static const Float32 dico2_lsf_3[] =
{
   12.2070F,
   17.3340F,
   -2.19727F,
   -82.5195F,
   -170.410F,
   -343.506F,
   24.9023F,
   -33.6914F,
   -200.195F,
   -75.6836F,
   -114.502F,
   -280.029F,
   101.074F,
   16.3574F,
   -65.1855F,
   258.789F,
   198.730F,
   351.807F,
   377.930F,
   332.031F,
   310.547F,
   428.223F,
   462.646F,
   405.518F,
   492.920F,
   520.752F,
   444.336F,
   441.406F,
   565.918F,
   450.439F,
   157.227F,
   -22.7051F,
   110.840F,
   209.473F,
   80.3223F,
   -33.2031F,
   119.385F,
   -62.9883F,
   -31.2500F,
   -48.3398F,
   -181.885F,
   -10.0098F,
   -12.6953F,
   -64.6973F,
   -240.479F,
   84.4727F,
   33.4473F,
   116.943F,
   -425.049F,
   -182.617F,
   -166.992F,
   -283.936F,
   -421.143F,
   -89.5996F,
   -218.506F,
   -279.541F,
   -191.406F,
   -119.141F,
   -230.957F,
   -236.328F,
   -20.7520F,
   -95.2148F,
   -177.002F,
   52.4902F,
   -83.0078F,
   -41.7480F,
   249.023F,
   223.633F,
   480.713F,
   137.695F,
   43.7012F,
   182.129F,
   161.621F,
   238.525F,
   423.340F,
   216.553F,
   151.855F,
   223.145F,
   229.248F,
   208.984F,
   284.424F,
   75.4395F,
   167.969F,
   196.045F,
   223.877F,
   39.3066F,
   139.160F,
   28.8086F,
   -4.88281F,
   -69.0918F,
   -199.219F,
   -10.2539F,
   49.8047F,
   -299.805F,
   -79.3457F,
   -112.793F,
   -235.107F,
   -49.3164F,
   -34.9121F,
   -241.211F,
   -118.164F,
   -88.1348F,
   -171.387F,
   -238.770F,
   -116.455F,
   -73.7305F,
   -192.871F,
   -290.039F,
   -24.4141F,
   -191.895F,
   -265.625F,
   -257.324F,
   -231.201F,
   -411.133F,
   -49.3164F,
   -205.811F,
   -190.918F,
   -253.662F,
   -336.426F,
   -219.971F,
   -152.344F,
   -26.8555F,
   -20.7520F,
   86.9141F,
   52.0020F,
   -2.44141F,
   -120.361F,
   88.8672F,
   188.965F,
   103.760F,
   200.684F,
   116.943F,
   -20.2637F,
   135.986F,
   126.953F,
   -242.188F,
   -380.859F,
   -139.648F,
   -147.217F,
   -180.908F,
   -6.34766F,
   -122.559F,
   -155.762F,
   -220.459F,
   51.0254F,
   74.7070F,
   35.8887F,
   -77.1484F,
   -144.775F,
   -145.508F,
   -20.7520F,
   -51.5137F,
   -54.9316F,
   -224.121F,
   -129.150F,
   28.5645F,
   56.8848F,
   -107.178F,
   -180.176F,
   268.799F,
   183.350F,
   154.541F,
   355.713F,
   418.945F,
   368.896F,
   430.908F,
   355.713F,
   222.168F,
   273.926F,
   282.227F,
   207.275F,
   330.566F,
   211.914F,
   114.746F,
   -212.646F,
   -280.762F,
   -438.477F,
   -212.646F,
   -210.205F,
   -242.188F,
   -28.8086F,
   37.8418F,
   51.7578F,
   -256.592F,
   -207.275F,
   -147.949F,
   -272.705F,
   -451.416F,
   -671.387F,
   -248.779F,
   -348.389F,
   -456.299F,
   90.3320F,
   -44.9219F,
   -101.074F,
   234.131F,
   120.361F,
   25.3906F,
   233.887F,
   253.662F,
   132.568F,
   37.5977F,
   159.424F,
   49.0723F,
   304.932F,
   123.779F,
   36.6211F,
   161.865F,
   122.803F,
   56.1523F,
   152.100F,
   189.697F,
   164.795F,
   160.889F,
   21.4844F,
   -26.8555F,
   205.811F,
   59.5703F,
   54.6875F,
   93.2617F,
   132.080F,
   73.7305F,
   176.758F,
   105.713F,
   162.598F,
   284.668F,
   179.199F,
   83.2520F,
   -33.6914F,
   4.88281F,
   -96.9238F,
   -288.818F,
   -103.516F,
   -11.2305F,
   -78.3691F,
   -85.9375F,
   -30.2734F,
   325.439F,
   249.268F,
   263.672F,
   63.9648F,
   89.3555F,
   176.514F,
   225.098F,
   69.0918F,
   -134.521F,
   7.56836F,
   -155.273F,
   -149.170F,
   -168.213F,
   -170.166F,
   -101.318F,
   -232.422F,
   -190.186F,
   -49.0723F,
   -324.463F,
   -145.996F,
   -87.6465F,
   -232.666F,
   -313.721F,
   40.5273F,
   120.361F,
   74.4629F,
   53.9551F,
   206.543F,
   171.631F,
   148.926F,
   205.078F,
   228.516F,
   188.965F,
   -176.514F,
   -323.242F,
   -307.861F,
   -87.1582F,
   -250.244F,
   -338.867F,
   -267.578F,
   -335.938F,
   -89.1113F,
   -345.703F,
   -459.229F,
   -148.438F,
   -438.965F,
   -421.631F,
   -164.551F,
   -133.057F,
   -286.377F,
   -171.631F,
   165.527F,
   191.895F,
   36.1328F,
   -30.0293F,
   169.922F,
   314.453F,
   157.227F,
   85.4492F,
   -2.44141F,
   101.074F,
   149.902F,
   3.66211F,
   33.4473F,
   83.9844F,
   -51.5137F,
   -198.730F,
   -369.141F,
   -199.951F,
   -95.4590F,
   -227.051F,
   -143.555F,
   11.4746F,
   -144.287F,
   -219.238F,
   -221.924F,
   -267.822F,
   -39.7949F,
   -310.547F,
   -284.912F,
   -38.3301F,
   -357.422F,
   -372.314F,
   -94.9707F,
   -311.035F,
   -290.039F,
   -152.344F,
   163.818F,
   52.0020F,
   110.840F,
   30.2734F,
   -66.8945F,
   -128.174F,
   -177.979F,
   -121.094F,
   -37.1094F,
   -328.125F,
   29.7852F,
   32.9590F,
   -709.229F,
   -143.799F,
   -96.1914F,
   -421.875F,
   107.666F,
   -12.2070F,
   360.352F,
   220.703F,
   192.139F,
   77.1484F,
   57.6172F,
   -107.422F,
   -84.7168F,
   52.9785F,
   100.830F,
   -222.412F,
   -223.877F,
   29.5410F,
   -111.084F,
   -227.539F,
   49.3164F,
   -22.4609F,
   -113.525F,
   -91.5527F,
   119.141F,
   95.2148F,
   115.723F,
   213.867F,
   177.979F,
   77.1484F,
   -443.115F,
   -320.313F,
   -163.330F,
   21.2402F,
   234.863F,
   105.469F,
   137.451F,
   -60.7910F,
   -258.301F,
   61.0352F,
   69.5801F,
   269.775F,
   278.564F,
   104.248F,
   169.922F,
   -253.418F,
   -406.250F,
   -386.230F,
   -231.445F,
   84.4727F,
   39.0625F,
   -75.4395F,
   -66.4063F,
   -209.473F,
   163.574F,
   152.344F,
   305.176F,
   -230.469F,
   -99.6094F,
   -162.598F,
   -147.949F,
   -78.1250F,
   -93.7500F,
   -120.117F,
   56.1523F,
   15.8691F,
   81.5430F,
   -12.2070F,
   -3.90625F,
   -3.90625F,
   -168.457F,
   -341.064F,
   437.256F,
   418.945F,
   341.553F,
   604.980F,
   503.662F,
   342.773F,
   303.955F,
   359.131F,
   348.145F,
   -93.2617F,
   -253.174F,
   -0.488281F,
   42.2363F,
   -97.1680F,
   279.541F,
   364.014F,
   494.141F,
   439.697F,
   188.477F,
   311.035F,
   367.676F,
   348.877F,
   423.584F,
   488.525F,
   263.428F,
   297.363F,
   310.791F,
   -281.738F,
   -451.904F,
   -324.463F,
   -197.266F,
   -276.611F,
   -267.578F,
   -110.107F,
   -252.197F,
   -420.410F,
   15.8691F,
   141.113F,
   -20.5078F,
   -360.352F,
   -594.238F,
   -434.082F,
   -186.768F,
   -333.496F,
   -120.605F,
   -53.2227F,
   -145.020F,
   -227.295F,
   82.2754F,
   -57.6172F,
   137.207F,
   575.439F,
   649.902F,
   473.145F,
   363.525F,
   311.523F,
   213.379F,
   46.1426F,
   87.4023F,
   91.3086F,
   -370.850F,
   -556.885F,
   -572.754F,
   -236.084F,
   -310.303F,
   -511.475F,
   -153.320F,
   -290.039F,
   -376.465F,
   405.518F,
   254.639F,
   133.301F,
   137.939F,
   259.033F,
   178.711F,
   -15.6250F,
   -204.102F,
   -105.957F,
   -106.445F,
   -23.4375F,
   49.5605F,
   263.184F,
   296.875F,
   399.414F,
   221.436F,
   374.512F,
   240.723F,
   79.5898F,
   235.596F,
   206.299F,
   34.6680F,
   -20.5078F,
   48.0957F,
   114.746F,
   580.811F,
   383.301F,
   276.611F,
   114.746F,
   296.387F,
   96.4355F,
   335.938F,
   292.969F,
   274.658F,
   254.395F,
   84.9609F,
   -132.568F,
   -301.270F,
   -91.7969F,
   -52.4902F,
   -44.1895F,
   117.432F,
   -475.342F,
   -395.752F,
   -51.2695F,
   -183.105F,
   -289.307F,
   95.2148F,
   7.08008F,
   -97.4121F,
   6.59180F,
   200.195F,
   301.758F,
   184.326F,
   169.678F,
   239.014F,
   99.8535F,
   -42.4805F,
   292.236F,
   252.686F,
   222.656F,
   331.055F,
   450.684F,
   -242.188F,
   -350.830F,
   118.164F,
   -362.549F,
   -415.039F,
   50.7813F,
   -100.586F,
   293.945F,
   349.609F,
   -66.1621F,
   218.750F,
   279.297F,
   -101.563F,
   433.838F,
   350.098F,
   -414.063F,
   -645.508F,
   -49.8047F,
   -436.768F,
   -378.662F,
   252.197F,
   -404.297F,
   -380.615F,
   318.115F,
   -305.908F,
   -387.939F,
   263.916F,
   -163.330F,
   -267.334F,
   -16.1133F,
   -166.504F,
   78.1250F,
   -84.2285F,
   160.889F,
   74.4629F,
   260.986F,
   -315.430F,
   -196.289F,
   -4.63867F,
   -399.170F,
   -315.186F,
   7.08008F,
   -410.889F,
   -121.338F,
   17.3340F,
   -70.0684F,
   -1.70898F,
   -24.4141F,
   -120.605F,
   -234.863F,
   -57.8613F,
   208.008F,
   459.229F,
   424.805F,
   -297.119F,
   -338.623F,
   55.4199F,
   -161.133F,
   73.7305F,
   91.0645F,
   23.4375F,
   265.381F,
   306.885F,
   -262.207F,
   -407.471F,
   39.0625F,
   118.408F,
   506.836F,
   438.965F,
   -228.027F,
   -53.7109F,
   134.766F,
   -145.508F,
   -149.414F,
   57.8613F,
   82.0313F,
   419.922F,
   214.600F,
   156.982F,
   153.564F,
   105.957F,
   309.326F,
   127.441F,
   398.682F,
   3.66211F,
   59.5703F,
   -107.666F,
   360.107F,
   175.049F,
   44.9219F,
   444.092F,
   388.184F,
   417.236F,
   241.211F,
   63.7207F,
   228.760F,
   510.986F,
   572.510F,
   371.094F,
   522.217F,
   453.613F,
   392.090F,
   -140.869F,
   -141.357F,
   -293.701F,
   -233.398F,
   32.9590F,
   -119.141F,
   -113.281F,
   12.4512F,
   -82.5195F,
   -153.564F,
   -84.9609F,
   -176.514F,
   279.785F,
   506.104F,
   352.051F,
   535.156F,
   357.910F,
   222.412F,
   -352.539F,
   -383.789F,
   -556.152F,
   341.797F,
   173.340F,
   316.650F,
   325.928F,
   154.541F,
   226.563F,
   350.098F,
   535.645F,
   633.301F,
   591.309F,
   538.086F,
   459.229F,
   239.746F,
   547.363F,
   452.637F,
   92.7734F,
   193.359F,
   279.541F,
   -15.3809F,
   -131.592F,
   101.074F,
   -61.5234F,
   -235.352F,
   -76.6602F,
   -307.861F,
   -166.748F,
   -190.430F,
   -202.881F,
   -128.418F,
   -245.361F,
   -406.738F,
   -277.100F,
   -103.516F,
   -393.311F,
   -110.352F,
   -72.9980F,
   309.570F,
   255.859F,
   156.738F,
   280.029F,
   208.252F,
   208.984F,
   -164.795F,
   -82.0313F,
   33.9355F,
   553.711F,
   327.881F,
   346.191F,
   7.08008F,
   187.500F,
   194.580F,
   -298.828F,
   103.271F,
   137.695F,
   -321.777F,
   -264.160F,
   59.8145F,
   -317.871F,
   -198.242F,
   139.893F,
   -316.895F,
   -394.775F,
   157.715F,
   -236.328F,
   203.613F,
   176.514F,
   242.432F,
   403.320F,
   494.873F,
   -46.6309F,
   -199.463F,
   105.469F,
   161.621F,
   14.6484F,
   48.3398F,
   152.832F,
   243.408F,
   324.707F,
   402.344F,
   479.248F,
   314.697F,
   -389.893F,
   -22.7051F,
   -10.9863F,
   -265.625F,
   9.03320F,
   -20.5078F,
   403.564F,
   636.475F,
   570.557F,
   260.010F,
   498.047F,
   580.322F,
   278.076F,
   567.871F,
   517.090F,
   209.717F,
   87.1582F,
   368.652F,
   162.109F,
   299.561F,
   268.311F,
   116.943F,
   332.031F,
   222.656F,
   463.135F,
   428.223F,
   492.920F,
   285.156F,
   466.064F,
   435.547F,
   97.4121F,
   8.30078F,
   62.5000F,
   -144.775F,
   -74.2188F,
   -257.080F,
   133.545F,
   413.574F,
   343.506F,
   157.959F,
   -24.1699F,
   -83.2520F,
   364.258F,
   402.100F,
   290.527F,
   9.27734F,
   -157.227F,
   -51.7578F,
   96.4355F,
   206.543F,
   54.1992F,
   -171.875F,
   -186.768F,
   -174.805F,
   -176.758F,
   -479.492F,
   -684.570F,
   -36.6211F,
   71.0449F,
   -20.0195F,
   301.025F,
   356.201F,
   245.850F,
   -34.1797F,
   -37.8418F,
   37.3535F,
   107.178F,
   72.5098F,
   382.813F,
   -373.291F,
   -100.098F,
   -155.273F,
   375.000F,
   111.084F,
   -57.8613F,
   -324.219F,
   -33.9355F,
   -63.4766F,
   129.639F,
   135.254F,
   211.914F,
   65.6738F,
   308.594F,
   147.949F,
   -56.8848F,
   215.576F,
   113.037F,
   181.152F,
   146.484F,
   -29.2969F,
   -17.8223F,
   102.783F,
   51.7578F,
   -107.178F,
   -14.1602F,
   196.289F,
   -313.965F,
   -302.979F,
   177.734F,
   71.7773F,
   -119.629F,
   12.2070F,
   -144.287F,
   -220.947F,
   -306.152F,
   10.2539F,
   -167.725F,
   35.8887F,
   -6.10352F,
   66.6504F,
   145.508F,
   -75.9277F,
   296.143F,
   146.729F,
   -184.082F,
   207.275F,
   142.578F,
   104.736F,
   148.193F,
   143.311F,
   -146.973F,
   -40.5273F,
   112.549F,
   -194.336F,
   -200.928F,
   189.697F,
   336.914F,
   222.168F,
   428.467F,
   29.0527F,
   345.947F,
   237.305F,
   -53.4668F,
   -214.844F,
   -389.648F,
   -256.104F,
   -246.582F,
   106.934F,
   -174.072F,
   -336.670F,
   19.0430F,
   0.000000F,
   -109.131F,
   -287.842F,
   -277.344F,
   -322.021F,
   -384.033F,
   548.828F,
   431.396F,
   319.580F,
   230.957F,
   386.475F,
   349.609F,
   280.762F,
   117.676F,
   106.445F,
   -114.502F,
   -270.508F,
   150.879F,
   -109.131F,
   -235.840F,
   265.625F,
   -305.664F,
   -369.873F,
   -27.8320F,
   -269.531F,
   -490.234F,
   -141.357F,
   51.2695F,
   149.658F,
   121.338F,
   -482.178F,
   -350.830F,
   156.738F,
   -309.814F,
   -208.984F,
   246.826F,
   -401.855F,
   -289.307F,
   259.521F,
   -379.639F,
   -164.063F,
   293.945F,
   -413.086F,
   -271.973F,
   152.100F,
   -239.014F,
   -323.730F,
   -311.768F,
   131.592F,
   -35.8887F,
   218.262F,
   -330.566F,
   -218.994F,
   -105.957F,
   216.797F,
   115.967F,
   104.492F,
   37.3535F,
   -93.7500F,
   82.5195F,
   -364.258F,
   -124.756F,
   87.6465F,
   -237.793F,
   -272.217F,
   -114.746F,
   25.6348F,
   -134.277F,
   165.283F,
   -228.760F,
   -279.541F,
   214.111F,
   92.7734F,
   -63.4766F,
   51.2695F,
   411.377F,
   225.586F,
   306.641F,
   433.350F,
   290.527F,
   267.334F,
   346.436F,
   154.053F,
   130.127F,
   153.076F,
   72.9980F,
   -84.7168F,
   -100.342F,
   -130.371F,
   157.959F,
   -158.691F,
   7.08008F,
   -145.264F,
   -92.2852F,
   -333.740F,
   381.592F,
   342.285F,
   273.682F,
   357.666F,
   265.869F,
   344.238F,
   158.203F,
   -511.719F,
   -266.113F,
   -1.46484F,
   75.9277F,
   -47.3633F,
   -212.158F,
   -156.006F,
   -202.881F,
   101.563F,
   -283.691F,
   -298.828F,
   329.346F,
   -304.443F,
   -229.736F,
   442.627F,
   -535.400F,
   -485.107F,
   110.596F,
   -151.123F,
   -333.740F,
   -233.398F,
   -392.090F,
   -481.445F,
   -367.920F,
   -286.865F,
   -258.057F,
   -269.531F,
   -92.0410F,
   146.729F,
   49.0723F,
   458.008F,
   201.416F,
   91.3086F,
   -104.980F,
   -322.998F,
   7.08008F,
   -341.064F,
   -304.932F,
   -324.951F,
   -245.850F,
   -367.188F,
   234.375F,
   -342.041F,
   -490.479F,
   48.0957F,
   -336.670F,
   -475.830F,
   -57.6172F,
   -262.939F,
   30.0293F,
   103.027F,
   150.146F,
   309.814F,
   133.301F,
   -74.7070F,
   372.559F,
   220.703F,
   291.504F,
   436.523F,
   287.354F,
   -152.832F,
   -215.820F,
   -372.559F,
   48.5840F,
   187.012F,
   367.188F,
   -260.010F,
   210.449F,
   48.0957F,
   -252.441F,
   -432.861F,
   -216.553F,
   -195.313F,
   35.4004F,
   146.240F,
   -276.855F,
   -126.709F,
   152.832F,
   -294.189F,
   -470.215F,
   122.070F,
   -222.168F,
   -254.150F,
   -340.576F,
   -360.352F,
   -382.568F,
   -236.572F,
   -127.686F,
   205.566F,
   8.30078F,
   437.988F,
   157.715F,
   210.449F,
   -294.678F,
   -460.938F,
   -244.629F,
   -19.0430F,
   -2.19727F,
   -164.063F,
   254.883F,
   185.303F,
   19.5313F,
   -146.484F,
   278.076F,
   248.779F,
   13.9160F,
   488.281F,
   347.168F,
   -203.369F,
   345.215F,
   273.682F,
   -293.457F,
   397.949F,
   307.617F,
   -112.549F,
   346.680F,
   303.711F,
   375.244F,
   238.037F,
   61.7676F,
   -69.0918F,
   79.1016F,
   -87.6465F,
   146.240F,
   -47.6074F,
   25.8789F,
   143.555F,
   15.1367F,
   -143.311F,
   -184.814F,
   157.471F,
   50.0488F,
   12.4512F,
   293.213F,
   185.059F,
   -295.166F,
   164.307F,
   -95.2148F,
   -152.344F,
   385.986F,
   229.736F,
   -36.8652F,
   249.756F,
   179.443F,
   688.477F,
   317.627F,
   168.457F,
   -73.7305F,
   127.930F,
   -24.1699F,
   -219.727F,
   -387.695F,
   -290.283F,
   264.648F,
   61.2793F,
   58.1055F,
   491.699F,
   437.500F,
   246.582F,
   303.955F,
   398.682F,
   425.049F,
   -299.561F,
   -375.977F,
   -294.922F,
   -151.611F,
   111.328F,
   -26.6113F,
   9.76563F,
   -15.8691F,
   192.383F,
   -196.533F,
   -170.654F,
   -329.590F,
   -142.334F,
   220.703F,
   203.125F,
   -195.557F,
   129.883F,
   145.020F,
   481.445F,
   343.750F,
   329.834F,
   -287.354F,
   -458.984F,
   -516.113F,
   -188.721F,
   138.672F,
   231.445F,
   -247.803F,
   263.428F,
   307.617F,
   -271.240F,
   117.676F,
   -31.7383F,
   434.082F,
   254.883F,
   190.430F,
   -364.014F,
   59.8145F,
   222.656F,
   -77.1484F,
   -278.564F,
   -223.877F,
   -130.859F,
   -352.051F,
   -572.754F,
   -191.650F,
   -377.441F,
   -485.352F,
   -489.014F,
   62.7441F,
   221.924F,
   -451.416F,
   -154.541F,
   -295.166F,
   -375.488F,
   -468.262F,
   -257.324F,
   392.090F,
   546.631F,
   384.766F,
   -138.428F,
   -366.211F,
   -376.953F,
   -312.256F,
   47.6074F,
   334.229F,
   -199.463F,
   71.5332F,
   297.607F,
   -128.174F,
   153.809F,
   292.236F,
   -414.551F,
   -592.041F,
   -449.219F,
   -73.9746F,
   178.467F,
   182.373F,
   -285.400F,
   -61.2793F,
   65.6738F,
   -231.934F,
   -18.3105F,
   411.133F,
   -288.574F,
   -110.596F,
   245.361F,
   -390.381F,
   142.822F,
   92.2852F,
   -506.592F,
   -139.404F,
   -104.248F,
   -129.150F,
   -282.959F,
   -285.889F,
   -69.0918F,
   -50.0488F,
   -137.695F,
   -194.336F,
   304.199F,
   175.049F,
   555.908F,
   226.318F,
   131.592F,
   -110.840F,
   136.475F,
   107.422F,
   -175.049F,
   356.445F,
   394.287F,
   -251.465F,
   256.836F,
   393.066F,
   -285.400F,
   -33.6914F,
   206.787F,
   55.1758F,
   9.52148F,
   -149.414F,
   -305.420F,
   -25.8789F,
   -177.979F,
   -158.936F,
   236.328F,
   317.871F,
   -174.316F,
   -155.273F,
   421.631F,
   86.1816F,
   260.986F,
   100.098F,
   -194.824F,
   -38.0859F,
   268.311F,
   -140.137F,
   224.121F,
   108.887F,
   -319.824F,
   247.070F,
   113.770F,
   343.750F,
   388.428F,
   186.768F,
   348.877F,
   336.914F,
   428.955F,
   475.830F,
   477.539F,
   580.566F,
   385.254F,
   499.756F,
   524.414F,
   223.633F,
   23.9258F,
   -1.70898F,
   462.158F,
   346.191F,
   522.705F,
   84.9609F,
   343.018F,
   385.498F,
   37.1094F,
   276.855F,
   439.697F,
   -65.1855F,
   37.5977F,
   340.576F,
   -284.668F,
   114.502F,
   257.324F,
   -278.809F,
   -98.8770F,
   -261.963F,
   -327.393F,
   -552.734F,
   -385.986F,
   -88.8672F,
   212.158F,
   416.504F,
   -283.691F,
   134.033F,
   378.418F,
   -299.072F,
   -471.680F,
   -406.738F,
   -362.549F,
   -482.666F,
   -501.709F,
   -421.631F,
   -221.191F,
   -23.9258F,
   -463.135F,
   56.8848F,
   364.258F,
   217.773F,
   26.3672F,
   -80.8105F,
   -421.875F,
   -285.645F,
   -415.039F,
   -258.789F,
   483.398F,
   437.012F,
   -261.230F,
   -425.049F,
   -466.064F,
   -2.68555F,
   375.732F,
   321.533F,
   -390.625F,
   22.9492F,
   121.338F,
   102.783F,
   108.154F,
   -48.0957F,
   -385.254F,
   -85.2051F,
   -242.676F,
   -146.240F,
   -131.592F,
   278.320F,
   -235.596F,
   -346.436F,
   -31.4941F,
   -327.393F,
   42.7246F,
   -109.131F,
   -91.5527F,
   320.068F,
   501.709F,
   -90.5762F,
   -158.691F,
   -74.9512F,
   -261.963F,
   147.705F,
   89.1113F,
   -502.197F,
   -27.5879F,
   104.980F,
   159.180F,
   223.145F,
   236.084F,
   -247.070F,
   -387.207F,
   -567.139F,
   367.432F,
   304.688F,
   136.475F,
   63.9648F,
   -118.652F,
   -97.9004F,
   -421.631F,
   327.637F,
   377.441F,
   12.2070F,
   13.6719F,
   105.469F,
   -80.5664F,
   29.0527F,
   -147.461F,
   -370.361F,
   -263.672F,
   -197.754F,
   230.957F,
   275.146F,
   257.568F,
   -341.797F,
   -415.771F,
   -417.969F,
   -310.059F,
   -171.875F,
   -321.533F,
   197.021F,
   444.580F,
   279.053F,
   673.828F,
   392.090F,
   530.029F,
   273.438F,
   99.8535F,
   -36.6211F,
   -35.8887F,
   98.6328F,
   234.131F,
   595.459F,
   466.553F,
   534.424F,
   -221.191F,
   -34.4238F,
   -211.426F,
   -220.703F,
   -34.6680F,
   -111.816F,
   -135.986F,
   -172.852F,
   -409.912F,
   -202.637F,
   -349.365F,
   -386.475F,
   -449.707F,
   -328.613F,
   -265.137F,
   -391.602F,
   -66.4063F,
   223.389F,
   -291.992F,
   188.477F,
   257.813F,
   -155.762F,
   -301.270F,
   -463.135F,
   -122.070F,
   -19.7754F,
   -200.684F,
   -314.697F,
   -393.799F,
   -179.443F,
   -28.5645F,
   191.650F,
   41.0156F,
   -266.113F,
   276.611F,
   225.098F,
   -267.578F,
   -182.129F,
   337.891F,
   70.0684F,
   -133.545F,
   -259.521F,
   -335.938F,
   -537.354F,
   -293.945F,
   -531.250F,
   -383.301F,
   -428.955F,
   -368.896F,
   -547.119F,
   -188.232F,
   -424.072F,
   268.311F,
   202.637F,
   -387.695F,
   176.758F,
   303.467F,
   -376.465F,
   169.189F,
   196.533F,
   -412.598F,
   -58.5938F,
   406.494F,
   -415.039F,
   -0.976563F,
   -163.086F,
   524.658F,
   199.219F,
   254.395F,
   -199.707F,
   -449.463F,
   5.37109F,
   -186.523F,
   -123.779F,
   109.619F,
   -281.006F,
   -150.635F,
   70.5566F,
   -205.811F,
   -389.648F,
   -58.5938F,
   121.582F,
   -57.1289F,
   -160.400F,
   -183.594F,
   117.188F,
   409.668F,
   -77.8809F,
   -117.432F,
   47.1191F,
   -197.998F,
   41.7480F,
   -29.0527F,
   -519.531F,
   -49.3164F,
   -207.031F,
   419.189F,
   278.320F,
   415.039F

};
static const Float32 dico3_lsf_3[] =
{
   16.3574F,
   -4.15039F,
   16.1133F,
   -2.92969F,
   -412.598F,
   -141.846F,
   -25.3906F,
   -66.4063F,
   -262.695F,
   -289.551F,
   -450.439F,
   -91.7969F,
   -278.320F,
   -226.074F,
   -102.539F,
   -14.1602F,
   -63.2324F,
   -160.156F,
   -276.855F,
   -135.010F,
   436.523F,
   299.561F,
   111.084F,
   31.4941F,
   112.793F,
   107.666F,
   -58.5938F,
   -128.906F,
   205.078F,
   125.488F,
   31.7383F,
   -18.3105F,
   271.973F,
   152.100F,
   37.3535F,
   52.7344F,
   260.742F,
   137.695F,
   -1.46484F,
   -67.3828F,
   273.193F,
   177.490F,
   46.3867F,
   -16.6016F,
   171.875F,
   74.7070F,
   29.0527F,
   -64.4531F,
   80.3223F,
   14.8926F,
   -24.4141F,
   38.0859F,
   88.8672F,
   30.0293F,
   44.6777F,
   -50.7813F,
   -41.7480F,
   -30.0293F,
   53.7109F,
   -15.8691F,
   -74.7070F,
   -15.1367F,
   98.1445F,
   4.15039F,
   -161.133F,
   -229.004F,
   -64.9414F,
   0.000000F,
   93.9941F,
   57.3730F,
   67.3828F,
   69.5801F,
   78.1250F,
   65.4297F,
   -82.0313F,
   -48.8281F,
   -176.758F,
   4.15039F,
   -20.5078F,
   93.0176F,
   -132.813F,
   104.736F,
   120.605F,
   126.709F,
   -28.5645F,
   70.3125F,
   74.2188F,
   80.3223F,
   156.982F,
   38.3301F,
   171.143F,
   124.023F,
   292.969F,
   152.588F,
   194.336F,
   148.438F,
   243.652F,
   102.783F,
   120.117F,
   154.297F,
   293.945F,
   190.430F,
   108.887F,
   32.2266F,
   306.885F,
   206.055F,
   133.545F,
   109.619F,
   202.393F,
   160.645F,
   132.080F,
   114.746F,
   276.367F,
   307.129F,
   224.121F,
   156.006F,
   133.545F,
   12.4512F,
   103.271F,
   68.1152F,
   2.19727F,
   95.7031F,
   20.2637F,
   22.9492F,
   132.324F,
   132.568F,
   55.9082F,
   -35.8887F,
   -48.3398F,
   31.4941F,
   47.3633F,
   -45.1660F,
   -210.693F,
   -322.510F,
   -73.7305F,
   7.32422F,
   -145.752F,
   -153.564F,
   -4.63867F,
   27.8320F,
   -219.727F,
   -263.916F,
   113.770F,
   86.1816F,
   -362.061F,
   -384.033F,
   3.66211F,
   -34.9121F,
   -416.992F,
   -502.686F,
   -183.350F,
   47.8516F,
   -458.008F,
   -504.639F,
   -156.738F,
   -62.9883F,
   -570.068F,
   -358.887F,
   -109.863F,
   -137.695F,
   -142.578F,
   -45.4102F,
   -212.891F,
   -101.074F,
   -440.674F,
   -241.211F,
   -274.658F,
   -319.824F,
   -177.246F,
   -275.635F,
   6.83594F,
   41.2598F,
   -253.662F,
   -210.938F,
   -175.293F,
   -60.0586F,
   118.164F,
   8.78906F,
   -56.8848F,
   -11.9629F,
   64.6973F,
   16.3574F,
   70.5566F,
   114.014F,
   43.4570F,
   132.568F,
   197.754F,
   131.836F,
   20.5078F,
   68.8477F,
   164.063F,
   171.631F,
   -238.037F,
   -189.697F,
   31.4941F,
   70.0684F,
   -229.004F,
   -55.4199F,
   233.154F,
   145.264F,
   -394.775F,
   -70.5566F,
   204.102F,
   158.447F,
   -450.928F,
   -52.4902F,
   270.020F,
   175.293F,
   -496.582F,
   -264.893F,
   158.691F,
   107.422F,
   -512.939F,
   -129.150F,
   221.436F,
   140.381F,
   -490.967F,
   -82.0313F,
   163.574F,
   49.8047F,
   -583.252F,
   -168.945F,
   87.8906F,
   33.4473F,
   -526.367F,
   -538.086F,
   -2.19727F,
   68.3594F,
   -64.9414F,
   29.0527F,
   9.52148F,
   47.1191F,
   19.0430F,
   -14.4043F,
   -29.2969F,
   55.1758F,
   -238.037F,
   -209.473F,
   -190.674F,
   -267.334F,
   -151.123F,
   -100.830F,
   -110.107F,
   -205.566F,
   -296.875F,
   -322.510F,
   -198.486F,
   -215.576F,
   -335.938F,
   -394.287F,
   -96.1914F,
   -104.492F,
   -179.932F,
   -271.729F,
   -134.033F,
   -192.871F,
   -214.844F,
   -238.037F,
   -236.084F,
   -156.738F,
   -240.479F,
   -216.309F,
   -310.791F,
   -332.275F,
   -115.479F,
   -196.289F,
   -342.041F,
   -343.506F,
   39.0625F,
   -64.6973F,
   -224.365F,
   -67.1387F,
   -60.5469F,
   -61.0352F,
   -175.293F,
   -92.7734F,
   23.6816F,
   -25.1465F,
   -91.5527F,
   -55.9082F,
   -101.318F,
   -47.1191F,
   -32.9590F,
   -135.498F,
   153.320F,
   88.1348F,
   29.0527F,
   52.7344F,
   141.357F,
   88.8672F,
   95.4590F,
   51.0254F,
   154.785F,
   127.441F,
   -37.5977F,
   -36.1328F,
   128.418F,
   94.9707F,
   41.5039F,
   8.05664F,
   25.6348F,
   65.1855F,
   15.6250F,
   92.7734F,
   -366.943F,
   -244.141F,
   -7.32422F,
   -90.0879F,
   -261.230F,
   14.1602F,
   157.959F,
   54.4434F,
   -371.094F,
   -71.0449F,
   151.611F,
   74.9512F,
   -373.779F,
   38.0859F,
   186.035F,
   98.6328F,
   -495.361F,
   34.4238F,
   179.199F,
   121.826F,
   -451.416F,
   -158.691F,
   74.7070F,
   125.000F,
   -45.6543F,
   -25.3906F,
   -14.4043F,
   106.934F,
   32.7148F,
   -56.1523F,
   38.0859F,
   -45.4102F,
   -14.8926F,
   -63.4766F,
   -3.90625F,
   2.44141F,
   -138.916F,
   -0.732422F,
   -102.783F,
   -72.5098F,
   -421.143F,
   -127.197F,
   -84.4727F,
   43.4570F,
   -332.520F,
   -14.4043F,
   -10.7422F,
   38.3301F,
   -523.926F,
   -112.549F,
   -114.746F,
   -85.2051F,
   -529.785F,
   -0.244141F,
   -90.0879F,
   -29.5410F,
   -385.498F,
   -91.0645F,
   -219.727F,
   -247.803F,
   -272.705F,
   -144.287F,
   -149.658F,
   -191.406F,
   -136.963F,
   29.7852F,
   -18.3105F,
   -109.619F,
   -0.976563F,
   -41.7480F,
   -30.0293F,
   -90.8203F,
   46.8750F,
   41.0156F,
   -18.5547F,
   -32.2266F,
   61.5234F,
   -26.1230F,
   83.0078F,
   51.2695F,
   95.7031F,
   124.268F,
   66.4063F,
   44.1895F,
   -26.6113F,
   35.4004F,
   53.2227F,
   29.0527F,
   -101.563F,
   -64.2090F,
   118.408F,
   64.6973F,
   -44.1895F,
   -1.95313F,
   -69.8242F,
   55.1758F,
   -59.5703F,
   -53.2227F,
   16.8457F,
   -70.8008F,
   -38.5742F,
   46.6309F,
   -0.244141F,
   -15.6250F,
   -144.531F,
   -21.9727F,
   52.0020F,
   -23.4375F,
   62.2559F,
   106.201F,
   43.4570F,
   -19.5313F,
   -90.0879F,
   -4.39453F,
   -8.05664F,
   -19.5313F,
   -10.2539F,
   101.318F,
   34.1797F,
   -54.1992F,
   279.053F,
   158.936F,
   158.447F,
   80.3223F,
   187.256F,
   135.742F,
   60.7910F,
   57.3730F,
   231.445F,
   100.830F,
   107.910F,
   68.1152F,
   34.4238F,
   82.7637F,
   86.9141F,
   135.986F,
   -114.746F,
   -41.5039F,
   24.1699F,
   57.8613F,
   -138.916F,
   -195.313F,
   85.9375F,
   137.939F,
   68.8477F,
   115.479F,
   114.746F,
   81.0547F,
   -48.5840F,
   -168.457F,
   -313.477F,
   -223.877F,
   -47.1191F,
   -104.004F,
   -195.313F,
   -273.926F,
   -6.34766F,
   -90.5762F,
   -119.629F,
   -47.1191F,
   155.518F,
   145.264F,
   126.709F,
   80.5664F,
   99.6094F,
   -28.0762F,
   19.2871F,
   2.92969F,
   116.455F,
   21.2402F,
   -25.1465F,
   -91.7969F,
   -162.598F,
   -84.7168F,
   -67.6270F,
   -71.0449F,
   -124.512F,
   -117.432F,
   41.2598F,
   72.5098F,
   -202.393F,
   -180.176F,
   -50.0488F,
   -41.7480F,
   -78.1250F,
   -131.836F,
   80.0781F,
   69.0918F,
   -209.717F,
   -233.887F,
   107.910F,
   -0.488281F,
   135.742F,
   167.480F,
   31.7383F,
   13.6719F,
   337.646F,
   247.070F,
   184.326F,
   104.248F,
   149.414F,
   180.908F,
   153.320F,
   135.010F,
   -82.7637F,
   -194.336F,
   32.7148F,
   67.6270F,
   -154.541F,
   -264.893F,
   -0.488281F,
   -60.0586F,
   -214.844F,
   -252.686F,
   -392.334F,
   -259.766F,
   -242.676F,
   -115.723F,
   -277.832F,
   -119.141F,
   -101.074F,
   -194.092F,
   17.8223F,
   -50.2930F,
   -1.95313F,
   -33.9355F,
   107.178F,
   49.8047F,
   -42.9688F,
   -141.113F,
   5.61523F,
   31.9824F,
   -65.6738F,
   -184.814F,
   -46.6309F,
   59.8145F,
   -26.6113F,
   -82.5195F,
   27.3438F,
   77.1484F,
   29.2969F,
   -99.1211F,
   -28.8086F,
   149.170F,
   -43.9453F,
   -45.4102F,
   -157.471F,
   28.0762F,
   -42.2363F,
   8.30078F,
   -126.465F,
   -119.385F,
   -36.8652F,
   14.8926F,
   -142.334F,
   -206.055F,
   53.7109F,
   -33.6914F,
   -166.260F,
   -249.023F,
   95.4590F,
   -4.15039F,
   -145.996F,
   -78.3691F,
   38.3301F,
   -72.0215F,
   31.4941F,
   37.8418F,
   -226.074F,
   -213.623F,
   -240.967F,
   69.5801F,
   58.8379F,
   -20.2637F,
   -30.5176F,
   -30.5176F,
   151.367F,
   145.752F,
   105.469F,
   22.4609F,
   95.9473F,
   19.0430F,
   99.8535F,
   14.8926F,
   -95.9473F,
   -180.420F,
   -100.830F,
   -182.617F,
   20.2637F,
   13.1836F,
   88.1348F,
   6.59180F,
   -264.648F,
   31.7383F,
   -82.2754F,
   -169.434F,
   -382.080F,
   72.5098F,
   77.6367F,
   -4.63867F,
   -457.275F,
   8.78906F,
   12.4512F,
   -77.3926F,
   -567.139F,
   -60.0586F,
   56.3965F,
   -20.5078F,
   -562.988F,
   -191.162F,
   9.76563F,
   -43.7012F,
   -545.166F,
   -227.051F,
   -115.723F,
   -112.793F,
   -184.082F,
   -20.9961F,
   -70.3125F,
   -152.832F,
   -588.623F,
   -111.084F,
   -15.3809F,
   41.7480F,
   -268.311F,
   -267.090F,
   -6.34766F,
   -34.9121F,
   -291.260F,
   -111.084F,
   -99.1211F,
   -93.0176F,
   -147.705F,
   -51.2695F,
   -23.4375F,
   -12.4512F,
   -141.602F,
   -116.211F,
   -67.3828F,
   -3.66211F,
   -291.748F,
   -154.785F,
   -293.701F,
   -215.088F,
   -92.2852F,
   -53.9551F,
   -163.330F,
   -232.422F,
   145.020F,
   43.4570F,
   -98.3887F,
   -165.039F,
   186.279F,
   79.8340F,
   146.729F,
   70.8008F,
   41.9922F,
   73.2422F,
   49.5605F,
   38.3301F,
   -13.6719F,
   -82.0313F,
   86.9141F,
   5.85938F,
   -55.6641F,
   -72.2656F,
   -63.2324F,
   -7.08008F,
   -45.4102F,
   64.2090F,
   101.563F,
   3.41797F,
   -86.1816F,
   91.0645F,
   -2.92969F,
   -52.7344F,
   62.7441F,
   23.4375F,
   42.4805F,
   13.9160F,
   -372.559F,
   -150.391F,
   -232.910F,
   -121.826F,
   -121.338F,
   -37.1094F,
   -81.2988F,
   30.5176F,
   25.6348F,
   48.8281F,
   43.7012F,
   -23.6816F,
   -80.8105F,
   -54.6875F,
   186.768F,
   170.166F,
   185.547F,
   62.5000F,
   73.4863F,
   14.4043F,
   111.084F,
   -20.7520F,
   49.8047F,
   70.3125F,
   -125.488F,
   58.5938F,
   61.2793F,
   -26.6113F,
   62.5000F,
   101.807F,
   -8.30078F,
   -100.830F,
   24.6582F,
   104.980F,
   93.7500F,
   38.0859F,
   -7.56836F,
   -2.44141F,
   50.2930F,
   104.004F,
   143.799F,
   35.4004F,
   34.9121F,
   17.3340F,
   197.266F,
   221.191F,
   81.2988F,
   85.2051F,
   240.723F,
   229.004F,
   143.799F,
   80.8105F,
   317.383F,
   201.172F,
   45.6543F,
   124.268F,
   259.277F,
   159.424F,
   92.5293F,
   113.770F,
   356.934F,
   228.760F,
   97.9004F,
   66.8945F,
   192.139F,
   210.205F,
   64.6973F,
   0.488281F,
   148.682F,
   135.010F,
   6.83594F,
   74.4629F,
   226.074F,
   83.0078F,
   25.8789F,
   94.2383F,
   58.8379F,
   -65.1855F,
   -35.8887F,
   54.9316F,
   -43.4570F,
   -130.371F,
   84.7168F,
   122.559F,
   -156.982F,
   -93.0176F,
   96.9238F,
   7.32422F,
   -158.936F,
   -178.955F,
   -106.201F,
   97.1680F,
   -99.3652F,
   -177.246F,
   -118.164F,
   -60.5469F,
   -192.627F,
   -223.145F,
   -106.934F,
   -116.211F,
   -121.582F,
   -95.2148F,
   18.3105F,
   -72.0215F,
   -235.352F,
   -144.043F,
   -147.949F,
   36.6211F,
   -29.5410F,
   -11.9629F,
   -37.8418F,
   -19.0430F,
   228.271F,
   134.277F,
   94.9707F,
   9.27734F,
   -78.3691F,
   31.0059F,
   103.516F,
   76.9043F,
   -69.5801F,
   -27.5879F,
   69.0918F,
   63.2324F,
   160.645F,
   49.5605F,
   78.6133F,
   118.652F,
   220.459F,
   123.291F,
   182.617F,
   101.807F,
   149.170F,
   103.271F,
   135.498F,
   125.000F,
   58.3496F,
   -20.2637F,
   -141.113F,
   -4.63867F,
   -82.7637F,
   -178.467F,
   85.2051F,
   3.17383F,
   -228.027F,
   -341.553F,
   -27.8320F,
   -87.8906F,
   26.1230F,
   168.945F,
   44.4336F,
   21.9727F,
   -303.467F,
   -375.488F,
   -378.662F,
   -177.002F,
   -138.672F,
   -220.459F,
   -332.764F,
   -128.174F,
   -126.221F,
   -208.252F,
   -210.205F,
   -245.117F,
   -41.0156F,
   -168.457F,
   -203.857F,
   15.3809F,
   -33.4473F,
   -135.742F,
   -133.545F,
   35.1563F,
   -69.8242F,
   -199.463F,
   118.408F,
   77.8809F,
   -35.8887F,
   -99.6094F,
   128.418F,
   60.0586F,
   -84.7168F,
   -105.957F,
   72.5098F,
   -6.83594F,
   -70.8008F,
   -114.990F,
   -270.996F,
   -313.721F,
   -112.305F,
   -87.6465F,
   -241.211F,
   -193.848F,
   328.857F,
   317.139F,
   168.457F,
   127.686F,
   296.875F,
   260.742F,
   267.090F,
   184.814F,
   201.416F,
   278.320F,
   183.594F,
   120.605F,
   305.664F,
   333.252F,
   291.748F,
   219.238F,
   127.197F,
   257.080F,
   129.883F,
   105.469F,
   -81.5430F,
   -52.7344F,
   -76.4160F,
   -64.2090F,
   -39.0625F,
   12.6953F,
   -115.234F,
   -37.8418F,
   31.0059F,
   33.2031F,
   -92.7734F,
   10.7422F,
   207.764F,
   100.098F,
   -39.5508F,
   -119.385F,
   30.0293F,
   -62.2559F,
   -194.336F,
   -162.842F,
   266.113F,
   223.877F,
   192.627F,
   120.361F,
   341.064F,
   292.236F,
   136.230F,
   49.3164F,
   -12.4512F,
   -28.8086F,
   -83.4961F,
   -171.143F,
   20.2637F,
   26.3672F,
   -10.2539F,
   -107.666F,
   14.8926F,
   23.1934F,
   70.0684F,
   62.5000F,
   -6.59180F,
   21.7285F,
   127.930F,
   129.639F,
   85.6934F,
   55.4199F,
   144.531F,
   133.057F,
   170.166F,
   37.8418F,
   -40.0391F,
   74.9512F,
   155.762F,
   66.8945F,
   -119.385F,
   -12.2070F,
   184.082F,
   58.5938F,
   -40.5273F,
   -30.2734F,
   -28.3203F,
   -141.357F,
   -295.898F,
   -15.3809F,
   46.3867F,
   -72.0215F,
   -253.906F,
   -316.406F,
   35.8887F,
   -91.7969F,
   -43.2129F,
   -27.5879F,
   205.322F,
   302.979F,
   256.592F,
   163.086F,
   0.488281F,
   71.5332F,
   134.521F,
   74.2188F,
   -267.578F,
   -232.666F,
   -60.5469F,
   91.7969F,
   -183.105F,
   -235.596F,
   21.2402F,
   125.977F,
   -67.1387F,
   -125.977F,
   168.213F,
   95.4590F,
   -92.5293F,
   -156.982F,
   213.867F,
   145.020F,
   -95.2148F,
   -247.314F,
   -157.471F,
   139.893F,
   -26.1230F,
   -138.672F,
   -168.213F,
   -201.660F,
   -250.244F,
   -6.59180F,
   -80.0781F,
   -49.5605F,
   210.205F,
   182.861F,
   133.789F,
   56.8848F,
   -405.273F,
   -254.639F,
   110.107F,
   26.3672F,
   -161.133F,
   -151.367F,
   104.980F,
   57.6172F,
   5.12695F,
   -96.6797F,
   -282.715F,
   -154.053F,
   334.961F,
   316.895F,
   236.084F,
   140.869F,
   274.658F,
   274.658F,
   143.799F,
   110.840F,
   -78.8574F,
   -211.182F,
   -114.014F,
   37.3535F,
   -114.258F,
   -170.654F,
   -196.289F,
   -124.268F,
   -95.7031F,
   -175.293F,
   -49.8047F,
   -8.54492F,
   -147.217F,
   -266.846F,
   -138.428F,
   -39.5508F,
   -123.291F,
   -245.117F,
   -24.9023F,
   85.4492F,
   53.4668F,
   54.6875F,
   103.271F,
   61.5234F,
   96.4355F,
   144.287F,
   148.438F,
   88.6230F,
   -182.129F,
   -23.4375F,
   91.0645F,
   41.9922F,
   41.7480F,
   72.0215F,
   174.316F,
   82.7637F,
   56.8848F,
   18.7988F,
   26.1230F,
   67.6270F,
   38.3301F,
   37.3535F,
   -121.826F,
   -86.9141F,
   377.686F,
   261.963F,
   140.625F,
   120.605F,
   -71.2891F,
   -82.7637F,
   -123.047F,
   -144.531F,
   -220.459F,
   -17.5781F,
   -151.123F,
   -117.432F,
   -389.160F,
   -272.705F,
   -138.428F,
   -62.0117F,
   -193.604F,
   -123.779F,
   -137.695F,
   -71.0449F,
   -120.117F,
   -129.883F,
   122.559F,
   136.719F,
   -93.2617F,
   104.248F,
   146.484F,
   56.1523F,
   -55.4199F,
   116.455F,
   61.2793F,
   18.3105F,
   69.5801F,
   205.566F,
   198.486F,
   116.211F,
   -319.824F,
   -325.439F,
   45.4102F,
   92.0410F,
   -143.311F,
   -223.877F,
   156.982F,
   93.0176F,
   -289.551F,
   -135.010F,
   100.342F,
   20.0195F,
   -275.146F,
   -200.195F,
   -42.4805F,
   -131.836F,
   -147.461F,
   29.0527F,
   132.568F,
   50.0488F,
   -92.7734F,
   160.400F,
   221.924F,
   138.428F,
   27.3438F,
   -72.7539F,
   -91.3086F,
   27.8320F,
   -209.229F,
   -61.2793F,
   13.6719F,
   38.8184F,
   97.9004F,
   84.2285F,
   -8.30078F,
   -34.1797F,
   -27.0996F,
   -148.193F,
   10.0098F,
   149.902F,
   86.6699F,
   -27.8320F,
   -18.7988F,
   115.723F,
   141.113F,
   13.6719F,
   354.004F,
   225.586F,
   268.066F,
   346.680F,
   180.908F,
   97.6563F,
   60.0586F,
   5.37109F,
   143.555F,
   76.4160F,
   -29.5410F,
   79.8340F,
   202.881F,
   115.234F,
   -277.832F,
   -148.438F,
   208.984F,
   134.766F,
   -302.979F,
   -261.719F,
   155.762F,
   146.484F,
   -87.4023F,
   62.0117F,
   -81.2988F,
   -73.9746F,
   -157.715F,
   180.420F,
   87.4023F,
   18.0664F,
   299.316F,
   407.959F,
   298.096F,
   207.275F,
   547.119F,
   396.484F,
   239.990F,
   155.273F,
   449.463F,
   360.596F,
   182.861F,
   93.7500F,
   85.4492F,
   64.2090F,
   21.2402F,
   31.2500F,
   -464.355F,
   -229.736F,
   -35.1563F,
   -15.6250F,
   -423.340F,
   -62.2559F,
   70.3125F,
   -7.56836F,
   -645.508F,
   -302.246F,
   89.3555F,
   57.3730F,
   -401.123F,
   -266.602F,
   -328.125F,
   -74.2188F,
   -132.080F,
   -262.451F,
   -272.461F,
   30.0293F,
   -287.598F,
   -61.5234F,
   -199.219F,
   -43.9453F,
   -248.047F,
   130.127F,
   137.939F,
   56.8848F,
   -118.896F,
   -104.980F,
   -45.8984F,
   81.5430F,
   211.670F,
   301.758F,
   130.371F,
   41.7480F,
   -388.184F,
   -392.334F,
   155.029F,
   153.809F,
   -536.133F,
   75.6836F,
   225.586F,
   100.586F,
   -575.684F,
   -80.0781F,
   233.398F,
   129.150F,
   -644.287F,
   -92.0410F,
   153.809F,
   67.8711F,
   -635.254F,
   77.3926F,
   195.068F,
   72.9980F,
   -587.402F,
   32.4707F,
   83.0078F,
   7.56836F,
   -526.367F,
   -358.398F,
   31.9824F,
   30.5176F,
   -289.063F,
   -119.629F,
   -33.9355F,
   11.2305F,
   -181.641F,
   109.131F,
   217.529F,
   137.695F,
   16.3574F,
   -110.107F,
   157.715F,
   147.461F,
   -135.010F,
   -104.736F,
   -213.867F,
   96.6797F,
   39.5508F,
   -16.1133F,
   318.604F,
   223.389F,
   116.943F,
   141.357F,
   265.625F,
   193.848F,
   109.863F,
   67.8711F,
   138.184F,
   79.1016F,
   -258.057F,
   -37.5977F,
   36.1328F,
   -43.2129F,
   -621.338F,
   41.0156F,
   261.230F,
   144.531F,
   -573.975F,
   -10.2539F,
   199.951F,
   84.2285F,
   -572.266F,
   -172.607F,
   176.025F,
   61.0352F,
   -531.006F,
   -365.479F,
   -75.4395F,
   29.7852F,
   -19.0430F,
   -17.8223F,
   29.2969F,
   42.2363F,
   -0.976563F,
   63.9648F,
   -64.2090F,
   -63.7207F,
   -105.225F,
   -15.6250F,
   -98.8770F,
   -178.711F,
   -636.963F,
   28.3203F,
   -20.2637F,
   -47.1191F,
   -372.314F,
   -230.469F,
   -116.455F,
   -177.002F,
   -124.023F,
   74.9512F,
   41.5039F,
   41.9922F,
   203.125F,
   101.807F,
   203.125F,
   167.480F,
   -54.9316F,
   43.2129F,
   218.262F,
   199.707F,
   -117.676F,
   -94.9707F,
   312.256F,
   253.662F,
   -93.5059F,
   49.0723F,
   -85.4492F,
   9.76563F,
   178.223F,
   155.029F,
   55.1758F,
   128.418F,
   122.803F,
   112.793F,
   82.5195F,
   97.1680F,
   130.615F,
   174.316F,
   9.76563F,
   -68.8477F,
   361.816F,
   359.131F,
   264.893F,
   178.467F,
   381.104F,
   261.719F,
   221.924F,
   169.189F,
   346.436F,
   312.988F,
   217.041F,
   214.600F,
   281.494F,
   177.734F,
   289.551F,
   205.078F,
   -55.1758F,
   275.879F,
   231.689F,
   168.213F,
   -120.605F,
   -240.723F,
   -379.883F,
   -31.2500F,
   -138.672F,
   -176.025F,
   -174.072F,
   -6.34766F,
   77.3926F,
   127.930F,
   17.0898F,
   32.9590F,
   -98.8770F,
   -211.182F,
   -431.152F,
   -159.180F,
   -42.4805F,
   -195.557F,
   216.064F,
   188.721F,
   -37.3535F,
   -22.2168F,
   268.311F,
   183.350F,
   -123.535F,
   -280.518F,
   208.252F,
   157.715F,
   58.8379F,
   190.918F,
   126.709F,
   131.592F,
   452.393F,
   415.039F,
   268.799F,
   166.992F,
   -304.932F,
   -362.793F,
   -113.281F,
   45.8984F,
   -218.018F,
   -343.994F,
   -320.313F,
   -83.2520F,
   -32.9590F,
   106.934F,
   -42.7246F,
   4.39453F,
   271.240F,
   238.281F,
   77.8809F,
   50.7813F,
   -349.121F,
   -431.641F,
   20.2637F,
   111.816F,
   -129.395F,
   -244.141F,
   74.9512F,
   31.4941F,
   -205.078F,
   -3.66211F,
   -7.08008F,
   -86.9141F,
   -222.412F,
   -225.586F,
   -280.029F,
   -59.0820F,
   -29.0527F,
   -128.906F,
   31.0059F,
   -32.4707F,
   -185.791F,
   -186.768F,
   46.3867F,
   -20.2637F,
   -76.9043F,
   218.506F,
   127.441F,
   56.3965F,
   -54.1992F,
   24.9023F,
   -15.3809F,
   -104.492F,
   77.1484F,
   170.654F,
   92.5293F,
   17.0898F,
   6.10352F,
   174.805F,
   76.6602F,
   -26.3672F,
   123.779F,
   213.379F,
   138.184F,
   58.1055F,
   26.3672F,
   229.736F,
   126.709F,
   47.6074F,
   103.760F,
   -14.6484F,
   -104.248F,
   62.7441F,
   33.9355F,
   -25.1465F,
   -153.809F,
   108.887F,
   81.5430F,
   90.3320F,
   100.586F,
   11.7188F,
   -41.9922F,
   -168.457F,
   -69.0918F,
   135.986F,
   45.6543F,
   -69.8242F,
   38.5742F,
   117.920F,
   34.1797F,
   65.9180F,
   -83.9844F,
   -154.053F,
   225.586F,
   141.357F,
   -28.3203F,
   32.2266F,
   34.6680F,
   113.770F,
   -16.6016F,
   -15.6250F,
   56.1523F,
   -35.4004F,
   -73.7305F,
   -132.324F,
   -196.045F,
   -222.656F,
   248.535F,
   179.932F,
   -188.721F,
   247.803F,
   153.809F,
   72.5098F,
   -633.789F,
   23.1934F,
   108.643F,
   82.0313F,
   -518.066F,
   119.873F,
   124.512F,
   46.6309F,
   -305.908F,
   39.3066F,
   -0.488281F,
   -79.1016F,
   -354.004F,
   -154.541F,
   -173.828F,
   -25.6348F,
   -205.566F,
   -62.0117F,
   -100.342F,
   24.4141F,
   -156.250F,
   -70.8008F,
   246.582F,
   186.279F,
   -158.691F,
   76.4160F,
   285.400F,
   178.223F,
   34.1797F,
   123.291F,
   251.465F,
   187.012F,
   188.477F,
   70.0684F,
   260.498F,
   200.928F,
   120.850F,
   182.861F,
   74.4629F,
   78.8574F,
   -40.0391F,
   112.793F,
   19.0430F,
   97.4121F,
   -83.4961F,
   -213.379F,
   16.8457F,
   145.752F,
   -3.90625F,
   151.367F,
   151.611F,
   82.2754F,
   -33.6914F,
   -108.398F,
   -64.6973F,
   53.2227F,
   20.5078F,
   -109.863F,
   232.666F,
   162.598F,
   -54.1992F,
   -196.045F,
   132.080F,
   147.461F,
   -224.854F,
   -335.938F,
   59.5703F,
   28.3203F,
   -205.322F,
   -176.514F,
   153.809F,
   143.555F,
   34.1797F,
   161.865F,
   71.7773F,
   89.8438F,
   228.271F,
   255.371F,
   215.088F,
   185.303F,
   426.270F,
   357.422F,
   223.633F,
   153.320F,
   106.445F,
   235.107F,
   68.6035F,
   0.244141F,
   -29.0527F,
   18.0664F,
   132.324F,
   52.0020F,
   0.244141F,
   -138.428F,
   73.4863F,
   58.8379F,
   63.4766F,
   106.201F,
   54.1992F,
   96.6797F,
   228.516F,
   233.643F,
   270.508F,
   171.631F,
   124.512F,
   123.535F,
   197.266F,
   116.699F,
   146.729F,
   169.434F,
   234.375F,
   151.367F,
   237.305F,
   180.908F,
   239.258F,
   146.484F,
   203.613F,
   175.049F,
   187.256F,
   166.992F,
   156.982F,
   237.305F,
   228.271F,
   155.762F,
   122.314F,
   161.377F,
   175.781F,
   207.764F,
   -25.6348F,
   -154.297F,
   -73.9746F,
   -28.5645F,
   -104.736F,
   31.7383F,
   192.627F,
   107.910F,
   -127.441F,
   -45.8984F,
   171.875F,
   91.0645F,
   -185.303F,
   10.2539F,
   198.730F,
   127.686F,
   -129.639F,
   -277.588F,
   91.0645F,
   141.113F,
   -166.504F,
   -293.701F,
   -111.084F,
   69.5801F,
   -283.936F,
   -385.010F,
   -268.066F,
   10.7422F,
   19.7754F,
   -20.0195F,
   173.828F,
   88.6230F,
   116.455F,
   60.0586F,
   232.910F,
   151.855F,
   391.602F,
   395.996F,
   311.768F,
   217.529F,
   343.994F,
   209.717F,
   225.586F,
   217.773F,
   188.965F,
   254.150F,
   231.201F,
   278.809F,
   9.76563F,
   -133.301F,
   -18.3105F,
   70.3125F,
   -150.391F,
   -25.8789F,
   -170.166F,
   -6.34766F,
   -41.2598F,
   -39.0625F,
   -217.529F,
   -180.420F,
   -68.1152F,
   -93.7500F,
   -251.221F,
   -85.4492F,
   434.814F,
   319.336F,
   255.371F,
   199.219F,
   385.742F,
   374.268F,
   359.375F,
   287.598F,
   367.432F,
   262.695F,
   296.875F,
   219.482F,
   217.285F,
   220.703F,
   137.695F,
   159.668F,
   224.609F,
   168.945F,
   249.268F,
   208.984F,
   -120.361F,
   32.2266F,
   43.2129F,
   123.291F,
   17.3340F,
   47.6074F,
   -6.83594F,
   23.6816F,
   111.328F,
   85.6934F,
   -40.0391F,
   21.4844F,
   107.178F,
   67.8711F,
   -9.76563F,
   85.4492F,
   340.576F,
   231.689F,
   57.1289F,
   -23.1934F,
   -196.533F,
   -115.234F,
   9.27734F,
   -39.7949F,
   89.5996F,
   -23.9258F,
   119.385F,
   127.686F,
   250.244F,
   287.598F,
   295.898F,
   221.191F,
   77.8809F,
   320.801F,
   198.730F,
   112.549F,
   -30.0293F,
   -132.568F,
   -196.289F,
   109.131F,
   -182.617F,
   -79.1016F,
   -218.994F,
   -275.146F,
   -179.932F,
   -122.314F,
   -192.627F,
   -174.072F,
   174.561F,
   189.697F,
   302.490F,
   225.098F,
   475.830F,
   473.389F,
   333.984F,
   211.182F,
   178.223F,
   214.844F,
   185.059F,
   94.7266F,
   -212.646F,
   110.840F,
   4.15039F,
   -61.2793F,
   -93.0176F,
   -197.754F,
   -386.475F,
   58.3496F,
   -127.197F,
   -235.840F,
   -193.359F,
   63.2324F,
   -217.285F,
   -331.543F,
   -187.988F,
   -17.8223F,
   40.5273F,
   85.2051F,
   -51.7578F,
   78.8574F,
   -205.078F,
   -73.4863F,
   115.479F,
   106.201F,
   -165.771F,
   -113.281F,
   177.734F,
   85.6934F,
   -38.0859F,
   -48.5840F,
   162.842F,
   105.469F,
   7.08008F,
   -61.5234F,
   101.318F,
   117.188F,
   -178.467F,
   -92.5293F,
   35.4004F,
   136.475F,
   -128.906F,
   -154.053F,
   -282.715F,
   -38.8184F,
   108.643F,
   66.6504F,
   30.0293F,
   156.006F,
   91.0645F,
   -30.7617F,
   195.313F,
   138.672F,
   20.5078F,
   -39.5508F,
   175.781F,
   173.828F,
   -202.637F,
   -130.859F,
   -45.1660F,
   54.1992F,
   99.6094F,
   110.352F,
   122.314F,
   188.232F,
   -218.994F,
   -330.811F,
   -16.3574F,
   107.910F,
   -193.359F,
   -343.262F,
   138.184F,
   146.973F,
   40.7715F,
   -79.5898F,
   124.268F,
   80.5664F,
   -23.1934F,
   -152.832F,
   -178.223F,
   -83.9844F,
   407.227F,
   297.119F,
   190.186F,
   111.084F,
   321.289F,
   202.148F,
   142.578F,
   175.537F,
   98.6328F,
   -7.56836F,
   247.314F,
   192.627F,
   21.7285F,
   26.1230F,
   217.529F,
   134.033F,
   212.646F,
   385.986F,
   223.877F,
   163.818F,
   211.426F,
   361.084F,
   314.697F,
   208.496F,
   95.4590F,
   260.742F,
   273.926F,
   198.242F,
   19.0430F,
   -137.207F,
   84.2285F,
   137.451F,
   104.736F,
   -25.1465F,
   101.807F,
   192.139F,
   -29.7852F,
   -106.689F,
   100.342F,
   192.383F,
   -222.900F,
   -101.807F,
   146.973F,
   184.082F,
   -55.1758F,
   -3.90625F,
   36.8652F,
   185.547F,
   -170.898F,
   28.8086F,
   -25.3906F,
   -3.41797F,
   -275.391F,
   11.7188F,
   69.3359F,
   95.9473F,
   -95.2148F,
   -102.295F,
   -156.006F,
   -28.3203F,
   -222.168F,
   74.7070F,
   77.1484F,
   -3.17383F,
   294.678F,
   240.234F,
   200.439F,
   163.330F,
   -291.748F,
   -169.189F,
   34.1797F,
   -52.0020F,
   -215.820F,
   -101.563F,
   -48.5840F,
   -136.230F,
   -150.391F,
   59.8145F,
   -98.6328F,
   -162.109F,
   63.9648F,
   13.6719F,
   -150.635F,
   -176.758F,
   -20.7520F,
   -119.873F,
   -78.1250F,
   -160.156F,
   -139.160F,
   -202.881F,
   -31.4941F,
   -128.906F,
   -367.676F,
   -15.3809F,
   -89.5996F,
   -93.9941F,
   -87.4023F,
   -78.3691F,
   0.976563F,
   12.4512F,
   -89.3555F,
   -52.2461F,
   77.8809F,
   124.756F,
   35.6445F,
   163.818F,
   -4.15039F,
   -71.0449F,
   -26.8555F,
   113.281F,
   -33.9355F,
   -121.094F,
   -49.3164F,
   53.7109F,
   -76.1719F,
   -154.053F,
   -161.133F,
   -17.8223F,
   -159.912F,
   -200.195F,
   -161.621F,
   -159.424F,
   -314.453F,
   -209.229F,
   -104.980F,
   -232.666F,
   -234.131F,
   -64.4531F,
   -11.9629F,
   -114.258F,
   -17.5781F,
   -93.0176F,
   -85.4492F,
   -137.451F,
   -47.1191F,
   -99.3652F,
   13.4277F,
   -99.6094F,
   -196.045F,
   2.68555F,
   -75.4395F,
   158.447F,
   45.8984F,
   -48.3398F,
   -125.000F,
   112.549F,
   -19.2871F,
   -111.816F,
   -321.777F,
   -64.2090F,
   -32.7148F,
   -127.686F,
   -404.541F,
   -106.201F,
   -120.850F,
   -186.768F,
   13.9160F,
   -84.7168F,
   -101.074F,
   105.957F,
   -278.564F,
   -59.0820F,
   -162.109F,
   -209.229F,
   8.30078F,
   -16.6016F,
   -172.607F,
   -82.5195F

};
#define MR515_3_SIZE  128

/* third codebook for MR475, MR515 */
static const Float32 mr515_3_lsf[] =
{
   102.295F,
   39.7949F,
   -7.32422F,
   -63.9648F,
   -111.084F,
   -192.627F,
   -349.121F,
   -176.025F,
   245.605F,
   162.109F,
   65.6738F,
   6.10352F,
   151.123F,
   63.4766F,
   44.6777F,
   23.4375F,
   -236.328F,
   -331.543F,
   -94.7266F,
   32.9590F,
   -169.189F,
   203.857F,
   111.328F,
   37.5977F,
   269.775F,
   171.631F,
   138.916F,
   88.6230F,
   396.729F,
   323.730F,
   240.479F,
   182.617F,
   -53.7109F,
   53.4668F,
   18.5547F,
   -50.7813F,
   -355.225F,
   -405.762F,
   11.9629F,
   36.3770F,
   -235.352F,
   -41.9922F,
   -183.594F,
   -82.0313F,
   152.588F,
   51.0254F,
   -61.0352F,
   -16.1133F,
   -248.291F,
   -204.590F,
   -0.488281F,
   77.3926F,
   -529.297F,
   -362.549F,
   -33.6914F,
   30.0293F,
   -458.008F,
   -512.451F,
   -127.197F,
   20.7520F,
   -236.084F,
   -89.3555F,
   -169.678F,
   -215.088F,
   -224.854F,
   -246.826F,
   -186.279F,
   -231.689F,
   -30.2734F,
   -62.5000F,
   -85.9375F,
   -161.133F,
   43.4570F,
   113.037F,
   86.4258F,
   74.2188F,
   -425.781F,
   -144.287F,
   -68.8477F,
   19.2871F,
   -549.072F,
   42.7246F,
   211.670F,
   121.826F,
   -33.6914F,
   -43.9453F,
   -44.1895F,
   -5.12695F,
   -559.326F,
   -302.979F,
   -112.305F,
   -126.953F,
   -188.232F,
   110.107F,
   -2.44141F,
   -75.1953F,
   66.1621F,
   -15.8691F,
   0.976563F,
   52.2461F,
   -68.1152F,
   -106.201F,
   -10.4980F,
   -84.9609F,
   -163.574F,
   8.54492F,
   -15.8691F,
   -51.5137F,
   196.777F,
   130.615F,
   20.7520F,
   72.5098F,
   13.9160F,
   58.3496F,
   176.270F,
   120.361F,
   54.9316F,
   161.377F,
   205.078F,
   133.545F,
   -131.836F,
   -91.7969F,
   3.41797F,
   85.2051F,
   114.502F,
   176.025F,
   80.8105F,
   39.5508F,
   -132.813F,
   -183.594F,
   -15.1367F,
   -2.44141F,
   97.1680F,
   -21.4844F,
   176.758F,
   171.143F,
   -4.63867F,
   -130.127F,
   -22.9492F,
   146.729F,
   33.2031F,
   -17.3340F,
   -166.260F,
   -182.373F,
   -40.5273F,
   -83.9844F,
   63.7207F,
   -12.2070F,
   39.3066F,
   -12.6953F,
   118.408F,
   82.2754F,
   -408.936F,
   12.2070F,
   46.3867F,
   -22.7051F,
   -557.129F,
   -56.3965F,
   -47.3633F,
   -20.0195F,
   -23.1934F,
   -145.264F,
   -37.5977F,
   31.2500F,
   218.262F,
   122.314F,
   143.555F,
   111.572F,
   -84.2285F,
   50.2930F,
   29.7852F,
   26.8555F,
   -154.053F,
   -55.4199F,
   -138.916F,
   0.732422F,
   99.6094F,
   58.3496F,
   96.9238F,
   55.1758F,
   -48.0957F,
   -0.488281F,
   31.2500F,
   119.873F,
   312.744F,
   220.703F,
   71.2891F,
   52.4902F,
   131.348F,
   74.7070F,
   63.2324F,
   124.268F,
   -165.283F,
   -255.615F,
   3.17383F,
   78.3691F,
   -165.771F,
   -143.555F,
   -87.4023F,
   -51.7578F,
   -136.230F,
   59.3262F,
   157.715F,
   116.943F,
   118.652F,
   83.4961F,
   154.785F,
   129.883F,
   26.1230F,
   195.801F,
   80.8105F,
   33.2031F,
   -27.3438F,
   -97.1680F,
   -251.709F,
   -69.8242F,
   -79.5898F,
   -172.119F,
   70.3125F,
   66.4063F,
   317.139F,
   279.297F,
   287.598F,
   209.961F,
   -103.271F,
   29.5410F,
   -93.9941F,
   -36.1328F,
   -72.0215F,
   -73.7305F,
   -203.613F,
   -199.951F,
   3.90625F,
   -5.85938F,
   -49.0723F,
   -116.211F,
   135.498F,
   22.2168F,
   -59.8145F,
   71.7773F,
   -9.27734F,
   -92.5293F,
   -234.863F,
   -298.096F,
   -290.771F,
   -370.605F,
   -66.6504F,
   -96.4355F,
   -95.2148F,
   -247.314F,
   -157.471F,
   139.893F,
   -449.951F,
   -251.465F,
   123.291F,
   114.258F,
   181.641F,
   231.201F,
   148.682F,
   120.361F,
   -168.213F,
   -286.133F,
   -153.320F,
   -32.9590F,
   -250.488F,
   47.6074F,
   100.342F,
   47.8516F,
   386.230F,
   280.029F,
   140.381F,
   82.2754F,
   -302.490F,
   -189.697F,
   -158.203F,
   -34.6680F,
   145.264F,
   201.416F,
   236.084F,
   179.443F,
   -294.434F,
   -236.816F,
   -19.7754F,
   -83.4961F,
   -181.885F,
   3.17383F,
   -17.5781F,
   91.5527F,
   110.840F,
   4.63867F,
   343.506F,
   224.854F,
   -402.100F,
   -41.9922F,
   210.205F,
   137.207F,
   226.563F,
   375.244F,
   259.521F,
   180.664F,
   -603.516F,
   -232.422F,
   64.4531F,
   20.0195F,
   -122.559F,
   -235.596F,
   -325.684F,
   30.0293F,
   211.670F,
   301.758F,
   130.371F,
   41.7480F,
   -566.406F,
   -112.305F,
   190.430F,
   88.6230F,
   -290.527F,
   -150.635F,
   61.5234F,
   -14.8926F,
   -42.4805F,
   8.30078F,
   246.826F,
   192.383F,
   -569.580F,
   60.3027F,
   103.271F,
   37.3535F,
   -3.90625F,
   -86.6699F,
   63.9648F,
   109.619F,
   -384.766F,
   -261.963F,
   -132.813F,
   -90.5762F,
   -150.146F,
   -74.4629F,
   256.592F,
   196.533F,
   167.725F,
   128.906F,
   1.46484F,
   -44.4336F,
   228.271F,
   213.623F,
   244.629F,
   197.510F,
   48.5840F,
   62.7441F,
   30.7617F,
   18.5547F,
   -142.578F,
   -277.832F,
   146.240F,
   135.742F,
   -269.775F,
   -339.600F,
   -388.428F,
   -126.709F,
   -238.525F,
   -323.486F,
   26.3672F,
   84.7168F,
   -176.270F,
   -238.037F,
   89.1113F,
   24.6582F,
   -35.4004F,
   166.260F,
   60.7910F,
   -37.3535F,
   0.000000F,
   -81.5430F,
   -139.160F,
   38.8184F,
   100.586F,
   69.5801F,
   -82.0313F,
   -150.635F,
   -232.666F,
   -235.840F,
   216.553F,
   168.213F,
   -305.420F,
   20.5078F,
   -45.1660F,
   -97.1680F,
   -144.531F,
   105.713F,
   254.883F,
   159.424F,
   20.7520F,
   80.3223F,
   -9.76563F,
   88.1348F,
   -105.713F,
   -172.119F,
   113.770F,
   140.137F,
   -37.5977F,
   159.668F,
   144.531F,
   70.8008F,
   -40.7715F,
   17.5781F,
   85.2051F,
   42.7246F,
   164.551F,
   72.5098F,
   238.525F,
   175.781F,
   301.514F,
   293.945F,
   184.814F,
   119.141F,
   -97.6563F,
   -65.6738F,
   131.348F,
   90.8203F,
   -329.590F,
   -338.623F,
   -291.504F,
   -22.2168F,
   308.105F,
   213.867F,
   189.209F,
   170.898F,
   -146.240F,
   -9.27734F,
   -104.980F,
   -176.270F,
   482.422F,
   397.949F,
   241.943F,
   148.438F,
   27.0996F,
   67.3828F,
   -55.1758F,
   -23.4375F,
   -231.201F,
   -94.7266F,
   -2.68555F,
   -1.70898F,
   -73.9746F,
   -129.639F,
   -204.834F,
   82.5195F,
   423.340F,
   417.480F,
   343.018F,
   247.314F,
   -125.977F,
   -208.740F,
   -157.471F,
   51.2695F,
   -167.969F,
   -101.563F,
   125.244F,
   56.1523F,
   -200.684F,
   -155.518F,
   -279.785F,
   -78.1250F,
   -232.422F,
   -160.645F,
   -169.434F,
   44.6777F,
   -27.8320F,
   -152.100F,
   199.707F,
   164.551F,
   -46.6309F,
   -49.8047F,
   178.467F,
   155.029F,
   12.4512F,
   298.096F,
   215.576F,
   140.625F,
   -232.910F,
   -105.225F,
   201.660F,
   145.996F,
   -83.4961F,
   -184.326F,
   -219.727F,
   -99.3652F,
   -274.902F,
   -86.4258F,
   -50.2930F,
   -125.000F,
   -133.545F,
   -197.754F,
   -87.1582F,
   -151.367F,
   16.1133F,
   125.732F,
   -17.8223F,
   -100.098F,
   -212.891F,
   -230.713F,
   -352.539F,
   -299.561F,
   46.6309F,
   -4.15039F,
   -132.813F,
   -56.3965F,
   -375.977F,
   -132.813F,
   -219.971F,
   -216.309F

};
#define MR795_1_SIZE  512

/* first codebook for MR795 */
static const Float32 mr795_1_lsf[] =
{
   -217.285F,
   -378.418F,
   -620.361F,
   -199.951F,
   -236.816F,
   42.7246F,
   -201.660F,
   -301.270F,
   -186.035F,
   -146.240F,
   -5.37109F,
   154.785F,
   -197.998F,
   -240.967F,
   -220.215F,
   -78.8574F,
   49.5605F,
   6.34766F,
   -93.5059F,
   -57.3730F,
   -190.674F,
   -97.4121F,
   308.105F,
   221.191F,
   -227.539F,
   -341.553F,
   -336.914F,
   -152.344F,
   22.7051F,
   21.2402F,
   -101.074F,
   -131.592F,
   -168.701F,
   9.03320F,
   154.541F,
   124.512F,
   -94.4824F,
   -116.211F,
   -324.707F,
   97.4121F,
   16.1133F,
   64.2090F,
   -99.3652F,
   -11.9629F,
   -81.7871F,
   -101.807F,
   254.150F,
   455.322F,
   -190.186F,
   -265.869F,
   -351.563F,
   -182.129F,
   -209.473F,
   203.125F,
   -141.846F,
   -185.303F,
   -90.5762F,
   -164.307F,
   -123.535F,
   509.766F,
   -136.719F,
   -154.785F,
   -287.842F,
   66.1621F,
   58.8379F,
   3.41797F,
   -106.934F,
   -59.5703F,
   -96.9238F,
   113.037F,
   293.457F,
   255.615F,
   -147.949F,
   -194.580F,
   -351.074F,
   -12.4512F,
   -78.8574F,
   117.432F,
   -54.6875F,
   -142.578F,
   -128.662F,
   120.605F,
   215.088F,
   166.504F,
   -105.713F,
   -74.7070F,
   -244.629F,
   135.254F,
   160.889F,
   54.1992F,
   41.7480F,
   -39.0625F,
   -86.1816F,
   166.260F,
   438.965F,
   382.080F,
   -208.008F,
   -288.330F,
   -413.818F,
   -82.0313F,
   -162.598F,
   27.8320F,
   -141.846F,
   -184.570F,
   -181.641F,
   -47.6074F,
   91.5527F,
   121.338F,
   -113.525F,
   -196.289F,
   -268.066F,
   37.5977F,
   68.8477F,
   -31.9824F,
   -12.2070F,
   -46.6309F,
   -175.537F,
   78.8574F,
   178.711F,
   376.465F,
   -176.270F,
   -199.951F,
   -342.773F,
   25.6348F,
   -61.0352F,
   45.1660F,
   -43.4570F,
   -122.559F,
   -181.152F,
   78.3691F,
   124.512F,
   271.240F,
   -78.8574F,
   -138.428F,
   -235.840F,
   31.0059F,
   118.164F,
   82.5195F,
   -39.0625F,
   12.6953F,
   -82.5195F,
   178.711F,
   333.740F,
   379.395F,
   -152.832F,
   -195.801F,
   -414.063F,
   -69.8242F,
   -143.066F,
   165.039F,
   -169.678F,
   -83.7402F,
   -90.3320F,
   -119.629F,
   72.0215F,
   462.158F,
   -153.809F,
   -140.137F,
   -247.559F,
   -19.5313F,
   157.471F,
   -16.8457F,
   -1.46484F,
   -77.6367F,
   -88.8672F,
   190.918F,
   354.004F,
   253.418F,
   -76.4160F,
   -178.955F,
   -340.576F,
   29.2969F,
   14.6484F,
   116.455F,
   -64.4531F,
   -142.822F,
   -30.0293F,
   173.584F,
   303.955F,
   154.541F,
   -22.2168F,
   -86.6699F,
   -248.047F,
   188.232F,
   185.059F,
   63.7207F,
   61.7676F,
   19.7754F,
   -115.723F,
   227.051F,
   540.771F,
   419.922F,
   -197.266F,
   -268.311F,
   -469.971F,
   -136.719F,
   -190.918F,
   41.2598F,
   -196.289F,
   -262.207F,
   -45.8984F,
   -152.832F,
   -13.4277F,
   343.018F,
   -169.434F,
   -174.805F,
   -291.504F,
   -161.133F,
   86.4258F,
   80.3223F,
   -125.488F,
   -13.4277F,
   -132.568F,
   89.3555F,
   252.197F,
   288.574F,
   -160.645F,
   -234.131F,
   -331.299F,
   -13.4277F,
   -44.9219F,
   22.7051F,
   -147.705F,
   -69.8242F,
   -161.621F,
   98.6328F,
   109.619F,
   201.904F,
   -69.8242F,
   -85.4492F,
   -308.350F,
   153.320F,
   74.7070F,
   55.4199F,
   -3.90625F,
   35.8887F,
   -152.100F,
   45.4102F,
   225.342F,
   523.926F,
   -164.551F,
   -217.285F,
   -392.090F,
   -108.154F,
   -55.6641F,
   82.7637F,
   -90.0879F,
   -192.871F,
   -99.8535F,
   56.3965F,
   20.9961F,
   358.643F,
   -109.375F,
   -141.846F,
   -259.033F,
   145.020F,
   109.863F,
   -43.2129F,
   -30.2734F,
   -41.5039F,
   -109.131F,
   163.818F,
   282.959F,
   342.773F,
   -116.211F,
   -162.842F,
   -368.896F,
   -18.7988F,
   -33.6914F,
   174.805F,
   -43.2129F,
   -90.8203F,
   -93.0176F,
   110.107F,
   228.027F,
   223.389F,
   -61.0352F,
   -105.469F,
   -200.684F,
   66.4063F,
   202.148F,
   108.887F,
   6.34766F,
   4.63867F,
   -7.56836F,
   170.410F,
   413.086F,
   529.297F,
   -157.715F,
   -238.525F,
   -469.727F,
   -43.7012F,
   -115.479F,
   65.4297F,
   -92.5293F,
   -181.885F,
   -168.701F,
   2.68555F,
   31.0059F,
   252.197F,
   -119.141F,
   -223.877F,
   -201.416F,
   14.8926F,
   78.8574F,
   32.9590F,
   35.8887F,
   -35.4004F,
   -167.480F,
   167.236F,
   191.895F,
   410.645F,
   -123.535F,
   -207.031F,
   -316.650F,
   8.54492F,
   21.9727F,
   54.1992F,
   -5.61523F,
   -84.4727F,
   -163.574F,
   111.084F,
   144.287F,
   314.209F,
   -49.5605F,
   -144.775F,
   -265.137F,
   159.180F,
   85.9375F,
   106.689F,
   9.52148F,
   15.3809F,
   -111.572F,
   205.322F,
   308.838F,
   513.916F,
   -126.953F,
   -215.332F,
   -386.719F,
   -80.0781F,
   -173.584F,
   346.924F,
   -145.508F,
   -83.4961F,
   -17.0898F,
   51.0254F,
   42.2363F,
   470.703F,
   -103.271F,
   -145.996F,
   -224.854F,
   102.783F,
   147.705F,
   -9.27734F,
   -0.488281F,
   -59.8145F,
   -31.0059F,
   218.750F,
   480.713F,
   277.100F,
   -92.5293F,
   -126.465F,
   -385.498F,
   42.2363F,
   28.8086F,
   183.838F,
   -13.4277F,
   -93.0176F,
   -12.6953F,
   240.479F,
   249.268F,
   183.838F,
   -0.488281F,
   -71.0449F,
   -217.529F,
   183.838F,
   242.188F,
   103.271F,
   64.4531F,
   31.9824F,
   -47.8516F,
   218.506F,
   555.176F,
   620.850F,
   -155.029F,
   -265.625F,
   -610.107F,
   -129.150F,
   -239.746F,
   128.418F,
   -186.523F,
   -202.637F,
   -133.789F,
   -106.445F,
   77.1484F,
   146.240F,
   -164.795F,
   -229.492F,
   -182.129F,
   -13.9160F,
   57.6172F,
   -2.68555F,
   -49.0723F,
   -19.7754F,
   -194.824F,
   3.90625F,
   206.299F,
   380.371F,
   -179.932F,
   -240.479F,
   -295.898F,
   -114.258F,
   4.15039F,
   70.8008F,
   -68.1152F,
   -142.578F,
   -170.898F,
   44.6777F,
   200.684F,
   172.119F,
   -64.6973F,
   -120.117F,
   -289.795F,
   102.783F,
   37.1094F,
   114.258F,
   -95.2148F,
   40.5273F,
   -65.4297F,
   9.52148F,
   378.418F,
   456.055F,
   -155.029F,
   -235.840F,
   -383.545F,
   -110.596F,
   -120.117F,
   222.168F,
   -69.3359F,
   -250.732F,
   -18.3105F,
   -44.1895F,
   -32.4707F,
   452.148F,
   -108.643F,
   -152.344F,
   -286.621F,
   102.539F,
   89.5996F,
   -11.9629F,
   -94.9707F,
   -51.7578F,
   -41.2598F,
   172.607F,
   261.963F,
   294.922F,
   -131.592F,
   -173.340F,
   -353.760F,
   20.2637F,
   -39.7949F,
   118.164F,
   -57.6172F,
   -132.568F,
   -86.6699F,
   82.5195F,
   286.865F,
   198.730F,
   -60.0586F,
   -75.4395F,
   -233.887F,
   147.949F,
   185.547F,
   14.6484F,
   40.5273F,
   -1.95313F,
   -39.7949F,
   -74.7070F,
   451.416F,
   625.732F,
   -182.373F,
   -250.244F,
   -435.303F,
   -102.295F,
   -108.887F,
   51.0254F,
   -175.293F,
   -138.184F,
   -130.371F,
   -123.535F,
   169.189F,
   209.229F,
   -113.037F,
   -170.166F,
   -264.160F,
   79.3457F,
   105.225F,
   -50.2930F,
   -3.66211F,
   -1.95313F,
   -186.279F,
   133.057F,
   224.365F,
   370.605F,
   -149.170F,
   -191.162F,
   -320.557F,
   62.5000F,
   -13.4277F,
   50.7813F,
   -40.2832F,
   -84.9609F,
   -161.621F,
   78.3691F,
   166.016F,
   227.051F,
   -79.5898F,
   -104.736F,
   -232.178F,
   118.164F,
   108.887F,
   139.160F,
   -48.0957F,
   17.5781F,
   -17.8223F,
   221.924F,
   355.225F,
   425.049F,
   -137.451F,
   -179.932F,
   -481.934F,
   -30.2734F,
   -101.563F,
   175.293F,
   -116.699F,
   -98.6328F,
   -76.6602F,
   -3.90625F,
   108.887F,
   399.414F,
   -134.521F,
   -131.104F,
   -183.105F,
   -14.1602F,
   155.762F,
   52.2461F,
   13.4277F,
   -45.1660F,
   -66.1621F,
   280.273F,
   317.627F,
   295.898F,
   -117.920F,
   -163.818F,
   -308.594F,
   28.5645F,
   69.5801F,
   132.568F,
   -49.8047F,
   -95.4590F,
   -27.0996F,
   125.244F,
   375.488F,
   208.496F,
   -27.8320F,
   -46.3867F,
   -238.770F,
   214.111F,
   145.264F,
   113.281F,
   63.4766F,
   63.4766F,
   -75.9277F,
   182.617F,
   557.373F,
   541.016F,
   -126.221F,
   -230.713F,
   -530.029F,
   -79.5898F,
   -172.852F,
   92.2852F,
   -198.242F,
   -168.701F,
   -56.6406F,
   -136.719F,
   167.725F,
   343.994F,
   -178.711F,
   -168.457F,
   -204.102F,
   -87.6465F,
   157.471F,
   94.2383F,
   -64.6973F,
   15.1367F,
   -165.527F,
   35.4004F,
   401.367F,
   294.922F,
   -135.498F,
   -241.211F,
   -301.025F,
   -19.0430F,
   3.41797F,
   27.8320F,
   -79.8340F,
   -87.4023F,
   -119.385F,
   95.7031F,
   165.283F,
   170.166F,
   -49.0723F,
   -57.6172F,
   -278.320F,
   169.189F,
   109.619F,
   43.4570F,
   -59.3262F,
   62.5000F,
   -105.713F,
   149.170F,
   338.135F,
   599.609F,
   -149.414F,
   -219.971F,
   -357.422F,
   -74.9512F,
   -4.15039F,
   121.826F,
   -76.9043F,
   -162.842F,
   -62.0117F,
   62.5000F,
   104.492F,
   357.178F,
   -118.652F,
   -103.027F,
   -257.813F,
   159.912F,
   90.3320F,
   4.39453F,
   -24.9023F,
   -45.1660F,
   -67.3828F,
   184.326F,
   385.254F,
   325.928F,
   -119.141F,
   -147.217F,
   -346.191F,
   44.4336F,
   -22.7051F,
   212.402F,
   -17.8223F,
   -111.816F,
   -84.9609F,
   203.857F,
   210.449F,
   233.643F,
   -68.8477F,
   -81.2988F,
   -182.129F,
   133.545F,
   204.834F,
   104.492F,
   66.6504F,
   -21.7285F,
   3.17383F,
   229.492F,
   416.992F,
   628.906F,
   -102.051F,
   -264.648F,
   -429.199F,
   -10.7422F,
   -87.4023F,
   63.2324F,
   -121.338F,
   -156.982F,
   -136.719F,
   24.1699F,
   135.986F,
   234.619F,
   -102.783F,
   -187.012F,
   -223.877F,
   72.0215F,
   79.5898F,
   44.9219F,
   42.7246F,
   3.66211F,
   -152.832F,
   129.883F,
   214.355F,
   483.643F,
   -108.154F,
   -187.500F,
   -311.279F,
   53.9551F,
   38.0859F,
   65.4297F,
   9.52148F,
   -88.6230F,
   -123.291F,
   169.678F,
   188.477F,
   278.320F,
   -39.5508F,
   -112.061F,
   -222.656F,
   173.096F,
   108.398F,
   160.645F,
   6.10352F,
   73.9746F,
   -76.1719F,
   309.570F,
   344.238F,
   418.701F,
   -72.5098F,
   -187.012F,
   -448.242F,
   -64.2090F,
   -26.3672F,
   261.230F,
   -99.1211F,
   -3.17383F,
   -31.4941F,
   13.9160F,
   106.934F,
   667.480F,
   -91.3086F,
   -118.896F,
   -203.857F,
   74.2188F,
   169.922F,
   40.0391F,
   25.3906F,
   -57.3730F,
   1.22070F,
   393.311F,
   463.867F,
   341.553F,
   -55.9082F,
   -142.090F,
   -323.486F,
   98.8770F,
   46.8750F,
   199.463F,
   -21.2402F,
   -106.934F,
   27.0996F,
   250.977F,
   292.725F,
   242.432F,
   16.6016F,
   -42.7246F,
   -228.027F,
   252.197F,
   272.705F,
   110.107F,
   116.699F,
   48.8281F,
   -60.5469F,
   519.287F,
   658.203F,
   498.535F,
   -203.857F,
   -322.998F,
   -520.264F,
   -195.068F,
   -168.945F,
   113.770F,
   -198.242F,
   -251.953F,
   -114.502F,
   -151.855F,
   70.3125F,
   224.609F,
   -171.143F,
   -205.322F,
   -261.230F,
   -100.342F,
   125.000F,
   1.95313F,
   -95.2148F,
   -22.2168F,
   -181.641F,
   -7.32422F,
   254.639F,
   283.447F,
   -200.684F,
   -280.273F,
   -282.227F,
   -71.7773F,
   -11.2305F,
   26.8555F,
   -100.342F,
   -91.3086F,
   -165.527F,
   52.2461F,
   129.639F,
   163.086F,
   -99.1211F,
   -102.539F,
   -291.504F,
   118.896F,
   56.6406F,
   73.9746F,
   -77.6367F,
   22.2168F,
   -115.234F,
   30.0293F,
   300.781F,
   596.924F,
   -176.270F,
   -232.422F,
   -364.990F,
   -180.176F,
   -164.795F,
   325.195F,
   -132.568F,
   -147.949F,
   -51.5137F,
   -23.1934F,
   -23.9258F,
   368.164F,
   -134.033F,
   -125.488F,
   -291.260F,
   115.479F,
   51.5137F,
   17.8223F,
   -70.3125F,
   -27.3438F,
   -94.9707F,
   131.104F,
   325.195F,
   307.129F,
   -138.428F,
   -184.326F,
   -377.197F,
   17.3340F,
   -69.0918F,
   154.297F,
   -41.5039F,
   -117.432F,
   -120.361F,
   166.260F,
   244.629F,
   199.463F,
   -86.9141F,
   -80.8105F,
   -214.111F,
   102.295F,
   172.363F,
   84.4727F,
   58.8379F,
   -8.30078F,
   -79.5898F,
   92.0410F,
   476.074F,
   459.717F,
   -177.490F,
   -262.451F,
   -396.729F,
   -56.8848F,
   -132.568F,
   28.3203F,
   -127.930F,
   -196.777F,
   -142.822F,
   -17.8223F,
   116.699F,
   177.979F,
   -70.3125F,
   -225.830F,
   -279.053F,
   42.2363F,
   109.131F,
   -12.6953F,
   16.6016F,
   -55.9082F,
   -147.949F,
   109.619F,
   129.150F,
   438.721F,
   -144.287F,
   -213.623F,
   -332.764F,
   44.6777F,
   -35.1563F,
   79.1016F,
   -25.1465F,
   -110.352F,
   -162.598F,
   152.100F,
   119.141F,
   287.109F,
   -58.1055F,
   -124.756F,
   -245.117F,
   79.5898F,
   134.766F,
   111.816F,
   33.2031F,
   26.3672F,
   -77.8809F,
   152.832F,
   327.881F,
   459.717F,
   -119.629F,
   -157.715F,
   -422.363F,
   -45.4102F,
   -109.619F,
   240.234F,
   -180.176F,
   -18.5547F,
   -41.5039F,
   -134.277F,
   184.326F,
   625.000F,
   -121.094F,
   -124.512F,
   -231.201F,
   51.2695F,
   169.434F,
   -12.6953F,
   20.5078F,
   -78.6133F,
   -48.5840F,
   266.113F,
   396.729F,
   298.828F,
   -91.7969F,
   -147.217F,
   -340.820F,
   83.7402F,
   18.0664F,
   154.297F,
   -42.7246F,
   -122.559F,
   -7.81250F,
   237.305F,
   325.195F,
   179.199F,
   12.6953F,
   -72.0215F,
   -271.729F,
   260.010F,
   224.121F,
   39.0625F,
   95.9473F,
   26.1230F,
   -96.9238F,
   296.387F,
   646.729F,
   425.049F,
   -154.297F,
   -293.213F,
   -461.670F,
   -175.537F,
   -67.6270F,
   86.1816F,
   -158.936F,
   -214.844F,
   -29.7852F,
   -51.5137F,
   51.0254F,
   326.660F,
   -137.207F,
   -174.316F,
   -258.545F,
   -50.7813F,
   94.7266F,
   38.8184F,
   -78.1250F,
   -14.8926F,
   -134.521F,
   71.5332F,
   266.602F,
   352.295F,
   -158.203F,
   -211.182F,
   -305.908F,
   -11.9629F,
   -34.9121F,
   74.4629F,
   -97.9004F,
   -55.4199F,
   -142.822F,
   136.963F,
   129.883F,
   226.318F,
   -28.5645F,
   -108.154F,
   -290.039F,
   123.779F,
   106.445F,
   71.2891F,
   -19.2871F,
   56.8848F,
   -111.816F,
   163.818F,
   250.244F,
   584.961F,
   -154.541F,
   -205.566F,
   -372.314F,
   -75.1953F,
   -69.8242F,
   156.250F,
   -91.0645F,
   -151.611F,
   -99.3652F,
   102.051F,
   61.7676F,
   318.604F,
   -76.9043F,
   -141.846F,
   -277.588F,
   139.648F,
   167.236F,
   -68.6035F,
   14.8926F,
   -16.6016F,
   -90.5762F,
   241.943F,
   268.799F,
   365.723F,
   -120.361F,
   -166.748F,
   -332.520F,
   -11.4746F,
   40.0391F,
   171.875F,
   -62.5000F,
   -76.6602F,
   -65.4297F,
   154.053F,
   231.689F,
   256.836F,
   -28.8086F,
   -84.9609F,
   -203.369F,
   16.6016F,
   288.086F,
   138.672F,
   37.1094F,
   28.5645F,
   8.30078F,
   271.729F,
   464.355F,
   546.631F,
   -146.729F,
   -234.131F,
   -416.504F,
   -34.9121F,
   -119.385F,
   117.188F,
   -81.0547F,
   -159.912F,
   -140.137F,
   13.1836F,
   86.1816F,
   291.016F,
   -112.793F,
   -159.180F,
   -194.336F,
   36.6211F,
   134.033F,
   27.3438F,
   47.6074F,
   -27.0996F,
   -125.732F,
   165.771F,
   270.508F,
   402.100F,
   -136.230F,
   -182.861F,
   -297.119F,
   -2.19727F,
   66.4063F,
   83.2520F,
   -12.9395F,
   -64.6973F,
   -130.615F,
   119.385F,
   205.811F,
   316.895F,
   -29.2969F,
   -117.676F,
   -251.953F,
   154.297F,
   132.568F,
   99.6094F,
   43.7012F,
   74.7070F,
   -128.418F,
   274.414F,
   357.422F,
   547.852F,
   -101.807F,
   -191.895F,
   -381.348F,
   -54.6875F,
   -93.7500F,
   333.008F,
   -92.0410F,
   -112.061F,
   -6.10352F,
   93.9941F,
   119.385F,
   530.762F,
   -81.0547F,
   -158.936F,
   -202.393F,
   132.813F,
   135.010F,
   14.8926F,
   5.37109F,
   -27.5879F,
   -21.7285F,
   275.391F,
   421.143F,
   372.070F,
   -52.7344F,
   -91.0645F,
   -403.564F,
   39.3066F,
   77.1484F,
   221.680F,
   -40.2832F,
   -54.1992F,
   -16.3574F,
   332.520F,
   286.865F,
   192.627F,
   17.8223F,
   -61.5234F,
   -187.256F,
   180.176F,
   227.539F,
   150.391F,
   88.3789F,
   60.0586F,
   -30.7617F,
   192.139F,
   647.949F,
   739.014F,
   -168.701F,
   -270.020F,
   -534.668F,
   -137.939F,
   -143.555F,
   127.930F,
   -144.043F,
   -239.014F,
   -119.629F,
   -64.2090F,
   96.9238F,
   239.746F,
   -140.869F,
   -204.346F,
   -230.713F,
   -5.37109F,
   106.201F,
   -11.9629F,
   -46.3867F,
   -28.8086F,
   -153.564F,
   -21.4844F,
   302.734F,
   369.385F,
   -155.273F,
   -256.592F,
   -248.779F,
   -71.0449F,
   46.1426F,
   63.2324F,
   -62.7441F,
   -114.746F,
   -153.564F,
   35.4004F,
   230.713F,
   218.262F,
   -79.5898F,
   -88.8672F,
   -267.090F,
   132.568F,
   63.4766F,
   153.809F,
   -49.3164F,
   46.1426F,
   -51.0254F,
   87.1582F,
   336.670F,
   510.498F,
   -138.916F,
   -262.451F,
   -353.760F,
   -174.316F,
   -58.3496F,
   224.365F,
   -102.539F,
   -172.119F,
   -20.5078F,
   -26.6113F,
   -27.8320F,
   587.646F,
   -100.830F,
   -129.150F,
   -287.354F,
   117.676F,
   89.8438F,
   31.9824F,
   -45.4102F,
   -17.5781F,
   -31.9824F,
   210.205F,
   306.396F,
   297.852F,
   -149.170F,
   -160.645F,
   -327.393F,
   55.4199F,
   -29.5410F,
   154.053F,
   -42.9688F,
   -119.385F,
   -53.2227F,
   181.885F,
   286.865F,
   233.643F,
   -78.3691F,
   -36.1328F,
   -228.516F,
   163.818F,
   235.840F,
   52.7344F,
   83.0078F,
   -0.732422F,
   -34.9121F,
   114.502F,
   451.172F,
   594.971F,
   -177.979F,
   -234.619F,
   -410.889F,
   -52.0020F,
   -62.0117F,
   78.3691F,
   -124.756F,
   -106.934F,
   -127.197F,
   -30.7617F,
   177.002F,
   220.459F,
   -83.0078F,
   -167.236F,
   -251.953F,
   77.1484F,
   117.188F,
   4.88281F,
   5.61523F,
   -21.7285F,
   -134.521F,
   86.1816F,
   256.592F,
   436.768F,
   -132.813F,
   -184.814F,
   -333.008F,
   72.7539F,
   -6.10352F,
   106.445F,
   -24.4141F,
   -95.7031F,
   -126.709F,
   114.014F,
   184.082F,
   263.184F,
   -51.2695F,
   -97.1680F,
   -263.184F,
   151.367F,
   160.645F,
   153.809F,
   8.05664F,
   35.8887F,
   -43.4570F,
   224.854F,
   411.865F,
   468.994F,
   -79.3457F,
   -128.906F,
   -482.910F,
   0.488281F,
   -69.5801F,
   222.168F,
   -90.5762F,
   -119.629F,
   -56.1523F,
   0.000000F,
   145.752F,
   490.723F,
   -121.094F,
   -96.4355F,
   -203.613F,
   9.03320F,
   230.713F,
   59.8145F,
   44.1895F,
   -39.0625F,
   -35.1563F,
   361.572F,
   335.205F,
   331.299F,
   -86.6699F,
   -146.729F,
   -310.059F,
   72.7539F,
   78.6133F,
   164.063F,
   -47.1191F,
   -82.0313F,
   18.7988F,
   265.869F,
   374.268F,
   225.098F,
   43.2129F,
   -9.52148F,
   -274.658F,
   243.164F,
   190.674F,
   130.859F,
   111.328F,
   89.3555F,
   -105.469F,
   345.459F,
   595.703F,
   556.396F,
   -113.770F,
   -185.059F,
   -567.627F,
   -73.9746F,
   -124.268F,
   94.4824F,
   -177.490F,
   -135.986F,
   16.1133F,
   -35.4004F,
   156.982F,
   304.688F,
   -132.813F,
   -165.039F,
   -223.633F,
   -54.9316F,
   210.449F,
   143.555F,
   -37.1094F,
   9.76563F,
   -130.127F,
   103.271F,
   347.412F,
   380.371F,
   -139.648F,
   -205.811F,
   -279.541F,
   -31.2500F,
   20.7520F,
   112.549F,
   -58.1055F,
   -62.7441F,
   -142.578F,
   147.705F,
   182.617F,
   210.205F,
   5.85938F,
   -49.3164F,
   -343.994F,
   194.580F,
   118.896F,
   73.9746F,
   -44.1895F,
   88.8672F,
   -44.4336F,
   150.391F,
   336.426F,
   718.262F,
   -120.605F,
   -208.008F,
   -351.807F,
   -71.2891F,
   14.8926F,
   198.242F,
   -20.5078F,
   -176.514F,
   -44.4336F,
   135.498F,
   129.883F,
   367.676F,
   -89.1113F,
   -120.361F,
   -258.057F,
   200.684F,
   143.555F,
   2.68555F,
   -3.41797F,
   -4.39453F,
   -56.1523F,
   244.385F,
   342.041F,
   354.248F,
   -115.723F,
   -138.916F,
   -315.430F,
   73.7305F,
   15.1367F,
   259.277F,
   -17.0898F,
   -91.7969F,
   -54.1992F,
   239.746F,
   237.793F,
   280.518F,
   -47.8516F,
   -57.1289F,
   -194.092F,
   116.943F,
   268.066F,
   121.826F,
   88.3789F,
   14.1602F,
   17.0898F,
   280.029F,
   505.127F,
   697.510F,
   -118.896F,
   -214.355F,
   -445.313F,
   17.8223F,
   -70.3125F,
   84.9609F,
   -87.4023F,
   -122.070F,
   -124.023F,
   48.5840F,
   176.025F,
   303.223F,
   -19.0430F,
   -170.166F,
   -194.092F,
   88.1348F,
   130.859F,
   47.8516F,
   91.3086F,
   26.8555F,
   -179.443F,
   206.787F,
   256.592F,
   462.891F,
   -89.3555F,
   -174.072F,
   -288.574F,
   76.9043F,
   78.1250F,
   104.736F,
   17.5781F,
   -52.4902F,
   -109.863F,
   185.303F,
   216.309F,
   332.764F,
   -7.32422F,
   -104.492F,
   -203.613F,
   210.205F,
   153.076F,
   194.336F,
   28.8086F,
   114.258F,
   -68.1152F,
   330.811F,
   459.717F,
   462.158F,
   -45.8984F,
   -156.738F,
   -393.555F,
   15.3809F,
   -42.7246F,
   292.480F,
   -102.051F,
   -51.5137F,
   12.4512F,
   101.074F,
   143.311F,
   635.010F,
   -57.1289F,
   -135.986F,
   -209.473F,
   103.516F,
   217.041F,
   54.1992F,
   33.2031F,
   -24.6582F,
   20.2637F,
   344.971F,
   556.152F,
   337.646F,
   -20.5078F,
   -108.643F,
   -339.111F,
   101.074F,
   76.4160F,
   255.127F,
   7.08008F,
   -83.7402F,
   15.8691F,
   378.906F,
   402.100F,
   239.258F,
   44.6777F,
   -22.2168F,
   -202.393F,
   310.791F,
   344.971F,
   87.8906F,
   135.010F,
   66.4063F,
   -26.1230F,
   387.451F,
   768.799F,
   635.498F

};

/*
 * Quantization tables for split_MQ of 2 sets of LSFs in a 20 ms frame.
 */
#define DICO1_SIZE_5 128
#define DICO2_SIZE_5 256
#define DICO3_SIZE_5 256
#define DICO4_SIZE_5 256
#define DICO5_SIZE_5 64
static const Float32 dico1_lsf_5[DICO1_SIZE_5 * 4] =
{
   -110.107F,
   -260.010F,
   -129.150F,
   -318.604F,
   -109.863F,
   -184.570F,
   -121.338F,
   -210.693F,
   -93.7500F,
   -151.123F,
   -100.830F,
   -163.330F,
   -77.3926F,
   -131.348F,
   -80.8105F,
   -135.742F,
   -101.074F,
   -124.023F,
   -103.516F,
   -92.2852F,
   -66.8945F,
   -79.1016F,
   -105.957F,
   -149.902F,
   -55.1758F,
   -122.070F,
   -56.6406F,
   -125.488F,
   -64.2090F,
   -92.0410F,
   -72.7539F,
   -100.098F,
   -36.8652F,
   -173.340F,
   -42.4805F,
   -199.707F,
   -36.3770F,
   -100.586F,
   -38.0859F,
   -104.736F,
   -70.3125F,
   -112.793F,
   -45.4102F,
   -49.5605F,
   -41.5039F,
   -73.7305F,
   -46.6309F,
   -78.3691F,
   -31.9824F,
   -35.8887F,
   -72.5098F,
   -96.4355F,
   -55.6641F,
   -52.2461F,
   -59.8145F,
   -46.8750F,
   -16.3574F,
   -77.1484F,
   -17.3340F,
   -79.8340F,
   -25.3906F,
   -50.0488F,
   -22.9492F,
   -44.6777F,
   -34.9121F,
   -9.27734F,
   -47.1191F,
   -23.1934F,
   3.90625F,
   -18.5547F,
   -30.2734F,
   -60.5469F,
   5.61523F,
   -57.8613F,
   5.85938F,
   -59.5703F,
   4.39453F,
   -33.2031F,
   10.7422F,
   -27.0996F,
   -8.05664F,
   -5.85938F,
   -6.10352F,
   0.000000F,
   36.3770F,
   4.63867F,
   5.61523F,
   -34.9121F,
   38.5742F,
   -41.2598F,
   42.4805F,
   -44.1895F,
   32.4707F,
   -13.4277F,
   40.2832F,
   -6.34766F,
   27.0996F,
   20.5078F,
   23.9258F,
   18.3105F,
   21.2402F,
   44.6777F,
   -28.0762F,
   -2.68555F,
   -1.95313F,
   31.7383F,
   2.68555F,
   41.5039F,
   62.0117F,
   18.7988F,
   50.0488F,
   4.15039F,
   44.6777F,
   27.3438F,
   63.9648F,
   47.3633F,
   49.3164F,
   70.0684F,
   23.1934F,
   46.1426F,
   -10.2539F,
   -25.6348F,
   57.1289F,
   43.7012F,
   9.52148F,
   45.4102F,
   39.7949F,
   84.2285F,
   81.0547F,
   48.5840F,
   72.9980F,
   39.3066F,
   -13.1836F,
   69.5801F,
   -19.0430F,
   68.6035F,
   -32.4707F,
   34.4238F,
   -44.4336F,
   27.0996F,
   60.7910F,
   83.2520F,
   66.1621F,
   88.8672F,
   22.7051F,
   98.3887F,
   18.3105F,
   95.4590F,
   22.4609F,
   124.512F,
   -33.6914F,
   53.7109F,
   -45.1660F,
   -7.08008F,
   -8.30078F,
   88.1348F,
   -28.0762F,
   78.1250F,
   0.732422F,
   135.254F,
   24.1699F,
   69.8242F,
   53.2227F,
   144.287F,
   -59.8145F,
   99.1211F,
   -65.4297F,
   110.596F,
   0.000000F,
   141.602F,
   6.10352F,
   147.949F,
   67.1387F,
   129.883F,
   36.1328F,
   109.863F,
   -17.8223F,
   180.420F,
   -69.5801F,
   126.465F,
   -70.3125F,
   22.9492F,
   -49.5605F,
   164.551F,
   -34.1797F,
   -18.0664F,
   50.0488F,
   174.316F,
   -27.8320F,
   72.9980F,
   42.9688F,
   225.342F,
   44.4336F,
   135.986F,
   58.5938F,
   172.119F,
   -3.90625F,
   125.244F,
   118.408F,
   144.775F,
   71.5332F,
   93.7500F,
   110.107F,
   150.635F,
   -9.27734F,
   12.2070F,
   137.451F,
   129.150F,
   73.9746F,
   51.0254F,
   112.061F,
   88.6230F,
   105.713F,
   110.352F,
   109.863F,
   110.840F,
   89.5996F,
   147.949F,
   116.455F,
   180.908F,
   105.469F,
   86.1816F,
   89.8438F,
   65.1855F,
   88.1348F,
   174.805F,
   66.6504F,
   142.334F,
   110.596F,
   40.5273F,
   124.512F,
   41.9922F,
   49.0723F,
   153.564F,
   66.8945F,
   46.6309F,
   138.672F,
   156.006F,
   73.7305F,
   72.7539F,
   154.785F,
   94.4824F,
   156.982F,
   85.4492F,
   143.311F,
   136.719F,
   149.414F,
   137.939F,
   146.484F,
   192.383F,
   118.896F,
   164.063F,
   125.000F,
   247.803F,
   78.3691F,
   81.2988F,
   87.1582F,
   208.496F,
   -30.5176F,
   100.830F,
   115.723F,
   173.828F,
   4.15039F,
   -36.8652F,
   137.695F,
   69.5801F,
   65.9180F,
   -58.8379F,
   237.061F,
   217.041F,
   119.385F,
   53.7109F,
   124.512F,
   218.750F,
   134.033F,
   225.586F,
   79.8340F,
   201.416F,
   70.8008F,
   222.412F,
   131.836F,
   270.508F,
   38.5742F,
   196.533F,
   48.5840F,
   233.643F,
   124.756F,
   178.223F,
   24.4141F,
   213.379F,
   3.17383F,
   193.115F,
   106.201F,
   154.297F,
   165.039F,
   237.305F,
   60.7910F,
   219.727F,
   114.014F,
   297.363F,
   190.674F,
   262.207F,
   142.822F,
   191.650F,
   -5.61523F,
   163.330F,
   65.1855F,
   254.639F,
   151.123F,
   264.648F,
   150.146F,
   279.541F,
   151.855F,
   220.947F,
   223.633F,
   256.104F,
   19.5313F,
   80.8105F,
   142.578F,
   262.451F,
   21.7285F,
   156.006F,
   241.211F,
   234.619F,
   187.988F,
   175.781F,
   194.824F,
   170.654F,
   120.117F,
   109.131F,
   219.482F,
   153.076F,
   66.1621F,
   290.039F,
   177.002F,
   325.439F,
   21.2402F,
   147.217F,
   203.125F,
   391.357F,
   150.391F,
   275.146F,
   217.285F,
   367.432F,
   244.141F,
   282.227F,
   211.426F,
   246.338F,
   242.920F,
   201.904F,
   280.518F,
   209.473F,
   199.463F,
   354.004F,
   188.721F,
   322.266F,
   122.070F,
   339.111F,
   76.1719F,
   281.494F,
   -4.88281F,
   264.648F,
   15.6250F,
   313.232F,
   0.488281F,
   286.133F,
   97.4121F,
   456.299F,
   125.488F,
   416.504F,
   122.559F,
   399.414F,
   216.309F,
   371.582F,
   101.563F,
   146.484F,
   276.123F,
   329.590F,
   311.279F,
   339.355F,
   217.041F,
   438.232F,
   223.145F,
   431.152F,
   55.4199F,
   288.818F,
   305.176F,
   445.801F,
   123.291F,
   452.637F,
   224.365F,
   574.463F,
   -48.5840F,
   105.225F,
   37.1094F,
   423.584F,
   -52.0020F,
   -6.83594F,
   95.7031F,
   325.684F,
   -37.3535F,
   -12.6953F,
   238.770F,
   281.006F,
   -78.8574F,
   -97.6563F,
   198.486F,
   415.771F,
   -33.2031F,
   20.5078F,
   353.760F,
   491.943F,
   -80.8105F,
   -34.9121F,
   -33.4473F,
   291.016F,
   -62.5000F,
   130.371F,
   -38.3301F,
   251.709F,
   -74.9512F,
   -107.178F,
   132.324F,
   178.467F,
   -80.3223F,
   -102.539F,
   -23.6816F,
   150.391F,
   -88.3789F,
   -41.0156F,
   -78.6133F,
   89.3555F,
   -60.3027F,
   -26.8555F,
   -51.5137F,
   21.7285F,
   -47.8516F,
   -75.4395F,
   4.88281F,
   14.4043F,
   -88.8672F,
   -113.037F,
   -69.8242F,
   21.7285F,
   -82.0313F,
   42.7246F,
   -105.469F,
   34.4238F,
   -92.5293F,
   -46.3867F,
   -105.957F,
   -47.8516F,
   -19.2871F,
   36.6211F,
   -67.8711F,
   -55.4199F,
   -68.3594F,
   40.5273F,
   -135.498F,
   -103.027F,
   -37.8418F,
   132.080F,
   -89.3555F,
   13.1836F,
   -7.08008F,
   -20.2637F,
   -73.4863F,
   -188.965F,
   45.4102F,
   153.320F,
   -96.9238F,
   -64.4531F,
   59.0820F,
   71.5332F,
   -48.0957F,
   -142.822F,
   30.2734F,
   100.098F,
   12.9395F,
   -32.4707F,
   2.44141F,
   83.0078F,
   -139.160F,
   -260.010F,
   15.8691F,
   -108.887F,
   16.6016F,
   -120.361F,
   93.5059F,
   228.760F,
   -87.1582F,
   -173.584F,
   -87.6465F,
   -61.0352F,
   -165.283F,
   -260.742F,
   71.2891F,
   -6.34766F,
   88.6230F,
   1.46484F,
   148.193F,
   320.557F,
   -31.0059F,
   -2.44141F,
   369.385F,
   460.449F,
   174.072F,
   237.305F,
   358.643F,
   532.471F,
   352.295F,
   492.188F

};
static const Float32 dico2_lsf_5[DICO2_SIZE_5 * 4] =
{
   -398.193F,
   -390.625F,
   -438.477F,
   -559.082F,
   -250.732F,
   -432.129F,
   -268.555F,
   -494.385F,
   -311.768F,
   -338.867F,
   -333.740F,
   -374.512F,
   -231.201F,
   -356.689F,
   -237.305F,
   -372.070F,
   -243.896F,
   -298.340F,
   -249.023F,
   -286.133F,
   -198.975F,
   -240.967F,
   -242.188F,
   -334.717F,
   -296.875F,
   -245.605F,
   -314.697F,
   -267.090F,
   -181.641F,
   -309.570F,
   -184.326F,
   -315.674F,
   -210.449F,
   -225.342F,
   -220.947F,
   -240.234F,
   -165.527F,
   -256.592F,
   -167.236F,
   -256.348F,
   -265.381F,
   -240.479F,
   -259.277F,
   -165.771F,
   -241.455F,
   -156.494F,
   -275.146F,
   -238.281F,
   -186.035F,
   -159.668F,
   -217.285F,
   -196.777F,
   -203.369F,
   -266.357F,
   -172.363F,
   -153.564F,
   -151.611F,
   -196.777F,
   -156.250F,
   -198.242F,
   -189.209F,
   -154.785F,
   -190.186F,
   -132.568F,
   -243.164F,
   -137.939F,
   -262.451F,
   -141.602F,
   -133.301F,
   -149.170F,
   -139.648F,
   -151.123F,
   -185.547F,
   -70.8008F,
   -214.600F,
   -128.418F,
   -200.928F,
   -112.793F,
   -194.092F,
   -61.7676F,
   -135.010F,
   -101.318F,
   -143.799F,
   -107.178F,
   -130.127F,
   -83.0078F,
   -168.945F,
   -228.271F,
   -123.291F,
   -188.477F,
   -171.387F,
   -276.123F,
   -64.2090F,
   -74.7070F,
   -237.061F,
   -117.920F,
   -108.643F,
   -18.0664F,
   -135.498F,
   -133.789F,
   -149.902F,
   -31.4941F,
   -169.189F,
   -57.1289F,
   -96.6797F,
   -60.0586F,
   -115.967F,
   -61.0352F,
   -64.6973F,
   -98.6328F,
   -91.7969F,
   -125.488F,
   -101.807F,
   -124.512F,
   -73.2422F,
   -76.4160F,
   -81.5430F,
   -162.109F,
   -113.037F,
   -198.730F,
   -94.2383F,
   -171.875F,
   -82.2754F,
   -150.146F,
   -57.1289F,
   -49.0723F,
   -56.8848F,
   -58.3496F,
   -40.7715F,
   -138.428F,
   -49.5605F,
   -151.123F,
   -35.8887F,
   -101.318F,
   -28.0762F,
   -85.9375F,
   -40.5273F,
   -183.105F,
   -41.7480F,
   -185.791F,
   -65.9180F,
   -214.600F,
   -64.4531F,
   -220.459F,
   -89.5996F,
   -181.641F,
   10.4980F,
   -115.967F,
   3.41797F,
   -159.424F,
   10.4980F,
   -163.574F,
   2.68555F,
   -109.375F,
   -14.4043F,
   -127.197F,
   -30.7617F,
   -29.0527F,
   -37.8418F,
   -149.658F,
   -10.2539F,
   -210.693F,
   -6.59180F,
   -227.295F,
   33.2031F,
   -117.920F,
   44.6777F,
   -114.258F,
   13.4277F,
   -72.7539F,
   13.4277F,
   -74.2188F,
   76.4160F,
   -148.682F,
   76.4160F,
   -175.781F,
   78.6133F,
   -40.7715F,
   24.4141F,
   -132.080F,
   -0.732422F,
   -29.0527F,
   -27.0996F,
   -45.6543F,
   56.8848F,
   -57.6172F,
   63.4766F,
   -57.1289F,
   6.34766F,
   -40.2832F,
   32.7148F,
   -10.9863F,
   -9.76563F,
   -134.033F,
   87.8906F,
   -49.5605F,
   92.2852F,
   -94.7266F,
   109.863F,
   -93.5059F,
   67.1387F,
   4.88281F,
   44.4336F,
   -25.1465F,
   60.0586F,
   -27.0996F,
   105.225F,
   9.03320F,
   112.793F,
   -35.6445F,
   118.896F,
   -38.3301F,
   -69.3359F,
   -14.4043F,
   122.803F,
   -44.9219F,
   5.85938F,
   12.9395F,
   -0.732422F,
   13.1836F,
   29.7852F,
   63.2324F,
   81.2988F,
   16.1133F,
   118.164F,
   25.3906F,
   106.445F,
   16.6016F,
   47.6074F,
   28.3203F,
   46.3867F,
   50.2930F,
   65.6738F,
   -2.19727F,
   117.676F,
   85.9375F,
   93.2617F,
   69.5801F,
   97.4121F,
   67.6270F,
   110.352F,
   62.5000F,
   16.8457F,
   45.4102F,
   3.17383F,
   72.5098F,
   -3.17383F,
   63.2324F,
   -23.1934F,
   7.32422F,
   13.6719F,
   96.1914F,
   47.8516F,
   103.760F,
   50.0488F,
   111.328F,
   68.6035F,
   140.869F,
   3.66211F,
   46.6309F,
   91.5527F,
   70.8008F,
   99.3652F,
   140.625F,
   -13.6719F,
   55.4199F,
   132.813F,
   98.8770F,
   0.000000F,
   134.033F,
   -22.4609F,
   128.906F,
   -55.9082F,
   85.6934F,
   -59.8145F,
   82.5195F,
   -88.3789F,
   106.201F,
   40.7715F,
   128.662F,
   -18.3105F,
   73.7305F,
   22.2168F,
   201.172F,
   31.4941F,
   146.240F,
   121.094F,
   165.771F,
   45.4102F,
   182.861F,
   37.3535F,
   179.932F,
   -68.6035F,
   146.484F,
   -84.9609F,
   150.146F,
   -57.6172F,
   187.744F,
   10.0098F,
   215.088F,
   9.27734F,
   217.285F,
   -53.7109F,
   205.322F,
   -87.1582F,
   215.576F,
   -95.9473F,
   220.459F,
   -154.785F,
   115.723F,
   -108.398F,
   207.520F,
   -42.7246F,
   165.527F,
   -120.361F,
   59.0820F,
   -126.709F,
   191.650F,
   -174.316F,
   142.090F,
   -132.080F,
   89.3555F,
   -132.568F,
   105.957F,
   -145.752F,
   122.070F,
   -186.768F,
   54.1992F,
   -171.387F,
   223.877F,
   -181.396F,
   234.863F,
   -212.158F,
   122.314F,
   -219.482F,
   133.789F,
   -92.5293F,
   48.8281F,
   -106.201F,
   38.3301F,
   -199.951F,
   52.2461F,
   -210.205F,
   38.3301F,
   -149.902F,
   9.76563F,
   -154.297F,
   22.9492F,
   -215.576F,
   -13.1836F,
   -180.908F,
   125.977F,
   -122.314F,
   72.7539F,
   -149.902F,
   -41.7480F,
   -212.402F,
   -39.3066F,
   -211.182F,
   -5.61523F,
   -199.707F,
   22.7051F,
   -247.803F,
   -65.1855F,
   -161.621F,
   -87.6465F,
   -134.033F,
   0.488281F,
   -107.910F,
   -29.5410F,
   -92.0410F,
   0.000000F,
   -55.4199F,
   8.05664F,
   -101.074F,
   -30.7617F,
   -31.4941F,
   51.7578F,
   -228.027F,
   8.30078F,
   -264.160F,
   -68.8477F,
   -273.193F,
   -65.4297F,
   -173.340F,
   -201.416F,
   -102.539F,
   -46.6309F,
   -262.695F,
   -226.563F,
   -223.877F,
   -22.7051F,
   -153.320F,
   -87.4023F,
   23.6816F,
   1.70898F,
   -50.2930F,
   -95.9473F,
   -24.6582F,
   5.85938F,
   -49.5605F,
   9.27734F,
   -41.0156F,
   20.2637F,
   -146.240F,
   -103.271F,
   -68.1152F,
   104.004F,
   -170.898F,
   28.8086F,
   -18.3105F,
   50.2930F,
   -239.502F,
   -164.307F,
   -166.016F,
   101.807F,
   -89.5996F,
   9.03320F,
   -68.1152F,
   115.723F,
   -31.4941F,
   -77.6367F,
   77.8809F,
   72.2656F,
   -152.832F,
   -9.52148F,
   83.7402F,
   146.973F,
   -169.922F,
   -9.52148F,
   -73.9746F,
   229.492F,
   25.3906F,
   56.8848F,
   -92.7734F,
   33.4473F,
   -8.78906F,
   65.6738F,
   -18.3105F,
   -52.2461F,
   29.2969F,
   10.4980F,
   -129.150F,
   -116.455F,
   112.061F,
   40.0391F,
   -49.3164F,
   -55.9082F,
   -11.9629F,
   -40.7715F,
   148.682F,
   193.359F,
   23.9258F,
   -53.7109F,
   223.389F,
   36.1328F,
   71.5332F,
   69.0918F,
   212.158F,
   22.2168F,
   140.381F,
   96.1914F,
   79.5898F,
   -19.0430F,
   175.049F,
   16.3574F,
   89.1113F,
   -78.8574F,
   150.391F,
   -8.78906F,
   178.467F,
   6.59180F,
   151.123F,
   58.1055F,
   154.297F,
   66.6504F,
   109.375F,
   24.1699F,
   195.557F,
   116.211F,
   212.158F,
   66.6504F,
   167.236F,
   15.6250F,
   192.627F,
   17.5781F,
   249.268F,
   52.9785F,
   193.604F,
   112.061F,
   179.199F,
   87.8906F,
   157.715F,
   117.188F,
   87.8906F,
   78.6133F,
   104.736F,
   113.281F,
   155.762F,
   104.980F,
   184.570F,
   88.6230F,
   244.141F,
   98.6328F,
   166.748F,
   128.906F,
   146.973F,
   150.146F,
   159.912F,
   100.830F,
   230.957F,
   167.725F,
   228.760F,
   146.973F,
   220.703F,
   147.461F,
   135.498F,
   179.932F,
   191.895F,
   161.621F,
   114.014F,
   159.668F,
   88.3789F,
   143.799F,
   226.807F,
   173.340F,
   121.582F,
   116.699F,
   101.318F,
   102.539F,
   169.189F,
   215.576F,
   198.486F,
   166.748F,
   190.674F,
   225.830F,
   222.900F,
   229.248F,
   177.246F,
   178.711F,
   119.873F,
   208.252F,
   129.639F,
   231.445F,
   179.199F,
   235.107F,
   76.9043F,
   197.266F,
   185.791F,
   184.326F,
   279.297F,
   185.547F,
   159.912F,
   262.695F,
   201.660F,
   258.057F,
   266.357F,
   204.590F,
   244.873F,
   197.266F,
   255.615F,
   276.611F,
   160.889F,
   268.799F,
   242.188F,
   256.348F,
   262.207F,
   262.451F,
   237.061F,
   169.434F,
   299.316F,
   257.324F,
   139.404F,
   205.322F,
   215.820F,
   342.773F,
   336.670F,
   267.578F,
   263.672F,
   210.205F,
   300.537F,
   179.443F,
   313.477F,
   185.547F,
   310.547F,
   241.943F,
   333.740F,
   257.080F,
   306.885F,
   170.898F,
   256.348F,
   130.371F,
   241.211F,
   110.596F,
   308.594F,
   146.240F,
   278.320F,
   165.771F,
   395.752F,
   198.975F,
   337.891F,
   127.197F,
   321.533F,
   95.9473F,
   381.836F,
   196.533F,
   353.516F,
   167.480F,
   260.742F,
   158.203F,
   213.623F,
   74.9512F,
   264.404F,
   88.1348F,
   255.615F,
   77.3926F,
   345.947F,
   235.352F,
   164.795F,
   139.404F,
   281.250F,
   19.2871F,
   271.973F,
   -11.4746F,
   373.535F,
   75.9277F,
   420.166F,
   76.6602F,
   284.668F,
   168.213F,
   125.488F,
   -22.9492F,
   85.2051F,
   68.8477F,
   344.727F,
   80.0781F,
   250.244F,
   118.896F,
   -15.8691F,
   13.9160F,
   196.533F,
   236.816F,
   8.78906F,
   15.1367F,
   187.744F,
   -64.2090F,
   193.115F,
   -84.4727F,
   155.518F,
   170.654F,
   -33.4473F,
   151.367F,
   130.371F,
   132.080F,
   -179.443F,
   47.3633F,
   173.584F,
   73.2422F,
   -65.4297F,
   -210.693F,
   226.074F,
   187.744F,
   -172.852F,
   -104.492F,
   123.535F,
   42.4805F,
   -217.773F,
   -153.809F,
   106.201F,
   133.545F,
   -350.342F,
   -62.9883F,
   151.611F,
   114.990F,
   -248.535F,
   -333.984F,
   -95.9473F,
   127.197F,
   -224.609F,
   -167.480F,
   -6.10352F,
   4.88281F,
   -239.746F,
   -282.227F,
   83.0078F,
   2.19727F,
   -380.371F,
   -277.100F,
   -85.9375F,
   11.7188F,
   -385.498F,
   -98.1445F,
   -216.553F,
   1.46484F,
   -282.227F,
   -216.797F,
   -133.789F,
   -85.9375F,
   -401.123F,
   -285.156F,
   -38.8184F,
   148.926F,
   -494.141F,
   -235.107F,
   -54.9316F,
   47.1191F,
   -404.297F,
   -478.516F,
   -59.8145F,
   -120.361F,
   -235.352F,
   -410.156F,
   -228.516F,
   -155.029F,
   -317.139F,
   -425.781F,
   -338.867F,
   -147.461F,
   -375.977F,
   -203.857F,
   -341.064F,
   -32.9590F,
   -387.695F,
   -70.8008F,
   -407.715F,
   -173.828F,
   -490.967F,
   -398.438F,
   -406.006F,
   -6.59180F,
   -551.270F,
   -197.998F,
   -282.471F,
   44.9219F,
   -308.838F,
   46.1426F,
   -333.740F,
   143.066F,
   -490.967F,
   49.0723F,
   -192.871F,
   173.828F,
   -295.410F,
   0.732422F,
   -252.197F,
   197.266F,
   -305.420F,
   202.637F,
   -27.0996F,
   155.029F,
   -399.414F,
   109.131F,
   -113.037F,
   -231.689F,
   -108.643F,
   -226.563F,
   -123.047F,
   -283.691F,
   -122.314F,
   -295.654F,
   35.1563F,
   -85.6934F,
   -90.8203F,
   -256.836F,
   -69.0918F,
   -258.545F,
   -68.1152F,
   -274.170F,
   -140.381F,
   -351.074F,
   -143.311F,
   -394.043F,
   -228.271F,
   -240.234F,
   55.9082F,
   168.457F,
   -224.854F,
   -175.537F,
   -98.3887F,
   332.520F,
   -167.236F,
   -113.525F,
   213.379F,
   96.9238F,
   -124.268F,
   -11.2305F,
   77.3926F,
   325.684F,
   -118.408F,
   111.328F,
   198.486F,
   107.178F,
   -100.342F,
   82.7637F,
   219.238F,
   260.498F,
   -103.760F,
   11.2305F,
   351.807F,
   121.338F,
   -221.924F,
   -195.313F,
   357.666F,
   255.371F,
   -62.0117F,
   -78.3691F,
   349.121F,
   284.424F,
   16.6016F,
   85.4492F,
   252.441F,
   162.598F,
   90.3320F,
   2.68555F,
   320.068F,
   192.871F,
   34.9121F,
   56.6406F,
   254.150F,
   381.348F,
   -27.8320F,
   161.865F,
   394.531F,
   263.184F,
   110.840F,
   141.357F,
   311.279F,
   253.906F,
   -18.5547F,
   221.924F,
   183.594F,
   260.498F,
   37.3535F,
   125.000F,
   84.9609F,
   296.387F,
   149.902F,
   93.9941F,
   449.951F,
   197.266F,
   65.6738F,
   252.441F,
   49.5605F,
   265.137F,
   159.180F,
   248.291F,
   435.303F,
   275.879F,
   104.736F,
   323.975F,
   94.4824F,
   337.891F,
   -11.9629F,
   288.818F,
   -17.5781F,
   296.631F,
   -101.563F,
   244.385F,
   132.813F,
   427.002F,
   -85.9375F,
   298.584F,
   -122.559F,
   292.725F,
   -143.799F,
   138.916F,
   -55.4199F,
   397.949F,
   -34.6680F,
   385.254F,
   -56.1523F,
   418.701F,
   -174.316F,
   314.453F,
   -204.590F,
   341.309F,
   276.123F,
   331.299F,
   -50.7813F,
   300.781F,
   106.689F,
   235.596F,
   -226.807F,
   199.707F,
   197.998F,
   344.238F,
   209.717F,
   367.920F,
   40.0391F,
   295.898F,
   338.623F,
   437.744F,
   118.164F,
   457.520F,
   111.328F,
   503.662F,
   243.164F,
   285.645F,
   323.730F,
   342.285F,
   321.289F,
   332.031F,
   277.100F,
   308.105F,
   301.270F,
   395.020F,
   332.275F,
   431.641F,
   346.924F,
   299.561F,
   386.719F,
   328.857F,
   208.496F,
   164.063F,
   411.377F,
   382.324F,
   278.076F,
   310.059F,
   492.188F,
   445.557F,
   432.861F,
   385.986F,
   374.023F,
   356.445F,
   363.037F,
   230.957F,
   405.029F,
   249.268F,
   425.781F,
   295.898F,
   339.844F,
   238.525F,
   432.617F,
   283.447F,
   445.801F,
   284.180F,
   419.434F,
   348.877F,
   481.689F,
   388.428F,
   289.307F,
   210.938F,
   520.508F,
   259.033F,
   439.209F,
   198.730F,
   448.730F,
   184.814F,
   513.672F,
   321.045F,
   501.465F,
   307.129F,
   515.869F,
   223.389F,
   569.092F,
   227.051F,
   358.154F,
   280.029F,
   632.324F,
   351.318F,
   548.096F,
   425.781F,
   510.254F,
   395.508F,
   575.684F,
   354.980F,
   650.879F,
   367.676F,
   458.008F,
   448.486F,
   505.371F,
   482.178F,
   424.561F,
   385.010F,
   166.504F,
   314.697F,
   386.719F,
   499.268F,
   354.980F,
   512.207F,
   609.863F,
   489.258F,
   661.865F,
   504.395F,
   177.246F,
   387.695F,
   672.852F,
   570.313F,
   55.6641F,
   206.787F,
   599.609F,
   405.029F,
   8.78906F,
   73.4863F,
   474.121F,
   477.783F,
   -108.887F,
   -23.4375F,
   525.879F,
   340.820F,
   374.268F,
   268.799F,
   3.41797F,
   148.438F,
   -225.342F,
   -178.711F,
   337.646F,
   483.887F,
   328.369F,
   232.422F,
   -166.016F,
   78.3691F,
   312.744F,
   309.570F,
   -389.160F,
   89.1113F,
   229.736F,
   230.957F,
   -424.072F,
   -200.684F,
   579.590F,
   680.420F,
   444.580F,
   680.664F

};
static const Float32 dico3_lsf_5[DICO3_SIZE_5 * 4] =
{
   -442.383F,
   -555.420F,
   -458.740F,
   -619.385F,
   -400.391F,
   -451.172F,
   -413.818F,
   -489.258F,
   -297.852F,
   -466.797F,
   -298.096F,
   -514.160F,
   -380.615F,
   -387.695F,
   -384.033F,
   -379.883F,
   -291.748F,
   -394.287F,
   -298.828F,
   -421.631F,
   -331.787F,
   -281.006F,
   -394.531F,
   -475.586F,
   -311.035F,
   -339.600F,
   -318.604F,
   -342.529F,
   -392.334F,
   -287.842F,
   -409.180F,
   -320.068F,
   -352.295F,
   -360.840F,
   -333.740F,
   -219.238F,
   -306.641F,
   -258.545F,
   -324.951F,
   -276.855F,
   -239.746F,
   -276.611F,
   -280.518F,
   -367.188F,
   -263.672F,
   -319.336F,
   -249.023F,
   -288.818F,
   -239.258F,
   -362.793F,
   -236.084F,
   -364.990F,
   -241.211F,
   -225.098F,
   -255.615F,
   -262.939F,
   -204.590F,
   -287.842F,
   -209.473F,
   -298.340F,
   -276.123F,
   -254.150F,
   -259.766F,
   -187.256F,
   -212.891F,
   -282.471F,
   -171.143F,
   -214.844F,
   -172.363F,
   -221.191F,
   -188.965F,
   -248.047F,
   -141.113F,
   -263.672F,
   -195.557F,
   -360.840F,
   -144.287F,
   -271.240F,
   -144.531F,
   -279.785F,
   -174.072F,
   -338.867F,
   -156.250F,
   -335.938F,
   -145.752F,
   -258.545F,
   -101.563F,
   -220.459F,
   -167.480F,
   -203.125F,
   -161.377F,
   -172.852F,
   -108.398F,
   -211.914F,
   -119.629F,
   -224.854F,
   -91.3086F,
   -189.453F,
   -151.123F,
   -285.645F,
   -142.822F,
   -134.033F,
   -187.744F,
   -194.092F,
   -106.201F,
   -160.889F,
   -129.395F,
   -180.908F,
   -121.582F,
   -204.346F,
   -87.1582F,
   -145.752F,
   -68.1152F,
   -212.646F,
   -59.3262F,
   -216.553F,
   -68.8477F,
   -162.354F,
   -68.3594F,
   -162.842F,
   -40.2832F,
   -136.719F,
   -96.1914F,
   -220.459F,
   -88.3789F,
   -100.098F,
   -109.375F,
   -142.334F,
   -99.8535F,
   -140.137F,
   -76.4160F,
   -87.1582F,
   -155.518F,
   -133.789F,
   -139.160F,
   -106.445F,
   -218.750F,
   -123.047F,
   -93.2617F,
   -184.814F,
   -14.1602F,
   -117.432F,
   -40.2832F,
   -150.879F,
   -46.6309F,
   -91.3086F,
   -57.1289F,
   -93.2617F,
   -54.1992F,
   -166.748F,
   -6.10352F,
   -117.188F,
   -102.051F,
   -87.6465F,
   -178.223F,
   -86.1816F,
   -79.1016F,
   -38.3301F,
   -105.469F,
   -78.6133F,
   -96.1914F,
   -73.9746F,
   -69.3359F,
   -25.3906F,
   -146.729F,
   -70.5566F,
   -135.742F,
   -47.8516F,
   -143.555F,
   -36.6211F,
   -160.889F,
   -148.438F,
   -115.479F,
   -5.85938F,
   -16.6016F,
   -109.375F,
   -115.723F,
   -1.95313F,
   -123.535F,
   -10.9863F,
   -182.617F,
   -44.9219F,
   -206.055F,
   -61.5234F,
   -219.971F,
   -22.2168F,
   -142.578F,
   -23.6816F,
   -159.180F,
   33.6914F,
   -186.523F,
   -31.9824F,
   -165.527F,
   -2.92969F,
   -163.574F,
   40.2832F,
   -63.2324F,
   -0.732422F,
   -205.078F,
   -26.1230F,
   -221.924F,
   9.03320F,
   -242.188F,
   10.7422F,
   -208.496F,
   -101.318F,
   -204.834F,
   3.17383F,
   -244.385F,
   -66.1621F,
   -250.488F,
   -75.4395F,
   -194.824F,
   -116.699F,
   -203.125F,
   -119.141F,
   -230.225F,
   41.0156F,
   -271.484F,
   -94.4824F,
   -289.307F,
   -24.6582F,
   -288.818F,
   -9.76563F,
   -229.736F,
   -77.1484F,
   -251.465F,
   -187.988F,
   -254.883F,
   -152.588F,
   -263.916F,
   -131.348F,
   -298.828F,
   -72.9980F,
   -320.313F,
   -106.445F,
   -292.236F,
   -161.865F,
   -284.912F,
   -39.3066F,
   -296.875F,
   -168.457F,
   -302.002F,
   -202.881F,
   -349.609F,
   -175.781F,
   -342.529F,
   -120.361F,
   -219.238F,
   -180.664F,
   -225.098F,
   -195.557F,
   -269.043F,
   -98.1445F,
   -385.498F,
   -235.352F,
   -259.033F,
   -155.762F,
   -309.814F,
   -351.074F,
   -365.967F,
   -228.027F,
   -366.699F,
   -218.506F,
   -390.137F,
   -137.695F,
   -420.654F,
   -175.049F,
   -147.949F,
   -145.752F,
   -284.668F,
   -264.893F,
   -334.229F,
   -114.258F,
   -475.098F,
   -364.502F,
   -448.730F,
   -232.666F,
   -471.680F,
   -227.295F,
   -365.967F,
   -45.8984F,
   -399.170F,
   -102.783F,
   -355.713F,
   -82.5195F,
   -353.516F,
   -5.37109F,
   -474.121F,
   -103.027F,
   -489.746F,
   -60.7910F,
   -121.094F,
   -27.8320F,
   -466.309F,
   -184.326F,
   -314.697F,
   42.4805F,
   -354.248F,
   -26.6113F,
   -117.676F,
   -62.7441F,
   -298.096F,
   -124.023F,
   -394.775F,
   36.8652F,
   -413.574F,
   50.7813F,
   -159.668F,
   26.1230F,
   -403.076F,
   7.08008F,
   -278.564F,
   68.1152F,
   -296.631F,
   74.7070F,
   -299.805F,
   -123.535F,
   -178.223F,
   -42.7246F,
   -301.758F,
   -24.6582F,
   -236.572F,
   134.521F,
   -212.402F,
   67.8711F,
   -200.928F,
   76.9043F,
   -137.451F,
   91.7969F,
   -256.592F,
   55.6641F,
   -123.779F,
   68.3594F,
   -146.240F,
   68.6035F,
   -185.059F,
   61.7676F,
   -74.4629F,
   92.5293F,
   -184.326F,
   -32.7148F,
   -149.170F,
   161.133F,
   -201.172F,
   130.859F,
   -199.463F,
   157.715F,
   -100.830F,
   11.9629F,
   -83.2520F,
   43.2129F,
   -110.596F,
   128.418F,
   -117.676F,
   143.799F,
   -17.3340F,
   82.7637F,
   -160.400F,
   64.4531F,
   -59.5703F,
   72.0215F,
   -57.8613F,
   76.9043F,
   -94.4824F,
   138.916F,
   -123.535F,
   -2.19727F,
   -92.0410F,
   3.41797F,
   -39.0625F,
   161.377F,
   -52.7344F,
   9.76563F,
   -75.1953F,
   -11.2305F,
   23.1934F,
   52.2461F,
   -59.0820F,
   40.7715F,
   -20.9961F,
   46.8750F,
   -13.6719F,
   6.59180F,
   -18.5547F,
   7.56836F,
   8.78906F,
   75.4395F,
   -25.8789F,
   -44.4336F,
   -27.5879F,
   18.0664F,
   -107.666F,
   -5.37109F,
   5.61523F,
   33.9355F,
   19.7754F,
   -2.68555F,
   10.7422F,
   3.66211F,
   -21.2402F,
   -33.4473F,
   -28.8086F,
   -50.5371F,
   -38.5742F,
   -14.1602F,
   66.4063F,
   -22.4609F,
   -38.0859F,
   -107.666F,
   1.95313F,
   -33.2031F,
   31.2500F,
   -53.9551F,
   24.6582F,
   -53.2227F,
   9.76563F,
   -48.0957F,
   -18.5547F,
   -111.328F,
   2.19727F,
   -108.643F,
   8.05664F,
   -103.271F,
   55.1758F,
   14.6484F,
   17.8223F,
   -54.1992F,
   38.0859F,
   -97.4121F,
   68.3594F,
   -77.6367F,
   59.8145F,
   -83.2520F,
   40.5273F,
   -121.826F,
   82.7637F,
   -46.3867F,
   79.8340F,
   -53.4668F,
   79.3457F,
   -33.4473F,
   -21.7285F,
   -145.508F,
   24.4141F,
   -153.076F,
   35.1563F,
   -165.283F,
   118.896F,
   6.83594F,
   61.5234F,
   -95.4590F,
   52.2461F,
   -10.0098F,
   68.8477F,
   -6.83594F,
   24.1699F,
   -69.8242F,
   80.8105F,
   11.9629F,
   112.061F,
   -94.7266F,
   137.939F,
   -90.0879F,
   106.445F,
   6.83594F,
   82.0313F,
   -2.19727F,
   96.9238F,
   -40.7715F,
   150.879F,
   8.30078F,
   145.508F,
   -4.15039F,
   136.963F,
   -34.1797F,
   72.9980F,
   19.2871F,
   127.441F,
   30.5176F,
   49.5605F,
   0.488281F,
   59.5703F,
   70.3125F,
   62.2559F,
   51.5137F,
   42.7246F,
   20.0195F,
   145.508F,
   45.6543F,
   126.221F,
   26.3672F,
   93.0176F,
   62.2559F,
   89.1113F,
   72.5098F,
   121.338F,
   85.9375F,
   79.8340F,
   -20.0195F,
   6.10352F,
   51.2695F,
   90.5762F,
   59.8145F,
   63.7207F,
   0.732422F,
   133.057F,
   109.619F,
   34.1797F,
   71.7773F,
   10.7422F,
   72.0215F,
   51.7578F,
   84.7168F,
   59.5703F,
   120.605F,
   80.8105F,
   128.906F,
   49.0723F,
   74.9512F,
   85.2051F,
   100.342F,
   149.658F,
   69.3359F,
   149.902F,
   100.830F,
   113.281F,
   78.6133F,
   152.344F,
   96.9238F,
   23.6816F,
   48.8281F,
   -39.0625F,
   93.7500F,
   36.3770F,
   88.3789F,
   120.850F,
   128.174F,
   65.6738F,
   142.822F,
   8.05664F,
   119.873F,
   -29.5410F,
   105.713F,
   104.248F,
   149.170F,
   121.582F,
   125.977F,
   41.7480F,
   108.154F,
   121.338F,
   162.598F,
   107.422F,
   67.1387F,
   138.184F,
   140.381F,
   35.6445F,
   156.006F,
   37.8418F,
   163.574F,
   -8.05664F,
   42.2363F,
   51.7578F,
   169.922F,
   -40.5273F,
   146.729F,
   -46.6309F,
   169.678F,
   -119.385F,
   122.803F,
   42.7246F,
   181.152F,
   52.2461F,
   116.211F,
   90.8203F,
   264.404F,
   141.113F,
   129.395F,
   143.066F,
   189.697F,
   103.760F,
   213.379F,
   76.9043F,
   205.322F,
   91.3086F,
   207.031F,
   -40.2832F,
   137.939F,
   8.54492F,
   241.943F,
   -9.52148F,
   259.277F,
   80.3223F,
   173.828F,
   191.895F,
   205.078F,
   157.471F,
   194.092F,
   161.377F,
   165.039F,
   139.404F,
   224.121F,
   154.297F,
   263.428F,
   164.307F,
   199.463F,
   77.6367F,
   94.7266F,
   213.379F,
   247.070F,
   137.695F,
   207.031F,
   214.844F,
   151.367F,
   135.986F,
   116.943F,
   163.818F,
   110.596F,
   168.945F,
   114.258F,
   205.078F,
   156.738F,
   206.055F,
   157.471F,
   123.535F,
   104.492F,
   218.994F,
   138.428F,
   204.346F,
   94.4824F,
   234.863F,
   121.826F,
   168.701F,
   136.963F,
   229.248F,
   226.074F,
   191.162F,
   72.2656F,
   192.871F,
   65.4297F,
   250.977F,
   129.395F,
   213.379F,
   80.3223F,
   133.789F,
   34.9121F,
   164.795F,
   71.0449F,
   122.803F,
   16.1133F,
   254.150F,
   87.6465F,
   191.895F,
   23.6816F,
   196.533F,
   8.05664F,
   204.346F,
   114.746F,
   124.756F,
   11.9629F,
   266.602F,
   79.8340F,
   286.621F,
   78.8574F,
   0.732422F,
   59.0820F,
   212.891F,
   115.723F,
   168.213F,
   104.736F,
   324.463F,
   165.527F,
   254.395F,
   151.367F,
   270.752F,
   162.109F,
   78.3691F,
   47.1191F,
   217.041F,
   231.934F,
   281.494F,
   213.379F,
   218.018F,
   155.029F,
   214.111F,
   210.449F,
   231.445F,
   222.900F,
   315.674F,
   162.354F,
   322.266F,
   156.006F,
   243.408F,
   193.604F,
   342.285F,
   251.465F,
   287.109F,
   247.070F,
   270.996F,
   234.131F,
   344.238F,
   225.830F,
   342.529F,
   223.389F,
   132.568F,
   210.449F,
   272.461F,
   298.340F,
   203.857F,
   290.527F,
   203.857F,
   290.527F,
   234.131F,
   280.273F,
   280.029F,
   335.938F,
   317.383F,
   291.260F,
   345.459F,
   300.537F,
   325.928F,
   327.393F,
   182.129F,
   266.602F,
   417.725F,
   313.232F,
   339.111F,
   261.963F,
   325.684F,
   382.324F,
   281.494F,
   360.107F,
   401.611F,
   277.588F,
   445.557F,
   297.852F,
   257.813F,
   337.402F,
   371.338F,
   422.363F,
   398.438F,
   377.197F,
   395.508F,
   376.465F,
   208.740F,
   389.648F,
   211.182F,
   406.982F,
   169.189F,
   216.064F,
   418.945F,
   370.850F,
   284.912F,
   316.406F,
   539.307F,
   429.688F,
   476.563F,
   364.502F,
   493.164F,
   361.816F,
   374.512F,
   455.566F,
   413.574F,
   490.234F,
   382.324F,
   182.617F,
   429.932F,
   201.416F,
   71.7773F,
   339.844F,
   264.648F,
   502.441F,
   151.611F,
   321.045F,
   89.1113F,
   314.209F,
   48.3398F,
   250.977F,
   119.141F,
   343.750F,
   60.7910F,
   98.3887F,
   247.559F,
   381.104F,
   79.1016F,
   88.6230F,
   401.611F,
   254.883F,
   47.1191F,
   89.5996F,
   496.582F,
   453.857F,
   -61.2793F,
   141.357F,
   183.105F,
   242.676F,
   -59.3262F,
   7.32422F,
   323.486F,
   214.600F,
   -6.83594F,
   -41.2598F,
   152.344F,
   223.877F,
   -110.596F,
   38.8184F,
   45.4102F,
   334.473F,
   -149.902F,
   1.46484F,
   131.104F,
   95.7031F,
   -22.9492F,
   -71.0449F,
   190.674F,
   55.9082F,
   -31.2500F,
   -72.7539F,
   59.8145F,
   119.873F,
   -171.143F,
   -158.203F,
   237.305F,
   192.627F,
   -122.314F,
   -156.250F,
   43.4570F,
   62.2559F,
   -89.1113F,
   -95.2148F,
   -62.2559F,
   77.3926F,
   -233.887F,
   -71.7773F,
   -46.6309F,
   55.6641F,
   -189.209F,
   -109.131F,
   38.3301F,
   -57.8613F,
   -160.400F,
   -175.781F,
   -99.3652F,
   22.4609F,
   -28.5645F,
   -149.170F,
   81.5430F,
   -56.1523F,
   -165.771F,
   -264.648F,
   -35.1563F,
   -77.3926F,
   -219.971F,
   -210.205F,
   -180.176F,
   -87.8906F,
   -20.7520F,
   -177.490F,
   -21.9727F,
   -192.139F,
   24.4141F,
   -5.37109F,
   -95.4590F,
   -64.2090F,
   -13.6719F,
   -17.8223F,
   -82.2754F,
   -184.082F,
   1.22070F,
   -46.1426F,
   -172.363F,
   -152.344F,
   21.7285F,
   -83.9844F,
   -32.9590F,
   -271.729F,
   -86.1816F,
   -57.8613F,
   -166.992F,
   -277.100F,
   -67.1387F,
   -269.043F,
   -65.6738F,
   -293.701F,
   37.1094F,
   35.4004F,
   -176.270F,
   -300.781F,
   11.9629F,
   19.5313F,
   -304.688F,
   -189.453F,
   -60.5469F,
   95.4590F,
   -178.711F,
   -133.545F,
   114.502F,
   53.2227F,
   -62.2559F,
   -210.938F,
   16.8457F,
   89.3555F,
   -40.5273F,
   -118.408F,
   -167.969F,
   46.6309F,
   -295.898F,
   -291.992F,
   -41.5039F,
   -41.2598F,
   -319.336F,
   -398.193F,
   78.3691F,
   114.746F,
   -346.436F,
   -303.467F,
   -15.6250F,
   66.4063F,
   -332.275F,
   -60.5469F,
   120.117F,
   137.939F,
   -176.025F,
   -148.682F,
   47.6074F,
   118.408F,
   -139.893F,
   -32.4707F,
   104.248F,
   49.3164F,
   -41.7480F,
   -28.8086F,
   48.5840F,
   140.381F,
   0.488281F,
   -7.56836F,
   169.434F,
   184.326F,
   -333.496F,
   -9.52148F,
   134.766F,
   135.986F,
   -119.385F,
   66.1621F,
   166.016F,
   131.104F,
   3.17383F,
   -110.596F,
   208.740F,
   232.910F,
   -32.4707F,
   -12.6953F,
   -19.7754F,
   180.176F,
   -285.400F,
   155.518F,
   257.568F,
   258.545F,
   -23.1934F,
   165.039F,
   307.373F,
   263.916F,
   119.385F,
   74.4629F,
   -109.619F,
   232.910F,
   -130.371F,
   243.164F,
   -236.572F,
   211.426F,
   -258.301F,
   258.545F,
   -315.918F,
   150.879F,
   -345.703F,
   150.635F,
   -111.816F,
   333.496F,
   -38.8184F,
   444.580F,
   -188.965F,
   -128.906F,
   -3.41797F,
   270.996F,
   -293.457F,
   -219.971F,
   -188.477F,
   105.713F,
   -306.641F,
   -306.396F,
   -246.826F,
   -73.7305F,
   -146.973F,
   -142.822F,
   -185.303F,
   -395.020F,
   -185.547F,
   -378.174F,
   -205.078F,
   -468.994F,
   -199.219F,
   -131.592F,
   -431.885F,
   -545.654F,
   -55.4199F,
   -8.78906F,
   -496.582F,
   -447.021F,
   -514.404F,
   -274.902F,
   -603.271F,
   -443.359F,
   -358.887F,
   61.5234F,
   -659.424F,
   -101.318F,
   -139.404F,
   -114.014F,
   368.408F,
   379.395F,
   532.227F,
   482.178F,
   567.871F,
   493.164F


};
static const Float32 dico4_lsf_5[DICO4_SIZE_5 * 4] =
{
   -453.369F,
   -410.400F,
   -453.369F,
   -428.467F,
   -501.953F,
   -280.762F,
   -520.996F,
   -403.809F,
   -395.264F,
   -268.311F,
   -416.016F,
   -276.123F,
   -328.369F,
   -392.578F,
   -331.787F,
   -399.902F,
   -326.660F,
   -315.674F,
   -323.486F,
   -308.838F,
   -406.250F,
   -402.588F,
   -363.037F,
   -207.764F,
   -328.613F,
   -447.266F,
   -344.971F,
   -534.180F,
   -312.988F,
   -166.260F,
   -435.791F,
   -402.588F,
   -235.840F,
   -264.160F,
   -288.818F,
   -409.180F,
   -257.324F,
   -261.963F,
   -278.809F,
   -282.715F,
   -294.678F,
   -181.641F,
   -311.035F,
   -243.408F,
   -228.027F,
   -337.646F,
   -226.318F,
   -345.703F,
   -246.582F,
   -318.604F,
   -191.162F,
   -233.154F,
   -256.104F,
   -219.727F,
   -242.432F,
   -199.463F,
   -179.932F,
   -200.928F,
   -237.305F,
   -290.283F,
   -180.176F,
   -267.090F,
   -180.176F,
   -281.738F,
   -191.406F,
   -195.557F,
   -197.754F,
   -191.895F,
   -217.773F,
   -126.953F,
   -244.141F,
   -199.707F,
   -157.227F,
   -235.596F,
   -140.869F,
   -215.332F,
   -132.080F,
   -169.434F,
   -163.818F,
   -223.877F,
   -145.264F,
   -156.738F,
   -157.715F,
   -150.146F,
   -233.398F,
   -151.611F,
   -225.830F,
   -125.732F,
   -177.490F,
   -117.920F,
   -198.975F,
   -118.408F,
   -205.078F,
   -141.113F,
   -107.422F,
   -174.072F,
   -141.113F,
   -79.3457F,
   -160.400F,
   -163.574F,
   -94.2383F,
   -139.160F,
   -107.666F,
   -162.598F,
   -125.488F,
   -192.139F,
   -95.7031F,
   -129.150F,
   -127.441F,
   -110.596F,
   -118.896F,
   -103.271F,
   -150.391F,
   -142.822F,
   -150.635F,
   -38.3301F,
   -161.621F,
   -65.4297F,
   -166.016F,
   -84.9609F,
   -78.6133F,
   -78.8574F,
   -154.297F,
   -108.398F,
   -74.2188F,
   -104.980F,
   -81.0547F,
   -111.816F,
   -67.6270F,
   -114.258F,
   -160.889F,
   -193.604F,
   -77.8809F,
   -155.273F,
   -55.4199F,
   -135.254F,
   -91.0645F,
   -84.7168F,
   -81.5430F,
   -51.2695F,
   -111.328F,
   -46.8750F,
   -129.395F,
   -59.0820F,
   -52.7344F,
   -48.3398F,
   -89.3555F,
   -90.3320F,
   -82.5195F,
   -39.3066F,
   -99.8535F,
   -182.617F,
   -26.1230F,
   -92.7734F,
   -71.7773F,
   -156.982F,
   -54.4434F,
   -162.354F,
   -57.1289F,
   -180.908F,
   -34.4238F,
   -121.094F,
   -31.7383F,
   -124.512F,
   -33.9355F,
   -79.8340F,
   -41.9922F,
   -74.4629F,
   -74.7070F,
   -141.602F,
   -40.0391F,
   -64.2090F,
   -63.9648F,
   -41.9922F,
   -16.3574F,
   -98.1445F,
   7.56836F,
   -89.3555F,
   -2.44141F,
   -106.445F,
   -20.9961F,
   -128.662F,
   17.3340F,
   -92.0410F,
   -5.37109F,
   -148.682F,
   -2.92969F,
   -165.527F,
   -16.3574F,
   -77.8809F,
   15.3809F,
   -46.6309F,
   8.54492F,
   -44.1895F,
   -9.52148F,
   -59.0820F,
   30.7617F,
   -40.7715F,
   -34.1797F,
   -132.813F,
   37.8418F,
   -72.5098F,
   42.4805F,
   -72.5098F,
   9.27734F,
   -1.95313F,
   28.5645F,
   -92.7734F,
   48.0957F,
   -110.352F,
   58.5938F,
   -127.441F,
   54.4434F,
   -25.1465F,
   26.8555F,
   -45.6543F,
   21.2402F,
   -37.8418F,
   41.2598F,
   -11.4746F,
   38.3301F,
   6.34766F,
   -20.2637F,
   -24.4141F,
   31.2500F,
   19.5313F,
   51.0254F,
   -15.1367F,
   1.46484F,
   1.70898F,
   5.37109F,
   1.22070F,
   77.6367F,
   -4.88281F,
   60.5469F,
   -10.9863F,
   -48.8281F,
   -15.3809F,
   38.0859F,
   -16.8457F,
   61.0352F,
   -44.6777F,
   90.0879F,
   -30.7617F,
   -27.5879F,
   -18.5547F,
   -34.6680F,
   -29.7852F,
   -15.6250F,
   -62.0117F,
   -7.56836F,
   8.54492F,
   -43.2129F,
   -17.3340F,
   -1.70898F,
   41.7480F,
   22.7051F,
   6.59180F,
   26.3672F,
   51.7578F,
   -80.5664F,
   -51.0254F,
   -30.0293F,
   -17.0898F,
   -68.1152F,
   23.1934F,
   -23.4375F,
   4.88281F,
   -45.8984F,
   -14.8926F,
   -76.6602F,
   21.2402F,
   -73.2422F,
   -19.0430F,
   -86.4258F,
   -32.7148F,
   2.68555F,
   29.7852F,
   -34.1797F,
   29.7852F,
   -67.1387F,
   37.1094F,
   -71.5332F,
   34.1797F,
   -20.0195F,
   33.6914F,
   -78.3691F,
   -27.0996F,
   -117.188F,
   -38.0859F,
   -87.6465F,
   18.5547F,
   -62.0117F,
   -9.76563F,
   -155.029F,
   -23.4375F,
   -127.441F,
   19.2871F,
   -123.779F,
   1.95313F,
   -65.4297F,
   73.9746F,
   -131.592F,
   16.6016F,
   -108.887F,
   14.8926F,
   -127.441F,
   74.7070F,
   27.0996F,
   46.1426F,
   -106.201F,
   29.7852F,
   -92.5293F,
   40.5273F,
   -139.404F,
   -97.1680F,
   -154.297F,
   -18.0664F,
   -182.373F,
   -23.1934F,
   -111.084F,
   47.3633F,
   -232.422F,
   20.2637F,
   -194.824F,
   46.8750F,
   -184.326F,
   46.8750F,
   -190.674F,
   -39.5508F,
   -151.123F,
   57.1289F,
   -161.865F,
   -72.5098F,
   -119.141F,
   -26.6113F,
   -235.352F,
   -32.2266F,
   -204.590F,
   -16.6016F,
   -205.811F,
   14.1602F,
   -271.484F,
   -20.9961F,
   -196.533F,
   -72.9980F,
   -230.469F,
   -61.7676F,
   -189.941F,
   -12.2070F,
   -235.596F,
   -134.033F,
   -85.9375F,
   -23.9258F,
   -242.188F,
   -83.7402F,
   -272.705F,
   -76.9043F,
   -272.705F,
   -74.9512F,
   -281.982F,
   -91.3086F,
   -155.518F,
   -56.1523F,
   -284.668F,
   -10.4980F,
   -317.139F,
   -24.4141F,
   -225.830F,
   -95.9473F,
   -311.035F,
   -146.484F,
   -168.213F,
   -31.7383F,
   -361.084F,
   -76.1719F,
   -322.510F,
   -62.0117F,
   -357.422F,
   -107.910F,
   -315.430F,
   -149.658F,
   -307.861F,
   -122.803F,
   -366.455F,
   -89.8438F,
   -322.754F,
   6.34766F,
   -349.609F,
   -16.1133F,
   -425.537F,
   -39.3066F,
   -401.367F,
   -114.014F,
   -429.688F,
   -133.789F,
   -340.088F,
   -138.672F,
   -379.883F,
   -212.646F,
   -364.990F,
   -252.441F,
   -338.623F,
   -139.404F,
   -468.018F,
   -128.906F,
   -435.303F,
   -30.0293F,
   -463.135F,
   -56.3965F,
   -501.465F,
   -78.8574F,
   -500.977F,
   -221.191F,
   -482.422F,
   -138.428F,
   -468.018F,
   -151.367F,
   -499.756F,
   -241.455F,
   -262.939F,
   -90.3320F,
   -495.850F,
   -171.875F,
   -574.951F,
   -182.861F,
   -668.945F,
   -265.869F,
   -466.064F,
   38.8184F,
   -491.211F,
   60.5469F,
   -152.832F,
   -30.0293F,
   -571.045F,
   -234.863F,
   -163.330F,
   -99.6094F,
   -336.670F,
   -286.621F,
   -110.352F,
   -88.8672F,
   -254.883F,
   -179.443F,
   -32.2266F,
   44.6777F,
   -395.508F,
   -183.594F,
   -133.545F,
   -74.9512F,
   -189.697F,
   -307.861F,
   -23.9258F,
   10.0098F,
   -214.844F,
   -266.357F,
   -62.7441F,
   23.6816F,
   -391.113F,
   -447.510F,
   7.56836F,
   -6.34766F,
   -157.227F,
   -136.963F,
   -43.9453F,
   -133.301F,
   -93.9941F,
   -267.334F,
   -100.098F,
   -195.801F,
   -101.074F,
   -201.904F,
   -111.572F,
   -236.816F,
   -119.629F,
   -270.752F,
   -52.4902F,
   -223.633F,
   -35.1563F,
   -228.760F,
   -120.361F,
   -309.814F,
   -126.221F,
   -367.920F,
   44.1895F,
   24.6582F,
   -81.0547F,
   -217.041F,
   -204.102F,
   -228.760F,
   -136.475F,
   -104.736F,
   -153.564F,
   -133.545F,
   -44.6777F,
   -82.2754F,
   -133.057F,
   -20.0195F,
   -61.0352F,
   -69.8242F,
   1.22070F,
   -32.2266F,
   -84.9609F,
   -61.5234F,
   -71.5332F,
   -115.234F,
   -38.5742F,
   24.4141F,
   -7.08008F,
   48.0957F,
   -57.6172F,
   -103.516F,
   -210.205F,
   -52.0020F,
   -34.1797F,
   -1.70898F,
   -104.248F,
   -108.154F,
   45.6543F,
   -23.6816F,
   -166.992F,
   -179.688F,
   -71.5332F,
   62.9883F,
   -89.8438F,
   -37.1094F,
   -36.6211F,
   95.7031F,
   -148.682F,
   42.7246F,
   -34.6680F,
   72.9980F,
   -33.6914F,
   37.1094F,
   -29.0527F,
   80.3223F,
   -118.652F,
   -12.6953F,
   71.5332F,
   48.3398F,
   -44.6777F,
   28.5645F,
   42.7246F,
   80.8105F,
   -14.1602F,
   -66.8945F,
   56.3965F,
   73.2422F,
   -70.3125F,
   80.5664F,
   -74.4629F,
   90.8203F,
   -27.0996F,
   99.8535F,
   -2.19727F,
   103.271F,
   20.2637F,
   62.5000F,
   16.3574F,
   89.5996F,
   -4.63867F,
   60.5469F,
   22.2168F,
   27.5879F,
   -8.54492F,
   99.1211F,
   -46.6309F,
   37.5977F,
   58.1055F,
   72.2656F,
   1.22070F,
   48.0957F,
   34.4238F,
   53.9551F,
   76.4160F,
   48.3398F,
   51.5137F,
   102.783F,
   59.5703F,
   81.5430F,
   21.4844F,
   104.004F,
   -59.3262F,
   110.840F,
   49.3164F,
   134.766F,
   -1.22070F,
   98.3887F,
   71.0449F,
   45.1660F,
   53.4668F,
   73.4863F,
   61.2793F,
   33.6914F,
   31.2500F,
   16.8457F,
   48.0957F,
   70.3125F,
   -34.1797F,
   -14.8926F,
   45.8984F,
   88.1348F,
   48.0957F,
   145.996F,
   107.910F,
   66.6504F,
   70.8008F,
   34.9121F,
   115.234F,
   117.676F,
   38.3301F,
   90.3320F,
   101.318F,
   78.3691F,
   90.8203F,
   93.9941F,
   98.1445F,
   134.766F,
   37.8418F,
   5.85938F,
   134.277F,
   64.2090F,
   -2.68555F,
   5.12695F,
   87.8906F,
   55.4199F,
   35.8887F,
   -62.0117F,
   103.516F,
   23.6816F,
   89.3555F,
   -3.17383F,
   91.5527F,
   34.4238F,
   109.619F,
   56.6406F,
   96.6797F,
   123.779F,
   115.723F,
   66.4063F,
   171.143F,
   79.1016F,
   88.3789F,
   -11.4746F,
   143.311F,
   36.1328F,
   132.568F,
   16.8457F,
   97.6563F,
   -12.4512F,
   136.963F,
   14.4043F,
   53.7109F,
   -2.44141F,
   85.9375F,
   35.8887F,
   50.2930F,
   51.5137F,
   159.424F,
   45.1660F,
   137.451F,
   72.5098F,
   137.939F,
   69.3359F,
   145.020F,
   29.5410F,
   187.012F,
   46.8750F,
   97.1680F,
   28.8086F,
   156.738F,
   105.957F,
   56.8848F,
   64.4531F,
   117.432F,
   114.014F,
   31.4941F,
   -40.2832F,
   170.654F,
   58.3496F,
   21.9727F,
   6.34766F,
   83.4961F,
   115.723F,
   -13.4277F,
   6.59180F,
   94.7266F,
   22.9492F,
   -41.9922F,
   0.000000F,
   177.002F,
   92.5293F,
   -14.6484F,
   82.2754F,
   90.3320F,
   113.525F,
   23.1934F,
   77.8809F,
   196.777F,
   145.264F,
   19.0430F,
   63.4766F,
   121.338F,
   207.764F,
   51.2695F,
   136.719F,
   111.816F,
   140.137F,
   -113.281F,
   49.3164F,
   121.338F,
   152.588F,
   -49.3164F,
   37.1094F,
   11.7188F,
   173.828F,
   -4.88281F,
   138.184F,
   24.4141F,
   174.561F,
   111.084F,
   114.258F,
   100.342F,
   147.705F,
   77.8809F,
   157.715F,
   47.6074F,
   150.146F,
   97.9004F,
   131.348F,
   166.016F,
   180.420F,
   49.0723F,
   162.842F,
   105.957F,
   232.910F,
   110.840F,
   103.760F,
   157.715F,
   119.873F,
   147.949F,
   166.260F,
   101.563F,
   124.023F,
   121.338F,
   200.684F,
   104.004F,
   198.975F,
   161.133F,
   157.959F,
   153.320F,
   174.805F,
   170.166F,
   113.770F,
   150.879F,
   111.572F,
   167.236F,
   112.305F,
   89.1113F,
   75.4395F,
   176.025F,
   138.428F,
   204.102F,
   146.729F,
   148.682F,
   73.2422F,
   201.416F,
   112.061F,
   230.225F,
   167.725F,
   166.260F,
   130.127F,
   223.389F,
   145.996F,
   144.287F,
   59.3262F,
   213.867F,
   110.107F,
   213.379F,
   102.539F,
   191.895F,
   77.3926F,
   178.711F,
   53.7109F,
   225.098F,
   77.3926F,
   270.508F,
   89.5996F,
   129.639F,
   113.770F,
   250.977F,
   158.447F,
   257.080F,
   150.146F,
   252.441F,
   135.010F,
   202.393F,
   146.973F,
   249.268F,
   195.068F,
   226.318F,
   196.045F,
   214.355F,
   186.279F,
   195.068F,
   121.094F,
   335.205F,
   188.721F,
   142.822F,
   187.988F,
   196.045F,
   227.051F,
   268.311F,
   193.604F,
   298.340F,
   210.449F,
   295.166F,
   218.506F,
   250.244F,
   177.490F,
   188.477F,
   206.299F,
   286.133F,
   272.217F,
   211.670F,
   249.268F,
   202.637F,
   247.314F,
   205.322F,
   222.168F,
   123.535F,
   171.631F,
   302.490F,
   262.939F,
   151.367F,
   199.951F,
   291.992F,
   264.404F,
   281.982F,
   263.916F,
   278.809F,
   221.436F,
   377.686F,
   273.682F,
   319.580F,
   158.203F,
   327.881F,
   149.414F,
   362.305F,
   241.211F,
   361.084F,
   228.760F,
   240.479F,
   324.219F,
   233.154F,
   327.393F,
   104.736F,
   222.168F,
   205.322F,
   326.660F,
   137.695F,
   287.842F,
   100.586F,
   282.227F,
   348.389F,
   322.266F,
   350.098F,
   324.707F,
   156.250F,
   185.547F,
   421.387F,
   344.238F,
   46.3867F,
   135.498F,
   261.963F,
   245.361F,
   104.004F,
   62.7441F,
   204.834F,
   239.258F,
   57.3730F,
   56.3965F,
   371.094F,
   284.912F,
   26.6113F,
   71.5332F,
   247.559F,
   383.057F,
   74.4629F,
   34.6680F,
   280.273F,
   131.592F,
   -71.0449F,
   -26.3672F,
   296.143F,
   237.305F,
   5.37109F,
   -52.7344F,
   162.842F,
   202.148F,
   -117.676F,
   106.934F,
   110.596F,
   349.365F,
   -141.846F,
   -103.027F,
   192.627F,
   94.4824F,
   -87.4023F,
   -110.840F,
   42.4805F,
   190.430F,
   -8.78906F,
   -90.8203F,
   95.2148F,
   -32.7148F,
   -153.564F,
   39.0625F,
   -74.7070F,
   183.350F,
   -307.129F,
   -80.8105F,
   43.2129F,
   127.441F,
   -60.5469F,
   140.137F,
   -61.2793F,
   156.006F,
   -129.639F,
   99.3652F,
   -145.508F,
   96.1914F,
   -102.295F,
   192.627F,
   -150.635F,
   195.557F,
   -240.723F,
   97.4121F,
   -209.229F,
   177.490F,
   -1.70898F,
   126.465F,
   -171.631F,
   75.6836F,
   -279.053F,
   -5.85938F,
   -244.629F,
   70.0684F,
   -234.375F,
   88.6230F,
   -317.139F,
   76.1719F,
   -374.512F,
   59.8145F,
   -380.127F,
   74.4629F,
   6.83594F,
   37.3535F,
   -209.717F,
   -42.7246F,
   -8.05664F,
   81.0547F,
   -341.309F,
   -37.5977F,
   51.7578F,
   100.098F,
   -144.775F,
   -48.0957F,
   -266.602F,
   -171.875F,
   -220.703F,
   -15.8691F,
   68.8477F,
   89.5996F,
   -224.121F,
   -167.480F,
   84.2285F,
   22.7051F,
   -62.9883F,
   -87.1582F,
   169.922F,
   157.227F,
   -169.189F,
   -6.83594F,
   109.375F,
   120.361F,
   -66.6504F,
   47.1191F,
   128.662F,
   133.301F,
   -59.3262F,
   -125.244F,
   93.7500F,
   -33.2031F,
   66.6504F,
   -86.1816F,
   125.000F,
   -34.6680F,
   131.104F,
   -48.3398F,
   229.736F,
   183.105F,
   20.2637F,
   60.5469F,
   141.113F,
   210.205F,
   -13.6719F,
   144.531F,
   205.566F,
   10.7422F,
   217.773F,
   5.85938F,
   8.05664F,
   217.285F,
   -3.90625F,
   239.746F,
   202.881F,
   341.309F,
   374.756F,
   463.379F,
   418.945F,
   335.938F,
   475.586F,
   357.666F
};
static const Float32 dico5_lsf_5[DICO5_SIZE_5 * 4] =
{
   -244.629F,
   -226.807F,
   -267.578F,
   -293.701F,
   -156.494F,
   -227.295F,
   -147.461F,
   -234.619F,
   -190.186F,
   -164.307F,
   -203.857F,
   -192.383F,
   -101.563F,
   -162.109F,
   -111.816F,
   -187.012F,
   -159.180F,
   -127.197F,
   -161.621F,
   -120.850F,
   -249.756F,
   -124.268F,
   -249.756F,
   -104.492F,
   -108.398F,
   -134.766F,
   -89.8438F,
   -109.619F,
   -116.943F,
   -51.5137F,
   -257.324F,
   -220.459F,
   -77.1484F,
   -60.7910F,
   -138.916F,
   -144.287F,
   -138.916F,
   -67.1387F,
   -132.080F,
   -46.6309F,
   -174.805F,
   -45.8984F,
   -205.566F,
   -64.4531F,
   -81.2988F,
   -60.5469F,
   -77.6367F,
   -55.6641F,
   -67.1387F,
   0.244141F,
   -138.428F,
   -55.6641F,
   -28.0762F,
   -53.9551F,
   -58.1055F,
   -91.3086F,
   -48.0957F,
   -123.779F,
   -54.1992F,
   -141.357F,
   -62.9883F,
   -105.469F,
   -14.8926F,
   -59.5703F,
   -84.2285F,
   0.488281F,
   -82.5195F,
   9.52148F,
   -52.4902F,
   -41.2598F,
   -14.1602F,
   0.000000F,
   -13.6719F,
   -1.46484F,
   -49.5605F,
   -31.9824F,
   0.244141F,
   -45.4102F,
   -1.22070F,
   -51.5137F,
   1.46484F,
   -92.7734F,
   2.68555F,
   -102.051F,
   -28.3203F,
   31.9824F,
   -32.7148F,
   27.5879F,
   21.7285F,
   -0.976563F,
   17.3340F,
   -0.488281F,
   -4.63867F,
   -46.8750F,
   63.9648F,
   5.85938F,
   46.1426F,
   36.8652F,
   -32.4707F,
   -26.6113F,
   45.4102F,
   -37.3535F,
   40.5273F,
   -53.4668F,
   9.03320F,
   33.9355F,
   47.1191F,
   41.7480F,
   82.2754F,
   30.2734F,
   38.5742F,
   -14.8926F,
   34.4238F,
   55.1758F,
   -3.17383F,
   46.3867F,
   56.3965F,
   8.30078F,
   86.4258F,
   26.6113F,
   77.1484F,
   49.0723F,
   59.5703F,
   40.0391F,
   80.5664F,
   -20.7520F,
   95.2148F,
   -20.5078F,
   62.0117F,
   79.8340F,
   62.7441F,
   81.7871F,
   119.873F,
   35.8887F,
   116.211F,
   25.6348F,
   13.1836F,
   18.7988F,
   106.689F,
   90.3320F,
   102.783F,
   76.6602F,
   109.619F,
   83.4961F,
   80.3223F,
   30.7617F,
   164.307F,
   71.2891F,
   139.404F,
   94.7266F,
   59.3262F,
   47.1191F,
   159.424F,
   78.1250F,
   151.611F,
   68.3594F,
   47.3633F,
   92.7734F,
   126.221F,
   141.846F,
   10.9863F,
   78.8574F,
   27.0996F,
   103.027F,
   119.385F,
   96.4355F,
   179.199F,
   130.371F,
   151.855F,
   133.301F,
   118.652F,
   122.559F,
   77.6367F,
   139.648F,
   46.1426F,
   134.277F,
   93.9941F,
   103.027F,
   -38.3301F,
   37.3535F,
   -30.5176F,
   93.2617F,
   -48.0957F,
   94.2383F,
   -64.2090F,
   81.5430F,
   55.6641F,
   170.166F,
   -45.8984F,
   0.244141F,
   12.4512F,
   72.5098F,
   -123.779F,
   52.0020F,
   -91.7969F,
   96.9238F,
   -5.85938F,
   62.2559F,
   -133.545F,
   21.7285F,
   -122.559F,
   -22.9492F,
   94.4824F,
   43.7012F,
   -151.367F,
   16.6016F,
   -166.992F,
   27.3438F,
   -156.738F,
   -85.4492F,
   -63.4766F,
   41.9922F,
   -106.934F,
   -79.1016F,
   64.4531F,
   158.203F,
   -235.352F,
   -0.976563F,
   -273.682F,
   1.70898F,
   -32.7148F,
   32.7148F,
   -276.611F,
   -74.7070F,
   34.9121F,
   23.4375F,
   -102.539F,
   -121.338F,
   -298.096F,
   -85.4492F,
   -372.803F,
   -167.236F,
   -39.3066F,
   17.5781F,
   213.135F,
   168.701F,
   178.711F,
   69.0918F,
   224.854F,
   86.1816F,
   81.5430F,
   115.967F,
   267.334F,
   200.439F,
   210.938F,
   127.930F,
   205.811F,
   121.338F,
   174.316F,
   173.584F,
   192.383F,
   183.105F,
   262.695F,
   174.316F,
   293.945F,
   183.838F,

};

/* Table for MR475 gain */
#define MR475_VQ_SIZE 256

/* The table contains the following data:
 *
 *    g_pitch(0)         // for sub-
 *    g_fac(0)           // frame 0 and 2
 *    g_pitch(1)         // for sub-
 *    g_fac(1)           // frame 1 and 3
 *
 */
static const Float32 table_gain_MR475[MR475_VQ_SIZE * 4] =
{
0.049561F, 0.031250F,
0.033081F, 0.034180F,
0.175354F, 0.277100F,
0.138306F, 0.830566F,
0.126160F, 0.137451F,
0.773743F, 0.157959F,
0.252197F, 0.438965F,
0.341858F, 1.290283F,
0.469299F, 0.091309F,
0.227966F, 0.107666F,
0.666016F, 0.644043F,
0.720642F, 0.608887F,
1.250610F, 0.194580F,
0.318481F, 0.164795F,
0.410400F, 2.039551F,
0.322388F, 0.414063F,
0.090820F, 0.104492F,
0.359009F, 0.110352F,
0.325439F, 0.994141F,
0.218689F, 0.309570F,
0.150696F, 0.219971F,
0.970093F, 0.245361F,
0.914429F, 0.798584F,
0.630554F, 1.186035F,
0.221863F, 0.493408F,
0.158447F, 0.203857F,
0.754395F, 1.194336F,
0.744995F, 0.416016F,
0.819824F, 0.391602F,
0.556519F, 0.571533F,
0.242188F, 0.375488F,
0.334412F, 2.425781F,
0.231628F, 0.101807F,
0.082947F, 0.101074F,
0.222168F, 0.383057F,
0.487976F, 0.864502F,
0.696106F, 0.157471F,
0.521973F, 0.155029F,
0.247559F, 0.336182F,
1.013672F, 1.495117F,
0.697021F, 0.170898F,
0.122498F, 0.148193F,
0.757751F, 0.385254F,
0.678650F, 1.136230F,
0.834961F, 0.416992F,
0.731812F, 0.300049F,
0.488037F, 1.781494F,
0.807556F, 1.395264F,
0.148193F, 0.282959F,
0.126526F, 0.473877F,
0.379517F, 1.494385F,
0.216431F, 0.440430F,
0.544739F, 0.439941F,
0.529724F, 0.385742F,
0.850525F, 0.873047F,
0.812561F, 1.522705F,
0.374878F, 0.275879F,
0.365845F, 0.423340F,
0.863098F, 1.138184F,
0.687927F, 0.810791F,
0.746216F, 0.378662F,
0.850281F, 0.736084F,
0.310120F, 2.554688F,
0.576416F, 1.637207F,
0.103088F, 0.166748F,
0.087646F, 0.318848F,
0.440186F, 0.960205F,
0.249146F, 0.662354F,
0.475647F, 0.171875F,
0.919800F, 0.195801F,
0.384460F, 1.272461F,
0.264709F, 1.307861F,
0.407471F, 0.132080F,
0.369995F, 0.152832F,
0.833191F, 0.903320F,
0.701782F, 0.587891F,
0.986084F, 0.174805F,
0.743225F, 0.183350F,
0.492249F, 2.804932F,
0.385376F, 0.550781F,
0.272583F, 0.121094F,
0.445129F, 0.127686F,
0.623352F, 0.935791F,
0.512329F, 0.741455F,
0.512878F, 0.235840F,
0.868408F, 0.458984F,
0.531189F, 1.320557F,
1.000671F, 1.187256F,
0.452881F, 0.483154F,
0.371643F, 0.300293F,
0.571960F, 1.073730F,
0.888550F, 0.821045F,
0.827576F, 0.701416F,
0.803406F, 0.357666F,
0.337769F, 0.409668F,
0.801880F, 3.606201F,
0.447876F, 0.146484F,
0.080444F, 0.133789F,
0.259521F, 0.864014F,
0.613037F, 0.577148F,
0.646179F, 0.151855F,
0.798828F, 0.163330F,
0.862183F, 0.860840F,
0.307556F, 2.388672F,
0.936157F, 0.151123F,
0.190125F, 0.144043F,
1.003540F, 0.735596F,
0.948608F, 1.017578F,
0.948303F, 0.393311F,
0.940247F, 0.299805F,
0.977966F, 2.270264F,
0.459839F, 1.214844F,
0.121460F, 0.296143F,
0.698669F, 0.282471F,
0.763672F, 1.347412F,
0.578308F, 0.645508F,
0.470947F, 0.496582F,
0.810547F, 0.546631F,
0.977234F, 1.243408F,
0.492310F, 1.650635F,
0.615417F, 0.344971F,
0.305298F, 0.572998F,
0.764343F, 1.458740F,
0.923218F, 0.707764F,
1.114746F, 0.558838F,
0.966003F, 0.616943F,
0.992737F, 2.503418F,
0.894226F, 2.763428F,
0.109680F, 0.082275F,
0.190125F, 0.096924F,
0.214233F, 0.714844F,
0.280273F, 0.651855F,
0.458923F, 0.153320F,
0.696716F, 0.160156F,
0.362915F, 0.594482F,
0.399414F, 1.798584F,
0.502808F, 0.202393F,
0.244141F, 0.210693F,
0.612305F, 0.608398F,
0.980042F, 0.866943F,
1.111084F, 0.257324F,
0.372498F, 0.354980F,
0.359131F, 1.928711F,
1.144531F, 0.846680F,
0.113770F, 0.132813F,
0.561401F, 0.166748F,
0.709412F, 1.015625F,
0.280396F, 0.401367F,
0.192749F, 0.282471F,
0.973694F, 0.625000F,
0.753723F, 0.911377F,
1.063232F, 1.284180F,
0.372681F, 0.489258F,
0.178040F, 0.425293F,
1.005066F, 1.283447F,
1.024597F, 0.410156F,
1.050110F, 0.429443F,
0.291321F, 0.788818F,
0.450806F, 1.473389F,
0.875366F, 2.444336F,
0.246277F, 0.107910F,
0.255981F, 0.111816F,
0.562378F, 0.547363F,
0.453308F, 1.029541F,
0.784912F, 0.195557F,
0.682739F, 0.201416F,
0.771973F, 0.508789F,
0.790771F, 1.600098F,
0.580139F, 0.243164F,
0.404602F, 0.240479F,
0.643127F, 0.610840F,
0.933167F, 1.222168F,
0.769165F, 0.501709F,
0.945068F, 0.403564F,
0.988403F, 1.689697F,
0.868591F, 1.413574F,
0.160278F, 0.202148F,
0.342712F, 0.411621F,
0.833923F, 1.410645F,
0.223877F, 0.379395F,
0.690491F, 0.642822F,
0.596313F, 0.356201F,
0.854675F, 1.155518F,
0.970276F, 1.535889F,
0.383179F, 0.456543F,
0.482788F, 0.557861F,
1.036255F, 1.115967F,
1.011719F, 0.939697F,
0.930664F, 0.564209F,
0.977966F, 0.791992F,
0.882507F, 4.347656F,
0.723083F, 0.674561F,
0.120911F, 0.343506F,
0.085449F, 0.213867F,
0.264587F, 0.865967F,
0.268005F, 1.027832F,
0.329895F, 0.166016F,
1.065735F, 0.190674F,
0.396790F, 1.249512F,
0.493835F, 1.874268F,
0.448914F, 0.193848F,
0.508606F, 0.288574F,
0.918030F, 0.771484F,
0.911133F, 0.741943F,
1.246399F, 0.214844F,
0.887756F, 0.208008F,
0.752991F, 3.590820F,
0.421387F, 0.468750F,
0.257874F, 0.227783F,
0.501587F, 0.265381F,
0.650574F, 0.997070F,
0.615356F, 1.106689F,
0.166931F, 0.205078F,
1.260803F, 0.263916F,
1.019958F, 1.456543F,
0.968811F, 1.117676F,
0.663513F, 0.628418F,
0.230286F, 0.284668F,
0.886169F, 0.987305F,
1.263367F, 0.641357F,
0.928894F, 0.667480F,
0.932251F, 0.533691F,
0.381897F, 0.787598F,
0.801086F, 4.755859F,
0.439209F, 0.227051F,
0.150269F, 0.395020F,
0.275574F, 0.754883F,
0.845459F, 1.044189F,
0.638428F, 0.203369F,
1.058289F, 0.197754F,
1.030945F, 0.558838F,
0.948853F, 2.006104F,
0.830261F, 0.411133F,
0.195129F, 0.447754F,
0.973389F, 0.688477F,
0.965088F, 1.300537F,
1.037964F, 0.595215F,
1.024658F, 0.323730F,
0.956482F, 1.991211F,
0.715698F, 2.088867F,
0.229614F, 0.501221F,
0.583679F, 0.321533F,
0.827698F, 1.657715F,
0.746277F, 0.472656F,
0.499268F, 0.866699F,
0.810974F, 0.434082F,
0.994812F, 1.611084F,
0.994324F, 1.894043F,
0.520081F, 0.622803F,
0.440979F, 0.645752F,
1.115051F, 1.828369F,
1.030579F, 0.548828F,
1.091431F, 0.704102F,
1.053772F, 0.812500F,
0.574768F, 4.922363F,
0.673950F, 2.031250F,
0.078491F, 0.151367F,
0.087341F, 0.142334F,
0.365784F, 0.558838F,
0.242798F, 0.885254F,
0.313965F, 0.183594F,
0.818420F, 0.202637F,
0.338928F, 0.698242F,
0.718018F, 1.442383F,
0.655334F, 0.136719F,
0.332397F, 0.137695F,
0.813049F, 0.734375F,
0.729126F, 0.899170F,
1.213806F, 0.194824F,
0.599670F, 0.177734F,
0.833923F, 2.135742F,
0.451111F, 0.745361F,
0.153503F, 0.189941F,
0.369263F, 0.203369F,
0.394836F, 1.238770F,
0.506897F, 0.601318F,
0.374817F, 0.455322F,
0.934326F, 0.308105F,
0.879395F, 1.110107F,
0.833923F, 1.102295F,
0.191467F, 0.728271F,
0.151306F, 0.307373F,
0.920898F, 1.134521F,
0.938843F, 0.636475F,
0.884521F, 0.583984F,
0.760620F, 0.619385F,
0.460144F, 0.719971F,
0.787903F, 2.944336F,
0.333740F, 0.166992F,
0.191711F, 0.176270F,
0.310120F, 0.311035F,
0.777100F, 1.025391F,
0.933960F, 0.166260F,
0.477234F, 0.144531F,
0.398804F, 0.493408F,
1.005737F, 2.133057F,
0.815674F, 0.215332F,
0.329407F, 0.219482F,
0.894531F, 0.531738F,
0.899719F, 1.031982F,
0.870972F, 0.316895F,
0.850159F, 0.495361F,
0.944641F, 1.826660F,
0.965271F, 1.116211F,
0.153870F, 0.491455F,
0.308960F, 0.440430F,
0.314880F, 1.606934F,
0.435181F, 0.878174F,
0.666504F, 0.393311F,
0.715881F, 0.416992F,
1.031677F, 0.843506F,
0.992920F, 1.621094F,
0.567993F, 0.245850F,
0.571838F, 0.514160F,
1.170776F, 1.229736F,
0.759338F, 1.042236F,
0.971619F, 0.325195F,
0.937317F, 0.857422F,
0.726196F, 3.452393F,
0.982727F, 1.673340F,
0.122681F, 0.179932F,
0.230652F, 0.210205F,
0.699097F, 0.703125F,
0.217529F, 0.864258F,
0.552795F, 0.302979F,
0.756287F, 0.218750F,
0.521606F, 1.130127F,
0.705627F, 1.410156F,
0.496155F, 0.143799F,
0.501587F, 0.143555F,
1.143066F, 0.916748F,
0.791809F, 0.768799F,
0.960022F, 0.185059F,
1.015259F, 0.184082F,
0.929077F, 2.719238F,
0.968689F, 0.539063F,
0.285217F, 0.148926F,
0.623657F, 0.165527F,
0.931213F, 1.012207F,
0.349670F, 0.812256F,
0.511292F, 0.407715F,
1.212280F, 0.566650F,
0.942993F, 1.345459F,
0.857788F, 1.338135F,
0.349609F, 0.705078F,
0.462646F, 0.328613F,
0.877930F, 1.300049F,
0.990967F, 0.964355F,
0.922729F, 0.914063F,
0.934204F, 0.350342F,
0.890930F, 0.993652F,
0.750793F, 3.832031F,
0.464905F, 0.414795F,
0.132446F, 0.215820F,
0.272156F, 1.114990F,
1.104370F, 0.807129F,
0.777710F, 0.198975F,
0.911011F, 0.221436F,
0.916504F, 1.045166F,
0.947144F, 2.042969F,
1.098267F, 0.265381F,
0.143921F, 0.211182F,
0.993713F, 0.924561F,
0.956970F, 1.122070F,
0.998291F, 0.374512F,
1.012207F, 0.539551F,
1.027405F, 2.367432F,
0.972229F, 1.101807F,
0.200012F, 0.264893F,
0.588562F, 0.533936F,
0.730774F, 1.481445F,
0.561462F, 1.089355F,
0.546570F, 0.397705F,
0.627991F, 0.747559F,
1.005920F, 1.258545F,
0.943970F, 1.725098F,
0.834839F, 0.620850F,
0.321838F, 0.555908F,
1.022095F, 1.516846F,
1.016541F, 0.832031F,
1.239258F, 0.821045F,
1.140625F, 0.484619F,
0.864990F, 3.141357F,
0.924927F, 3.833740F,
0.273010F, 0.249023F,
0.102600F, 0.216309F,
0.263123F, 1.050049F,
0.546387F, 0.892822F,
0.359680F, 0.280029F,
0.710876F, 0.354492F,
0.969604F, 0.543701F,
0.279663F, 1.622070F,
0.422913F, 0.294189F,
0.379639F, 0.195068F,
0.757751F, 0.832275F,
0.974609F, 0.946533F,
1.212097F, 0.514893F,
0.591370F, 0.522705F,
0.899780F, 2.155762F,
0.883789F, 0.634521F,
0.110901F, 0.302246F,
0.474304F, 0.198486F,
1.164490F, 1.076660F,
0.338989F, 0.503906F,
0.225037F, 0.694336F,
1.064148F, 0.550781F,
1.019104F, 1.095215F,
0.988708F, 1.315430F,
0.489990F, 0.674561F,
0.207825F, 0.517334F,
1.063599F, 1.337158F,
0.836060F, 0.680176F,
1.213318F, 0.664063F,
0.555298F, 0.947266F,
1.109131F, 1.179932F,
1.058105F, 2.980225F,
0.312256F, 0.243164F,
0.301208F, 0.241211F,
0.603516F, 0.752197F,
0.367065F, 1.311279F,
0.969299F, 0.406982F,
0.513000F, 0.288818F,
0.920837F, 0.577637F,
1.207092F, 1.709473F,
0.730164F, 0.381348F,
0.444275F, 0.275391F,
1.028992F, 0.374023F,
0.962036F, 1.313721F,
0.897705F, 0.504150F,
1.227112F, 0.526123F,
1.047241F, 2.012939F,
1.093201F, 1.542480F,
0.333679F, 0.367188F,
0.250244F, 0.628418F,
1.063293F, 1.662598F,
0.346252F, 0.705078F,
1.015381F, 0.825684F,
0.547791F, 0.447021F,
1.230408F, 1.156494F,
1.193237F, 1.780762F,
0.406372F, 0.678955F,
0.700195F, 0.810791F,
1.183899F, 1.233887F,
1.152222F, 1.152832F,
1.003357F, 0.579346F,
1.246948F, 1.070801F,
0.692932F, 6.476074F,
0.707581F, 0.749023F,
0.174927F, 0.382324F,
0.311768F, 0.261230F,
0.586792F, 1.199951F,
0.301453F, 0.863281F,
0.460266F, 0.214355F,
1.264465F, 0.207764F,
0.423462F, 1.072998F,
1.025330F, 1.887939F,
0.618713F, 0.248779F,
0.600891F, 0.235352F,
0.945679F, 0.965576F,
0.943542F, 0.837402F,
1.151306F, 0.239746F,
1.228027F, 0.235107F,
1.030029F, 3.146240F,
0.874878F, 1.025391F,
0.402771F, 0.297852F,
0.562866F, 0.198730F,
1.034058F, 1.253418F,
0.345520F, 1.195801F,
0.334961F, 0.438965F,
1.236450F, 0.967285F,
1.037903F, 1.508301F,
1.094299F, 1.447510F,
0.571594F, 0.834961F,
0.456177F, 0.481201F,
1.192444F, 1.263916F,
1.159851F, 0.733887F,
1.004272F, 0.924805F,
0.980835F, 0.577881F,
0.528809F, 0.669678F,
0.576477F, 6.435059F,
0.662598F, 0.314209F,
0.216980F, 0.246338F,
0.342163F, 0.884277F,
1.187317F, 1.345703F,
0.760071F, 0.194580F,
1.259399F, 0.222412F,
0.941589F, 0.748535F,
1.039856F, 2.508545F,
1.152527F, 0.642822F,
0.238831F, 0.309570F,
1.191345F, 0.822998F,
1.101807F, 1.276855F,
1.177429F, 0.409668F,
1.190674F, 0.774414F,
1.103027F, 2.625488F,
1.014709F, 1.671143F,
0.191284F, 0.561035F,
0.663391F, 0.594971F,
0.950928F, 1.687744F,
0.768860F, 0.825439F,
0.678467F, 0.804932F,
1.023071F, 0.591797F,
1.150696F, 1.639404F,
1.048035F, 2.413818F,
0.777771F, 0.635986F,
0.545471F, 0.766602F,
1.161682F, 1.895508F,
1.119812F, 0.947266F,
1.249695F, 0.898926F,
1.196411F, 0.825195F,
0.796143F, 4.729736F,
0.642456F, 5.645508F,

};

/*
 *    g_pitch        ,
 *    g_fac          , (g_code = g_code0*g_fac),
 */
/* table used in 'high' rates: MR67 MR74 MR102 */
#define VQ_SIZE_HIGHRATES 128
static const Float32 table_highrates[VQ_SIZE_HIGHRATES * 2] =
{
   /*g_pit,    g_fac,   */
   0.0352173F,   0.161621F,
   0.0491943F,   0.448242F,
   0.189758F,   0.256836F,
   0.255188F,   0.338623F,
   0.144836F,   0.347900F,
   0.198242F,   0.484619F,
   0.111511F,   0.566406F,
   0.0574341F,   0.809082F,
   0.143494F,   0.726807F,
   0.220703F,   0.590820F,
   0.210632F,   0.755859F,
   0.180359F,   1.05005F,
   0.112793F,   1.09863F,
   0.237061F,   1.32227F,
   0.0724487F,   1.76025F,
   0.188171F,   2.19727F,
   0.450684F,   0.215576F,
   0.363892F,   0.367676F,
   0.314636F,   0.520996F,
   0.484863F,   0.490479F,
   0.397156F,   0.549316F,
   0.468140F,   0.671875F,
   0.363281F,   0.736328F,
   0.298950F,   0.918945F,
   0.426575F,   0.875977F,
   0.498901F,   0.971191F,
   0.370117F,   1.07520F,
   0.470520F,   1.24194F,
   0.337097F,   1.46997F,
   0.474182F,   1.73975F,
   0.369873F,   1.93799F,
   0.341431F,   2.80444F,
   0.645813F,   0.331055F,
   0.552307F,   0.389893F,
   0.597778F,   0.496826F,
   0.546021F,   0.589600F,
   0.628418F,   0.630859F,
   0.574158F,   0.667480F,
   0.531006F,   0.785645F,
   0.595520F,   0.828857F,
   0.621155F,   0.950195F,
   0.559692F,   1.10547F,
   0.619629F,   1.22168F,
   0.556274F,   1.40015F,
   0.640869F,   1.52979F,
   0.617065F,   1.86304F,
   0.539795F,   2.13062F,
   0.546631F,   3.05078F,
   0.788818F,   0.238281F,
   0.697937F,   0.428467F,
   0.740845F,   0.568359F,
   0.695068F,   0.578125F,
   0.653076F,   0.748047F,
   0.752686F,   0.698486F,
   0.715454F,   0.812256F,
   0.687866F,   0.903320F,
   0.662903F,   1.07739F,
   0.737427F,   1.10669F,
   0.688660F,   1.27075F,
   0.729980F,   1.53931F,
   0.681580F,   1.83936F,
   0.740234F,   2.03345F,
   0.669495F,   2.63110F,
   0.628662F,   4.24219F,
   0.848328F,   0.410400F,
   0.767822F,   0.499268F,
   0.809631F,   0.595459F,
   0.856506F,   0.729736F,
   0.821045F,   0.756348F,
   0.756592F,   0.893066F,
   0.824585F,   0.922852F,
   0.786133F,   1.04297F,
   0.825989F,   1.18677F,
   0.773132F,   1.33228F,
   0.845581F,   1.49072F,
   0.795349F,   1.58276F,
   0.827454F,   1.88501F,
   0.790833F,   2.27319F,
   0.837036F,   2.82007F,
   0.768494F,   3.71240F,
   0.922424F,   0.375977F,
   0.919922F,   0.569580F,
   0.886658F,   0.613037F,
   0.896729F,   0.781006F,
   0.938843F,   0.869141F,
   0.862610F,   0.966797F,
   0.921753F,   1.03418F,
   0.874756F,   1.17773F,
   0.906128F,   1.33081F,
   0.934204F,   1.48511F,
   0.874573F,   1.68164F,
   0.919189F,   1.87720F,
   0.879272F,   2.30127F,
   0.939148F,   2.37817F,
   0.904785F,   3.48413F,
   0.830078F,   6.08862F,
   1.00073F,   0.480713F,
   1.02643F,   0.691406F,
   0.959045F,   0.694092F,
   0.982910F,   0.814453F,
   1.00000F,   0.967529F,
   1.03394F,   1.11792F,
   0.958923F,   1.12280F,
   0.990112F,   1.33008F,
   1.02734F,   1.55811F,
   0.960999F,   1.74341F,
   0.996460F,   1.82349F,
   1.01385F,   2.10547F,
   1.03931F,   2.54346F,
   0.970764F,   2.88501F,
   1.03015F,   3.58643F,
   1.00800F,   5.09521F,
   1.10730F,   0.508545F,
   1.18414F,   0.775879F,
   1.06860F,   0.836426F,
   1.22400F,   0.983154F,
   1.10284F,   1.03735F,
   1.15674F,   1.23682F,
   1.08099F,   1.31885F,
   1.21063F,   1.51172F,
   1.09558F,   1.71240F,
   1.30115F,   1.92310F,
   1.09314F,   2.26782F,
   1.16846F,   2.26807F,
   1.25226F,   2.77856F,
   1.10321F,   3.53638F,
   1.22064F,   4.36572F,
   1.15002F,   7.99902F
};


/* table used in 'low' rates: MR475, MR515, MR59 */
#define VQ_SIZE_LOWRATES 64
static const Float32 table_lowrates[VQ_SIZE_LOWRATES * 2] =
{
   /*g_pit,    g_fac */
   0.659973F,   7.01978F,
   1.25000F,   0.679932F,
   1.14996F,   1.60986F,
   0.379944F,   1.80981F,
   1.04999F,   2.54980F,
   1.31995F,   0.309814F,
   1.28998F,   1.07983F,
   0.689941F,   0.379883F,
   1.15997F,   3.12988F,
   1.06000F,   0.609863F,
   1.08997F,   1.17993F,
   0.609985F,   0.609863F,
   1.06995F,   1.91992F,
   0.869995F,   0.459961F,
   0.969971F,   0.769775F,
   0.409973F,   0.439941F,
   1.10999F,   4.92993F,
   1.09998F,   0.739990F,
   1.01996F,   1.42993F,
   0.539978F,   0.979980F,
   0.969971F,   2.18994F,
   1.09998F,   0.339844F,
   1.01996F,   1.00000F,
   0.500000F,   0.159912F,
   0.929993F,   3.39990F,
   0.869995F,   0.759766F,
   0.859985F,   1.13989F,
   0.329956F,   0.659912F,
   0.819946F,   1.59985F,
   0.759949F,   0.219971F,
   0.759949F,   0.649902F,
   0.229980F,   0.159912F,
   0.899963F,   5.73999F,
   1.16998F,   0.599854F,
   1.22998F,   1.23999F,
   0.419983F,   1.00000F,
   1.25000F,   2.08984F,
   1.19995F,   0.179932F,
   1.15997F,   1.03979F,
   0.479980F,   0.509766F,
   0.699951F,   3.00000F,
   0.969971F,   0.359863F,
   0.959961F,   1.12988F,
   0.559998F,   0.349854F,
   0.979980F,   1.70996F,
   0.904968F,   0.179932F,
   0.919983F,   0.549805F,
   0.309998F,   0.299805F,
   0.809998F,   4.22998F,
   1.00995F,   0.569824F,
   0.919983F,   1.41992F,
   0.239990F,   0.899902F,
   0.869995F,   2.09985F,
   1.02997F,   0.189941F,
   0.919983F,   0.929932F,
   0.369995F,   0.149902F,
   0.569946F,   2.25977F,
   0.809998F,   0.429932F,
   0.809998F,   0.859863F,
   0.149963F,   0.479980F,
   0.699951F,   1.34985F,
   0.639954F,   0.179932F,
   0.709961F,   0.779785F,
   0.0899658F,   0.189941F
};


#define DTX_VQ_SIZE 47
static const Word32 qua_gain_code_MR122[NB_QUA_CODE+VQ_SIZE_HIGHRATES+VQ_SIZE_LOWRATES+(MR475_VQ_SIZE*2)+DTX_VQ_SIZE+1] =
{
      - 3776,
      - 3394,
      - 3005,
      - 2615,
      - 2345,
      - 2138,
      - 1932,
      - 1726,
      - 1518,
      - 1314,
      - 1106,
      - 900,
      - 694,
      - 487,
      - 281,
      - 75,
      133,
      339,
      545,
      752,
      958,
      1165,
      1371,
      1577,
      1784,
      1991,
      2197,
      2404,
      2673,
      3060,
      3448,
      3836,
      /* high_rates */
      - 2692,
      - 1185,
      - 2008,
      - 1600,
      - 1560,
      - 1070,
      - 840,
      - 313,
      - 471,
      - 777,
      - 414,
      72,
      139,
      413,
      835,
      1163,
      - 2267,
      - 1478,
      - 963,
      - 1052,
      - 885,
      - 588,
      - 452,
      - 125,
      - 196,
      - 43,
      107,
      320,
      569,
      818,
      977,
      1523,
      - 1633,
      - 1391,
      - 1033,
      - 780,
      - 681,
      - 597,
      - 356,
      - 277,
      - 75,
      148,
      296,
      497,
      628,
      919,
      1117,
      1648,
      - 2119,
      - 1252,
      - 835,
      - 810,
      - 429,
      - 530,
      - 307,
      - 150,
      110,
      150,
      354,
      637,
      900,
      1049,
      1429,
      2135,
      - 1316,
      - 1026,
      - 766,
      - 465,
      - 413,
      - 167,
      - 119,
      62,
      253,
      424,
      590,
      678,
      937,
      1213,
      1532,
      1938,
      - 1445,
      - 832,
      - 723,
      - 365,
      - 207,
      - 50,
      50,
      242,
      422,
      584,
      768,
      930,
      1231,
      1280,
      1844,
      2669,
      - 1082,
      - 545,
      - 539,
      - 303,
      - 49,
      165,
      171,
      421,
      655,
      821,
      888,
      1100,
      1379,
      1565,
      1887,
      2406,
      - 999,
      - 375,
      - 264,
      - 25,
      54,
      314,
      409,
      610,
      795,
      966,
      1210,
      1210,
      1510,
      1866,
      2177,
      3072,
      /* low_rates */
      2879,
      - 570,
      703,
      876,
      1383,
      - 1731,
      113,
      - 1430,
      1686,
      - 731,
      244,
      - 731,
      964,
      - 1147,
      - 387,
      - 1213,
      2357,
      - 445,
      528,
      - 30,
      1158,
      - 1594,
      0,
      - 2708,
      1808,
      - 406,
      193,
      - 614,
      694,
      - 2237,
      - 637,
      - 2708,
      2582,
      - 755,
      318,
      0,
      1089,
      - 2534,
      58,
      - 995,
      1623,
      - 1510,
      180,
      - 1552,
      793,
      - 2534,
      - 884,
      - 1780,
      2131,
      - 831,
      518,
      - 156,
      1096,
      - 2454,
      - 107,
      - 2804,
      1204,
      - 1247,
      - 223,
      - 1084,
      443,
      - 2534,
      - 367,
      - 2454,
      /* MR475 */
      - 5120,
      - 4988,
      - 1896,
      - 274,
      - 2932,
      - 2726,
      - 1216,
      376,
      - 3536,
      - 3293,
      - 650,
      - 733,
      - 2418,
      - 2664,
      1053,
      - 1303,
      - 3337,
      - 3256,
      - 9,
      - 1732,
      - 2237,
      - 2076,
      - 332,
      252,
      - 1044,
      - 2350,
      262,
      - 1296,
      - 1385,
      - 827,
      - 1447,
      1309,
      - 3375,
      - 3386,
      - 1418,
      - 215,
      - 2731,
      - 2754,
      - 1610,
      594,
      - 2610,
      - 2821,
      - 1409,
      189,
      - 1292,
      - 1779,
      853,
      492,
      - 1865,
      - 1103,
      593,
      - 1211,
      - 1213,
      - 1407,
      - 201,
      621,
      - 1903,
      - 1270,
      191,
      - 310,
      - 1435,
      - 453,
      1386,
      728,
      - 2646,
      - 1689,
      - 60,
      - 609,
      - 2602,
      - 2409,
      356,
      396,
      - 2991,
      - 2775,
      - 150,
      - 785,
      - 2577,
      - 2506,
      1524,
      - 881,
      - 3119,
      - 3041,
      - 98,
      - 442,
      - 2134,
      - 1151,
      411,
      254,
      - 1075,
      - 1777,
      105,
      - 291,
      - 524,
      - 1519,
      - 1318,
      1895,
      - 2838,
      - 2972,
      - 216,
      - 812,
      - 2785,
      - 2677,
      - 221,
      1286,
      - 2792,
      - 2863,
      - 454,
      26,
      - 1379,
      - 1780,
      1211,
      287,
      - 1798,
      - 1868,
      440,
      - 647,
      - 1034,
      - 892,
      322,
      740,
      - 1572,
      - 823,
      558,
      - 511,
      - 860,
      - 714,
      1356,
      1502,
      - 3690,
      - 3448,
      - 496,
      - 632,
      - 2770,
      - 2706,
      - 768,
      867,
      - 2360,
      - 2301,
      - 734,
      - 211,
      - 2005,
      - 1530,
      970,
      - 246,
      - 2982,
      - 2646,
      23,
      - 1349,
      - 1868,
      - 694,
      - 137,
      369,
      - 1056,
      - 1263,
      369,
      - 1317,
      - 1249,
      - 351,
      572,
      1320,
      - 3289,
      - 3237,
      - 890,
      43,
      - 2411,
      - 2367,
      - 998,
      694,
      - 2089,
      - 2105,
      - 728,
      296,
      - 1019,
      - 1341,
      775,
      511,
      - 2362,
      - 1311,
      508,
      - 1432,
      - 653,
      - 1525,
      214,
      634,
      - 1158,
      - 862,
      162,
      - 92,
      - 846,
      - 345,
      2171,
      - 582,
      - 1579,
      - 2279,
      - 213,
      40,
      - 2653,
      - 2448,
      329,
      928,
      - 2424,
      - 1836,
      - 383,
      - 441,
      - 2272,
      - 2320,
      1888,
      - 1119,
      - 2186,
      - 1960,
      - 4,
      150,
      - 2341,
      - 1968,
      555,
      164,
      - 686,
      - 1856,
      - 19,
      - 656,
      - 597,
      - 928,
      - 353,
      2304,
      - 2190,
      - 1372,
      - 416,
      64,
      - 2353,
      - 2394,
      - 860,
      1028,
      - 1313,
      - 1187,
      - 551,
      388,
      - 767,
      - 1666,
      1017,
      1088,
      - 1020,
      - 1676,
      747,
      - 1107,
      - 211,
      - 1233,
      704,
      943,
      - 700,
      - 646,
      891,
      - 886,
      - 518,
      - 307,
      2354,
      1047,
      - 2789,
      - 2880,
      - 860,
      - 180,
      - 2504,
      - 2358,
      - 531,
      541,
      - 2940,
      - 2929,
      - 456,
      - 157,
      - 2416,
      - 2552,
      1121,
      - 434,
      - 2454,
      - 2353,
      316,
      - 752,
      - 1162,
      - 1739,
      154,
      144,
      - 469,
      - 1743,
      186,
      - 668,
      - 795,
      - 708,
      - 485,
      1595,
      - 2644,
      - 2564,
      - 1725,
      37,
      - 2651,
      - 2858,
      - 1044,
      1119,
      - 2269,
      - 2240,
      - 933,
      47,
      - 1698,
      - 1038,
      890,
      162,
      - 1049,
      - 1211,
      701,
      - 192,
      - 1379,
      - 1292,
      - 251,
      714,
      - 2073,
      - 983,
      305,
      61,
      - 1660,
      - 227,
      1830,
      760,
      - 2534,
      - 2304,
      - 520,
      - 216,
      - 1764,
      - 2245,
      181,
      508,
      - 2865,
      - 2868,
      - 128,
      - 389,
      - 2492,
      - 2500,
      1478,
      - 913,
      - 2813,
      - 2657,
      18,
      - 307,
      - 1326,
      - 839,
      438,
      430,
      - 516,
      - 1644,
      388,
      - 54,
      - 133,
      - 1550,
      - 9,
      1985,
      - 1300,
      - 2265,
      161,
      - 317,
      - 2385,
      - 2227,
      65,
      1055,
      - 1960,
      - 2297,
      - 116,
      170,
      - 1451,
      - 912,
      1273,
      143,
      - 1963,
      - 927,
      581,
      126,
      - 1362,
      - 430,
      340,
      806,
      - 704,
      - 868,
      615,
      - 272,
      - 291,
      - 1070,
      1691,
      1985,
      - 2054,
      - 2262,
      72,
      - 168,
      - 1881,
      - 1532,
      - 900,
      715,
      - 1808,
      - 2415,
      - 271,
      - 81,
      - 981,
      - 959,
      1135,
      - 672,
      - 1768,
      - 2389,
      109,
      - 1013,
      - 539,
      - 881,
      134,
      405,
      - 582,
      - 974,
      429,
      - 569,
      - 605,
      - 80,
      244,
      1613,
      - 2089,
      - 2101,
      - 421,
      400,
      - 1328,
      - 1835,
      - 811,
      792,
      - 1424,
      - 1905,
      - 1453,
      403,
      - 1012,
      - 949,
      1033,
      640,
      - 1480,
      - 686,
      751,
      - 516,
      - 283,
      - 1190,
      215,
      852,
      - 572,
      - 310,
      310,
      210,
      - 806,
      101,
      2760,
      - 427,
      - 1420,
      - 1983,
      269,
      - 217,
      - 2275,
      - 2321,
      104,
      939,
      - 2055,
      - 2137,
      - 52,
      - 262,
      - 2110,
      - 2139,
      1693,
      37,
      - 1789,
      - 2387,
      334,
      264,
      - 1216,
      - 49,
      607,
      546,
      - 267,
      - 1081,
      346,
      - 457,
      - 116,
      - 810,
      - 592,
      2750,
      - 1710,
      - 2070,
      - 182,
      439,
      - 2418,
      - 2221,
      - 428,
      1359,
      - 653,
      - 1732,
      - 288,
      361,
      - 1318,
      - 378,
      1426,
      759,
      - 854,
      - 767,
      773,
      - 283,
      - 321,
      - 775,
      730,
      1302,
      - 669,
      - 393,
      945,
      - 80,
      - 158,
      - 284,
      2295,
      2557,
      /* dtx */
      - 1921,
      - 1878,
      - 1836,
      - 1793,
      - 1751,
      - 1708,
      - 1666,
      - 1623,
      - 1581,
      - 1538,
      - 1495,
      - 1453,
      - 1410,
      - 1368,
      - 1325,
      - 1283,
      - 1240,
      - 1198,
      - 1155,
      - 1113,
      - 1070,
      - 1028,
      - 985,
      - 943,
      - 900,
      - 858,
      - 815,
      - 773,
      - 730,
      - 688,
      - 645,
      - 602,
      - 560,
      - 517,
      - 475,
      - 432,
      - 390,
      - 347,
      - 305,
      - 262,
      - 220,
      - 177,
      - 135,
      - 92,
      - 50,
      - 7,
      0,
      /* init value */
      - 2381
};


static const Word32 qua_gain_code[NB_QUA_CODE+VQ_SIZE_HIGHRATES+VQ_SIZE_LOWRATES+(MR475_VQ_SIZE*2)+DTX_VQ_SIZE+3] =
{
      - 22731,
      - 20428,
      - 18088,
      - 15739,
      - 14113,
      - 12867,
      - 11629,
      - 10387,
      - 9139,
      - 7906,
      - 6656,
      - 5416,
      - 4173,
      - 2931,
      - 1688,
      - 445,
      801,
      2044,
      3285,
      4530,
      5772,
      7016,
      8259,
      9501,
      10745,
      11988,
      13231,
      14474,
      16096,
      18429,
      20763,
      23097,
      /* high rates */
      - 16214,
      - 7135,
      - 12086,
      - 9629,
      - 9394,
      - 6442,
      - 5056,
      - 1885,
      - 2838,
      - 4681,
      - 2490,
      434,
      836,
      2484,
      5030,
      7002,
      - 13647,
      - 8900,
      - 5800,
      - 6335,
      - 5327,
      - 3537,
      - 2724,
      - 751,
      - 1177,
      - 260,
      645,
      1928,
      3426,
      4926,
      5885,
      9172,
      - 9831,
      - 8380,
      - 6220,
      - 4700,
      - 4099,
      - 3595,
      - 2144,
      - 1669,
      - 454,
      891,
      1781,
      2993,
      3782,
      5534,
      6728,
      9921,
      - 12753,
      - 7539,
      - 5024,
      - 4872,
      - 2580,
      - 3192,
      - 1848,
      - 904,
      663,
      902,
      2132,
      3837,
      5420,
      6312,
      8604,
      12853,
      - 7921,
      - 6179,
      - 4610,
      - 2802,
      - 2482,
      - 1006,
      - 713,
      374,
      1523,
      2552,
      3551,
      4084,
      5639,
      7304,
      9221,
      11667,
      - 8700,
      - 5007,
      - 4352,
      - 2197,
      - 1247,
      - 300,
      298,
      1454,
      2542,
      3518,
      4623,
      5602,
      7413,
      7706,
      11102,
      16067,
      - 6517,
      - 3283,
      - 3248,
      - 1825,
      - 294,
      992,
      1030,
      2537,
      3945,
      4944,
      5343,
      6622,
      8303,
      9423,
      11360,
      14483,
      - 6013,
      - 2257,
      - 1589,
      - 151,
      326,
      1890,
      2461,
      3676,
      4784,
      5816,
      7283,
      7284,
      9089,
      11234,
      13108,
      18494,
      /* low_rates */
      17333,
      - 3431,
      4235,
      5276,
      8325,
      - 10422,
      683,
      - 8609,
      10148,
      - 4398,
      1472,
      - 4398,
      5802,
      - 6907,
      - 2327,
      - 7303,
      14189,
      - 2678,
      3181,
      - 180,
      6972,
      - 9599,
      0,
      - 16305,
      10884,
      - 2444,
      1165,
      - 3697,
      4180,
      - 13468,
      - 3833,
      - 16305,
      15543,
      - 4546,
      1913,
      0,
      6556,
      - 15255,
      347,
      - 5993,
      9771,
      - 9090,
      1086,
      - 9341,
      4772,
      - 15255,
      - 5321,
      - 10714,
      12827,
      - 5002,
      3118,
      - 938,
      6598,
      - 14774,
      - 646,
      - 16879,
      7251,
      - 7508,
      - 1343,
      - 6529,
      2668,
      - 15255,
      - 2212,
      - 14774,
      /* MR475 */
      - 30825,
      - 30028,
      - 11416,
      - 1652,
      - 17651,
      - 16414,
      - 7323,
      2266,
      - 21289,
      - 19823,
      - 3914,
      - 4413,
      - 14560,
      - 16037,
      6338,
      - 7842,
      - 20089,
      - 19604,
      - 53,
      - 10430,
      - 13469,
      - 12497,
      - 2001,
      1517,
      - 6283,
      - 14145,
      1579,
      - 7801,
      - 8339,
      - 4977,
      - 8713,
      7881,
      - 20321,
      - 20385,
      - 8535,
      - 1296,
      - 16442,
      - 16581,
      - 9696,
      3577,
      - 15714,
      - 16981,
      - 8484,
      1135,
      - 7780,
      - 10708,
      5136,
      2962,
      - 11229,
      - 6643,
      3572,
      - 7294,
      - 7304,
      - 8473,
      - 1208,
      3739,
      - 11455,
      - 7646,
      1150,
      - 1866,
      - 8638,
      - 2726,
      8342,
      4384,
      - 15933,
      - 10167,
      - 362,
      - 3665,
      - 15663,
      - 14504,
      2142,
      2387,
      - 18006,
      - 16707,
      - 905,
      - 4726,
      - 15513,
      - 15088,
      9173,
      - 5305,
      - 18778,
      - 18307,
      - 591,
      - 2661,
      - 12849,
      - 6927,
      2472,
      1527,
      - 6470,
      - 10701,
      632,
      - 1754,
      - 3155,
      - 9145,
      - 7938,
      11408,
      - 17085,
      - 17891,
      - 1301,
      - 4889,
      - 16764,
      - 16117,
      - 1333,
      7744,
      - 16808,
      - 17234,
      - 2732,
      154,
      - 8300,
      - 10715,
      7291,
      1731,
      - 10824,
      - 11244,
      2652,
      - 3894,
      - 6227,
      - 5372,
      1937,
      4457,
      - 9466,
      - 4954,
      3358,
      - 3075,
      - 5176,
      - 4297,
      8162,
      9040,
      - 22215,
      - 20758,
      - 2986,
      - 3807,
      - 16679,
      - 16291,
      - 4626,
      5220,
      - 14210,
      - 13852,
      - 4420,
      - 1271,
      - 12073,
      - 9212,
      5842,
      - 1481,
      - 17956,
      - 15933,
      137,
      - 8120,
      - 11244,
      - 4180,
      - 826,
      2224,
      - 6359,
      - 7605,
      2219,
      - 7927,
      - 7518,
      - 2111,
      3447,
      7949,
      - 19803,
      - 19487,
      - 5360,
      259,
      - 14515,
      - 14253,
      - 6011,
      4180,
      - 12577,
      - 12676,
      - 4384,
      1784,
      - 6135,
      - 8071,
      4665,
      3078,
      - 14220,
      - 7896,
      3060,
      - 8621,
      - 3931,
      - 9182,
      1285,
      3816,
      - 6974,
      - 5192,
      975,
      - 554,
      - 5091,
      - 2075,
      13071,
      - 3502,
      - 9504,
      - 13719,
      - 1280,
      244,
      - 15972,
      - 14740,
      1981,
      5587,
      - 14593,
      - 11054,
      - 2308,
      - 2656,
      - 13678,
      - 13966,
      11370,
      - 6739,
      - 13158,
      - 11799,
      - 27,
      901,
      - 14092,
      - 11849,
      3344,
      989,
      - 4132,
      - 11176,
      - 114,
      - 3951,
      - 3596,
      - 5586,
      - 2124,
      13869,
      - 13187,
      - 8262,
      - 2502,
      384,
      - 14166,
      - 14416,
      - 5176,
      6192,
      - 7906,
      - 7147,
      - 3320,
      2336,
      - 4615,
      - 10032,
      6125,
      6551,
      - 6144,
      - 10092,
      4495,
      - 6666,
      - 1273,
      - 7423,
      4241,
      5680,
      - 4212,
      - 3891,
      5366,
      - 5337,
      - 3121,
      - 1847,
      14175,
      6302,
      - 16793,
      - 17341,
      - 5176,
      - 1085,
      - 15076,
      - 14199,
      - 3195,
      3257,
      - 17698,
      - 17635,
      - 2746,
      - 946,
      - 14548,
      - 15365,
      6748,
      - 2615,
      - 14774,
      - 14166,
      1904,
      - 4525,
      - 6998,
      - 10472,
      928,
      865,
      - 2821,
      - 10493,
      1122,
      - 4019,
      - 4785,
      - 4261,
      - 2922,
      9604,
      - 15920,
      - 15438,
      - 10388,
      222,
      - 15959,
      - 17204,
      - 6283,
      6737,
      - 13658,
      - 13488,
      - 5618,
      280,
      - 10222,
      - 6248,
      5358,
      977,
      - 6319,
      - 7294,
      4218,
      - 1156,
      - 8300,
      - 7780,
      - 1514,
      4296,
      - 12479,
      - 5917,
      1839,
      367,
      - 9992,
      - 1369,
      11020,
      4578,
      - 15255,
      - 13873,
      - 3133,
      - 1298,
      - 10621,
      - 13518,
      1087,
      3057,
      - 17250,
      - 17265,
      - 774,
      - 2339,
      - 15006,
      - 15053,
      8896,
      - 5497,
      - 16938,
      - 15998,
      107,
      - 1850,
      - 7980,
      - 5053,
      2639,
      2590,
      - 3108,
      - 9898,
      2333,
      - 323,
      - 800,
      - 9329,
      - 57,
      11948,
      - 7827,
      - 13638,
      967,
      - 1906,
      - 14361,
      - 13410,
      392,
      6353,
      - 11799,
      - 13831,
      - 698,
      1024,
      - 8736,
      - 5489,
      7665,
      861,
      - 11816,
      - 5582,
      3495,
      761,
      - 8201,
      - 2588,
      2045,
      4850,
      - 4240,
      - 5223,
      3705,
      - 1636,
      - 1754,
      - 6443,
      10180,
      11952,
      - 12365,
      - 13618,
      433,
      - 1009,
      - 11322,
      - 9225,
      - 5421,
      4302,
      - 10883,
      - 14537,
      - 1633,
      - 489,
      - 5904,
      - 5771,
      6831,
      - 4047,
      - 10643,
      - 14383,
      656,
      - 6097,
      - 3245,
      - 5305,
      809,
      2438,
      - 3502,
      - 5862,
      2584,
      - 3429,
      - 3642,
      - 482,
      1471,
      9712,
      - 12577,
      - 12649,
      - 2533,
      2410,
      - 7996,
      - 11047,
      - 4882,
      4768,
      - 8575,
      - 11470,
      - 8747,
      2427,
      - 6092,
      - 5713,
      6222,
      3854,
      - 8911,
      - 4132,
      4521,
      - 3108,
      - 1704,
      - 7162,
      1293,
      5132,
      - 3445,
      - 1866,
      1868,
      1264,
      - 4855,
      608,
      16615,
      - 2571,
      - 8552,
      - 11940,
      1620,
      - 1308,
      - 13699,
      - 13976,
      626,
      5652,
      - 12374,
      - 12868,
      - 312,
      - 1579,
      - 12703,
      - 12877,
      10194,
      222,
      - 10773,
      - 14372,
      2009,
      1590,
      - 7323,
      - 296,
      3655,
      3289,
      - 1605,
      - 6506,
      2082,
      - 2752,
      - 696,
      - 4878,
      - 3567,
      16558,
      - 10297,
      - 12462,
      - 1094,
      2640,
      - 14560,
      - 13370,
      - 2577,
      8180,
      - 3931,
      - 10430,
      - 1733,
      2173,
      - 7938,
      - 2274,
      8585,
      4567,
      - 5141,
      - 4619,
      4655,
      - 1707,
      - 1931,
      - 4666,
      4396,
      7837,
      - 4026,
      - 2364,
      5687,
      - 482,
      - 948,
      - 1709,
      13820,
      15394,
      /* dtx */
      - 11560,
      - 11304,
      - 11048,
      - 10792,
      - 10536,
      - 10280,
      - 10024,
      - 9768,
      - 9512,
      - 9256,
      - 9000,
      - 8744,
      - 8488,
      - 8232,
      - 7976,
      - 7720,
      - 7464,
      - 7208,
      - 6952,
      - 6696,
      - 6440,
      - 6184,
      - 5928,
      - 5672,
      - 5416,
      - 5160,
      - 4904,
      - 4648,
      - 4392,
      - 4136,
      - 3880,
      - 3624,
      - 3368,
      - 3112,
      - 2856,
      - 2600,
      - 2344,
      - 2088,
      - 1832,
      - 1576,
      - 1320,
      - 1064,
      - 808,
      - 552,
      - 296,
      - 40,
      0,
      /* init value */
      - 14336,
      /* min */
      - 32767,
      /* max */
      18284

};
/*
 * LP analysis is performed twice per frame using two different asymmetric windows.
 * The second window has its weight concentrated at the fourth subframe
 * and it consists of two parts:
 * the first part is half a Hamming window and
 * the second part is a quarter of a cosine function cycle.
 */
static const Float32 window_200_40[240] =
{
   0.0800000F,
   0.0800570F,
   0.0802281F,
   0.0805132F,
   0.0809123F,
   0.0814251F,
   0.0820517F,
   0.0827919F,
   0.0836454F,
   0.0846121F,
   0.0856917F,
   0.0868840F,
   0.0881886F,
   0.0896053F,
   0.0911337F,
   0.0927733F,
   0.0945239F,
   0.0963849F,
   0.0983560F,
   0.100437F,
   0.102626F,
   0.104924F,
   0.107330F,
   0.109843F,
   0.112463F,
   0.115189F,
   0.118020F,
   0.120956F,
   0.123996F,
   0.127139F,
   0.130384F,
   0.133731F,
   0.137178F,
   0.140726F,
   0.144372F,
   0.148117F,
   0.151959F,
   0.155897F,
   0.159930F,
   0.164058F,
   0.168278F,
   0.172591F,
   0.176995F,
   0.181489F,
   0.186072F,
   0.190743F,
   0.195500F,
   0.200343F,
   0.205270F,
   0.210280F,
   0.215372F,
   0.220544F,
   0.225795F,
   0.231125F,
   0.236531F,
   0.242012F,
   0.247567F,
   0.253195F,
   0.258893F,
   0.264662F,
   0.270499F,
   0.276402F,
   0.282371F,
   0.288404F,
   0.294499F,
   0.300655F,
   0.306871F,
   0.313144F,
   0.319473F,
   0.325857F,
   0.332295F,
   0.338783F,
   0.345322F,
   0.351909F,
   0.358543F,
   0.365221F,
   0.371943F,
   0.378707F,
   0.385510F,
   0.392352F,
   0.399231F,
   0.406144F,
   0.413091F,
   0.420069F,
   0.427077F,
   0.434112F,
   0.441175F,
   0.448261F,
   0.455371F,
   0.462501F,
   0.469650F,
   0.476817F,
   0.484000F,
   0.491197F,
   0.498405F,
   0.505624F,
   0.512852F,
   0.520086F,
   0.527325F,
   0.534567F,
   0.541811F,
   0.549054F,
   0.556295F,
   0.563532F,
   0.570763F,
   0.577986F,
   0.585201F,
   0.592403F,
   0.599593F,
   0.606768F,
   0.613927F,
   0.621067F,
   0.628187F,
   0.635285F,
   0.642360F,
   0.649409F,
   0.656431F,
   0.663424F,
   0.670387F,
   0.677317F,
   0.684213F,
   0.691073F,
   0.697896F,
   0.704680F,
   0.711423F,
   0.718124F,
   0.724780F,
   0.731390F,
   0.737953F,
   0.744467F,
   0.750930F,
   0.757341F,
   0.763698F,
   0.770000F,
   0.776245F,
   0.782430F,
   0.788556F,
   0.794620F,
   0.800622F,
   0.806558F,
   0.812428F,
   0.818231F,
   0.823965F,
   0.829628F,
   0.835220F,
   0.840738F,
   0.846182F,
   0.851550F,
   0.856840F,
   0.862052F,
   0.867184F,
   0.872235F,
   0.877204F,
   0.882089F,
   0.886889F,
   0.891603F,
   0.896230F,
   0.900769F,
   0.905218F,
   0.909577F,
   0.913844F,
   0.918018F,
   0.922098F,
   0.926084F,
   0.929974F,
   0.933767F,
   0.937463F,
   0.941060F,
   0.944558F,
   0.947955F,
   0.951251F,
   0.954446F,
   0.957537F,
   0.960525F,
   0.963409F,
   0.966187F,
   0.968860F,
   0.971427F,
   0.973886F,
   0.976238F,
   0.978482F,
   0.980617F,
   0.982643F,
   0.984559F,
   0.986365F,
   0.988060F,
   0.989644F,
   0.991117F,
   0.992478F,
   0.993726F,
   0.994862F,
   0.995885F,
   0.996796F,
   0.997592F,
   0.998276F,
   0.998846F,
   0.999301F,
   0.999644F,
   0.999872F,
   0.999986F,
   1.00000F,
   0.999219F,
   0.996878F,
   0.992981F,
   0.987533F,
   0.980544F,
   0.972023F,
   0.961985F,
   0.950444F,
   0.937420F,
   0.922932F,
   0.907002F,
   0.889657F,
   0.870923F,
   0.850828F,
   0.829406F,
   0.806688F,
   0.782711F,
   0.757511F,
   0.731129F,
   0.703605F,
   0.674983F,
   0.645307F,
   0.614623F,
   0.582980F,
   0.550426F,
   0.517013F,
   0.482792F,
   0.447818F,
   0.412145F,
   0.375828F,
   0.338924F,
   0.301491F,
   0.263587F,
   0.225272F,
   0.186605F,
   0.147647F,
   0.108458F,
   0.0690994F,
   0.0296333F
};

/*
 * LP analysis is performed twice per frame using two different asymmetric windows.
 * The second window has its weight concentrated at the fourth subframe
 * and it consists of two parts:
 * the first part is half a Hamming window and
 * the second part is a quarter of a cosine function cycle.
 */
static const Float32 window_232_8[240] =
{
   0.0800000F,
   0.0800424F,
   0.0801694F,
   0.0803812F,
   0.0806775F,
   0.0810585F,
   0.0815240F,
   0.0820739F,
   0.0827082F,
   0.0834267F,
   0.0842292F,
   0.0851157F,
   0.0860859F,
   0.0871398F,
   0.0882770F,
   0.0894975F,
   0.0908009F,
   0.0921870F,
   0.0936556F,
   0.0952064F,
   0.0968391F,
   0.0985534F,
   0.100349F,
   0.102226F,
   0.104183F,
   0.106220F,
   0.108338F,
   0.110534F,
   0.112810F,
   0.115165F,
   0.117598F,
   0.120108F,
   0.122696F,
   0.125361F,
   0.128102F,
   0.130919F,
   0.133812F,
   0.136779F,
   0.139820F,
   0.142935F,
   0.146123F,
   0.149384F,
   0.152717F,
   0.156121F,
   0.159596F,
   0.163140F,
   0.166754F,
   0.170437F,
   0.174188F,
   0.178007F,
   0.181892F,
   0.185843F,
   0.189859F,
   0.193939F,
   0.198084F,
   0.202291F,
   0.206561F,
   0.210892F,
   0.215283F,
   0.219734F,
   0.224245F,
   0.228813F,
   0.233439F,
   0.238121F,
   0.242859F,
   0.247652F,
   0.252498F,
   0.257397F,
   0.262349F,
   0.267351F,
   0.272404F,
   0.277506F,
   0.282656F,
   0.287854F,
   0.293098F,
   0.298388F,
   0.303722F,
   0.309100F,
   0.314520F,
   0.319981F,
   0.325484F,
   0.331025F,
   0.336606F,
   0.342223F,
   0.347877F,
   0.353567F,
   0.359290F,
   0.365048F,
   0.370837F,
   0.376657F,
   0.382508F,
   0.388387F,
   0.394295F,
   0.400229F,
   0.406189F,
   0.412174F,
   0.418182F,
   0.424213F,
   0.430265F,
   0.436337F,
   0.442428F,
   0.448537F,
   0.454663F,
   0.460805F,
   0.466961F,
   0.473131F,
   0.479313F,
   0.485507F,
   0.491710F,
   0.497922F,
   0.504142F,
   0.510369F,
   0.516601F,
   0.522837F,
   0.529077F,
   0.535318F,
   0.541561F,
   0.547803F,
   0.554043F,
   0.560281F,
   0.566516F,
   0.572745F,
   0.578969F,
   0.585185F,
   0.591393F,
   0.597591F,
   0.603779F,
   0.609955F,
   0.616119F,
   0.622268F,
   0.628402F,
   0.634520F,
   0.640620F,
   0.646702F,
   0.652764F,
   0.658805F,
   0.664825F,
   0.670822F,
   0.676794F,
   0.682741F,
   0.688662F,
   0.694556F,
   0.700421F,
   0.706257F,
   0.712062F,
   0.717835F,
   0.723576F,
   0.729282F,
   0.734954F,
   0.740590F,
   0.746189F,
   0.751750F,
   0.757272F,
   0.762755F,
   0.768196F,
   0.773595F,
   0.778951F,
   0.784263F,
   0.789530F,
   0.794751F,
   0.799925F,
   0.805051F,
   0.810129F,
   0.815156F,
   0.820133F,
   0.825059F,
   0.829932F,
   0.834751F,
   0.839517F,
   0.844227F,
   0.848881F,
   0.853478F,
   0.858018F,
   0.862499F,
   0.866920F,
   0.871282F,
   0.875582F,
   0.879820F,
   0.883996F,
   0.888109F,
   0.892157F,
   0.896141F,
   0.900059F,
   0.903911F,
   0.907696F,
   0.911413F,
   0.915061F,
   0.918641F,
   0.922151F,
   0.925590F,
   0.928958F,
   0.932255F,
   0.935480F,
   0.938632F,
   0.941710F,
   0.944714F,
   0.947644F,
   0.950499F,
   0.953278F,
   0.955981F,
   0.958607F,
   0.961157F,
   0.963628F,
   0.966022F,
   0.968338F,
   0.970574F,
   0.972731F,
   0.974809F,
   0.976806F,
   0.978723F,
   0.980559F,
   0.982314F,
   0.983988F,
   0.985579F,
   0.987089F,
   0.988516F,
   0.989861F,
   0.991123F,
   0.992302F,
   0.993398F,
   0.994410F,
   0.995338F,
   0.996183F,
   0.996943F,
   0.997619F,
   0.998212F,
   0.998719F,
   0.999143F,
   0.999481F,
   0.999735F,
   0.999905F,
   0.999989F,
   1.00000F,
   0.979530F,
   0.918958F,
   0.820763F,
   0.688967F,
   0.528964F,
   0.347305F,
   0.151428F
};

/*
 * LP analysis is performed twice per frame using two different asymmetric windows.
 * The first window has its weight concentrated at the second subframe
 * and it consists of two halves of Hamming windows with different sizes
 */
static const Float32 window_160_80[240] =
{
   0.0800000F,
   0.0800898F,
   0.0803591F,
   0.0808079F,
   0.0814359F,
   0.0822430F,
   0.0832287F,
   0.0843928F,
   0.0857347F,
   0.0872539F,
   0.0889499F,
   0.0908220F,
   0.0928695F,
   0.0950914F,
   0.0974871F,
   0.100056F,
   0.102796F,
   0.105707F,
   0.108787F,
   0.112036F,
   0.115451F,
   0.119033F,
   0.122779F,
   0.126688F,
   0.130758F,
   0.134988F,
   0.139376F,
   0.143920F,
   0.148619F,
   0.153471F,
   0.158473F,
   0.163625F,
   0.168924F,
   0.174367F,
   0.179953F,
   0.185680F,
   0.191545F,
   0.197546F,
   0.203681F,
   0.209947F,
   0.216342F,
   0.222863F,
   0.229508F,
   0.236274F,
   0.243159F,
   0.250160F,
   0.257273F,
   0.264498F,
   0.271829F,
   0.279266F,
   0.286804F,
   0.294441F,
   0.302174F,
   0.310000F,
   0.317916F,
   0.325918F,
   0.334004F,
   0.342170F,
   0.350414F,
   0.358731F,
   0.367119F,
   0.375575F,
   0.384095F,
   0.392676F,
   0.401314F,
   0.410007F,
   0.418750F,
   0.427540F,
   0.436375F,
   0.445250F,
   0.454162F,
   0.463107F,
   0.472083F,
   0.481085F,
   0.490109F,
   0.499154F,
   0.508214F,
   0.517287F,
   0.526369F,
   0.535456F,
   0.544544F,
   0.553631F,
   0.562713F,
   0.571786F,
   0.580846F,
   0.589891F,
   0.598915F,
   0.607917F,
   0.616893F,
   0.625838F,
   0.634750F,
   0.643625F,
   0.652460F,
   0.661250F,
   0.669993F,
   0.678686F,
   0.687324F,
   0.695905F,
   0.704425F,
   0.712881F,
   0.721269F,
   0.729586F,
   0.737830F,
   0.745996F,
   0.754082F,
   0.762084F,
   0.770000F,
   0.777826F,
   0.785559F,
   0.793196F,
   0.800734F,
   0.808171F,
   0.815502F,
   0.822727F,
   0.829840F,
   0.836841F,
   0.843726F,
   0.850492F,
   0.857137F,
   0.863658F,
   0.870053F,
   0.876319F,
   0.882454F,
   0.888455F,
   0.894320F,
   0.900047F,
   0.905633F,
   0.911076F,
   0.916375F,
   0.921527F,
   0.926529F,
   0.931381F,
   0.936080F,
   0.940624F,
   0.945012F,
   0.949242F,
   0.953312F,
   0.957221F,
   0.960967F,
   0.964549F,
   0.967964F,
   0.971213F,
   0.974293F,
   0.977204F,
   0.979944F,
   0.982513F,
   0.984909F,
   0.987131F,
   0.989178F,
   0.991050F,
   0.992746F,
   0.994265F,
   0.995607F,
   0.996771F,
   0.997757F,
   0.998564F,
   0.999192F,
   0.999641F,
   0.999910F,
   1.00000F,
   1.00000F,
   0.999636F,
   0.998546F,
   0.996730F,
   0.994193F,
   0.990937F,
   0.986968F,
   0.982292F,
   0.976917F,
   0.970851F,
   0.964104F,
   0.956687F,
   0.948610F,
   0.939887F,
   0.930532F,
   0.920560F,
   0.909986F,
   0.898826F,
   0.887100F,
   0.874824F,
   0.862019F,
   0.848705F,
   0.834903F,
   0.820635F,
   0.805922F,
   0.790790F,
   0.775260F,
   0.759359F,
   0.743111F,
   0.726542F,
   0.709677F,
   0.692545F,
   0.675171F,
   0.657584F,
   0.639810F,
   0.621879F,
   0.603818F,
   0.585657F,
   0.567423F,
   0.549146F,
   0.530854F,
   0.512577F,
   0.494343F,
   0.476182F,
   0.458121F,
   0.440190F,
   0.422416F,
   0.404829F,
   0.387455F,
   0.370323F,
   0.353458F,
   0.336889F,
   0.320641F,
   0.304740F,
   0.289210F,
   0.274078F,
   0.259365F,
   0.245097F,
   0.231295F,
   0.217981F,
   0.205176F,
   0.192900F,
   0.181174F,
   0.170014F,
   0.159440F,
   0.149468F,
   0.140113F,
   0.131390F,
   0.123313F,
   0.115896F,
   0.109149F,
   0.103083F,
   0.0977077F,
   0.0930321F,
   0.0890632F,
   0.0858073F,
   0.0832696F,
   0.0814541F,
   0.0803637F,
   0.0800000F
};

/* correlation weights	*/
static const Float32 corrweight[251] =
{
   0.624805F,
   0.625813F,
   0.626820F,
   0.627827F,
   0.628834F,
   0.630024F,
   0.631031F,
   0.632221F,
   0.633229F,
   0.634419F,
   0.635426F,
   0.636616F,
   0.637623F,
   0.638813F,
   0.640034F,
   0.641224F,
   0.642415F,
   0.643605F,
   0.644826F,
   0.646016F,
   0.647206F,
   0.648427F,
   0.649617F,
   0.651021F,
   0.652211F,
   0.653615F,
   0.654805F,
   0.656209F,
   0.657430F,
   0.658834F,
   0.660207F,
   0.661611F,
   0.663015F,
   0.664418F,
   0.665822F,
   0.667226F,
   0.668630F,
   0.670217F,
   0.671621F,
   0.673208F,
   0.674612F,
   0.676229F,
   0.677816F,
   0.679434F,
   0.681021F,
   0.682607F,
   0.684225F,
   0.685812F,
   0.687613F,
   0.689230F,
   0.691031F,
   0.692831F,
   0.694632F,
   0.696432F,
   0.698233F,
   0.700034F,
   0.702017F,
   0.703818F,
   0.705832F,
   0.707816F,
   0.709616F,
   0.711814F,
   0.713828F,
   0.715812F,
   0.718009F,
   0.720237F,
   0.722221F,
   0.724631F,
   0.726829F,
   0.729026F,
   0.731437F,
   0.733818F,
   0.736229F,
   0.738609F,
   0.741234F,
   0.743614F,
   0.746208F,
   0.748833F,
   0.751610F,
   0.754234F,
   0.757012F,
   0.760033F,
   0.762810F,
   0.765831F,
   0.768822F,
   0.772027F,
   0.775018F,
   0.778222F,
   0.781610F,
   0.785028F,
   0.788415F,
   0.792016F,
   0.795618F,
   0.799219F,
   0.803034F,
   0.807031F,
   0.811029F,
   0.815027F,
   0.819239F,
   0.823634F,
   0.828028F,
   0.832636F,
   0.837428F,
   0.842219F,
   0.847224F,
   0.852412F,
   0.857814F,
   0.863216F,
   0.869015F,
   0.874813F,
   0.881039F,
   0.887417F,
   0.894040F,
   0.901028F,
   0.908231F,
   0.915616F,
   0.923429F,
   0.931639F,
   0.940214F,
   0.960021F,
   1.00000F,
   1.00000F,
   1.00000F,
   1.00000F,
   1.00000F,
   1.00000F,
   1.00000F,
   0.960021F,
   0.940214F,
   0.931639F,
   0.923429F,
   0.915616F,
   0.908231F,
   0.901028F,
   0.894040F,
   0.887417F,
   0.881039F,
   0.874813F,
   0.869015F,
   0.863216F,
   0.857814F,
   0.852412F,
   0.847224F,
   0.842219F,
   0.837428F,
   0.832636F,
   0.828028F,
   0.823634F,
   0.819239F,
   0.815027F,
   0.811029F,
   0.807031F,
   0.803034F,
   0.799219F,
   0.795618F,
   0.792016F,
   0.788415F,
   0.785028F,
   0.781610F,
   0.778222F,
   0.775018F,
   0.772027F,
   0.768822F,
   0.765831F,
   0.762810F,
   0.760033F,
   0.757012F,
   0.754234F,
   0.751610F,
   0.748833F,
   0.746208F,
   0.743614F,
   0.741234F,
   0.738609F,
   0.736229F,
   0.733818F,
   0.731437F,
   0.729026F,
   0.726829F,
   0.724631F,
   0.722221F,
   0.720237F,
   0.718009F,
   0.715812F,
   0.713828F,
   0.711814F,
   0.709616F,
   0.707816F,
   0.705832F,
   0.703818F,
   0.702017F,
   0.700034F,
   0.698233F,
   0.696432F,
   0.694632F,
   0.692831F,
   0.691031F,
   0.689230F,
   0.687613F,
   0.685812F,
   0.684225F,
   0.682607F,
   0.681021F,
   0.679434F,
   0.677816F,
   0.676229F,
   0.674612F,
   0.673208F,
   0.671621F,
   0.670217F,
   0.668630F,
   0.667226F,
   0.665822F,
   0.664418F,
   0.663015F,
   0.661611F,
   0.660207F,
   0.658834F,
   0.657430F,
   0.656209F,
   0.654805F,
   0.653615F,
   0.652211F,
   0.651021F,
   0.649617F,
   0.648427F,
   0.647206F,
   0.646016F,
   0.644826F,
   0.643605F,
   0.642415F,
   0.641224F,
   0.640034F,
   0.638813F,
   0.637623F,
   0.636616F,
   0.635426F,
   0.634419F,
   0.633229F,
   0.632221F,
   0.631031F,
   0.630024F,
   0.628834F,
   0.627827F,
   0.626820F,
   0.625813F,
   0.624805F,
   0.623615F,
   0.622608F,
   0.621632F,
   0.620624F
};


const struct
{
   Word8 max_frac_lag; /* lag up to which fractional lags are used */
   Word8 flag3; /* enable 1/3 instead of 1/6 fract. resolution */
   Word8 first_frac; /* first fractional to check */
   Word8 last_frac; /* last fractional to check */
   Word8 delta_int_low; /* integer lag below TO to start search from */
   Word8 delta_int_range; /* integer range around T0 */
   Word8 delta_frc_low; /* fractional below T0 */
   Word8 delta_frc_range; /* fractional range around T0 */
   Word8 pit_min; /* minimum pitch */

} static mode_dep_parm[8] =
{
   /*
    * MR475
    */
   {
      84, 1, - 2, 2, 5, 10, 5, 9, PIT_MIN
   },
   /*
    * MR515
    */
   {
      84, 1, - 2, 2, 5, 10, 5, 9, PIT_MIN
   },
   /*
    * MR59
    */
   {
      84, 1, - 2, 2, 3, 6, 5, 9, PIT_MIN
   },
   /*
    * MR67
    */
   {
      84, 1, - 2, 2, 3, 6, 5, 9, PIT_MIN
   },
   /*
    * MR74
    */
   {
      84, 1, - 2, 2, 3, 6, 5, 9, PIT_MIN
   },
   /*
    * MR795
    */
   {
      84, 1, - 2, 2, 3, 6, 10, 19, PIT_MIN
   },
   /*
    * MR102
    */
   {
      84, 1, - 2, 2, 3, 6, 5, 9, PIT_MIN
   },
   /*
    * MR122
    */
   {
      94, 0, - 3, 3, 3, 6, 5, 9, PIT_MIN_MR122
   }
};

/* table used inbase 2 logharithm computation */
static const Word32 log2_table[33] =
{
   0,
   1455,
   2866,
   4236,
   5568,
   6863,
   8124,
   9352,
   10549,
   11716,
   12855,
   13967,
   15054,
   16117,
   17156,
   18172,
   19167,
   20142,
   21097,
   22033,
   22951,
   23852,
   24735,
   25603,
   26455,
   27291,
   28113,
   28922,
   29716,
   30497,
   31266,
   32023,
   32767
};

/* table used in 2 to the power computation */
static const Word32 pow2_table[33] =
{
   16384,
   16743,
   17109,
   17484,
   17867,
   18258,
   18658,
   19066,
   19484,
   19911,
   20347,
   20792,
   21247,
   21713,
   22188,
   22674,
   23170,
   23678,
   24196,
   24726,
   25268,
   25821,
   26386,
   26964,
   27554,
   28158,
   28774,
   29405,
   30048,
   30706,
   31379,
   32066,
   32767
};


#endif
