# Orca
#
# Copyright 2004-2006 Sun Microsystems Inc.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Manages the default speech server for orca.  A script can use this
as its speech server, or it can feel free to create one of its own."""

__id__        = "$Id: speech.py,v 1.55 2006/06/10 00:54:47 wwalker Exp $"
__version__   = "$Revision: 1.55 $"
__date__      = "$Date: 2006/06/10 00:54:47 $"
__copyright__ = "Copyright (c) 2005-2006 Sun Microsystems Inc."
__license__   = "LGPL"

import threading
import time
import BaseHTTPServer

import debug
import orca
import platform
import settings

from acss import ACSS
from orca_i18n import _           # for gettext support

class _SpeakRequestHandler(BaseHTTPServer.BaseHTTPRequestHandler):
    """Provides support for communicating with speech via a HTTP.
    This is to support self-voicing applications that want to use
    Orca as a speech service.

    The protocol is simple: POST content is 'stop' or 'speak:text'

    To test this, run:

      wget --post-data='speak:hello world' localhost:20433

    """

    def do_GET(self):
        self.send_response(200)
        self.send_header("Content-type", "text/html")
        self.end_headers()
        self.wfile.write("<html><body><p>Orca %s</p></body></html>" \
                         % platform.version)

    def do_POST(self):
        contentLength = self.headers.getheader('content-length')
        if contentLength:
            contentLength = int(contentLength)
            inputBody = self.rfile.read(contentLength)
            debug.println(debug.LEVEL_FINEST,
                          "speech._SpeakRequestHandler received %s" \
                          % inputBody)
            if inputBody.startswith("speak:"):
                speak(inputBody[6:])
                self.send_response(200, 'OK')
            elif inputBody == "stop":
                stop()
                self.send_response(200, 'OK')
            elif inputBody == "isSpeaking":
                self.send_response(200, 'OK')
                self.send_header("Content-type", "text/html")
                self.end_headers()
                self.wfile.write("%s" % isSpeaking())
        else:
            debug.println(debug.LEVEL_FINEST,
                          "speech._SpeakRequestHandler received no data")


class _SpeakRequestThread(threading.Thread):
    """Runs a _SpeakRequestHandler in a separate thread."""

    def run(self):
        httpd = BaseHTTPServer.HTTPServer(('',
                                           settings.speechServerPort),
                                          _SpeakRequestHandler)
        httpd.serve_forever()

# The speech server to use for all speech operations.
#
__speechserver = None

def getSpeechServerFactories():
    """Imports all known SpeechServer factory modules.  Returns a list
    of modules that implement the getSpeechServers method, which
    returns a list of speechserver.SpeechServer instances.
    """

    factories = []

    moduleNames = settings.speechFactoryModules
    for moduleName in moduleNames:
        try:
            module =  __import__(moduleName,
                                 globals(),
                                 locals(),
                                 [''])
            factories.append(module)
        except:
            debug.printException(debug.LEVEL_OFF)

    return factories

_speakRequestThread = None

def init():

    global __speechserver

    if __speechserver:
        return

    # First, find the factory module to use.  We will
    # allow the user to give their own factory module,
    # thus we look first in the global name space, and
    # then we look in the orca namespace.
    #
    moduleName = settings.speechServerFactory

    debug.println(debug.LEVEL_CONFIGURATION,
                  "Using speech server factory: %s" % moduleName)

    factory = None
    try:
        factory =  __import__(moduleName,
                              globals(),
                              locals(),
                              [''])
    except:
        try:
            moduleName = moduleName.replace("orca.","",1)
            factory =  __import__(moduleName,
                                  globals(),
                                  locals(),
                                  [''])
        except:
            debug.printException(debug.LEVEL_SEVERE)

    # Now, get the speech server we care about.
    #
    speechServerInfo = settings.speechServerInfo
    if speechServerInfo:
        __speechserver = factory.SpeechServer.getSpeechServer(speechServerInfo)
    else:
        __speechserver = factory.SpeechServer.getSpeechServer()

    # If we want to listen for speak commands from a separate port,
    # do so.  We make it a daemon so it will die automatically when
    # orca dies.
    #
    global _speakRequestThread
    if settings.speechServerPort and (not _speakRequestThread):
        try:
            _speakRequestThread = _SpeakRequestThread()
            _speakRequestThread.setDaemon(True)
            _speakRequestThread.start()
        except:
            debug.printException(debug.LEVEL_SEVERE)

def __resolveACSS(acss=None):
    if acss:
        return acss
    else:
        voices = settings.voices
        return voices[settings.DEFAULT_VOICE]

def sayAll(utteranceIterator, progressCallback):
    if settings.silenceSpeech:
        return
    if __speechserver:
        __speechserver.sayAll(utteranceIterator, progressCallback)

def speak(text, acss=None, interrupt=True):
    """Speaks all queued text immediately.  If text is not None,
    it is added to the queue before speaking.

    Arguments:
    - text:      optional text to add to the queue before speaking
    - acss:      acss.ACSS instance; if None,
                 the default voice settings will be used.
                 Otherwise, the acss settings will be
                 used to augment/override the default
                 voice settings.
    - interrupt: if True, stops any speech in progress before
                 speaking the text
    """

    # We will not interrupt a key echo in progress.
    #
    if orca.lastKeyEchoTime:
        interrupt = interrupt \
            and ((time.time() - orca.lastKeyEchoTime) > 0.5)

    if settings.silenceSpeech:
        return
    if __speechserver:
        debug.println(debug.LEVEL_INFO, "SPEECH OUTPUT: '" + text + "'")
        __speechserver.speak(text, __resolveACSS(acss), interrupt)

def isSpeaking():
    """"Returns True if the system is currently speaking."""
    if __speechserver:
        return __speechserver.isSpeaking()
    else:
        return False

def speakUtterances(utterances, acss=None, interrupt=True):
    """Speaks the given list of utterances immediately.

    Arguments:
    - list:      list of strings to be spoken
    - acss:      acss.ACSS instance; if None,
                 the default voice settings will be used.
                 Otherwise, the acss settings will be
                 used to augment/override the default
                 voice settings.
    - interrupt: if True, stop any speech currently in progress.
    """

    # We will not interrupt a key echo in progress.
    #
    if orca.lastKeyEchoTime:
        interrupt = interrupt \
            and ((time.time() - orca.lastKeyEchoTime) > 0.5)

    if settings.silenceSpeech:
        return
    if __speechserver:
        for utterance in utterances:
            debug.println(debug.LEVEL_INFO,
                          "SPEECH OUTPUT: '" + utterance + "'")
        __speechserver.speakUtterances(utterances,
                                       __resolveACSS(acss),
                                       interrupt)

def stop():
    if __speechserver:
        __speechserver.stop()

def increaseSpeechRate(script=None, inputEvent=None):
    if __speechserver:
        __speechserver.increaseSpeechRate()
    return True

def decreaseSpeechRate(script=None, inputEvent=None):
    if __speechserver:
        __speechserver.decreaseSpeechRate()
    return True

def increaseSpeechPitch(script=None, inputEvent=None):
    if __speechserver:
        __speechserver.increaseSpeechPitch()
    return True

def decreaseSpeechPitch(script=None, inputEvent=None):
    if __speechserver:
        __speechserver.decreaseSpeechPitch()
    return True

def shutdown():
    global __speechserver
    if __speechserver:
        __speechserver.shutdownActiveServers()
        __speechserver = None

def reset(text=None, acss=None):
    global __speechserver
    if __speechserver:
        __speechserver.reset(text, acss)

def testNoSettingsInit():
    init()
    speak("testing")
    speak("this is higher", ACSS({'average-pitch' : 7}))
    speak("this is slower", ACSS({'rate' : 3}))
    speak("this is faster", ACSS({'rate' : 80}))
    speak("this is quiet",  ACSS({'gain' : 2}))
    speak("this is loud",   ACSS({'gain' : 10}))
    speak("this is normal")

def test():
    import speechserver
    factories = getSpeechServerFactories()
    for factory in factories:
        print factory.__name__
        servers = factory.SpeechServer.getSpeechServers()
        for server in servers:
            try:
                print "    ", server.getInfo()
                for family in server.getVoiceFamilies():
                    name = family[speechserver.VoiceFamily.NAME]
                    print "      ", name
                    acss = ACSS({ACSS.FAMILY : family})
                    server.speak(name, acss)
                    server.speak("testing")
                server.shutdown()
            except:
                debug.printException(debug.LEVEL_OFF)
                pass
