/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <glib.h>

#include <string.h>

#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/article-actions.h>
#include <pan/message-window.h>
#include <pan/nntp.h>
#include <pan/queue.h>
#include <pan/task.h>
#include <pan/task-bodies.h>
#include <pan/task-func-ptr.h>

/**
 * Find a matching article without comparing by message-id.
 * This is because many news servers alter the message-id of
 * an article being posted, so we can't rely on searching by
 * the message-id that Pan generated.
 *
 * If a match is found, the refcount of Pan.sent is incremented
 * by one and must be decremented by the caller.
 */
static Article*
find_matching_sent_article (const Article * article)
{
	Group * sent;
	Article * retval = NULL;
	GPtrArray * articles;
	guint i;
	debug_enter ("find_matching_sent_article");

       	sent = serverlist_get_named_folder (PAN_SENT);

	/* sanity clause */
	g_return_val_if_fail (article!=NULL, NULL);
	g_return_val_if_fail (sent!=NULL, NULL);

	group_ref_articles (sent, NULL);
	articles = group_get_article_array (sent);

	for (i=0; retval==NULL && i!=articles->len; ++i)
	{
		gchar * body_1;
		gchar * body_2;
		Article * compare = ARTICLE(g_ptr_array_index(articles,i));

		/* comparison #1: subject must match */
		if (pan_strcmp (article_get_subject(compare), article_get_subject(article)))
			continue;

		/* comparison #2: author address must match */
		if (pan_strcmp (compare->author_addr, article->author_addr))
			continue;

		/* comparison #3: body address must match */
		body_1 = article_get_body (article);
		body_2 = article_get_body (compare);
		if (body_1!=NULL && body_2!=NULL)
		{
			g_strstrip (body_1);
			g_strstrip (body_2);
			if (!strcmp (body_1, body_2))
				retval = compare;
		}
		g_free (body_1);
		g_free (body_2);
	}

	if (retval == NULL)
		group_unref_articles (sent, NULL);

	debug_exit ("find_matching_sent_article");
	g_ptr_array_free (articles, TRUE);
	return retval;
}

static gint
article_cancel_run (TaskFuncPtr* task, gpointer user_data)
{
        int status = nntp_cancel (STATUS_ITEM(task),
	                          ARTICLE(user_data),
	                          TASK(task)->sock);

	if (status == TASK_SUCCESS)
	{
		/* FIXME: add something to the article to denote that it's been cancelled */
	}

	return 0;
}
static gchar*
article_cancel_describe (const StatusItem * status)
{
	return g_strdup (_("Cancelling article"));
}
static void
article_cancel_dtor (gpointer data)
{
	debug_enter ("article_cancel_dtor");

	group_unref_articles (serverlist_get_named_folder(PAN_SENT), NULL);

	debug_exit ("article_cancel_dtor");
}
void
article_cancel (Article * article)
{
	Server * server;
	Article * ours;
	const gchar * newsgroups;

	/* sanity clause */
	g_return_if_fail (article != NULL);
	server = article->group==NULL ? NULL : article->group->server;
	g_return_if_fail (server != NULL);
	newsgroups = article_get_header (article, HEADER_NEWSGROUPS);
	g_return_if_fail (is_nonempty_string (newsgroups));

	ours = find_matching_sent_article (article);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to cancel article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		queue_add (TASK(task_func_ptr_new (server,
			article_cancel_describe,
			article_cancel_run,
			article_cancel_dtor, 
			ours, FALSE)));
	}
}

/****
*****
****/

void
article_supersede (const Article * article)
{
	const Article * ours;
	Group * sendlater;
	debug_enter ("article_supersede");

       	sendlater = serverlist_get_named_folder (PAN_SENDLATER);

	/* sanity clause */
	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (sendlater != NULL);

	/* find the original copy so that the headers will match */
	ours = find_matching_sent_article (article);
	if (ours == NULL)
	{
		log_add_va (LOG_ERROR|LOG_URGENT,
		            _("Unable to supersede article: Couldn't find matching article in folder `pan.sent'!"));
	}
	else
	{
		gchar * message_id = gnksa_generate_message_id_from_email_addr (ours->author_addr);
		Article * super = article_dup (sendlater, ours);
		article_set_header (super, HEADER_MESSAGE_ID, message_id, DO_CHUNK);
		article_set_header (super, HEADER_SUPERSEDES, article_get_message_id(article), DO_CHUNK);

		/* edit the copy */
		message_edit_window (super);

		/* cleanup */
		g_free (message_id);
		group_unref_articles (ours->group, NULL);
		/* FIXME: super dangles */
	}

	debug_exit ("article_supersede");
}

/****
*****
****/

static gint
article_copy_articles_to_folder_cb (gpointer obj, gpointer arg, gpointer data)
{
	gint status;
	TaskBodies * task;
	GPtrArray * articles;
	Group * folder;
	debug_enter ("article_copy_articles_to_folder_cb");

	/* sanity clause */
	g_return_val_if_fail (obj!=NULL, 0);
	g_return_val_if_fail (data!=NULL, 0);
	g_return_val_if_fail (group_is_folder(GROUP(data)), 0);

	status = GPOINTER_TO_INT(arg);
	task = TASK_BODIES(obj);
	articles = task->articles;
	folder = GROUP(data);

	if (status == TASK_SUCCESS)
	{
		gint i;
		GPtrArray * addme;

		/* init */
		group_ref_articles (folder, NULL);

		/* create an array of articles to add. */
		addme = g_ptr_array_new ();
		pan_g_ptr_array_reserve (addme, articles->len);
		for (i=0; i!=articles->len; ++i)
		{
			const Article * a_old;
			Article * a_new;
			gchar * body;

			a_old = ARTICLE(g_ptr_array_index(articles,i));
			body = article_get_body (a_old);
			a_new = article_dup (folder, a_old);
			article_set_header (a_new, PAN_BODY, body, DO_CHUNK);
			g_ptr_array_add (addme, a_new);

			g_free (body);
		}

		/* smoke 'em if you've got 'em */
		if (addme->len != 0)
			group_add_articles (folder, addme, NULL, NULL, NULL);

		/* clean up */
		group_unref_articles (folder, NULL);
		g_ptr_array_free (addme, TRUE);
	}

	debug_exit ("articlelist_article_to_folder_cb");
	return 0;
}

gboolean
article_copy_articles_to_folder (Group * folder, Article ** articles, gint qty)
{
	gint i;
	Task * task;
	GPtrArray * a;
	debug_enter ("article_copy_articles_to_folder");

	/* sanity clause */
	g_return_val_if_fail (folder!=NULL, FALSE);
	g_return_val_if_fail (group_is_folder(folder), FALSE);
	g_return_val_if_fail (articles!=NULL, FALSE);
	g_return_val_if_fail (qty>0, FALSE);
	for (i=0; i<qty; ++i)
		g_return_val_if_fail (article_is_valid(articles[i]), FALSE);

	/* make a task to download the bodies if we don't already have them */
	a = g_ptr_array_new ();
	pan_g_ptr_array_assign (a, (gpointer*)articles, qty);
	task = TASK(task_bodies_new (articles[0]->group, a));
	g_ptr_array_free (a, TRUE);
	if (task != NULL) {
		pan_callback_add (task->task_ran_callback, article_copy_articles_to_folder_cb, folder);
		queue_add (task);
	}

	debug_exit ("article_copy_articles_to_folder");
	return TRUE;
}
