/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/server.h>
#include <pan/base/group.h>
#include <pan/base/util-mime.h>

const gchar * default_incoming_name_addr;
const gchar * default_incoming_name_real;

static void fire_articles_changed (Group*, Article**, gint article_qty, ArticleChangeType);

/***
****
****  LIFE CYCLE
****
***/

void
article_destructor (PanObject* o)
{
	Article * a = ARTICLE(o);
        g_return_if_fail (a != NULL);

	/* clean up article parts */
	debug1 (DEBUG_PAN_OBJECT, "article_destructor: %p", a);
	if (a->headers != NULL) {
		g_hash_table_destroy (a->headers);
		a->headers = NULL;
	}
	if (a->threads != NULL) {
		g_slist_free (a->threads);
		a->threads = NULL;
	}
	a->date = 0;
	a->number = 0;
	a->linecount = 0;
	a->message_id = NULL;
	a->xref = NULL;
	a->references = NULL;
	a->subject = NULL;
	a->author_addr = NULL;
	a->author_real = NULL;
	a->group = NULL;

	/* clean up parent */
	pan_object_destructor (o);
}

void
article_constructor (Article * article, Group * group)
{
	g_return_if_fail (article!=NULL);
	g_return_if_fail (group!=NULL);

	/* init the superclass */
	pan_object_constructor (PAN_OBJECT(article), article_destructor);

	/* init this class */
        debug1 (DEBUG_PAN_OBJECT, "article_constructor: %p", article);
	article->group = group;
	article->self_passes_filter = FALSE;
	article->tree_passes_filter = FALSE;
	article->part = (gint16)0;
	article->parts = (gint16)0;
	article->score = (gint16)0;
	article->crosspost_qty = (gint8)0;
	article->linecount = (guint16)0;
	article->state = (guint16)0;
	article->unread_children = (guint16)0;
	article->new_children = (guint16)0;
	article->date = (time_t)0;
	article->number = (gulong)0;
	article->subject = NULL;
	article->author_real = NULL;
	article->author_addr = NULL;
	article->message_id = NULL;
	article->xref = NULL;
	article->references = NULL;
	article->parent = NULL;
	article->threads = NULL;
	article->headers = NULL;
}

Article*
article_new (Group * group)
{
	Article* article = g_new (Article, 1);
	article_constructor (article, group);
	return article;
}


static void
article_dup_ghfunc (gpointer key, gpointer val, gpointer user_data)
{
	Article * article = ARTICLE(user_data);
	article_init_header (article, key, val, DO_CHUNK_SHARE);
}
Article*
article_dup (Group * group, const Article * article)
{
	Article * retval;

	g_return_val_if_fail (article_is_valid(article), NULL);
	g_return_val_if_fail (group!=NULL, NULL);

#define chunk_if_exists(A, share) \
	is_nonempty_string (A) ? group_chunk_string (group, A, share) : NULL

	retval                      = article_new (group);
	retval->self_passes_filter  = FALSE;
	retval->tree_passes_filter  = FALSE;
	retval->unread_children     = 0;
	retval->new_children        = 0;
	retval->group               = group;
	retval->parts               = article->parts;
	retval->part                = article->part;
	retval->linecount           = article->linecount;
	retval->crosspost_qty       = article->crosspost_qty;
	retval->state               = article->state;
	retval->date                = article->date;
	retval->number              = article->number;
	retval->xref                = chunk_if_exists (article->xref, FALSE);
	retval->subject             = chunk_if_exists (article->subject, TRUE);
	retval->author_real         = chunk_if_exists (article->author_real, TRUE);
	retval->author_addr         = chunk_if_exists (article->author_addr, TRUE);
	retval->message_id          = chunk_if_exists (article->message_id, FALSE);
	retval->references          = chunk_if_exists (article->references, TRUE);
	retval->parent              = NULL;
	retval->threads             = NULL;
	retval->headers             = NULL;

	if (article->headers != NULL)
		g_hash_table_foreach (article->headers, article_dup_ghfunc, retval);

	return retval;
}

/***
****
***/

static void
articles_set_dirty (Article ** articles, gint qty)
{
	Group * group;

	/* sanity check */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles != NULL);
	g_return_if_fail (articles[0] != NULL);

	/* mark the articles' group as read */
	group = articles[0]->group;
	group_set_articles_dirty (group);
	fire_articles_changed (group, articles, qty, ARTICLE_CHANGED_DIRTY);
}



/**
 * Remove an article from its parent/child threadings.  The children and
 * parent are linked together instead.  This is useful for not breaking
 * the threading of articles surrounding an article which is about to be
 * destroyed.
 */
void
article_isolate (Article* article)
{
	GSList *l = NULL;
	Article *parent = article->parent;

	if (parent)
		parent->threads = g_slist_remove (parent->threads, article);

	for (l=article->threads; l!=NULL; l=l->next) {
		Article *child = ARTICLE(l->data);
		child->parent = parent;
		if (parent)
			parent->threads = g_slist_append (parent->threads, child);
	}

	g_slist_free (article->threads);
	article->threads = NULL;
	article->parent = NULL;
}


gboolean
article_is_new (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);
	return a->number > a->group->article_high_old;
}


/***
****
****  READ / UNREAD
****
***/

gboolean
article_is_read (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);

	return newsrc_is_article_read (group_get_newsrc(a->group), a->number);
}

static void
articles_set_read_numbers_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	gint i;
	gint changed_qty;
	Group * g = GROUP(key);
	GArray * numbers = (GArray*) value;
	gboolean read = user_data!=NULL;
	Newsrc * newsrc = group_get_newsrc (g);

	/* for each number, mark it read */
	for (i=changed_qty=0; i<numbers->len; ++i) {
		const gulong number = g_array_index (numbers, gulong, i);
		if (read != newsrc_mark_article (newsrc, number, read))
			++changed_qty;
	}

	/* if changed, alter the group's read qty */
	if (changed_qty != 0)
		group_inc_article_read_qty (g, read ? changed_qty : -changed_qty);

	g_array_free (numbers, TRUE);
}

static void
articles_set_read_articles_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	gint i;
	GPtrArray * changed = g_ptr_array_new ();
	Group * g = GROUP(key);
	GArray * articles = (GArray*) value;
	gboolean read = user_data!=NULL;
	Newsrc * newsrc = group_get_newsrc (g);

	/* for each article, mark it read */
	for (i=0; i<articles->len; ++i) {
		Article * a = ARTICLE(g_array_index(articles,Article*,i));
		if (read != newsrc_mark_article (newsrc, a->number, read))
			g_ptr_array_add (changed, a);
	}

	/* if changed, alter the group's read qty & fire an event */
	if (changed->len != 0) {
		group_inc_article_read_qty (g, read ? changed->len : -changed->len);
		fire_articles_changed (g, (Article**)changed->pdata, changed->len, ARTICLE_CHANGED_READ);
	}

	g_array_free (articles, TRUE);
}

static GArray*
get_group_array_from_hash (GHashTable * hash, Group * group, gint size)
{
	GArray * arr = (GArray*) g_hash_table_lookup (hash, group);
	if (arr == NULL)
		g_hash_table_insert (hash, group, arr = g_array_new (FALSE, FALSE, size));
	return arr;
}

static void
mark_read_xreffunc (Group * g, gulong number, gpointer user_data)
{
	GHashTable ** group_to_numbers = (GHashTable**) user_data;
	if (group_is_subscribed(g)) {
		if (*group_to_numbers == NULL)
			*group_to_numbers = g_hash_table_new (g_direct_hash, g_direct_equal);
		g_array_append_val (get_group_array_from_hash(*group_to_numbers,g,sizeof(gulong)), number);
	}
}

void
articles_set_read (Article ** articles, int article_qty, gboolean read)
{
	int i;
	GHashTable * group_to_articles;
	GHashTable * group_to_numbers = NULL;

	debug_enter ("articles_set_read");

	/* sanity check */
	g_return_if_fail (articles != NULL);
	g_return_if_fail (article_qty > 0);

	group_to_articles = g_hash_table_new (g_direct_hash, g_direct_equal);
	for (i=0; i<article_qty; ++i)
	{
		GSList * l;
		GArray * arr;
		Article * a = articles[i];

		/* add this article to the group's array */
	       	arr = get_group_array_from_hash (group_to_articles, a->group, sizeof(Article*));
		g_array_append_val (arr, a);
		article_xref_foreach (a, mark_read_xreffunc, &group_to_numbers, TRUE);

		/* if it's a multipart, add parts 2...n as well */
		if (a->part>0 && a->parts>1 && a->threads!=NULL) {
			for (l=a->threads; l!=NULL; l=l->next) {
				Article * child = ARTICLE(l->data);
				if (child->part > 0) {
					g_array_append_val (arr, child);
					article_xref_foreach (child, mark_read_xreffunc, &group_to_numbers, TRUE);
				}
			}
		}
	}

	/* mark the passed-in articles as read/unread */
	g_hash_table_foreach (group_to_articles, articles_set_read_articles_ghfunc, GINT_TO_POINTER(read));
	g_hash_table_destroy (group_to_articles);

	/* mark their crosspost counterparts, if any, as read/unread */
	if (group_to_numbers != NULL) {
		g_hash_table_foreach (group_to_numbers, articles_set_read_numbers_ghfunc, GINT_TO_POINTER(read));
		g_hash_table_destroy (group_to_numbers);
	}
}


/**
*** PUBLIC MUTATORS
**/

void
articles_add_flag (Article  ** articles,
                   gint        qty,
                   gushort     flag)
{
	gint i;

	g_return_if_fail (articles != NULL);
	g_return_if_fail (qty >= 1);

	for (i=0; i<qty; ++i)
		articles[i]->state |= flag;

	articles_set_dirty (articles, qty);
}


void
articles_remove_flag (Article         ** articles,
                      gint               qty,
                      gushort            flag)
{
	gint i;

	g_return_if_fail (articles != NULL);
	g_return_if_fail (qty >= 1);

	for (i=0; i<qty; ++i)
		articles[i]->state &= ~flag;

	articles_set_dirty (articles, qty);
}



/***
****
****  THREADS
****
***/

gchar*
article_get_thread_message_id (const Article* article)
{
	char * pch;
	const gchar * refs = article_get_header (article, HEADER_REFERENCES);
	const gchar * msg_id = article_get_message_id (article);

	g_return_val_if_fail (article!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(msg_id), NULL);

	if (is_nonempty_string(refs))
	{
		pch = get_next_token_str (refs, ' ', NULL);
	}
	else /* top of the thread */
	{
		pch = g_strdup (msg_id);
	}

	return pch;
}

static void
article_get_entire_thread_impl (Article* top, GPtrArray* setme)
{
	GSList *l;
	g_ptr_array_add (setme, top);
	for (l=top->threads; l!=NULL; l=l->next)
		article_get_entire_thread_impl (ARTICLE(l->data), setme);
}

static void
article_get_thread_impl (Article * article,
                         GPtrArray * setme,
                         gboolean children_only)
{
	Article* top = article;

	if (top != NULL)
	{
		if (!children_only)
			while (top->parent != NULL)
				top = top->parent;

		pan_g_ptr_array_reserve (setme, 128);
		article_get_entire_thread_impl (top, setme);
	}
}

void
article_get_subthread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, TRUE);
}

void
article_get_entire_thread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, FALSE);
}

void
article_get_references (Article* article, GPtrArray* setme)
{
	g_return_if_fail (setme!=NULL);
	g_return_if_fail (article!=NULL);

	pan_g_ptr_array_reserve (setme, 128);
	while (article != NULL) {
		g_ptr_array_add (setme, article);
		article = article->parent;
	}
}


/***
****
****  OTHER HEADERS
****
***/

void
article_remove_header (Article * a, const gchar * key)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(key));
		 
	if (!strcmp(key, HEADER_MESSAGE_ID))
		a->message_id = NULL;
	else if (!strcmp(key, HEADER_XREF))
		a->xref = NULL;
	else if (!strcmp(key, HEADER_SUBJECT))
		a->subject = NULL;
	else if (!strcmp(key, HEADER_REFERENCES))
		a->references = NULL;
	else if (!strcmp(key, HEADER_FROM))
	{
		a->author_addr = NULL;
		a->author_real = NULL;
	}
	else if (a->headers!=NULL)
	{
		g_hash_table_remove (a->headers, key);
		if (!g_hash_table_size(a->headers))
		{
			g_hash_table_destroy (a->headers);
			a->headers = NULL;
		}
	}
}

const gchar*
article_get_header (const Article * a, const gchar * key)
{
	const gchar * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(a), NULL);
	g_return_val_if_fail (is_nonempty_string(key), NULL);

	if (key == HEADER_MESSAGE_ID) retval = a->message_id;
	else if (key == HEADER_SUBJECT) retval = a->subject;
	else if (key == HEADER_REFERENCES) retval = a->references;
	else if (key == HEADER_FROM) pan_warn_if_reached ();
	else if (!strcmp(key, HEADER_MESSAGE_ID)) retval = a->message_id;
	else if (!strcmp (key, HEADER_SUBJECT)) retval = a->subject;
	else if (!strcmp (key, HEADER_REFERENCES)) retval = a->references;
	else if (!strcmp (key, HEADER_FROM)) pan_warn_if_reached ();
	else if (a->headers!=NULL)
		retval = (const gchar*) g_hash_table_lookup (a->headers, key);

	return retval;
}

gboolean
article_header_is_internal (const gchar * key)
{
	return is_nonempty_string(key) && !g_strncasecmp(key,"Pan-",4);
}

gboolean
article_header_is_extra (const gchar * key)
{
	/* sanity check */
	if (!is_nonempty_string(key)) return FALSE;

	/* pan internals aren't user-specified headers */
	if (article_header_is_internal(key)) return FALSE;

	/* other headers that are handled explicitly elsewhere */
	if (!strcmp(key,HEADER_FOLLOWUP_TO)) return FALSE;
	if (!strcmp(key,HEADER_NEWSGROUPS)) return FALSE;
	if (!strcmp(key,HEADER_ORGANIZATION)) return FALSE;
	if (!strcmp(key,HEADER_REPLY_TO)) return FALSE;

	return TRUE;
}

static void
ensure_extra_headers_exists (Article * a)
{
	if (a->headers == NULL)
		a->headers = g_hash_table_new (g_str_hash, g_str_equal);
}

void
article_xref_foreach (const Article       * a,
                      ArticleXRefFunc       func,
                      gpointer              user_data,
                      gboolean              skip_group_a)
{
	gchar * str;
	Server * server;
	const gchar * xref;
	const gchar * march;

	/* sanity clause */
	g_return_if_fail (article_is_valid(a));
	g_return_if_fail (a->group!=NULL);
	g_return_if_fail (a->group->server!=NULL);

	/* get the xref header */
	xref = article_get_header (a, HEADER_XREF);
	if (!is_nonempty_string(xref))
		return;

	/* skip the servername; we've got the server already */
	skip_next_token (xref, ' ', &march);

	/* walk through the xrefs, of format "group1:number group2:number" */
	server = a->group->server;
	while ((str = get_next_token_str (march, ' ', &march)))
	{
		gchar * delimit = strchr (str, ':');

		if (delimit != NULL)
		{
			const gulong number = strtoul (delimit+1, NULL, 10);
			Group * g;
			*delimit = '\0';
			g = server_get_named_group (server, str);

			if (g!=NULL && (a->group!=g || !skip_group_a))
				(*func)(g, number, user_data);
		}

		g_free (str);
	}
}

static void
article_set_header_impl (Article         * a,
                         const gchar     * key,
                         const gchar     * val,
                         HeaderAction      action,
                         gboolean          init)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(key));

	/* chunk if necessary */
	if (action==DO_CHUNK || action==DO_CHUNK_SHARE)
		val = group_chunk_string (a->group, val, action==DO_CHUNK_SHARE);

	/* if we're possibly erasing a header, remove the old value */
	if (!is_nonempty_string (val))
		article_remove_header (a, key);

	/* add the new header */
	if (is_nonempty_string (val))
	{
		if      (!strcmp (key, HEADER_MESSAGE_ID)) a->message_id = val;
		else if (!strcmp (key, HEADER_SUBJECT)) a->subject = val;
		else if (!strcmp (key, HEADER_REFERENCES)) a->references = val;
		else if (!strcmp (key, HEADER_XREF)) a->xref = val;
		else if (!strcmp (key, HEADER_FROM))
		{
			if (init)
				article_init_author_from_header (a, val);
			else
				article_set_author_from_header (a, val);
		}
		else
		{
			ensure_extra_headers_exists (a);
			key = group_chunk_string (a->group, key, TRUE);
			g_hash_table_insert (a->headers,
			                     (gpointer)key,
			                     (gpointer)val);
			if (!init)
				articles_set_dirty (&a, 1);
		}
	}
}

void
article_init_header (Article         * a,
                     const gchar     * key,
                     const gchar     * val,
                     HeaderAction      action)
{
	article_set_header_impl (a, key, val, action, TRUE);
}

void
article_set_header (Article         * a,
                    const gchar     * key,
                    const gchar     * val,
                    HeaderAction      action)
{
	article_set_header_impl (a, key, val, action, FALSE);
}

static void
article_get_all_headers_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	GPtrArray * a = (GPtrArray*)user_data;
	g_ptr_array_add (a, key);
	g_ptr_array_add (a, value);
}

GPtrArray*
article_get_all_headers (const Article * a)
{
	GPtrArray * retval = g_ptr_array_new ();

	/* sanity clause */
	g_return_val_if_fail (a!=NULL, retval);

	/* add the headers */
	if (a->headers != NULL)
	{
		pan_g_ptr_array_reserve (retval, g_hash_table_size(a->headers)*2);
		g_hash_table_foreach (a->headers, article_get_all_headers_ghfunc, retval);
	}

	return retval;
}

gboolean
article_has_attachment (const Article *a)
{
	g_return_val_if_fail (article_is_valid(a), FALSE);

	return article_get_header(a,PAN_ATTACH_FILE) != NULL;
}

/***
****
****  BODY
****
***/

gboolean
article_has_body (const Article * a)
{
	gboolean retval = FALSE;

	g_return_val_if_fail (article_is_valid(a), FALSE);

	/* maybe the article's in a folder so the body
	 * is hidden as a header... */
	if (!retval) {
		const gchar * body = article_get_header (a, PAN_BODY);
		if (body != NULL)
			retval = TRUE;
	}

	/* try to get it from acache... */
	if (!retval && is_nonempty_string(a->message_id))
		retval = acache_has_message (a->message_id);

	return retval;
}

gchar*
article_get_author_str (const Article * a)
{
        gchar * retval;

	if (a==NULL)
	{
		retval = g_strdup ("");
	}
	else if (is_nonempty_string(a->author_addr))
	{
		if (is_nonempty_string(a->author_real))
		{
			retval = g_strdup_printf ("\"%s\" <%s>", a->author_real, a->author_addr);
		}
		else
		{
			retval = g_strdup (a->author_addr);
		}
	}
	else if (is_nonempty_string(a->author_real))
	{
		retval = g_strdup (a->author_real);
	}
	else
	{
		retval = g_strdup ("");
	}

	g_strstrip (retval);

	return retval;
}

gchar*
article_get_short_author_str (const Article * a)
{
	gchar * author;

	if (is_nonempty_string (a->author_real) && strcmp(a->author_real,default_incoming_name_real))
	{
		author = g_strdup (a->author_real);
	}
	else if (is_nonempty_string (a->author_addr) && strcmp(a->author_addr,default_incoming_name_addr))
	{
		author = g_strdup (a->author_addr);
		if (author != NULL) {
			gchar * pch = strchr (author, '@');
			if (pch != NULL)
				*pch = '\0';
		}
	}
	else
	{
		author = g_strdup (default_incoming_name_real);
	}

	return author;
}


gchar*
article_get_message (const Article * a)
{
	gchar * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(a), NULL);

	if (retval==NULL && a!=NULL && acache_has_message (a->message_id))
	{
		retval = acache_get_message (a->message_id);
	}

	if (retval==NULL && a!=NULL)
	{
		GString * s = g_string_new (NULL);
		gchar * pch;

		pch = article_get_headers (a);
		g_string_append (s, pch);
		g_free (pch);

		g_string_append (s, "\n");

		pch = article_get_body (a);
		g_string_append (s, pch);
		g_free (pch);

		retval = s->str;
		g_string_free (s, FALSE);
	}

	return retval;
}

gchar*
article_get_headers (const Article * a)
{
	gchar * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), g_strdup(""));

	/* do we have it in acache? */
	if (retval==NULL && a!=NULL && acache_has_message (a->message_id))
	{
		gchar * s = acache_get_message (a->message_id);
		gchar * pch = pan_strstr (s, "\n\n");
		if (pch != NULL)
			retval = g_strndup (s, (gint)(pch-s));
		g_free (s);
	}

	/* can we build it from a? */
	if (retval==NULL && a!=NULL)
	{
		gchar * tmp;
		const gchar * c_tmp;
		GString * s = g_string_new (NULL);

		/* subject */
		c_tmp = a->subject;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, "Subject: %s\n", c_tmp);

		/* author */
	       	tmp = article_get_author_str (a);
		if (is_nonempty_string (tmp))
			g_string_sprintfa (s, "From: %s\n", tmp);
		g_free (tmp);

		/* date */
		if (a->date != 0) {
			tmp = rfc822_date_generate (a->date);
			g_string_sprintfa (s, "Date: %s\n", tmp);
			g_free (tmp);
		}

		/* message-id */
		c_tmp = a->message_id;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, "Message-Id: %s\n", c_tmp);

		/* references */
		c_tmp = a->references;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, "References: %s\n", c_tmp);

		/* extra headers */
		if (1) {
			gint i;
			GPtrArray * z = article_get_all_headers (a);
			for (i=0; i<z->len; i+=2) {
				const char* key = (const char*) g_ptr_array_index (z, i);
				const char* val = (const char*) g_ptr_array_index (z, i+1);
				if (!article_header_is_internal (key))
					g_string_sprintfa (s, "%s: %s\n",  key, val);
			}
			g_ptr_array_free (z, TRUE);
		}

		retval = s->str;
		g_string_free (s, FALSE);
	}

	return retval;
}

gchar*
article_get_body (const Article * a)
{
	gchar * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), NULL);

	/* see if we've got the body hidden in a header... */
	if (retval==NULL && a!=NULL)
	{
		const gchar * body = article_get_header (a, PAN_BODY);
		if (body != NULL)
			retval = g_strdup (body);
	}

	/* see if we've got it in acache... */
	if (retval==NULL && a!=NULL && acache_has_message(a->message_id))
	{
		gchar * s = acache_get_message (a->message_id);
		gboolean is_html = FALSE;
		GMimeMessage * m = pan_g_mime_parser_construct_message (s);
		retval = g_mime_message_get_body (m, TRUE, &is_html);
		g_mime_message_destroy (m);
		g_free (s);
	}

	return retval;
}

static void
article_set_author_from_header_impl (Article       * a,
                                     const gchar   * header_from,
                                     gboolean        init)
{
	gchar * pch;
	gchar * addr = NULL;
	gchar * real = NULL;

	g_return_if_fail (a!=NULL);
	g_return_if_fail (a->group!=NULL);
	g_return_if_fail (is_nonempty_string(header_from));

	/* note that strict is FALSE here because we can't control
	 * how other people have posted to usenet */
	gnksa_do_check_from (header_from, &addr, &real, FALSE);
	replace_gstr (&addr, g_mime_utils_8bit_header_decode(addr));
	replace_gstr (&real, g_mime_utils_8bit_header_decode(real));

	/* use the real mail address, or fill in a default */
	pch = addr;
	if (!is_nonempty_string (pch))
		pch = (gchar*)default_incoming_name_addr;
	a->author_addr = group_chunk_string (a->group, pch, TRUE);

	/* use the real name, or fill in a default. */
	if (is_nonempty_string(real)) {
		gnksa_strip_realname (real);
		a->author_real = group_chunk_string (a->group, real, TRUE);
	}
	else if (addr!=NULL && ((pch=strchr(addr,'@'))!=NULL)) {
		gchar * tmp = g_strndup (addr, pch-addr);
		a->author_real = group_chunk_string (a->group, tmp, TRUE);
		g_free (tmp);
	}
	else {
		const gchar * cpch = (gchar*)default_incoming_name_real;
		a->author_real = group_chunk_string (a->group, cpch, TRUE);
	}

	if (!init)
		articles_set_dirty (&a, 1);

	/* cleanup */
	g_free (addr);
	g_free (real);
}

void
article_init_author_from_header (Article         * a,
                                 const gchar     * header_from)
{
	article_set_author_from_header_impl (a, header_from, TRUE);
}
void
article_set_author_from_header (Article         * a,
                                const gchar     * header_from)
{
	article_set_author_from_header_impl (a, header_from, FALSE);
}


void
article_set_from_raw_message   (Article         * a,
                                const gchar     * text)
{
	guint i;
	guint16 lines;
	GMimeMessage * msg;
	GPtrArray * arr;
	gchar * body;
	const gchar * pch;
	gboolean foo;

	/* get the message */
	g_return_if_fail (a != NULL);
	g_return_if_fail (is_nonempty_string(text));
	msg = pan_g_mime_parser_construct_message (text);
	g_return_if_fail (msg != NULL);

	/* body */
	a->date = msg->header->date;
	body = g_mime_message_get_body (msg, TRUE, &foo);
	article_init_header (a, PAN_BODY, body, DO_CHUNK);
	for (lines=0, pch=body; pch!=NULL && *pch; ++pch)
		if (*pch=='\n')
			++lines;
	a->linecount = lines;
	g_free (body);

	/* headers */
	article_init_header (a, HEADER_FROM, msg->header->from, DO_CHUNK_SHARE);
	article_init_header (a, HEADER_REPLY_TO, msg->header->reply_to, DO_CHUNK_SHARE);
	article_init_header (a, HEADER_SUBJECT, msg->header->subject, DO_CHUNK_SHARE);
	article_init_header (a, HEADER_MESSAGE_ID, msg->header->message_id, DO_CHUNK);
	arr = msg->header->arbitrary_headers;
	for (i=0; i!=arr->len; ++i)
	{
		GMimeHeader * h = (GMimeHeader*) g_ptr_array_index (arr, i);
		const gchar * name = h->name;
		if (!pan_strcmp (name, "Author")) /* back compatability with 0.9.2 */
			name = HEADER_FROM;
		article_init_header (a, h->name, h->value, DO_CHUNK_SHARE);
	}

	/* check to handle buggy <= 0.9.6 mboxes.  The problem was that Pan
	   translated the header names, so instead of "From:" we get "Von:"
	   and so on. */
	if (!is_nonempty_string(a->message_id))
	{
		g_warning (_("Couldn't parse header info mbox message; trying harder."));

		/* <= 0.9.6 wrote headers in a particular order: subject, author, date,
		   message-id, references. Since improper i18n is the most likely glitch
		   we're going to encounter here, do some simple checks to detect it. */
		if (arr->len>= 5
			&& (strchr(((GMimeHeader*)g_ptr_array_index(arr,1))->value,'@')!=NULL)
			&& (((GMimeHeader*)g_ptr_array_index(arr,3))->value[0]=='<'))
		{
			const GMimeHeader * h;
g_message ("looks like a <= 0.9.6 mbox");
			h = (GMimeHeader*) g_ptr_array_index (arr, 0);
			article_init_header (a, HEADER_SUBJECT, h->value, DO_CHUNK_SHARE);
			h = (GMimeHeader*) g_ptr_array_index (arr, 1);
			article_init_header (a, HEADER_FROM, h->value, DO_CHUNK_SHARE);
			h = (GMimeHeader*) g_ptr_array_index (arr, 2);
			a->date = g_mime_utils_header_decode_date (h->value, NULL);
			h = (GMimeHeader*) g_ptr_array_index (arr, 3);
			article_init_header (a, HEADER_MESSAGE_ID, h->value, DO_CHUNK);
		}
		else
		{
			gchar * id = gnksa_generate_message_id_from_email_addr (a->author_addr);
g_message ("dunno what this is");
			article_init_header (a, HEADER_MESSAGE_ID, id, DO_CHUNK);
			g_free (id);
			article_init_header (a, HEADER_SUBJECT, _("Unparseable Subject"), DO_CHUNK_SHARE);
		}
	}

	/* state */
	pch = article_get_header (a, "Status");
	if (pch != NULL) {
		const gboolean read = strchr (pch, 'R') != NULL;
		articles_set_read (&a, 1, read);
		article_remove_header (a, "Status");
	}

	/* references */
	pch = article_get_header (a, HEADER_REFERENCES);
	if (pch != NULL)
		a->references = pch;

	/* cleanup */
	g_mime_message_destroy (msg);
}


gboolean
article_is_valid (const Article * a)
{
#if 0
	g_assert (a!=NULL);
	g_assert (a->pan_object.ref_count>0);
	g_assert (a->message_id!=NULL && *a->message_id=='<');
	g_assert (is_nonempty_string(a->subject));
	g_assert (a->number!=0);
	g_assert (!a->references || *a->references=='<');
#else
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->pan_object.ref_count>0, FALSE);
	g_return_val_if_fail (a->message_id!=NULL && *a->message_id=='<', FALSE);
	g_return_val_if_fail (is_nonempty_string(a->subject), FALSE);
	g_return_val_if_fail (a->number!=0, FALSE);
	g_return_val_if_fail (!a->references || *a->references=='<', FALSE);
#endif
	return TRUE;
}

const gchar*
article_get_message_id (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), "");
	return a->message_id;
}
 
const gchar*
article_get_subject (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), "");
	return is_nonempty_string(a->subject) ? a->subject : "";
}



/***
****  EVENTS
***/

PanCallback*
article_get_articles_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_articles_changed (Group              * group,
                       Article           ** articles,
                       gint                 article_qty,
                       ArticleChangeType    type)
{
	ArticleChangeEvent e;
	debug_enter ("fire_articles_changed");

	g_return_if_fail (group!=NULL);
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (article_qty>0);
	g_return_if_fail (type==ARTICLE_CHANGED_READ || type==ARTICLE_CHANGED_DIRTY);

	e.group = group;
	e.articles = articles;
	e.article_qty = article_qty;
	e.type = type;
	pan_callback_call (article_get_articles_changed_callback(), &e, NULL);

	debug_exit ("fire_articles_changed");
}
