/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <limits.h>
#include <string.h>
#include <stdlib.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article-thread.h>
#include <pan/base/debug.h>
#include <pan/base/file-headers.h>
#include <pan/base/group.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/newsrc.h>
#include <pan/base/pan-callback.h>
#include <pan/base/server.h>

static void fire_group_changed (Group *);
static void fire_group_cleared (Group *);
static void fire_group_marked_read (Group *, gboolean read);

/**
***
**/

Group*
group_new (const char * name)
{
	Group * g = g_new (Group, 1);
	debug1 (DEBUG_PAN_OBJECT, "group_new: %p", g);
	pan_warn_if_fail (is_nonempty_string(name));
	group_constructor (g, group_destructor, name);
	return g;
}

void
group_constructor (Group                * g,
                   PanObjectDestructor    dtor,
		   const char           * name)
{
	pan_warn_if_fail (g!=NULL);
	pan_warn_if_fail (is_nonempty_string(name));

	pan_object_constructor (PAN_OBJECT(g), dtor);

	debug1 (DEBUG_PAN_OBJECT, "group_constructor: %p", g);
	g->articles_dirty = FALSE;
	g->articles_threading_dirty = FALSE;
	g->flags = (gint8)0;
	g->sort_style = (gint8)0;
	g->filter_bits = ~((guint16)STATE_FILTER_IGNORED);
	g->filter_show = FILTER_SHOW_MATCHES_AND_REFERENCES;
	g->article_qty = (gint32)0;
	g->article_read_qty = (gint32)0;
	g->article_low = (gulong)0;
	g->article_high_old = (gulong)0;
	g->permission = '?';
	g->article_high = (gulong)0;
	g->name = g_strdup (name);
	g->filter_name = NULL;
	g->description = NULL;
	g->readable_name = NULL;
	g->download_dir = NULL;
	g->chunk = NULL;
	g->server = NULL;

	g->_articles_refcount = (gint)0;
	g->_newsrc = NULL;
	g->_purged = NULL;
	g->_articles = NULL;
	g->_one_big_chunk = NULL;
}

void
group_destructor (PanObject * o)
{
	Group * g = GROUP(o);

	g_return_if_fail (o!=NULL);

	debug1 (DEBUG_PAN_OBJECT, "group_destructor: %p", g);

	/* strings */
	replace_gstr (&g->name, NULL);
	replace_gstr (&g->description, NULL);
	replace_gstr (&g->readable_name, NULL);
	replace_gstr (&g->download_dir, NULL);
	replace_gstr (&g->filter_name, NULL);

	/* newsrc */
	if (g->_newsrc != NULL)
		pan_object_unref (PAN_OBJECT(g->_newsrc));
	if (g->_purged != NULL)
		pan_object_unref (PAN_OBJECT(g->_purged));

	/* articles */
	pan_warn_if_fail (g->_articles_refcount >= 0);
	replace_gstr (&g->_one_big_chunk, NULL);
	if (g->chunk != NULL) {
		g_string_chunk_free (g->chunk);
		g->chunk = NULL;
	}
	if (g->_articles != NULL) {
		g_hash_table_destroy (g->_articles);
		g->_articles = NULL;
	}

	pan_object_destructor (o);
}

/***
****
****  PUBLIC ACCESSORS / MUTATORS
****
***/

void
group_set_new (Group * group, gboolean is_new)
{
	g_return_if_fail (group != NULL);

	if (is_new)
		group_set_flags (group, group->flags|GROUP_NEW);
	else
		group_set_flags (group, group->flags&~GROUP_NEW);
}

gboolean 
group_is_new (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return (group->flags & GROUP_NEW) ? 1 : 0;
}

void
group_set_subscribed (Group * group, gboolean subscribed)
{
	ServerGroupsType type;

	g_return_if_fail (group != NULL);

	type = SERVER_GROUPS_SUBSCRIBED|SERVER_GROUPS_UNSUBSCRIBED;
	if (group->server != NULL)
		server_set_group_type_dirty (group->server, type);

	if (subscribed)
		group_set_flags (group, group->flags|GROUP_SUBSCRIBED);
	else
		group_set_flags (group, group->flags&~GROUP_SUBSCRIBED);
}

gboolean 
group_is_subscribed (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);

	return (group->flags & GROUP_SUBSCRIBED) ? 1 : 0;
}

void
group_set_dirty (Group * group)
{
	g_return_if_fail (group != NULL);

	if (group->server != NULL)
		server_set_group_type_dirty (group->server, server_get_group_type (group));
}

void
group_set_download_dir (Group * group, const gchar* download_dir)
{
	g_return_if_fail (group!=NULL);

	replace_gstr (&group->download_dir, g_strdup(download_dir));
}

void
group_set_flags (Group * group, guint flags)
{
	g_return_if_fail (group!=NULL);

	group->flags = flags;
	group_set_dirty (group);
}

void
group_set_filter (Group         * group,
                  guint           filter_bits,
                  gulong          filter_show,
                  const gchar   * filter_name)
{
	g_return_if_fail (group!=NULL);

	group->filter_bits = filter_bits;
	group->filter_show = filter_show;
	replace_gstr (&group->filter_name, g_strdup(filter_name));

	group_set_dirty (group);
}

void
group_set_sort_style (Group * group, int sort_style)
{
	g_return_if_fail (group != NULL);

	group->sort_style = sort_style;
	group_set_dirty (group);
}

void
group_set_article_range (Group    * g,
                         gulong     article_low,
                         gulong     article_high)
{
	g_return_if_fail (g!=NULL);

	g->article_low = article_low;
	g->article_high = article_high;
	newsrc_set_group_range (group_get_newsrc(g), g->article_low, g->article_high);
	newsrc_set_group_range (group_get_purged(g), g->article_low, g->article_high);

	group_set_dirty (g);
}

void
group_get_article_range (const Group   * g,
                         gulong        * article_low,
                         gulong        * article_high)
{
	g_return_if_fail (g!=NULL);
	g_return_if_fail (article_low!=NULL);
	g_return_if_fail (article_high!=NULL);

	*article_low = g->article_low;
	*article_high = g->article_high;
}


void
group_mark_new_article_number (Group         * group,
                               gulong          everything_above_this_is_new)
{
	group->article_high_old = everything_above_this_is_new;
}

void
group_set_article_qty (Group * group, gint article_qty)
{
	g_return_if_fail (group != NULL);

	if (article_qty != group->article_qty)
	{
		group->article_qty = article_qty;
		group_set_dirty (group);
		fire_group_changed (group);
	}
}

void
group_set_article_read_qty (Group * group, gint article_read_qty)
{
	g_return_if_fail (group != NULL);

	article_read_qty = MAX (article_read_qty, 0);
	group->article_read_qty = article_read_qty;
	group_set_dirty (group);
	fire_group_changed (group);
}
void
group_inc_article_read_qty (Group * group, gint inc)
{
	if (inc != 0)
		group_set_article_read_qty (group, group->article_read_qty + inc);
}

void
group_mark_all_read (Group * g, gboolean read)
{
	Newsrc * newsrc;

	g_return_if_fail (g != NULL);

	newsrc = group_get_newsrc (g);
	newsrc_set_group_range (newsrc, g->article_low, g->article_high);
	newsrc_mark_all (newsrc, read);
	group_set_article_read_qty (g, (read ? g->article_qty : 0));
	fire_group_marked_read (g, read);
}

gboolean
group_is_read_only (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->permission == 'n';
}

gboolean
group_is_moderated (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->permission == 'm';
}

gboolean
group_is_folder (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->flags & GROUP_FOLDERS ? 1 : 0;
}

void
group_set_is_folder (Group * group, gboolean folder)
{
	if (!folder != !group_is_folder(group))
	{
		if (folder)
			group->flags |= GROUP_FOLDERS;
		else
			group->flags &= ~GROUP_FOLDERS;
		group_set_dirty (group);
	}
}

/***
****
****  DELETE GROUP
****
***/

static gboolean
real_group_empty_ghrfunc (gpointer key, gpointer value, gpointer data)
{
	pan_object_unref (PAN_OBJECT(value));
	return TRUE;
}

void
group_empty (Group * group, gboolean clear_counts)
{
	g_return_if_fail (group != NULL);

	/* unload group (if it's loaded) */
	fire_group_cleared (group);

	/* clear out the article headers */
	file_headers_destroy (group);

	/* clear out the articles... */
	if (group->_articles != NULL)
		g_hash_table_foreach_remove (group->_articles, real_group_empty_ghrfunc, NULL);
	group_set_article_qty (group, 0);
	group_set_article_read_qty (group, 0);

	/* if user wants to clear the range, do that too */
	if (clear_counts)
		group_set_article_range (group, 0, 0);

	/* flush the flags */
	if (group->flags & GROUP_SUBSCRIBED)
		group_set_flags (group, GROUP_SUBSCRIBED);
	else
		group_set_flags (group, 0);
}

/**
***  ARTICLE LIST
**/

static void
group_add_articles_ghfunc (gpointer key, gpointer val, gpointer data)
{
	guint i;
	Group * group = GROUP(key);
	GPtrArray * articles = (GPtrArray*)val;

	/* this is so that unreffing from the old group doesn't kill
	   the article -- this ref, plus group_remove_article's unref,
	   has a net change of zero */
	for (i=0; i!=articles->len; ++i)
		pan_object_ref (PAN_OBJECT(g_ptr_array_index(articles,i)));

	/* remove the articles from the other group */
	group_remove_articles (group, articles);

	/* clean up */
	g_ptr_array_free (articles, TRUE);
}

void
group_add_article (Group * group, Article * article)
{
	GPtrArray * a;
	debug_enter ("group_add_article");

	a = g_ptr_array_new ();
	g_ptr_array_add (a, article);
	group_add_articles (group, a, NULL, NULL, NULL);
	g_ptr_array_free (a, TRUE);

	debug_exit ("group_add_article");
}

 
void
group_add_articles_remove_unused (Group            * group,
                                  GPtrArray        * articles,
                                  StatusItem       * status)
{
	GPtrArray * ignored;
	GPtrArray * used;

	/* entry assertions */
	g_return_if_fail (group != NULL);
	g_return_if_fail (articles != NULL);
	g_return_if_fail (articles->len != 0);

	/* add the articles */
	ignored  = g_ptr_array_new ();
	used = g_ptr_array_new ();
	group_add_articles (group, articles, status, used, ignored);

	/* destroy unused articles & remove from the incoming array */
	pan_g_ptr_array_assign (articles, used->pdata, used->len);
	pan_g_ptr_array_foreach (ignored, (GFunc)pan_object_unref, NULL);

	/* cleanup */
	g_ptr_array_free (ignored, TRUE);
	g_ptr_array_free (used, TRUE);
}

void
group_add_articles (Group             * group,
                    const GPtrArray   * articles,
                    StatusItem        * status,
                    GPtrArray         * used,
                    GPtrArray         * ignored)
{
	gint old_qty;
	gint new_read_qty;
	guint i;
	gulong low;
	gulong high;
	gboolean tmp_ref;
	GPtrArray * tmp_used;
	GHashTable * ours;
	Newsrc * newsrc;
	Newsrc * purged;
	debug_enter ("group_add_articles");

	/* sanity checks */
	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->_articles_refcount>=0);
	g_return_if_fail (articles!=NULL);
	if (!articles->len) return;
	g_return_if_fail (articles->len>0);

	/* warn the caller if they're doing something inefficient by adding
	   articles to a group whose refcount is zero.  The old articles in
	   this group have to be loaded too; otherwise they'll be overwritten
	   when we save this group to disk without them. */
	if ((tmp_ref = !group->_articles_refcount))
		group_ref_articles (group, status);

	/**
	***  Remove the articles from their old groups.
	***  Batching the removal makes the code uglier but
	***  it runs faster.
	**/

	if (1)
	{
		GHashTable * remove_from_old = NULL;

		for (i=0; i!=articles->len; ++i)
		{
			Article * article = ARTICLE(g_ptr_array_index(articles,i));
			Group * oldgroup = article->group;
			gboolean array_existed;
			GPtrArray * to_remove;

			if (oldgroup==NULL || oldgroup==group)
				continue;

			/* ensure the hash table exists */
			if (remove_from_old == NULL)
				remove_from_old = g_hash_table_new (g_direct_hash, g_direct_equal);

			/* add this array to the array of articles to remove from its group */
			to_remove = g_hash_table_lookup (remove_from_old, oldgroup);
			array_existed = to_remove != NULL;
			if (!array_existed)
				to_remove = g_ptr_array_new ();
			g_ptr_array_add (to_remove, article);

			/* we hash the 'remove these' array by group */
			if (!array_existed)
				g_hash_table_insert (remove_from_old, oldgroup, to_remove);
		}

		if (remove_from_old != NULL)
		{
			g_hash_table_foreach (remove_from_old, group_add_articles_ghfunc, NULL);
			g_hash_table_destroy (remove_from_old);
		}
	}

	/**
	***  Get a handle on our current stats...
	**/

	low = group->article_low;
	high = group->article_high;
	ours = group_get_articles (group);
	old_qty = (int) g_hash_table_size (ours);

	/**
	***  Now try to add the articles that the client passed in
	**/

	new_read_qty = 0;
	tmp_used = g_ptr_array_new ();
	newsrc = group_get_newsrc (group);
	purged = group_get_purged (group);
	pan_g_ptr_array_reserve (tmp_used, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		const gchar * message_id = article_get_message_id (a);
		gboolean is_purged = newsrc_is_article_read(group->_purged, a->number);

		if (!is_purged && g_hash_table_lookup(ours,message_id) == NULL)
		{
			a->group = group;
			g_hash_table_insert (ours, (gpointer)message_id, a);
			g_ptr_array_add (tmp_used, a);

			if (a->number<low)
				low  = a->number;
			if (a->number>high)
				high = a->number;
			if (newsrc_is_article_read (newsrc, a->number))
				++new_read_qty;
		}
		else if (ignored != NULL)
		{
			g_ptr_array_add (ignored, a);
		}
	}

	/**
	***  Now update our stats if articles were added
	**/

	if (tmp_used->len != 0)
	{
		/* maybe update the article range */
		if (high!=group->article_high || low!=group->article_low)
			group_set_article_range (group, low, high);

		/* update the group article stats */
		group->articles_threading_dirty = TRUE;
		group_set_article_qty (group, g_hash_table_size(ours));
		if (old_qty != 0)
			new_read_qty += group->article_read_qty;
		group_set_article_read_qty (group, new_read_qty);
		group_set_articles_dirty (group);
		fire_group_changed (group);

		pan_callback_call (group_get_articles_added_callback(), group, tmp_used);
	}

	if (used != NULL)
		pan_g_ptr_array_assign (used, tmp_used->pdata, tmp_used->len);

	/* cleanup */
	g_ptr_array_free (tmp_used, TRUE);
	if (tmp_ref)
		group_unref_articles (group, status);
	debug_exit ("group_add_articles");
}

void
group_set_articles_dirty (Group * group)
{
	g_return_if_fail (group!=NULL);

	group->articles_dirty = TRUE;
}

void
group_thread_if_needed (Group * group, StatusItem * status)
{
	debug_enter ("group_thread_if_needed");

	g_return_if_fail (group != NULL);

	if (group->articles_threading_dirty && group->_articles!=NULL && g_hash_table_size(group->_articles)!=0)
	{
		GPtrArray * a;

		a = group_get_article_array (group);
		thread_articles (a, status);
		check_multipart_articles (a);
		group->articles_threading_dirty = FALSE;

		g_ptr_array_free (a, TRUE);
	}

	debug_exit ("group_thread_if_needed");
}

 
gpointer
group_get_article_by_message_id (Group * group, const gchar * message_id)
{
	g_return_val_if_fail (group!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);
	return g_hash_table_lookup (group_get_articles(group), message_id);
}

void
group_remove_article (Group * group, Article * article)
{
	GPtrArray * a = g_ptr_array_new ();
	g_ptr_array_add (a, article);
	group_remove_articles (group, a);
	g_ptr_array_free (a, TRUE);
}

void
group_remove_articles (Group * group, const GPtrArray * articles)
{
	guint i;
	int removed_read;
	int removed_total;
	GPtrArray * tmp_remove;
	GHashTable * ours;
	debug_enter ("group_remove_articles");

	/* sanity clause */
	g_return_if_fail (group != NULL);
	g_return_if_fail (articles != NULL);
	if (!articles->len) return;
	for (i=0; i!=articles->len; ++i)
		g_return_if_fail (ARTICLE(g_ptr_array_index(articles,i))->group==group);

	/* let everyone know that articles are going to be removed */
	tmp_remove = pan_g_ptr_array_dup ((GPtrArray*)articles);
	sort_articles ((Article**)tmp_remove->pdata, tmp_remove->len, ARTICLE_SORT_MSG_ID, TRUE);
	pan_callback_call (group_get_articles_removed_callback(), group, tmp_remove);

	/* is this the last gasp for these articles? */
	for (i=0; i!=tmp_remove->len; ++i) {
		Article * article = ARTICLE(g_ptr_array_index(tmp_remove,i));
		if (PAN_OBJECT(article)->ref_count == 1) /* about to die */
			article_isolate (article);
	}

	/**
	***  Try to remove the articles
	**/

	ours = group_get_articles (group);
	removed_total = 0;
	removed_read = 0;
	for (i=0; i!=tmp_remove->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index(tmp_remove,i));
		const gchar * message_id = article_get_message_id (a);
		Article * our_a = ARTICLE(g_hash_table_lookup (ours, message_id));

		if (our_a != NULL)
		{
			g_hash_table_remove (ours, message_id);

			++removed_total;
			if (article_is_read (our_a))
				++removed_read;

			our_a->group = NULL;
			pan_object_unref (PAN_OBJECT(our_a));
		}
	}

	/**
	***  Update the stats
	**/

	if (removed_total != 0)
	{
		/* update the articles stats */
		group->articles_threading_dirty = TRUE;
		group->article_qty -= removed_total;
		group->article_read_qty -= removed_read;
		group_set_articles_dirty (group);
		fire_group_changed (group);
	}

	/* cleanup */
	g_ptr_array_free (tmp_remove, TRUE);
	debug_exit ("group_remove_articles");
}

void
group_expire_articles_not_in_range (Group * g, gulong low, gulong high)
{
	guint i;
	GPtrArray * tmp;
	GPtrArray * removeme;
	debug_enter ("group_expire_articles_not_in_range");

	g_return_if_fail (g != NULL);

	/* find old articles to remove */
	tmp = group_get_article_array (g);
	removeme = g_ptr_array_new ();
	pan_g_ptr_array_reserve (removeme, tmp->len);
	for (i=0; i!=tmp->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(tmp,i));
		if ((a->number<low) || (a->number>high))
			g_ptr_array_add (removeme, a);
	}

	/* remove the old articles */
	if (removeme->len != 0)
	{
		log_add_va (LOG_INFO, _("Expired %u articles from `%s'"),
			removeme->len,
			group_get_readable_name(g));

		group_remove_articles (g, removeme);
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
	g_ptr_array_free (removeme, TRUE);
	debug_exit ("group_expire_articles_not_in_range");
}


/***
****
***/

void
group_ref_articles (Group * group, StatusItem * status)
{
	debug_enter ("group_ref_articles");

	/* sanity clause */
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles_refcount >= 0);

	/* if we're up from zero, load the headers */
	if (++group->_articles_refcount == 1)
		file_headers_load (group, status);

	/* g_message ("group %s refcount inc to %d", group->name, group->_articles_refcount); */
	group_set_article_qty (group, g_hash_table_size(group->_articles));
	debug_exit ("group_ref_articles");
}

static void
group_unref_articles_ghfunc (gpointer key, gpointer val, gpointer data)
{
	pan_object_unref (PAN_OBJECT(val));
}
void
group_unref_articles (Group * group, StatusItem * status)
{
	debug_enter ("group_unref_articles");

	/* sanity clause */
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles_refcount > 0);

	/* if down to zero, save & flush the articles */
	if (!--group->_articles_refcount)
	{
		file_headers_save (group, status);

		/* free the articles */
		g_hash_table_foreach (group->_articles, group_unref_articles_ghfunc, NULL);
		g_hash_table_destroy (group->_articles);
		group->_articles = NULL;

		/* free the memory used by those articles */
		replace_gstr (&group->_one_big_chunk, NULL);
		if (group->chunk != NULL) {
			g_string_chunk_free (group->chunk);
			group->chunk = NULL;
		}
	}

	/* g_message ("group %s refcount dec to %d", group->name, group->_articles_refcount); */
	debug_exit ("group_unref_articles");
}

/**
***
**/


const gchar*
group_get_readable_name (const Group * group)
{
	const gchar * retval;

	if (group == NULL)
	{
		pan_warn_if_reached ();
		retval = _("(No Group)");
	}
	else
	{
		retval = is_nonempty_string (group->readable_name)
			? group->readable_name
			: group->name;
	}

	return retval;
}

/**
***
**/

const gchar*
group_chunk_string (Group         * group,
                    const gchar   * chunkme,
                    gboolean        share)
{
	gchar * retval = NULL;

	g_return_val_if_fail (group!=NULL, NULL);

	if (chunkme != NULL)
	{
		if (group->chunk == NULL)
			group->chunk = g_string_chunk_new (81920);

		if (share)
			retval = g_string_chunk_insert_const (group->chunk, chunkme);
		else
			retval = g_string_chunk_insert (group->chunk, chunkme);
	}

	return retval;
}


/***
****
****
****
***/

Newsrc*
group_get_newsrc (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_newsrc == NULL)
		group->_newsrc = newsrc_new (NULL, group->article_low, group->article_high);

	return group->_newsrc;
}

gboolean
group_has_newsrc (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->_newsrc != NULL;
}

/**
 * When crossposts are deleted, we don't want to load in all the groups
 * containing a crosspost just to delete the article and save the whole
 * group back to disk again.  Likewise the crosspost may not even have been
 * downloaded yet, which would thwart the deletion.
 *
 * So the solution Pan uses is to have a newsrc-like range of article
 * numbers of these deleted articles.  When a client tries to add articles
 * to the group, group_add_articles() checks them against the deleted list
 * before allowing them to be added.
 * 
 * Also important is that we update the deleted list's bounds whenever
 * the group's own low/high numbers are updated, in group_set_article_range(),
 * so that the deleted article list doesn't get too long when saved to disk.
 */
Newsrc*
group_get_purged (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_purged == NULL)
		group->_purged = newsrc_new (NULL, group->article_low, group->article_high);

	return group->_purged;
}

static gint
compare_ppA_to_ppA_by_number (const void * va, const void * vb)
{
	const Article * a = *(const Article **)va;
	const Article * b = *(const Article **)vb;
	if (a->number < b->number) return -1;
	if (a->number > b->number) return 1;
	return 0;
}

void
group_remove_crossposts (Group     * g,
		         gulong    * numbers,
                         gint        number_qty)
{

	g_return_if_fail (g!=NULL);
	g_return_if_fail (numbers!=NULL);
	g_return_if_fail (number_qty>0);

	/* remove articles if they're already present */
	if (g->_articles_refcount>0)
	{
		gint i;
		GPtrArray * articles;
		GPtrArray * remove;

		/* get a list of articles sorted by number for fast lookup */
		remove = g_ptr_array_new ();
		articles = group_get_article_array (g);
		qsort (articles->pdata, articles->len, sizeof(Article*), compare_ppA_to_ppA_by_number);

		/* for each number to remove, if we've got it, add it to the "remove" ptr array */
		for (i=0; i<number_qty; ++i)
		{
			Article tmp;
			gint index;
			gboolean exact_match = FALSE;

			tmp.number = numbers[i];
			index = lower_bound (&tmp,
			                     articles->pdata,
			                     articles->len,
			                     sizeof(Article*),
			                     compare_ppA_to_ppA_by_number,
			                     &exact_match);

			if (exact_match) {
				Article * a = ARTICLE(g_ptr_array_index(articles,index));
				odebug2 ("group [%s] removing crosspost [%s]", g->name, a->subject);
				g_ptr_array_add (remove, a);
			}
		}

		if (remove->len != 0)
			group_remove_articles (g, remove);

		g_ptr_array_free (remove, TRUE);
		g_ptr_array_free (articles, TRUE);
	}

	/* flag articles for deletion in case they get downloaded later */
	if (1)
	{
		guint i;
		Newsrc * n;
		
		n = group_get_purged (g);
		for (i=0; i<number_qty; ++i) {
			odebug2 ("group [%s] is removing xref [%lu]", g->name, numbers[i]);
			newsrc_mark_article (n, numbers[i], TRUE);
		}
	}
}


GHashTable*
group_get_articles (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_articles == NULL)
		group->_articles = g_hash_table_new (g_str_hash, g_str_equal);

	return group->_articles;
}

GPtrArray*
group_get_article_array (Group * group)
{
	GHashTable * hash;
	GPtrArray * retval;

	/* sanity clause */
	retval = g_ptr_array_new ();
	g_return_val_if_fail (group!=NULL, retval);

	/* populate retval */
	hash = group_get_articles (group);
	pan_hash_to_ptr_array (hash, retval);

	return retval;
}

/***
****
****  EVENTS
****
***/
 
/**
***  Groups Removed
**/
 
PanCallback*
group_get_group_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_group_changed (Group * g)
{
	g_return_if_fail (g!=NULL);

	pan_callback_call (group_get_group_changed_callback(), g, NULL);
}

PanCallback*
group_get_articles_added_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

PanCallback*
group_get_articles_removed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

PanCallback*
group_get_group_marked_read_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_group_marked_read (Group * group, gboolean read)
{
	g_return_if_fail (group!=NULL);

	pan_callback_call (group_get_group_marked_read_callback(),
	                   group,
	                   GINT_TO_POINTER(read));
}

PanCallback*
group_get_group_emptied_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_group_cleared (Group * group)
{
	g_return_if_fail (group!=NULL);

	pan_callback_call (group_get_group_emptied_callback(), group, NULL);
}
