/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <glib.h>

#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/filters/filter-saved.h>

const gchar * FILTER_SAVED_CLASS_ID = "PanObject::Filter::FilterSaved";

/************
*************  PROTECTED
************/

static Filter*
filter_saved_dup (const Filter * f_old)
{
	Filter * f_new = filter_saved_new ();
	FilterSaved * fb_old = FILTER_SAVED(f_old);
	FilterSaved * fb_new = FILTER_SAVED(f_new);

	/* duplicate superclass */
	filter_class_dup (f_old, f_new);

	/* duplicate saved */
	fb_new->state = fb_old->state;

	return f_new;
}

static gchar*
filter_saved_to_string (const Filter * filter)
{
	gchar * retval = NULL;
	const gboolean neg = filter->negate;
	FilterSavedState state = FILTER_SAVED(filter)->state;

	if (state==FILTER_SAVED_SAVED) {
		if (neg)
			retval = g_strdup (_("Article has not been saved or decoded"));
		else
			retval = g_strdup (_("Article has been saved or decoded"));
	}
	else if (state==FILTER_SAVED_QUEUED) {
		if (neg)
			retval = g_strdup (_("Article is not queued for saving or decoding"));
		else
			retval = g_strdup (_("Article is queued for saving or decoding"));
	}
	else if (state==FILTER_SAVED_NOT_SAVED) {
		if (neg)
			retval = g_strdup (_("Article is either saved or queued for saving"));
		else
			retval = g_strdup (_("Article is neither saved nor queued for saving"));
	}
	else retval = g_strdup (_("Error"));

	return retval;
}

static void
filter_saved_test_articles (Filter           * filter,
                            const Article   ** articles,
                            gint               article_qty,
			    gboolean         * passfail)
{
	gint i;
	FilterSaved * fb = FILTER_SAVED(filter);
	FilterSavedState state = fb->state;

	for (i=0; i<article_qty; ++i)
	{
		const Article * a = articles[i];
		gboolean is_saved = article_flag_on (a, STATE_DECODED);
		gboolean is_queued = article_flag_on (a, STATE_DECODE_QUEUED);
		gboolean b;

		switch (state)
		{
			case FILTER_SAVED_SAVED:
				b = is_saved;
				break;
			case FILTER_SAVED_QUEUED:
				b = is_queued;
				break;
			case FILTER_SAVED_NOT_SAVED:
				b = !is_saved && !is_queued;
				break;
			default:
				pan_warn_if_reached ();
				b = FALSE;
				break;
		}

		passfail[i] = b;
	}
}

static void
filter_saved_destructor (PanObject * o)
{
	filter_destructor (o);
}

static void
filter_saved_constructor (FilterSaved * f)
{
	filter_constructor ((Filter*)f,
	                    filter_saved_destructor,
	                    filter_saved_test_articles,
	                    filter_saved_to_string,
	                    filter_saved_dup,
	                    FILTER_SAVED_CLASS_ID);
	f->state = FILTER_SAVED_SAVED;
}

/************
*************  PUBLIC
************/

Filter*
filter_saved_new (void)
{
	FilterSaved * f = g_new0 (FilterSaved, 1);
	filter_saved_constructor (f);
	return FILTER(f);
}

void
filter_saved_set_state (FilterSaved        * filter_saved,
                        FilterSavedState     state)
{
	g_return_if_fail (filter_saved!=NULL);
	g_return_if_fail (state==FILTER_SAVED_SAVED ||
	                  state==FILTER_SAVED_QUEUED ||
	                  state==FILTER_SAVED_NOT_SAVED);

	filter_saved->state = state;
}
