/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <ctype.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>

#include <gnome.h>

#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-file.h>
#include <pan/base/util-wrap.h>

#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/message-check-ui.h>
#include <pan/message-send.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/print.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h> /* for colors of fg, bg, and quoted text */
#include <pan/util.h>

typedef struct
{
	GtkWidget * window; /* GnomeApp */
	GtkWidget * appbar; /* GnomeAppBar */

	GtkWidget * read_info_pane;

	GtkWidget * from;
	GtkWidget * custom_headers_text;
	GtkWidget * user_agent_tb;
	GtkWidget * organization;
	GtkWidget * newsgroups;
	GtkWidget * users;
	GtkWidget * subject;
        GtkWidget * followup_to;
        GtkWidget * reply_to;
	GtkWidget * body;

	/* Attachment */
	GtkWidget * filename;
	GtkWidget * lines;
	GtkWidget * lines_per_part;
	GtkWidget * parts;

	Article * article;
	MessageWindowType type;
}
MessageWindow;

extern GtkTooltips *ttips;

static void rot13_cb                       (GtkWidget*, MessageWindow*);
static void wrap_cb                        (GtkWidget*, MessageWindow*);
static void save_cb                        (GtkWidget*, MessageWindow*);
static void send_now_cb                    (GtkWidget*, MessageWindow*);
static void send_later_cb                  (GtkWidget*, MessageWindow*);
static void message_window_save_cb         (GtkWidget*, MessageWindow*);
static void message_window_cut_cb          (GtkWidget*, MessageWindow*);
static void message_window_copy_cb         (GtkWidget*, MessageWindow*);
static void message_window_paste_cb        (GtkWidget*, MessageWindow*);
static void message_window_close           (GtkWidget*, MessageWindow*);
static void message_window_print           (GtkWidget*, MessageWindow*);
static void message_window_appbar_toggle   (GtkWidget*, MessageWindow*);
static void message_reply_external         (GtkWidget*, MessageWindow*);

/* Not ready -- Matt
static void attachment_file_selector       (GtkWidget*, MessageWindow*);
static GtkWidget * attachment_page         (MessageWindow*);
static void attachments_lines_per_part_changed (GtkWidget*, MessageWindow*);
*/
static GtkWidget * extra_headers_page      (MessageWindow*);
static GtkWidget * create_body_pane_nolock (MessageWindow*);
/*static void message_edit_external (GtkWidget *widget, Message *msg);*/

/***
****  SHUTDOWN
***/

static gint
window_delete_event_cb (GtkWidget * w, GdkEvent * e, gpointer data)
{
	MessageWindow * mw = (MessageWindow*) data;
	gui_save_window_size (mw->window, "message_window");
	return FALSE;
}
static void
window_destroy_cb (GtkWidget * window, MessageWindow * mw)
{
	if (mw->article != NULL)
		group_unref_articles (mw->article->group, NULL);
	g_free (mw);
}

/***
****  EXPERIMENTAL WRAP CODE
***/

static gboolean experimental_toggle_active = TRUE;

static void
wrap_tb_toggled_cb (GtkToggleButton * togglebutton, gpointer user_data)
{
	experimental_toggle_active = gtk_toggle_button_get_active (togglebutton);
}


static void
text_inserted_cb (GtkEditable * editable,
                  gchar * new_text,
                  gint new_text_length,
                  gint * position,
                  gpointer user_data)
{
	static gboolean dampen_feedback = FALSE;

	if (!dampen_feedback && experimental_toggle_active)
	{
		gchar * text = gtk_editable_get_chars (editable, 0, *position);
		gint text_len = text ? strlen(text) : 0;
		gchar * linefeed = strrchr (text, '\n');
		gboolean need_break;

		/* do we need to break the line up? */
		need_break = linefeed==NULL
			? text_len >= wrap_column
			: &text[text_len-1] - linefeed >= wrap_column;

		/* break the line up... */
		if (need_break)
		{
			gchar * pch;
			gchar * del_end_pos = NULL;

			/* try to find a place to break... */
			for (pch=text+text_len; pch!=text; --pch) {
				if (isspace((int)pch[-1])) {
					del_end_pos = pch;
					break;
				}
			}

			/* break... */
			if (del_end_pos != NULL)
			{
				gint del_start_index;
				gint del_end_index;
				gint tmp;

				/* find the beginning of the whitespace */
				for (pch=del_end_pos; pch!=text; --pch)
					if (!isspace((int)pch[-1]))
						break;
				del_end_index = del_end_pos - text;
				del_start_index = pch - text;

				/* replace the space(s) with a linefeed */
				dampen_feedback = TRUE;
				tmp = del_start_index;
				gtk_editable_delete_text (editable, tmp, del_end_index);
				gtk_editable_insert_text (editable, "\n", 1, &tmp);
				dampen_feedback = FALSE;

				/* update the position of the original text */
				*position += 1; /* for '\n' */
				*position -= (del_end_index - del_start_index);
			}
		}

		g_free (text);
	}
}

static void
experimental_wrap_handler (GtkText * text, GtkToggleButton * tb)
{
	pan_lock ();
	gtk_toggle_button_set_active (tb, experimental_toggle_active);
	gtk_signal_connect (GTK_OBJECT(text), "insert-text", text_inserted_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(tb), "toggled", wrap_tb_toggled_cb, NULL);
	pan_unlock ();
}

/***
****
***/

static gchar*
article_get_attribution_string (const Article * a)
{
	gchar * pch = NULL;
	gchar * retval = NULL;
	const gchar * cpch = NULL;
	GString * gstr = NULL;

	pan_warn_if_fail (is_nonempty_string(attribution_line));

	/* get the unsubstituted string */
	gstr = g_string_new (attribution_line);

	/* substitutions: message-id */
	cpch = article_get_message_id (a);
	pan_g_string_replace (gstr, "%i", cpch);

	/* substitutions: date */
	pch = a==NULL ? g_strdup("") : rfc822_date_generate(a->date);
	pan_g_string_replace (gstr, "%d", pch);
	g_free (pch);

	/* substitutions: author */
	pch = a==NULL ? g_strdup("") : article_get_author_str (a);
	pan_g_string_replace (gstr, "%a", pch);
	g_free (pch);

	/* substitutions: author name */
	pch = a==NULL ? g_strdup("") : article_get_short_author_str (a);
	pan_g_string_replace (gstr, "%n", pch);
	g_free (pch);

	/* cleanup & return the result */
	retval = gstr->str;
	g_string_free (gstr, FALSE);
	return retval;
}

/***
****
***/

static GnomeUIInfo message_edit_file_menu[] =
{
	GNOMEUIINFO_ITEM_STOCK (N_("Save Changes"), N_("Save Changes"), 
				save_cb, GNOME_STOCK_MENU_SAVE),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_file_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Send Now"),
		N_("Send Now"),
		send_now_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_SND,
		GDK_Return, GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Send Later"),
		N_("Send Later"),
		send_later_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TIMER,
		'D', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_file_menu[] =
{
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (message_window_save_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PRINT_ITEM (message_window_print, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM (message_window_close, NULL),	
	GNOMEUIINFO_END
};

static GnomeUIInfo message_edit_menu[] =
{
	GNOMEUIINFO_MENU_CUT_ITEM (message_window_cut_cb, NULL),
	GNOMEUIINFO_MENU_COPY_ITEM (message_window_copy_cb, NULL),
	GNOMEUIINFO_MENU_PASTE_ITEM (message_window_paste_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_NONE (
		N_("_Rot13 Selected Text"),
		N_("Rot13 Selected Text"),
		&rot13_cb),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_view_menu[] =
{
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_ ("Show Status Bar"), 0,
		message_window_appbar_toggle, 0, 0, 0, 0, 0, 0, 0
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_message_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Post to newsgroup"),
		N_("Post a new message to the current group."),
		message_post_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Followup to newsgroup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply by _E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply with External Mailer"),
		N_("Create a mail reply to the sender with your favorite MUA."),
		message_reply_external,
		NULL
       	},
	GNOMEUIINFO_END
};

static GnomeUIInfo message_read_main_menu[] =
{
	GNOMEUIINFO_MENU_FILE_TREE (message_read_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &message_view_menu),
	GNOMEUIINFO_SUBTREE (N_("_Message"), &message_read_message_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_main_menu[] =
{
	GNOMEUIINFO_MENU_FILE_TREE (message_post_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &message_view_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_edit_main_menu[] =
{
	GNOMEUIINFO_MENU_FILE_TREE (message_edit_file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (message_edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &message_view_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};


static GnomeUIInfo message_read_toolbar[] =
{
	GNOMEUIINFO_ITEM_STOCK(
		N_("Post Followup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window,
		GNOME_STOCK_PIXMAP_MAIL_NEW),
	GNOMEUIINFO_ITEM_STOCK(
		N_("E-mail Reply"),
		N_("Create a mail reply to the sender."),
		message_reply_window,
		GNOME_STOCK_PIXMAP_MAIL_RPL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Print"), N_("Print this message"),
				message_window_print, GNOME_STOCK_PIXMAP_PRINT),

	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Close"), N_("Close this Window"),
				message_window_close, GNOME_STOCK_PIXMAP_CLOSE),
	GNOMEUIINFO_END
};

static GnomeUIInfo message_post_toolbar[] =
{
	GNOMEUIINFO_ITEM_STOCK
	(
		N_("Send Now"),
		N_("Send this Message Now"), 
		send_now_cb,
		GNOME_STOCK_PIXMAP_MAIL_SND
	),
	GNOMEUIINFO_ITEM_STOCK
	(
		N_("Send Later"),
		N_("Send this Message Later"), 
		send_later_cb,
		GNOME_STOCK_PIXMAP_TIMER
	),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK
	(
		N_("Rewrap"),
		N_("Rewrap all the text in the Message Body."),
		wrap_cb,
		GNOME_STOCK_PIXMAP_ALIGN_JUSTIFY
	),
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Wrap Text"),
		N_("Turn line wrap on/off in the Message Body"),
		NULL,
		NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_ALIGN_JUSTIFY,
		0, (GdkModifierType)0, NULL
	},
	GNOMEUIINFO_END
};


/**
***  
**/

static void
append_signature_file (GString * body)
{
	gchar * sig = NULL;
	gchar * pch;

        /* load the signature */	
	pch = gnome_config_get_string ("/Pan/User/Signature_File");
	if (is_nonempty_string (pch)) {
		GArray * array;
		replace_gstr (&pch, pan_substitute (pch,"~",g_get_home_dir()));
		array = read_file (pch);
		if (array != NULL) {
			sig = array->data;
			g_array_free (array, FALSE);
		}
	}
	g_free (pch);

	/* append the signature */
	if (sig!=NULL) {
		/* we insert "-- " if it's not already there */
		gboolean delimiter_exists = (
			    (!strncmp (sig, "-- \r\n", 5))
			 || (!strncmp (sig, "-- \n", 4))
			 || (pan_strstr(sig, "\n-- \r")!=NULL)
			 || (pan_strstr(sig, "\n-- \n")!=NULL));
		g_string_append (body, "\n\n");
		if (!delimiter_exists)
			g_string_append (body, "\n-- \n");
		g_string_append (body, sig);
		g_free (sig);
	}
}

/**
 *  1. If the user selected an area, we use that.
 *  2. If the user didn't select an area, use the whole body.
 *  3. Quote fresh text with "> "
 *  4. Requote older quoted text with ">"
 *  5. Strip the signature if user didn't select it.
 *  6. Prepend attribution line.
 *  7. Wrap everything.
 *  8. Add our own signature to the end.
 */
static gchar*
create_reply_body (Article * article)
{
	gboolean marked;
	GString * out = g_string_new (NULL);
	gchar * pch;

	/* body */
	pch = g_strdup (article_get_header (article, PAN_REPLY_PORTION));
	marked = pch != NULL;
	if (pch == NULL)
		pch = article_get_body (article);
	if (is_nonempty_string(pch)) {
		gchar ** sd = g_strsplit (pch, "\n", -1);
		gint i;
		for (i=0; sd!=NULL && sd[i]!=NULL; ++i) {
			const gchar * l = sd[i];
			if (!marked && (!strcmp(l,"-- ") || !strcmp(l,"-- \r")))
				break;
			g_string_sprintfa (out,
				"%s%s\n", (*l=='>' ? ">" : "> "), l);
		}
		g_strfreev (sd);
	}
	g_free (pch);

	/* attribution */
	pch = article_get_attribution_string (article);
	if (is_nonempty_string(pch)) {
		g_string_prepend (out, "\n\n");
		g_string_prepend (out, pch);
	}
	g_free (pch);

	if (wrap_body_on_reply)
	{
		/* wrap what we've got so far */
		pch = fill_body (out->str, wrap_column);
		g_string_assign (out, pch);
		g_free (pch);
	}

	/* append signature (AFTER the wrap) */
	append_signature_file (out);

	/* return the result */
	pch = out->str;
	g_string_free (out, FALSE);
	return pch;
}

/***
****
***/

static gboolean
is_posting (MessageWindowType type)
{
	return type==NNTP_POST
		|| type==NNTP_REPLY
		|| type==EMAIL_AND_POST_REPLY;
}

static gchar*
make_reply_string (const gchar* string)
{
	gchar * retval = NULL;

	if (string != NULL)
	{
		if (g_strncasecmp ("Re: ", string, 4))
			retval = g_strconcat ("Re: ", string, NULL);
		else
			retval = g_strdup (string);
	}

	return retval;
}

static void
message_window_appbar_toggle (GtkWidget *widget, MessageWindow * mw)
{
	pan_lock ();

	if (mw->appbar == NULL)
	{
		mw->appbar = gnome_appbar_new (0, 1, GNOME_PREFERENCES_USER);
		gnome_app_set_statusbar (GNOME_APP(mw->window),
		                         GTK_WIDGET(mw->appbar));
		gnome_app_install_menu_hints (GNOME_APP (mw->window),
			(mw->type==NNTP_READ
			 ? message_read_main_menu
			 : message_post_main_menu));
	}
	else if (GTK_IS_WIDGET (mw->appbar->parent))
	{
		gtk_widget_ref (mw->appbar);
		gtk_container_remove (GTK_CONTAINER(mw->appbar->parent),
		                      mw->appbar);
	}
	else
	{
		gtk_box_pack_start (GTK_BOX(GNOME_APP(mw->window)->vbox),
		                    mw->appbar, FALSE, FALSE, 0);
		gtk_widget_unref (mw->appbar);
	}
	pan_unlock ();
}

static void
rot13_cb (GtkWidget * widget, MessageWindow * mw)
{
	GtkEditable * e = GTK_EDITABLE(mw->body);
	gint a = e->selection_start_pos;
	gint b = e->selection_end_pos;
	gint start = MIN (a, b);
	gint end = MAX (a, b);

	if (end > start)
	{
		gchar * str = gtk_editable_get_chars (e, start, end);
		replace_gstr (&str, rot13(str));
		gtk_editable_delete_text (e, start, end);
		gtk_editable_insert_text (e, str, strlen(str), &start);
	}
}

static void
wrap_cb (GtkWidget *widget, MessageWindow * mw)
{
	gchar * body;
	gchar * new_body;
	gboolean b;
	debug_enter ("wrap_cb");

	/* get the current body */
	pan_lock();
       	body = gtk_editable_get_chars (GTK_EDITABLE(mw->body), 0, -1);
	pan_unlock ();

	new_body = fill_body (body, wrap_column);

	/* turn off our own wrapping while we fill the body pane */
	b = experimental_toggle_active;
	experimental_toggle_active = FALSE;
	update_body_pane (mw->body, new_body, FALSE);
	experimental_toggle_active = b;

	/* cleanup */
	g_free (body);
	g_free (new_body);
	debug_exit ("wrap_cb");
}

/**
***
**/

static void
message_window_cut_cb (GtkWidget * widget, MessageWindow * mw)
{
	g_return_if_fail (mw->body);
	pan_lock ();
	gtk_editable_cut_clipboard (GTK_EDITABLE(mw->body));
	pan_unlock ();
}


static void
message_window_copy_cb (GtkWidget * widget, MessageWindow * mw)
{
	g_return_if_fail (mw->body);
	pan_lock ();
	gtk_editable_copy_clipboard (GTK_EDITABLE(mw->body));
	pan_unlock ();
}


static void
message_window_paste_cb (GtkWidget *widget, MessageWindow * mw)
{
	pan_lock();
	gtk_editable_paste_clipboard (GTK_EDITABLE(mw->body));
	pan_unlock();
}


static void
message_window_destroy (MessageWindow *mw)
{
	pan_lock();
/*	gtk_container_foreach (GTK_CONTAINER (mw->window), GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL); */
	gtk_widget_destroy (mw->window);
	pan_unlock();
}

static void
message_window_print (GtkWidget * widget, MessageWindow * mw)
{
	print_cb (widget, mw->article);
}

static void
message_window_close (GtkWidget * widget, MessageWindow * mw)
{
	message_window_destroy (mw);
}


static GtkWidget *
create_read_info_pane (MessageWindow *mw)
{
	GtkWidget * table;

	/* create the table */
	pan_lock();
	table = gtk_table_new (7, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gui_headers_set_default (table, mw->article, FALSE);
	pan_unlock();

	return table;
}


static void
update_title_with_subject (GtkWidget * widget, MessageWindow * mw)
{
	gtk_window_set_title (GTK_WINDOW (mw->window),
	                      gtk_entry_get_text (GTK_ENTRY (widget)));
}

/* 
static GtkWidget *
attachment_page (MessageWindow *mw)
{
	GtkWidget *table;
	GtkWidget *w;
	GtkObject *a;

	table = gtk_table_new (6, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(table), GNOME_PAD);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new (_("Filename:"));
	gtk_misc_set_alignment (GTK_MISC (w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 0, 1,
                    GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	mw->filename = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), mw->filename, 1, 5, 0, 1,
                    GTK_EXPAND | GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	w = gtk_button_new_with_label (_("Select File"));
	gtk_signal_connect (GTK_OBJECT (w), "clicked",
                      GTK_SIGNAL_FUNC (attachment_file_selector), mw);
	gtk_table_attach (GTK_TABLE (table), w, 5, 6, 0, 1,
                    GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	w = gtk_label_new (_("Lines per Part:"));
	gtk_misc_set_alignment (GTK_MISC (w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 0, 1, 1, 2,
                    GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	a = gtk_adjustment_new (7000, 100, 15000, 25, 10, 10);
	mw->lines_per_part = gtk_spin_button_new (GTK_ADJUSTMENT (a), 1, 0);
	gtk_table_attach (GTK_TABLE (table), mw->lines_per_part, 1, 2, 1, 2,
                    GTK_EXPAND | GTK_FILL, GTK_FILL, GNOME_PAD, 0);
 	gtk_signal_connect (GTK_OBJECT (a), "value_changed",
                           GTK_SIGNAL_FUNC (attachments_lines_per_part_changed),
                           (gpointer) mw);

	w = gtk_label_new (_("Lines:"));
	gtk_misc_set_alignment (GTK_MISC (w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 2, 3, 1, 2,
                    GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	mw->lines = gtk_label_new (_("0"));
	gtk_misc_set_alignment (GTK_MISC (mw->lines), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), mw->lines, 3, 4, 1, 2,
                    GTK_EXPAND | GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	w = gtk_label_new (_("Parts:"));
	gtk_misc_set_alignment (GTK_MISC (w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), w, 4, 5, 1, 2,
                    GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	mw->parts = gtk_label_new (_("0"));
	gtk_misc_set_alignment (GTK_MISC (mw->parts), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), mw->parts, 5, 6, 1, 2,
                    GTK_EXPAND | GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	return table;
}
*/

static GtkWidget *
extra_headers_page (MessageWindow * mw)
{
	gchar * pch;
	GtkWidget * table;
	GtkWidget * scroll;
	GtkWidget * frame;
	GtkWidget * vbox;
	GtkWidget * w;
	GtkWidget * eventbox; 
	int row = 0;

	/**
	***  The top table: common headers
	**/

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	w = gtk_label_new ( _("Followup-To:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The newsgroup or newsgroups where replies to "
		  "your posted message should go.  This is only "
		  "needed if it differs from the \"Post To Groups\" "
		  "header. "
		  "\nTo direct all replies to your email address, "
		  "use \"Followup-To: poster\""), "");
	gtk_table_attach (GTK_TABLE(table), eventbox, 0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->followup_to = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->followup_to,
			  1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	++row;

	w = gtk_label_new ( _("Reply-To:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The e-mail account where mail replies to "
		  "your posted message should go.  This is only "
		  "needed if it differs from the \"From\" "
		  "header."), "");
	gtk_table_attach (GTK_TABLE(table), eventbox,
			  0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->reply_to = gtk_entry_new ();
	pch = gnome_config_get_string ("/Pan/User/Reply_To");
	if (pch != NULL) {
		gtk_entry_set_text (GTK_ENTRY(mw->reply_to), pch);
		g_free (pch);
	}
	
	gtk_table_attach (GTK_TABLE(table), mw->reply_to,
			  1, 2, row, row+1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	++row;

	w = gtk_label_new ( _("Organization:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	eventbox = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(eventbox), w);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), eventbox,
		_("The organization you're associated with."), "");
	gtk_table_attach (GTK_TABLE(table), eventbox,
			  0, 1, row, row+1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->organization = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(table), mw->organization,
			  1, 2, row, row+1,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);

	++row;

	/**
	***  The extra headers edit field
	**/

	w = gtk_text_new (NULL, NULL);
	gtk_text_set_word_wrap (GTK_TEXT(w), FALSE);
	gtk_text_set_line_wrap (GTK_TEXT(w), FALSE);
	gtk_text_set_editable (GTK_TEXT(w), TRUE);
	mw->custom_headers_text = w;
	scroll = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_set_border_width (GTK_CONTAINER(scroll), GNOME_PAD_SMALL);
	gtk_container_add (GTK_CONTAINER(scroll), w);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC);

	w = gtk_check_button_new_with_label (_("Don't add the \"User-Agent\" identification header"));
	mw->user_agent_tb = w;

	w = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(w), GNOME_PAD);
	gtk_box_pack_start (GTK_BOX(w), scroll, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(w), mw->user_agent_tb, FALSE, TRUE, 0);

	frame = gtk_frame_new (_("Custom Headers"));
	gtk_container_set_border_width (GTK_CONTAINER(frame), GNOME_PAD);
	gtk_container_add (GTK_CONTAINER(frame), w);

	/**
	***  Tie the top and bottom together
	**/

	vbox = gtk_vbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(vbox), GNOME_PAD);
	gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	return vbox;
}

/*--------------------------------------------------------------------
 * Newsgroups:
 * CC:
 * Subject:
 *--------------------------------------------------------------------*/
static GtkWidget*
create_post_info_pane (MessageWindow *mw)
{
	GtkWidget * w;
	GtkWidget * main_page;
	GtkWidget * headers;
	GtkWidget * notebook;

	pan_lock();
	
	/*-----
	 * Headers Page
	 * ---- */
	
	headers = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(headers), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(headers), GNOME_PAD_SMALL);

	w = gtk_label_new ( _("From:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);

	mw->from = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(headers), mw->from, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);


	/* set the subject of the new message */
	w = gtk_label_new ( _("Subject:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->subject = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(headers), mw->subject,
	                  1, 2, 1, 2,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);
	gtk_signal_connect (GTK_OBJECT (mw->subject),
	                    "changed",
	                    update_title_with_subject,
	                    mw);

	/* set the Post To Groups: */
	w = gtk_label_new ( _("Post To Groups:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w,
	                  0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->newsgroups = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE(headers), mw->newsgroups,
	                  1, 2, 2, 3,
			  GTK_FILL|GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);


	/* set the Mail To: */
	w = gtk_label_new ( _("Mail To:") );
	gtk_misc_set_alignment (GTK_MISC(w), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE(headers), w, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, GNOME_PAD, 0);
	mw->users = gtk_entry_new();
	gtk_table_attach (GTK_TABLE(headers), mw->users,
	                  1, 2, 3, 4,
			  GTK_FILL | GTK_EXPAND, GTK_FILL, GNOME_PAD, 0);


	/* create the main hbox */
	main_page = gtk_vbox_new (FALSE, GNOME_PAD);
	gtk_container_set_border_width (GTK_CONTAINER(main_page), GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(main_page), headers, FALSE, TRUE, 0);
	w = create_body_pane_nolock (mw);
	gtk_box_pack_start (GTK_BOX(main_page), w, TRUE, TRUE, 0);

	
	/*-----
	 * Fill the notebook
	 * ---- */
	
	notebook = gtk_notebook_new ();

        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), main_page,
			gtk_label_new (_("Message")));
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), extra_headers_page(mw),
			gtk_label_new (_("More Headers")));

/* This isn't ready for the stable release -- Matt
 * gtk_notebook_append_page (GTK_NOTEBOOK (notebook), attachment_page(mw),
			gtk_label_new (_("Attachments")));
*/

	pan_unlock();

	return notebook;
}

static void
populate_post_info_pane (MessageWindow *mw)
{
	gchar * p;
	const gchar * cp;
	Article * a;

	g_return_if_fail (mw != NULL);

	a = mw->article;

	/**
	***  Populate the extra headers
	**/

	{
		gchar * pch;
		const gchar * cpch;
		GString * gstr = g_string_new (NULL);

		/* start with "X-Comment-To:" header for Fido users. */
		pch = NULL;
		if (a!=NULL)
			pch = article_get_author_str (a);
		if (!is_nonempty_string(pch))
			pch = g_strdup ("ALL");
		g_string_sprintfa (gstr, "X-Comment-To: %s\n", pch);
		g_free (pch);

		/* add user's defaults */
		cpch = extra_posting_headers;
		g_string_append (gstr, cpch);

		/* walk through all of the article's headers.
		   If it's an extra one & wasn't covered in user's defaults add it */
		if (a!=NULL && group_is_folder(a->group)) {
			guint i;
			GPtrArray * h = article_get_all_headers (a);
			for (i=0; i!=h->len; i+=2) {
				char* key = (char*)g_ptr_array_index(h,i);
				char* val = (char*)g_ptr_array_index(h,i+1);
				gboolean extra = article_header_is_extra (key);
				if (extra && pan_strstr(pch,key)==NULL)
					g_string_sprintfa (gstr, "%s: %s\n", key, val);
			}
			g_ptr_array_free (h, TRUE);
		}

		/* add the headers */
		if (1) {
			gint foo = 0;
			GtkText * text = GTK_TEXT(mw->custom_headers_text);
			guint len = gtk_text_get_length (text);
			gtk_text_set_point (text, len);
			gtk_editable_insert_text (GTK_EDITABLE(text), gstr->str, gstr->len, &foo);
			if (len)
				gtk_editable_delete_text (GTK_EDITABLE(text), 0, len);
		}

		/* cleanup */
		g_string_free (gstr, TRUE);
	}

	/**
	***  Subject
	**/

	cp = a != NULL ? a->subject : NULL;
	p = NULL;
	if (cp != NULL) {
		if (mw->type == EDIT_ORIGINAL)
			p = g_strdup (cp);
		else if (mw->type == EMAIL_FORWARD)
			p = g_strdup_printf ("[%s] %s", a->group->name, cp);
		else
			p = make_reply_string (cp);
	}
	if (is_nonempty_string(p))
		gtk_entry_set_text (GTK_ENTRY(mw->subject), p);
	g_free (p);


	/**
	***  Populate the other pieces
	**/

	if (mw->type == EDIT_ORIGINAL)
	{
		gchar * pch;
		const gchar * header;

		/* author */
		g_assert (mw->from!=NULL);
		pch = article_get_author_str (a);
		if (is_nonempty_string(pch))
			gtk_entry_set_text (GTK_ENTRY(mw->from), pch);
		g_free (pch);

		/* organization */
		g_assert (mw->organization!=NULL);
		header = article_get_header (a, HEADER_ORGANIZATION);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->organization), header);

		/* newsgroups */
		g_assert (mw->newsgroups!=NULL);
		header = article_get_header (a, HEADER_NEWSGROUPS);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), header);

		/* mail-to */
		g_assert (mw->users!=NULL);
		header = article_get_header (a, PAN_MAIL_TO);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->users), header);

		/* reply-to */
		g_assert (mw->reply_to!=NULL);
		header = article_get_header (a, HEADER_REPLY_TO);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->reply_to), header);

		/* followup-to */
		g_assert (mw->followup_to!=NULL);
		header = article_get_header (a, HEADER_FOLLOWUP_TO);
		if (is_nonempty_string(header))
			gtk_entry_set_text (GTK_ENTRY(mw->followup_to), header);

		/* editor */
		g_assert (mw->body!=NULL);
		pch = article_get_body (a);
		update_body_pane (mw->body, pch, FALSE);
		g_free (pch);

	}
	else /* a reply */
	{
		gchar * p;
		gboolean default_newsgroups = TRUE;

		/**
		***  BODY
		**/

		g_assert (mw->body!=NULL);
		if (a != NULL) {
			p = create_reply_body (a);
			update_body_pane (mw->body, p, FALSE);
			g_free (p);
		} else {
			GString * s = g_string_new (NULL);
			append_signature_file (s);
			update_body_pane (mw->body, s->str, FALSE);
			g_string_free (s, TRUE);
		}

		/**
		***   FROM
		**/

		p = get_default_author_from ();
		if (is_nonempty_string (p))
			gtk_entry_set_text (GTK_ENTRY(mw->from), p);
		g_free (p);

		/**
		***  ORGANIZATION
		**/

		p = gnome_config_get_string ("/Pan/User/Organization");
		if (p != NULL) {
			gtk_entry_set_text (GTK_ENTRY(mw->organization), p);
			g_free (p);
		}

		/**
		***   NEWSGROUPS
		***   MAILTO
		**/

		p = NULL;

		/* are we posting an article? */
		if (a!=NULL && is_posting(mw->type))
		{
			const gchar * followup = article_get_header (a, HEADER_FOLLOWUP_TO);
			const gchar * newsgroups = article_get_header (a, HEADER_NEWSGROUPS);

			/* if user set followup-to, use that;
			 * otherwise, use newsgroups;
			 * otherwise, use the currently-selected group */
			if (is_nonempty_string(followup))
			{
				/* Followup-To: poster */
				if (!g_strcasecmp ("poster", followup)) /* explicit reply by mail */
				{
					const gchar * replyto = article_get_header (a, HEADER_REPLY_TO);

					gnome_ok_dialog_parented (
						_("``Followup-To: poster'': sending email to author."),
						GTK_WINDOW(mw->window));

					if (is_nonempty_string(replyto))
						gtk_entry_set_text (GTK_ENTRY(mw->users), replyto);
					else {
						gchar * pch = article_get_author_str (a);
						gtk_entry_set_text (GTK_ENTRY(mw->users), pch);
						g_free (pch);
					}
				}
				else /* explicit followup groups */
				{
					gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), followup);
					default_newsgroups = FALSE;
				}
			}
			else if (is_nonempty_string(newsgroups)) /* explicit groups */
			{
				gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), newsgroups);
				default_newsgroups = FALSE;
			}
		}

		/* are we sending mail? */
		if (mw->type==EMAIL_REPLY || mw->type==EMAIL_AND_POST_REPLY)
		{
			/* figure out who to send mail to by checking reply-to: and from: */
			if (a!=NULL)
			{
				const gchar * replyto = article_get_header (a, HEADER_REPLY_TO);
				if (is_nonempty_string(replyto))
					gtk_entry_set_text (GTK_ENTRY(mw->users), replyto);
				else {
					gchar * pch = article_get_author_str (a);
					gtk_entry_set_text (GTK_ENTRY(mw->users), pch);
					g_free (pch);
				}
			}
		}



		/* no explicit newsgroup specified for this post,
		 * so let's guess where the user might want to post */
		if (default_newsgroups && is_posting(mw->type))
		{
			GString * str = g_string_new (NULL);

			if (a!=NULL && a->group!=NULL)
				g_string_sprintfa (str, "%s,", a->group->name);
			else {
				gint i;
				Group * thread_group = articlelist_get_group ();
				GPtrArray * ggroups = grouplist_get_selected_groups ();
				if (ggroups->len<2 && thread_group!=NULL)
					g_string_sprintfa (str, "%s,", thread_group->name);
				else for (i=0; i<ggroups->len; ++i) {
					Group * g = GROUP(g_ptr_array_index(ggroups,i));
					if (g!=NULL && !group_is_folder(g))
						g_string_sprintfa (str, "%s,", g->name);
				}
				g_ptr_array_free (ggroups, TRUE);
			}

			if (str->len != 0) {
				g_string_truncate (str, str->len-1); /* zotz last comma */
				gtk_entry_set_text (GTK_ENTRY(mw->newsgroups), str->str);
			}

			g_string_free (str, TRUE);
		}
	}
}

/*
static void
attachments_update_lines (MessageWindow *mw)
{
	struct stat statd;
   	char *filename = gtk_entry_get_text (GTK_ENTRY(mw->filename));

	if (filename && (0 == stat (filename, &statd))) {
		char *p;
		unsigned long lines;
		unsigned long lines_per_part;

		lines = statd.st_size / 45;
		p = g_strdup_printf ("%lu", lines);
		gtk_label_set_text (GTK_LABEL(mw->lines), p); 
		g_free (p);
		lines_per_part = gtk_spin_button_get_value_as_int (
	                      GTK_SPIN_BUTTON(mw->lines_per_part));
		p = g_strdup_printf ("%lu", (lines / lines_per_part) + 1);
		gtk_label_set_text (GTK_LABEL(mw->parts), p); 
		g_free (p);
	}
}
*/

/*
static void
attachments_lines_per_part_changed (GtkWidget *w, MessageWindow *mw)
{
	attachments_update_lines (mw);
}

static void
attachments_page_file_selected (GtkWidget *w, GtkWidget *f)
{
	char *selected_filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(f));
	MessageWindow * mw = gtk_object_get_user_data (GTK_OBJECT(f));
   	gtk_entry_set_text (GTK_ENTRY(mw->filename), selected_filename);
	attachments_update_lines (mw);
}

static void
attachment_file_selector (GtkWidget *menuitem, MessageWindow * mw) 
{
	GtkWidget *file_selector;

	file_selector = gtk_file_selection_new(_("Select the file to attach."));
   	gtk_object_set_user_data (GTK_OBJECT(file_selector), mw);

	gtk_signal_connect (
		GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
		"clicked", GTK_SIGNAL_FUNC (attachments_page_file_selected), 
		(gpointer) file_selector);
                            
    	gtk_signal_connect_object (
		GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->ok_button),
		"clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
		(gpointer) file_selector);
	gtk_signal_connect_object (GTK_OBJECT (
		GTK_FILE_SELECTION(file_selector)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
		(gpointer) file_selector);
    
    	gtk_widget_show (file_selector);
}
*/


static GtkWidget*
create_body_pane_nolock (MessageWindow * mw)
{
	GtkWidget * scrolled_window;
	GtkStyle * style;
	gboolean editable = mw->type != NNTP_READ;

	g_return_val_if_fail (mw!=NULL, NULL);

	mw->body = gtk_text_new (NULL, NULL);

	/* set the text widget's properties */
	pan_widget_set_font (GTK_WIDGET(mw->body), message_body_font);
	style = gtk_widget_get_style(mw->body);
	style->text[0] = text_fg_color;
	style->base[0] = text_bg_color;
	gtk_text_set_word_wrap (GTK_TEXT (mw->body), FALSE);
	gtk_text_set_line_wrap (GTK_TEXT (mw->body), FALSE);
	gtk_text_set_editable (GTK_TEXT (mw->body), editable);
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled_window), mw->body);

	return scrolled_window;
}


void
message_post_window (void)
{
	message_window_new (NULL, NNTP_POST);
}

static void
message_post_window_create (MessageWindow *mw)
{
	GtkWidget * w;

	pan_lock();
	mw->window = gnome_app_new ("Post", "New Message");
	gnome_app_create_menus_with_data (GNOME_APP (mw->window),
	                                  message_post_main_menu, mw);
	gnome_app_create_toolbar_with_data (GNOME_APP (mw->window),
	                                    message_post_toolbar, mw);
	pan_unlock();

	w = create_post_info_pane (mw);
	populate_post_info_pane (mw);
	experimental_wrap_handler (GTK_TEXT(mw->body),
	                           GTK_TOGGLE_BUTTON(message_post_toolbar[4].widget));

	pan_lock ();
	gnome_app_set_contents (GNOME_APP(mw->window), w);
	pan_unlock();
}


void
message_read_window (Article * article)
{
	if (article == NULL)
		article = get_current_article();

	if (article != NULL)
	{
		gchar * body;
		gboolean have_body;

		/* do we have the body already? */
		body = article_get_body (article);
		have_body = is_nonempty_string (body);
		g_free (body);

		if (have_body)
		{
			/* have the body, pop up the window now... */
			message_window_new (article, NNTP_READ);
		}
		else
		{
			/* don't have the body, go download it then pop up the window... */
			Task * task = TASK(task_body_new (article, TRUE, 
				NNTP_READ));
			queue_add (task);
		}
	}
}

/*--------------------------------------------------------------------
 *
 *--------------------------------------------------------------------*/
static void
message_read_window_create (MessageWindow *mw)
{
	GtkWidget * vbox = NULL;

	/* create a new window */
	pan_lock();
	mw->window = gnome_app_new ("Read", article_get_subject(mw->article));
	vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (mw->window), vbox);
	gnome_app_create_menus_with_data (GNOME_APP(mw->window),
	                                  message_read_main_menu, mw);
	gnome_app_create_toolbar_with_data (GNOME_APP(mw->window),
	                                    message_read_toolbar, mw);
	pan_unlock();

	/* create read info pane */
	mw->read_info_pane = create_read_info_pane (mw);

	pan_lock();
	gtk_box_pack_start (GTK_BOX(vbox),
	                    mw->read_info_pane,
	                    FALSE, FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (vbox),
	                    gtk_hseparator_new(),
	                    FALSE, FALSE, 2);
	gtk_box_pack_start (GTK_BOX(vbox),
	                    create_body_pane_nolock (mw),
	                    TRUE, TRUE, 0);
	pan_unlock();

	/* populate the body pane */
	if (1) {
		gchar * text = article_get_body (mw->article);
		update_body_pane (mw->body, text, text_get_mute_quoted());
		g_free (text);
	}
}


/*--------------------------------------------------------------------
 *
 *--------------------------------------------------------------------*/

static void
message_reply_external (GtkWidget       * widget,
                        MessageWindow   * mw)
{
	gchar * sub = NULL;
	gchar * body = NULL;
	gchar * author = NULL;
	Article * article;

	g_return_if_fail (mw!=NULL);
	g_return_if_fail (mw->article!=NULL);

	article = mw->article;

	/* What are we replying to anyway ? */
	if (article == NULL)
		article = get_current_article();
	if (article != NULL) {
		sub = make_reply_string (article_get_subject(article));
		author = article_get_author_str (article);
		body = article_get_body (article);
	}

	/* if we are replying to something, then go ahead and reply... */
	if (body!=NULL && author!=NULL && sub!=NULL)
	{
		/* create the command string and expand it */
		GString * exec = g_string_new (external_mailer);
		pan_g_string_replace (exec, "%r", author);
		pan_g_string_replace (exec, "%s", sub);

		/* If %t is being used, write the body to a temporary file */
		if ((pan_strstr(external_mailer, "%t")) != NULL) {
			FILE * f = NULL;
			gchar * filename = pan_make_temp (&f);
			if (filename != NULL) {
				gchar * out = create_reply_body (article);
				fputs (out, f);
				g_free (out);
				pan_g_string_replace (exec, "%t", filename);
				fclose (f);
			}
			g_free (filename);
		}

		/* Launch the command */
		gnome_execute_shell (NULL, exec->str);

		g_string_free (exec, TRUE);
	}
	
	/* clean up */
	g_free (sub);
	g_free (author);
	g_free (body);
}

static void
make_reply_window (Article * article, MessageWindowType type)
{
	gchar * body;

	g_return_if_fail (article!=NULL);
	g_return_if_fail (type==EMAIL_REPLY
		|| type==NNTP_REPLY
		|| type==EMAIL_AND_POST_REPLY
		|| type==EMAIL_FORWARD);

	/* if this isn't the same article in the article reading window,
	 * then fire up a task to retreive the body from the local cache
	 * or on the net.  FIXME task-body really needs to be decoupled
	 * from viewing.
	 */
	if (article != get_current_article()) {
		text_set_from_article (article, type, TRUE);
		return;
	}

	/* open up a populated reply window. */
	body = text_get_message_to_reply_to ();
	article_set_header (article, PAN_REPLY_PORTION, body, DO_CHUNK);
	message_window_new (article, type);
	g_free (body);
}

void
message_forward_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_FORWARD);
}

void
message_edit_window (Article * a)
{
	if (a == NULL)
		a = get_current_article();
	if (a != NULL)
		message_window_new (a, EDIT_ORIGINAL);
}

void
message_reply_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_REPLY);
}

void
message_followup_reply_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, EMAIL_AND_POST_REPLY);
}


void
message_followup_window (void)
{
	Article * a = get_current_article();
	if (a != NULL)
		make_reply_window (a, NNTP_REPLY);
}

static void
message_edit_window_create (MessageWindow *mw)
{
	GtkWidget * w;

	pan_lock ();
	mw->window = gnome_app_new ("Edit", article_get_subject(mw->article));
	gnome_app_create_menus_with_data (GNOME_APP(mw->window),
	                                  message_edit_main_menu,
	                                  mw);
	gnome_app_create_toolbar_with_data (GNOME_APP(mw->window),
	                                    message_post_toolbar,
	                                    mw);
	pan_unlock ();

	w = create_post_info_pane (mw);
	populate_post_info_pane (mw);
	experimental_wrap_handler (GTK_TEXT(mw->body),
	                           GTK_TOGGLE_BUTTON(message_post_toolbar[4].widget));

	pan_lock ();
	gnome_app_set_contents (GNOME_APP(mw->window), w);
	gtk_text_set_editable (GTK_TEXT(mw->body), TRUE);
	pan_unlock();
}
static void
message_reply_window_create (MessageWindow * mw)
{
	GtkWidget * w;
	gchar * title = make_reply_string (article_get_subject(mw->article));

	pan_lock();
	mw->window = gnome_app_new ("Reply", title);
	gnome_app_create_menus_with_data (GNOME_APP(mw->window),
	                                  message_post_main_menu, mw);
	gnome_app_create_toolbar_with_data (GNOME_APP(mw->window),
	                                    message_post_toolbar, mw);
	pan_unlock();

	w = create_post_info_pane (mw);
	populate_post_info_pane (mw);
	experimental_wrap_handler (GTK_TEXT(mw->body),
	                           GTK_TOGGLE_BUTTON(message_post_toolbar[4].widget));

	pan_lock ();
	gnome_app_set_contents (GNOME_APP (mw->window), w);
	pan_unlock();

	/* cleanup */
	g_free (title);
}

void
message_window_new (Article * article, MessageWindowType type)
{
	MessageWindow *mw;

	if (article != NULL)
		group_ref_articles (article->group, NULL);

	mw = g_new0 (MessageWindow, 1);
	mw->article = article;
	mw->type = type;

	switch (type)
	{
		case NNTP_READ:
			message_read_window_create (mw);
			break;

		case NNTP_POST:
			message_post_window_create (mw);
			break;

		case NNTP_REPLY:
		case EMAIL_REPLY:
		case EMAIL_AND_POST_REPLY:
		case EMAIL_FORWARD:
			message_reply_window_create (mw);
			break;

		case EDIT_ORIGINAL:
			message_edit_window_create (mw);
			break;

		default:
			break;
	}

	if (mw->window != NULL)
	{
		pan_lock ();
		gtk_window_set_default_size (GTK_WINDOW (mw->window), 550, 410);
		gtk_signal_connect (GTK_OBJECT(mw->window), "delete_event",
		                    GTK_SIGNAL_FUNC(window_delete_event_cb), mw);
		gtk_signal_connect (GTK_OBJECT(mw->window), "destroy",
		                    GTK_SIGNAL_FUNC(window_destroy_cb), mw);
		gtk_window_set_policy (GTK_WINDOW(mw->window), TRUE, TRUE, TRUE);
		pan_unlock ();

		gui_restore_window_size (mw->window, "message_window");

		pan_lock ();
		gtk_widget_show_all (mw->window);
		pan_unlock ();
	}
}

/***
****
****  SENDING THE MESSAGE
****
***/

static gchar*
get_text_from_editable (GtkWidget* w)
{
	gchar * pch;
	g_return_val_if_fail (GTK_IS_EDITABLE(w), NULL);
	pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
	g_strstrip (pch);
	return pch;
}

static gchar*
get_text_from_maybe_editable (GtkWidget * w)
{
	gchar * pch = NULL;

	if (w!=NULL && GTK_IS_EDITABLE(w))
	{
		pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
		g_strstrip (pch);
	}

	return pch;
}

#if 0
static gchar*
message_window_get_lines_per_part (const MessageWindow* mw)
{
	int lpp; 

	g_return_val_if_fail (mw!=NULL, NULL);
	
	lpp = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(mw->lines_per_part));

	return g_strdup_printf ("%d", lpp);
}
#endif

#if 0
static gchar*
message_window_get_filename (const MessageWindow* mw)
{
	gchar * pch;

	g_return_val_if_fail (mw!=NULL, NULL);

	pch = get_text_from_maybe_editable (mw->filename);

	return pch;
}
#endif

static gchar*
message_window_get_reply_to (const MessageWindow* mw)
{
	gchar * pch;

	g_return_val_if_fail (mw!=NULL, NULL);

	pch = get_text_from_maybe_editable (mw->reply_to);
	if (pch == NULL)
		pch = g_strdup ("");

	return pch;
}

static gchar*
message_window_get_group_str (GtkWidget * e)
{
	gchar * pch;

	g_return_val_if_fail (e!=NULL, NULL);

	pch = get_text_from_maybe_editable (e);
	if (pch == NULL)
	{
		pch = g_strdup ("");
	}
	else
	{
		gchar * tmp;

		/* ensure all the delimiters are commas */
		g_strdelimit (pch, ":; ", ',');

		/* remove leading/trailing commas */
		for (tmp=pch; *tmp==','; ++tmp);
		g_memmove (pch, tmp, strlen(tmp)+1);
		for (tmp=pch+strlen(pch)-1; tmp>pch && *tmp==','; --tmp);
		tmp[1] = '\0';

		/* remove empty entries */
		while ((tmp=pan_strstr(pch,",,")) != NULL)
			g_memmove (tmp, tmp+1, strlen(tmp+1)+1);
	}

	return pch;
}

static gchar*
message_window_get_organization (const MessageWindow* mw)
{
	gchar * pch = NULL;
	g_return_val_if_fail (mw!=NULL, NULL);
	pch = get_text_from_maybe_editable (mw->organization);
	if (pch == NULL)
		pch = gnome_config_get_string ("/Pan/User/Organization");
	if (pch == NULL)
		pch = g_strdup ("");
	if (pch != NULL)
		g_strstrip (pch);
	return pch;
}

static gchar*
message_window_get_reverse_path (const MessageWindow* mw)
{
	gchar * from = get_text_from_editable (mw->from);
	gchar * address = NULL;
	gchar * realname = NULL;
	gchar * retval = NULL;
	int status = gnksa_do_check_from (from, &address, &realname, FALSE);
	g_free (from);
	g_free (realname);
	if (status == GNKSA_OK)
		retval = address;
	else
		g_free (address);
	return retval;
}


static void
populate_article_from_mw (Article * article, const MessageWindow * mw)
{
	gint i;
	gchar ** sd;
	gchar * pch;
	gchar * author_addr;
	gchar * author_real;

	/* date */
	article->date = time(0);

	/* group's article number */
	article->number = 1;

	/* author */
	pch = get_text_from_editable (mw->from);
	gnksa_do_check_from (pch, &author_addr, &author_real, FALSE);
	if (is_nonempty_string (author_real))
		gnksa_strip_realname (author_real);
	article->author_addr = is_nonempty_string (author_addr)
		? group_chunk_string (article->group, author_addr, TRUE)
		: NULL;
	article->author_real = is_nonempty_string (author_real)
		? group_chunk_string (article->group, author_real, TRUE)
		: NULL;
	g_free (author_addr);
	g_free (author_real);
	g_free (pch);

	/* message-id */
	pch = gnksa_generate_message_id_from_email_addr (article->author_addr);
	article_init_header (article, HEADER_MESSAGE_ID, pch, DO_CHUNK);
	g_free (pch);

	/* subject */
	pch = get_text_from_editable (mw->subject);
	article_init_header (article, HEADER_SUBJECT, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* references (for posting articles) */
	if ((mw->type==NNTP_REPLY || mw->type==EMAIL_REPLY || mw->type==EMAIL_AND_POST_REPLY)
	    && (mw->article != NULL))
	{
		const gchar * refs = article_get_header (mw->article, HEADER_REFERENCES);
		const gchar * msg_id = article_get_message_id (mw->article);
		pch = gnksa_generate_references (refs, msg_id);
		if (pch != NULL) {
			article_init_header (article, HEADER_REFERENCES, pch, DO_CHUNK_SHARE);
			g_free (pch);
		}
	}

	/* organization */
	pch = message_window_get_organization(mw);
	article_init_header (article, HEADER_ORGANIZATION, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* followup-to */
	pch = message_window_get_group_str(mw->followup_to);
	article_init_header (article, HEADER_FOLLOWUP_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* reply-to */
	pch = message_window_get_reply_to (mw);
	article_init_header (article, HEADER_REPLY_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-body */
	pch = get_text_from_editable (mw->body);
	article_init_header (article, PAN_BODY, pch, DO_CHUNK);
	g_free (pch);

	/* newsgroups (for posting articles) */
	pch = message_window_get_group_str (mw->newsgroups);
	article_init_header (article, HEADER_NEWSGROUPS, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-mail-to (for sending mail) */
	pch = get_text_from_editable (mw->users);
	article_init_header (article, PAN_MAIL_TO, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* pan-reverse-path (for sending mail) */
	pch = message_window_get_reverse_path (mw);
	if (is_nonempty_string(pch))
		article_init_header (article, PAN_REVERSE_PATH, pch, DO_CHUNK_SHARE);
	g_free (pch);

	/* attribution header (for message-check) */
	if (mw->article!=NULL) {
		gchar * attribution = article_get_attribution_string (mw->article);
		if (is_nonempty_string(attribution))
			article_init_header (article, PAN_ATTRIBUTION, attribution, DO_CHUNK);
		g_free (attribution);
	}

#if 0
	/* attachment: filename */
	pch = message_window_get_filename (mw);
	if (pch != NULL) {
		article_set_header (article, PAN_ATTACH_FILE, pch, DO_CHUNK);
		g_free (pch);
	}

	/* attachment: lines per article */
	pch = message_window_get_lines_per_part (mw);
	if (pch != NULL) {
		article_set_header (article, PAN_LINES_PER_PART, pch, DO_CHUNK);
		g_free (pch);
	}
#endif

	/* custom headers */
	pch = gtk_editable_get_chars (GTK_EDITABLE(mw->custom_headers_text), 0, -1);
	sd = g_strsplit (pch, "\n", -1);
	for (i=0; sd!=NULL && sd[i]!=NULL; ++i) {
		gchar * delimit;
		g_strstrip (sd[i]);
		delimit = strchr (sd[i], ':');
		if (delimit != NULL) {
			gchar * key = g_strndup (sd[i], delimit-sd[i]);
			gchar * val = g_strdup (delimit + 1);
			g_strstrip (key);
			g_strstrip (val);
			article_init_header (article, key, val, DO_CHUNK_SHARE);
			g_free (key);
			g_free (val);
		}
	}
	g_strfreev (sd);
	g_free (pch);
}

static Article*
build_article_from_window (const MessageWindow * mw)
{
	Group * folder;
	Article * article;
	debug_enter ("build_article_from_window");

	folder  = serverlist_get_named_folder (PAN_SENDLATER);
	article = article_new (folder);
	populate_article_from_mw (article, mw);

	debug_exit ("build_article_from_window");
	return article;
}

static void
save_cb (GtkWidget * button, MessageWindow * mw)
{
	Article * a = mw->article;
	Article * tmp;
	guint i;
	GPtrArray * headers;

	/* update the article fields from changes made in the gui */
	tmp = article_new (a->group);
	populate_article_from_mw (tmp, mw);
	a->author_addr = tmp->author_addr;
	a->author_real = tmp->author_real;
	a->subject = tmp->subject;

	/* replace old headers with new headers */
	headers = article_get_all_headers (a);
	for (i=0; i!=headers->len; i+=2) {
		gchar * key = (gchar*) g_ptr_array_index (headers, i);
		article_remove_header (a, key);
	}
	g_ptr_array_free (headers, TRUE);
	headers = article_get_all_headers (tmp);
	for (i=0; i!=headers->len; i+=2) {
		gchar * key = (gchar*) g_ptr_array_index (headers, i);
		gchar * val = (gchar*) g_ptr_array_index (headers, i+1);
		article_set_header (a, key, val, 0);
	}
	g_ptr_array_free (headers, TRUE);

	/* cleanup */
	pan_object_unref (PAN_OBJECT(tmp));
	message_window_destroy (mw);
}

typedef enum
{
	SEND_NOW,
	SEND_LATER
}
SendMode;

static void
post (MessageWindow * mw, SendMode mode)
{
	Article * article;
	Server * server;
	GPtrArray * errors;
	GoodnessLevel goodness;
	debug_enter ("post");

       	server = serverlist_get_active_server ();
       	errors = g_ptr_array_new ();

	/* Get the article pointer, either through repopulating the
	 * current article (if we're editing an existing message) or
	 * by creating a new article (if this is a new message).
	 */
	if (mw->type == EDIT_ORIGINAL) {
		article = mw->article;
		populate_article_from_mw (article, mw);
	} else {
		article = build_article_from_window (mw);
	}

	/* if the article's okay, then send or queue */
	check_article_and_prompt (article, server, errors, &goodness);
	if (goodness == OKAY)
	{
		Group * sendlater = serverlist_get_named_folder (PAN_SENDLATER);

		/* gotta ref the articles to make sure that, after sendlater
		 * takes over memory management of article, it stays alive
		 * until after we've called queue_article_for_posting. */
		group_ref_articles (sendlater, NULL);

		/* store which server to post through, in case the article
		 * winds up waiting in pan.sendlater */
		if (!is_nonempty_string(article_get_header (article, PAN_SERVER)))
			article_set_header (article, PAN_SERVER, server->name, DO_CHUNK_SHARE);

		/* add this article to the sendlater group */
		group_add_article (sendlater, article);

		/* try to post right now, if desired... */
		if (mode == SEND_NOW)
			queue_article_for_posting (server, article, FALSE);

		/* close this window */
		message_window_destroy (mw);

		group_unref_articles (sendlater, NULL);
	}

	/* cleanup */
	pan_g_ptr_array_foreach (errors, (GFunc)g_free, NULL);
	g_ptr_array_free (errors, TRUE);
	if (goodness!=OKAY && mw->type!=EDIT_ORIGINAL)
		group_remove_article (serverlist_get_named_folder(PAN_SENDLATER), article);

	debug_exit ("post");
}

static void
send_now_cb (GtkWidget * button, MessageWindow * mw)
{
	post (mw, SEND_NOW);
}
static void
send_later_cb (GtkWidget * button, MessageWindow * mw)
{
	post (mw, SEND_LATER);
}

/**
***  Save as File
**/

static void
message_window_save_ok_clicked (GtkWidget * widget, GtkWidget * file_entry)
{
	const gchar * filename;
	gchar * writeme = NULL;
	const MessageWindow * mw;

	/* get information from the widget */
	pan_lock();
	mw = (const MessageWindow*) gtk_object_get_data (GTK_OBJECT(file_entry), "mw");
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_entry));
	pan_unlock();

	/* get the article text to save... */
	if (mw->type == NNTP_READ)
	{
		writeme = article_get_message (mw->article);
	}
	else
	{
		Article * a = build_article_from_window (mw);
		writeme = article_get_message (a);
		pan_object_unref (PAN_OBJECT(a));
	}

	/* write the text */
	if (is_nonempty_string(writeme))
	{
		FILE * msg_dump = fopen (filename, "w");
		if (msg_dump != NULL)
		{
			fprintf (msg_dump, "%s\n", writeme);
			fclose (msg_dump);

			pan_lock();
			gtk_widget_destroy (file_entry);
			pan_unlock();
		}
		else
		{
		}
	}

	/* cleanup */
	g_free (writeme);
}

static void
message_window_save_cb (GtkWidget *widget, MessageWindow * mw)
{
	GtkWidget *file_entry = NULL;
	
	pan_lock();
	
	file_entry = gtk_file_selection_new (_("Save message to file"));

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (file_entry));

	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (file_entry)->cancel_button),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (file_entry));

	/* message_id is const, but we don't alter it here */
	gtk_object_set_data (GTK_OBJECT (file_entry), "mw", (gpointer)mw);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (file_entry)->ok_button),
				"clicked",
				GTK_SIGNAL_FUNC (message_window_save_ok_clicked),
				file_entry);

	gtk_widget_show_all (file_entry);
	pan_unlock();
}
