/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001 Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

#include <gnome.h>

#ifdef USE_GTKHTML
#include <gtkhtml/gtkhtml.h>
#include <gtkhtml/htmltext.h>
#include <gtkhtml/htmlengine.h>
#include <gtkhtml/htmlfontmanager.h>
#include <gtkhtml/htmlpainter.h>
#include <gtkhtml/htmltextslave.h>
#include <gtkhtml/htmlselection.h>
#include <gtkhtml/gtkhtml-properties.h>
#include <gal/widgets/e-font.h>
#endif

#include <gmime/gmime.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-object.h>
#include <pan/base/util-file.h>
#include <pan/base/util-mime.h>
#include <pan/base/util-wrap.h>

#include <pan/articlelist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/globals.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h>
#include <pan/util.h>

static GdkColor headers_col = {0, 0xbbbb, 0, 0};
#ifndef USE_GTKHTML
static GtkWidget * header_table = NULL;
#endif

static void ensure_header_col_inited (void);
static void realize_text_cb (GtkWidget *text, gpointer data);
static void text_button_press_cb (GtkWidget *widget, GdkEventButton *bevent);
static void show_all_headers_cb (GtkCheckMenuItem* item, gpointer data);
static void mute_quoted_text_cb (GtkCheckMenuItem* item, gpointer data);

static void
rot13_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_rot13 (item->active);
}

/***
****
****  POPUP MENUS
****
***/

static GnomeUIInfo respond_menu [] =
{
        {
		GNOME_APP_UI_ITEM,
		N_("Followup to newsgroup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RPL,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply by E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RPL,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Followup to newsgroup and Reply by E-mail"),
		N_("Send a reply both to the author in mail, and to the news server."),
		message_followup_reply_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RPL,
		0, 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Forward article by E-mail"),
		N_("Forward article by E-mail"),
		message_forward_window
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo filter_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Watch Thread"),
		N_("Watch Thread"),
		articlelist_selected_thread_watch
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Ignore Thread"),
		N_("Ignore Thread"),
		articlelist_selected_thread_ignore
	},
	GNOMEUIINFO_END
};

static void
text_read_body_in_new_window (void)
{
	Article * article = get_current_article ();
	if (article != NULL)
		message_read_window (article);
}


static GnomeUIInfo text_menu_popup[] =
{
	{ /* 0 */
		GNOME_APP_UI_ITEM,
		N_("Open in New Window"),
		N_("Open this message for reading."),
		text_read_body_in_new_window
	},
	GNOMEUIINFO_SEPARATOR, /* 1 */
        { /* 2 */
		GNOME_APP_UI_SUBTREE,
		N_("Respond"),
		N_("Respond"),
		&respond_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{ /* 3 */
		GNOME_APP_UI_SUBTREE,
		N_("Filter"),
		N_("Filter"),
		&filter_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR, /* 4 */
	{ /* 5 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show All Headers in Body"),
		N_("Show All Headers in Body"),
		show_all_headers_cb
	},
	{ /* 6 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Mute Quoted Text"),
		N_("Mute Quoted Text"),
		mute_quoted_text_cb
	},
	{ /* 7 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Rot13 the Body"),
		N_("Rot13 the Body"),
		rot13_cb
	},
	GNOMEUIINFO_END
};

static gchar welcome[] =
{
#ifdef USE_GTKHTML
"<html><body><font face=\"times\" size=\"12\">"
"<b>Pan " VERSION "</b><br>"
"Copyright (C) 1999, 2000, 2001 <br>"
"Pan Development Team"
"<p>"
"This is beta software.  If you find a bug, please <a href=\"http://pan.rebelbase.com/bugreport.html\">report it</a>."
"<p>"
"<a href=\"http://pan.rebelbase.com/contact.html\">Request a Feature</a><br>"
"<a href=\"http://pan.rebelbase.com/bugreport.html\">Report a Bug</a><br>"
"<a href=\"http://pan.rebelbase.com/download.html\">Upgrade Pan</a>"
"<p>"
"This program is free software; you can redistribute it "
"and/or modify it under the terms of the GNU General Public "
"License as published by the Free Software Foundation; "
"either version 2 of the License, or (at your option) any "
"later version."
"<p>"
"This program is distributed in the hope that it will be "
"useful, but WITHOUT ANY WARRANTY; without even the implied "
"warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR "
"PURPOSE.  See the GNU General Public License for more "
"details."
"<p>"
"The GNU Public License can be found from the menu above "
"in Help|About|License."
"</font></body></html>"
#else
"Pan " VERSION "\n"
"Copyright (C) 1999, 2000, 2001 \n"
"Pan Development Team (pan@rebelbase.com) \n"
"\n"
"  This is beta software.  If you find a bug, please report it.\n"
"\n"
"  http://pan.rebelbase.com/contact.html   -  Request a Feature\n"
"  http://pan.rebelbase.com/bugreport.html -  Report a Bug\n"
"  http://pan.rebelbase.com/download.html  -  Upgrade Pan\n"
"\n"
"This program is free software; you can redistribute it \n"
"and/or modify it under the terms of the GNU General Public \n"
"License as published by the Free Software Foundation; \n"
"either version 2 of the License, or (at your option) any \n"
"later version. \n"
"\n"
"This program is distributed in the hope that it will be \n"
"useful, but WITHOUT ANY WARRANTY; without even the implied \n"
"warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR \n"
"PURPOSE.  See the GNU General Public License for more \n"
"details. \n"
"\n"
"The GNU Public License can be found from the menu above \n"
"in Help|About|License. \n"
#endif
};

static GtkWidget * text_rightclick_popup = NULL;
PanCallback * current_article_changed = NULL;
static GMimeMessage * mm = NULL;
static gboolean rot13_enabled = FALSE;

static Article * current_article = NULL;
GdkColor text_fg_color;
GdkColor text_bg_color;
GdkColor text_quoted_color;


static void
mute_quoted_text_cb (GtkCheckMenuItem * item, gpointer data)
{
	text_set_mute_quoted (item->active);
}

static void
show_all_headers_cb (GtkCheckMenuItem * item, gpointer data)
{
	text_set_show_all_headers (item->active);
}

void
text_select_all (void)
{
	debug_enter ("text_select_all");

	pan_lock();
	gtk_editable_select_region (GTK_EDITABLE(Pan.text), 0, -1);
	pan_unlock();

	debug_exit ("text_select_all");
}

/**
***  HTML
**/

static void
sylpheed_textview_smooth_scroll_do (GtkAdjustment  * vadj,
                                    gfloat           old_value,
                                    gfloat           last_value,
                                    gint             step)
{
	gint i;
	gint change_value;
	gboolean up;

	if (old_value < last_value) {
		change_value = last_value - old_value;
		up = FALSE;
	} else {
		change_value = old_value - last_value;
		up = TRUE;
	}

	gdk_key_repeat_disable ();

	for (i=step; i<=change_value; i+=step)
		gtk_adjustment_set_value (vadj, old_value+(up?-i:i));
	gtk_adjustment_set_value (vadj, last_value);

	gdk_key_repeat_restore ();
}

void
text_read_more (void)
{
#ifdef USE_GTKHTML
	GtkAdjustment * v = GTK_LAYOUT(Pan.text)->vadjustment;
#else
	GtkAdjustment * v = GTK_TEXT(Pan.text)->vadj;
#endif

	/* try to go down a page */
	const gfloat val = CLAMP (v->value + v->page_size,
	                          v->lower,
	                          MAX(v->upper,v->page_size)-MIN(v->upper,v->page_size));

	/* if successful, then update the viewpane; otherwise, go to the next article */
	if (val == v->value)
		articlelist_view_next ();
	else if (text_window_smooth_scrolling)
		sylpheed_textview_smooth_scroll_do (v, v->value, val,
		                                    text_window_smooth_scrolling_speed);
	else
		gtk_adjustment_set_value (v, val);
}


#ifdef USE_GTKHTML

static void
add_header (GString * buf, const gchar * header_name, const gchar * val)
{
	gchar * tmp_val;
	debug_enter ("add_header");

       	tmp_val = pan_str_escape (val);
	if (is_nonempty_string (tmp_val))
		g_string_sprintfa (buf, "<tr><td><b>%s</b> %s</td></tr>", header_name, tmp_val);

	g_free (tmp_val);
	debug_exit ("add_header");
}

typedef struct
{
	gint i;
	const Article * a;
	GString * str;
}
AttachmentWorkStruct;

static void
add_inline_images_mpfunc (GMimePart * part, gpointer user_data)
{
	AttachmentWorkStruct * data = (AttachmentWorkStruct*) user_data;
	GString * str = data->str;
	const gchar * content_id;
	const GMimeContentType * type;

	/*if (pan_strcmp ("inline", g_mime_part_get_content_disposition (part)))
		return;*/ /* let all images get shown */

	type = g_mime_part_get_content_type (part);
	if (!g_mime_content_type_is_type (type, "image", "*"))
		return;

	content_id = g_mime_part_get_content_id (part);
	if (!is_nonempty_string(content_id)) /* uh oh, no cid, let's make one */
	{
		GString * id = g_string_new (article_get_message_id(data->a));
		g_string_erase (id, id->len-1, 1); /* '>' */
		g_string_sprintfa (id, "___%d>", data->i++);
		g_mime_part_set_content_id (part, id->str);
		content_id = g_mime_part_get_content_id (part);
		g_string_free (id, TRUE);
	}

	g_string_sprintfa (str, "<img alt=\"%s\" src=\"cid:%s\"><p>\n", content_id, content_id);
}

static gchar*
colorize_html (const gchar * body)
{
	gint line_len;
	const gchar * line_start;
	GString * buf;
	gchar * retval;
	gchar quoted_color[64];
	gchar normal_color[64];

	/* sanity clause */
	g_return_val_if_fail (body!=NULL, NULL);

	/* init the color strings */
	sprintf (quoted_color, "<font color=\"#%02x%02x%02x\">",
	                       text_quoted_color.red   >>8,
	                       text_quoted_color.green >>8,
	                       text_quoted_color.blue  >>8);
	sprintf (normal_color, "<font color=\"#%02x%02x%02x\">",
	                       text_fg_color.red   >>8,
	                       text_fg_color.green >>8,
	                       text_fg_color.blue  >>8);

	/* colorize each line */
	line_len = 0;
	line_start = NULL;
	buf = g_string_sized_new (strlen(body) * 2);
	while (get_next_token_range (body, '\n', &body, &line_start, &line_len)) {
		const gboolean is_reply = !strncmp (line_start, "&gt;", 4);
		g_string_append (buf, is_reply ? quoted_color : normal_color);
		pan_g_string_append_len (buf, line_start, line_len);
		g_string_append (buf, "</font>\n");
	}

	retval = buf->str;
	g_string_free (buf, FALSE);
	return retval;
}

static void
text_get_article_body (const Article * a, GString * buf)
{
	gchar * body = NULL;
	GMimePart * mp = NULL;
	gboolean is_html = FALSE;
	const GMimeContentType * type;

	debug_enter ("text_get_article_body");

	g_return_if_fail (mm!=NULL);
	mp = mm->mime_part;
	type = g_mime_part_get_content_type (mp);
	body = g_mime_message_get_body (mm, FALSE, &is_html);

	if (g_mime_content_type_is_type (type, "image", "*"))
	{
		const gchar * content_id = g_mime_part_get_content_id (mp);
		if (!is_nonempty_string(content_id))
		{
			GString * id = g_string_new (article_get_message_id(a));
			g_string_erase (id, id->len-1, 1); /* '>' */
			g_string_sprintfa (id, "___%d>", 0);
			g_mime_part_set_content_id (mp, id->str);
			content_id = g_mime_part_get_content_id (mp);
			g_string_free (id, TRUE);
		}
		body = g_strdup_printf ("<img alt=\"%s\" src=\"cid:%s\"><p>\n", content_id, content_id);
		is_html = TRUE;
	}

	if (body==NULL && mp->children==NULL)
	{
		g_string_append (buf, _("No Body Found"));
	}

	if (is_html)
	{
		gchar * begin;
		gchar * end;

 		begin = pan_stristr (body, "<body>");
		if (begin == NULL)
			begin = body;
		end = pan_stristr (begin, "</body>");
		if (end == NULL)
			end = pan_stristr (begin, "</html>");
		if (end != NULL)
			*end = '\0';

		g_string_append (buf, begin);
	}
	else
	{
		/* the traditional body */
		if (is_nonempty_string(body))
		{
			if (rot13_enabled)
				replace_gstr (&body, rot13(body));
			if (text_get_mute_quoted())
				replace_gstr (&body, mute_quoted_text_in_body(body));
			if (text_get_wrap())
				replace_gstr (&body, fill_body(body,wrap_column));
			replace_gstr (&body, pan_str_escape (body));
			replace_gstr (&body, colorize_html (body));
			replace_gstr (&body, linkify_text(body));
		}

		/* the headers */
		if (text_get_show_all_headers())
		{
			gchar * headers = g_mime_message_get_headers (mm);
			replace_gstr (&headers, pan_str_escape (headers));
			replace_gstr (&body, g_strdup_printf ("%s\n\n%s", headers, (body?body:"")));
			g_free (headers);
		}

		/* add the body */
		g_string_append (buf, "<pre>");
		g_string_append (buf, body?body:"");
		g_string_append (buf, "</pre>");

		/* attachments */
		if (1) {
			AttachmentWorkStruct tmp;
			tmp.i = 0;
			tmp.a = a;
			tmp.str = g_string_new (NULL);
			g_mime_part_foreach (mp, add_inline_images_mpfunc, &tmp);
			g_string_append (buf, tmp.str->str);
			g_string_free (tmp.str, TRUE);
		}
			
	}

	/* cleanup */
	g_free (body);
	debug_exit ("text_get_article_body");
}

static void
text_get_article_headers (const Article * a, GString * buf)
{
	debug_enter ("text_get_article_headers");

	g_return_if_fail (a!=NULL);
	g_return_if_fail (buf!=NULL);

	g_string_append (buf, "<table width=\"100%\" border=\"0\" cellspacing=\"0\" cellpadding=\"5\">");
	g_string_append (buf, "<tr> <td width=\"20\" bgcolor=\"#008899\">&nbsp; </td><td><table>");

	if (header_flags & UI_HEADER_SUBJECT) {
		gchar * subject = pan_str_escape (article_get_subject(a));
		g_string_sprintfa (buf, "<tr><td><b>%s</b></td></tr>", subject);
		g_free (subject);
	}
	if (header_flags & UI_HEADER_AUTHOR) {
		gchar * author = article_get_author_str (a);
		replace_gstr (&author, pan_str_escape(author));
		if (is_nonempty_string(a->author_addr))
			replace_gstr (&author, g_strdup_printf("<a href=\"mailto:%s\">%s</a>", a->author_addr, author));
		g_string_sprintfa (buf, "<tr><td><b>%s:</b> %s</td></tr>", _("From"), author);
		g_free (author);
	}
	if (header_flags & UI_HEADER_MESSAGE_ID)
		add_header (buf, _("Message-ID:"), article_get_message_id(a));
	if (header_flags & UI_HEADER_REFERENCES)
		add_header (buf, _("References:"), a->references);
	if (header_flags & UI_HEADER_REPLY_TO)
		add_header (buf, _("Reply-To:"), article_get_header(a, HEADER_REPLY_TO));
	if (header_flags & UI_HEADER_FOLLOWUP_TO)
		add_header (buf, _("FollowUp-To:"), article_get_header(a, HEADER_FOLLOWUP_TO));
	if (header_flags & UI_HEADER_NEWSGROUPS)
		add_header (buf, _("Newsgroups:"), article_get_header(a, HEADER_NEWSGROUPS));
	if (header_flags & UI_HEADER_DATE) {
		gchar * date = get_date_display_string (a->date, body_date_format);
		add_header (buf, _("Date:"), date);
		g_free (date);
	}

	g_string_append (buf, "</table></td></tr>");

	/* References */
	g_string_append (buf, "<tr><td colspan=\"2\" bgcolor=\"#008899\">&nbsp;");
	if (is_nonempty_string(a->references))
	{
		Group * g = a->group;
		gchar ** ids = g_strsplit (a->references, " ", -1);
		int i;
		int ref_nr = 0;

		g_string_sprintfa (buf, "<b>%s</b>&nbsp;", _("References:"));
		for (i=0; ids!=NULL && ids[i]!=NULL; ++i)
		{
			gchar * id = ids[i];

			if (*id == 0) continue;

			if (group_get_article_by_message_id (g, id) != NULL)
				g_string_sprintfa (buf, "<a href=\"message-id:%s\">%d</a>&nbsp;", id, 
					++ref_nr);
			else
				g_string_sprintfa (buf, "%d&nbsp;", ++ref_nr);
		}
		g_strfreev (ids);
	}
	g_string_append (buf, "</td></tr>");

	/* end of headers */
	g_string_append (buf, "</table>");
	debug_exit ("text_get_article_headers");
}

static void
set_text_from_article (const Article * a)
{
	gboolean set_face = is_nonempty_string (message_body_font);
	GString * buf = g_string_new (NULL);
	debug_enter ("set_text_from_article");

	/* build the html string */
	g_string_append (buf, "<html><body>");
	if (set_face)
		g_string_sprintfa (buf, "<font face=\"%s\">", message_body_font);
	text_get_article_headers (a, buf);
	text_get_article_body (a, buf);
	if (set_face)
		g_string_append (buf, "</font>");
	g_string_append (buf, "</body></html>");

	/* update the text window */
	text_set_raw (buf->str);

	/* cleanup */
	g_string_free (buf, TRUE);
	debug_exit ("set_text_from_article");
}

static void
url_requested (GtkHTML *html, const char *url, GtkHTMLStream *handle, gpointer data)
{
	int end = GTK_HTML_STREAM_ERROR;
	debug_enter ("url_requested");

	g_return_if_fail (html != NULL);
	g_return_if_fail (is_nonempty_string(url));

	if (!g_strncasecmp (url, "cid:", 4))
	{
		char * cid = NULL;
		const GMimePart * part = NULL;
		const gchar * content = NULL;
		guint len = 0;

		cid = g_strdup (url + 4);
		if (*cid != '<')
			replace_gstr (&cid, g_strdup_printf ("<%s>", cid));
		if (mm!=NULL && mm->mime_part!=NULL)
			part = g_mime_part_get_subpart_from_content_id (mm->mime_part, cid);
		if (part != NULL)
			content = g_mime_part_get_content (part, &len);
		if (content != NULL) {
			gtk_html_write (html, handle, content, len);
			end = GTK_HTML_STREAM_OK;
		}
	}
	else
	{
		GArray * a = read_file (url);
		if (a != NULL)
		{
			gtk_html_write (html, handle, a->data, a->len);
			end = GTK_HTML_STREAM_OK;
			g_array_free (a, TRUE);
		}
	}

	gtk_html_end (html, handle, end);
	debug_exit ("url_requested");
}

static void
load_done (GtkHTML *html)
{
}

static void
on_url (GtkHTML *html, const gchar *url, gpointer data)
{
}

static void
on_set_base (GtkHTML *html, const gchar *url, gpointer data)
{
}  

static void
on_link_clicked (GtkHTML *html, const gchar *url, gpointer data)
{
	debug_enter ("on_link_clicked");

	if (!strncmp (url, "message-id:", 11))
	{
		const gchar * message_id = url + 11;
		articlelist_view_message_id (message_id);
	}
	else
	{
		pan_url_show (url);
	}
	debug_exit ("on_link_clicked");
}

static void
on_redirect (GtkHTML *html, const gchar *url, int delay, gpointer data)
{
}

static gboolean
object_requested_cmd (GtkHTML *html, GtkHTMLEmbedded *eb, void *data)
{
	return TRUE;
}


#else /* PLAIN TEXT */


/**
***
**/

typedef struct
{
	const Article * a;
	GString * str;
}
SetTextFromArticleIdleStruct;

static gint
set_text_from_article_idle (gpointer data)
{
	Group * g;
	SetTextFromArticleIdleStruct * s = (SetTextFromArticleIdleStruct*) data;
	debug_enter ("set_text_from_article_idle");

	/* update the gui */
	pan_lock_unconditional ();
	gui_headers_set_default (header_table, s->a, FALSE);
	update_body_pane_nolock (Pan.text, s->str->str, text_get_mute_quoted());
	pan_unlock_unconditional ();

	/* undo the reference cound added in set_text_from_article */
	g = s->a->group;
	pan_object_unref (PAN_OBJECT(s->a));
	group_unref_articles (g, NULL);

	/* cleanup */
	g_string_free (s->str, TRUE);
	g_free (s);
	debug_exit ("set_text_from_article_idle");
	return 0;
}
static void
set_text_from_article (const Article * a)
{
	GString * buf;
	GMimePart * mp;
	debug_enter ("set_text_from_article");

	g_assert (mm != NULL);
	g_return_if_fail (a!=NULL);

	buf = g_string_new (NULL);

	/* headers */
	if (text_get_show_all_headers()) {
		gchar * pch = g_mime_message_get_headers (mm);
		g_string_append (buf, pch);
		g_string_append_c (buf, '\n');
		g_free (pch);
	}

	/* body */
	if (1) {
		gboolean is_html = FALSE;
		gchar * pch = g_mime_message_get_body (mm, FALSE, &is_html);
		if (is_nonempty_string(pch)) {
			if (rot13_enabled)
				replace_gstr (&pch, rot13(pch));
                	if (text_get_wrap())
				replace_gstr (&pch, fill_body(pch,wrap_column));      
			g_string_append (buf, pch);
		}
		g_free (pch);
	}

	/* attachments table */
	mp = mm->mime_part;
	if (mp->children != NULL)
	{
		GList * l;
		g_string_sprintfa (buf, "\n\n");
		for (l=mp->children; l!=NULL; l=l->next)
		{
			GMimePart * tmp = (GMimePart*) l->data;
			gchar * content_type;
			const gchar * encoding_string;
			const gchar * filename;

		       	content_type = g_mime_content_type_to_string (tmp->mime_type);
		       	encoding_string = g_mime_part_encoding_to_string (tmp->encoding);
			filename = g_mime_part_get_filename (tmp);

			g_string_sprintfa (buf, _("Attachment: %s - %s - %s\n"),
				is_nonempty_string(content_type) ? content_type : _("Unknown Content Type"),
				is_nonempty_string(encoding_string) ? encoding_string : _("Unknown Encoding"),
				is_nonempty_string(filename) ? filename : _("No Filename"));

			g_free (content_type);
		}
	}

	if (1)
	{
		/* create a callback struct */
		SetTextFromArticleIdleStruct * foo = g_new (SetTextFromArticleIdleStruct, 1);
		foo->str = buf;
		foo->a = a;

		/* make sure the article's still around by the time we finish */
		pan_object_ref (PAN_OBJECT(a));
		group_ref_articles (a->group, NULL);

		/* tell the UI to refresh when it's got some free time */
		pan_lock ();
		gtk_idle_add (set_text_from_article_idle, foo);
		pan_unlock ();
	}
	debug_exit ("set_text_from_article");
}
#endif

/**
***
**/

static gint
text_set_raw_idle (gpointer data)
{
	gchar * text = (gchar*) data;
	debug_enter ("text_set_raw_idle");

	if (text == NULL)
		text = g_strdup (" ");

	pan_lock_unconditional ();
	if (1)
	{
#ifdef USE_GTKHTML
		GtkHTML * html = GTK_HTML(Pan.text);
		GtkHTMLStream * stream = gtk_html_begin (html);
		gtk_html_write (html, stream, text, strlen(text));
		gtk_html_end (html, stream, GTK_HTML_STREAM_OK);
#else
		gint position = 0;
		GtkText * text_w = GTK_TEXT(Pan.text);
		GtkEditable * editable_w = GTK_EDITABLE(Pan.text);
		gtk_text_freeze (text_w);
		gtk_text_set_point (text_w, 0);
		gtk_editable_delete_text (editable_w, 0, -1);
		if (is_nonempty_string(text))
			gtk_editable_insert_text (editable_w, text, strlen(text), &position);
		gtk_text_thaw (text_w);
#endif
		g_free (text);
	}
	pan_unlock_unconditional ();
	debug_exit ("text_set_raw_idle");
	return 0;
}
void
text_set_raw (const gchar * text)
{
	debug_enter ("text_set_raw");
	pan_lock ();
	gtk_idle_add (text_set_raw_idle, g_strdup(text));
	pan_unlock ();
	debug_exit ("text_set_raw");
}

/***
****
***/

void
text_refresh (void)
{
	debug_enter ("text_refresh");

	if (current_article != NULL)
		set_text_from_article (current_article);

	debug_exit ("text_refresh");
}

void
text_set_rot13 (gboolean on)
{
	debug_enter ("text_set_rot13");

	if (on != rot13_enabled)
	{
		rot13_enabled = on;

		if (current_article != NULL)
			text_update (current_article);
	}

	debug_exit ("text_set_rot13");
}



/* Called when the widget is first drawn, puts our welcome text into view */
static void
realize_text_cb (GtkWidget *text, gpointer data)
{
	debug_enter ("realize_text_cb");

	gtk_idle_add (text_set_raw_idle, g_strdup(welcome));
	gtk_signal_disconnect_by_func (
		GTK_OBJECT(text), GTK_SIGNAL_FUNC(realize_text_cb), NULL);

	debug_exit ("realize_text_cb");
}

/* Process a mouse button events inside the text widget */
static void
text_button_press_cb (GtkWidget *widget, GdkEventButton *bevent)
{
	debug_enter ("text_button_press_cb");

	gtk_widget_grab_focus (GTK_WIDGET (Pan.text));

	if (bevent->button != 1)
		gtk_signal_emit_stop_by_name (
			GTK_OBJECT(widget), "button_press_event");
	
	if (bevent->button == 3)
	{
		/* show the menu */
		gnome_popup_menu_do_popup (text_rightclick_popup, NULL,
					   NULL, bevent, NULL);
	}

	debug_exit ("text_button_press_cb");
}


/**
***  CURRENT ARTICLE
**/

Article*
get_current_article (void)
{
	return current_article;
}

static gint
set_current_article_idle (gpointer data)
{
	Article * a = ARTICLE(data);
	Article * old_article;
	GMimeMessage * old_mm;

	debug_enter ("set_current_article_idle");

	/**
	***  Update the current article
	**/

	old_mm = mm;
	old_article = current_article;
	current_article = a;
	mm = NULL;
	if (a != NULL)
	{
		gchar * raw = article_get_message (a);
		mm = pan_g_mime_parser_construct_message (raw);
		g_free (raw);
	}
	pan_callback_call (current_article_changed, old_article, current_article);

	/**
	***  Cleanup
	**/

	if (old_mm != NULL)
	{
		g_mime_message_destroy (old_mm);
	}
	if (old_article != NULL)
	{
		Group * g = old_article->group;
		pan_object_unref (PAN_OBJECT(old_article));
		group_unref_articles (g, NULL);
	}
	if (current_article != NULL)
	{
		set_text_from_article (current_article);
	}
	else
	{
#ifndef USE_GTKHTML
		gui_headers_set_default (header_table, NULL, TRUE);
#endif
		text_set_raw (NULL);
	}

	debug_exit ("set_current_article_idle");
	return 0;
}
void
set_current_article (Article * a)
{
	debug_enter ("set_current_article");

	if (a != NULL) {
		pan_object_ref (PAN_OBJECT(a));
		group_ref_articles (a->group, NULL);
	}
	set_current_article_idle (a);

	debug_enter ("set_current_article");
}

/**
***
**/

static int
articlelist_group_changed_cb (gpointer call_obj,
                              gpointer call_arg,
                              gpointer client_data)
{
	Group * group = GROUP(call_arg);

	if (current_article!=NULL && current_article->group!=group)
		set_current_article (NULL);

	return 0;
}

static int
compare_pgchar_pparticle_msgid (const void * a, const void *b)
{
	const gchar * msgid_a = (const gchar*) a;
	const gchar * msgid_b = article_get_message_id (*(const Article**)b);
	pan_warn_if_fail (is_nonempty_string(msgid_a));
	pan_warn_if_fail (is_nonempty_string(msgid_b));
	return pan_strcmp (msgid_a, msgid_b);
}     

static int
group_articles_removed_cb (gpointer call_obj,
                           gpointer call_arg,
                           gpointer client_data)
{
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;
	debug_enter ("group_articles_removed_cb");

	/* sanity checks */
	g_return_val_if_fail (group!=NULL, 0);
	g_return_val_if_fail (removed!=NULL, 0);
	g_return_val_if_fail (removed->len>0, 0);

	/* unset the current article if we need to */
	if (current_article!=NULL)
	{
		gboolean exact_match = FALSE;

		lower_bound (article_get_message_id(current_article),
		             removed->pdata,
		             removed->len,
			     sizeof(gpointer),
			     compare_pgchar_pparticle_msgid,
			     &exact_match);

		if (exact_match)
			set_current_article (NULL);
	}

	debug_exit ("group_articles_removed_cb");
	return 0;
}
/**
***
**/

GtkWidget *
text_create (void)
{
	GtkWidget * scrolled_window;
#ifdef USE_GTKHTML
	GtkWidget * frame;
#else
	GtkStyle * style;
#endif
	GtkWidget * retval;

	ensure_header_col_inited();

#ifdef USE_GTKHTML
	Pan.text = gtk_html_new ();
	/* this kludge kills gtkhtml's GDK_space handling -- we need it for space reading */
	gtk_binding_entry_remove (gtk_binding_set_by_class(((GtkObject*)(Pan.text))->klass),
	                          GDK_space, 0);
	text_set_font (message_body_font);
#else
        Pan.text = gtk_text_new (NULL, NULL);
        gtk_text_set_editable (GTK_TEXT(Pan.text), FALSE);
        pan_widget_set_font (GTK_WIDGET (Pan.text), message_body_font);
        style = gtk_widget_get_style(GTK_WIDGET(Pan.text));
        style->text[0] = text_fg_color;
        style->base[0] = text_bg_color;                                                         
#endif

	current_article_changed = pan_callback_new ();

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER(scrolled_window), GNOME_PAD_SMALL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);

	gtk_signal_connect (GTK_OBJECT (Pan.text), "realize",
			    GTK_SIGNAL_FUNC(realize_text_cb), NULL);
	gtk_signal_connect (GTK_OBJECT (Pan.text), "button_press_event",
			    GTK_SIGNAL_FUNC(text_button_press_cb), NULL);
#ifdef USE_GTKHTML
        gtk_signal_connect (GTK_OBJECT (Pan.text), "url_requested",
                            GTK_SIGNAL_FUNC (url_requested), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "load_done",
                            GTK_SIGNAL_FUNC (load_done), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "on_url",
                            GTK_SIGNAL_FUNC (on_url), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "set_base",
                            GTK_SIGNAL_FUNC (on_set_base), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "link_clicked",
                            GTK_SIGNAL_FUNC (on_link_clicked), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "redirect",
                            GTK_SIGNAL_FUNC (on_redirect), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "object_requested",
                            GTK_SIGNAL_FUNC (object_requested_cmd), NULL); 
#endif

	gtk_container_add (GTK_CONTAINER(scrolled_window), Pan.text);
#ifdef USE_GTKHTML
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(frame), scrolled_window);
	retval = frame;
#else
	/* text_box holds the header info and the scrolled text window */
        text_box = gtk_vbox_new (FALSE, 0);
        header_table = gtk_table_new ( 0, 0, FALSE);
        gtk_box_pack_start (GTK_BOX(text_box), header_table, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX(text_box), scrolled_window, TRUE, TRUE, 0); 
	retval = text_box;
#endif
	text_rightclick_popup = gnome_popup_menu_new (text_menu_popup);

	/* watch for dying articles */
	pan_callback_add (group_get_articles_removed_callback(),
	                  group_articles_removed_cb,
	                  NULL);

	/* watch for the articlelist group changing */
	pan_callback_add (articlelist_get_group_changed_callback(),
	                  articlelist_group_changed_cb,
			  NULL);

	return retval;
}

static void
ensure_header_col_inited (void)
{
	static gboolean headers_col_inited = FALSE;

	if (!headers_col_inited)
	{
		pan_lock ();
		if (gdk_color_alloc (cmap, &headers_col))
			headers_col_inited = TRUE;
		else
			g_error (_("couldn't allocate \"headers\" color"));
		pan_unlock ();
	}
}

/**
***
**/

void
text_update (Article * article)
{
	debug_enter ("text_update");

        /* sanity checks */
        g_return_if_fail (article!=NULL);

	/* update the current article */
	set_current_article (article);
	gui_page_set (MESSAGE_PAGE, Pan.text);

	debug_exit ("text_update");
}

void
text_set_from_article (Article *article,
		       MessageWindowType type,
		       gboolean needs_window)
{
	debug_enter ("text_set_from_article");

	/* sanity checks */
	g_return_if_fail (article!=NULL);

	queue_add (TASK(task_body_new(article, needs_window, type)));

	debug_exit ("text_set_from_article");
}

gchar*
text_get_message_to_reply_to (void)
{
	gchar * body = NULL;
	gboolean has_selection;
	debug_enter ("text_get_message_to_reply_to");

	pan_lock ();
#ifdef USE_GTKHTML
	{
		GtkHTML * html = GTK_HTML(Pan.text);
		HTMLEngine * engine = html->engine;
		body = html_engine_get_selection_string (engine);
		has_selection = is_nonempty_string (body);
	}
#else
	{
		GtkEditable * editable = GTK_EDITABLE(Pan.text);
		has_selection = editable->has_selection
			&& editable->selection_end_pos!=editable->selection_start_pos;
		if (has_selection) {
			gulong a = editable->selection_start_pos;
			gulong b = editable->selection_end_pos;
			gulong start_pos = MIN (a, b);
			gulong end_pos = MAX (a, b);
			body = gtk_editable_get_chars (editable, start_pos, end_pos);
		}
	}
#endif
	pan_unlock ();
 
	/* no selection, so user is replying to whole message sans signature */
	if (mm!=NULL && !has_selection)
	{
		gboolean is_html = FALSE;
		gchar * pch = NULL;
		replace_gstr (&body, g_mime_message_get_body (mm, FALSE, &is_html));
		if (body != NULL)
			pch = pan_strstr (body, "\n-- \n");
		if (pch != NULL)
			pch[1] = '\0';
	}

	debug_exit ("text_get_message_to_reply_to");
	return body;
}

/**
***  Wrapping
**/

static gboolean _do_wrap = FALSE;

static PanCallback * _text_fill_body_changed_callback = NULL;

PanCallback*
text_get_fill_body_changed_callback (void)
{
	if (_text_fill_body_changed_callback == NULL)
		_text_fill_body_changed_callback = pan_callback_new ();

	return _text_fill_body_changed_callback;
}

void
text_set_wrap (gboolean wrap)
{
	debug_enter ("text_set_wrap");

	if (wrap != _do_wrap)
	{
		_do_wrap = wrap;

		pan_callback_call (text_get_fill_body_changed_callback(), NULL, GINT_TO_POINTER(wrap));
		text_refresh ();                            
	}

	debug_exit ("text_set_wrap");
}

gboolean
text_get_wrap (void)
{
	return _do_wrap;
}

/**
***  Show All Headers
**/

static gboolean _show_all_headers;

static PanCallback * _show_all_headers_changed_callback = NULL;

PanCallback*
text_get_show_all_headers_changed_callback (void)
{
	if (_show_all_headers_changed_callback == NULL)
		_show_all_headers_changed_callback = pan_callback_new ();

	return _show_all_headers_changed_callback;
}

void
text_set_show_all_headers (gboolean show)
{
	debug_enter ("text_set_show_all_headers");

	if (_show_all_headers != show)
	{
		_show_all_headers = show;
		pan_callback_call (text_get_show_all_headers_changed_callback(), NULL, GINT_TO_POINTER(show));
		text_refresh ();

		pan_lock ();
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(text_menu_popup[5].widget), show);
		pan_unlock ();
	}

	debug_exit ("text_set_show_all_headers");
}

gboolean
text_get_show_all_headers (void)
{
	return _show_all_headers;
}

/**
***  Mute Quoted
**/

static gboolean _mute_quoted = FALSE;

static PanCallback * _mute_quoted_changed_callback = NULL;

PanCallback*
text_get_mute_quoted_changed_callback (void)
{
	if (_mute_quoted_changed_callback == NULL)
		_mute_quoted_changed_callback = pan_callback_new ();

	return _mute_quoted_changed_callback;
}

void
text_set_mute_quoted (gboolean quoted)
{
	debug_enter ("text_set_mute_quoted");

	if (_mute_quoted != quoted)
	{
		_mute_quoted = quoted;
		pan_callback_call (text_get_mute_quoted_changed_callback(), NULL, GINT_TO_POINTER(quoted));
		text_refresh ();

		pan_lock ();
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(text_menu_popup[6].widget), quoted);
		pan_unlock ();
	}

	debug_exit ("text_set_mute_quoted");
}

gboolean
text_get_mute_quoted (void)
{
	return _mute_quoted;
}

/**
***  Font
**/

#ifdef USE_GTKHTML
static HTMLFont *
alloc_this_font_and_squeal_like_a_pig (HTMLPainter      * painter,
                                       gchar            * face,
                                       gdouble            size,
                                       GtkHTMLFontStyle   style)
{
	HTMLFont * retval = NULL;
	EFont * font;

	font  = e_font_from_gdk_name (message_body_font);
	if (font != NULL)
	{
		EFontStyle efs = E_FONT_PLAIN;
		if (style & GTK_HTML_FONT_STYLE_BOLD)   efs |= E_FONT_BOLD;
		if (style & GTK_HTML_FONT_STYLE_ITALIC) efs |= E_FONT_ITALIC;
		retval = html_font_new (font, e_font_utf8_text_width (font, efs, " ", 1));
	}

	return retval;
}
#endif

void
text_set_font (const gchar * font_name)
{
#ifndef USE_GTKHTML
	pan_widget_set_font (GTK_WIDGET(Pan.text), message_body_font);
#else
	GtkHTML * html = GTK_HTML (Pan.text);

	if (html->engine)
	{
		static struct _HTMLPainterClass * painter_class = NULL;
		static HTMLFont *(*old_alloc_font)(HTMLPainter*,gchar*,gdouble,GtkHTMLFontStyle) = NULL;
		static gboolean painter_found = FALSE;

		if (!painter_found) {
			painter_class = HTML_PAINTER_CLASS(GTK_OBJECT(html->engine->painter)->klass);
			old_alloc_font = painter_class->alloc_font;
			painter_found = TRUE;
		}

		painter_class->alloc_font = is_nonempty_string(message_body_font)
			? alloc_this_font_and_squeal_like_a_pig
			: old_alloc_font;
	}
#endif
	text_refresh ();
}

/***
****
***/

void
update_body_pane_nolock (GtkWidget    * body_widget,
                         const char   * body,
		         gboolean       mute_quotes)
{
	GtkText * text;
	gulong len;
	gchar * buf = NULL;
	debug_enter ("update_body_pane_nolock");

	/* sanity checks */
	g_return_if_fail (body_widget!=NULL);
	g_return_if_fail (GTK_IS_TEXT(body_widget));

	/* mute the quoted text, if desired */
	if (mute_quotes)
		body = buf = mute_quoted_text_in_body (body);

	/* prepare the widget */
	text = GTK_TEXT(body_widget);
	gtk_text_freeze (text);
	len = gtk_text_get_length (text);
	gtk_text_set_point (text, len);

	/* add the new body */
	if (is_nonempty_string(body))
	{
       		GString * str = g_string_new (NULL);
		GString * chunk = g_string_new (NULL);
		gint chunk_quote = -1;

		while (get_next_token_g_str (body, '\n', &body, str))
		{
			const gint is_quote = str->len && *str->str=='>' ? 1 : 0;
			g_string_append_c (str, '\n');

			if (chunk_quote==-1 || chunk_quote==is_quote)
			{
				g_string_append (chunk, str->str);
			}
			else if (chunk->len)
			{
				GdkColor * pFG = chunk_quote ? &text_quoted_color : &text_fg_color;
				gtk_text_insert (text, NULL, pFG, NULL, chunk->str, chunk->len);
				g_string_assign (chunk, str->str);
			}

			chunk_quote = is_quote;
		}

		if (chunk->len)
		{
			GdkColor * pFG = chunk_quote ? &text_quoted_color : &text_fg_color;
			gtk_text_insert (text, NULL, pFG, NULL, chunk->str, chunk->len);
		}

		g_string_free (str, TRUE);
		g_string_free (chunk, TRUE);
	}

	/* remove the old text, if any */
	if (len != 0)
		gtk_editable_delete_text (GTK_EDITABLE(body_widget), 0, len);

	/* done with text widget */
	gtk_text_thaw (text);

	/* cleanup */
	g_free (buf);
	debug_exit ("update_body_pane_nolock");
}

void
update_body_pane (GtkWidget    * body_widget,
                  const char   * body,
		  gboolean       mute_quotes)
{
	debug_enter ("update_body_pane");

	pan_lock ();
	update_body_pane_nolock (body_widget, body, mute_quotes);
	pan_unlock ();

	debug_exit ("update_body_pane");
}
