/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __SERVER_H__
#define __SERVER_H__

#include <time.h>
#include <glib.h>

#include <pan/base/pan-callback.h>
#include <pan/base/pan-object.h>
#include <pan/base/status-item.h>

#define SERVER(a) ((Server*)a)

 
typedef enum
{
	SERVER_GROUPS_SUBSCRIBED      = (1<<0),
	SERVER_GROUPS_UNSUBSCRIBED    = (1<<1),
	SERVER_GROUPS_ALL             = (SERVER_GROUPS_SUBSCRIBED|SERVER_GROUPS_UNSUBSCRIBED)
}
ServerGroupsType;

typedef struct
{
	/* Parent */
	PanObject parent;

	/*Fields*/
	guint posting              : 1;  /* is posting allowed? */
	guint gen_msgid            : 1;
	guint need_auth            : 1;  /* need authorization ? */
	guint online               : 1;  /* is this server online? */
	guint online_prompted      : 1;  /* true if user's been asked to go online for some tasks. */
	guint reserve_connection_for_bodies     : 1;
	gint8 max_connections;           /* max connections to server */
        gint16 idle_secs_before_timeout;
	gint port;	                 /* port number.  This is usually 119 for NNTP. */
	time_t last_newgroup_list_time;
	gchar * address;                 /* server net address */
	gchar * name;	                 /* pretty name, i.e. Erols */
	gchar * username;                /* auth username */
	gchar * password;                /* auth password */

	ServerGroupsType _groups_dirty;
	ServerGroupsType _groups_loaded;
	GMemChunk * _group_memchunk;
	GPtrArray * _groups;             /* don't access directly; use server_get_groups instead */
}
Server;

#define INTERNAL_SERVER_NAME "folders"

/**
***  PROTECTED
**/

void       server_constructor            (Server*, PanObjectDestructor);

void       server_destructor             (PanObject*);

/**
***  PUBLIC - Life Cycle
**/

Server*    server_new (void);

gboolean   server_has_group              (const Server*, const char* groupname);

gboolean   server_has_group_in_type      (const Server      * server,
                                          const char        * groupname,
                                          ServerGroupsType    type);

/**
***  PUBLIC - Online
**/

void       server_set_online             (Server*, gboolean);

/**
***  PUBLIC - Server Groups
**/

struct _Group;

struct _Group* server_alloc_new_group    (Server            * server);

struct _Group* server_get_named_group    (Server            * server,
                                          const gchar       * name);

void       server_add_groups             (Server            * server,
                                          struct _Group    ** groups,
                                          int                 group_qty,
                                          GPtrArray         * fillme_used,
                                          GPtrArray         * fillme_not_used);

gboolean    server_add_group             (Server * server, struct _Group * group);

void        server_remove_groups          (Server * server, struct _Group ** groups, gint qty);

void        server_destroy_groups         (Server * server, struct _Group ** groups, gint qty);

void        server_ensure_groups_loaded   (Server * server, ServerGroupsType type);

GPtrArray*  server_get_groups             (Server * server, ServerGroupsType type);

void        server_save_dirty_groups      (Server * server, StatusItem * status);

ServerGroupsType server_get_group_type    (const struct _Group * group);

void        server_set_group_type_dirty   (Server * server, ServerGroupsType type);


/**
***  Events
**/

/**
 * @call_obj: Server*
 * @call_arg: GPtrArray* of Group*
 */
PanCallback*   server_get_groups_removed_callback (void);

/**
 * @call_obj: Server*
 * @call_arg: GPtrArray* of Group*
 */
PanCallback*   server_get_groups_added_callback (void);

/**
 * @call_obj: Server*
 * @call_arg: NULL if offline, non-NULL if online
 */
PanCallback*   server_get_online_status_changed_callback (void);


#endif /* __SERVER_H__ */
