/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/pan-i18n.h>
#include <pan/base/debug.h>
#include <pan/base/file-grouplist.h>
#include <pan/base/file-headers.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/server.h>
#include <pan/base/serverlist.h>
#include <pan/base/util-file.h>

#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/server-ui.h>
#include <pan/util.h>

/**
***
***  MENU
***
**/

static int server_qty = 0;
static gchar * first_server_name = NULL;
static GnomeUIInfo * server_menu = NULL;
static GnomeUIInfo * server_menu_radiobox = NULL;
static GnomeUIInfo end = GNOMEUIINFO_END;
static GHashTable * server_to_menubutton = NULL;
static gboolean server_activate_registered = FALSE;

static void
server_selected_cb (GtkWidget  * w,
                    Server     * selected_server)
{
	Server * cur_server;
	debug_enter ("server_selected_cb");

	cur_server = serverlist_get_active_server ();
	if (selected_server!=cur_server && GTK_CHECK_MENU_ITEM(w)->active)
		serverlist_set_active_server (selected_server);

	debug_exit ("server_selected_cb");
}

static void
server_activated_cb (gpointer server, gpointer foo, gpointer bar)
{
	GnomeUIInfo * menu_info = NULL;

	if (server != NULL)
		menu_info = g_hash_table_lookup (server_to_menubutton, server);
	if (menu_info != NULL)
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(menu_info->widget), TRUE);
}

static gint
sort_servers_by_name (const void * va, const void * vb)
{
	const Server * a = *(const Server **)va;
	const Server * b = *(const Server **)vb;
	return g_strcasecmp (a->name, b->name);
}

void
server_menu_update (void)
{
	GnomeUIInfo * buf;
	GnomeUIInfo * radio_box;
	GPtrArray * servers;
	gchar * path;
	gchar * new_first_server_name;
	gint i;
	gint widget_index;
	gint sel_index;
	Server * cur_server;
	debug_enter ("server_menu_update");

	cur_server = serverlist_get_active_server ();

	/* make sure we update the menu when a server is activated */
	if (!server_activate_registered)
	{
		server_activate_registered = TRUE;

		pan_callback_add (serverlist_get_server_activated_callback(),
		                  server_activated_cb,
		                  NULL);
	}

	/* build the new menu */
	if (server_to_menubutton != NULL)
		g_hash_table_destroy (server_to_menubutton);
	server_to_menubutton = g_hash_table_new (g_direct_hash, g_direct_equal);

	/* get a list of servers, sorted alphabetically */
	servers = g_ptr_array_new ();
	serverlist_get_servers (servers);
	qsort (servers->pdata, servers->len, sizeof(gpointer), sort_servers_by_name);

	buf = g_new0 (GnomeUIInfo, servers->len+2);
	widget_index = 0;
	if (!servers->len)
	{
		GnomeUIInfo * info = buf + widget_index++;
		info->type = GNOME_APP_UI_ITEM;
		info->label = _("No Servers Defined");
		info->hint = _("No Servers Defined");
		info->moreinfo = NULL;
		info->user_data = NULL;
	}

	sel_index = -1;
	new_first_server_name = NULL;
	for (i=0; i<servers->len; ++i)
	{
		GnomeUIInfo * info = buf + widget_index;
		Server * s = SERVER(g_ptr_array_index(servers,i));

		/* skip 'invisible' servers used for folders, etc. */
		if (!strncmp (s->name, INTERNAL_SERVER_NAME, strlen(INTERNAL_SERVER_NAME)))
			continue;

		g_hash_table_insert (server_to_menubutton, s, info);
		info->type = GNOME_APP_UI_ITEM;
		info->label = s->name;
		info->hint = s->name;
		info->moreinfo = server_selected_cb;
		info->user_data = s;
		if (new_first_server_name == NULL)
			new_first_server_name = g_strdup (s->name);
		if (s == cur_server)
			sel_index = widget_index;

		++widget_index;
	}
	buf[widget_index] = end;

	radio_box = g_new0 (GnomeUIInfo, 2);
	radio_box[0] = radio_box[1] = end;
	radio_box[0].type = GNOME_APP_UI_RADIOITEMS;
	radio_box[0].label = radio_box[0].hint = _("Servers");
	radio_box[0].moreinfo = buf;

	/* remove the old menus */
	if (is_nonempty_string(first_server_name)) {
		gchar* pch = g_strdup_printf ("Servers/%s", first_server_name);
		gnome_app_remove_menus (GNOME_APP(Pan.window), pch, server_qty);
		g_free (pch);
	}

	/* insert the menus */
	path = g_strdup_printf ("%s/", _("_Servers"));
	gnome_app_insert_menus (GNOME_APP(Pan.window), path, radio_box);
	g_free (path);

	/* select the current server, if any */
	if (sel_index != -1) {
		GtkWidget * w = buf[sel_index].widget;
		gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(w), TRUE);
	}

	/* update the local varialbes */
	g_free (first_server_name);
	g_free (server_menu);
	g_free (server_menu_radiobox);
	first_server_name = new_first_server_name;
	server_menu = buf;
	server_menu_radiobox = radio_box;
	server_qty = widget_index;

	/* cleanup */
	debug_exit ("server_menu_update");
}

/**
***
***  ONLINE PROMPT
***
**/

static void
online_prompt_cb (gint reply,
                  gpointer user_data)
{
	const gboolean online = !reply;
	Server * server = SERVER(user_data);
	if (server->online != online)
	{
		server_set_online (SERVER(user_data), online);
		server->online_prompted = FALSE;
	}
}

void
server_prompt_to_go_online (Server* server)
{
	g_return_if_fail (server!=NULL);

	if (!server->online && !server->online_prompted)
	{
		gchar* question;

		server->online_prompted = TRUE;

		question = g_strdup_printf (_("Go Online for Server \"%s\"?"),
		                            server->name);

		pan_lock ();
		gnome_question_dialog_parented (question,
		                                online_prompt_cb,
		                                server,
		                                GTK_WINDOW(Pan.window));
		pan_unlock ();

		g_free (question);
	}
}
