/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbas.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/nntp.h>
#include <pan/task-headers.h>
#include <pan/rules/rule-manager.h>

/*********************
**********************  Defines / Enumerated types
*********************/

#define SAMPLE_SIZE_DEFAULT 150

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static gint task_headers_run (Task* item);

static gchar* task_headers_describe (const StatusItem *item);

static gint task_get_header_range (TaskHeaders   * task,
                                   gulong        * setme_lo_article_in_group,
                                   gulong        * setme_hi_article_in_group,
                                   gulong        * setme_lo_article_to_fetch,
                                   gulong        * setme_hi_article_to_fetch,
                                   gulong        * setme_total_in_group,
                                   const char   ** setme_progress_fmt);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******  TASK LIFE CYCLE
*****/

static void
task_headers_destructor (PanObject* o)
{
	TaskHeaders * task_h = TASK_HEADERS(o);

	/* destruct this class: cleanup articles... */
	g_ptr_array_free (task_h->headers_needing_bodies, TRUE);
	task_h->headers_needing_bodies = NULL;

	/* destruct parent class... */
	task_destructor (o);
}

static PanObject*
real_task_headers_new (Group                * group,
		       HeaderDownloadType     dl_type,
		       gboolean               download_bodies,
                       int                    sample_size)
{
	TaskHeaders * item;
	debug_enter ("real_task_headers_new");

	g_return_val_if_fail (group_is_valid(group), NULL);
	g_return_val_if_fail (!group_is_folder(group), NULL);
	g_return_val_if_fail (group->server!=NULL, NULL);
	g_return_val_if_fail (dl_type==HEADERS_ALL || dl_type==HEADERS_NEW || dl_type==HEADERS_SAMPLE, NULL);

       	item = g_new0 (TaskHeaders, 1);
	debug1 (DEBUG_PAN_OBJECT, "task_headers_new: %p", item);

	task_constructor (TASK(item),
	                  task_headers_destructor,
	                  task_headers_describe,
	                  task_headers_run,
	                  group->server,
	                  TRUE, /* high priority */
	                  TRUE);
	item->group = group;
	item->download_type = dl_type;
	item->download_bodies = download_bodies;
	item->headers_needing_bodies = g_ptr_array_new ();
	item->body_index = 0;
	item->sample_size = sample_size;

	debug_exit ("real_task_headers_new");
	return PAN_OBJECT(item);
}

PanObject*
task_headers_new (Group               * group,
                  HeaderDownloadType    dl_type)
{
	return real_task_headers_new (group,
	                              dl_type,
	                              FALSE,
	                              SAMPLE_SIZE_DEFAULT);
}

PanObject*
task_headers_new_with_bodies   (Group               * group,
				HeaderDownloadType    dl_type)
{
	return real_task_headers_new (group,
	                              dl_type,
	                              TRUE,
	                              SAMPLE_SIZE_DEFAULT);
}

PanObject*
task_headers_new_sample (Group     * group,
			 guint       sample_size,
                         gboolean    download_bodies)
{
	return real_task_headers_new (group,
	                              HEADERS_SAMPLE,
	                              download_bodies,
	                              sample_size);
}

/*****
******
*****/

static gint
task_headers_run (Task* task)
{
	StatusItem * status = STATUS_ITEM(task);
	TaskHeaders * task_h = TASK_HEADERS(task);
	gulong low_in_group, high_in_group;
	gulong low_to_fetch, high_to_fetch;
	gulong total_in_group;
	const char * progress_fmt;
	gboolean reffed = FALSE;
	gboolean retval_set = FALSE;
	gint retval = TASK_SUCCESS;
	gint result;
	debug_enter ("task_headers_run");

	/* get up-to-date information about the group */
	progress_fmt = NULL;
	low_in_group = high_in_group = low_to_fetch = high_to_fetch = 0;
	result = task_get_header_range (task_h,
	                                &low_in_group,
	                                &high_in_group,
	                                &low_to_fetch,
	                                &high_to_fetch,
	                                &total_in_group,
	                                &progress_fmt);
	if (result != TASK_SUCCESS) {
		retval_set = TRUE;
		retval = result;
	}


	/* where to start downloading? */
	if (task_h->download_type == HEADERS_NEW)
		low_to_fetch = MAX (low_to_fetch, task_h->group->article_high);

	/* no articles to fetch */
	if (!retval_set && (high_to_fetch==0 || low_to_fetch>high_to_fetch)) {
		retval_set = TRUE;
		retval = TASK_SUCCESS;
	}

	/* if articles to fetch, try fetching them */
	if (!retval_set)
	{
		GPtrArray * articles;
		gint steps;

		/* load the existing headers */
		if (!reffed) {
			group_ref_articles (task_h->group, status);
			reffed = TRUE;
		}

		/* init the status item for the header download */
		switch (task_h->download_type) {
			case HEADERS_ALL:
			case HEADERS_NEW:
				steps = high_to_fetch - low_to_fetch;
				if (steps < 0)
					steps = 0;
				break;
			case HEADERS_SAMPLE:
				steps = task_h->sample_size;
				break;
			default:
				steps = 0;
				pan_warn_if_reached();
		}
		status_item_emit_init_steps (status, steps);

		/* get the article headers... */
		articles = g_ptr_array_new ();
		result = nntp_download_headers (status,
		                                task->sock,
		                                task_h->group,
		                                low_to_fetch,
		                                high_to_fetch,
		                                &task->hint_abort,
		                                progress_fmt,
		                                articles);
		if (result != TASK_SUCCESS) {
			retval_set = TRUE;
			retval = result;
		}

		/* if we got new articles, process them */
		if (articles->len != 0)
		{
			/* log */
			log_add_va (LOG_INFO, _("Fetched %u headers for `%s'"),
				articles->len,
				group_get_readable_name(TASK_HEADERS(task)->group));

			/* add the articles to the group */
			group_mark_new_article_number (task_h->group, task_h->group->article_high);
			group_add_articles_remove_unused (task_h->group, articles, status);
			group_expire_articles_not_in_range (task_h->group, low_in_group, high_in_group);
			group_set_article_range (task_h->group, low_in_group, high_in_group);
			group_thread_if_needed (task_h->group, status);

			/* apply the rules to the new headers */
			rule_manager_process_incoming_articles (articles);

			/* do we need to download these? */
			if (task_h->download_bodies && articles->len)
				pan_g_ptr_array_append (task_h->headers_needing_bodies,
				                        articles->pdata,
				                        articles->len);
		}

		/* cleanup */
		g_ptr_array_free (articles, TRUE);
	}

	/* Download bodies, if any */
	if (!retval_set && task_h->headers_needing_bodies->len!=0)
	{
		/* refresh the ui */
		status_item_emit_init_steps (status, task_h->headers_needing_bodies->len);
		status_item_emit_set_step (status, task_h->body_index);

		/* try downloading the bodies */
		result = nntp_download_bodies (status,
		                               task_h->group,
		                               task->sock,
		                               &task->hint_abort,
		                               task_h->headers_needing_bodies,
		                               &task_h->body_index,
					       FALSE, FALSE);
		if (result != TASK_SUCCESS) {
			retval_set = TRUE;
			retval = result;
		}
	}

	/* cleanup */
	status_item_emit_progress (status, 0);
	if (reffed)
		group_unref_articles (task_h->group, STATUS_ITEM(status));

	/* if we made it this far without fail, return success. */
	if (!retval_set)
		retval = TASK_SUCCESS;

	debug_exit ("task_headers_run");
	return retval;
}

/*****
******
*****/

static int
task_get_header_range (TaskHeaders   * task,
		       gulong        * setme_lo_article_in_group,
		       gulong        * setme_hi_article_in_group,
                       gulong        * setme_lo_article_to_fetch,
		       gulong        * setme_hi_article_to_fetch,
		       gulong        * setme_total_in_group,
		       const char   ** setme_progress_fmt)
{
	const gboolean * const abort = &TASK(task)->hint_abort;
	PanSocket *sock = TASK(task)->sock;
	Server *server = TASK(task)->server;
	Group *group = TASK_HEADERS(task)->group;
	HeaderDownloadType dl_type = TASK_HEADERS(task)->download_type;
	const char * progress_fmt;
	gulong lo_article_to_fetch;
	gulong hi_article_to_fetch;
	gulong lo_article_in_group;
	gulong hi_article_in_group;
	gulong total_in_group;
	gint result;
	debug_enter ("task_get_header_range");

	/* sanity checks */
	g_return_val_if_fail (sock!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (server!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (group_is_valid(group), TASK_FAIL_HOPELESS);

	/* change to the right group */
	lo_article_in_group = 0;
	hi_article_in_group = 0;
	total_in_group = 0;
	result = nntp_get_group_info (STATUS_ITEM(task),
	                              sock,
	                              group->name,
	                              &total_in_group,
	                              &lo_article_in_group,
	                              &hi_article_in_group,
	                              abort);
	if (result != TASK_SUCCESS)
		return result;

	/* figure out which articles to ask for, based on the first/last
	 * returned by the server and the dl_type of this task.
	 */
	if (dl_type==HEADERS_NEW)
	{
		gulong p_last = group->article_high;
		if (p_last!=0 && p_last>hi_article_in_group)
		{
			status_item_emit_status_va (STATUS_ITEM(task),
				_("No new articles in group \"%s\""),
			        group_get_readable_name(group));
		}

		lo_article_to_fetch = p_last!=0 ? p_last : lo_article_in_group;
		hi_article_to_fetch = hi_article_in_group;
		progress_fmt = _("Getting new Headers %lu of %lu");
	}
	else if (dl_type==HEADERS_SAMPLE)
	{
		const int qty = MIN (total_in_group, task->sample_size);
		status_item_emit_status_va (
			STATUS_ITEM(task), _("Sampling %d articles"), qty);
		lo_article_to_fetch = hi_article_in_group - qty;
		hi_article_to_fetch = hi_article_in_group;
		progress_fmt = _("Sampling Headers %lu of %lu");
	}
	else if (dl_type==HEADERS_ALL)
	{
		lo_article_to_fetch = lo_article_in_group;
		hi_article_to_fetch = hi_article_in_group;
		progress_fmt = _("Getting Headers %lu of %lu");
	}
	else
	{
		pan_warn_if_reached ();
		lo_article_to_fetch  = 0;
		hi_article_to_fetch = 0;
		progress_fmt = _("Error");
	}

	*setme_lo_article_in_group = lo_article_in_group;
	*setme_hi_article_in_group = hi_article_in_group;
	*setme_lo_article_to_fetch = lo_article_to_fetch;
	*setme_hi_article_to_fetch = hi_article_to_fetch;
	*setme_total_in_group      = total_in_group;
	*setme_progress_fmt        = progress_fmt;

	debug_exit ("task_get_header_range");
	return TASK_SUCCESS;
}


static gchar*
task_headers_describe (const StatusItem* item)
{
	const gchar* action = NULL;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, NULL);

	switch (TASK_HEADERS(item)->download_type)
	{
		case HEADERS_ALL: action=_("all headers"); break;
		case HEADERS_NEW: action=_("new headers"); break;
		case HEADERS_SAMPLE: action=_("sample"); break;
		default: pan_warn_if_reached(); action=_("BUG IN CODE"); break;
	}

	return g_strdup_printf ( _("Getting %s for \"%s\""),
		action,
		group_get_readable_name(TASK_HEADERS(item)->group));
}
