/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001 Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

extern void* alloca (size_t);

#include <glib.h>
#include <libgnomeui/libgnomeui.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gdk-pixbuf-loader.h>
#include <gmime/gmime.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-object.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/util-file.h>
#include <pan/base/util-mime.h>
#include <pan/base/util-wrap.h>

#include <pan/articlelist.h>
#include <pan/gui.h>
#include <pan/gui-headers.h>
#include <pan/globals.h>
#include <pan/message-window.h>
#include <pan/prefs.h>
#include <pan/queue.h>
#include <pan/task-body.h>
#include <pan/text.h>
#include <pan/util.h>

static gchar welcome[] =
{
"Pan " VERSION "\n"
"Copyright (C) 1999, 2000, 2001 \n"
"Pan Development Team (pan@rebelbase.com) \n"
"\n"
"  This is beta software.  If you find a bug, please report it.\n"
"\n"
"  http://pan.rebelbase.com/contact.html   -  Request a Feature\n"
"  http://pan.rebelbase.com/bugreport.html -  Report a Bug\n"
"  http://pan.rebelbase.com/download.html  -  Upgrade Pan\n"
"\n"
"This program is free software; you can redistribute it \n"
"and/or modify it under the terms of the GNU General Public \n"
"License as published by the Free Software Foundation; \n"
"either version 2 of the License, or (at your option) any \n"
"later version. \n"
"\n"
"This program is distributed in the hope that it will be \n"
"useful, but WITHOUT ANY WARRANTY; without even the implied \n"
"warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR \n"
"PURPOSE.  See the GNU General Public License for more \n"
"details. \n"
"\n"
"The GNU Public License can be found from the menu above \n"
"in Help|About|License. \n"
};

static GMimeMessage * current_mm = NULL;
static gboolean rot13_enabled = FALSE;
static gboolean still_on_welcome = TRUE;
static Article * current_article = NULL;
static GtkWidget * header_table = NULL;
static GtkWidget * text_vbox = NULL;
static GtkWidget * scrolled_window = NULL;
static GPtrArray * pixmaps = NULL;

GdkColor text_fg_color;
GdkColor text_bg_color;
GdkColor text_quoted_color[3];
GdkColor text_url_color;
PanCallback * current_article_changed = NULL;

static GnomeUIInfo text_pane_popup_uiinfo [];
static GtkWidget * text_pane_popup_menu;

void update_body_pane_nolock (GtkWidget*, const gchar*, gboolean);


/****
*****
*****    LOW-LEVEL TEXT UPDATE
*****
****/

typedef struct
{
	gchar * uri;

	guint start;
	guint end;
}
RemoteURI;

typedef struct
{
	GtkWidget * top;
	GtkText * text;

	GPtrArray * uris;
}
TextView;

static void
textview_clear_uris (TextView * v)
{
	guint i;

	g_return_if_fail (v!=NULL);
	g_return_if_fail (v->uris!=NULL);

	for (i=0; i<v->uris->len; ++i) {
		RemoteURI * ru = (RemoteURI*) g_ptr_array_index (v->uris, i);
		g_free (ru->uri);
		g_free (ru);
	}

	g_ptr_array_set_size (v->uris, 0);
}

static void
textview_destroy (TextView * v)
{
	g_return_if_fail (v!=NULL);

	textview_clear_uris (v);
	g_free (v);
}


static gboolean
get_uri_part (const gchar  * start,
              const gchar  * scanpos,
              const gchar ** bp,
              const gchar ** ep)
{
	const gchar * pch;

	g_return_val_if_fail (start!=NULL, FALSE);
	g_return_val_if_fail (scanpos!=NULL, FALSE);
	g_return_val_if_fail (bp!=NULL, FALSE);
	g_return_val_if_fail (ep!=NULL, FALSE);

	*bp = scanpos;

	for (pch=scanpos; *pch; ++pch)
		if (!isgraph((int)*pch) || !isascii((int)*pch) || strchr("()<>\"", *pch))
			break;

#define IS_REAL_PUNC(ch) (ispunct(ch) && ((ch!='/')))
	for (; pch-1>scanpos+1 && IS_REAL_PUNC((int)*(pch-1)); --pch)
		;

	*ep = pch;
	return TRUE;
}

static gchar*
make_uri_string (const gchar * bp,
                 const gchar * ep)
{
	return g_strndup (bp, ep-bp);
}

#define ADD_TXT_POS(bp_, ep_, pti_) \
	if ((last->next = alloca(sizeof(struct txtpos))) != NULL) { \
		last = last->next; \
		last->bp = (bp_); last->ep = (ep_); last->pti = (pti_); \
		last->next = NULL; \
	} else { \
		g_warning("alloc error scanning URIs\n"); \
		gtk_text_insert(text, font, fg_color, NULL, linebuf, -1); \
		return; \
	}

static void
make_clickable_parts (GtkText      * text,
                      GdkFont      * font,
                      GdkColor     * fg_color,
                      GdkColor     * uri_color,
                      const gchar  * linebuf,
                      GPtrArray    * uris)
{
	/* parse table - in order of priority */
	struct table
	{
		/* token */
		const gchar * needle;

		/* token search function */
		gchar *(*search) (const gchar * haystack,
		                  const gchar * needle);

		/* part parsing function */
		gboolean (*parse) (const gchar * start,
		                   const gchar * scanpos,
		                   const gchar ** bp,
		                   const gchar ** ep);

		/* part to URL function */
		gchar *(*build_uri) (const gchar * bp,
		                     const gchar * ep);
	};

	static struct table parser[] = {
		{"http://",   pan_stristr,   get_uri_part,  make_uri_string},
		{"https://",  pan_stristr,   get_uri_part,  make_uri_string},
		{"ftp://",    pan_stristr,   get_uri_part,  make_uri_string}
	};
	const gint PARSE_ELEMS = sizeof(parser) / sizeof(parser[0]);

	gint n;
	const gchar * walk;
	const gchar * bp;
	const gchar * ep;

	struct txtpos
	{
		const gchar * bp;
		const gchar * ep;
		gint pti;
		struct txtpos * next;
	}
	head = { NULL, NULL, 0, NULL}, *last = &head;

	if (uris != NULL)
	{
		for (walk=linebuf, n=0;;)
		{
			gint last_index = PARSE_ELEMS;
			gchar * scanpos = NULL;

			for (n=0; n<PARSE_ELEMS; ++n) {
				gchar * tmp = parser[n].search (walk, parser[n].needle);
				if (tmp!=NULL && (scanpos==NULL || tmp<scanpos)) {
					scanpos = tmp;
					last_index = n;
				}
			}

			if (!scanpos)
				break;

			/* check if URI can be parsed */
			if (parser[last_index].parse (linebuf, scanpos, &bp, &ep)
			    && (ep-bp-1) > strlen(parser[last_index].needle)) {
				ADD_TXT_POS (bp, ep, last_index);
				walk = ep;
			} else
				walk = scanpos + strlen(parser[last_index].needle);
		}
	}

	/* colorize this line */

	if (!head.next)
	{
		gtk_text_insert (text, font, fg_color, NULL, linebuf, -1);
	}
	else
	{
		const gchar * normal_text = linebuf;

		/* insert URIs */
		for (last = head.next;
		     last!=NULL;
		     normal_text = last->ep, last=last->next)
		{
			RemoteURI * uri = g_new0 (RemoteURI, 1);

			if (last->bp - normal_text > 0)
				gtk_text_insert (text, font, fg_color, NULL, normal_text, last->bp - normal_text);

			uri->uri = parser[last->pti].build_uri (last->bp, last->ep);

			uri->start = gtk_text_get_point (text);
			gtk_text_insert (text, font, uri_color, NULL, last->bp, last->ep - last->bp);
			uri->end = gtk_text_get_point (text);
			g_ptr_array_add (uris, uri);
		}

		if (*normal_text)
			gtk_text_insert (text, font, fg_color, NULL, normal_text, -1);
	}
}

static void
text_button_pressed (GtkWidget * text, GdkEventButton * event, TextView * view)
{
	g_return_if_fail (view!=NULL);
	g_return_if_fail (view->uris!=NULL);

	if (event==NULL)
	       return;

	if (event->button==2 || (event->button==1 && event->type==GDK_2BUTTON_PRESS))
	{
		guint i;
		guint current_pos = GTK_EDITABLE(text)->current_pos;

		for (i=0; i<view->uris->len; ++i)
		{
			const RemoteURI * uri = (const RemoteURI *) g_ptr_array_index (view->uris, i);

			if (uri->start<=current_pos && current_pos<uri->end)
				pan_url_show (uri->uri);
		}
	}
	else if (event->button==3)
	{
		gtk_menu_popup (GTK_MENU(text_pane_popup_menu),
		                NULL, NULL, NULL, NULL,
		                event->button, event->time);

	}
}

static GdkColor*
get_quote_color (const gchar * line)
{
	GdkColor * retval = &Pan.text->style->text[0];

	if (is_nonempty_string(line))
	{
		gint depth = 0;

		while (isspace((int)*line))
			++line;
		while (*line=='>' || isspace((int)*line)) {
			if (*line=='>')
				++depth;
			++line;
		}

		if (depth > 0)
			retval = &text_quoted_color[(depth-1)%3];
	}

	return retval;
}

void
update_body_pane_nolock (GtkWidget    * pane,
                         const char   * body,
		         gboolean       mute_quotes)
{
	gint len;
	GtkText * text;
	TextView * view;
	gchar * buf = NULL;
	debug_enter ("update_body_pane_nolock");

	/* sanity checks */
	g_return_if_fail (pane!=NULL);
	g_return_if_fail (GTK_IS_TEXT(pane));

	/* mute the quoted text, if desired */
	if (mute_quotes)
		body = buf = mute_quoted_text_in_body (body);

	/* prepare the widget */
	text = GTK_TEXT(pane);
	gtk_text_freeze (text);
	len = gtk_text_get_length (text);
	gtk_text_set_point (text, len);
	gtk_text_backward_delete (text, len);
	gtk_text_thaw (text);

	/* clear out old uris */
	view = (TextView*) gtk_object_get_data (GTK_OBJECT(pane), "view");
	if (view != NULL)
		textview_clear_uris (view);

	/* add the new body */
	gtk_text_freeze (text);
	if (is_nonempty_string(body))
	{
       		GString * str = g_string_new (NULL);

		while (get_next_token_g_str (body, '\n', &body, str))
		{
			GdkColor * fg = get_quote_color (str->str);
			g_string_append_c (str, '\n');
			make_clickable_parts (text, NULL, fg, &text_url_color, str->str, view?view->uris:NULL);
		}

		g_string_free (str, TRUE);
	}
	gtk_text_thaw (text);

	/* cleanup */
	g_free (buf);
	debug_exit ("update_body_pane_nolock");
}

void
update_body_pane (GtkWidget    * body_widget,
                  const char   * body,
		  gboolean       mute_quotes)
{
	debug_enter ("update_body_pane");

	pan_lock ();
	update_body_pane_nolock (body_widget, body, mute_quotes);
	pan_unlock ();

	debug_exit ("update_body_pane");
}


/**
***  Font
**/

static const gchar*
fontname (void)
{
	return text_use_fixed_font ? message_body_font_fixed
	                           : message_body_font;
}

void
text_set_font (void)
{
	pan_widget_set_font (GTK_WIDGET(Pan.text), fontname());
	text_refresh ();
}



/***
****
****   SPACE READING
****
***/

static void
sylpheed_textview_smooth_scroll_do (GtkAdjustment  * vadj,
                                    gfloat           old_value,
                                    gfloat           last_value,
                                    gint             step)
{
	gint i;
	gint change_value;
	gboolean up;

	if (old_value < last_value) {
		change_value = last_value - old_value;
		up = FALSE;
	} else {
		change_value = old_value - last_value;
		up = TRUE;
	}

	gdk_key_repeat_disable ();

	for (i=step; i<=change_value; i+=step)
		gtk_adjustment_set_value (vadj, old_value+(up?-i:i));
	gtk_adjustment_set_value (vadj, last_value);

	gdk_key_repeat_restore ();
}

void
text_read_more (void)
{
	GtkAdjustment * v = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window));

	/* try to go down a page */
	const gfloat val = CLAMP (v->value + v->page_size,
	                          v->lower,
	                          MAX(v->upper,v->page_size)-MIN(v->upper,v->page_size));

	/* if successful, then update the viewpane; otherwise, go to the next article */
	if ((v->upper < v->page_size) || (val == v->value))
		articlelist_read_next (FALSE, TRUE);
	else if (Pan.text->parent==scrolled_window && text_window_smooth_scrolling)
		sylpheed_textview_smooth_scroll_do (v, v->value, val,
		                                    text_window_smooth_scrolling_speed);
	else
		gtk_adjustment_set_value (v, val);
}


/****
*****
*****   SETTING THE TEXT FROM A RAW TEXT MESSAGE
*****
****/

static gint
text_set_raw_mainthread (gpointer data)
{
	gchar * text = (gchar*) data;
	debug_enter ("text_set_raw_mainthread");

	if (text == NULL)
		text = g_strdup (" ");

	pan_lock_unconditional ();
	if (GTK_IS_OBJECT(Pan.text))
	{
		gint position = 0;
		GtkText * text_w = GTK_TEXT(Pan.text);
		GtkEditable * editable_w = GTK_EDITABLE(Pan.text);
		gtk_text_freeze (text_w);
		gtk_text_set_point (text_w, 0);
		gtk_editable_delete_text (editable_w, 0, -1);
		if (is_nonempty_string(text))
			gtk_editable_insert_text (editable_w, text, strlen(text), &position);
		gtk_text_thaw (text_w);
		g_free (text);
	}
	pan_unlock_unconditional ();

	debug_exit ("text_set_raw_mainthread");
	return 0;
}

void
text_set_raw (const gchar * text)
{
	pan_lock ();
	run_in_main_thread_nolock (text_set_raw_mainthread, g_strdup(text));
	pan_unlock ();
}


/****
*****
*****   SETTING THE TEXT FROM AN ARTICLE
*****
****/


/**
 * Generates a GtkPixmap object from a given GMimePart that contains an image.
 * Used for displaying attached pictures inline.
 */
static GtkWidget*
get_gtk_pixmap_from_gmime_part (const GMimePart * part)
{
	guint len;
	const gchar * content;
	GdkPixbuf * pixbuf = NULL;
	GdkPixmap * pixmap = NULL;
	GdkBitmap * mask = NULL;
	GtkWidget * w = NULL;
	GdkPixbufLoader * l = NULL;

	/* create the loader */
	l = gdk_pixbuf_loader_new ();

	/* create the pixbuf */
	content = g_mime_part_get_content (part, &len);
	gdk_pixbuf_loader_write (l, content, len);
	pixbuf = gdk_pixbuf_loader_get_pixbuf (l);
	content = NULL;

	/* check for success */
	if (!pixbuf) {
		g_warning(_("Can't load the image."));
		return NULL;
	} 

	/* create a pixmap widget */
	gdk_pixbuf_render_pixmap_and_mask (pixbuf, &pixmap, &mask, 0);
	w = gtk_pixmap_new (pixmap, mask);
	if (pixmap != NULL)
		gtk_object_set_data_full (
			GTK_OBJECT(w), "pixmap", pixmap, (GtkDestroyNotify)gdk_pixmap_unref);
	if (mask != NULL)
		gtk_object_set_data_full (
			GTK_OBJECT(w), "mask", mask, (GtkDestroyNotify)gdk_pixmap_unref);

	gdk_pixbuf_loader_close (l);
	gtk_object_unref (GTK_OBJECT(l));

	return w;
}

/**
 * Called recursively to count the number of images attached to this mime message.
 */
static void
count_image_parts_gmpfunc (GMimePart * part, gpointer data)
{
	gint * pi = (gint*) data;
	const GMimeContentType * type = g_mime_part_get_content_type (part);
	if (g_mime_content_type_is_type (type, "image", "*"))
		++*pi;
}

/**
 * Appends all the text/plain mime parts into a GString* passed in the data arg.
 */
static void
extract_text_cb (GMimePart * part, gpointer data)
{
	GString * body = (GString*) data;

	const GMimeContentType * type = g_mime_part_get_content_type (part);
	if (g_mime_content_type_is_type (type, "text", "plain"))
	{
		guint len;
		const gchar * content;
		gchar * pch;

		/* get it as a string */
		content = g_mime_part_get_content (part, &len);
		pch = g_strndup (content, len);
		content = NULL;

		if (is_nonempty_string(pch))
		{
			if (rot13_enabled)
				replace_gstr (&pch, rot13(pch));
                	if (text_get_wrap())
				replace_gstr (&pch, fill_body(pch,wrap_column));      
		
			if (body->len)
				g_string_append (body, "\n\n");

			g_string_append (body, pch);
		}

		g_free (pch);
	}
}

/**
 * Builds the body string, attachment tables, and the pixmap objects of any
 * attached images, and places them in the text container widget.
 */
static void
set_text_from_article_nolock (const Article * article)
{
	GString * body = NULL;
	GMimePart * mp = NULL;
	debug_enter ("set_text_from_article");

	body = g_string_new (NULL);

	/* headers */
	if (current_mm!=NULL && text_get_show_all_headers()) {
		gchar * pch = g_mime_message_get_headers (current_mm);
		g_string_append (body, pch);
		g_string_append_c (body, '\n');
		g_free (pch);
	}

	/* append all the text parts into the body string */
	if (current_mm!=NULL)
		g_mime_message_foreach_part (current_mm, extract_text_cb, body);

	/* build the attachments table at the end of the body string */
	if (current_mm!=NULL)
		mp = current_mm->mime_part;
	if (mp!=NULL && mp->children!=NULL) {
		GList * l;
		g_string_sprintfa (body, "\n\n");
		for (l=mp->children; l!=NULL; l=l->next) {
			GMimePart * tmp = (GMimePart*) l->data;
			gchar * content_type;
			const gchar * filename;
			const gchar * encoding_string = NULL;

		       	content_type = g_mime_content_type_to_string (tmp->mime_type);
			if (tmp->content != NULL)
		       		encoding_string = g_mime_part_encoding_to_string (tmp->content->encoding);
			filename = g_mime_part_get_filename (tmp);

			g_string_sprintfa (body, _("Attachment: %s - %s - %s\n"),
				is_nonempty_string(content_type) ? content_type : _("Unknown Content Type"),
				is_nonempty_string(encoding_string) ? encoding_string : _("Unknown Encoding"),
				is_nonempty_string(filename) ? filename : _("No Filename"));

			g_free (content_type);
		}
	}

	/* update the widgets */
	if (1) {
		gint pix_qty = 0;
		gboolean pix_before;
		gboolean need_child_change;
		GMimePart * mp = NULL;

		/* Decide if there's been a change between this time, and last time,
		 * in whether or not we need to show pictures. */
		mp = current_mm!=NULL ? current_mm->mime_part : NULL;
		pix_before = pixmaps!=NULL && pixmaps->len;
		if (current_mm != NULL)
			g_mime_message_foreach_part (current_mm, count_image_parts_gmpfunc, &pix_qty);
		need_child_change = !!pix_before != !!pix_qty;

		/* Remove old Pixmaps, if any */
		if (pix_before) {
			gint i;
			for (i=0; i<pixmaps->len; ++i) {
				GtkWidget * w = GTK_WIDGET(g_ptr_array_index(pixmaps,i));
				gtk_pixmap_set (GTK_PIXMAP(w), NULL, NULL);
				gtk_container_remove (GTK_CONTAINER(text_vbox), w);
			}
			g_ptr_array_set_size (pixmaps, 0);
		}

		/* Update the text widget.
		 * If we're showing pictures thie time around then there are no scrollbars around
		 * the text widget, so we set it to full size to ensure that all the text is shown.
		 * Otherwise the widget will have scrollbars, so clear out the usize so that the
		 * viewpane can manage the widget's size.
		 */
		gui_headers_set_default_nolock (header_table, article);
		update_body_pane_nolock (Pan.text, body->str, text_get_mute_quoted());
		if (pix_qty > 0)
			gtk_widget_set_usize (Pan.text, -1, GTK_TEXT(Pan.text)->vadj->upper + 5);
		else
			gtk_widget_set_usize (Pan.text, -1, -1);

		/* Change the scrolled window's child if needed. */
		if (need_child_change)
		{
			gboolean text_need_realize = FALSE;

			/* remove the text widget to reparent it.  We unrealize it first to
			 * avoid gtktext warnings. */
			if (Pan.text->parent != NULL) {
				text_need_realize = TRUE;
				gtk_widget_unrealize (Pan.text);
				gtk_container_remove (GTK_CONTAINER(Pan.text->parent), Pan.text);
			}

			/* reparent the text vbox */
			if (text_vbox->parent != NULL)
				gtk_container_remove (GTK_CONTAINER(text_vbox->parent), text_vbox);

			/* remove all the children from the scrolled window */
			if (GTK_BIN(scrolled_window)->child != NULL)
				gtk_container_remove (GTK_CONTAINER(scrolled_window),
						      GTK_BIN(scrolled_window)->child);

			/* add the children to the scrolled window */
			if (pix_qty <= 0)
				gtk_container_add (GTK_CONTAINER(scrolled_window), Pan.text);
			else {
				gtk_box_pack_start (GTK_BOX(text_vbox), Pan.text, TRUE, TRUE, 0);
				gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(scrolled_window), text_vbox);
			}

			if (text_need_realize)
				gtk_widget_realize (Pan.text);

			gtk_widget_show_all (scrolled_window);
		}

		/* Add Images if Necessary */
		if (pix_qty > 0)
		{
			GList * l;
			for (l=mp->children; l!=NULL; l=l->next)
			{
				GMimePart * part = (GMimePart*) l->data;
				const GMimeContentType * type = g_mime_part_get_content_type (part);
				if (g_mime_content_type_is_type (type, "image", "*"))
				{
					GtkWidget * w = get_gtk_pixmap_from_gmime_part (part);
					if (w != NULL) {
						gtk_container_add (GTK_CONTAINER(text_vbox), w);
						if (pixmaps == NULL)
							pixmaps = g_ptr_array_new ();
						g_ptr_array_add (pixmaps, w);
					}
				}
			}
			gtk_widget_show_all (text_vbox);
			gtk_adjustment_set_value (
				gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW(scrolled_window)), 0.0);
		}
	}

	/* cleanup */
	g_string_free (body, TRUE);

	debug_exit ("set_text_from_article_nolock");
}

/****
*****
*****  CURRENT ARTICLE
*****
****/

Article*
get_current_article (void)
{
	return current_article;
}

static gint
set_current_article_mainthread (gpointer data)
{
	Article * a = ARTICLE(data);
	Article * old_article;
	GMimeMessage * old_mm;

	debug_enter ("set_current_article_mainthread");

	/**
	***  Update the current article
	**/

	old_mm = current_mm;
	old_article = current_article;
	current_article = a;
	current_mm = NULL;
	if (a != NULL)
	{
		gchar * raw = article_get_message (a);
		current_mm = pan_g_mime_parser_construct_message (raw);
		g_free (raw);
	}
	pan_callback_call (current_article_changed, old_article, current_article);

	if (old_mm != NULL)
	{
		g_mime_message_destroy (old_mm);
	}
	if (old_article != NULL)
	{
		group_unref_articles (old_article->group, NULL);
	}
	if (current_article != NULL)
	{
		pan_lock_unconditional ();
		set_text_from_article_nolock (current_article);
		pan_unlock_unconditional ();
	}
	else
	{
		pan_lock_unconditional ();
		gui_headers_set_default_nolock (header_table, NULL);
		set_text_from_article_nolock (NULL);
		pan_unlock_unconditional ();
	}

	debug_exit ("set_current_article_mainthread");
	return 0;
}

static void
set_current_article (Article * article)
{
	debug_enter ("set_current_article");

	still_on_welcome = FALSE;

	if (article!=NULL && article_is_valid(article))
		group_ref_articles (article->group, NULL);

	pan_lock ();
	run_in_main_thread_nolock (set_current_article_mainthread, article);
	pan_unlock ();

	debug_enter ("set_current_article");
}

void
text_clear_nolock (void)
{
	GMimeMessage * old_mm;
	Article * old_article;

	/* update model */
	old_mm = current_mm;
	old_article = current_article;
	current_article = NULL;
	current_mm = NULL;
	pan_callback_call (current_article_changed, old_article, current_article);

	/* clear out old */
	if (old_mm != NULL)
		g_mime_message_destroy (old_mm);
	if (old_article != NULL)
		group_unref_articles (old_article->group, NULL);

	/* update UI */
	gui_headers_set_default_nolock (header_table, NULL);
	set_text_from_article_nolock (NULL);
}

void
text_refresh (void)
{
	if (Pan.text != NULL) {
		if (current_article != NULL)
			text_set_from_cached_article (current_article);
		else if (still_on_welcome) {
			pan_lock ();
			update_body_pane_nolock (Pan.text, welcome, FALSE);
			pan_unlock ();
		}
	}
}

void
text_set_from_cached_article (Article * article)
{
	debug_enter ("text_set_from_cached_article");

	g_return_if_fail (article!=NULL);
	g_return_if_fail (article_has_body(article));

	set_current_article (article);
	gui_page_set (MESSAGE_PAGE, Pan.text);

	debug_exit ("text_set_from_cached_article");
}

void
text_set_from_article (Article *article)
{
	debug_enter ("text_set_from_article");

	g_return_if_fail (article!=NULL);

	if (article_has_body (article))
		text_set_from_cached_article (article);
	else
		queue_add (TASK(task_body_new(article)));

	debug_exit ("text_set_from_article");
}

/****
*****
*****    CALLBACKS
*****
****/

static void
articlelist_group_changed_cb (gpointer call_obj, gpointer call_arg, gpointer client_data)
{
	Group * group = GROUP(call_arg);

	if (current_article!=NULL && current_article->group!=group)
		set_current_article (NULL);
}

static int
compare_pgchar_pparticle_msgid (const void * a, const void *b)
{
	const gchar * msgid_a = (const gchar*) a;
	const gchar * msgid_b = article_get_message_id (*(const Article**)b);
	pan_warn_if_fail (is_nonempty_string(msgid_a));
	pan_warn_if_fail (is_nonempty_string(msgid_b));
	return pan_strcmp (msgid_a, msgid_b);
}     

static void
group_articles_removed_cb (gpointer call_obj, gpointer call_arg, gpointer client_data)
{
	Group * group = GROUP(call_obj);
	GPtrArray * removed = (GPtrArray*) call_arg;
	debug_enter ("group_articles_removed_cb");

	/* sanity checks */
	g_return_if_fail (group!=NULL);
	g_return_if_fail (removed!=NULL);
	g_return_if_fail (removed->len>0);

	/* unset the current article if we need to */
	if (current_article!=NULL)
	{
		gboolean exact_match = FALSE;

		lower_bound (article_get_message_id(current_article),
		             removed->pdata,
		             removed->len,
			     sizeof(gpointer),
			     compare_pgchar_pparticle_msgid,
			     &exact_match);

		if (exact_match)
			set_current_article (NULL);
	}

	debug_exit ("group_articles_removed_cb");
}
void
text_set_rot13 (gboolean on)
{
	debug_enter ("text_set_rot13");

	if (on != rot13_enabled)
	{
		rot13_enabled = on;

		if (current_article != NULL)
			text_set_from_cached_article (current_article);
	}

	debug_exit ("text_set_rot13");
}

/****
*****
*****    TEXT WIDGET STARTUP
*****
****/

static void
realize_text_cb (GtkWidget *text, gpointer data)
{
	debug_enter ("realize_text_cb");

	/* set the font and colors */
	if (!use_system_fg || !use_system_bg)  {
		GtkStyle * style = gtk_style_copy (gtk_widget_get_style(Pan.text));
		if (!use_system_fg)
			style->text[0] = text_fg_color;
		if (!use_system_bg)
			style->base[0] = text_bg_color;
		gtk_widget_set_style (Pan.text, style);
	}
        pan_widget_set_font (GTK_WIDGET(Pan.text), fontname());

	/* show the welcome message */
	update_body_pane_nolock (Pan.text, welcome, FALSE);

	/* this funtion only needs to be called once */
	gtk_signal_disconnect_by_func (GTK_OBJECT(text), GTK_SIGNAL_FUNC(realize_text_cb), NULL);

	debug_exit ("realize_text_cb");
}

GtkWidget *
text_create (void)
{
	TextView * view;

        Pan.text = gtk_text_new (NULL, NULL);
        gtk_text_set_editable (GTK_TEXT(Pan.text), FALSE);

	text_pane_popup_menu = gnome_popup_menu_new (text_pane_popup_uiinfo);

	current_article_changed = pan_callback_new ();

	/* the larger scrolled window */
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER(scrolled_window), 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	text_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(scrolled_window), Pan.text);

	gtk_widget_ref (text_vbox);
	gtk_widget_ref (Pan.text);

	gtk_signal_connect (GTK_OBJECT (Pan.text), "realize",
			    GTK_SIGNAL_FUNC(realize_text_cb), NULL);

	/* text_box holds the header info and the scrolled text window */
        text_box = gtk_vbox_new (FALSE, 4);
	gtk_container_set_border_width (GTK_CONTAINER(text_box), 4);
        header_table = gtk_table_new ( 0, 0, FALSE);
        gtk_box_pack_start (GTK_BOX(text_box), header_table, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX(text_box), scrolled_window, TRUE, TRUE, 0); 

	pan_callback_add (group_get_articles_removed_callback(),
	                  group_articles_removed_cb,
	                  NULL);
	pan_callback_add (articlelist_get_group_changed_callback(),
	                  articlelist_group_changed_cb,
			  NULL);

	/* create view object */
	view = g_new0 (TextView, 1);
	view->uris = g_ptr_array_new ();
	view->text = GTK_TEXT(Pan.text);
	view->top = text_box;
	gtk_object_set_data_full (GTK_OBJECT(Pan.text), "view", view, (GtkDestroyNotify)textview_destroy);
	gtk_signal_connect(GTK_OBJECT(view->text), "button_press_event",
	                   GTK_SIGNAL_FUNC(text_button_pressed), view);

	return view->top;
}

/****
*****
*****    UTILITY FUNCTIONS FOR OUTSIDE CLIENTS
*****
****/

gchar*
text_get_message_to_reply_to (void)
{
	gchar * body = NULL;
	gboolean has_selection;
	debug_enter ("text_get_message_to_reply_to");

	pan_lock ();
	{
		GtkEditable * editable = GTK_EDITABLE(Pan.text);
		has_selection = editable->has_selection
			&& editable->selection_end_pos!=editable->selection_start_pos;
		if (has_selection) {
			gulong a = editable->selection_start_pos;
			gulong b = editable->selection_end_pos;
			gulong start_pos = MIN (a, b);
			gulong end_pos = MAX (a, b);
			body = gtk_editable_get_chars (editable, start_pos, end_pos);
		}
	}
	pan_unlock ();
 
	/* no selection, so user is replying to whole message sans signature */
	if (current_mm!=NULL && !has_selection)
	{
		gboolean is_html = FALSE;
		gchar * pch = NULL;
		replace_gstr (&body, g_mime_message_get_body (current_mm, FALSE, &is_html));
		if (body != NULL)
			pch = pan_strstr (body, "\n-- \n");
		if (pch != NULL)
			pch[1] = '\0';
	}

	debug_exit ("text_get_message_to_reply_to");
	return body;
}

/****
*****
*****    MANIPULATORS:  WRAPPING
*****
****/

static gboolean _do_wrap = FALSE;

static PanCallback * _text_fill_body_changed_callback = NULL;

PanCallback*
text_get_fill_body_changed_callback (void)
{
	if (_text_fill_body_changed_callback == NULL)
		_text_fill_body_changed_callback = pan_callback_new ();

	return _text_fill_body_changed_callback;
}

void
text_set_wrap (gboolean wrap)
{
	debug_enter ("text_set_wrap");

	if (wrap != _do_wrap)
	{
		_do_wrap = wrap;

		pan_callback_call (text_get_fill_body_changed_callback(), NULL, GINT_TO_POINTER(wrap));
		text_refresh ();                            
	}

	debug_exit ("text_set_wrap");
}

gboolean
text_get_wrap (void)
{
	return _do_wrap;
}

/****
*****
*****    MANIPULATORS:  HEADERS
*****
****/

static gboolean _show_all_headers;

static PanCallback * _show_all_headers_changed_callback = NULL;

PanCallback*
text_get_show_all_headers_changed_callback (void)
{
	if (_show_all_headers_changed_callback == NULL)
		_show_all_headers_changed_callback = pan_callback_new ();

	return _show_all_headers_changed_callback;
}

void
text_set_show_all_headers (gboolean show)
{
	debug_enter ("text_set_show_all_headers");

	if (_show_all_headers != show)
	{
		_show_all_headers = show;
		pan_callback_call (text_get_show_all_headers_changed_callback(), NULL, GINT_TO_POINTER(show));
		text_refresh ();
	}

	debug_exit ("text_set_show_all_headers");
}

gboolean
text_get_show_all_headers (void)
{
	return _show_all_headers;
}

/****
*****
*****    MANIPULATORS:  MUTE QUOTED
*****
****/

static gboolean _mute_quoted = FALSE;

static PanCallback * _mute_quoted_changed_callback = NULL;

PanCallback*
text_get_mute_quoted_changed_callback (void)
{
	if (_mute_quoted_changed_callback == NULL)
		_mute_quoted_changed_callback = pan_callback_new ();

	return _mute_quoted_changed_callback;
}

void
text_set_mute_quoted (gboolean quoted)
{
	debug_enter ("text_set_mute_quoted");

	if (_mute_quoted != quoted)
	{
		_mute_quoted = quoted;
		pan_callback_call (text_get_mute_quoted_changed_callback(), NULL, GINT_TO_POINTER(quoted));
		text_refresh ();
	}

	debug_exit ("text_set_mute_quoted");
}

gboolean
text_get_mute_quoted (void)
{
	return _mute_quoted;
}

/**/

void
text_select_all (void)
{
	debug_enter ("text_select_all");

	pan_lock();
	gtk_editable_select_region (GTK_EDITABLE(Pan.text), 0, -1);
	pan_unlock();

	debug_exit ("text_select_all");
}

#define GNOMEUIINFO_ITEM_STOCK_ACCEL(label, tooltip, callback, stock_id, accel, modifier) \
        { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)callback, NULL, NULL, \
		GNOME_APP_PIXMAP_STOCK, stock_id, accel, (GdkModifierType)modifier,  NULL }


static GnomeUIInfo text_pane_popup_uiinfo [] =
{
	GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Follow-Up to newsgroup"),
	                              N_("Post a reply to the message on the news server."),
	                              message_followup_window,
	                              GNOME_STOCK_MENU_MAIL_RPL, 'F', 0),
        GNOMEUIINFO_ITEM_STOCK_ACCEL (N_("_Reply by E-mail"),
	                              N_("Create a mail reply to the sender."),
	                              message_reply_window,
	                              GNOME_STOCK_MENU_MAIL_RPL, 'R', 0),
	GNOMEUIINFO_ITEM_STOCK       (N_("Follow-Up _and Reply"),
	                              N_("Send a reply both to the author in mail, and to the news server."),
	                              message_followup_reply_window,
	                              GNOME_STOCK_MENU_MAIL_RPL),
	GNOMEUIINFO_ITEM_NONE        (N_("For_ward article by E-mail"),
	                              N_("Forward article by E-mail"),
	                              message_forward_window)
};

