/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/globals.h>
#include <pan/gui-headers.h>
#include <pan/prefs.h>
#include <pan/util.h>

gulong header_flags = ~0;
const gint max_header_width = 60;
extern GtkTooltips * ttips;


/*****
******
*****/

static void
showmore_cb (GtkButton *button, gpointer user_data)
{
	const gchar* showme = (const gchar*) gtk_object_get_data (GTK_OBJECT(button), "text");
	gnome_ok_dialog_parented (showme, GTK_WINDOW(Pan.window));
}

static gint
find_breakpoint (const gchar  * str,
                 gchar          delimiter,
                 gint           max_width)
{
	const gint len = strlen(str);
	gint retval;
	debug_enter ("find_breakpoint");

	if (!delimiter || len<=max_width)
	{
		retval = -1;
	}
	else
	{
		const gchar * pch = str;
		const gchar * last_match = NULL;
		for (; *pch; ++pch)
		{
			if (*pch == delimiter)
				last_match = pch;
			if (pch-str > max_width)
				break;
		}

		retval = last_match ? last_match-str : max_width;
	}

	debug_exit ("find_breakpoint");
	return retval;
}

static void
add_header_nolock (GtkWidget        * table,
	           gint               row,
	           const gchar      * str1,
	           const gchar      * str2,
	           gchar              delimiter,
                   gint               max_width)
{
	GtkTable * t;
	GtkWidget * w1;
	GtkWidget * w2 = NULL;
	gint breakpoint;
	debug_enter ("add_header_nolock");
       
       	t = GTK_TABLE(table);
       	w1 = gtk_label_new (str1);

	breakpoint = find_breakpoint (str2, delimiter, max_width);
	if (breakpoint != -1)
	{
		GtkWidget * hbox = gtk_hbox_new (FALSE, 0);
		GtkWidget * l = NULL;
		GtkWidget * l2 = gtk_label_new (_("<More>"));
		gchar * full = NULL;
		gchar * pch = NULL;

		/* create the truncated label */
		pch = g_strdup(str2);
		pch[breakpoint] = '\0';
		l = gtk_label_new (pch);
		g_free (pch);

		/* add them both to the button */
		gtk_box_pack_start(GTK_BOX(hbox), l, FALSE, FALSE, 0);
		gtk_box_pack_end(GTK_BOX(hbox), l2, FALSE, FALSE, 0);

		/* break the line into easier-to-read pieces
		   for the popup dialog */
		pch = full = g_strdup(str2);
		for (;;) {
			int index = find_breakpoint (pch, delimiter, max_width);
			if (index == -1) break;
			pch[index] = '\n';
			pch += (index + 1);
		}

		/* create a button */
		w2 = gtk_button_new ( );
		gtk_button_set_relief (GTK_BUTTON(w2), GTK_RELIEF_NONE);
		gtk_container_add (GTK_CONTAINER(w2), hbox);
		gtk_widget_show (l);
		gtk_widget_show (l2);
		gtk_widget_show (hbox);
		gtk_object_set_data_full (
			GTK_OBJECT(w2), "text", full, g_free);
		gtk_signal_connect (
			GTK_OBJECT(w2), "clicked", showmore_cb, NULL);
		gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w2, str2, NULL);
	}
	else
	{
		/* string is short; show it all */
		w2 = gtk_label_new  (str2);
		gtk_misc_set_alignment (GTK_MISC(w2), 0.0, 0.5);
	}

	gtk_misc_set_alignment (GTK_MISC(w1), 1.0, 0.5);

	gtk_table_resize (t, row+1,2);

	gtk_table_attach (t, w1, 0,1, row, row+1, GTK_FILL, 0,2,1 );
	gtk_table_attach (t, w2, 1,2, row, row+1, GTK_EXPAND|GTK_FILL, 0,2,1 );
	gtk_widget_show (w1);
	gtk_widget_show (w2);

	debug_exit ("add_header_nolock");
}

void
gui_headers_set_nolock (GtkWidget         * table,
		        const Article     * article,
		        gulong              header_fields)
{
	const gchar * newsgroups;
	const gchar * pch;
	gint row = 0;
	debug_enter ("gui_headers_set");

	g_return_if_fail (table != NULL);

	/* remove children */
	gtk_container_foreach (
		GTK_CONTAINER (table),
		GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);

	/* if no article, that's all we need to do */
	if (article == NULL)
		return;

	/* if author turned on, show it unconditionally */
	if (header_fields & UI_HEADER_AUTHOR)
	{
		gchar * author = article_get_author_str (article);
		add_header_nolock (table, row++, _("From: "), author, ' ', max_header_width);
		g_free (author);
	}

	/* if newsgroups turned on, and this is a crosspost, show it */
	newsgroups = pch = article_get_header (article, HEADER_NEWSGROUPS);
	if ((header_fields & UI_HEADER_NEWSGROUPS)
		&& is_nonempty_string (pch)
		&& strchr (pch, ',')!=NULL)
	{
		add_header_nolock (table, row++, _("Newsgroups: "), pch, ',', max_header_width);
	}

	/* if message-id turned on, show it unconditionally */
	if (header_fields & UI_HEADER_MESSAGE_ID)
	{
		const gchar * s = article_get_message_id (article);
		add_header_nolock (table, row++, _("Message-Id: "), s, '\0', max_header_width);
	}

	/* if references turned on, and references exist, show them */
	if (header_fields & UI_HEADER_REFERENCES)
	{
		const gchar * s = article_get_header (article, HEADER_REFERENCES);
		if (is_nonempty_string (s))
			add_header_nolock (table, row++, _("References: "), s, ' ', max_header_width);
	}

	/* if date turned on, show it unconditionally. */
	if (header_fields & UI_HEADER_DATE)
	{
		gchar buf[256];
		get_date_display_string (article->date, body_date_format, buf, sizeof(buf));
		add_header_nolock (table, row++, _("Date: "), buf, '\0', max_header_width);
	}

	/* if reply_to turned on, and reply_to exists and != author, show it */
	pch = article_get_header (article, HEADER_REPLY_TO);
	if ((header_fields & UI_HEADER_REPLY_TO)
		&& is_nonempty_string (pch)
		&& pan_strcmp (pch, article->author_addr))
	{
		add_header_nolock (table, row++, _("Reply-To: "), pch, '\0', max_header_width);
	}

	/* if followups turned on, followup_to exists and != newsgroups, show */
	pch = article_get_header (article, HEADER_FOLLOWUP_TO);
	if ((header_fields & UI_HEADER_FOLLOWUP_TO)
		&& is_nonempty_string (pch)
		&& pan_strcmp (pch, newsgroups))
	{
		add_header_nolock (table, row++, _("Followup-To: "), pch, ',', max_header_width);
	}

	if (header_fields & UI_HEADER_SUBJECT)
	{
		add_header_nolock (table, row++, _("Subject: "), article_get_subject(article), ' ', max_header_width);
	}

	debug_exit ("gui_headers_set");
}

void
gui_headers_set_default_nolock (GtkWidget      * table,
			        const Article  * article)
{
	gui_headers_set_nolock (table, article, header_flags);
}
