/*
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <glib.h>

#include <gnome-xml/parser.h>
#include <gnome-xml/xmlmemory.h>

#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

#include <pan/identities/identity-xml.h>

/*****
******
******    UTIL
******
*****/

static void
newline_depth (GString * appendme, gint depth)
{
	gint i;
	g_string_append_c (appendme, '\n'); 
	for (i=0; i<depth; ++i)
		g_string_append_c (appendme, '\t');
}

static void
pan_g_string_append_escaped (GString * gstr, gchar * escapeme)
{
	gchar * pch = pan_str_escape (escapeme);
	g_string_append (gstr, pch);
	g_free (pch);
}


/*****
******
******    WRITING
******
*****/

static void
write_identity (GString        * gstr, 
                const Identity * id,
                gint             depth)
{
	gint i;

	g_return_if_fail (gstr!=NULL);
	g_return_if_fail (id!=NULL);

	/* top level */
	newline_depth (gstr, depth);
	g_string_append (gstr, "<profile name=\"");
	pan_g_string_append_escaped (gstr, id->name);
	g_string_append (gstr, "\">");
	depth++;

	/* author name */
	if (is_nonempty_string(id->author_real))
	{
		newline_depth (gstr, depth);
		g_string_append (gstr, "<author_real>");
		pan_g_string_append_escaped (gstr, id->author_real);
		g_string_append (gstr, "</author_real>");
	}

	/* author address */
	if (is_nonempty_string(id->author_addr))
	{
		newline_depth (gstr, depth);
		g_string_append (gstr, "<author_addr>");
		pan_g_string_append_escaped (gstr, id->author_addr);
		g_string_append (gstr, "</author_addr>");
	}

	/* signature */
	if (is_nonempty_string(id->signature))
	{
		newline_depth (gstr, depth);
		g_string_append (gstr, "<signature>");
		pan_g_string_append_escaped (gstr, id->signature);
		g_string_append (gstr, "</signature>");
	}

	/* organization */
	if (is_nonempty_string(id->organization))
	{
		newline_depth (gstr, depth);
		g_string_append (gstr, "<organization>");
		pan_g_string_append_escaped (gstr, id->organization);
		g_string_append (gstr, "</organization>");
	}

	/* reply_to */
	if (is_nonempty_string(id->reply_to))
	{
		newline_depth (gstr, depth);
		g_string_append (gstr, "<reply_to>");
		pan_g_string_append_escaped (gstr, id->reply_to);
		g_string_append (gstr, "</reply_to>");
	}

	/* attribution */
	if (is_nonempty_string(id->attribution))
	{
		newline_depth (gstr, depth);
		g_string_append (gstr, "<attribution>");
		pan_g_string_append_escaped (gstr, id->attribution);
		g_string_append (gstr, "</attribution>");
	}

	/* custom headers */
	if (id->custom_headers->len > 0)
	{
		newline_depth (gstr, depth++);
		g_string_append (gstr, "<custom_headers>");

		for (i=0; i<id->custom_headers->len; i++)
		{
			Header * header;

			header = g_ptr_array_index (id->custom_headers, i);

			newline_depth (gstr, depth+1);
			g_string_append (gstr, "<custom_header name=\"");
			pan_g_string_append_escaped (gstr, header->name);
			g_string_append (gstr, "\">");
			pan_g_string_append_escaped (gstr, header->value);
			g_string_append (gstr, "</custom_header>");
		}

		newline_depth (gstr, --depth);
		g_string_append (gstr, "</custom_headers>");
	}
	
	/* trailer */
	newline_depth (gstr, --depth);
	g_string_append (gstr, "</profile>");
}

char *
identity_xml_to_string (Identity    ** identities,
                        gint           identities_qty,
                        const gchar  * default_news_id,
                        const gchar  * default_mail_id)
{
	gint      i;
	GString * str;
	GString * hdr;
	gchar   * retval;

	g_return_val_if_fail (identities_qty>=0, NULL);

	/* write the identities to a buf */
	str = g_string_sized_new (4096);
	for (i=0; i<identities_qty; i++)
	{
		write_identity (str, identities[i], 1);
		if (i!=identities_qty-1)
			g_string_append (str, "\n\n");
	}

	/* header */
	hdr = g_string_sized_new (4096);
	g_string_assign (hdr, 
	        "<?xml version=\"1.0\" ?>\n"
	        "<!DOCTYPE profiles SYSTEM \"profiles.dtd\">\n"
	        "<profiles");
	if (is_nonempty_string (default_news_id))
	{
		g_string_append (hdr, " default_news=\"");
		pan_g_string_append_escaped (hdr, (gchar *) default_news_id);
		g_string_append (hdr, "\"");
	}
	if (is_nonempty_string (default_mail_id))
	{
		g_string_append (hdr, " default_mail=\"");
		pan_g_string_append_escaped (hdr, (gchar *) default_mail_id);
		g_string_append (hdr, "\"");
	}
	g_string_append (hdr, ">\n");
	g_string_prepend(str, hdr->str);

	/* trailer */
	g_string_append (str, "\n</profiles>");

	/* return the string */
	retval = str->str;
	g_string_free (str, FALSE);
	g_string_free (hdr, TRUE);
	return retval;
}

static void
identity_xml_fwrite (FILE         * fp,
                     Identity    ** identities,
                     gint           identities_qty,
                     const gchar  * default_news_id,
                     const gchar  * default_mail_id)
{
        gchar * pch;

        g_return_if_fail (fp!=NULL);
        g_return_if_fail (identities!=NULL);
        g_return_if_fail (identities_qty>0);

        pch = identity_xml_to_string ((Identity**)identities, identities_qty,
		default_news_id, default_mail_id);
        fwrite (pch, sizeof(char), strlen(pch), fp);
        g_free (pch);
}

void
identity_xml_write (const gchar  * filename,
                    Identity    ** identities,
                    gint           identities_qty,
                    const gchar  * default_news_id,
                    const gchar  * default_mail_id) 
{
        /* sanity clause */
        g_return_if_fail (is_nonempty_string(filename));
        g_return_if_fail (identities_qty>=0);

        /* write the file */

        if (identities_qty == 0)
        {
                unlink (filename);
        }
        else
        {
                FILE * fp = fopen (filename, "w+");

                if (fp == NULL) {
                        log_add_va (LOG_ERROR,
                                _("Can't write profiles to `%s': %s"),
                                filename, g_strerror(errno));
                } else {
                        identity_xml_fwrite (fp, identities, identities_qty,
				default_news_id, default_mail_id);
                        fclose (fp);
                        log_add_va (LOG_INFO, _("Wrote %d profiles to '%s'"),
                                identities_qty, filename);
                }
        }
}


/*****
******
******    READING
******
*****/

static void
get_attributes (xmlDoc             * doc,
                xmlNode            * node,
                GHashTable        ** newme_hash,
                GStringChunk      ** newme_chunk)
{
	const xmlAttr * attr = NULL;
	GHashTable * hash = g_hash_table_new (g_str_hash, g_str_equal);
	GStringChunk * strchunk = g_string_chunk_new (128);

	for (attr=node->properties; attr!=NULL; attr=attr->next) {
		const gchar * name = attr->name;
		xmlChar * value = xmlNodeListGetString (doc, attr->val, FALSE);
		gchar * clear_value = pan_str_unescape (value);
		g_hash_table_insert (hash,
		                     g_string_chunk_insert(strchunk,name),
		                     g_string_chunk_insert(strchunk,clear_value));
		g_free (clear_value);
		xmlFree (value);
	}

	*newme_hash = hash;
	*newme_chunk = strchunk;
}

static Identity *
create_identity (xmlDoc *doc, xmlNode *node)
{
	Identity     * id = NULL;
	const gchar  * tag = node->name;

	xmlNode      * child;
	const gchar  * cpch;
	GHashTable   * attr = NULL;
	GStringChunk * strchunk = NULL;

	if (pan_strcmp (tag, "profile") != 0)
		return NULL;

	/* init toplevel */
	id = identity_new ();
	get_attributes (doc, node, &attr, &strchunk);

	/* populate toplevel */
	cpch = (const char *) g_hash_table_lookup (attr, "name");
	replace_gstr(&id->name, g_strdup(cpch));

	/* cleanup toplevel */
	g_hash_table_destroy (attr);
	g_string_chunk_free (strchunk);

	/* walk through children */
	for (child=node->xmlChildrenNode; child!=NULL; child=child->next)
	{

		if (!pan_strcmp (child->name, "custom_headers"))
		{
			xmlNode * hdr;

			/* walk through custom headers */
			for (hdr=child->xmlChildrenNode; hdr!=NULL; hdr=hdr->next)
			{
				if (!pan_strcmp (hdr->name, "custom_header"))
				{
					Header  * header = g_new0 (Header, 1);
					xmlChar * raw;
				 	gchar   * contents;


					/* name */
					get_attributes (doc, hdr, &attr, &strchunk);
					cpch = (const char *) g_hash_table_lookup (attr, "name");

					/* value */
					raw = xmlNodeListGetString (doc, hdr->xmlChildrenNode, FALSE);
					contents = pan_str_unescape (raw);

					replace_gstr (&header->name, g_strdup (cpch));
					replace_gstr (&header->value, g_strdup (contents));

					g_ptr_array_add (id->custom_headers, header);

					/* cleanup toplevel */
					g_hash_table_destroy (attr);
					g_string_chunk_free (strchunk);
					xmlFree (raw);
					g_free (contents);
				}
			}
		}
		else
		{
			gchar  ** fldptr = NULL;

			if (!pan_strcmp (child->name, "author_real"))
				fldptr = &id->author_real;
			else
			if (!pan_strcmp (child->name, "author_addr"))
				fldptr = &id->author_addr;
			else
			if (!pan_strcmp (child->name, "signature"))
				fldptr = &id->signature;
			else
			if (!pan_strcmp (child->name, "organization"))
				fldptr = &id->organization;
			else
			if (!pan_strcmp (child->name, "reply_to"))
				fldptr = &id->reply_to;
			else
			if (!pan_strcmp (child->name, "attribution"))
				fldptr = &id->attribution;

			if (fldptr != NULL && child->xmlChildrenNode != NULL)
			{
				xmlChar * raw;
			 	gchar * contents;

				/* retrieve contents */
				raw = xmlNodeListGetString (doc, 
					child->xmlChildrenNode, FALSE);
				contents = pan_str_unescape (raw);

				/* copy the field */
				replace_gstr (fldptr, g_strdup (contents));

				/* cleanup toplevel */
				xmlFree (raw);
				g_free (contents);
			}

		}
	}
			
	return id;
}

void
identity_xml_read (const gchar  * filename,
                   GPtrArray    * appendme_identities,
                   gchar       ** default_news_id,
                   gchar       ** default_mail_id)
{
	xmlDocPtr doc;
	xmlNodePtr cur;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string (filename));
	g_return_if_fail (appendme_identities!=NULL);
	if (!file_exists (filename)) return;

	doc = xmlParseFile (filename);
	g_return_if_fail (doc!=NULL);

	cur = xmlDocGetRootElement (doc);
	g_return_if_fail (cur!=NULL);

	if (pan_strcmp(cur->name, "profiles"))
		g_warning ("%s does not appear to be a valid datafile\n", filename);
	else
	{
		GHashTable   * attr = NULL;
		GStringChunk * strchunk = NULL;
		const gchar  * cpch;

		get_attributes (doc, cur, &attr, &strchunk);

		cpch = (const char *) g_hash_table_lookup (attr, "default_news");
		if (is_nonempty_string (cpch))
			replace_gstr (default_news_id, g_strdup (cpch));

		cpch = (const char *) g_hash_table_lookup (attr, "default_mail");
		if (is_nonempty_string (cpch))
			replace_gstr (default_mail_id, g_strdup (cpch));

		g_hash_table_destroy (attr);
		g_string_chunk_free (strchunk);

		
		cur = cur->xmlChildrenNode;

		for (; cur!=NULL; cur=cur->next)
		{
			Identity * id = create_identity (doc, cur);

			if (id != NULL)
			{
				g_ptr_array_add (appendme_identities, id);
			}
		}
	}

	xmlFreeDoc (doc);
}
