/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>

#include "acache.h"
#include "prefs.h" /* for data_dir */
#include "util.h" /* for read_file */
#include "debug.h"

static GHashTable *files_in_use = NULL;

static pthread_mutex_t fiu_lock = PTHREAD_MUTEX_INITIALIZER;

/***
****
****  PATHS & FILENAMES
****
***/

static gchar*
acache_get_path (void)
{
	return g_strdup_printf ("/%s/cache/", data_dir);
}

gchar*
acache_get_filename (const gchar * message_id)
{
	gchar * retval = NULL;

	if (is_nonempty_string(message_id))
	{
		GString * s = g_string_new (NULL);

		/* directory */
		if (1) {
			gchar * pch = acache_get_path ();
			g_string_assign (s, pch);
			g_free (pch);
		}

		/* strip out all the dangerous characters */
		if (1) {
			const gchar * cpch;
			for (cpch=message_id; *cpch!='\0'; ++cpch)
				if (strchr("%$@<>/\\", *cpch) == NULL)
					g_string_append_c (s, *cpch);
		}

		/* file suffix */
		g_string_append (s, ".msg");

		retval = s->str;
		g_string_free (s, FALSE);
	}

	if (retval != NULL)
		replace_gstr (&retval, pan_normalize_filename(retval));

	return retval;
}

/***
****
****  INIT / SHUTDOWN
****
***/

void
acache_init (void)
{
	gchar * pch;

	/* ensure the cache directory exists */
	pch = acache_get_path ();
	directory_check (pch);
	g_free (pch);

	/* init the hash table */
	files_in_use = g_hash_table_new (g_str_hash, g_str_equal);
}

static void
acache_clean_tmp (void)
{
	gchar * path = acache_get_path ();
	DIR * dir_p = opendir (path);

	/* delete the tmp files */
	if (dir_p != NULL)
	{
		GString * f = g_string_new (NULL);
		struct dirent * dirent_p = NULL;

		while ((dirent_p = readdir(dir_p))) {
			if (string_ends_with (dirent_p->d_name, ".tmp")) {
				g_string_sprintf (f, "/%s/%s", path, dirent_p->d_name);
				unlink (f->str);
			}	
		}

		g_string_free (f, TRUE);
	}

	/* cleanup */
	closedir (dir_p);
	g_free (path);
}

void
acache_shutdown (void)
{
	if (gnome_config_get_bool("/Pan/Cache/FlushOnExit"))
		acache_expire_all ();

	acache_clean_tmp ();
}

/***
****
****  CHECKIN / CHECKOUT
****
***/

void
acache_file_checkout (const gchar *m_id)
{
	gchar * filename = acache_get_filename (m_id);
	gpointer p = NULL;

	debug1 (DEBUG_ACACHE, "File Checkout for `%s'", filename);

	pthread_mutex_lock (&fiu_lock);
	if ((p = g_hash_table_lookup (files_in_use, filename)))
	{
		gint refcount = GPOINTER_TO_INT(p);
		g_hash_table_insert (files_in_use, filename, GINT_TO_POINTER(++refcount));
		debug1 (DEBUG_ACACHE, "new  value: %d", refcount);
	}
	else
	{
		g_hash_table_insert (files_in_use, g_strdup(filename), GINT_TO_POINTER(1));
		debug0 (DEBUG_ACACHE, "new value: 1");
	}
	pthread_mutex_unlock (&fiu_lock);

	g_free (filename);
}

void
acache_file_checkin (const gchar *m_id)
{
	gchar * filename = acache_get_filename (m_id);
	gpointer p = NULL;

	debug1 (DEBUG_ACACHE, "File Checkin  for `%s'", filename);

	pthread_mutex_lock (&fiu_lock);
	if ((p = g_hash_table_lookup (files_in_use, filename)))
	{
		gint refcount = GPOINTER_TO_INT(p);
		debug1 (DEBUG_ACACHE, "previous value: %d", refcount);
		refcount--;
		
		if (refcount <= 0)
		{
			gpointer key = NULL;
			gpointer value = NULL;
			g_hash_table_lookup_extended (files_in_use, filename, &key, &value);
			g_hash_table_remove (files_in_use, filename);
			g_free (key);
		}
		else
		{
			g_hash_table_insert (files_in_use, g_strdup(filename), GINT_TO_POINTER(refcount));
		}
	}
	pthread_mutex_unlock (&fiu_lock);

	/* if we've checked in over a half meg of stuff so far, expire */
	if (1) {
		static size_t checked_in = 0;
		checked_in += get_filesize (filename);
		if (checked_in > 524288) {
			acache_expire ();
			checked_in = 0;
		}
	}

	g_free (filename);
}

static gboolean
acache_file_in_use (const gchar * filename)
{
	gpointer p;
	gchar * path = acache_get_path ();
	gchar * str = g_strdup_printf ("%s/%s", path, filename);

	replace_gstr (&str, pan_normalize_filename(str));

	debug2 (DEBUG_ACACHE, "[%s][%s]", path, str);

	pthread_mutex_lock (&fiu_lock);
	p = g_hash_table_lookup (files_in_use, str);
	pthread_mutex_unlock (&fiu_lock);

	debug2 (DEBUG_ACACHE, "acache_file_in_use check: %s - %s in use", str, p ? "Is" : "Not");

	g_free (str);
	g_free (path);

	return p!=NULL;
}

/***
****
****  EXPIRE
****
***/

typedef struct
{
	char* filename;
	size_t size;
	time_t date;
}
DataFile;

static gint
pdf_ppdf_compare_by_youth (gconstpointer a, gconstpointer b)
{
	const time_t aval = (*(DataFile*)a).date;
	const time_t bval = (**(DataFile**)b).date;
	return (gint) difftime (aval, bval);
}

static gint
acache_expire_to_size (int cache_max)
{
	GStringChunk * string_chunk = g_string_chunk_new (4096);
	GMemChunk * datafile_chunk = g_mem_chunk_create (DataFile, 256, G_ALLOC_ONLY);
	GPtrArray * files = g_ptr_array_new ();
	GString * path = g_string_new (NULL);

	gchar * dirpath = NULL;
	struct dirent * dirent_p = NULL;
	DIR * dir_p = NULL;
	gint files_removed = 0;
	gint cache_size = 0;
	gint i = 0;

	/* Get a list of files sorted by size */
       	dirpath = acache_get_path ();
	dir_p = opendir (dirpath);
	while ((dirent_p = readdir(dir_p)))
	{
		struct stat stat_p;

		if (!string_ends_with(dirent_p->d_name, ".msg"))
			continue;

		if (acache_file_in_use (dirent_p->d_name))
			continue;

		g_string_sprintf (path, "/%s/%s", dirpath, dirent_p->d_name);
		if (stat(path->str, &stat_p) != 0)
		{
			debug2 (DEBUG_ACACHE,
			        "stat() failed for '%s': %s\n",
			        path->str, strerror(errno));
			continue;
		}

		if (S_ISREG(stat_p.st_mode))
		{
			gint index;
			DataFile * df;

			df = g_chunk_new (DataFile, datafile_chunk);
			df->filename = g_string_chunk_insert (string_chunk, path->str);
			df->size = stat_p.st_size;
			df->date = stat_p.st_mtime;

			cache_size += df->size;

			index = lower_bound (df,
			                     files->pdata,
			                     files->len,
			                     sizeof(gpointer),
			                     pdf_ppdf_compare_by_youth,
			                     NULL);

			pan_g_ptr_array_insert (files, df, index);
		}
	}
	closedir (dir_p);

	/* Start blowing away files */
	i = files->len;
	while (i>0 && cache_max<cache_size)
	{
		DataFile * df;

		/* delete a file */
		df = (DataFile*) g_ptr_array_index (files, i-1);
		unlink (df->filename);

		--i;
		++files_removed;
		cache_size -= df->size;
	}

	/*  cleanup */
	g_string_chunk_free (string_chunk);
	g_mem_chunk_destroy (datafile_chunk);
	g_ptr_array_free (files, TRUE);
	g_string_free (path, TRUE);
	g_free (dirpath);

	/* done */
	return files_removed;
}

gint 
acache_expire (void)
{
	gint cache_max = gnome_config_get_int_with_default("/Pan/Cache/MaxSize=1242880", NULL);
	return acache_expire_to_size (cache_max);
}

gint
acache_expire_all (void)
{
	gint retval;
	printf ("Pan is flushing article cache... ");
	retval = acache_expire_to_size(0);
	printf ("%d files erased.\n", retval);
	fflush (NULL);
	return retval;
}

/***
****
****  GET / SET
****
***/

void
acache_set_message (const gchar * message_id,
                    const gchar * message)
{
       	FILE * fp;
	gchar * filename;

	g_return_if_fail (is_nonempty_string(message));
	g_return_if_fail (is_nonempty_string(message_id));

	filename = acache_get_filename (message_id);
	fp = fopen (filename, "w+");
	if (fp != NULL)
	{
		const gchar * line = message;
		while (is_nonempty_string(line))
		{
			const gchar * eoln;

			/* find the end of line */
			eoln = strchr (line, '\n');
			if (eoln == NULL)
				eoln = line + strlen(line);
			++eoln;

			/* collapse double periods at line start: RFC977 2.4.1 */
			if (line[0]=='.' && line[1]=='.')
				++line;

			fwrite (line, sizeof(char), eoln-line, fp);
			skip_next_token (line, '\n', &line);
		}
		fclose (fp);
	}

	g_free (filename);
}

gchar*
acache_get_message (const gchar * message_id)
{
	gchar * buf = NULL;

	gchar * filename = acache_get_filename (message_id);
	if (is_nonempty_string (filename))
	{
		GArray* file = read_file (filename);
		if (file != NULL)
		{
			buf = file->data;
			g_array_free (file, FALSE);
		}
		g_free (filename);
	}

	return buf;
}

gboolean
acache_has_message (const gchar * message_id)
{
	gchar * filename = acache_get_filename (message_id);
	gboolean retval = file_exists (filename);
	g_free (filename);
	return retval;
}

void
acache_delete (const gchar * message_id)
{
	gchar * filename = acache_get_filename (message_id);
	debug1 (DEBUG_ACACHE, "Cache delete: %s", message_id);
	unlink (filename);
	g_free (filename);
}
