/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "acache.h"
#include "debug.h"
#include "decode.h"
#include "log.h"
#include "util.h"
#include "util-mime.h"

/**
*** Private Routines
**/

static gchar*
create_filename (const gchar       * path,
                 const gchar       * subject,
                 const gchar       * default_filename,
                 const GMimePart   * part)
{
	gchar * retval;
	GString * filename;

	g_return_val_if_fail (is_nonempty_string(path), NULL);
	g_return_val_if_fail (part!=NULL, NULL);

       	filename = g_string_new (NULL);

	/* first try the filename specified by the user */
	if (!filename->len && is_nonempty_string(default_filename))
		g_string_assign (filename, default_filename);

	/* otherwise try the filename specified by the article */
	if (!filename->len) {
		const gchar * pch = g_mime_part_get_filename (part);
		if (is_nonempty_string(pch))
			g_string_assign (filename, pch);
	}

	/* otherwise try the article's subject */
	if (!filename->len && is_nonempty_string(subject))
		g_string_assign (filename, subject);

	/* otherwise punt */
	if (!filename->len)
		g_string_assign (filename, _("UNKNOWN"));

	/* filter out any wacky stuff that might make fopen fail */
	if (1) {
		GString * tmp = g_string_new (NULL);
		const gchar * pch;
		for (pch=filename->str; *pch; ++pch) {
			const gchar ch = strchr("<>/\\ ",*pch)==NULL ? *pch : '_';
			g_string_append_c (tmp, ch);
		}
		g_string_assign (filename, tmp->str);
		g_string_free (tmp, TRUE);
	}

	/* add path */
	if (is_nonempty_string(path)) {
		g_string_prepend_c (filename, G_DIR_SEPARATOR);
		g_string_prepend (filename, path);
	}

	/* make sure we've got a unique file */
	if (file_exists (filename->str)) {
		const gint len = filename->len;
		int i = 1;
		for (;;) {
			g_string_truncate (filename, len);
			g_string_sprintfa (filename, ".%d", i++);
			if (!file_exists (filename->str))
				break;
		}
	}

	g_message (_("Using filename [%s]"), filename->str);
	retval = filename->str;
	g_string_free (filename, FALSE);
	return retval;
}

typedef struct
{
	gchar * text;
}
Body;

static gchar*
create_big_article (GPtrArray * articles)
{
	Body * bodies;
	gchar * full = NULL;
	gulong full_len = 0;
	gint i;

	/* sanity clause */
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len, NULL);

	/* get the articles & find out how big the buffer must be */
	full_len = 0;
	bodies = g_new (Body, articles->len);
	for (i=0; i!=articles->len; ++i)
	{
		const Article * a = ARTICLE(g_ptr_array_index(articles,i));
		Body * body = bodies+i;
		body->text = acache_get_message (a->message_id);
		full_len += body->text==NULL ? 0 : strlen(body->text);
	}

	/* make the buffer */
	if (full_len != 0)
	{
		size_t offset = 0;

		/* allocate the buffer */
		full = g_new (gchar, full_len+1);

		/* move the articles into it */
		for (i=0; i!=articles->len; ++i)
		{
			const gchar * pch = bodies[i].text;
			const gchar * tmp;

			/* article [2..n] trim the headers out */
			if (pch!=NULL && i!=0) {
				tmp = strstr (pch, "\n\n");
				if (tmp != NULL)
					pch = tmp;
				while (isspace((int)*pch))
					++pch;
			}

			/* all articles strip out leading space */
			while (isspace((int)*pch))
				++pch;

			/* all articles strip out trailing space after last linefeed */
			tmp = pch + strlen(pch) - 1;
			while (tmp>pch && isspace((int)*tmp))
				--tmp;
			if (tmp[1]=='\n')
				++tmp;

			/* do the copying */
			if (pch!=NULL) {
				const size_t len = 1 + (tmp-pch);
				memcpy (full+offset, pch, len);
				offset += len;
			}
		}

		/* make sure the big item is zero-terminated */
		full[offset] = '\0';
	}

	/* cleanup */
	for (i=0; i!=articles->len; ++i)
		g_free (bodies[i].text);
	g_free (bodies);

	return full;
}

static void
get_array_of_decodable_parts_func (GMimePart * part, gpointer data)
{
	GPtrArray * a = (GPtrArray*) data;
	const GMimeContentType * type = g_mime_part_get_content_type (part);
	if ((is_nonempty_string(g_mime_part_get_filename (part))) ||
		(!g_mime_content_type_is_type (type, "text", "*") && !g_mime_content_type_is_type (type, "multipart", "*")))
		g_ptr_array_add (a, part);
}

static void
get_decodable_parts (GPtrArray       * articles,
                     GMimeMessage   ** setme_msg,
                     GPtrArray      * fillme_array)
{
	gchar * text;

	/* sanity check */
	g_return_if_fail (articles!=NULL);

	/* get the message */
	text = create_big_article (articles);
	*setme_msg = pan_g_mime_parser_construct_message (text);
	g_mime_message_foreach_part (*setme_msg, get_array_of_decodable_parts_func, fillme_array);

	/* cleanup */
	g_free (text);
}

/**
*** Public Routines
**/

void
decode_article (const decode_data * dd)
{
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
	GPtrArray * articles;
	GMimeMessage * mm;
	GPtrArray * attachments;
	GSList * p;
	gint i = 0;
	gboolean success = TRUE;
	Article * first;

	/* sanity clause */
	g_return_if_fail (dd!=NULL);
	g_return_if_fail (dd->server!=NULL);
	g_return_if_fail (dd->articles!=NULL);
	for (p=dd->articles; p!=NULL; p=p->next) {
		Article * a = ARTICLE(p->data);	
		g_return_if_fail (a!=NULL);
		g_return_if_fail (is_nonempty_string(a->message_id));
		g_return_if_fail (acache_has_message(a->message_id));
	}

	first = ARTICLE(dd->articles->data);

	/* This is sad, really. individual GMimeMessages are safe enough;
	   the reason we use mutexes here is that decoding a big mp3 takes
	   up so _much_ memory that running two at once makes my system
	   swap to its knees. */
	status_item_emit_status_va (dd->item, _("Waiting to decode \"%s\""), first->subject);
	g_static_mutex_lock (&mutex);

	/* let the client know what we're doing */
	status_item_emit_status_va (dd->item, _("Processing \"%s\""), first->subject);

	/* make a GPtrArray of the articles */
	articles = g_ptr_array_new ();
	for (p=dd->articles; p!=NULL; p=p->next)
		g_ptr_array_add (articles, p->data);

	/* get the decodable parts */
	status_item_emit_status_va (dd->item, _("Decoding \"%s\""), first->subject);
	mm = NULL;
	attachments = g_ptr_array_new ();
	get_decodable_parts (articles, &mm, attachments);

	/* decode & save the parts */
	success = attachments->len != 0;
	for (i=0; success && i!=attachments->len; ++i)
	{
		guint len;
		const gchar * content;
		FILE * fp = NULL;
		const GMimePart * part = (const GMimePart*) g_ptr_array_index (attachments, i);
		gchar * filename = create_filename (dd->path, first->subject, dd->filename, part);

		/* make sure there's content to write */
		content = g_mime_part_get_content (part, &len);
		if (content==NULL || len==0)
			success = FALSE;

		/* try to open a file for writing */
		if (success)
		{
			gchar * path = g_dirname (filename);
			directory_check (path);
			g_free (path);
			fp = fopen (filename, "w+");
			if (fp == NULL)
				success = FALSE;
		}

		/* write the content */
		if (success) {
			fwrite (content, sizeof(char), len, fp);
			fclose (fp);
			fp = NULL;
			status_item_emit_status_va (dd->item, _("Saved \"%s\""), filename);
		}

		/* log the success */
		if (success) {
			gchar * msg = g_strdup_printf (
				_("Decoded \"%s\" from group \"%s\", \"%s\" part #%d"),
				filename,
				(dd->group ? dd->group->name : "Unknown"),
				first->subject, i+1);
			log_add (msg);
			debug0 (DEBUG_DECODE, msg);
			g_free (msg);
		}

		/* remember the filename */
		if (success && !article_get_header (first, PAN_HEADER_FILENAME))
			article_set_header (first, PAN_HEADER_FILENAME, filename, DO_CHUNK);

                /* maybe open the file */
                if (success && dd->open)
                        open_outside_file (filename);

		/* cleanup this iteration */
		g_free (filename);
	}

	/* update the node */
	article_remove_flag (first, STATE_DECODE_QUEUED);
	article_add_flag (first, (success?STATE_DECODED:STATE_DECODE_FAILED));

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	g_mime_message_destroy (mm);
	g_ptr_array_free (attachments, TRUE);
	g_static_mutex_unlock (&mutex);
}
