/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/* TODO:
 * [ ] add tool tips
 * [ ] background should be white
 * [*] re-write with a simpler 2-3 items per page design and nice big 
 *     description of exactly what information is needed and why.
 * [*] druid is modal, i.e., don't pop up Pan's window on first run, wait
 *     for druid to finish.
 */

#include <config.h>

#include <gnome.h>
#include <gdk_imlib.h>

#include "dialogs.h"
#include "globals.h"
#include "pan.h"
#include "prefs.h"
#include "util.h"

#include "xpm/pan_druid.xpm"

typedef struct
{
	GtkWidget *radio_button;
	GtkWidget *target_a;
	GtkWidget *target_b;
}
DruidChkBoxTemp;

typedef struct
{
	GtkWidget * full_name;
	GtkWidget * organization;

	GtkWidget * email;
	GtkWidget * smtp_server;
	GtkWidget * smtp_port;

	
	GtkWidget * server_address;
	GtkWidget * server_port;
	GtkWidget * auth_toggle;

	GtkWidget * server_username_label;
	GtkWidget * server_username;
	GtkWidget * server_password_label;
	GtkWidget * server_password;

	GtkWidget * profile_name;
}
Druid;


static void       druid_cancel        (GtkObject        * object);

static void       druid_finish        (GnomeDruidPage   * page,
                                       GnomeDruid       * druid,
                                       gpointer           window);

static gboolean   complex_druid_next  (GnomeDruidPage   * page,
                                       GnomeDruid       * druid,
                                       DruidChkBoxTemp  * data);

static gboolean   druid_next          (GnomeDruidPage   * page,
                                       GnomeDruid       * druid,
                                       GnomeDruidPage   * next);

static gboolean   druid_back_cb       (GnomeDruidPage   * page,
                                       GnomeDruid       * druid,
                                       gpointer           data);


/*---[ druid_finish ]-------------------------------------------------
 * 'finish' button has been clicked, grab and store all the data
 * that the user entered in.
 *--------------------------------------------------------------------*/
static void
druid_finish (GnomeDruidPage  * page,
              GnomeDruid      * druid,
              gpointer          window)
{
	GString * buf = g_string_new (NULL);
	Druid * ddata
		= gtk_object_get_data (GTK_OBJECT (window), "data");
	gint server_port
		= atoi(gtk_entry_get_text (GTK_ENTRY(ddata->server_port)));
	gint auth_toggle
		= GTK_TOGGLE_BUTTON (ddata->auth_toggle)->active;
	const gchar * profile_name
		= gtk_entry_get_text (GTK_ENTRY(ddata->profile_name));
	const gchar * server_address
		= gtk_entry_get_text (GTK_ENTRY(ddata->server_address));
	const gchar * server_username
		= gtk_entry_get_text(GTK_ENTRY(ddata->server_username));
	const gchar * server_password
		= gtk_entry_get_text(GTK_ENTRY(ddata->server_password));
	const gchar * full_name
		= gtk_entry_get_text (GTK_ENTRY(ddata->full_name));
	const gchar * email
		= gtk_entry_get_text (GTK_ENTRY(ddata->email));
	const gchar * organization
		= gtk_entry_get_text (GTK_ENTRY(ddata->organization));
	const gchar * smtp_server
		= gtk_entry_get_text (GTK_ENTRY(ddata->smtp_server));
	const gchar * smtp_port
		= gtk_entry_get_text (GTK_ENTRY(ddata->smtp_port));
						

	
	if (profile_name && server_address)
	{
		Server * server = server_new ();
                server->name = g_strdup (profile_name);
		server->address = g_strdup (server_address);
                server->port = server_port;
                server->need_auth = auth_toggle;
                server->username = auth_toggle ? g_strdup (server_username) : NULL;
		server->password = auth_toggle ? g_strdup (server_password) : NULL;
                server->gen_msgid = TRUE;
		server->max_connections = 4;
		server->idle_secs_before_timeout = 180;
		server_add_new_server (server);
	}

	/* save user stats */
	gnome_config_set_string ("/Pan/User/Full_Name", full_name);
	gnome_config_set_string ("/Pan/User/Email", email);
	gnome_config_set_string ("/Pan/User/Organization", organization);
	gnome_config_set_string ("/Pan/Mail/smtp_port", smtp_port);
	gnome_config_set_string ("/Pan/Mail/smtp_address", smtp_server);
	mail_server_address = g_strdup (smtp_server);

	
	/* start Pan proper */
	gnome_config_sync ();
	pan_init ();
	gtk_widget_show_all (Pan.window);

	/* cleanup */
	g_free (ddata);
	g_string_free (buf, TRUE);
	gtk_widget_destroy (GTK_WIDGET (window));
}


/*---[ druid_cancel ]-------------------------------------------------
 * Cancel was pressed in the druid, kill the wizard and then exit
 * Pan.
 *--------------------------------------------------------------------*/
static void
druid_cancel (GtkObject *object)
{
	gtk_widget_destroy (GTK_WIDGET (object));
	pan_shutdown();
}

static gboolean
complex_druid_next (GnomeDruidPage   * page,
                    GnomeDruid       * druid,
                    DruidChkBoxTemp  * data)
{
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->radio_button)))
	{
		gtk_object_set_data (GTK_OBJECT (data->target_a), "back", page);
		gnome_druid_set_page (druid, GNOME_DRUID_PAGE (data->target_a));
	}
	else
	{
		gtk_object_set_data (GTK_OBJECT (data->target_b), "back", page);
		gnome_druid_set_page (druid, GNOME_DRUID_PAGE (data->target_b));
	}

	return TRUE;
}

static gboolean
druid_back_cb (GnomeDruidPage *page, GnomeDruid *druid, gpointer data)
{
        GtkWidget * back_page = gtk_object_get_data (GTK_OBJECT(page), "back");
        if (back_page != NULL)
	{
                gtk_object_set_data (GTK_OBJECT (page), "back", NULL);
                gnome_druid_set_page (druid, GNOME_DRUID_PAGE (back_page));
        }

	return back_page!=NULL;
}

/*---[ druid_next ]---------------------------------------------------
 * 'next' was clicked, flip to next wizard page, and set the previous
 * page while we're at it.
 *--------------------------------------------------------------------*/
static gboolean
druid_next (GnomeDruidPage *page, GnomeDruid *druid, GnomeDruidPage *next)
{
	gtk_object_set_data (GTK_OBJECT (next), "back", page);
	gnome_druid_set_page (druid, next);
	return TRUE;
}


/*
 * Page #1:
 *  "Your Name"
 */
static GtkWidget*
druid_name_page (GtkWidget *vbox, Druid *ddata)
{
	GtkWidget *table;
	GtkWidget *label;

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("When you post a message to a newsgroup, your display name will appear in the From field.  Type your name in as you would like it to appear."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("Display Name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->full_name = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), ddata->full_name, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);
	gtk_entry_set_text (GTK_ENTRY(ddata->full_name), g_get_real_name());

	label = gtk_label_new (_("For example: John Doe"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("If you belong to an organization and would like for your organization's name to appear with your posts, type in your organization name below.  This information is optional, you may leave it blank."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 3, 4,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("Organization:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->organization = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), ddata->organization, 1, 2, 4, 5,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	return table;
}


/*
 * Page #2:
 *  "Internet News E-mail Address"
 */
static GtkWidget*
druid_email_page (Druid *ddata)
{
	GtkWidget * table;
	GtkWidget * label;
	gchar * email;
	gchar * smtp_server;
	gchar * smtp_port;
	gchar * host;
	gchar * fqdn;

	table = gtk_table_new (6, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("People can reply to your posts by sending you an e-mail message at the following address.") );
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("E-mail address:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->email = gtk_entry_new ();
	email = get_default_author_address ();
	gtk_entry_set_text (GTK_ENTRY(ddata->email), email);
	gtk_table_attach (GTK_TABLE (table), ddata->email, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("For example: user@host.com"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);


	/* Jan */
	label = gtk_label_new (_("Please enter the name of the mail server to use when sending e-mail messages."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 3, 4,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);


	label = gtk_label_new (_("SMTP Server:") );
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 4, 5,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	
	ddata->smtp_server = gtk_entry_new ();
	host = get_host_name();
	fqdn = get_fqdn (host);
	smtp_server = g_strdup_printf ("%s", fqdn);
	gtk_entry_set_text (GTK_ENTRY(ddata->smtp_server), smtp_server);
	gtk_table_attach (GTK_TABLE (table), ddata->smtp_server, 1, 2, 4, 5,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);
	
	label = gtk_label_new (_("SMTP Port:") );
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 5, 6,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->smtp_port = gtk_entry_new ();
	smtp_port = g_strdup ("25");
	gtk_entry_set_text (GTK_ENTRY(ddata->smtp_port), smtp_port);
	gtk_table_attach (GTK_TABLE (table), ddata->smtp_port, 1, 2, 5, 6,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	/* cleanup */
	/* cleanup */
	g_free (host);
	g_free (fqdn);
	g_free (email);
	g_free (smtp_port);
	g_free (smtp_server);
	return table;
}


/*
 * Page #3:
 *  "Internet News Server Name"
 */
static GtkWidget*
druid_server_page (Druid *ddata, DruidChkBoxTemp *chkbox)
{
	GtkWidget *table;
	GtkWidget *label;
	gchar *env_nntp_adr;

	table = gtk_table_new (5, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new(_("Type the name of your Internet news server."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 0, 1,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("News (NNTP) server:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_address = gtk_entry_new();
	if ((env_nntp_adr = g_getenv ("NNTPSERVER")) != NULL)
		gtk_entry_set_text (GTK_ENTRY(ddata->server_address), env_nntp_adr);
	else
		gtk_entry_set_text (GTK_ENTRY(ddata->server_address), "news");
	gtk_table_attach (GTK_TABLE (table), ddata->server_address, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("Port:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_port = gtk_entry_new();
	gtk_entry_set_text (GTK_ENTRY (ddata->server_port), "119");
	gtk_table_attach (GTK_TABLE (table), ddata->server_port, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("If you must log on to your news (NNTP) server, then select the check box below."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 3, 4,
	 		  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->auth_toggle = gtk_check_button_new_with_label (_("My server requires me to log on."));
	chkbox->radio_button = ddata->auth_toggle;
 	gtk_table_attach (GTK_TABLE (table), chkbox->radio_button, 0, 2, 4, 5,
	 		  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	return table;
}


/*
 * Page #4 (if the box is checked on #3):
 *  "Internet News Server Logon"
 */
static GtkWidget*
druid_auth_page (Druid *ddata)
{
	GtkWidget *table;
	GtkWidget *label;

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (
		_("Type your user name and password for your news server."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 0, 1,
	 		  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("Account name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
 	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
	 		  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);


	ddata->server_username = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), ddata->server_username, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("Password:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
 	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
	 		  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->server_password = gtk_entry_new ();
	gtk_entry_set_visibility (GTK_ENTRY(ddata->server_password), FALSE);
	gtk_table_attach (GTK_TABLE (table), ddata->server_password, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	return table;
}


/*
 * Page #5:
 *  "Save Profile"
 */
static GtkWidget *
druid_profile_page (Druid *ddata)
{
	GtkWidget *table = NULL;
	GtkWidget *label = NULL;

	table = gtk_table_new (4, 2, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);
	gtk_table_set_col_spacings (GTK_TABLE(table), GNOME_PAD_SMALL);

	label = gtk_label_new (_("Type in the name you would like Pan to show when referring to this profile you have just setup.  You can name the profile anything you want to."));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
 	gtk_table_attach (GTK_TABLE (table), label, 0, 2, 0, 1,
	 		  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("Profile Name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	ddata->profile_name = gtk_entry_new();
	gtk_entry_set_text (GTK_ENTRY(ddata->profile_name), "Default");
	gtk_table_attach (GTK_TABLE (table), ddata->profile_name, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	label = gtk_label_new (_("For example: UUNET Server"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD_SMALL);

	return table;
}


/*---[ druid_spawn ]--------------------------------------------------
 * spawn the first-run wizard
 *--------------------------------------------------------------------*/
void
dialog_newuser (void)
{
	GtkWidget *window;
	GtkWidget *druid;
	GtkWidget *page_start;
	GtkWidget * page_a;
	GtkWidget * page_b;
	GtkWidget * page_c;
	GtkWidget * page_d;
	GtkWidget * page_e;
	GtkWidget * page_finish;
	GdkImlibImage * logo;
	GdkImlibImage *watermark = NULL;
	gchar *fname;
	Druid *ddata;
	DruidChkBoxTemp *data;
	
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER_ALWAYS);
	gtk_window_set_title (GTK_WINDOW (window), _("Pan Setup Wizard"));

	druid = gnome_druid_new ();
	ddata = g_new (Druid, 1);

	/* create the logo iamge */
	logo = NULL;
	fname = gnome_pixmap_file ("gnome-logo-icon.png");
	if (fname != NULL)
		logo = gdk_imlib_load_image (fname);
	g_free (fname);

	watermark = gdk_imlib_create_image_from_xpm_data (pan_druid_xpm);

	gtk_object_set_data (GTK_OBJECT(window), "data", ddata);

	/* Initialize the druid pages */
	page_start = gnome_druid_page_start_new_with_vals (
		_("Welcome to Pan"),
		_("Welcome to Pan!\nSince this is your first time running Pan,\nwe'll need to gather some information.\n"),
		logo,
		watermark);
	page_a = gnome_druid_page_standard_new_with_vals (_("Your Name"), logo);
	page_b = gnome_druid_page_standard_new_with_vals (_("E-mail"), logo);
	page_c = gnome_druid_page_standard_new_with_vals (_("News Server Name"), logo);
	page_d = gnome_druid_page_standard_new_with_vals (_("News Server Logon"), logo);
	page_e = gnome_druid_page_standard_new_with_vals (_("Save Profile"), logo);
	page_finish = gnome_druid_page_finish_new_with_vals (
		_("Congratulations"),
		_("You have successfullly entered all of\nthe information required to set up Pan.\n\nTo save these settings, click Finish."),
		logo,
		watermark);

	data = g_new (DruidChkBoxTemp, 1);
	data->radio_button = gtk_check_button_new_with_label (_("My server requires me to log on."));
	data->target_a = page_d;
	data->target_b = page_e;

	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_a)->vbox),
			    druid_name_page (GNOME_DRUID_PAGE_STANDARD(page_a)->vbox, ddata), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_b)->vbox),
			    druid_email_page (ddata), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_c)->vbox),
			    druid_server_page (ddata, data), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_d)->vbox),
			    druid_auth_page (ddata), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page_e)->vbox),
			    druid_profile_page (ddata), TRUE, TRUE, 0);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (page_a), "next",
			    GTK_SIGNAL_FUNC (druid_next),
			    (gpointer) page_b);
	gtk_signal_connect (GTK_OBJECT (page_a), "back",
			    GTK_SIGNAL_FUNC (druid_back_cb), NULL);
	gtk_signal_connect (GTK_OBJECT (page_b), "next",
			    GTK_SIGNAL_FUNC (druid_next),
			    (gpointer) page_c);
	gtk_signal_connect (GTK_OBJECT (page_b), "back",
			    GTK_SIGNAL_FUNC (druid_back_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (page_c), "next",
			    GTK_SIGNAL_FUNC (complex_druid_next),
			    (gpointer) data);
	gtk_signal_connect (GTK_OBJECT (page_c), "back",
			    GTK_SIGNAL_FUNC (druid_back_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (page_d), "next",
			    GTK_SIGNAL_FUNC (druid_next),
			    (gpointer) page_e);
	gtk_signal_connect (GTK_OBJECT (page_d), "back",
			    GTK_SIGNAL_FUNC (druid_back_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (page_e), "next",
			    GTK_SIGNAL_FUNC (druid_next),
			    (gpointer) page_finish);
	gtk_signal_connect (GTK_OBJECT (page_e), "back",
			    GTK_SIGNAL_FUNC (druid_back_cb), NULL);

	gtk_signal_connect (GTK_OBJECT (page_finish), "finish",
			    GTK_SIGNAL_FUNC (druid_finish),
			    window);
	gtk_signal_connect (GTK_OBJECT (page_finish), "back",
			    GTK_SIGNAL_FUNC (druid_back_cb), NULL);


	gtk_signal_connect_object (GTK_OBJECT (druid), "cancel",
				   GTK_SIGNAL_FUNC (druid_cancel),
				   GTK_OBJECT (window));

	/* tie it all together */
	gtk_container_add (GTK_CONTAINER (window), druid);		
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_start));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_a));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_b));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_c));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_d));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_e));
	gnome_druid_append_page (GNOME_DRUID (druid),
				 GNOME_DRUID_PAGE (page_finish));

	/* set the first page */
	gnome_druid_set_page (GNOME_DRUID (druid),
			      GNOME_DRUID_PAGE (page_start));

	gtk_widget_show_all (window);
}
