/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "article.h"
#include "file-headers.h"
#include "gnksa.h"
#include "group.h"
#include "status-item.h"
#include "prefs.h"
#include "util.h"

/***
****
****
****
***/

static gboolean
file_headers_load_group (Group * group, StatusItem * status)
{
	gboolean success = FALSE;
        gchar * path;
	GArray *dat, *idx;

	g_return_val_if_fail (group!=NULL, FALSE);

	if (status!=NULL)
		status_item_emit_status_va (status, _("Loading %s"), group_get_readable_name(group));

	/* open the index file */
	path = g_strdup_printf ("/%s/%s/%s.idx", data_dir, group->server->name, group->name);
	idx = read_file (path);
	g_free (path);

	/* open the data file */
	path = g_strdup_printf ("/%s/%s/%s.dat", data_dir, group->server->name, group->name);
	dat = read_file (path);
	g_free (path);

	/* allocate the articles array */
	pan_warn_if_fail (group->_articles == NULL);
	group_get_articles (group);

	if (dat!=NULL && idx!=NULL)
	{
		const gchar * dat_str = dat->data;
		const gchar * march = idx->data;
		const glong version = get_next_token_int (march, '\n', &march);

		if (version==1 || version==2 || version==3)
		{
			int i;
			long l;
			const long qty = get_next_token_long (march, '\n', &march);
			GPtrArray * addme = g_ptr_array_new ();

			group->_one_big_chunk = dat->data;
			pan_g_ptr_array_reserve (group->_articles, qty);

			for (i=0; i!=qty; ++i)
			{
				Article * a = article_new (group);
				int extra_header_qty;
				int j;

				/* message id */
				l = get_next_token_long (march, '\n', &march);
				if (l != -1)
					a->message_id = dat_str + l;

				/* author */
				if (version<2) /* version 2 split author into 2 fields */
				{
					l = get_next_token_long (march, '\n', &march);
					if (l != -1)
						article_set_author_from_header (a, dat_str+l);
				}
				else
				{
					l = get_next_token_long (march, '\n', &march);
					if (l != -1)
						a->author_addr = dat_str+l;

					l = get_next_token_long (march, '\n', &march);
					if (l != -1)
						a->author_real = dat_str+l;
				}

				/* subject */
				l = get_next_token_long (march, '\n', &march);
				if (l != -1 )
					a->subject = dat_str + l;

				/* date string - removed in version 3 */
				if (version<3)
					skip_next_token (march, '\n', &march);

				/* references */
				l = get_next_token_long (march, '\n', &march);
				if (l != -1 )
					article_set_header (a, HEADER_REFERENCES, dat_str+l, 0);

				/* numeric fields */
				a->part           = (gint16) get_next_token_int (march, '\n', &march);
				a->parts          = (gint16) get_next_token_int (march, '\n', &march);
				a->linecount      = (guint16) get_next_token_int (march, '\n', &march);
				a->crosspost_qty  = (gint8) get_next_token_int (march, '\n', &march);
				a->state          = (guint16) get_next_token_int (march, '\n', &march);
				a->date           = (time_t) get_next_token_ulong (march, '\n', &march);
				a->number         = (gulong) get_next_token_ulong (march, '\n', &march);

				/* extra headers */
				extra_header_qty  = get_next_token_int (march, '\n', &march);
				for (j=0; j<extra_header_qty; ++j)
				{
					const long key_idx = get_next_token_long (march, '\n', &march);
					const long val_idx = get_next_token_long (march, '\n', &march);
					if (key_idx!=-1 && val_idx!=-1)
					{
						const gchar * key = dat_str + key_idx;
						const gchar * val = dat_str + val_idx;
						article_set_header (a, key, val, 0);
					}
				}

				/* let the user know what we're doing */
				if (status != NULL) {
					status_item_emit_next_step (status);
					if (!(addme->len % 256))
						status_item_emit_status_va (status,
							_("Loaded %d of %d articles"), i, qty);
				}

				/* add the article to the group */
				g_ptr_array_add (addme, a);
			}

			group_add_articles_remove_unused (group, addme, status);
			group->articles_dirty = FALSE;
			success = TRUE;
			g_ptr_array_free (addme, TRUE);
		}
		else
		{
			pan_error_dialog (
				_("Unsupported data version for %s headers : %d"),
					group->name,
					version);
		}
	}

	if (dat != NULL) g_array_free (dat, !success);
	if (idx != NULL) g_array_free (idx, TRUE);

	return success;
}

static gchar*
get_mbox_filename (Group * folder)
{
	gchar * retval;

	g_return_val_if_fail (folder!=NULL, NULL);
	g_return_val_if_fail (group_is_folder(folder), NULL);

	retval = g_strdup_printf ("/%s/%s/%s.mbox",
		data_dir,
		folder->server->name,
		folder->name);

	return retval;
}

static gboolean
file_headers_load_folder (Group * folder, StatusItem * status)
{
	gint i;
	gchar * text;
	const gchar * march;
	GPtrArray * articles;

	g_return_val_if_fail (folder!=NULL, FALSE);
	g_return_val_if_fail (group_is_folder(folder), FALSE);

	/* get the mbox file */
	{
		GArray * array;
		gchar * fname = get_mbox_filename (folder);
		array = read_file (fname);
		g_free (fname);
		if (array == NULL)
			return FALSE;
		text = array->data;
		g_array_free (array, FALSE);
	}

	/* allocate the articles array */
	pan_warn_if_fail (folder->_articles == NULL);
	group_get_articles (folder);

	/* walk through the mbox */
	i = 0;
	march = text;
	articles = g_ptr_array_new ();
	while (march != NULL)
	{
		gchar * from = NULL;
		gchar * msg = NULL;

		march = mbox_get_next_message (march, &from, &msg);
		if (msg != NULL)
		{
			Article * a = article_new (folder);
			a->number = ++i;
			article_set_from_raw_message (a, msg);
			g_ptr_array_add (articles, a);
			g_free (msg);
		}

		g_free (from);
	}

	/* if we've got articles then add them; otherwise, clean up */
	if (articles->len != 0)
		group_add_articles_remove_unused (folder, articles, status);
	else {
		g_ptr_array_free (folder->_articles, TRUE);
		folder->_articles = NULL;
	}
	folder->articles_dirty = FALSE;

	/* cleanup */
	g_free (text);

	return folder->_articles!=NULL && folder->_articles->len!=0;
}

void
file_headers_load (Group * group, StatusItem * status)
{
	struct timeval start;
	struct timeval finish;
	double diff;
	gboolean success = FALSE;

	/* start the stopwatch */
	gettimeofday (&start, NULL);

	if (group_is_folder (group))
	{
		success = file_headers_load_folder (group, status);
		if (!success)
			g_message (_("Couldn't load mbox for %s; trying old-style folders..."), group->name);
	}
	if (!success)
		success = file_headers_load_group (group, status);

	/* expire the old articles, if any */
	if (success)
		group_expire_old_articles (group);

	/* timing stats */
	gettimeofday (&finish, NULL);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/1000000.0;
	g_message (_("Got %d articles in %.1f seconds (%.0f art/sec)"),
		group->_articles->len,
		diff,
		group->_articles->len/(fabs(diff)<0.001?0.001:diff));
}

/**
 * This is used to make sure we don't write the same string to the output
 * file more than once.  This way we can get some space savings from authors
 * and subjects that are repeated many times.
 */
long
get_string_offset (GHashTable * hash,
                   const gchar * str,
                   FILE * fp,
                   long * pos)
{
	gpointer p;
	long retval;

	g_assert (fp != NULL);
	g_assert (pos != NULL);

	if (!is_nonempty_string(str)) /* nothing to write */
	{
		retval = -1;
	}
	else if (hash == NULL) /* don't bother weeding duplicates */
	{
		retval = *pos;
		*pos += fwrite (str, 1, strlen(str)+1, fp);
	}
	else if ((p = g_hash_table_lookup(hash, str)) != NULL) /* a match! */
	{
		retval = GPOINTER_TO_INT(p);
	}
	else
	{
		retval = *pos;
		g_hash_table_insert (hash, (gpointer)str, GINT_TO_POINTER(retval));
		*pos += fwrite (str, 1, strlen(str)+1, fp);
	}

	return retval;
}


static void
file_headers_save_folder (Group * folder, StatusItem * status)
{
	FILE * fp;
	gchar * fname;
	gchar * fname_tmp;

	g_return_if_fail (group_is_folder(folder));

	/* get the filenames & start writing to the outfile */
	fname = get_mbox_filename (folder);
	fname_tmp = g_strdup_printf ("%s.tmp", fname);
	fp = fopen (fname_tmp, "w+");

	if (fp != NULL)
	{
		guint i = 0;
		GPtrArray * array = folder->_articles;
		GString * s = g_string_new (NULL);

		for (i=0; i!=array->len; ++i)
		{
			Article * a = ARTICLE(g_ptr_array_index(array,i));
			gchar buf[32], *pch, *mbox;

			/* add headers */
			pch = article_get_headers (a);
			g_string_assign (s, pch);
			g_free (pch);

			/* add Status: header */
			*(pch=buf)='\0';
			if (article_is_read(a)) *pch++ = 'R';
			if (!article_is_new(a)) *pch++ = 'O';
			*pch = '\0';
			if (*buf != '\0')
				g_string_sprintfa (s, _("Status: %s\n"), buf);
  
			/* add header/body separator */ 
			g_string_append (s, "\n");
  
			/* add body */ 
			pch = article_get_body (a);
			g_string_append (s, pch);
			g_free (pch);

			/* write the message in mbox format */
			mbox = mbox_format_message (s->str, a->author_addr, a->date);
			fputs (mbox, fp);
			g_free (mbox);
		}
		fclose (fp);

		/* wrote to tmp file okay... now make it permanent */
		remove (fname);
		rename (fname_tmp, fname);

		/* cleanup */
		g_string_free (s, TRUE);
	}

	g_free (fname);
	g_free (fname_tmp);
}

static void
file_headers_save_group (Group * group, StatusItem * status)
{
	guint i;
	FILE * idx_fp;
	FILE * dat_fp;
	gchar * pch;
	gchar * idx_path;
	gchar * dat_path;
	long pos = 0;
	GHashTable * hash_name = NULL;
	GHashTable * hash_mail = NULL;
	GHashTable * hash_refs = NULL;
	GHashTable * hash_misc = NULL;

	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->server!=NULL);
	g_return_if_fail (group->_articles!=NULL);

	/* trivial case #1: no change, so no need to save */
	if (group->articles_dirty == FALSE)
		return;

	/* trivial case #2: nothing to save */
	if (group->_articles->len == 0) {
		file_headers_destroy (group);
		return;
	}

	/**
	***  Save the Headers
	**/

	g_message (_("Saving \"%s\""), group_get_readable_name(group));

	/* open index file */
	idx_path = g_strdup_printf ("/%s/%s/%s.idx.tmp", data_dir, group->server->name, group->name);
	idx_fp = fopen (idx_path, "w+");
	if (idx_fp == NULL) {
		g_free (idx_path);
		return;
	}

	/* open data file */
	dat_path = g_strdup_printf ("/%s/%s/%s.dat.tmp", data_dir, group->server->name, group->name);
	dat_fp = fopen (dat_path, "w+");
	if (dat_fp == NULL) {
		fclose (idx_fp);
		remove (idx_path);
		g_free (idx_path);
		g_free (dat_path);
		return;
	}

	/* Write DATBASE_VERSION */
	fprintf (idx_fp, "3\n%ld\n", (long)group->_articles->len);

	/* Write the article information... */
	pos = 0;
	hash_name = g_hash_table_new (g_str_hash, g_str_equal);
	hash_mail = g_hash_table_new (g_str_hash, g_str_equal);
	hash_refs = g_hash_table_new (g_str_hash, g_str_equal);
	hash_misc = g_hash_table_new (g_str_hash, g_str_equal);
	for (i=0; i!=group->_articles->len; ++i)
	{
		Article * a = ARTICLE(g_ptr_array_index (group->_articles, i));
		GPtrArray * extra_headers = article_get_all_headers (a);
		guint j;

		const long id_idx          = get_string_offset (NULL, a->message_id,  dat_fp, &pos);
		const long author_addr_idx = get_string_offset (hash_mail, a->author_addr, dat_fp, &pos);
		const long author_real_idx = get_string_offset (hash_name, a->author_real, dat_fp, &pos);
		const long subj_idx        = get_string_offset (hash_misc, a->subject,     dat_fp, &pos);
		const long refs_idx        = get_string_offset (hash_refs, a->references,  dat_fp, &pos);

		pan_warn_if_fail (a->number != 0);

		/* write the non-string fields. */
		fprintf (idx_fp,
			"%ld\n" "%ld\n" "%ld\n" "%ld\n" "%ld\n"
			"%d\n" "%d\n"
			"%u\n"
			"%d\n"
			"%d\n"
			"%lu\n"
			"%lu\n",
			id_idx, author_addr_idx, author_real_idx, subj_idx, refs_idx,
			(int)a->part,
			(int)a->parts,
			(unsigned int)a->linecount,
			(int)a->crosspost_qty,
			(int)a->state,
			(unsigned long)a->date,
			(unsigned long)a->number);

		/* write the extra headers */
		g_assert (!(extra_headers->len % 2));
		fprintf (idx_fp, "%d\n", extra_headers->len/2); /* extra header qty */
		for (j=0; j!=extra_headers->len; j+=2)
		{
			const gchar * key = (const gchar*) g_ptr_array_index (extra_headers, j);
			const gchar * val = (const gchar*) g_ptr_array_index (extra_headers, j+1);
			if (is_nonempty_string(key) && is_nonempty_string(val))
			{
				const long key_index = get_string_offset (hash_misc, key, dat_fp, &pos);
				const long val_index = get_string_offset (hash_misc, val, dat_fp, &pos);
				fprintf (idx_fp, "%ld\n%ld\n", key_index, val_index);
			}
		}

		g_ptr_array_free (extra_headers, TRUE);
	}
	g_hash_table_destroy (hash_name); hash_name = NULL;
	g_hash_table_destroy (hash_mail); hash_mail = NULL;
	g_hash_table_destroy (hash_refs); hash_refs = NULL;
	g_hash_table_destroy (hash_misc); hash_misc = NULL;

	/* the write went okay; move the idx file over */
	fclose (idx_fp);
	pch = g_strdup_printf ("/%s/%s/%s.idx", data_dir, group->server->name, group->name);
	remove (pch);
	rename (idx_path, pch);
	g_free (idx_path);
	g_free (pch);

	/* the write went okay; move the dat file over */
	fclose (dat_fp);
	pch = g_strdup_printf ("/%s/%s/%s.dat", data_dir, group->server->name, group->name);
	remove (pch);
	rename (dat_path, pch);
	g_free (dat_path);
	g_free (pch);

	/* cleanup */
	group->articles_dirty = FALSE;
}
void
file_headers_save (Group * group, StatusItem * status)
{
	struct timeval start;
	struct timeval finish;
	double diff;

	gettimeofday (&start, NULL);

	/* trivial case #1: no change, so no need to save */
	if (group->articles_dirty == FALSE)
		return;

	/* trivial case #2: nothing to save */
	if (group->_articles->len == 0) {
		file_headers_destroy (group);
		return;
	}

	/* save the group */
	if (group_is_folder (group))
		file_headers_save_folder (group, status);
	else
		file_headers_save_group (group, status);

	/* timing stats */
	gettimeofday (&finish, NULL);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/1000000.0;
	g_message (_("Saved %d articles in %.1f seconds (%.0f art/sec)"),
		group->_articles->len,
		diff,
		group->_articles->len/(fabs(diff)<0.001?0.001:diff));
}

void
file_headers_destroy (const Group * group)
{
	gchar * path;

	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->server!=NULL);
	g_return_if_fail (is_nonempty_string(data_dir));

	path = g_strdup_printf ("/%s/%s/%s.idx", data_dir, group->server->name, group->name);
	g_message (_("Deleting file \"%s\""), path);
	remove (path);
	g_free (path);

	path = g_strdup_printf ("/%s/%s/%s.dat", data_dir, group->server->name, group->name);
	g_message (_("Deleting file \"%s\""), path);
	remove (path);
	g_free (path);
}
