/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktogglebutton.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-stock.h>

#include "acache.h"
#include "article-thread.h"
#include "articlelist.h"
#include "debug.h"
#include "file-headers.h"
#include "globals.h"
#include "group.h"
#include "grouplist.h"
#include "log.h"
#include "newsrc.h"
#include "nntp.h"
#include "pan-callback.h"
#include "prefs.h"
#include "gui.h"
#include "queue.h"
#include "server.h"
#include "task-bodies.h"
#include "task-headers.h"
#include "task-grouplist.h"
#include "util.h"

/**
***
**/

static PanCallback * group_articles_removed = NULL;

static PanCallback * group_articles_added = NULL;

PanCallback*
group_get_articles_added_callback (void)
{
	if (group_articles_added == NULL)
		group_articles_added = pan_callback_new ();
	return group_articles_added;
}

PanCallback*
group_get_articles_removed_callback (void)
{
	if (group_articles_removed == NULL)
		group_articles_removed = pan_callback_new ();
	return group_articles_removed;
}

/**
***
**/

Group*
group_new (Server * server, const char * name)
{
	Group * g = g_new (Group, 1);
	debug1 (DEBUG_PAN_OBJECT, "group_new: %p", g);
	pan_warn_if_fail (is_nonempty_string(name));
        group_constructor (g, group_destructor, server, name);
	return g;
}

void
group_constructor (Group                * g,
                   PanObjectDestructor    dtor,
		   Server               * server,
		   const char           * name)
{
	g_return_if_fail (server != NULL);
	pan_warn_if_fail (is_nonempty_string(name));

	pan_object_constructor (PAN_OBJECT(g), dtor);

	debug1 (DEBUG_PAN_OBJECT, "group_constructor: %p", g);
	g->articles_dirty = FALSE;
	g->articles_threading_dirty = FALSE;
	g->flags = (gint8)0;
	g->sort_style = (gint8)0;
	g->state_filter = ~((guint16)STATE_FILTER_IGNORED);
	g->article_qty = (gint32)0;
	g->article_read_qty = (gint32)0;
	g->article_low = (gulong)0;
	g->article_high_old = (gulong)0;
	g->permission = '?';
	g->article_high = (gulong)0;
	g->server = server;
	g->name = server_chunk_string (server, name);
	g->description = NULL;
	g->readable_name = NULL;
	g->download_dir = NULL;
	g->chunk = NULL;

	g->_articles_refcount = (gint)0;
	g->_newsrc = NULL;
	g->_articles = NULL;
	g->_one_big_chunk = NULL;
}

void
group_destructor (PanObject * o)
{
	Group * g = GROUP(o);

	g_return_if_fail (o!=NULL);

	debug1 (DEBUG_PAN_OBJECT, "group_destructor: %p", g);

	/* newsrc */
	if (g->_newsrc != NULL)
		pan_object_unref (PAN_OBJECT(g->_newsrc));

	/* articles */
	/* FIXME: make sure the refcount is down to 0 */
	g_free (g->_one_big_chunk);
	if (g->chunk != NULL)
		g_string_chunk_free (g->chunk);
	if (g->_articles != NULL) {
		g_ptr_array_free (g->_articles, TRUE);
		g->_articles = NULL;
	}

	pan_object_destructor (o);
}

/***
****
****  PUBLIC ACCESSORS / MUTATORS
****
***/

void
group_set_new (Group * group, gboolean is_new)
{
	g_return_if_fail (group != NULL);

	if (is_new)
		group_set_flags (group, group->flags|GROUP_NEW);
	else
		group_set_flags (group, group->flags&~GROUP_NEW);
}

gboolean 
group_is_new (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return (group->flags & GROUP_NEW) ? 1 : 0;
}

void
group_set_subscribed (Group * group, gboolean subscribed)
{
	g_return_if_fail (group != NULL);

	if (subscribed)
		group_set_flags (group, group->flags|GROUP_SUBSCRIBED);
	else
		group_set_flags (group, group->flags&~GROUP_SUBSCRIBED);
}

gboolean 
group_is_subscribed (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);

	return (group->flags & GROUP_SUBSCRIBED) ? 1 : 0;
}

void
group_set_dirty (Group * group)
{
	g_return_if_fail (group != NULL);
	group->server->groups_dirty = TRUE;
}

void
group_set_download_dir (Group * group, const gchar* download_dir)
{
	group_set_string_field (group, &group->download_dir, download_dir);
}

void
group_set_flags (Group * group, guint flags)
{
	group->flags = flags;
	group_set_dirty (group);
}

void
group_set_state_filter (Group * group, guint state_filter)
{
	g_return_if_fail (group != NULL);

	group->state_filter = state_filter;
	group_set_dirty (group);
}

void
group_set_sort_style (Group * group, int sort_style)
{
	g_return_if_fail (group != NULL);

	group->sort_style = sort_style;
	group_set_dirty (group);
}

void
group_set_article_range (Group * group,
                         gulong article_low,
			 gulong article_high)
{
	group->article_low = article_low;
	group->article_high = article_high;
	group_set_dirty (group);
}

void
group_mark_new_article_number (Group         * group,
                               gulong          everything_above_this_is_new)
{
	group->article_high_old = everything_above_this_is_new;
}

void
group_set_article_qty (Group * group, gint article_qty)
{
	g_return_if_fail (group != NULL);

	if (article_qty != group->article_qty)
	{
		const Group * const_group = (const Group*) group;
		group->article_qty = article_qty;
		group_set_dirty (group);
		grouplist_update_groups (&const_group, 1);
	}
}

void
group_set_article_read_qty (Group * group, gint article_read_qty)
{
	const Group * const_group = (const Group*) group;

	g_return_if_fail (group != NULL);

	article_read_qty = MAX (article_read_qty, 0);
	group->article_read_qty = article_read_qty;
	group_set_dirty (group);
	grouplist_update_groups (&const_group, 1);
}
void
group_inc_article_read_qty (Group * group, gint inc)
{
	group_set_article_read_qty (group, group->article_read_qty + inc);
}

void
group_mark_all_read (Group * g, gboolean read)
{
	Newsrc * newsrc;

	g_return_if_fail (g != NULL);

	newsrc = group_get_newsrc (g);
	newsrc_set_group_range (newsrc, g->article_low, g->article_high);
	newsrc_mark_all (newsrc, read);
	group_set_article_read_qty (g, (read ? g->article_qty : 0));

	/* FIXME: this should be a fired event to decouple group/articlelist */
	if (g == articlelist_get_group())
		articlelist_reload ();
}

gboolean
group_is_read_only (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->permission == 'n';
}

gboolean
group_is_moderated (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->permission == 'm';
}

gboolean
group_is_folder (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->flags & GROUP_FOLDERS ? 1 : 0;
}

void
group_set_is_folder (Group * group, gboolean folder)
{
	if (!folder != !group_is_folder(group))
	{
		if (folder)
			group->flags |= GROUP_FOLDERS;
		else
			group->flags &= ~GROUP_FOLDERS;
		group_set_dirty (group);
	}
}

/***
****
****  DELETE GROUP
****
***/

/* empty the group, delete the group database file */
static void
real_group_empty (Group * group, gboolean clear_counts)
{
	g_return_if_fail (group != NULL);

	/* unload group (if it's loaded) */
	if (articlelist_get_group() == group)
		articlelist_set_group (NULL);

	/* clear out the article headers */
	file_headers_destroy (group);

	/* wipe out all the articles... */
	if (group->_articles != NULL) {
		pan_g_ptr_array_foreach (group->_articles, (GFunc)pan_object_unref, NULL);
		g_ptr_array_set_size (group->_articles, 0);
	}

	group_set_article_qty (group, 0);
	group_set_article_read_qty (group, 0);

	if (clear_counts)
		group_set_article_range (group, 0, 0);


	/* flush the flags */
	if (group->flags & GROUP_SUBSCRIBED)
		group_set_flags (group, GROUP_SUBSCRIBED);
	else
		group_set_flags (group, 0);
}

/* callback for button clicks in group_empty_dialog */
static void
group_empty_cb (GnomeDialog *dialog,
		int button_number,
		gpointer group)
{
	switch (button_number)
	{
		case 0: /* 'yes' */
		{
			GtkWidget * w = gtk_object_get_data (GTK_OBJECT(dialog), "reset");
			gboolean clear_counts = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));
			real_group_empty (group, clear_counts);
			break;
		}
		case 1: /* 'no' */
			break;

		default: /* cancel */
			break;
	};

	gtk_widget_destroy (GTK_WIDGET(dialog));
}

void
group_empty_dialog (Group *group)
{
	GString * str = g_string_new (NULL);
	GtkWidget * dialog = NULL;
	GtkWidget * w = NULL;

	/* sanity check */
	g_return_if_fail (group!=NULL);

	pan_lock();

	/* create dialog */
	g_string_sprintf (str,
	                  _("Clean group \"%s\""),
	                  group_get_readable_name(group));
	dialog = gnome_dialog_new (str->str,
				   GNOME_STOCK_BUTTON_YES,
				   GNOME_STOCK_BUTTON_NO, NULL);

	/* body */
	g_string_sprintf (str,
	                  _("Remove all messages from group: \n%s ?"),
	                  group_get_readable_name(group));
	w = gtk_label_new (str->str);
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);

	/* "clear count?" check button */
	w = gtk_check_button_new_with_label (_("Reset new message counter"));
	gtk_container_add (GTK_CONTAINER(GNOME_DIALOG(dialog)->vbox), w);
	gtk_object_set_data (GTK_OBJECT(dialog), "reset", w);

	/* listen for clicks */
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
			    GTK_SIGNAL_FUNC (group_empty_cb), group);

        gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock();

	g_string_free (str, TRUE);
}


/**
***  ARTICLE LIST
**/

static int
compare_pgchar_pparticle_msgid (const void * a, const void *b )
{
	const gchar * msgid_a = (const gchar*) a;
	const gchar * msgid_b = (*(const Article**)b)->message_id;
	pan_warn_if_fail (is_nonempty_string(msgid_a));
	pan_warn_if_fail (is_nonempty_string(msgid_b));
	return pan_strcmp (msgid_a, msgid_b);
}
static int
compare_particle_pparticle_msgid (const void * a, const void * b)
{
	const gchar * msgid_a = ((const Article*)a)->message_id;
	return compare_pgchar_pparticle_msgid (msgid_a, b);
}
static int
compare_pparticle_pparticle_msgid (const void * a, const void * b)
{
	return compare_particle_pparticle_msgid (*(const Article**)a, b);
}

static void
group_add_articles_ghfunc (gpointer key, gpointer val, gpointer data)
{
	guint i;
	Group * group = GROUP(key);
	GPtrArray * articles = (GPtrArray*)val;

	/* this is so that unreffing from the old group doesn't kill
	   the article -- this ref, plus group_remove_article's unref,
	   has a net change of zero */
	for (i=0; i!=articles->len; ++i)
		pan_object_ref (PAN_OBJECT(g_ptr_array_index(articles,i)));

	/* remove the articles from the other group */
	group_remove_articles (group, articles);

	/* clean up */
	g_ptr_array_free (articles, TRUE);
}

void
group_add_article (Group * group, Article * article)
{
	GPtrArray * a = g_ptr_array_new ();
	g_ptr_array_add (a, article);
	group_add_articles (group, a, NULL, NULL, NULL);
	g_ptr_array_free (a, TRUE);
}

 
void
group_add_articles_remove_unused (Group            * group,
                                  GPtrArray        * articles,
                                  StatusItem       * status)
{
	GPtrArray * ignored;
	GPtrArray * used;

	/* entry assertions */
	g_return_if_fail (group != NULL);
	g_return_if_fail (articles != NULL);
	g_return_if_fail (articles->len != 0);

	/* add the articles */
	ignored  = g_ptr_array_new ();
	used = g_ptr_array_new ();
	group_add_articles (group, articles, status, used, ignored);

	/* destroy unused articles & remove from the incoming array */
	pan_g_ptr_array_assign (articles, used->pdata, used->len);
	pan_g_ptr_array_foreach (ignored, (GFunc)pan_object_unref, NULL);

	/* cleanup */
	g_ptr_array_free (ignored, TRUE);
	g_ptr_array_free (used, TRUE);
}

void
group_add_articles (Group       * group,
                    GPtrArray   * articles,
                    StatusItem  * status,
                    GPtrArray   * used,
                    GPtrArray   * ignored)
{
	gint read_qty;
	guint i, j;
	gulong low;
	gulong high;
	gboolean changed = FALSE;
	gboolean tmp_ref;
	GPtrArray * old_art;
	GPtrArray * new_art;
	GPtrArray * merged;
	GHashTable * remove_from_old = NULL;

	/* sanity checks */
	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->_articles_refcount>=0);
	g_return_if_fail (articles!=NULL);
	if (!articles->len) return;
	g_return_if_fail (articles->len>0);

	/* warn the caller if they're doing something inefficient by adding
	   articles to a group whose refcount is zero.  The old articles in
	   this group have to be loaded too; otherwise they'll be overwritten
	   when we save this group to disk without them. */
	if ((tmp_ref = !group->_articles_refcount))
		group_ref_articles (group, status);

	/**
	***  Remove the articles from their old groups;
	***  do this in batches for efficiency.
	***  This is faster but makes the code a lot uglier.
	**/

	for (i=0; i!=articles->len; ++i)
	{
		Article * article = ARTICLE(g_ptr_array_index(articles,i));
		Group * oldgroup = article->group;
		if (oldgroup!=NULL && oldgroup!=group)
		{
			gboolean array_existed;
			GPtrArray * to_remove;

			/* ensure the hash table exists */
			if (remove_from_old == NULL)
				remove_from_old = g_hash_table_new (
					g_direct_hash, g_direct_equal);

			/* see if any articles are queued for removing
			   from that group */
			to_remove = g_hash_table_lookup (remove_from_old,
			                                 oldgroup);
			array_existed = to_remove != NULL;
			if (!array_existed)
				to_remove = g_ptr_array_new ();
			g_ptr_array_add (to_remove, article);
			if (!array_existed)
				g_hash_table_insert (remove_from_old,
				                     oldgroup,
				                     to_remove);
		}
	}
	if (remove_from_old != NULL) {
		g_hash_table_foreach (remove_from_old,
		                      group_add_articles_ghfunc,
		                      NULL);
		g_hash_table_destroy (remove_from_old);
		remove_from_old = NULL;
	}

	/**
	***  Now that the articles don't belong to another group,
	***  add them to this one.  We do this by sorting the incoming
	***  articles and merging two sorted lists.
	**/

	old_art = group_get_articles (group);
	new_art = pan_g_ptr_array_dup (articles);
	pan_g_ptr_array_sort (new_art, compare_pparticle_pparticle_msgid);
	merged = g_ptr_array_new ();
	pan_g_ptr_array_reserve (merged, new_art->len + old_art->len);
	if (used != NULL)
		pan_g_ptr_array_reserve (used, used->len + new_art->len);
	if (ignored != NULL)
		pan_g_ptr_array_reserve (ignored, ignored->len + new_art->len);

	/* now merge the two lists */
	low = group->article_low;
	high = group->article_high;
	for (i=j=0; i!=new_art->len || j!=old_art->len; )
	{
		Article * addme = NULL;

		if (i==new_art->len)
		{
			addme = g_ptr_array_index (old_art, j++);
		}
		else if (j==old_art->len)
		{
			addme = g_ptr_array_index (new_art, i++);
			changed = TRUE;
			if (used != NULL)
				g_ptr_array_add (used, addme);
		}
		else
		{
			Article * a1 = ARTICLE(g_ptr_array_index(new_art,i));
			Article * a2 = ARTICLE(g_ptr_array_index(old_art,j));
			const int comp = pan_strcmp (a1->message_id, a2->message_id);

			if (comp<0) {
				addme = a1;
				changed = TRUE;
				++i;
				if (used != NULL)
					g_ptr_array_add (used, addme);
			} else {
				addme = a2;
				++j;
				if (!comp && a1!=a2) {
					if (ignored != NULL)
						g_ptr_array_add (ignored, a1);
					++i;
				}
			}
		}

		if (addme != NULL)
		{
			addme->group = group;

			if (addme->number < low)
				low = addme->number;
			if (addme->number > high)
				high = addme->number;

			g_ptr_array_add (merged, addme);
		}
	}

	/* update the article range */
	if (high > group->article_high || low<group->article_low)
		group_set_article_range (group, low, high);

	/* update the group 'article' field */
	group->articles_dirty = TRUE;
	pan_g_ptr_array_assign (old_art, merged->pdata, merged->len);
	group_set_article_qty (group, old_art->len);

	/* update the group 'article read' field */
	read_qty = 0;
	for (i=0; i!=old_art->len; ++i)
		if (article_is_read (ARTICLE(g_ptr_array_index(old_art,i))))
			++read_qty;
	group_set_article_read_qty (group, read_qty);

	/* let everyone know that articles were just added */
	if (changed) {
		group->articles_threading_dirty = TRUE;
		pan_callback_call (group_get_articles_added_callback(), group, articles);
	}

#if 0
	/* folders are for permanent storage, so re-save *NOW*.
	   This is especially important for pan.sendlater. */
	if (group_is_folder(group))
		file_headers_save (group, status);
#endif

	/* cleanup */
	g_ptr_array_free (merged, TRUE);
	g_ptr_array_free (new_art, TRUE);
	if (tmp_ref)
		group_unref_articles (group, status);
}

void
group_thread_if_needed (Group * group, StatusItem * status)
{
	g_return_if_fail (group != NULL);

	if (group->articles_threading_dirty && group->_articles!=NULL && group->_articles->len)
	{
		thread_articles (group->_articles, status);
		check_multipart_articles (group->_articles);
		group->articles_threading_dirty = FALSE;
	}
}

 
gpointer
group_get_article_by_message_id (Group * group, const gchar * message_id)
{
	int index = 0;
	GPtrArray * a;
	gpointer retval = NULL;
	gboolean exact_match = FALSE;

	g_return_val_if_fail (group!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);

	a = group_get_articles (group);
        index = lower_bound (message_id,
		             a->pdata,
		             a->len,
		             sizeof(gpointer),
	                     compare_pgchar_pparticle_msgid,
		             &exact_match);
	if (exact_match)
		retval = g_ptr_array_index (a, index);

	return retval;
}

void
group_remove_article (Group * group, Article * article)
{
	GPtrArray * a = g_ptr_array_new ();
	g_ptr_array_add (a, article);
	group_remove_articles (group, a);
	g_ptr_array_free (a, TRUE);
}

void
group_remove_articles (Group * group, GPtrArray * articles)
{
	guint i;
	guint old_len;
	int removed_read = 0;
	int removed_total = 0;
	GPtrArray * a;
	GPtrArray * tmp;
	GPtrArray * removed;

	/* sanity checks */
	g_return_if_fail (group != NULL);
	g_return_if_fail (articles != NULL);
	if (!articles->len) return;
	for (i=0; i!=articles->len; ++i)
		g_assert (ARTICLE(g_ptr_array_index(articles,i))->group==group);

	/* let everyone know that articles are going to be removed */
	pan_callback_call (group_get_articles_removed_callback(), group, articles);

	/* is this the last gasp for these articles? */
	for (i=0; i!=articles->len; ++i) {
		Article * article = ARTICLE(g_ptr_array_index(articles,i));
		if (PAN_OBJECT(article)->ref_count == 1) { /* about to die */
			article_isolate (article);
			acache_delete (article->message_id);
		}
	}

	a = group_get_articles (group);
	old_len = a->len;

	/* sort the articles being removed for lower_bound used below */
	qsort (articles->pdata,
	       articles->len,
	       sizeof(gpointer),
	       compare_pparticle_pparticle_msgid);

	/* build a tmp array of all the articles not being removed */
	tmp = g_ptr_array_new ();
	removed = g_ptr_array_new ();
	pan_g_ptr_array_reserve (tmp, a->len);
	for (i=0; i!=a->len; ++i)
	{
		Article * article = ARTICLE(g_ptr_array_index(a,i));
		gboolean do_remove = FALSE;
		lower_bound (article,
		             articles->pdata,
		             articles->len,
		             sizeof(gpointer),
		             compare_particle_pparticle_msgid,
		             &do_remove);
		if (do_remove)
		{
			++removed_total;
			if (article_is_read(article))
				++removed_read;

			g_ptr_array_add (removed, article);
		}
		else
		{
			g_ptr_array_add (tmp, article);
		}
	}

	g_message (_("Removed %d articles from group %s"),
	           removed_total,
		   group->name);

	/* free the old articles */
	for (i=0; i!=removed->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(removed,i));
		a->group = NULL;
		pan_object_unref (PAN_OBJECT(a));
	}

	/* update the group array */
	pan_g_ptr_array_assign (a, tmp->pdata, tmp->len);

	/* if articles were removed, update our stats */
	if (a->len != old_len)
	{
		const Group * const_group = (const Group*) group;
		group->articles_dirty = TRUE;
		group->article_qty -= removed_total;
		group->article_read_qty -= removed_read;
		group->articles_threading_dirty = TRUE;
		grouplist_update_groups (&const_group, 1);
		if (group_is_folder (group))
			file_headers_save (group, NULL);
	}

	/* cleanup */
	g_ptr_array_free (tmp, TRUE);
	g_ptr_array_free (removed, TRUE);
}

void
group_expire_old_articles (Group * g)
{
	guint i;
	GPtrArray * removeme;

	g_return_if_fail (g != NULL);
	g_return_if_fail (g->_articles != NULL);

	removeme = g_ptr_array_new ();

	/* find old articles to remove */
	for (i=0; i!=g->_articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(g->_articles,i));
		if ((a->number<g->article_low) || (a->number>g->article_high))
			g_ptr_array_add (removeme, a);
	}

	/* remove the old articles */
	if (removeme->len != 0)
		group_remove_articles (g, removeme);

	/* cleanup */
	g_ptr_array_free (removeme, TRUE);
}


/***
****
***/

void
group_ref_articles (Group * group, StatusItem * status)
{
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles_refcount >= 0);

	if (++group->_articles_refcount == 1)
		file_headers_load (group, status);

g_message ("group %s refcount inc to %d", group->name, group->_articles_refcount);

	group_set_article_qty (group, group->_articles->len);
}

void
group_unref_articles (Group * group, StatusItem * status)
{
	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles_refcount > 0);

	if (!--group->_articles_refcount)
	{
		file_headers_save (group, status);

		/* free the articles */
		pan_g_ptr_array_foreach (group->_articles,
		                         (GFunc)pan_object_unref,
		                         NULL);
		g_ptr_array_free (group->_articles, TRUE);
		group->_articles = NULL;

		/* free the memory used by those articles */
		replace_gstr (&group->_one_big_chunk, NULL);
		if (group->chunk != NULL) {
			g_string_chunk_free (group->chunk);
			group->chunk = NULL;
		}
	}

g_message ("group %s refcount dec to %d", group->name, group->_articles_refcount);
}



/**
***
**/

Group*
folder_get_by_name (const gchar * foldername)
{
	Server * server;

	/* get the folder server... */
	server = server_get_by_name (INTERNAL_SERVER_NAME);
	g_return_val_if_fail (server!=NULL, NULL);

	/* get the folder... */
	return group_get_by_name (server, foldername);
}


const gchar*
group_get_readable_name (const Group * group)
{
	const gchar * retval;

	if (group == NULL)
	{
		pan_warn_if_reached ();
		retval = "(No Group)";
	}
	else
	{
		retval = is_nonempty_string (group->readable_name)
			? group->readable_name
			: group->name;
	}

	return retval;
}

/**
***
**/

const gchar*
group_chunk_string (Group         * group,
                    const gchar   * chunkme)
{
	gchar * retval = NULL;

	if (group->chunk == NULL)
		group->chunk = g_string_chunk_new (8192);

	if (chunkme != NULL)
		retval = g_string_chunk_insert_const (group->chunk, chunkme);

	return retval;
}

void
group_set_string_field (Group         * group,
	                const gchar  ** field_ptr,
			const gchar   * chunkme_replacement)
{
	g_return_if_fail (group!=NULL);
	g_return_if_fail (group->server!=NULL);
	g_return_if_fail (field_ptr!=NULL);

	*field_ptr = server_chunk_string (group->server, chunkme_replacement);
}


/***
****
****  NEW FOLDER DIALOG
****
***/

static void
new_folder_cb (gchar* string, gpointer data)
{
	Group * group = NULL;
	Server * server = server_get_by_name (INTERNAL_SERVER_NAME);
	gchar * tmp;

	/* user hit cancel */
	if (string == NULL)
		return;

	/* make sure the user entered something */
	tmp = g_strdup (string);
	g_strstrip (tmp);
	if (!is_nonempty_string (tmp)) {
		pan_error_dialog (_("Invalid folder name: \"%s\""), string);
		g_free (tmp);
		return;
	}

	/* make sure there's not already a folder with that name */
	g_strdown (tmp);
	group = folder_get_by_name (tmp);
	if (group != NULL) {
		pan_error_dialog (_("Folder \"%s\" already exists."), string);
		g_free (tmp);
		return;
	}

	/* make sure it's not in the reserved namespace */
	if (!strncmp(tmp, "pan.", 4)) {
		pan_error_dialog (_("Please don't begin folders with \"pan.\"; it confuses me terribly."));
		g_free (tmp);
		return;
	}

	/* create new folder */
	group = group_new (server, tmp);
	group_set_is_folder (group, TRUE);
	group_set_dirty (group);
	server_add_groups (server, &group, 1, NULL, NULL);

	/* cleanup */
	g_free (tmp);
}

void
group_new_folder_dialog (void)
{
	gchar * prompt = g_strdup (_("New Folder Name:"));
	gnome_request_dialog (FALSE, prompt, NULL, 128,
			      new_folder_cb,
			      NULL, GTK_WINDOW(Pan.window));
	g_free (prompt);
}

void
group_download_all_articles (Group * group)
{
	Task * task;

	g_return_if_fail (group != NULL);
	g_return_if_fail (group->_articles != NULL);

	group_ref_articles (group, NULL);
	task = TASK(task_bodies_new (group, group->_articles));
	if (task != NULL)
		queue_add (task);
	group_unref_articles (group, NULL);
}

Newsrc*
group_get_newsrc (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_newsrc == NULL)
		group->_newsrc = newsrc_new (NULL, group->article_low, group->article_high);

	return group->_newsrc;
}

gboolean
group_has_newsrc (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->_newsrc != NULL;
}

GPtrArray*
group_get_articles (Group * group)
{
	g_return_val_if_fail (group!=NULL, NULL);

	if (group->_articles == NULL)
		group->_articles = g_ptr_array_new ();

	return group->_articles;
}

gboolean
group_articles_reffed (const Group * group)
{
	g_return_val_if_fail (group!=NULL, FALSE);
	return group->_articles_refcount > 0; 
}

