/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include <sys/time.h>          

#include "article-find.h"
#include "article-thread.h"
#include "articlelist.h"
#include "dialogs/dialogs.h"
#include "dialogs/rules_interface.h"
#include "globals.h"
#include "gnksa.h"
#include "gui.h"
#include "gui-notebook.h"
#include "gui-paned.h"
#include "grouplist.h"
#include "log.h"
#include "message-send.h"
#include "message-window.h"
#include "pan.h"
#include "prefs.h"
#include "print.h"
#include "status-item.h"
#include "status-item-view.h"
#include "save.h"
#include "text.h"
#include "task-manager.h"
#include "util.h"
#include "queue.h"

GtkTooltips * ttips = NULL;
GdkColormap * cmap = NULL;

#include "xpm/envelope.xpm"
#include "xpm/envelope_new.xpm"

static unsigned int gui_refresh_timeout_id = 0;
static double KBps = 0.0;
static GtkWidget * taskbar;

#define VIEW_QTY 3
typedef struct {
	StatusItem * status_item;
	StatusItemView * view;
} ViewStruct;
static ViewStruct views[VIEW_QTY];

/**
*** private function prototypes
**/

static void exit_cb (void);
static gboolean gui_key_press_cb (GtkWidget*, GdkEventKey*, gpointer);
static int window_delete_event_cb (GtkWidget*, GdkEvent*, gpointer);
static int gui_refresh_timeout_cb (gpointer);
static int gui_message_state_filter_changed_cb  (gpointer, gpointer, gpointer);
static int update_menus_cb                      (gpointer, gpointer, gpointer);
static int thread_changed_cb                    (gpointer, gpointer, gpointer);
static int gui_articlelist_sort_changed_cb      (gpointer, gpointer, gpointer);
static int gui_text_show_all_headers_changed_cb (gpointer, gpointer, gpointer);
static int status_item_active_changed_cb        (gpointer, gpointer, gpointer);

static void
server_import_newsrc_dialog_menu (void)
{
	Server * server = grouplist_get_server ();
	if (server != NULL)
		server_import_newsrc_dialog (server);
}
static void
server_export_newsrc_dialog_menu (void)
{
	Server * server = grouplist_get_server ();
	if (server != NULL)
		server_export_newsrc_dialog (server);
}


static GnomeUIInfo file_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Save Article As..."),
		N_("Save the article headers and body to a text file."),
		save_current_article, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Save Binary Attachment"),
		N_("Download and decode attachments in the selected messages."),
		articlelist_selected_decode, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Save Binary Attachment _As..."),
		N_("Download and decode attachments in the selected messages, and save as..."),
		articlelist_selected_decode_as, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'\0', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Open Binary Attachment..."),
		N_("Open Binary Attachment..."),
		articlelist_selected_open, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PRINT_ITEM (print_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),
	GNOMEUIINFO_END
};

static void
killfile_cb (void)
{
	dialog_rules ();
}
static void
bozo_cb (void)
{
	GtkWidget * dialog = create_bozo_dialog ();
	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
}

GtkWidget* create_rules_dialog (void);


static GnomeUIInfo edit_menu[] =
{
	GNOMEUIINFO_MENU_SELECT_ALL_ITEM (gui_select_all, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_FIND_ITEM(articlelist_find_text_cb, NULL),
	GNOMEUIINFO_MENU_FIND_AGAIN_ITEM(articlelist_find_next_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Rules..."),
		N_("Rules Tool."),
		killfile_cb, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_NEW,
		'R', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Bozos..."),
		N_("Bozo Management."),
		bozo_cb
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(prefs_spawn, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo post_menu [] =
{
        {
		GNOME_APP_UI_ITEM,
		N_("_Post to newsgroup"),
		N_("Post a new message to the current group."),
		message_post_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_NEW,
		'P', 0, NULL
	},
        {
		GNOME_APP_UI_ITEM,
		N_("_Followup to newsgroup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RPL,
		'F', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Reply by E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RPL,
		'R', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Followup to newsgroup _and Reply by E-mail"),
		N_("Send a reply both to the author in mail, and to the news server."),
		message_followup_reply_window, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_RPL,
		0, 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("For_ward article by E-mail"),
		N_("Forward article by E-mail"),
		message_forward_window
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Mail/Post messages from folder \"pan.sendlater\""),
		N_("Mail/Post messages from folder \"pan.sendlater\""),
		flush_sendlater_articles, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_MAIL_SND,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Cancel Selected Article you Posted"),
		N_("Cancel Selected Article you Posted"),
		articlelist_selected_cancel
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Supersede Selected Article you Posted"),
		N_("Supersede Selected Article you Posted"),
		articlelist_selected_supersede
	},
	GNOMEUIINFO_END
};

static int current_layout = -1;

static
void zoom_cb (void)
{
	if (current_layout == GUI_PANED)
	{
		GtkWidget * w;
		int page;

		if (GTK_WIDGET_HAS_FOCUS(Pan.text)) {
			page = MESSAGE_PAGE;
			w = Pan.text;
		}
		else if (GTK_WIDGET_HAS_FOCUS(Pan.article_ctree)) {
			page = ARTICLELIST_PAGE;
			w = Pan.article_ctree;
		}
		else {
			page = GROUPS_PAGE;
			w = Pan.group_clist;
		}
        	gui_set_layout (GUI_NOTEBOOK);
		gui_page_set (page, w);
	}
	else /* notebook */
	{
		GtkWidget * w = NULL;
		int notebook_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook));
        	gui_set_layout (GUI_PANED);
		switch (notebook_page) {
			case 0: w = Pan.group_clist; break;
			case 1: w = Pan.article_ctree; break;
			case 2: w = Pan.text; break;
		}
		gtk_widget_grab_focus (w);
	}
}

static GnomeUIInfo navigate_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Next Unread _Group"),
		N_("Move to the Next Group with Unread Messages."),
		grouplist_activate_next_unread_group, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
		'G', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Next Group"),
		N_("Move to the Next Group."),
		grouplist_activate_next_group, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
		'G', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Up to Select the Previous Article"),
		N_("Up to Select the Previous Article"),
		articlelist_skip_to_prev, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UP,
		'U', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Down to Select the Next Article"),
		N_("Down to Select the Next Article"),
		articlelist_skip_to_next, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'D', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Select the _Next Unread Message"),
		N_("Select the Next Unread Message"),
		articlelist_skip_to_next_unread, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'N', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Select the Next Unread Message _Body"),
		N_("Select the Next Unread Message Body"),
		articlelist_skip_to_next_unread_body, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'B', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Up to Read the Previous Article"),
		N_("Up to Read the Previous Article"),
		articlelist_view_prev, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UP,
		'U', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Down to Read the Next Article"),
		N_("Down to Read the Next Article"),
		articlelist_view_next, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'D', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Read the Next Unread Message"),
		N_("Read the Next Unread Message"),
		articlelist_view_next_unread, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'N', 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Read the Next Unread Message Body"),
		N_("Read the Next Unread Message Body"),
		articlelist_view_next_unread_body, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'B', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Up to Select the Previous Thread"),
		N_("Up to Select the Previous Thread"),
		articlelist_skip_to_prev_thread, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UP,
		'U', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Down to Select the Next Thread"),
		N_("Down to Select the Next Thread"),
		articlelist_skip_to_next_thread, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		'D', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo online_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Online/Offline Settings..."),
		N_("Online/Offline Settings..."),
		prefs_spawn_to_news_connections, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Get List of _All Groups"),
		N_("Refresh a list of groups from the selected servers."),
		grouplist_get_all, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get List of _New Groups"),
		N_("Download a list of new groups from the selected servers."),
		grouplist_get_new, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
		0, 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Import .newsrc..."),
		N_("Import .newsrc..."),
		server_import_newsrc_dialog_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REDO,
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Export .newsrc..."),
		N_("Export .newsrc..."),
		server_export_newsrc_dialog_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UNDO,
	},
	GNOMEUIINFO_END
};

static void
pan_homepage_url (void)
{
	gnome_url_show ("http://www.superpimp.org/");
}

static void
pan_manual_url (void)
{
	gnome_url_show ("http://www.superpimp.org/manual/");
}

static void
send_feedback_cb (GtkWidget * w)
{
	gchar * from = get_default_author_from ();
	gchar * msg_id = gnksa_generate_message_id ("superpimp.org");
	Group * sendlater = folder_get_by_name (PAN_SENDLATER);
	Article * a = article_new (sendlater);

	article_set_header (a, HEADER_SUBJECT, "Pan " VERSION " Feedback", DO_CHUNK);
	article_set_header (a, PAN_MAIL_TO, "pan@superpimp.org", DO_CHUNK);
	article_set_header (a, HEADER_FROM, from, DO_CHUNK);
	article_set_header (a, HEADER_MESSAGE_ID, msg_id, DO_CHUNK);
	message_edit_window (NULL, a);      

	g_free (msg_id);
	g_free (from);
}

static void
dialog_about_cb (void)
{
	pan_lock ();
	dialog_about (Pan.window);
	pan_unlock ();
}

GnomeUIInfo help_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Pan _Homepage"),
		N_("Pan _Homepage"),
		pan_homepage_url, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK,
		GNOME_STOCK_MENU_HOME,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Online Users _Manual"),
		N_("Online Users _Manual"),
		pan_manual_url, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK,
		GNOME_STOCK_PIXMAP_HELP,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Send Feedback"),
		N_("Send Feedback"),
		send_feedback_cb, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK,
		GNOME_STOCK_PIXMAP_MAIL,
		0, 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_ABOUT_ITEM(dialog_about_cb, NULL),
	GNOMEUIINFO_END
};


static gboolean dampen_feedback_sort = FALSE;

static void
sort_cb (GtkRadioMenuItem* item, gpointer data)
{
	if (!dampen_feedback_sort && GTK_CHECK_MENU_ITEM(item)->active)
		articlelist_set_sort_type (GPOINTER_TO_INT(data));
}

static void
thread_articles_cb (GtkCheckMenuItem* item, gpointer data)
{
	articlelist_set_threaded (item->active);
}

static void
rot_13_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_rot13 (item->active);
}

static void
fill_body_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_wrap (item->active);
}

static void
show_all_headers_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_show_all_headers (item->active);
}

static void
mute_quoted_text_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_mute_quoted (item->active);
}

static GnomeUIInfo article_sort_menu_2[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("+ Author"),
		N_("+ Author"),
		sort_cb,
		GINT_TO_POINTER(+ARTICLE_SORT_AUTHOR)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("- Author"),
		N_("- Author"),
		sort_cb,
		GINT_TO_POINTER(-ARTICLE_SORT_AUTHOR)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("+ Date"),
		N_("+ Date"),
		sort_cb,
		GINT_TO_POINTER(+ARTICLE_SORT_DATE)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("- Date"),
		N_("- Date"),
		sort_cb,
		GINT_TO_POINTER(-ARTICLE_SORT_DATE)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("+ Line Count"),
		N_("+ Line Count"),
		sort_cb,
		GINT_TO_POINTER(+ARTICLE_SORT_LINES)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("- Line Count"),
		N_("- Line Count"),
		sort_cb,
		GINT_TO_POINTER(-ARTICLE_SORT_LINES)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("+ Subject"),
		N_("+ Subject"),
		sort_cb,
		GINT_TO_POINTER(+ARTICLE_SORT_SUBJECT)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("- Subject"),
		N_("- Subject"),
		sort_cb,
		GINT_TO_POINTER(-ARTICLE_SORT_SUBJECT)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("+ Unread Count"),
		N_("+ Unread Count"),
		sort_cb,
		GINT_TO_POINTER(+ARTICLE_SORT_UNREAD_CHILDREN)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("- Unread Count"),
		N_("- Unread Count"),
		sort_cb,
		GINT_TO_POINTER(-ARTICLE_SORT_UNREAD_CHILDREN)
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo article_sort_menu[] =
{
	{
		GNOME_APP_UI_RADIOITEMS,
		N_("Sort Articles"),
		N_("Sort Articles"),
		article_sort_menu_2
	},
	GNOMEUIINFO_END
};

static void
article_filter_cb (GtkCheckMenuItem* item, gpointer data)
{
	const guint flag = GPOINTER_TO_UINT(data);
	articlelist_poke_state_filter (flag, item->active);
}

static GnomeUIInfo article_filter_menu[] =
{
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show New Messages"),
		N_("Show New Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_NEW)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Unread Messages"),
		N_("Show Unread Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_UNREAD)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Read Messages"),
		N_("Show Read Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_READ)
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Watched Threads"),
		N_("Show Watched Threads"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_WATCHED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Killfiled Messages"),
		N_("Show Killfiled Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_IGNORED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Others"),
		N_("Show Others"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_NORMAL_RANK)
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Complete Binary Articles"),
		N_("Show Complete Binary Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_COMPLETE_BINARIES)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Incomplete Binary Articles"),
		N_("Show Incomplete Binary Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_INCOMPLETE_BINARIES)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Non-Binary Articles"),
		N_("Show Non-Binary Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_NONBINARIES)
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Saved Articles"),
		N_("Show Saved Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_SAVED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Queued Articles"),
		N_("Show Queued Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_QUEUED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Others"),
		N_("Show Others"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_IDLE)
	},
	GNOMEUIINFO_END
};

static void
zoom_page_cb (gpointer menu, gpointer data)
{
	const guint page = GPOINTER_TO_UINT(data);
	GtkWidget * w = NULL;
	switch (page) {
		case GROUPS_PAGE:      w = Pan.group_clist;   break;
		case ARTICLELIST_PAGE: w = Pan.article_ctree; break;
		case MESSAGE_PAGE:     w = Pan.text;          break;
	}
       	gui_set_layout (GUI_NOTEBOOK);
	gui_page_set (page, w);
}

static GnomeUIInfo view_menu[] =
{
	{
		/* 0 */
		GNOME_APP_UI_ITEM,
		N_("_Zoom/Unzoom Window"),
		N_("Zoom/Unzoom Window"),
		zoom_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'Z', 0, NULL
	},
	{
		/* 1 */
		GNOME_APP_UI_ITEM,
		N_("Zoom the Newsgroup Pane"),
		N_("Zoom the Newsgroup Pane"),
		zoom_page_cb, GUINT_TO_POINTER(GROUPS_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'1', 0, NULL
	},      
	{
		/* 2 */
		GNOME_APP_UI_ITEM,
		N_("Zoom the Message Pane"),
		N_("Zoom the Message Pane"),
		zoom_page_cb, GUINT_TO_POINTER(ARTICLELIST_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'2', 0, NULL
	},      
	{
		/* 3 */
		GNOME_APP_UI_ITEM,
		N_("Zoom the Message Body Pane"),
		N_("Zoom the Message Body Pane"),
		zoom_page_cb, GUINT_TO_POINTER(MESSAGE_PAGE),
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		'3', 0, NULL
	},      
	GNOMEUIINFO_SEPARATOR, /* 4 */
	{
		/* 5 */
		GNOME_APP_UI_SUBTREE,
		N_("Article Filter"),
		N_("Article Filter"),
		&article_filter_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		/* 6 */
		GNOME_APP_UI_SUBTREE,
		N_("Sort Articles"),
		N_("Sort Articles"),
		&article_sort_menu, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR, /* 7 */
	{
		/* 8 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Thread Articles"),
		N_("Thread Articles"),
		thread_articles_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		/* 9 */
		GNOME_APP_UI_ITEM,
		N_("Expand all Threads"),
		N_("Expand all Threads"),
		articlelist_expand_all_threads, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		/* 10 */
		GNOME_APP_UI_ITEM,
		N_("Collapse all Threads"),
		N_("Collapse all Threads"),
		articlelist_collapse_all_threads, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR, /* 11 */
	{
		/* 12 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Rot13 the Message Body"),
		N_("Rot13 the Message Body"),
		rot_13_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		/* 13 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Fill/Wrap the Message Body"),
		N_("Fill/Wrap the Message Body"),
		fill_body_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'\0', 0, NULL
	},
	{
		/* 14 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show All _Headers in the Message Body"),
		N_("Show All Headers in the Message Body"),
		show_all_headers_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'H', 0, NULL
	},
	{
		/* 15 */
		GNOME_APP_UI_TOGGLEITEM,
		N_("Mute Quoted Text"),
		N_("Mute Quoted Text"),
		mute_quoted_text_cb, NULL,
		NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK,
		'Q', 0, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Log Viewer"),
		N_("Open up the Log Viewer."),
		dialog_log_viewer, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
		'L', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Task Manager"),
		N_("Open up the Task Manager."),
		task_manager_spawn, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
		'T', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

static gint
mute_quoted_text_changed_cb (gpointer data,
                             gpointer call,
                             gpointer client)
{
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[15].widget),
	                                call!=NULL);
	return 0;
}

static gint
show_all_headers_changed_cb (gpointer data,
                             gpointer call,
                             gpointer client)
{
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[14].widget),
	                                call!=NULL);
	return 0;
}



GnomeUIInfo group_menu [] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Download..."),
		N_("Download..."),
		grouplist_selected_download_dialog
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Get New Headers in Subscribed Groups"),
		N_("Get New Headers in Subscribed Groups"),
		grouplist_subscribed_download_new,
		NULL, NULL,
		GNOME_APP_PIXMAP_DATA, envelope_new_xpm,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get New Headers in Selected Groups"),
		N_("Get New Headers in Selected Groups"),
		grouplist_selected_download_new
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get All Headers in Selected Groups"),
		N_("Get All Headers in Selected Groups"),
		grouplist_selected_download_all
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Sample Headers from Selected Groups"),
		N_("Sample Headers from Selected Groups"),
		grouplist_selected_download_sample
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Get New Headers & Bodies in Subscribed Groups"),
		N_("Get New Headers & Bodies in Subscribed Groups"),
		grouplist_subscribed_download_new_bodies,
		NULL, NULL,
		GNOME_APP_PIXMAP_DATA, envelope_new_xpm,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get New Headers & Bodies in Selected Groups"),
		N_("Get New Headers & Bodies in Selected Groups"),
		grouplist_selected_download_new_bodies
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get All Headers & Bodies in Selected Groups"),
		N_("Get All Headers & Bodies in Selected Groups"),
		grouplist_selected_download_all_bodies
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Sample Headers & Bodies in Selected Groups"),
		N_("Sample Headers & Bodies in Selected Groups"),
		grouplist_selected_download_sample_bodies
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Refresh Article _Count"),
		N_("Refresh Article Count"),
		grouplist_selected_update_count_info
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Mark Selected Group as Read"),
		N_("Mark Selected Group as Read"),
		grouplist_selected_mark_read
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Subscribe"),
		N_("Subscribe to the selected group(s)"),
		grouplist_selected_subscribe, NULL, NULL,
		GNOME_APP_PIXMAP_DATA, envelope_xpm,
		0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Unsubscribe"),
		N_("Unsubscribe from the selected group(s)"),
		grouplist_selected_unsubscribe
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Delete..."),
		N_("Delete the selected group(s) from Pan."),
		grouplist_selected_destroy
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Empty"),
		N_("Remove all articles from the selected group(s)."),
		grouplist_selected_empty
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Rename..."),
		N_("Rename the selected group."),
		grouplist_selected_rename
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Properties..."),
		N_("Set the properties for the selected group."),
		grouplist_selected_properties, NULL, NULL,
		GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
		0, 0, NULL
	},
	GNOMEUIINFO_END
};

GnomeUIInfo folder_menu [] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("New Folder..."),
		N_("Create a new folder."),
		group_new_folder_dialog
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Rename Folder..."),
		N_("Rename the selected folder."),
		grouplist_selected_rename
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Delete Folder..."),
		N_("Delete the selected folder."),
		grouplist_selected_destroy
	},
	GNOMEUIINFO_END
};


extern GnomeUIInfo articlelist_menu [];

static GnomeUIInfo pan_main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &view_menu),
	GNOMEUIINFO_SUBTREE (N_("_Online"), &online_menu),
	GNOMEUIINFO_SUBTREE (N_("_Post"), &post_menu),
	GNOMEUIINFO_SUBTREE (N_("_Navigate"), &navigate_menu),
	GNOMEUIINFO_SUBTREE (N_("_Group"), &group_menu),
	GNOMEUIINFO_SUBTREE (N_("Fo_lder"), &folder_menu),
	GNOMEUIINFO_SUBTREE (N_("_Messages"), &articlelist_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};


/** 
 ** global variables
 **/

static GtkWidget * appbar = NULL;
static GtkWidget * queue_qty_label = NULL;
static GtkWidget * queue_qty_button = NULL;
static GtkWidget * connection_qty_label = NULL;
static GtkWidget * connection_qty_button = NULL;

guint context_id;

GtkWidget * groups_vbox;
GtkWidget * articlelist_ctree;
GtkWidget * text_box;
GtkWidget * contents_vbox;

/*---[ gui_select_all ]-----------------------------------------------
 * Intelligently handle the Edit->Select All callback so the
 * active pane or page gets all it's items selected, in either layout.
 *--------------------------------------------------------------------*/
void
gui_select_all (void)
{
	if (Pan.viewmode == GUI_PANED)
		gui_paned_select_all ();
	else if (Pan.viewmode == GUI_NOTEBOOK)
		gui_notebook_select_all ();
}

static void
connection_button_clicked (GtkButton *button,
                           gpointer user_data)
{
	prefs_spawn_to_news_connections ();
}

static void
queue_button_clicked (GtkButton *button,
                      gpointer user_data)
{
	task_manager_spawn ();
}

static void
gui_create_appbar (GtkWidget* window)
{
	int i;
	GtkWidget * w;
	GtkWidget * frame;

	pan_lock();

	appbar = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(appbar), GNOME_PAD_SMALL);

	/* connection status */
	connection_qty_label = gtk_label_new ("");
	w = connection_qty_button = gtk_button_new();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Open the Connection Manager"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", connection_button_clicked, NULL);
	gtk_container_add (GTK_CONTAINER(w), connection_qty_label);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(frame), w);
	gtk_box_pack_start (GTK_BOX(appbar), frame, FALSE, FALSE, 0);

	/* task status */
	queue_qty_label = gtk_label_new ("");
	w = queue_qty_button = gtk_button_new();
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Open the Task Manager"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", queue_button_clicked, NULL);
	gtk_container_add (GTK_CONTAINER(w), queue_qty_label);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER(frame), w);
	gtk_box_pack_start (GTK_BOX(appbar), frame, FALSE, FALSE, 0);

	/* remainder */
	taskbar = gtk_table_new (1, VIEW_QTY, TRUE);
	for (i=0; i<VIEW_QTY; ++i) {
		GtkWidget * w = status_item_view_new ();
                views[i].view = STATUS_ITEM_VIEW(w);
		gtk_table_attach (GTK_TABLE(taskbar), w, i, i+1, 0, 1, ~0, ~0, 0, 0);
	}
	gtk_box_pack_start (GTK_BOX(appbar), taskbar, TRUE, TRUE, 0);

	gnome_app_set_statusbar (GNOME_APP(window), appbar);
	pan_unlock();

	gui_set_queue_size (0, 0);
	gui_set_connection_size (0);
}

/*---[ gui_shutdown ]-------------------------------------------------
 * save geometry states into config files
 *--------------------------------------------------------------------*/
static guint
warn_about_tasks_running (void)
{
	guint num_tasks = queue_get_running_task_count ();
	int retval = TRUE;

	if (num_tasks != 0)
	{
		guint button = 0;
		gchar *msg;
		GtkWidget * dialog;

		if (num_tasks == 1)
		{
			msg = g_strdup(
				_("1 task still active or queued.\n"
				  "Are you sure you want to exit Pan?"));
		}
		else
		{
			msg = g_strdup_printf (
				_("%d tasks still active or queued.\n"
				  "Are you sure you want to exit Pan?"),
				  num_tasks);
		}

		dialog = gnome_message_box_new (msg,
		                                GNOME_MESSAGE_BOX_QUESTION,
						GNOME_STOCK_BUTTON_YES,
						GNOME_STOCK_BUTTON_NO,
						NULL);

		g_free (msg);

		button = gnome_dialog_run_and_close (GNOME_DIALOG(dialog));
		if (button == 1) /* NO */
			retval = 0;
	}

	return retval;
}

static void
exit_cb (void)
{
	if (warn_about_tasks_running())
	{
		gui_shutdown ();
		gtk_object_destroy (GTK_OBJECT(Pan.window));
	}
}

static int
window_delete_event_cb (GtkWidget *widget,
			GdkEvent *event,
			gpointer  data)
{
	if (warn_about_tasks_running())
	{
		gui_shutdown ();
		return FALSE; /* causes "destroy" signal to be fired */
	}

	return TRUE;
}

static int
text_fill_body_changed_cb (gpointer call_obj,
                               gpointer call_arg,
                               gpointer user_data)
{
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(view_menu[13].widget),
		call_arg!=NULL);
	return 0;
}


void
gui_shutdown (void)
{
	gint x = 0, y = 0;
	gint width = 640, height = 400;

	/**
	***  Stop updating the UI
	**/

	pan_callback_remove (text_get_fill_body_changed_callback(),
	                     text_fill_body_changed_cb,
	                     NULL);
	pan_callback_remove (status_item_get_active_callback(),
	                     status_item_active_changed_cb,
	                     NULL);
	pan_callback_remove (articlelist_state_filter_changed,
	                     gui_message_state_filter_changed_cb,
	                     NULL);
	pan_callback_remove (text_get_show_all_headers_changed_callback(),
	                     show_all_headers_changed_cb,
	                     NULL);
	pan_callback_remove (text_get_mute_quoted_changed_callback(),
	                     mute_quoted_text_changed_cb,
	                     NULL);
	pan_callback_remove (current_article_changed,
	                     update_menus_cb,
	                     NULL);
	pan_callback_remove (grouplist_group_selection_changed,
	                     update_menus_cb,
	                     NULL);
	pan_callback_remove (articlelist_group_changed,
	                     update_menus_cb,
	                     NULL);
	pan_callback_remove (articlelist_selection_changed,
	                     update_menus_cb,
	                     NULL);
	pan_callback_remove (articlelist_sort_changed,
	                     gui_articlelist_sort_changed_cb,
	                     NULL);
	pan_callback_remove (articlelist_thread_changed,
	                     thread_changed_cb,
	                     NULL);
	pan_callback_remove (text_get_show_all_headers_changed_callback(),
	                     gui_text_show_all_headers_changed_cb,
	                     NULL);
	g_source_remove (gui_refresh_timeout_id);
	gui_refresh_timeout_id = 0;

	/**
	***  Save window/clist positioning
	**/

	gdk_window_get_position (Pan.window->window, &x, &y);
	gdk_window_get_size (Pan.window->window, &width, &height);

	gnome_config_set_bool ("/Pan/General/do_wrap",
	                       text_get_wrap());
	gnome_config_set_bool ("/Pan/Display/Show_All_Headers_In_Body",
	                       text_get_show_all_headers());
	gnome_config_set_bool ("/Pan/Display/Mute_Quoted_text",
	                       text_get_mute_quoted());

	gnome_config_set_int ("/Pan/Geometry/pan_x", x);
	gnome_config_set_int ("/Pan/Geometry/pan_y", y);
	gnome_config_set_int ("/Pan/Geometry/width", width);
	gnome_config_set_int ("/Pan/Geometry/height", height);
	gnome_config_set_int ("/Pan/State/viewmode", Pan.viewmode);
	gnome_config_set_int ("/Pan/State/group_mode", grouplist_get_view_mode());

	if (Pan.viewmode == GUI_NOTEBOOK) {
		gnome_config_set_int ("/Pan/State/page", gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)));
	}
	else if (Pan.viewmode == GUI_PANED) {
		gnome_config_set_int ("/Pan/Geometry/vpaned", articlelist_ctree->allocation.height);
		gnome_config_set_int ("/Pan/Geometry/hpaned", groups_vbox->allocation.width);
	}

	gui_save_column_widths (Pan.group_clist, "group");
	gui_save_column_widths (Pan.article_ctree, "articlelist");
	gnome_config_sync();

	/**
	***  Clear out the UI pieces
	**/

	articlelist_set_group (NULL);
	set_current_article (NULL);       
}


static gint
update_menus_idle (gpointer data)
{
	gint i;
	GnomeUIInfo * menu = NULL;
	const Article * article;
	gboolean have_article;
	gboolean have_current_article;
	const Group * alist_group;
	const Group * glist_group;
	gboolean have_group;
	gboolean have_folder;
	const Group * sendlater;
	gboolean sendlater_empty;
	gboolean posted_here;

	article = articlelist_get_selected_article ();
	have_article = article!=NULL;
	have_current_article = get_current_article()!=NULL;
	posted_here = article!=NULL;

	alist_group = articlelist_get_group ();
	glist_group = grouplist_get_selected_group();
	have_group = alist_group!=NULL || glist_group!=NULL;
	have_folder = have_group && group_is_folder(alist_group?alist_group:glist_group);

	sendlater = folder_get_by_name(PAN_SENDLATER);
	sendlater_empty = sendlater!=NULL && !sendlater->article_qty;

	pan_lock_unconditional ();

	/* update post menu */
	menu = post_menu;
	i = -1;
	++i; /* i is post */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* i is followup */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* i is reply */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* i is reply & followup */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* i is separator */
	++i; /* i is forward by email */
	gtk_widget_set_sensitive (menu[i].widget, have_current_article);
	++i; /* separator */
	++i; /* flush pan.sendlater */
	gtk_widget_set_sensitive (menu[i].widget, !sendlater_empty);
	++i; /* separator */
	++i; /* cancel */
	gtk_widget_set_sensitive (menu[i].widget, posted_here);
	++i; /* supercede */
	gtk_widget_set_sensitive (menu[i].widget, posted_here);


	/* update navigate menu */
	menu = navigate_menu;
	i = -1;
	++i; /* i is next unread group */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* i is next group */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* i is separator */
	++i; /* i is select previous article */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is select next article */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is select next unread article */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is select next unread article body */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is separator */
	++i; /* i is read previous article */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is read next article */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is read next unread article */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is read next unread article body */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is separator */
	++i; /* i is prev thread */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is next thread */
	gtk_widget_set_sensitive (menu[i].widget, have_group);

	/* update group menu */
	menu = group_menu;
	i = -1;
	++i; /* i is download dialog */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* i is separator */
	++i; /* i is new headers for subscribed */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is separator */
	++i; /* new headers + bodies for subscribed */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is separator */
	++i; /* i is refresh article count */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is catch up */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is separator */
	++i; /* i is subscribe */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is unsubscribe */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is separator */
	++i; /* i is delete group */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is clear group */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is rename group */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is group properties */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	
	menu = folder_menu;
	i = -1;
	++i; /* i is new folder */
	gtk_widget_set_sensitive (menu[i].widget, TRUE);
	++i; /* i is rename folder or group */
	gtk_widget_set_sensitive (menu[i].widget, have_group);
	++i; /* i is delete folder */
	gtk_widget_set_sensitive (menu[i].widget, have_folder);

	/* update article_menus */
	menu = articlelist_menu;
	i = -1;
	++i; /* i is read */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* i is read in new window */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* separator */
	++i; /* binary attachments */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* search offsite */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* filter */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* separator */
	++i; /* mark as read */
	++i; /* mark as unread */
	++i; /* download bodies */
	++i; /* folder tool */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* delete */
	++i; /* separator */
	++i; /* select all articles */
	gtk_widget_set_sensitive (menu[i].widget, have_group);


	/* update file menu */
	menu = file_menu;
	i = -1;
	++i; /* save article */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* save binary attachment */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* save binary attachment as */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* open binary attachment */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* separator */
	++i; /* print article */
	gtk_widget_set_sensitive (menu[i].widget, have_article);
	++i; /* separator */
	++i; /* exit */

	pan_unlock_unconditional ();

	/* update the title */
	gui_set_title (alist_group);

	return 0;
}
static int
update_menus_cb (gpointer call_object,
                 gpointer call_arg,
                 gpointer user_data)
{
	pan_lock ();
	gtk_idle_add (update_menus_idle, NULL);
	pan_unlock ();
	return 0;
}

static int
gui_text_show_all_headers_changed_cb (gpointer call_object,
				      gpointer call_arg,
				      gpointer user_data)
{
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(view_menu[14].widget),
		call_arg!=NULL);

	return 0;
}


static int
thread_changed_cb (gpointer call_object,
		   gpointer call_arg,
		   gpointer user_data)
{
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(view_menu[8].widget), 
		call_arg!=NULL);

	return 0;
}

static int
gui_articlelist_sort_changed_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	int sort = GPOINTER_TO_INT(call_arg);
	int i = 0;
	GtkRadioMenuItem* w = NULL;

	switch (sort) {
		case ARTICLE_SORT_AUTHOR: i=0; break;
		case -ARTICLE_SORT_AUTHOR: i=1; break;
		case ARTICLE_SORT_DATE: i=2; break;
		case -ARTICLE_SORT_DATE: i=3; break;
		case ARTICLE_SORT_LINES: i=4; break;
		case -ARTICLE_SORT_LINES: i=5; break;
		case ARTICLE_SORT_SUBJECT: i=6; break;
		case -ARTICLE_SORT_SUBJECT: i=7; break;
		case ARTICLE_SORT_UNREAD_CHILDREN: i=8; break;
		case -ARTICLE_SORT_UNREAD_CHILDREN: i=9; break;
		default: pan_warn_if_reached(); break;
	}

	w = GTK_RADIO_MENU_ITEM(article_sort_menu_2[i].widget);
	dampen_feedback_sort = TRUE;
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(w), TRUE);
	dampen_feedback_sort = FALSE;

	return 0;
}


static gint
gui_message_state_filter_changed_idle (gpointer data)
{
	guint filter = GPOINTER_TO_UINT (data);

	pan_lock_unconditional();

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[0].widget),
		filter & STATE_FILTER_NEW);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[1].widget),
		filter & STATE_FILTER_UNREAD);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[2].widget),
		filter & STATE_FILTER_READ);

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[4].widget),
		filter & STATE_FILTER_WATCHED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[5].widget),
		filter & STATE_FILTER_IGNORED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[6].widget),
		filter & STATE_FILTER_NORMAL_RANK);

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[8].widget),
		filter & STATE_FILTER_COMPLETE_BINARIES);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[9].widget),
		filter & STATE_FILTER_INCOMPLETE_BINARIES);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[10].widget),
		filter & STATE_FILTER_NONBINARIES);

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[12].widget),
		filter & STATE_FILTER_SAVED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[13].widget),
		filter & STATE_FILTER_QUEUED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[14].widget),
		filter & STATE_FILTER_IDLE);

	pan_unlock_unconditional ();

	return 0;
}
static gint
gui_message_state_filter_changed_cb (gpointer call_object,
				     gpointer call_arg,
				     gpointer user_data)
{
	pan_lock ();
	gtk_idle_add (gui_message_state_filter_changed_idle, call_arg);
	pan_unlock ();
	return 0;
}


/*---[ gui_construct ]------------------------------------------------
 * @geometry:
 *
 * initialize and place the basic interface components
 *--------------------------------------------------------------------*/
void
gui_construct (const gchar * geometry)
{
	GtkWidget * w;

	memset (views, '\0', sizeof(ViewStruct)*VIEW_QTY);
	ttips = gtk_tooltips_new();
        cmap = gdk_colormap_get_system ();

	Pan.window = gnome_app_new ("Pan", "Pan");
	gtk_signal_connect (GTK_OBJECT(Pan.window), "delete_event", 
			    GTK_SIGNAL_FUNC (window_delete_event_cb), NULL);
	gtk_signal_connect (GTK_OBJECT(Pan.window), "destroy",
			    GTK_SIGNAL_FUNC (pan_shutdown), NULL);

	gtk_widget_realize (Pan.window);

	/** geometry parsing **/
	/* The following chunk of code will allow command line arguments
	 * to override the config. So if this was invoked as part of a 
	 * session we 'Do the right thing' :)
	 */
	if (geometry != NULL)
	  {
	    gint x, y, w, h;
	    if (gnome_parse_geometry (geometry, &x, &y, &w, &h) )
	      {
		if (x != -1)
		  gtk_widget_set_uposition (Pan.window, x, y);
		if (w != -1)
		  gtk_window_set_default_size (GTK_WINDOW (Pan.window), w, h);
	      }
	    else
	      {
		g_error (_("Unable to parse the geometry string \"%s\""), geometry);
	      }
	  }
	else
	  {                  /* Lets see if the config has a geometry */
	    gint w, h;

	    w = gnome_config_get_int ("/Pan/Geometry/width");
	    h = gnome_config_get_int ("/Pan/Geometry/height");
	    if (w!=0 && h!= 0)
		    gtk_window_set_default_size (GTK_WINDOW (Pan.window), w, h);
	    else
		    gtk_window_set_default_size (GTK_WINDOW (Pan.window), 640, 400);

/* While this is nice for some people, it's typically much better to let
 * the window manager do the position saving and restoring... we have similar
 * code at the bottom of this function...

	    x = gnome_config_get_int ("/Pan/Geometry/pan_x");
	    y = gnome_config_get_int ("/Pan/Geometry/pan_y");
	    if (x != 0 && y != 0)
	      gtk_widget_set_uposition (Pan.window, x, y);
*/
	  }

/* Groups vbox */
	groups_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_set_usize (GTK_WIDGET (groups_vbox), 260, -1);
	gtk_box_pack_start (GTK_BOX (groups_vbox), GTK_WIDGET(grouplist_create()), TRUE, TRUE, 1);
        gtk_signal_connect (GTK_OBJECT (Pan.group_clist), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb),  NULL);

/* Articlelist_ctree */
	articlelist_ctree = gtk_vbox_new (FALSE, 0);
	w = create_articlelist_ctree ();
	gtk_box_pack_start (GTK_BOX (articlelist_ctree), w, TRUE, TRUE, 0);

/* Text */
	text_box = text_create ();

	gui_create_appbar (Pan.window);

	gnome_app_create_menus (GNOME_APP (Pan.window), pan_main_menu);
	gnome_app_install_menu_hints (GNOME_APP (Pan.window), pan_main_menu);

	contents_vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (Pan.window), contents_vbox);

	/* connect signals */
        gtk_signal_connect (GTK_OBJECT (Pan.article_ctree), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb), NULL);

        gui_set_layout (gnome_config_get_int ( "Pan/State/viewmode=1" ));

	pan_callback_add (status_item_get_active_callback(),
	                  status_item_active_changed_cb,
	                  NULL);

	pan_callback_add (text_get_fill_body_changed_callback(),
	                  text_fill_body_changed_cb,
	                  NULL);

	/* listen for changes to the articlelist "state filter"
	   so that we can update the menus accordingly */
	pan_callback_add (articlelist_state_filter_changed,
			  gui_message_state_filter_changed_cb,
			  NULL);

	/* listen for changes to the articlelist active group
	   so that we can sensitize/desensitize menus accordingly */
	update_menus_cb (NULL, NULL, NULL);
	pan_callback_add (text_get_show_all_headers_changed_callback(),
	                  show_all_headers_changed_cb,
	                  NULL);
	pan_callback_add (text_get_mute_quoted_changed_callback(),
	                  mute_quoted_text_changed_cb,
	                  NULL);
	pan_callback_add (current_article_changed,
			  update_menus_cb,
	                  NULL);
	pan_callback_add (grouplist_group_selection_changed,
			  update_menus_cb,
			  NULL);
	pan_callback_add (articlelist_group_changed,
			  update_menus_cb,
			  NULL);
	pan_callback_add (articlelist_selection_changed,
			  update_menus_cb,
			  NULL );
	pan_callback_add (articlelist_sort_changed,
			  gui_articlelist_sort_changed_cb,
			  NULL);

	thread_changed_cb (NULL, GINT_TO_POINTER(1), NULL);
	pan_callback_add (articlelist_thread_changed,
			  thread_changed_cb,
			  NULL);


	gui_text_show_all_headers_changed_cb (NULL,
	                                      GINT_TO_POINTER(text_get_show_all_headers()),
					      NULL);
	pan_callback_add (text_get_show_all_headers_changed_callback(),
			  gui_text_show_all_headers_changed_cb,
			  NULL );

	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(view_menu[13].widget),
	                                text_get_wrap());

	text_set_show_all_headers (gnome_config_get_bool("/Pan/Display/Show_All_Headers_In_Body=false"));

	text_set_mute_quoted (gnome_config_get_bool("/Pan/Display/Mute_Quoted_Text=false"));

	gui_set_title (NULL);

	gui_refresh_timeout_id = gtk_timeout_add (2192, gui_refresh_timeout_cb, NULL); 
}



/*---[ gui_layout ]---------------------------------------------------
 * main menu callback
 * gui_set_layout (layout)
 *--------------------------------------------------------------------*/

void
gui_set_layout (int new_layout)
{
	switch (new_layout)
	{
		case GUI_PANED:
			gui_paned_construct ();
			break;

		case GUI_NOTEBOOK:
			gui_notebook_construct ();
			break;
	}

	current_layout = new_layout;
}


/*---[ gui_page_set ]-------------------------------------------------
 * flip to a different notebook page, setting keyboard focus on a 
 * widget in the new page
 *--------------------------------------------------------------------*/
void
gui_page_set (int page, GtkWidget *focus_item)
{
	if (Pan.viewmode == GUI_NOTEBOOK)
		gui_notebook_page_set (page, focus_item);
}


/*---[ gui_page_change ]----------------------------------------------
 * flip to a differnt notebook page, setting keyboard focus in new page.  
 * This is different from gui_page_set because it flips the page based on 
 * a offset, like -1 or 1, to flip back one page or forward one.  It is 
 * automatic in handling the focus grabbing too.
 *--------------------------------------------------------------------*/
static void
gui_page_change (int change)
{
	int cur_page;

	if (Pan.viewmode == GUI_PANED)
		cur_page = gui_paned_get_current_pane ();
	if (Pan.viewmode == GUI_NOTEBOOK)
		cur_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
	else
		return;

	cur_page += change;
	if ((cur_page < 0) || (cur_page > 2)) 
		return;

	if (change < 0) /* Escape was pressed */
	{
		switch (cur_page)
		{
			case GROUPS_PAGE:
				pan_lock ();
				gtk_widget_grab_focus (Pan.group_clist);
				pan_unlock ();
				break;

			case ARTICLELIST_PAGE:
				pan_lock ();
				gtk_widget_grab_focus (Pan.article_ctree);
				pan_unlock ();
				break;

			case MESSAGE_PAGE:
				pan_lock ();
				gtk_widget_grab_focus (Pan.text);
				pan_unlock ();
				break;

			default:
				break;
		}
	}
	else /* enter key hit */
	{
		/* FIXME: segfaults if selection is zero, but when user hits
		 * escape, it automatically sends a "undo_selection" signal,
		 * deselecting all */

		switch (cur_page)
		{
			case GROUPS_PAGE:
				gtk_widget_grab_focus (Pan.group_clist);
				break;

			case ARTICLELIST_PAGE:
			{
				/* update the articlelist */
				Group * g = grouplist_get_selected_group ();
				if (g != NULL)
				{
					articlelist_set_group_thread (g);

					pan_lock ();
					gtk_widget_grab_focus (Pan.article_ctree);
					pan_unlock ();
				}
				break;
			}

			case MESSAGE_PAGE:
			{
				Article * a = articlelist_get_selected_article();
				if (a != NULL)
				{
					text_set_from_article (a, NNTP_READ, FALSE);

					pan_lock ();
					gtk_widget_grab_focus (Pan.text);
					pan_unlock ();
				}
				break;
			}

			default:
				break;
		}
	}

	if (Pan.viewmode == GUI_NOTEBOOK)
		gtk_notebook_set_page (GTK_NOTEBOOK (notebook), cur_page);
}

static void
gui_enter_clicked (void)
{
	if (Pan.viewmode == GUI_NOTEBOOK)
		gui_page_change (1);
	else if (Pan.viewmode == GUI_PANED)
		gui_paned_enter_clicked ();
}


/*---[ gui_set_title ]------------------------------------------------
 * @group:
 *
 * set Pan's title to reflect the current open group
 *--------------------------------------------------------------------*/

static gint
gui_set_title_idle (gpointer data)
{
	const Group * group = GROUP(data);
	GString * s = g_string_new (NULL);
	const gchar * servername = NULL;
	const gchar * groupname = NULL;

	/* look for active server & group */
	if (group == NULL) {
		Server * server = grouplist_get_server ();
		if (server != NULL)
			servername = server->name;
	} else {
		groupname = group_get_readable_name (group);
		servername = group->server->name;
	}

	/* build the title */
	g_string_sprintf (s, _("Pan %s "), VERSION);
	if (servername != NULL)
		g_string_sprintfa (s, _(" - %s"), servername);
	if (groupname != NULL)
		g_string_sprintfa (s, _(" - %s"), groupname);

	/* set the title in the UI */
	pan_lock_unconditional ();
	gtk_window_set_title (GTK_WINDOW (Pan.window), s->str);
	pan_unlock_unconditional ();

	/* cleanup */
	g_string_free (s, TRUE);

	return 0;
}
void
gui_set_title (const Group * group)
{
	pan_lock ();
	gtk_idle_add (gui_set_title_idle, (gpointer)group);
	pan_unlock ();
}


/*---[ gui_key_press_cb ]---------------------------------------------
 * key press callback, when a key is pressed in the group list, article
 * list, or in the message view.  This allows better keyboard navigation
 * and the potential for new key bindings to imitate Ag*nt.
 *--------------------------------------------------------------------*/
static gboolean
gui_key_press_cb (GtkWidget      * widget,
                  GdkEventKey    * event,
                  gpointer         data)
{
	switch (event->keyval)
	{
		case GDK_Q:
		case GDK_q:
			gui_page_change (-1);
			break;

		case GDK_Delete:
			articlelist_selected_delete ();
			break;

		case GDK_Return:
			gui_enter_clicked ();
			break;

		case GDK_space:
		case GDK_KP_Space:
			text_read_more ();
			break;

		default:
			break;
	}

	return TRUE;
}


/*---[ gui_restore_column_widths ]------------------------------------
 * @clist:
 * @type:
 *
 * 
 *--------------------------------------------------------------------*/
void
gui_restore_column_widths (GtkWidget     * clist,
                           const gchar   * type)
{
	int i;
	GtkCList * list = GTK_CLIST (clist);
	const int cols = list->columns;
	int * widths = g_new (int, cols);
	GString * buf = g_string_new (NULL);

	/* get width from config... */
	for (i=0; i!=cols; ++i) {
		g_string_sprintf (buf, "/Pan/Geometry/%s_column_%d", type, i);
		widths[i] = gnome_config_get_int (buf->str);
	}

	/* set ui.. */
	pan_lock();
	for (i=0; i!=cols; ++i)
		if (widths[i] != 0)
			gtk_clist_set_column_width (list, i, widths[i]);
	pan_unlock();

	/* cleanup */
	g_string_free (buf, TRUE);
	g_free (widths);
}

void
gui_save_column_widths (GtkWidget    * clist,
                        const gchar  *type)
{
	int i;
	int cols = GTK_CLIST (clist)->columns;
	GtkCList * list = GTK_CLIST (clist);
	GString * buf = g_string_new (NULL);
	
	for (i=0; i<cols; i++)
	{
		g_string_sprintf (buf, "/Pan/Geometry/%s_column_%d", type, i);
		gnome_config_set_int (buf->str, (int)list->column[i].width);
	}

	g_string_free (buf, TRUE);
	gnome_config_sync ();
}

/*****
******
*****/

static int connection_size = -1;

static gboolean _is_connecting = FALSE;

static void
gui_refresh_connection_label_nolock (void)
{
	char str[128];

	if (connection_size != 0)
		sprintf (str, _("%d @ %.1f KBps"), connection_size, KBps);
	else if (_is_connecting)
		sprintf (str, _("Connecting"));
	else
		sprintf (str, _("Offline"));

	if (GTK_IS_LABEL(connection_qty_label))
		gtk_label_set_text (GTK_LABEL(connection_qty_label), str);
}

void
gui_set_connecting_flag (gboolean is_connecting)
{
	if (_is_connecting != is_connecting)
	{
		_is_connecting = is_connecting;

		pan_lock ();
		gui_refresh_connection_label_nolock ();
		pan_unlock ();
	}
}

void
gui_set_connection_size (int size)
{
	if (size != connection_size)
	{
		connection_size = size;

		pan_lock ();
		gui_refresh_connection_label_nolock ();
		pan_unlock ();
	}
}

void
gui_inc_connection_size (int inc)
{
	gui_set_connection_size (connection_size + inc);
}

static int
gui_refresh_timeout_cb (gpointer user_data)
{
	/* update KBps */
	if (1)
	{
		static struct timeval last_time;
		static unsigned long last_KB = 0;
		static gboolean last_time_inited = FALSE;

		if (!last_time_inited)
		{
			gettimeofday (&last_time, NULL);
			last_time_inited = TRUE;
		}
		else
		{
			double time_diff;
			double KB_diff;
			struct timeval new_time;
			unsigned long new_KB;

			gettimeofday (&new_time, NULL);
			time_diff = (new_time.tv_sec - last_time.tv_sec) * 1000000.0;
			time_diff += (new_time.tv_usec - last_time.tv_usec);
			last_time = new_time;

			new_KB = pan_socket_get_total_xfer_K ();
			KB_diff = new_KB - last_KB;
			last_KB = new_KB;

			KB_diff *= 1000000.0;
			KBps = KB_diff / time_diff;
		}
	}

	pan_lock_unconditional ();
	gui_refresh_connection_label_nolock ();
	pan_unlock_unconditional ();

	return 1;
}

/*****
******
*****/

void
gui_set_queue_size (int running, int size)
{
	char str[128];

	if (size == 0)
		sprintf (str, _("No Tasks"));
	else
		sprintf (str, _("Tasks: %d/%d"), running, size);

	pan_lock();
	gtk_label_set_text (GTK_LABEL(queue_qty_label), str);
	pan_unlock();
}

static void
gui_add_status_item (StatusItem * item)
{
	int i;

	pan_lock ();

	for (i=0; i<VIEW_QTY; ++i)
		if (views[i].status_item == NULL)
			break;
	if (i!=VIEW_QTY) {
		views[i].status_item = item;
		status_item_view_set_item (STATUS_ITEM_VIEW(views[i].view), item);
	}

	pan_unlock ();
}

static void
gui_remove_status_item (StatusItem * item)
{
	int i;

	pan_lock ();

	for (i=0; i<VIEW_QTY; ++i)
		if (views[i].status_item == item)
			break;
	if (i!=VIEW_QTY) {
		StatusItemView * view = views[i].view;
		status_item_view_set_item (STATUS_ITEM_VIEW(view), NULL);
		views[i].status_item = NULL;
	}

	pan_unlock();
}

static int
status_item_active_changed_cb (gpointer call_obj,
                               gpointer call_arg,
                               gpointer user_data)
{
	StatusItem * item = STATUS_ITEM(call_obj);
	gboolean is_active = GPOINTER_TO_INT(call_arg) != 0;
	if (is_active)
		gui_add_status_item (item);
	else
		gui_remove_status_item (item);
	return 0;
}

/*****
******
*****/

void
widget_set_font (GtkWidget * w, const char * font_name)
{
        GtkStyle* style = NULL;

        g_return_if_fail (w!=NULL);

        style = gtk_style_copy (gtk_widget_get_style(Pan.window));
	if (is_nonempty_string(font_name))
	{
        	GdkFont * font = use_gdk_fontset_load
			? gdk_fontset_load (font_name)
			: gdk_font_load (font_name);

        	if (font != NULL)
			style->font = font;
		else
			g_warning (_("Couldn't load font \"%s\""), font_name);
	}
        gtk_widget_set_style (w, style);
}
