/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>

#include "debug.h"
#include "newsrc.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	gulong low;
	gulong high;
}
Range;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

static void
parse_read_str (const char * str,
                GArray * setme_read)
{
	const char * pch = str;

	g_return_if_fail (str!=NULL);

	if (setme_read != NULL)
	{
		gboolean error = FALSE;
		while (*pch && !error)
		{
			gulong high;
			gulong low = strtol (pch, (char**)&pch, 10);
			switch (*pch)
			{
				case ',' : /* single article */
					high = low;
					++pch;
					break;
				case '\0' : /* single article at end of line */
					high = low;
					break;
				case '-' : /* start of a range */
					++pch;
					high = strtol (pch, (char**)&pch, 10);
					if (*pch)
						++pch;
					break;
				default: /* some invalid range? */
					high = low;
					error = TRUE;
					pan_warn_if_reached ();
					g_message ("the line was [%s] end is [%s]", str, pch);
					break;
			}

			if (!error)
			{
				Range range;
				range.low = low;
				range.high = high;
				g_array_append_val (setme_read, range);
			}
		}
	}
}

/*****
******
*****/

/**
 * This makes sure that n->read exists.  If it doesn't already exist,
 * it's created based on the information found in n->read_str.
 */
static void
init_read (Newsrc * n)
{
	g_return_if_fail (n != NULL);
	g_return_if_fail (n->read == NULL);

	n->read = g_array_new (FALSE, FALSE, sizeof(Range));
	parse_read_str (n->read_str, n->read);
}

static int
compare_pulong_to_pRange (const void * a, const void * b)
{
	gulong number = *(gulong *)a;
	const Range * range = (const Range*) b;
	int retval;

	if (number < range->low)
		retval = -1;
	else if (number > range->high)
		retval = 1;
	else
		retval = 0;

	return retval;
}

static gboolean
newsrc_mark_article_read (Newsrc * n, gulong number)
{
	int i;
	GArray * a;
	gboolean exact_match = FALSE;
	Range * next = NULL;
	Range * prev = NULL;
	gboolean merge_next = FALSE;
	gboolean merge_prev = FALSE;

	g_return_val_if_fail (n!=NULL, FALSE);

	if (n->read == NULL)
		init_read (n);


	a = n->read;
	i = lower_bound (&number,
	                 a->data,
	                 a->len,
	                 sizeof(Range),
	                 compare_pulong_to_pRange,
	                 &exact_match);

	if (exact_match) /* already read */
		return TRUE;

	if (i>0) { /* if it's adjacent to prev, merge */
		prev = &g_array_index (a, Range, i-1);
		if (prev->high == number-1) {
			prev->high = number;
			merge_prev = TRUE;
		}
	}
	if (a->len!=0 && i<a->len) { /* if it's adjacent to next, merge */
		next = &g_array_index (a, Range, i);
		if (next->low == number+1) {
			next->low = number;
			merge_next = TRUE;
		}
	}

	if (merge_prev && merge_next) {
		next->low = prev->low;
		g_array_remove_index (a, i-1);
	}

	/* if it's not absorbed by one of the read
	   near it, add it as a new range. */
	if (!merge_prev && !merge_next) {
		Range r;
		r.low = r.high = number;
		g_array_insert_val (a, i, r);
	}

	return FALSE;
}

static gboolean
newsrc_mark_article_unread (Newsrc * n, gulong number)
{
	int i;
	GArray * a;
	gboolean exact_match = FALSE;
	Range * range;

	g_return_val_if_fail (n!=NULL, FALSE);

	if (n->read == NULL)
		init_read (n);

	a = n->read;
	i = lower_bound (&number,
	                 a->data,
	                 a->len,
	                 sizeof(Range),
	                 compare_pulong_to_pRange,
	                 &exact_match);

	if (!exact_match) /* wasn't read anyway */
		return FALSE;

	range = &g_array_index (a, Range, i);
	if (range->low == range->high) /* simple -- just remove the range */
	{
		g_array_remove_index (a, i);
	}
	else if (number == range->low) /* snip lower bound */
	{
		++range->low;
	}
	else if (number == range->high) /* snip upper bound */
	{
		--range->high;
	}
	else /* break the range into two pieces */
	{
		Range r;
		r.low = range->low;
		r.high = number - 1;
		g_array_insert_val (a, i, r);
		range->low = number + 1;
	}

	return TRUE;
}


/************
*************  PROTECTED
************/

void
newsrc_constructor (Newsrc * n,
                    const char * read_str,
                    gulong low,
                    gulong high)
{
	/* construct parent */
	pan_object_constructor (PAN_OBJECT(n), newsrc_destructor);

	/* construct this class' bits */
	debug1 (DEBUG_PAN_OBJECT, "newsrc_ctor: %p", n);
	n->group_low = 0;
	n->group_high = 0;
	n->read = NULL;
	n->read_str = NULL;
	newsrc_init (n, read_str, low, high);
}

void
newsrc_destructor (PanObject * obj)
{
	Newsrc * n = NEWSRC (obj);

	debug1 (DEBUG_PAN_OBJECT, "newsrc_dtor: %p", n);
	g_free (n->read_str);
	if (n->read != NULL)
		g_array_free (n->read, TRUE);

	pan_object_destructor (obj);
}

/************
*************  PUBLIC
************/

void
newsrc_init (Newsrc * n, const char * read_str, gulong low, gulong high)
{
	n->group_low = low;
	n->group_high = high;
	replace_gstr (&n->read_str, g_strdup (read_str?read_str:""));
	g_strstrip (n->read_str);
}

Newsrc*
newsrc_new (const char * read_str, gulong low, gulong high)
{
	Newsrc * e = g_new (Newsrc, 1);
	newsrc_constructor (e, read_str, low, high);
	return e;
}

/*****
******
*****/

void
newsrc_set_group_range (Newsrc * e, gulong low, gulong high)
{
	g_return_if_fail (e != NULL);

	e->group_low = low;
	e->group_high = high;
}

/*****
******
*****/

void
newsrc_mark_all (Newsrc * n, gboolean read)
{
	if (n->read == NULL)
		init_read (n);

	/* clear the ranges, effectively marking all unread... */
	g_array_set_size (n->read, 0);

	if (read)
	{
		/* create an all-inclusive range, marking all read... */
		Range range;
		range.low = 0;
		range.high = n->group_high;
		g_array_append_val (n->read, range);
	}
}

gboolean
newsrc_is_article_read (const Newsrc * n, gulong article_number)
{
	gboolean exact_match = FALSE;

	g_return_val_if_fail (n!=NULL, FALSE);
	g_return_val_if_fail (article_number!=0, FALSE);

	if (n->read == NULL)
		init_read ((Newsrc*)n);

	lower_bound (&article_number,
	             n->read->data,
	             n->read->len,
	             sizeof(Range),
	             compare_pulong_to_pRange,
	             &exact_match);
	
	return exact_match;
}

/*****
******
*****/

gboolean
newsrc_mark_article (Newsrc * n, gulong number, gboolean read)
{
	gboolean retval;

	g_return_val_if_fail (n!=NULL, FALSE);

	retval = read
		? newsrc_mark_article_read (n, number)
		: newsrc_mark_article_unread (n, number);

	return retval;
}

/*****
******
*****/

void
newsrc_import (Newsrc * n, gboolean * setme_subscribed, const char * import)
{
	const char * pch = import;
	gulong low, high;
	GArray * a;

	g_return_if_fail (n != NULL);
	g_return_if_fail (setme_subscribed != NULL);
	g_return_if_fail (import != NULL);

	/* subscription */
	while (*pch && *pch!=' ') ++pch;
	--pch;
	*setme_subscribed = *pch==':';

	/* article numbers */
	++pch;
	while (*pch==' ') ++pch;
	replace_gstr (&n->read_str, g_strdup(pch));
	g_strstrip (n->read_str);
	pch = NULL;

	/* remove the old ranges & add the new */
	if (n->read != NULL)
		g_array_free (n->read, TRUE);
	a = n->read = g_array_new (FALSE, FALSE, sizeof(Range));
	parse_read_str (n->read_str, n->read);

	/* update the low/high based on the imported ranges */
	low = high = 0;
	if (a->len != 0) {
		low = g_array_index(a,Range,0).low;
		high = g_array_index (a, Range, a->len-1).high;
	}
	newsrc_set_group_range (n, low, high);
}

/*****
******
*****/

gchar*
newsrc_get_read_str (const Newsrc * n)
{
	gchar * retval = NULL;

	if (n!=NULL && n->read!=NULL) /* someone's been changing the read/unread */
	{
		int i;
		GString * str = g_string_new (NULL);

		for (i=0; i<n->read->len; ++i)
		{
			const Range * range = &g_array_index (n->read, Range, i);

			/* if this range is too old, don't bother writing it */
			if (range->high < n->group_low)
				continue;

			g_string_sprintfa (str, "%lu", range->low);

			if (range->low != range->high)
				g_string_sprintfa (str, "-%lu", range->high);

			if (i != n->read->len-1)
				g_string_append_c (str, ',');
		}

		retval =  str->str;
		g_string_free (str, FALSE);
	}
	else if (n!=NULL && is_nonempty_string(n->read_str)) /* just use the fields passed in */
	{
		retval = g_strdup (n->read_str);
	}

	return retval;
}

void
newsrc_export (const Newsrc  * n,
               const char    * group_name,
               gboolean        subscribed,
	       GString       * appendme)
{
	gchar * buf;

	g_return_if_fail (is_nonempty_string(group_name));

	g_string_append (appendme, group_name);
	g_string_append_c (appendme, subscribed ? ':' : '!');
	g_string_append_c (appendme, ' ');

	buf = NULL;
	if (n != NULL)
		buf = newsrc_get_read_str (n);
	if (buf != NULL) {
		g_string_append (appendme, buf);
		g_free (buf);
	}
}
