/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "bozo.h"
#include "debug.h"
#include "fnmatch.h"
#include "group.h"
#include "rule.h"
#include "thread-watch.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

/*****
******
*****/

/*****
******
*****/

/************
*************  PROTECTED
************/

void
rule_criteria_constructor (RuleCriteria * r)
{
	g_return_if_fail (r!=NULL);

	pan_object_constructor (PAN_OBJECT(r), rule_criteria_destructor);
	debug1 (DEBUG_PAN_OBJECT, "rule_criteria_ctor: %p", r);

	r->type = 0;
	r->not = FALSE;
	r->children = NULL;
	r->phrase_type = 0;
	r->phrase_does = FALSE;
	r->phrase_is_regex = FALSE;
	r->phrase = NULL;
	r->exceeds_n = 0;
	r->thread_type = 0;
	r->marked_type = 0;
	r->binary_type = 0;
}

void
rule_criteria_destructor (PanObject * obj)
{
	RuleCriteria * r = RULE_CRITERIA(obj);

	debug1 (DEBUG_PAN_OBJECT, "rule_criteria_dtor: %p", r);

	/* free the bits of the children */
	if (r->children != NULL) {
		pan_g_ptr_array_foreach (r->children, (GFunc)pan_object_unref, NULL);
		g_ptr_array_free (r->children, TRUE);
		r->children = NULL;
	}

	/* free the bits of this */
	rule_criteria_set_phrase (r, NULL, FALSE, FALSE);

	/* free the superclass */
	pan_object_destructor (obj);
}

/************
*************  PUBLIC
************/

RuleCriteria*
rule_criteria_new (void)
{
	RuleCriteria * r = g_new0 (RuleCriteria, 1);
	rule_criteria_constructor (r);
	return r;
}

/*****
******
*****/

RuleCriteria*
rule_criteria_dup (const RuleCriteria * src)
{
	RuleCriteria * tgt;

	g_return_val_if_fail (src!=NULL, NULL);

	tgt = rule_criteria_new ();

	/* duplicate this object's bits */
	tgt->type = src->type;
	tgt->not = src->not;
	tgt->phrase_type = src->phrase_type;
	tgt->phrase_does = src->phrase_does;
	tgt->phrase_is_regex = src->phrase_is_regex;
	tgt->phrase = g_strdup (src->phrase);
	tgt->exceeds_n = src->exceeds_n;
	tgt->thread_type = src->thread_type;
	tgt->marked_type = src->marked_type;
	tgt->binary_type = src->binary_type;

	/* duplicate the children */
	if (src->children != NULL) {
		guint i;
		tgt->children = g_ptr_array_new ();
		pan_g_ptr_array_reserve (tgt->children, src->children->len);
		for (i=0; i!=src->children->len; ++i) {
			RuleCriteria * child = RULE_CRITERIA (
				g_ptr_array_index(src->children,i));
			g_ptr_array_add (tgt->children,
			                 rule_criteria_dup(child));
		}
	}

	return tgt;
}

/*****
******
*****/

void
rule_criteria_remove_failures (const RuleCriteria * c,
                               GPtrArray * articles)
{
	guint i;

	g_return_if_fail (c!=NULL);
	g_return_if_fail (articles!=NULL);

	for (i=0; i!=articles->len; )
	{
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		gboolean pass = rule_criteria_test_article (c, a);
		if (pass)
			++i;
		else
			g_ptr_array_remove_index_fast (articles, i);
	}
}


gboolean
rule_criteria_test_article (const RuleCriteria * r,
                            const Article * a)
{
	gboolean retval;

	g_return_val_if_fail (r!=NULL, FALSE);
	g_return_val_if_fail (a!=NULL, FALSE);

	switch (r->type)
	{
		case RULE_CRITERIA_AND:
		{
			guint i;
			retval = TRUE;
			if (r->children==NULL || !r->children->len) /* ?? */
				retval = FALSE;
			else for (i=0; r->children!=NULL && i!=r->children->len; ++i) {
				RuleCriteria * child = RULE_CRITERIA(g_ptr_array_index(r->children,i));
				if (!rule_criteria_test_article (child,a)) {
					retval = FALSE;
					break;
				}
			}
			break;
		}
		case RULE_CRITERIA_OR:
		{
			guint i;
			retval = FALSE;
			for (i=0; r->children!=NULL && i!=r->children->len; ++i) {
				RuleCriteria * c = RULE_CRITERIA(g_ptr_array_index(r->children,i));
				if (rule_criteria_test_article (c,a)) {
					retval = TRUE;
					break;
				}
			}
			break;
		}
		case RULE_CRITERIA_PHRASE:
		{
			/* find what arcticle field we need to check... */
			gchar * checkme;
			switch (r->phrase_type) {
				case RULE_CRITERIA_PHRASE_SUBJECT:
					checkme = g_strdup (a->subject);
					break;
				case RULE_CRITERIA_PHRASE_AUTHOR:
					checkme = article_get_author_str (a);
					break;
				default:
					pan_warn_if_reached ();
					checkme = g_strdup("");
					break;
			}

			if (r->phrase_is_regex)
			{
				gint val = regexec (&r->phrase_regex, checkme, 0, NULL, 0);
				if (!r->phrase_does)
					val = !val;
				retval = !val;
			}
			else
			{
				/* case-insensitive substring search */
				gchar * tmp = g_strdup_printf ("*%s*", r->phrase); 
				gboolean pass = !fnmatch(tmp,checkme,PAN_CASEFOLD);
				g_free (tmp);
				if (!r->phrase_does)
					pass = !pass;
				retval = pass;
			}

			g_free (checkme);
			break;
		}
#if 0
		case RULE_CRITERIA_IS_REPLY_TO_MINE:
		{
			retval = FALSE;
			break;
		}
#endif
		case RULE_CRITERIA_EXCEEDS_N_LINES:
		{
			retval = a->linecount > r->exceeds_n;
			break;
		}
		case RULE_CRITERIA_EXCEEDS_N_GROUPS:
		{
			retval = a->crosspost_qty > r->exceeds_n;
			break;
		}
		case RULE_CRITERIA_EXCEEDS_N_DAYS_OLD:
		{
			const time_t now = time(0);
			const time_t then = a->date;
			const time_t secs_since_posting = now - then;
			const int secs_in_day = 24 * 60 * 60;
			const int days_since_posting = secs_since_posting / secs_in_day;
			retval = days_since_posting > r->exceeds_n;
			break;
		}
		case RULE_CRITERIA_THREAD:
		{
			const gint state = thread_get_state (a);
			switch (r->thread_type)
			{
				case RULE_CRITERIA_THREAD_WATCHED:
					retval = state == THREAD_WATCHED;
					break;
				case RULE_CRITERIA_THREAD_IGNORED:
					retval = state == THREAD_IGNORED;
					break;
				default:
					pan_warn_if_reached ();
					retval = FALSE;
					break;
			}
			break;
		}
		case RULE_CRITERIA_BINARY:
		{
			switch (r->binary_type) {
				case RULE_CRITERIA_BINARY_COMPLETE:
					retval = article_flag_on (a, STATE_MULTIPART_ALL);
					break;
				case RULE_CRITERIA_BINARY_INCOMPLETE:
					retval = article_flag_on (a, STATE_MULTIPART_SOME);
					break;
				case RULE_CRITERIA_BINARY_NONBINARY:
					retval = !article_flag_on(a,STATE_MULTIPART_ALL) &&
					         !article_flag_on(a,STATE_MULTIPART_SOME);
					break;
				default:
					pan_warn_if_reached ();
					retval = FALSE;
			}
			break;
		}
		case RULE_CRITERIA_MARKED_AS:
		{
			switch (r->marked_type) {
				case RULE_CRITERIA_MARKED_READ:
					retval = article_is_read (a);
					break;
				case RULE_CRITERIA_MARKED_UNREAD:
					retval = !article_is_read (a);
					break;
				default:
					pan_warn_if_reached ();
					retval = FALSE;
			}
			break;
		}
		case RULE_CRITERIA_BOZO:
		{
			retval = is_bozo (a->author_addr, a->author_real);
			break;
		}
		default:
		{
			pan_warn_if_reached ();
			retval = FALSE;
		}
	}

	if (r->not)
		retval = !retval;

	return retval;
}

/*****
******
*****/

void
rule_criteria_set_phrase (RuleCriteria * r,
                          const gchar * phrase,
                          gboolean phrase_does,
                          gboolean is_regex)
{
	g_return_if_fail (r!=NULL);

	/* clear out the old */
	r->phrase_does = FALSE;
	if (r->phrase != NULL) {
		g_free (r->phrase);
		r->phrase = NULL;
	}
	if (r->phrase_is_regex) {
		r->phrase_is_regex = FALSE;
		regfree (&r->phrase_regex);
	}

	/* set the new */
	if (is_nonempty_string (phrase)) {
		r->phrase_does = phrase_does;
		r->phrase = g_strdup (phrase);
		if (is_regex) {
			regcomp (&r->phrase_regex, r->phrase, REG_ICASE);
			r->phrase_is_regex = TRUE;
		}
	}
}

/****
*****
****/

void
rule_criteria_add_child (RuleCriteria * parent,
                         RuleCriteria * child)
{
	g_return_if_fail (parent!=NULL);
	g_return_if_fail (child!=NULL);

	if (!parent->children)
		parent->children = g_ptr_array_new ();

	g_ptr_array_add (parent->children, child);
}

/****
*****
*****   RuleCriteria stringifying / parsing
*****
****/

static gchar*
get_depth_string (int depth)
{
	gchar * s;
	depth *= 3;
	s = g_malloc (depth+1);
	memset (s, (int)' ', depth);
	s[depth] = '\0';
	return s;
}

static int
line_get_depth (const gchar * pch)
{
	const gchar * tmp = pch;
	while (*tmp && *tmp==' ') ++tmp;
	return (tmp-pch)/3;
}

RuleCriteria*
string_to_rule_criteria_single (const gchar * line,
                                gint * setme_depth)
{
	RuleCriteria * r = rule_criteria_new ();
	gchar * not = _("NOT ");
	gchar * work = g_strdup (line);
	gchar * const freeme = work;

	if (setme_depth != NULL)
		*setme_depth = line_get_depth (line);

	g_strstrip (work);

	if (!strncmp (work, not, strlen(not)))
	{
		r->not = TRUE;
		work += strlen(not);
	}

	if (!strcmp (work, _("ALL of:")))
	{
		r->type = RULE_CRITERIA_AND;
	}
	else if (!strcmp (work, _("ANY of:")))
	{
		r->type = RULE_CRITERIA_OR;
	}
	else if (strstr(work, _("contains"))!=NULL || strstr(work,_("does not contain"))!=NULL)
	{
		gchar * phrase;
		gchar * field;
		const gchar * march = work;
		gboolean is_regex;
		gboolean phrase_does;
		gint phrase_type;

		r->type = RULE_CRITERIA_PHRASE;

		/* determine what field is being searched */
		field = get_next_token_str (march, ' ', &march);
		if (!strcmp (field, _("subject")))
			phrase_type = RULE_CRITERIA_PHRASE_SUBJECT;
		else if (!strcmp (field, _("author")))
			phrase_type = RULE_CRITERIA_PHRASE_AUTHOR;
		else {
			phrase_type = RULE_CRITERIA_PHRASE_AUTHOR;
			pan_warn_if_reached ();
		}
		g_free (field);

		is_regex = strstr (work, _("reg. expr.")) != NULL;
		phrase_does = strstr (work, _("does not contain")) == NULL;
		march = strchr (work, '"') + 1;
		phrase = g_strdup (march);
		*strrchr(phrase, '"') = '\0';

		/* update the rule criteria */
		rule_criteria_set_phrase (r, phrase, phrase_does, is_regex);
		r->phrase_type = phrase_type;

		/* cleanup */
		g_free (phrase);
	}
	else if (!strcmp (work, _("binary attachments")))
	{
		r->type = RULE_CRITERIA_BINARY;

		if (strstr(work, _(" incomplete ")) != NULL)
			r->binary_type = RULE_CRITERIA_BINARY_INCOMPLETE;
		else if (strstr(work, _(" complete ")) != NULL)
			r->binary_type = RULE_CRITERIA_BINARY_COMPLETE;
		else if (strstr(work, _(" no ")) != NULL)
			r->binary_type = RULE_CRITERIA_BINARY_NONBINARY;
		else {
			r->binary_type = RULE_CRITERIA_BINARY_INCOMPLETE;
			pan_warn_if_reached ();
		}
	}
#if 0
	else if (!strcmp (work, _("is a reply to an article I posted")))
	{
		r->type = RULE_CRITERIA_IS_REPLY_TO_MINE;
	}
#endif
	else if (strstr (work, _("binary attachments")) != NULL)
	{
		r->type = RULE_CRITERIA_BINARY;

		/* set the binary type */
		if (strstr (work, _("incomplete")) != NULL)
			r->binary_type = RULE_CRITERIA_BINARY_INCOMPLETE;
		else if (strstr (work, _("complete")) != NULL)
			r->binary_type = RULE_CRITERIA_BINARY_COMPLETE;
		else if (strstr (work, _("no")) != NULL)
			r->binary_type = RULE_CRITERIA_BINARY_NONBINARY;
		else {
			pan_warn_if_reached ();
			r->binary_type = RULE_CRITERIA_BINARY_NONBINARY;
		}
	}
	else if (strstr (work, _("article exceeds")) != NULL)
	{
		gchar * pch;
		r->type = RULE_CRITERIA_EXCEEDS_N_LINES;
		for (pch=work; !isdigit((int)*pch); ) ++pch;
		r->exceeds_n = atoi(pch);
	}
	else if (strstr (work, _("article was posted to more than")) != NULL)
	{
		gchar * pch;
		r->type = RULE_CRITERIA_EXCEEDS_N_GROUPS;
		for (pch=work; !isdigit((int)*pch); ) ++pch;
		r->exceeds_n = atoi(pch);
	}
	else if (strstr (work, _("article is more than")) != NULL)
	{
		gchar * pch;
		r->type = RULE_CRITERIA_EXCEEDS_N_DAYS_OLD;
		for (pch=work; !isdigit((int)*pch); ) ++pch;
		r->exceeds_n = atoi(pch);
	}
	else if (strstr (work, _("article is in a")) != NULL)
	{
		r->type = RULE_CRITERIA_THREAD;

		if (strstr (work, _("watched")) != NULL)
			r->thread_type = RULE_CRITERIA_THREAD_WATCHED;
		else if (strstr (work, _("ignored")) != NULL)
			r->thread_type = RULE_CRITERIA_THREAD_IGNORED;
		else {
			pan_warn_if_reached ();
			r->thread_type = RULE_CRITERIA_THREAD_IGNORED;
		}
	}
	else if (strstr (work, _("author is a bozo")) != NULL)
	{
		r->type = RULE_CRITERIA_BOZO;
	}
	else if (strstr (work, _("article is marked as")) != NULL)
	{
		r->type = RULE_CRITERIA_MARKED_AS;

		if (strstr (work, _("unread")) != NULL)
			r->marked_type = RULE_CRITERIA_MARKED_UNREAD;
		else if (strstr (work, _("read")) != NULL)
			r->marked_type = RULE_CRITERIA_MARKED_READ;
		else {
			pan_warn_if_reached ();
			r->marked_type = RULE_CRITERIA_MARKED_READ;
		}
	}
	else
	{
		pan_warn_if_reached ();
	}

	g_free (freeme);
	return r;
}

RuleCriteria*
string_to_rule_criteria_recursive  (const gchar * str)
{
	RuleCriteria * retval;
	GPtrArray * rules = g_ptr_array_new ();
	gchar ** lines = g_strsplit (str, "\n", -1);
	gint i;

	for (i=0; lines[i]!=NULL; ++i)
	{
		const gchar * line = lines[i];
		gint depth = 0;
		RuleCriteria * parent = NULL;
		RuleCriteria * rc = string_to_rule_criteria_single (line, &depth);

		/* link with the parent */
		g_assert ((depth <= rules->len+1) && "this rule is too deep!");
		g_assert ((rules->len ? depth>0 : TRUE) && "must be only one root node");
		if (depth != 0) {
			parent = RULE_CRITERIA (g_ptr_array_index(rules,depth-1));
			rule_criteria_add_child (parent, rc);
		}

		/* update the parent hierarchy */
		g_ptr_array_set_size (rules, depth+1);
		g_ptr_array_index (rules, depth) = rc;
	}

	retval = RULE_CRITERIA(g_ptr_array_index(rules,0));

	/* cleanup */
	g_strfreev (lines);
	g_ptr_array_free (rules, TRUE);

	return retval;
}


gchar*
rule_criteria_tostring_single (const RuleCriteria * r,
                               int depth)
{
	gchar * retval = NULL;

	switch (r->type)
	{
		case RULE_CRITERIA_AND:
		{
			retval = g_strdup_printf (_("ALL of:"));
			break;
		}
		case RULE_CRITERIA_OR:
		{
			retval = g_strdup_printf (_("ANY of:"));
			break;
		}
		case RULE_CRITERIA_PHRASE:
		{
			gchar * search_field;
			gchar * phrase_type;

			/* which field to search... */
			switch (r->phrase_type)
			{
				case RULE_CRITERIA_PHRASE_SUBJECT:
					search_field = _("subject");
					break;
				case RULE_CRITERIA_PHRASE_AUTHOR:
					search_field = _("author");
					break;
				default:
					pan_warn_if_reached ();
					search_field = _("Hello frob!");
					break;
			}

			phrase_type = r->phrase_is_regex ? _("reg. expr.") : _("phrase");

			if (r->phrase_does)
				/* Translators, leave quotes exactly \"%s\" */
				retval = g_strdup_printf (_("%s contains %s \"%s\""),
					search_field,
					phrase_type,
					r->phrase);
			else
				/* Translators, leave quotes exactly \"%s\" */
				retval = g_strdup_printf (_("%s does not contain %s \"%s\""),
					search_field,
					phrase_type,
					r->phrase);
			break;
		}
#if 0
		case RULE_CRITERIA_IS_REPLY_TO_MINE:
		{
			retval = g_strdup_printf (_("is a reply to an article I posted"));
			break;
		}
#endif
		case RULE_CRITERIA_EXCEEDS_N_LINES:
		{
			retval = g_strdup_printf (_("article exceeds %d lines"), r->exceeds_n);
			break;
		}
		case RULE_CRITERIA_EXCEEDS_N_GROUPS:
		{
			retval = g_strdup_printf (_("article was posted to more than %d groups"), r->exceeds_n);
			break;
		}
		case RULE_CRITERIA_EXCEEDS_N_DAYS_OLD:
		{
			retval = g_strdup_printf (_("article is more than %d days old"), r->exceeds_n);
			break;
		}
		case RULE_CRITERIA_THREAD:
		{
			gchar * type;

			switch (r->thread_type) {
				case RULE_CRITERIA_THREAD_WATCHED:
					type = _("watched");
					break;
				case RULE_CRITERIA_THREAD_IGNORED:
					type = _("ignored");
					break;
				default:
					pan_warn_if_reached ();
					type = _("Hello kail!");
					break;
			}

			retval = g_strdup_printf (_("article is in a %s thread"), type);
			break;
		}
		case RULE_CRITERIA_BINARY:
		{
			gchar * type;
			switch (r->binary_type) {
				case RULE_CRITERIA_BINARY_COMPLETE:
					type = _("complete");
					break;
				case RULE_CRITERIA_BINARY_INCOMPLETE:
					type = _("incomplete");
					break;
				case RULE_CRITERIA_BINARY_NONBINARY:
					type = _("no");
					break;
				default:
					type = _("hello tov!");
					break;
			}

			retval = g_strdup_printf (_("article has %s binary attachments"), type);
			break;
		}
		case RULE_CRITERIA_BOZO:
		{
			retval = g_strdup (_("author is a bozo"));
			break;
		}
		case RULE_CRITERIA_MARKED_AS:
		{
			gchar * type;

			switch (r->marked_type) {
				case RULE_CRITERIA_MARKED_READ:
					type = _("read");
					break;
				case RULE_CRITERIA_MARKED_UNREAD:
					type = _("unread");
					break;
				default:
					pan_warn_if_reached ();
					type = _("Hello minmax!");
					break;
			}

			retval = g_strdup_printf (_("article is marked as %s"), type);
			break;
		}
		default:
		{
			pan_warn_if_reached ();
			break;
		}
	}

	if (r->not)
	{
		gchar * tmp = g_strdup_printf (_("NOT %s"), retval);
		g_free (retval);
		retval = tmp;
	}

	if (depth > 0)
	{
		gchar * depth_str = get_depth_string (depth);
		gchar * tmp = g_strdup_printf ("%s%s", depth_str, retval);
		g_free (depth_str);
		g_free (retval);
		retval = tmp;
	}

	return retval;
}

void
rule_criteria_tostring_recursive (const RuleCriteria * r,
                                  GString * appendme,
                                  gint depth)
{
	gchar * pch;
	guint i;

	/* write this criteria */
	pch = rule_criteria_tostring_single (r, depth);
	g_string_append (appendme, pch);
	g_string_append_c (appendme, '\n');

	/* write the children */
	if (r->children!=NULL && r->children->len) {
		for (i=0; i!=r->children->len; ++i) {
			RuleCriteria * child = RULE_CRITERIA(g_ptr_array_index(r->children,i));
			rule_criteria_tostring_recursive (child, appendme, depth+1);
		}
	}

	/* cleanup */
	g_free (pch);
}
