/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <libgnome/libgnome.h>

#include "rule.h"
#include "rule-manager.h"
#include "rule-manager-p.h"
#include "file-rules.h"
#include "util.h"

/***
****
***/

static GPtrArray * rules = NULL;

static gboolean dirty = FALSE;

/***
****
***/

static void
add_canned_rules (GPtrArray * rules)
{
	Rule * r;
	RuleCriteria * rc;

	g_return_if_fail (rules!=NULL);

	/* create "Ignore Bozos" rule */
	r = rule_new ();
	r->name = g_strdup (_("Delete Bozos"));
	rc = rule_criteria_new ();
	rc->type = RULE_CRITERIA_AND;
	r->criteria = rc;
	rc = rule_criteria_new ();
	rc->type = RULE_CRITERIA_BOZO;
	rule_criteria_add_child (r->criteria, rc);
	r->action->flags = RULE_ACTION_DISCARD;
	r->dirty = TRUE;
	r->apply_to_incoming = FALSE;
	g_ptr_array_add (rules, r);

	/* create "Delete Old Articles" rule */
	r = rule_new ();
	r->name = g_strdup (_("Delete Old Articles"));
	rc = rule_criteria_new ();
	rc->type = RULE_CRITERIA_AND;
	r->criteria = rc;
	rc = rule_criteria_new ();
	rc->type = RULE_CRITERIA_EXCEEDS_N_DAYS_OLD;
	rc->exceeds_n = 21;
	rule_criteria_add_child (r->criteria, rc);
	r->action->flags = RULE_ACTION_DISCARD;
	r->dirty = TRUE;
	r->apply_to_incoming = FALSE;
	g_ptr_array_add (rules, r);

	dirty = TRUE;
}

/***
****
***/

static void
rule_manager_module_init (void)
{
	rules = g_ptr_array_new ();
	file_rules_load (rules, NULL);
	dirty = FALSE;

	/* create canned rules? */
	if (1) {
		gint canned_created = gnome_config_get_int ("/Pan/General/canned_rules=0");
		if (!canned_created) {
			add_canned_rules (rules);
			gnome_config_set_int ("/Pan/General/canned_rules", 1);
			gnome_config_sync ();
		}
	}
}


/***
****
***/

void
rule_manager_process_incoming_articles (GPtrArray * articles)
{
	guint i;

	g_return_if_fail (articles!=NULL);

	if (rules == NULL)
		rule_manager_module_init ();

	for (i=0; i!=rules->len; ++i) {
		Rule * r = RULE(g_ptr_array_index(rules,i));
		if (r->apply_to_incoming)
			rule_apply (r, articles);
	}
}

/***
****
***/

gboolean
rule_manager_has_rule (const gchar * name)
{
	guint i;

	g_return_val_if_fail (is_nonempty_string(name), FALSE);

	if (rules == NULL)
		rule_manager_module_init ();

	for (i=0; i!=rules->len; ++i)
		if (!g_strcasecmp (name, RULE(g_ptr_array_index(rules,i))->name))
			return TRUE;

	return FALSE;
}

static gchar*
rule_manager_get_new_rule_name (void)
{
	int i = 0;

	for (;;)
	{
		gchar * name = g_strdup_printf ("Rule #%d", ++i);

		if (!rule_manager_has_rule (name))
			return name;

		g_free (name);
	}
}

void
rule_manager_destroy_rule (Rule * rule)
{
	g_return_if_fail (rule!=NULL);
	g_ptr_array_remove (rules, rule);
	dirty = TRUE;
	pan_object_unref (PAN_OBJECT(rule));
}

void
rule_manager_add_new_rule (Rule * rule)
{
	/* sanity checks */
	g_return_if_fail (rule!=NULL);
	g_return_if_fail (is_nonempty_string(rule->name) ? !rule_manager_has_rule (rule->name) : TRUE);

	if (rules == NULL)
		rule_manager_module_init ();

	/* make sure this rule has a unique name */
	if (!is_nonempty_string(rule->name) || rule_manager_has_rule(rule->name))
		replace_gstr (&rule->name, rule_manager_get_new_rule_name());

	g_ptr_array_add (rules, rule);
	dirty = TRUE;
}

/***
****
***/

void
rule_manager_save_rules (void)
{
	if (rules != NULL)
	{
		guint i;

		file_rules_save (rules, NULL);

		for (i=0; i!=rules->len; ++i)
			RULE(g_ptr_array_index(rules,i))->dirty = FALSE;
	}

	dirty = FALSE;
}

const GPtrArray*
rule_manager_get_rules (void)
{
	if (rules == NULL)
		rule_manager_module_init ();

	return rules;
}

void
rule_manager_shutdown_module (void)
{
	if (rules != NULL)
	{
		guint i;

		/* see if the rule manager, or the rules, have changed */
		gboolean tmp_dirty =  dirty;
		if (!tmp_dirty) {
			guint i;
			for (i=0; i!=rules->len && !tmp_dirty; ++i)
				if (RULE(g_ptr_array_index(rules,i))->dirty)
					tmp_dirty = TRUE;
		}

		/* if they've changed, save them */
		if (tmp_dirty)
			rule_manager_save_rules ();

		/* clean out the rules */
		for (i=0; i!=rules->len; ++i)
			pan_object_unref (PAN_OBJECT(g_ptr_array_index(rules,i)));
		g_ptr_array_free (rules, TRUE);
		rules = NULL;
	}
}

/***
****
***/
