/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>

#include <gmime/gmime-utils.h>

#include "article.h"
#include "gnksa.h"
#include "prefs.h"
#include "util.h"
#include "smtp.h"
#include "sockets.h"
#include "status-item.h"

extern const gchar* sockread_err_msg;
extern const gchar* sockwrite_err_msg;

/**
 * examine response from server.
 *
 * If a read error occurs, a sockread_err is emitted by the StatusItem.
 * If expected is >0 and differs from the return value, an error is emitted.
 * 
 * @param item the StatusItem to whom errors are emitted
 * @param sock the socket to read from.
 * @param expected the expected numeric value.
 * @return the received numeric value.
 */
static int
smtp_get_response (StatusItem   * item,
                   PanSocket    * sock,
                   int            expected)
{
	const gchar *buf = NULL;
	int retval = 0;

	if (pan_socket_getline (sock, &buf)) {
		status_item_emit_error (item, sockread_err_msg);
		return -1;
	}

	retval = atoi(buf);
	if (expected>0 && retval!=expected)
		status_item_emit_error_va (
			item,
			_("Got unexpected response from mail server: "
			  "expected %d; got %s"),
			expected,
			buf);

	return retval;
}

/**
 * send a line to the server.
 *
 * If a write error occurs, a sockwrite_err is emitted from StatusItem.
 *
 * @param item the StatusItem to whom errors are emitted
 * @param sock the socket to write to.
 * @param string a string to write.
 * @return zero on success (no write errors), nonzero on failure.
 */
static int
smtp_send_line (StatusItem   * item,
                PanSocket    * sock,
                const char   * str)
{
	int retval = 0;
	if (pan_mute)
	{
		g_message ("(MUTE) to smtp: [%s]", str);
		retval = 0;
	}
	else
	{
		const int retval = pan_socket_putline (sock, str);
		if (retval)
			status_item_emit_error (item, sockwrite_err_msg);
	}
	return retval;
}


/**
 * send a line to the server.
 *
 * If a write error occurs, a sockwrite_err is emitted from StatusItem.
 *
 * @param item the StatusItem to whom errors are emitted
 * @param sock the socket to write to.
 * @param fmt a printf-style format string
 * @return zero on success (no write errors), nonzero on failure.
 */
static int
smtp_send_line_va (StatusItem  * item,
                   PanSocket   * sock,
                   const char  * fmt,
                   ...)
{
	va_list ap;
	int retval = 0;
	char *line = NULL;

	va_start (ap, fmt);
	line = g_strdup_vprintf (fmt, ap);
	va_end (ap);
	retval = smtp_send_line (item, sock, line);
	g_free (line);

	return retval;
}


/**
 * Sends a message to the specified smtp socket.
 *
 * Possible emissions:
 * * If the "From" line is empty and there's no fallback in the
 *     Pan config file, a "No Sender Specified" error is emitted.
 * * If the server doesn't know any of the recipients, a "No Known Recipients"
 *     error is emitted.
 * * If no recipients are specified, a "No Recipients!" error is emitted.  
 * * If a socket write error occurs, a sockwrite_err is emitted.
 * * If a socket read error occurs, a sockread_err is emitted.
 *
 * @param item the StatusItem through which emissions are made.
 * @param socket the smtp socket where the message is mailed.
 * @param msg the message being sent.
 * @return zero on success, nonzero on failure
 */
static int
smtp_send_article (StatusItem    * item,
                   PanSocket     * sock,
                   Article       * article,
                   const gchar   * leader)
{
	gchar * body = NULL;
	gchar * from = NULL;
	const gchar * reverse_path;
	const gchar * reply_to;
	const gchar * subject;
	const gchar * mail_to;
	const gchar * references;
	gboolean okay = TRUE;
	gboolean maybe_okay = FALSE;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, -1);
	g_return_val_if_fail (sock!=NULL, -1);
	g_return_val_if_fail (article!=NULL, -1);

	status_item_emit_status (item, _("Preparing to Send Mail"));

	/**
	***  Generate/Locate headers
	**/

	/* who is this mail to? */
	mail_to = article_get_header (article, PAN_MAIL_TO);
	pan_warn_if_fail (is_nonempty_string (mail_to));
	if (!is_nonempty_string(mail_to)) {
		status_item_emit_error (item, _("No Recipients!"));
		okay = FALSE;
	}

	/* who is this mail from?  (part 1) */
	reverse_path = NULL;
	if (okay) {
		reverse_path = article_get_header (article, PAN_REVERSE_PATH);
		if (!is_nonempty_string(reverse_path)) {
			g_message (_("Can't parse the From: line; hope for the best."));
			reverse_path = article->author_addr;
		}
		if (!is_nonempty_string(reverse_path)) {
			pan_warn_if_reached ();
			okay = FALSE;
		}
	}

	/* who is this mail from?  (part 2) */
	from = NULL;
	if (okay) {
		gchar * author = article_get_author_str (article);
		if (is_nonempty_string (author))
			from = g_strdup (author);
		if (from == NULL)
			from = gnome_config_get_string ("/Pan/User/Email");
		if (from == NULL) {
			status_item_emit_error(item, _("No Sender Specified!"));
			okay = FALSE;
		}
		g_free (author);
	}

	/**
	*** start sending a letter
	**/
	if (okay)
		status_item_emit_status (item, _("Sending Mail"));

	if (okay) {
		const char *fmt = "MAIL FROM:<%s>\r\n";
		okay = !smtp_send_line_va (item, sock, fmt, reverse_path);
	}
	if (okay)
		okay = pan_mute || smtp_get_response(item, sock, 250)==250;

	/**
	*** tell the server who we're sending the letter to
	**/

	maybe_okay = FALSE;
	if (okay)
	{
		guint i;
		GPtrArray* to_array = gnksa_split_addresses (mail_to);
		for (i=0; i!=to_array->len; ++i)
		{
			if (okay)
			{
				const gchar * fmt = "RCPT TO: <%s>\r\n";
				const char * name = (const char*) g_ptr_array_index (to_array, i);
				gchar * address;
				gchar * realname;

				gnksa_do_check_from (name, &address, &realname);
				okay = !smtp_send_line_va (item, sock, fmt, address);

				g_free (address);
				g_free (realname);
			}
			if (okay)
			{
				const int retval = pan_mute
					? 250
					: smtp_get_response(item, sock, -1);

				/* >=1 known recepient */
				maybe_okay = retval==250;

				/* keep going even if one recipient is unknown */
				okay = retval==250 || retval==550;
			}
		}

		pan_g_ptr_array_foreach (to_array, (GFunc)g_free, NULL);
		g_ptr_array_free (to_array,  TRUE);
	}

	/* don't proceed unless we have at least one known recipient */
	if (okay) {
		okay = maybe_okay;
		if (!okay)
			status_item_emit_error(item,
				_("No Known Recipients"));
	}

	/**
	*** tell the server we're about to start sending a letter
	**/

	if (okay)
		okay = !smtp_send_line (item, sock, "DATA\r\n");
	if (okay)
		okay = pan_mute || smtp_get_response (item, sock, 354)==354;

	/**
	*** send the message
	**/
        
	references = article_get_header (article, HEADER_REFERENCES);

	reply_to = article_get_header (article, HEADER_REPLY_TO);

	subject = article->subject;

	body = g_strstrip (article_get_body (article));


	if (okay) {
		gchar * tmp = g_strdup ("User-Agent: Pan/" VERSION " (Unix)");
		replace_gstr (&tmp, g_mime_utils_8bit_header_encode (tmp));
		okay = !smtp_send_line_va (item, sock, "%s\r\n", tmp);
		g_free (tmp);
	}
	if (okay) {
		gchar * tmp = g_mime_utils_8bit_header_encode (from);
		okay = !smtp_send_line_va (item, sock, "From: %s\r\n", tmp);
		g_free (tmp);
	}
	if (okay) {
		gchar * tmp = g_mime_utils_8bit_header_encode (mail_to);
		okay = !smtp_send_line_va (item, sock, "To: %s\r\n", tmp);
		g_free (tmp);
	}
	if (okay) {
		gchar * tmp = g_mime_utils_8bit_header_encode (subject);
		okay = !smtp_send_line_va (item, sock, "Subject: %s\r\n", tmp);
		g_free (tmp);
	}
	if (okay && references!=NULL) {
		gchar * tmp = g_mime_utils_8bit_header_encode (references);
		okay = !smtp_send_line_va (item, sock, "References: %s\r\n", tmp);
		g_free (tmp);
	}
	if (okay && reply_to) {
		gchar * tmp = g_mime_utils_8bit_header_encode (reply_to);
		okay = !smtp_send_line_va (item, sock, "Reply-To: %s\r\n", tmp);
		g_free (tmp);
	}
	if (okay)
		okay = !smtp_send_line (item, sock, "\r\n" );
	if (okay && leader && *leader)
		okay = !smtp_send_line_va (item, sock, "%s\r\n\r\n", leader);
	if (okay) {
		int i;
		gchar ** sd;
		GString * tmp = g_string_sized_new (120);
		replace_gstr (&body, pan_substitute(body,"\r\n","\n"));
	       	sd = g_strsplit (body, "\n", -1);
		for (i=0; sd[i]!=NULL; ++i) {
			g_string_sprintf (tmp, "%s\r\n", sd[i]);
	 		if (okay) okay = !smtp_send_line (item, sock, tmp->str);
		}
		g_strfreev (sd);
		g_string_free (tmp, TRUE);
	}
	if (okay)
		okay = !smtp_send_line (item, sock, ".\r\n");
	if (okay)
		okay = pan_mute || smtp_get_response (item, sock, 250)==250;
	if (okay)
		status_item_emit_status (item, _("Mail Sent!"));
	
	/* clean up */
	g_free (from);
	g_free (body);

	return okay ? 0 : -1;
}


/* begin a SMTP session */
static int
smtp_handshake (StatusItem   * item,
                PanSocket    * sock)
{
	gchar my_name[256+1];
	struct hostent *hp;

	status_item_emit_status (item, _("Handshaking with mail server"));

	gethostname (my_name, sizeof(my_name));
	hp = gethostbyname (my_name);
	strcpy (my_name, hp->h_name);

	if (smtp_send_line_va (item, sock, "HELO %s\r\n", my_name))
		return -1;
	if (!pan_mute && smtp_get_response (item, sock, 250) != 250)
		return -1;

	return 0;
}


/* end a SMTP session */
static void
smtp_disconnect (StatusItem   * item,
                 PanSocket    * sock)
{
	if (sock != NULL)
	{
		smtp_send_line (item, sock, "QUIT\r\n");

		if (!pan_mute)
			smtp_get_response (item, sock, 221);
	}
}

/* open a connection to a mail server */
static PanSocket*
smtp_get_connection (StatusItem *item)
{
	PanSocket *sock = NULL;

	/* get smtp address/port */
	status_item_emit_status (item, _("Connecting to mail server"));

	/* get the socket... */
	sock = pan_socket_new (mail_server_address, mail_server_port);
	pan_object_ref (PAN_OBJECT(sock));
	pan_object_sink (PAN_OBJECT(sock));
	if (sock->sockfd < 0) {
		status_item_emit_error_va (item,
			_("Unable to open connection to mail server \"%s\""),
			mail_server_address);
		pan_object_unref (PAN_OBJECT(sock));
		sock = NULL;
	}

	/* get the "ready" message from the socket... */
	if (sock!=NULL && smtp_get_response (item, sock, 220) != 220) {
		status_item_emit_error_va (item, _("Mail Server Not Ready"));
		pan_object_unref (PAN_OBJECT(sock));
		sock = NULL;
	}

	/* handshake */
	if (sock!=NULL && smtp_handshake (item, sock)) {
		pan_object_unref (PAN_OBJECT(sock));
		sock = NULL;
	}

	return sock;
}

/***
****
****  PUBLIC FUNCTIONS
****
***/

int
smtp_send (StatusItem      * item,
           Article         * article,
           const gchar     * leader)
{
	PanSocket * sock = NULL;
	int retval;

	/* sanity checks */
	g_return_val_if_fail (item!=NULL, -1);
	g_return_val_if_fail (article!=NULL, -1);

	/* connect to the smtp server */
	sock = smtp_get_connection (item);
	if (!sock)
		return -1;

	/* send the message */
	retval = smtp_send_article (item, sock, article, leader);

	/* say goodbye and disconnect */
	smtp_disconnect (item, sock);
	pan_object_unref (PAN_OBJECT(sock));
	return retval;
}
