/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <fcntl.h>
#include <pthread.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "sockets.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

#define BUFSIZE 2048

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static gulong total_bytes = 0;

#if 0
static int reads_til_fail = 500;
#endif

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

PanSocket*
pan_socket_new (const gchar    * address,
                gint             port)
{
	PanSocket * sock;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string (address), NULL);
	g_return_val_if_fail (port>=0, NULL);

	/* create the socket */
	sock = g_new0 (PanSocket, 1);	
	debug1 (DEBUG_PAN_OBJECT, "pan_socket_new: %p", sock);
	pan_socket_constructor (sock, pan_socket_destructor, address, port);
	return sock;
}

void
pan_socket_constructor (PanSocket             * sock,
			PanObjectDestructor     destructor,
			const gchar           * address,
			gint                    port)
{
	struct sigaction act, oact;
	struct sockaddr_in serv_addr;
	struct in_addr inaddr;
	int on;
	struct hostent *hp;
#if (defined(sun) || defined(__sun)) && (defined(__svr4) || defined(__SVR4) || defined(__svr4__))
	unsigned long tinaddr;
#endif

	/* sanity clause */
	g_return_if_fail (sock!=NULL);
	g_return_if_fail (is_nonempty_string(address));
	g_return_if_fail (port>=0);

	/* constructor */
	pan_object_constructor (PAN_OBJECT(sock), destructor);
	sock->host = g_strdup (address);
	sock->sockfd = -1;
	sock->bytes_read = 0;
	sock->bytes_written = 0;
	sock->byte_count_start_time = time(0);
	sock->inbuf = g_malloc0 (BUFSIZE);
	sock->endbuf = sock->inbuf;
	sock->inptr = sock->inbuf;
	sock->line_buffer = g_array_new (TRUE, FALSE, 1);
	sock->error = FALSE;
	sock->nntp_group_name = NULL;
	sock->nntp_username = NULL;
	sock->nntp_password = NULL;

	/* set up the serv_addr struct */
	memset ( &serv_addr, 0, sizeof(serv_addr) );
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons(port);

	/* Try to convert host name as dotted decimal */
#if (defined(sun) || defined(__sun)) && (defined(__svr4) || defined(__SVR4) || defined(__svr4__))
	tinaddr = inet_addr(address);
	if ( tinaddr != -1 )
#else
       	if ( inet_aton(address, &inaddr) != 0 ) 
#endif
	{
		memcpy(&serv_addr.sin_addr, &inaddr, sizeof(inaddr));
	}
	else /* If that failed, then look up the host name */
	{
		while (NULL==(hp = gethostbyname(address)))
		{
			const char* error_str = NULL;
			if (errno)
				error_str = strerror(errno);
			g_warning (
				"Can't resolve %s -- PanSocket not created\n%s",
				address,
				(error_str ? error_str : ""));
			return;
		}
		memcpy(&serv_addr.sin_addr, hp->h_addr, hp->h_length);
	}

	sock->sockfd = socket(AF_INET, SOCK_STREAM, 0);
	if (sock->sockfd < 0) {
		g_warning ("Can't create socket: %s", strerror(errno));
		return;
	}

	if (connect (sock->sockfd, (struct sockaddr*)&serv_addr, sizeof(serv_addr))<0)
	{
		close (sock->sockfd);
		g_warning ("socket connect to %s, port %d failed: %s", address, port, strerror(errno));
		return;
	}

	on = 1;
	if (setsockopt (sock->sockfd, SOL_SOCKET, SO_KEEPALIVE,
	                (char*)&on, sizeof(on)) < 0)
	{
		close(sock->sockfd);
		g_warning ("socket keepalive option failed: %s",
			   strerror(errno));
		return;
	}

	/* If we write() to a dead socket, then let write return EPIPE rather
	   than crashing the program. */
	act.sa_handler = SIG_IGN;
	sigemptyset (&act.sa_mask);
	act.sa_flags = 0;
	sigaction (SIGPIPE, &act, &oact);

	g_message ("Pan: socket created: %p for %s, port %d, sockfd %d", sock, address, port, sock->sockfd);
	debug3 (DEBUG_SOCKET,
	        "socket created: %p for %s, port %d", sock, address, port);
        debug1 (DEBUG_PAN_OBJECT, "pan_socket_constructor: %p", sock);
}


void
pan_socket_destructor (PanObject *obj)
{
	PanSocket * sock;

	g_return_if_fail (obj != NULL);
	sock = PAN_SOCKET(obj);
	debug1 (DEBUG_SOCKET, "socket closed: %p", sock);
        debug1 (DEBUG_PAN_OBJECT, "pan_socket_destructor: %p", sock);

	/* free up our pieces */
	close (sock->sockfd);
	g_free (sock->nntp_group_name);
	g_free (sock->nntp_username);
	g_free (sock->nntp_password);
	g_free (sock->inbuf);
	g_free (sock->host);
	g_array_free (sock->line_buffer, TRUE);

	/* pass along to superclass */
	pan_object_destructor (obj);
}

 
void
pan_socket_set_nntp_auth (PanSocket            * socket,
                          const gchar          * username,
                          const gchar          * password)
{
	g_return_if_fail (socket!=NULL);

	replace_gstr (&socket->nntp_username, g_strdup(username));
	replace_gstr (&socket->nntp_password, g_strdup(password));
}

/*****
******
*****/

/* simple read buffering to hopefully speed things up. */
static gint
read_from_socket (PanSocket* ps)
{
	gint read_count = 0;
	gint retval = 0;
	struct timeval tv;
	fd_set read_fds;

	/* clear the input buffer */
	memset (ps->inbuf, 0, BUFSIZE);

	/* we want to listen for "ready-to-read" from this socket */
	FD_ZERO (&read_fds);
	FD_SET (ps->sockfd, &read_fds);

	/* wait 60 seconds for an OK to read */
	errno = 0;
	tv.tv_sec = 60;
	tv.tv_usec = 0;
	retval = select (ps->sockfd+1, &read_fds, NULL, NULL, &tv);

	/* did select fail? */
	if (retval<=0) {
		if (errno)
			g_message (_("Timed out waiting to read from the server: %s"), g_strerror(errno));
		else
			g_message (_("Timed out waiting to read from the server."));
		ps->error = TRUE;
		return -1;
	}

	/* try to read from the server... */
	errno = 0;
	read_count = read (ps->sockfd, ps->inbuf, BUFSIZE);
	ps->endbuf = ps->inbuf + read_count;
	ps->inptr = ps->inbuf;

	/* did the server hang up? */
	if (read_count == 0) {
		g_message (_("Lost connection to server"));
		return -1;
	}

	/* did we encounter some other error? */
	if (read_count < 0) {
		if (errno)
			g_message (_("Error: reading from socket returned %d: %s"), read_count, g_strerror(errno));
		else
			g_message (_("Error: reading from socket returned %d"), read_count);
		return read_count;
	}

	return 0;
}

/*
 * get_server - read from a socket until \n
 */
int
pan_socket_getline (PanSocket     * sock,
                    const gchar  ** setme)
{
	register char * pch;
	register gboolean done = FALSE;

	g_array_set_size (sock->line_buffer, 0);
	sock->last_action_time = time(0);

	while (!done)
	{
		/* make sure we have text to read */
		if (sock->inptr == sock->endbuf)
		{
			errno = 0;

			if (read_from_socket(sock)<0)
			{
				char * msg;

				if (errno)
					msg = g_strdup_printf (_("socket [%p] failed its read:\n%s"), sock, strerror(errno));
				else
					msg = g_strdup_printf (_("socket [%p] failed its read"), sock);
				g_warning (msg);

				g_free (msg);
				return -1;
			}
		}

		/* find out how many bytes to copy */
		for (pch=sock->inptr; pch!=sock->endbuf; ++pch) {
			if (*pch=='\n') {
				done = TRUE;
				++pch; /* keep the \n in the line_buffer */
				break;
			}
		}

		/* copy the data */
		g_array_append_vals (sock->line_buffer, sock->inptr, pch-sock->inptr);
		if (done) {
			char ch = '\0';
			g_array_append_val (sock->line_buffer, ch);
		}

		/* update stats */
		sock->bytes_read += pch-sock->inptr;
		total_bytes += (pch - sock->inptr);
		sock->inptr = pch;
	}

	debug2 (DEBUG_SOCKET_INPUT,
	       "socket [%p] received [%s]", sock,
	       (char*)sock->line_buffer->data);

	*setme = (const char*) sock->line_buffer->data;
	return 0;
}


gint 
pan_socket_putline (PanSocket     * sock,
                    const gchar   * line)
{
	const gchar * pch = line;
	gint nbytes = strlen (line);
	gint nleft = nbytes;

	sock->last_action_time = time(0);

	debug3 (DEBUG_SOCKET_OUTPUT,
	        "[%ld][socket %p putline][%s]", time(0), (void*)sock, line);

	while (nleft > 0)
	{
		struct timeval tv;
		fd_set write_fds;
		gint nwritten;
		gint retval;

		/* we want to listen for "ready-to-write" on this socket */
		FD_ZERO (&write_fds);
		FD_SET (sock->sockfd, &write_fds);

		/* Wait up to 60 seconds for a ready-to-write acknowledgment */
		tv.tv_sec = 60;
		tv.tv_usec = 0;
		retval = select (sock->sockfd+1, NULL, &write_fds, NULL, &tv);

		/* did select fail? */
		if (retval<=0) {
			if (errno)
				g_warning (_("Timed out waiting for the goahead to write to the server: %s"), g_strerror(errno));
			else
				g_warning (_("Timed out waiting for the goahead to write to the server."));
			sock->error = TRUE;
			return -1;
		}

		/* try to write to the server... */
		nwritten = write (sock->sockfd, pch, nleft);
		sock->bytes_written += nwritten;
		if (nwritten == -1) {
			g_message (_("An error occurred writing to the server."));
			sock->error = TRUE;
			return -1;
		}
		nleft -= nwritten;
		pch += nwritten;
	}

	return nleft;
}

gboolean 
pan_socket_putline_va (PanSocket    *sock,
                       const gchar  *format,
                       ...)
{
	va_list args;
	char* line;
	int retval;

	g_return_val_if_fail (format!=NULL, -1);

	va_start(args, format);
	line = g_strdup_vprintf(format, args);
	va_end(args);

	retval = pan_socket_putline (sock, line);

	g_free (line);
	return retval;
}

/**
***
**/

void
pan_socket_reset_statistics (PanSocket* sock)
{
	sock->bytes_read = 0;
	sock->bytes_written = 0;
	sock->byte_count_start_time = time(0);
}

double
pan_socket_get_xfer_rate_KBps (const PanSocket* sock)
{
	const double bytes = sock->bytes_read + sock->bytes_written;
	const double K = bytes / 1024.0;
	const time_t time_elapsed = time(0) - sock->byte_count_start_time;
	const double KBps = time_elapsed ? (K / time_elapsed) : 0;
	return KBps;
}

gulong
pan_socket_get_total_xfer_K (void)
{
	return total_bytes / (gulong)1024;
}
