/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/ 

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "gui.h"
#include "status-item-view.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static void status_item_view_class_init (StatusItemViewClass *klass);
static void status_item_view_init (StatusItemView *object);
static void status_item_view_destroy (GtkObject *object);
static void status_item_view_shutdown (GtkObject *object);

static gchar* string_shorten_nolock (GdkFont* font, int width, const gchar* original);

static void status_item_view_unregister (StatusItemView *view);

static int progress_cb      (gpointer, gpointer, gpointer);
static int status_cb        (gpointer, gpointer, gpointer);
static int error_cb         (gpointer, gpointer, gpointer);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static GtkObjectClass *parent_class = NULL;

static GtkTooltips *tips = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

GtkType
status_item_view_get_type (void)
{
	static GtkType status_item_view_type = 0;

	if (!status_item_view_type)
	{
		static const GtkTypeInfo status_item_view_info =
		{
			"StatusItemView",
			sizeof(StatusItemView),
			sizeof(StatusItemViewClass),
			(GtkClassInitFunc) status_item_view_class_init,
			(GtkObjectInitFunc) status_item_view_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		status_item_view_type = gtk_type_unique (
			GTK_TYPE_VBOX,
			&status_item_view_info);
	}

	return status_item_view_type;
}

/*****
******
*****/

static void
status_item_view_class_init (StatusItemViewClass *klass)
{
	GtkObjectClass * object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);

	object_class->destroy = status_item_view_destroy;
	object_class->shutdown = status_item_view_shutdown;

	tips = gtk_tooltips_new ();
}

static void
status_item_view_init (StatusItemView *view)
{
	view->progress = NULL;
	view->last_status = NULL;
	view->item = NULL;
}


/*****
******
*****/

static void
status_item_view_shutdown (GtkObject* object)
{
	StatusItemView *view = STATUS_ITEM_VIEW(object);
	status_item_view_unregister (view);

	(*GTK_OBJECT_CLASS (parent_class)->shutdown) (object);
}

static void
status_item_view_destroy (GtkObject* object)
{
	StatusItemView *item = STATUS_ITEM_VIEW(object);
	debug1 (DEBUG_QUEUE, "%p status_item_view_destroy", item);

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_STATUS_ITEM_VIEW (object));
	g_return_if_fail (GTK_OBJECT_CONSTRUCTED (object));

	replace_gstr (&item->last_status, NULL);

	(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

GtkWidget*
status_item_view_new (void)
{
	StatusItemView * view;

        /* gtk instantiation */
	view = STATUS_ITEM_VIEW(gtk_type_new(TYPE_STATUS_ITEM_VIEW));
	debug1 (DEBUG_QUEUE, "%p status_item_view_new", view);

	/* parent setup */
	GTK_CONTAINER(view)->border_width = 0;
	GTK_BOX(view)->spacing = 0;
	GTK_BOX(view)->homogeneous = FALSE;

	/* status_item_view setup: progress bar */
	view->progress = gtk_progress_bar_new ();
	gtk_progress_set_format_string (GTK_PROGRESS(view->progress), "");
	gtk_progress_set_show_text (GTK_PROGRESS(view->progress), TRUE);

	/* give the progressbar an eventbox backer for tooltips */
	view->eventbox = gtk_event_box_new();
	gtk_container_add (GTK_CONTAINER(view->eventbox), view->progress);
	gtk_box_pack_start (GTK_BOX(view), view->eventbox, TRUE, TRUE, 0);

	/* return what we've got */
	gtk_widget_show_all (GTK_WIDGET(view));
	return GTK_WIDGET(view);
}

GtkWidget*
status_item_view_new_with_item (StatusItem * item)
{
	StatusItemView * l = STATUS_ITEM_VIEW(status_item_view_new());
	status_item_view_set_item (l, item);
	return GTK_WIDGET(l);
}

/*****
******
*****/

static void
status_item_view_unregister (StatusItemView *view)
{
	StatusItem *item = NULL;

	item = STATUS_ITEM(view->item);
	if (!item)
		return;

	/* stop listening to the item... */
	pan_callback_remove (item->progress, progress_cb, view);
	pan_callback_remove (item->status, status_cb, view);
	pan_callback_remove (item->error, error_cb, view);

	/* unref the item... */
	pan_object_unref(PAN_OBJECT(item));
	view->item = NULL;
}

void
status_item_view_set_item (StatusItemView* view, StatusItem* item)
{
	if (item != view->item)
	{
		GtkProgress* prog = GTK_PROGRESS(view->progress);

		status_item_view_unregister (view);

		/* update last_status; update progressbar w/o a resize */
		gtk_progress_set_percentage (prog, 0);
		replace_gstr (&view->last_status, g_strdup(""));
		replace_gstr (&prog->format, g_strdup(""));
		GTK_PROGRESS_CLASS (GTK_OBJECT(prog)->klass)->update (prog);
		gtk_tooltips_set_tip (tips, GTK_WIDGET(view->eventbox), NULL, NULL);

		if (item != NULL) {
			gchar * description = status_item_describe (item);
			pan_object_ref (PAN_OBJECT(item));
			pan_callback_add (item->progress, progress_cb, view);
			pan_callback_add (item->status, status_cb, view);
			pan_callback_add (item->error, error_cb, view);
			gtk_tooltips_set_tip (tips, GTK_WIDGET(view->eventbox), description, NULL);
			g_free (description);
		}

		view->item = item;
	}
}

/*****
******
*****/

static gint
progress_cb (gpointer call_object,
             gpointer call_arg,
             gpointer user_data)
{
	const gint of_100 = GPOINTER_TO_INT(call_arg);

	/* update the percentage */
	if (0<=of_100 && of_100<=100)
	{
		StatusItemView * view;
		GtkProgress * prog;

		pan_lock();
 		view = STATUS_ITEM_VIEW(user_data);
		prog = GTK_PROGRESS(view->progress);
		gtk_progress_set_percentage (prog, of_100/100.0);
		pan_unlock();
	}

	return 0;
}

/**
***
**/

static int
status_cb (gpointer call_object,
           gpointer call_arg,
           gpointer user_data)
{
	pan_lock ();
	if (1)
	{
		StatusItemView * v = STATUS_ITEM_VIEW(user_data);
		GtkProgress * prog = GTK_PROGRESS(v->progress);
		const gchar * str = (const gchar*)call_arg;
		char * s = string_shorten_nolock (
			gtk_widget_get_style(v->progress)->font,
			v->progress->allocation.width,
			str);

		/* update last_status; update progressbar w/o a resize */
		replace_gstr (&v->last_status, g_strdup(str));
		replace_gstr (&prog->format, s);
		GTK_PROGRESS_CLASS (GTK_OBJECT(prog)->klass)->update (prog);
	}
	pan_unlock();

	/* cleanup */
	return 0;
}

static int
error_cb (gpointer call_object,
          gpointer call_arg,
          gpointer user_data)
{
	const gchar *error = (const gchar*)call_arg;
	StatusItemView *v = STATUS_ITEM_VIEW(user_data);

	g_warning ("%s",error);
	debug1 (DEBUG_QUEUE,"Error: %s", error);
	status_cb (NULL, (gpointer)error, v);

	return 0;
}

/**
***
**/

static void
string_zotz_char (gchar   * str,
                  gint      i)
{
	gchar *p = str+i;
	for (;;) {
		++p;
		p[-1] = *p;
		if (!*p)
			break;
	}
}

static gchar*
string_shorten_nolock (GdkFont           * font,
                       gint                max_width,
                       const gchar       * original)
{
	gchar *clone = NULL;
	size_t len = 0;
	gint i, j;
	gboolean i_turn = TRUE;
	gint target_width = max_width - 4 - 20; /* 20 is just a magic safety number */

	/* if original is NULL, so is shortened */
	if (!original)
		return NULL;

	/* if original fits, use it */
	len = strlen(original);
	if (gdk_text_width (font, original, len) < target_width)
		return g_strdup (original);

	/* add in the middle "..." */
	clone = g_malloc (len + strlen("...") + 1);
	i = len/2;
	memcpy (clone, original, i);
	memcpy (clone+i, "...", 3);
	j = i+3;
	memcpy (clone+j, original+i, len+1-i);
	original = NULL;

	/* start trimming out text until it fits */	
	while (gdk_text_width(font, clone, strlen(clone)) >= target_width)
	{
		int *pi = i_turn ? &i : &j;
		*pi -= 1;
		if (*pi<0 || *pi>=strlen(clone))
			break;
		string_zotz_char (clone, *pi);
		i_turn = !i_turn;
	}

	return clone;
}
