/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "gui.h"
#include "log.h"
#include "status-item.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static PanCallback * status_item_active_callback = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

/*****
******
*****/

char*
status_item_describe (const StatusItem* item)
{
	g_return_val_if_fail (item!=NULL, NULL);
	return (*item->describe)(item);
}

StatusItem *
status_item_new (StatusItemDescribeFunc describe)
{
	StatusItem *item = g_new (StatusItem, 1);
        debug1 (DEBUG_PAN_OBJECT, "status_item_new: %p", item);
	status_item_constructor (item, status_item_destructor, describe);
	return item;
}

void
status_item_constructor (StatusItem              *item,
                         PanObjectDestructor      destructor,
                         StatusItemDescribeFunc   describe)
{
	PanObject *obj = PAN_OBJECT(item);

	/* init the superclass bits */
	pan_object_constructor (obj, destructor);

	/* init the status-item bits */
        debug1 (DEBUG_PAN_OBJECT, "status_item_constructor: %p", item);
	item->progress = pan_callback_new ();
	item->status = pan_callback_new ();
	item->error = pan_callback_new ();
	item->done = pan_callback_new ();
	item->describe = describe;
	item->errors = NULL;
	item->progress_val = 0;
	item->steps = 0;
	item->step = 0;
}

void
status_item_destructor (PanObject *po)
{
	StatusItem *item = STATUS_ITEM(po);
	g_return_if_fail (item != NULL);

        debug1 (DEBUG_PAN_OBJECT, "status_item_destructor: %p", item);

	/* make sure nobody thinks this item is active */
	status_item_set_active (item, FALSE);

	/* clear out the callback lists */
	pan_callback_free (item->progress);
	pan_callback_free (item->status);
	pan_callback_free (item->error);
	pan_callback_free (item->done);

	/* clear out the errors */
	g_slist_foreach (item->errors, (GFunc)g_free, NULL);
	g_slist_free (item->errors);
	item->errors = NULL;

	pan_object_destructor (PAN_OBJECT(item));
}

/*****
******
*****/

PanCallback*
status_item_get_active_callback (void)
{
	if (status_item_active_callback == NULL)
		status_item_active_callback = pan_callback_new ();

	return status_item_active_callback;
}

void
status_item_set_active (StatusItem       * item,
                        gboolean           active)
{
	g_return_if_fail (item != NULL);
 
	pan_callback_call (status_item_get_active_callback(),
	                   item,
	                   GINT_TO_POINTER((int)active));
}

/*****
******
*****/

void
status_item_emit_init_steps (StatusItem  * item,
                             int           steps)
{
	g_return_if_fail (item != NULL);
	g_return_if_fail (steps >= 0);

	item->step = 0;
	item->steps = steps;
}

void
status_item_emit_next_step (StatusItem * item)
{
	g_return_if_fail (item != NULL);

	status_item_emit_set_step (item, item->step+1);
}

void
status_item_emit_set_step (StatusItem    * item,
                           int             step)
{
	g_return_if_fail (item != NULL);
	g_return_if_fail (step >= 0);

	item->step = step;

	status_item_emit_progress (
		item,
		(int)(item->steps?((item->step*100.0)/item->steps):0));
}

void
status_item_emit_progress (StatusItem    * item,
                           int             out_of_100)
{
	g_return_if_fail (item != NULL);

	if (0<=out_of_100
		&& out_of_100<=100
		&& item->progress_val!=out_of_100)
	{
		item->progress_val = out_of_100;

		pan_callback_call (item->progress,
                                   item,
                                   GINT_TO_POINTER(out_of_100));
	}
}

void
status_item_emit_next_step_inside_lock (StatusItem * item)
{
        int out_of_100;

        /* increment the step */
        ++item->step;

        /* if the progress changed, unlock and fire an update */
        out_of_100 = (int)(item->steps?((item->step*100.0)/item->steps):0);
        if (0<=out_of_100
                && out_of_100<=100
                && item->progress_val!=out_of_100)
        {
                gpointer p = GINT_TO_POINTER(out_of_100);
                item->progress_val = out_of_100;

                pan_unlock ();
                pan_callback_call (item->progress, item, p);
                pan_lock ();
        }
}


void
status_item_emit_done (StatusItem * item,
                       int          status)
{
	g_return_if_fail (item != NULL);
	pan_callback_call (item->done, item, GINT_TO_POINTER(status));
}

void
status_item_emit_status (StatusItem    * item,
                         const char    * status)
{
	g_return_if_fail (item!=NULL);
	g_return_if_fail (status!=NULL);

	debug0 (DEBUG_QUEUE, status);
	pan_callback_call (item->status, item, (gpointer)status);
}

void
status_item_emit_status_va (StatusItem    * item,
                            const char    * format,
                            ...)
{
	char *pch = NULL;
       	va_list args;

	g_return_if_fail (item!=NULL);
	g_return_if_fail (format!=NULL);

	va_start (args, format);
	pch = g_strdup_vprintf (format, args);
	va_end (args);

	status_item_emit_status (item, pch);
	g_free (pch);
}

void
status_item_emit_error (StatusItem  * item,
                        const char  * error)
{
	gchar * copy;

	g_return_if_fail (item != NULL);
	g_return_if_fail (error != NULL);

	copy = g_strdup (error);
	g_message (copy);
	debug2 (DEBUG_QUEUE, "StatusItem %p has this error: %s", item, copy);
	item->errors = g_slist_append (item->errors, copy);
	pan_callback_call (item->error, item, copy);
}

void
status_item_emit_error_va (StatusItem  * item,
                           const char  * fmt,
                           ...)
{
	char *pch = NULL;
       	va_list args;

	g_return_if_fail (item != NULL);
	g_return_if_fail (fmt != NULL);

	va_start (args, fmt);
	pch = g_strdup_vprintf (fmt, args);
	va_end (args);

	status_item_emit_error (item, pch);
	g_free (pch);
}

void
status_item_next_step_gfunc (gpointer group,
                             gpointer p_status_item)
{
	status_item_emit_next_step (STATUS_ITEM(p_status_item));
}

int
status_item_get_progress_of_100 (const StatusItem * item)
{
	g_return_val_if_fail (item!=NULL, 0);
	return item->progress_val;
}
