/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "globals.h"
#include "acache.h"
#include "article.h"
#include "debug.h"
#include "log.h"
#include "message-window.h"
#include "nntp.h"
#include "queue.h"
#include "task-body.h"
#include "text.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static int task_body_run (Task * task);

static gchar * task_body_describe (const StatusItem * status);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
task_body_destructor (PanObject * o)
{
	TaskBody *item = TASK_BODY(o);

	if (item->checkout)
	{
		item->checkout = FALSE;
		acache_file_checkin (
			article_get_header (
				item->article, HEADER_MESSAGE_ID));
	}

	pan_object_unref (PAN_OBJECT(item->article));
	item->article = NULL;

	task_destructor (o);
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_body_new_defaults (Article  * article)
{
	return task_body_new (article, FALSE, NNTP_READ);
}

PanObject*
task_body_new (Article            * a,
               gboolean             open_in_new_window,
               MessageWindowType    message_window_type)
{
	TaskBody * body = NULL;
	gboolean body_exists;
	gboolean needs_socket;
	gboolean is_folder = group_is_folder (a->group);
	const gchar * message_id = article_get_header (a, HEADER_MESSAGE_ID);

	/* sanity checks */
	g_return_val_if_fail (a!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(message_id), NULL);

	pan_object_ref (PAN_OBJECT(a));

	/* do we already have the body for this message? */
	body_exists = article_has_body (a);

	/* do we need to pull this body from the network? */
	needs_socket = TRUE;
	if (is_folder || body_exists)
	{
		acache_file_checkout (message_id);
		needs_socket = FALSE;
	}

	/* create the object */
       	body = g_new0 (TaskBody, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_body_new: %p", body);

	/* initialize the parent class */
	task_constructor (TASK(body),
		task_body_destructor,
		task_body_describe,
		task_body_run,
		a->group->server, TRUE, needs_socket);
	TASK(body)->is_article_reader = TRUE;


	/* initialize the task-body */
	debug1 (DEBUG_PAN_OBJECT, "task_body: %p", a);
	body->article = a;
	body->open_in_new_window = open_in_new_window;
	body->message_window_type = message_window_type;
	body->checkout = !needs_socket;

	return PAN_OBJECT(body);
}

/*****
******
*****/

static gchar*
task_body_describe (const StatusItem * status)
{
	TaskBody * t = TASK_BODY(status);
	const gchar * message_id =
		article_get_header (t->article, HEADER_MESSAGE_ID);

	g_return_val_if_fail (is_nonempty_string(message_id), "");
	g_return_val_if_fail (t->article->group!=NULL, "");

	if (t->checkout)
	{
		return g_strdup_printf (
			_("Cached article \"%s\" from group %s"),
			message_id,
		        group_get_readable_name(t->article->group));
	}
	else
	{
		return g_strdup_printf (
			_("Downloading body of \"%s\" from group %s"),
			message_id,
		        group_get_readable_name(t->article->group));
	}
}

/*****
******
*****/

static int
task_body_run (Task * task)
{
	StatusItem * status = STATUS_ITEM(task);
	TaskBody * body = TASK_BODY(task);
	Server * const server = task->server;
	Group * const group = body->article->group;
	Article * const article = body->article;
	const gboolean * const abort = &task->hint_abort;
	const gchar * message_id =article_get_header(article,HEADER_MESSAGE_ID);
	gboolean is_folder;
	
	/* sanity checks */
	g_return_val_if_fail (server!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (group!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(group->name), TASK_FAIL_HOPELESS);
	g_return_val_if_fail (article!=NULL, TASK_FAIL_HOPELESS);
	g_return_val_if_fail (is_nonempty_string(message_id), TASK_FAIL_HOPELESS);

	/* let the user know what's happening */
	if (body->checkout)
		status_item_emit_status_va (status,
			_("Cached Article \"%s\""), message_id );

	/* Check the cache before we download */
	is_folder = group_is_folder (article->group);
	if (!is_folder && !article_has_body(article))
	{
		gint result = 0;

		if (*abort)
			return TASK_FAIL;

		/* set the group */
		acache_file_checkout (message_id);
	        result = nntp_set_group (status, task->sock, group->name);
	        if (result != TASK_SUCCESS) {
			acache_file_checkin (message_id);
			status_item_emit_error_va (status,
				_("Couldn't select group \"%s\""),
				group_get_readable_name(group));
			return result;
		}

		/* download the article */
		if (*abort) {
			acache_file_checkin (message_id);
			return TASK_FAIL;
		}

		result = nntp_article_download (status,
		                                task->sock,
		                                group,
		                                article,
		                                &task->hint_abort,
		                                NNTP_VERBOSE);
		if (result != TASK_SUCCESS)
		{
			status_item_emit_status_va (status,
				_("Downloading article \"%s\" failed."),
				article->subject);

			acache_file_checkin (message_id);

			/* if user's expecting to see something but we've failed,
			   show them the error messages instead */
			if (task->high_priority && status->errors!=NULL)
			{
				GString * tmp = g_string_sized_new (1024);
				GSList * l = status->errors;
				while (l != NULL)
				{
					g_string_append (tmp, (gchar*)l->data);
					g_string_append_c (tmp, '\n');
					l = l->next;
				}
				text_set_raw (tmp->str);
				g_string_free (tmp, TRUE);
			}

			return result;
		}
	}

	/**
	***  Display the message?
	**/

	if (body->open_in_new_window) /* message window */
	{
		message_window_new (article, body->message_window_type);
		article_set_read (article, TRUE);
	}
	else if (task->high_priority) /* use default text window */
	{
		text_update (article);

		/* check in this file.. */
		if (body->checkout) {
			body->checkout = FALSE;
			acache_file_checkin (message_id);
		}
		article_set_read (article, TRUE);
	}

	return TASK_SUCCESS;
}
