/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <glib.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkvseparator.h>
#include <gtk/gtkwindow.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>

#include "debug.h"
#include "gui.h"
#include "prefs.h"
#include "queue.h"
#include "status-item-view.h"
#include "task-manager.h"
#include "util.h"

#include "xpm/arrow_up.xpm"
#include "xpm/arrow_top.xpm"
#include "xpm/arrow_down.xpm"
#include "xpm/arrow_bottom.xpm"
#include "xpm/delete_row.xpm"
#include "xpm/delete_sheet.xpm"
#include "xpm/requeue.xpm"
#include "xpm/server.xpm"
#if 0 /* soon, soon */
#include "xpm/queue_load.xpm"
#include "xpm/queue_save.xpm"
#endif

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget    * window;
	GtkWidget    * top;
	GtkWidget    * list;

	guint          timeout_id;

	gboolean       titlebar_needs_refresh;
	gboolean       dampen_move_feedback_loop;
}
ManagerUI;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Constants
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

extern GtkTooltips * ttips;

/***********
************  Public
***********/

/***********
************  Private
***********/

static ManagerUI * manager_ui = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE ROUTINES
************/

/**
***  Internal Utility
**/

static GPtrArray*
get_selected_tasks (ManagerUI * ui)
{
	GPtrArray * a = g_ptr_array_new ();
	GtkCList *clist = GTK_CLIST(ui->list);
        GList * l;
	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int index = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, index);
		g_ptr_array_add (a, task);
	}
	return a;
}

/**
***  User Interactions
**/

static void
up_clicked_cb (GtkButton   * button,
               gpointer      user_data)
{
        ManagerUI *ui = (ManagerUI*) user_data;
	GtkCList *clist = GTK_CLIST(ui->list);
	GList *l;
	int last_index = -1;

	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int index = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, index);
		int new_index = MAX (last_index+1, index-1);
		queue_task_move (task, new_index);
		last_index = new_index;
	}
}

static void
top_clicked_cb (GtkButton    * button,
                gpointer       user_data)
{
        ManagerUI *ui = (ManagerUI*) user_data;
	GtkCList *clist = GTK_CLIST(ui->list);
	GList *l;
	int new_row = -1;

	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int old_row = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, old_row);
		queue_task_move (task, ++new_row);
	}
}

static void
down_clicked_cb (GtkButton   * button,
                 gpointer      user_data)
{
	ManagerUI *ui = (ManagerUI*) user_data;
	GtkCList *clist = GTK_CLIST(ui->list);
	GList *l;
	for (l=g_list_last(clist->selection); l!=NULL; l=l->prev)
	{
		const int index = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, index);
		queue_task_move (task, index+1);
	}
}

static void
bottom_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
        ManagerUI *ui = (ManagerUI*) user_data;
	GtkCList *clist = GTK_CLIST(ui->list);
	GList *l;
	int new_row = 0;

	for (l=clist->selection; l!=NULL; l=l->next)
	{
		const int old_row = GPOINTER_TO_INT (l->data);
		Task * const task = gtk_clist_get_row_data (clist, old_row);
		queue_task_move (task, clist->rows - ++new_row);
	}
}

static void
cancel_clicked_cb (GtkButton   * button,
                   gpointer      user_data)
{
	ManagerUI *ui = (ManagerUI*) user_data;
	GPtrArray * a = get_selected_tasks (ui);
	pan_g_ptr_array_foreach (a, (GFunc)queue_task_remove, NULL);
	g_ptr_array_free (a, TRUE);
}

static void
clear_queue_clicked_cb (GtkButton   * b,
                        gpointer      user_data)
{
	GSList * queue = g_slist_copy ((GSList*)queue_get_tasks());
	GSList * l = NULL;

	for (l=queue; l!=NULL; l=l->next) {
		Task * task = TASK(l->data);
		if (queue_get_task_status(task)==QUEUE_TASK_STATUS_QUEUED)
			queue_task_remove(task);
	}

	g_slist_free (queue);
}

static void
requeue_cb (GtkButton   * button,
            gpointer      user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	GPtrArray * a = get_selected_tasks (ui);
	guint i;
	for (i=0; i!=a->len; ++i) {
		Task * task = (Task*) g_ptr_array_index (a, i);
		if (queue_get_task_status(task) == QUEUE_TASK_STATUS_FAILED)
			queue_task_requeue_failed (task, 0);
	}
	g_ptr_array_free (a, TRUE);
}


/**
***  Display
**/

static int
get_secs_left (const Task * task)
{
	int retval = 0;

	if (task->sock)
	{
		double elapsed_time = time(0) - task->sock->byte_count_start_time;
		double progress = status_item_get_progress_of_100 (STATUS_ITEM(task)) / 100.0;
		double safe_progress = MAX(progress, 0.01); /* don't divide by 0 */
		const double estimated_total_time = elapsed_time/safe_progress;
		retval = estimated_total_time - elapsed_time;
	}

	return retval;
}

static void
get_percent_str (const Task   * task,
                 gchar        * buf)
{
	int percent = 0;

	if (task != NULL)
		percent = status_item_get_progress_of_100 (STATUS_ITEM(task));

	if (percent == 0)
		*buf = '\0';
	else
		sprintf (buf, "%d%%", percent);
}

static void
get_xfer_str (const Task    * task,
              gchar         * buf)
{
	if (!task->sock)
	{
		*buf = '\0';
	}
	else
	{
		int secs_left = get_secs_left (task);
		const int h = secs_left / 3600;
		const int m = (secs_left % 3600) / 60;
		const int s = secs_left % 60;
		const double KBps = pan_socket_get_xfer_rate_KBps (task->sock);
		sprintf (buf, _("%.2f KBps, %d:%02d:%02d remaining"), KBps, h, m, s);
	}
}

static void
get_status_str (const Task     * task,
                gchar          * buf)
{
	const char* status_str;
	QueueTaskStatus status = queue_get_task_status(task);

	switch (status)
	{
		case QUEUE_TASK_STATUS_NOT_QUEUED:
			status_str = _("Not Queued");
			break;
		case QUEUE_TASK_STATUS_QUEUED:
			status_str = _("Queued");
			break;
		case QUEUE_TASK_STATUS_RUNNING:
			status_str = _("Running");
			break;
		case QUEUE_TASK_STATUS_FAILED:
			status_str = _("Failed");
			break;
		case QUEUE_TASK_STATUS_ABORTING:
			status_str = _("Aborting");
			break;
		case QUEUE_TASK_STATUS_CONNECTING:
			status_str = _("Connecting");
			break;
		default:
			status_str = _("???");
			break;
	}

	if (!task->server->online && status==QUEUE_TASK_STATUS_QUEUED)
		sprintf (buf, _("Offline"));
	else if (!task->tries)
		sprintf (buf, "%s", status_str);
	else
		sprintf (buf, _("%s (%d tries)"), status_str, task->tries);
}


static void
add_task_to_list (ManagerUI     * ui,
                  Task          * task,
                  int             index)
{
	int row;
	char statusbuf[64];
	char percentbuf[8];
	char xferbuf[64];
	GtkCList * list = GTK_CLIST(ui->list);
	char * description = status_item_describe(STATUS_ITEM(task));
	char * text [5];
       
	text[0] = statusbuf;
	text[1] = percentbuf;
	text[2] = task->server->name;
	text[3] = xferbuf;
	text[4] = description;

	get_xfer_str (task, xferbuf);
	get_status_str (task, statusbuf);
	get_percent_str (task, percentbuf);

	pan_lock ();
	row = gtk_clist_insert (list, index, text);
	gtk_clist_set_row_data (list, row, task);
	pan_unlock ();

	g_free (description);
}


static void
ui_populate (ManagerUI *ui)
{
	int i;
	const GSList* l = NULL;
	for (i=0, l=queue_get_tasks(); l!=NULL; l=l->next, ++i)
		add_task_to_list (ui, l->data, i);
}

static void
queue_row_move_cb (GtkCList   * clist,
                   int          old_index,
                   int          new_index,  
                   gpointer     user_data)
{
	ManagerUI *ui = (ManagerUI*) user_data;
	Task *task = TASK(gtk_clist_get_row_data (clist, old_index));
	if (!ui->dampen_move_feedback_loop)
		queue_task_move (task, new_index);
}


/**
***  Queue Changes
**/

static int
task_added_cb (gpointer call_obj,
               gpointer call_arg,
               gpointer client_arg)
{
	Task * task = TASK(call_obj);
	int index = GPOINTER_TO_INT(call_arg);
	ManagerUI * ui = (ManagerUI*) client_arg;

	add_task_to_list (ui, task, index);

	return 0;
}

static int
task_removed_cb (gpointer call_obj,
                 gpointer call_arg,
                 gpointer client_arg)
{
	Task * task = TASK(call_obj);
	int index = GPOINTER_TO_INT(call_arg);
	ManagerUI * ui = (ManagerUI*) client_arg;
	GtkCList * list = GTK_CLIST(ui->list);

	g_return_val_if_fail (gtk_clist_get_row_data(list,index)==task, 0);
	pan_lock ();
	gtk_clist_remove (list, index);
	pan_unlock ();

	return 0;
}

static int
task_moved_cb (gpointer call_obj,
               gpointer call_arg,
               gpointer client_arg)
{
	ManagerUI * ui = (ManagerUI*) client_arg;
	int old_row = ((int*)call_arg)[0];
	int new_row = ((int*)call_arg)[1];

	pan_lock ();
	gtk_clist_row_move (GTK_CLIST(ui->list), old_row, new_row);
	pan_unlock ();

	return 0;
}

static void
update_titlebar_lock_unconditional (ManagerUI * ui)
{
	int running, queued, failed;
	const GSList* l;
	gchar* pch;

	g_return_if_fail (ui!=NULL);

	running = queued = failed = 0;
	for (l=queue_get_tasks(); l!=NULL; l=l->next) {
		QueueTaskStatus status = queue_get_task_status (TASK(l->data));
		if (status==QUEUE_TASK_STATUS_QUEUED)
			++queued;
		else if (status==QUEUE_TASK_STATUS_RUNNING)
			++running;
		else if (status==QUEUE_TASK_STATUS_ABORTING || status==QUEUE_TASK_STATUS_FAILED)
			++failed;
	}

	if (failed)
		pch = g_strdup_printf (_("Pan %s Task Manager (%d Queued, %d Running, %d Failed)"), VERSION, queued, running, failed);
	else if (queued || running)
		pch = g_strdup_printf (_("Pan %s Task Manager (%d Queued, %d Running)"), VERSION, queued, running);
	else 
		pch = g_strdup_printf (_("Pan %s Task Manager"), VERSION);
	pan_lock_unconditional ();
	gtk_window_set_title (GTK_WINDOW(ui->window), pch);
	pan_unlock_unconditional ();
	g_free (pch);
}

static int
periodic_update_timer (gpointer user_data)
{
	ManagerUI * ui = (ManagerUI*) user_data;
	GtkCList * clist = GTK_CLIST(ui->list);
	gint row;

	debug0 (DEBUG_QUEUE, "task manager periodic ui refresh");

	/* update the ui */
	pan_lock_unconditional ();
	gtk_clist_freeze (clist);
	for (row=0; row<clist->rows; ++row)
	{
		if (gtk_clist_row_is_visible (clist, row))
		{
			char buf[256];
			char * pch;
			Task * task;

			task = gtk_clist_get_row_data(clist, row);
			pan_object_ref (PAN_OBJECT(task));

			/* have the index, now update it */
			get_status_str (task, buf);
			gtk_clist_set_text (clist, row, 0, buf);
			get_percent_str (task, buf);
			gtk_clist_set_text (clist, row, 1, buf);
			gtk_clist_set_text (clist, row, 2, task->server->name);
			get_xfer_str (task, buf);
			gtk_clist_set_text (clist, row, 3, buf);
			pch = status_item_describe(STATUS_ITEM(task));
			gtk_clist_set_text (clist, row, 4, pch);

			/* cleanup */
			g_free (pch);
			pan_object_unref (PAN_OBJECT(task));
		}
	}

	gtk_clist_thaw (clist);
	pan_unlock_unconditional ();

	if (ui->titlebar_needs_refresh) {
		update_titlebar_lock_unconditional (ui);
		ui->titlebar_needs_refresh = FALSE;
	}

	return 1;
}

static void
ui_destroy_cb (GtkObject   * o,
               gpointer      user_data)
{
	ManagerUI *ui = (ManagerUI*) user_data;
	gui_save_column_widths (ui->list, "task_manager");

	pan_callback_remove (queue_task_added, task_added_cb, ui);
	pan_callback_remove (queue_task_removed, task_removed_cb, ui);
	pan_callback_remove (queue_task_moved, task_moved_cb, ui);

	g_source_remove (ui->timeout_id);

	pan_warn_if_fail (ui == manager_ui);
	g_free (manager_ui);
	manager_ui = NULL;
}

/**
***
**/

static ManagerUI*
task_manager_build_ui (void)
{
	ManagerUI * ui = g_new0 (ManagerUI, 1);
	GtkWidget * w;
	GtkWidget * hbox;
	GtkWidget * vbox;
	char* list_titles [5];

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	/* button bar */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_container_set_border_width (GTK_CONTAINER(hbox), GNOME_PAD_SMALL);

	/* up button */
	w = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", up_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (arrow_up_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) Up"), "");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* top button */
	w = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", top_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (arrow_top_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) To Top"), "");
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD);

	/* down button */
	w = gtk_button_new ();
	gtk_signal_connect (GTK_OBJECT(w), "clicked", down_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (arrow_down_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) Down"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* bottom button */
	w = gtk_button_new ();
	gtk_signal_connect (GTK_OBJECT(w), "clicked", bottom_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (arrow_bottom_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Move Selected Task(s) To Bottom"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD);

	/* requeue button */
	w = gtk_button_new ();
	gtk_signal_connect (GTK_OBJECT(w), "clicked", requeue_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (requeue_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Requeue selected failed Task(s)"), "");
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* separator */
	w = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_BIG);

        /* edit per-server maximum */
	w = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (server_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Set Per-Server Connection Limits"), "");
	gtk_signal_connect (GTK_OBJECT(w), "clicked", prefs_spawn_to_news_connections, NULL);
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);

	/* clear queue */
	w = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Clear Queue"), "");
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (delete_sheet_xpm));
	gtk_signal_connect (GTK_OBJECT(w), "clicked", clear_queue_clicked_cb, ui);
	gtk_box_pack_end (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_BIG);

	/* cancel button */
	w = gtk_button_new ();
	gtk_signal_connect (GTK_OBJECT(w), "clicked", cancel_clicked_cb, ui);
	gtk_container_add (GTK_CONTAINER(w), gnome_pixmap_new_from_xpm_d (delete_row_xpm));
	gtk_tooltips_set_tip (GTK_TOOLTIPS(ttips), w, _("Cancel Selected Task(s)"), "");
	gtk_button_set_relief (GTK_BUTTON(w),GTK_RELIEF_NONE);
	gtk_box_pack_end (GTK_BOX(hbox), w, FALSE, FALSE, GNOME_PAD_BIG);

	/* add button bar to the queued window */
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

        /* queue list */
        list_titles[0] = _("Status");
	list_titles[1] = _("% Done");
	list_titles[2] = _("Server");
        list_titles[3] = _("Transfer Rate");
	list_titles[4] = _("Description");
	ui->list = w = gtk_clist_new_with_titles (5, list_titles);
	gtk_clist_set_column_width (GTK_CLIST(w), 0, 200);

	gui_restore_column_widths (ui->list, "task_manager");

	gtk_clist_set_selection_mode (GTK_CLIST(w), GTK_SELECTION_EXTENDED);
	gtk_signal_connect (GTK_OBJECT(w), "row_move", queue_row_move_cb, ui);
	w = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(w), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (w), ui->list);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);

	pan_callback_add (queue_task_added, task_added_cb, ui);
	pan_callback_add (queue_task_removed, task_removed_cb, ui);
	pan_callback_add (queue_task_moved, task_moved_cb, ui);

	ui->titlebar_needs_refresh = TRUE;
	ui->timeout_id = gtk_timeout_add (2000, periodic_update_timer, ui);
	ui->top = vbox;
        gtk_signal_connect (GTK_OBJECT(ui->top), "destroy", ui_destroy_cb, ui);
	return ui;
}

/**
***  Titlebar dirty
**/

static int
update_titlebar_cb (gpointer call_obj,
                    gpointer call_arg,
                    gpointer user_data)
{
	((ManagerUI*)(user_data))->titlebar_needs_refresh = TRUE;
	return 0;
}

static void
remove_update_titlebar_callbacks_cb (GtkObject * o, gpointer user_data)
{
	pan_callback_remove (queue_task_added, update_titlebar_cb, user_data);
	pan_callback_remove (queue_task_removed, update_titlebar_cb, user_data);
	pan_callback_remove (queue_task_status_changed, update_titlebar_cb, user_data);
}

/************
*************  PUBLIC ROUTINES
************/

void
task_manager_spawn (void)
{
	/* There can be only one */
	if (manager_ui != NULL)
	{
		pan_lock();
		if (!GTK_WIDGET_MAPPED(GTK_WIDGET(manager_ui->window)))
			gtk_widget_map ( GTK_WIDGET(manager_ui->window));
		gdk_window_raise (manager_ui->window->window);
		gdk_window_show (manager_ui->window->window);
		pan_unlock();
		return;
	}

	manager_ui = task_manager_build_ui ();
	ui_populate (manager_ui);
	manager_ui->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

	pan_callback_add (queue_task_added, update_titlebar_cb, manager_ui);
	pan_callback_add (queue_task_removed, update_titlebar_cb, manager_ui);
	pan_callback_add (queue_task_status_changed, update_titlebar_cb, manager_ui);
	gtk_signal_connect (GTK_OBJECT(manager_ui->window), "destroy", remove_update_titlebar_callbacks_cb, manager_ui);

	gtk_window_set_title (GTK_WINDOW(manager_ui->window), _("Pan - Task Manager"));
	gtk_container_add (GTK_CONTAINER(manager_ui->window), manager_ui->top);
	gtk_window_set_default_size (GTK_WINDOW (manager_ui->window), 600, 400);
	gtk_widget_show_all (manager_ui->window);
}

