/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include "../config.h"

#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include <libgnome/libgnome.h>

#include "util-mime.h"
#include "util.h"

/***
****
***/

#define uu_is_beginning(text) \
	(text!=NULL \
	 && text[0]=='b' && text[1]=='e' && text[2]=='g' && text[3]=='i' && text[4]=='n' \
	 && text[5]==' ' \
	 && '0'<=text[6] && text[6]<='7' \
	 && '0'<=text[7] && text[7]<='7' \
	 && '0'<=text[8] && text[8]<='7' \
	 && text[9]==' ' \
	 && text[10]!='\0')

#define uu_is_ending(line) \
	(line!=NULL \
	 && line[0]=='e' && line[1]=='n' && line[2]=='d' \
	 && (line[3]=='\0' || line[3]=='\n' || line[3]==' '))

static void
uu_get_file_info (const gchar       * begin,
                  gchar            ** setme_filename,
                  gulong            * setme_mode)
{
	gchar * filename;
	const gchar * end;

	g_return_if_fail (uu_is_beginning(begin));
	g_return_if_fail (setme_filename != NULL);
	g_return_if_fail (setme_mode != NULL);

	*setme_mode = strtol (begin+6, NULL, 8);
	
	begin += 10;
	end = strchr (begin, '\n');
	filename = g_strndup (begin, end-begin);
	g_strstrip (filename);
	*setme_filename = filename;
}

#define skip_to_next_line(pch, stop) \
	while (pch<stop && *pch!='\n') \
		++pch; \
	if (pch<stop && *pch=='\n') \
		++pch; \

static gboolean
find_uu_content (const gchar      * text,
                 gulong             text_len,
                 const gchar     ** setme_begin,
                 gulong           * setme_len)
{
	const gchar * line;
	const gchar * begin = NULL;
	const gchar * end = NULL;
	const gchar * const stop = text + text_len;

	g_return_val_if_fail (is_nonempty_string(text), FALSE);
	g_return_val_if_fail (setme_begin!=NULL, FALSE);
	g_return_val_if_fail (setme_len!=NULL, FALSE);

	/* find beginning and end */
	line = text;
	end = NULL;
	while (line<stop && is_nonempty_string(line))
	{
		if (begin==NULL && uu_is_beginning(line))
		{
			begin = line;
		}
		else if (uu_is_ending (line))
		{
			skip_to_next_line (line, stop);
			end = line;
			break;
		}

		/* skip to the next line */
		skip_to_next_line (line, stop);
	}
	if (begin!=NULL && end==NULL && line!=NULL)
		end = line;

	/* return what we found */
	if (begin!=NULL && end!=NULL) {
		*setme_begin = begin;
		*setme_len = end - begin;
		return TRUE;
	} else {
		*setme_begin = NULL;
		*setme_len = 0;
		return FALSE;
	}
}

static void
guess_part_type_from_filename (const gchar   * filename,
                               gchar        ** setme_type,
                               gchar        ** setme_subtype)
{
	const gchar * type = gnome_mime_type (filename);
	const gchar * delimiter = strchr (type, '/');
	*setme_type = g_strndup (type, delimiter-type);
	*setme_subtype = g_strdup (delimiter+1);
}

static void
look_for_uuencoded_data (GMimePart * part, gpointer data)
{
	guint content_len = 0;
	gulong uu_len = 0;
	const gchar * uu_start = NULL;
	const gchar * body = g_mime_part_get_content (part, &content_len);
	gboolean force_check = GPOINTER_TO_INT(data)!=0;
	gboolean uu_found = FALSE;

	if (force_check || g_mime_content_type_is_type (part->mime_type, "text", "plain"))
		uu_found = find_uu_content (body, content_len, (const gchar**)&uu_start, &uu_len);

	if (uu_found)
	{
		GMimeContentType * mime_type;
		GMimePart * plain_child = NULL;
		GMimePart * uu_child = NULL;

		/* add anything from outside the encoded data as a plaintext child */
		if (1)
		{
			GByteArray * array = g_byte_array_new ();

			if (uu_start!=body)
				g_byte_array_append (array, body, uu_start-body);
			if (uu_start+uu_len!=body+content_len)
				g_byte_array_append (array, uu_start+uu_len, (body+content_len)-(uu_start+uu_len));
			if (array->len == 0) /* make sure we have _something_, it might be the body */
				g_byte_array_append (array, " ", 1);

			plain_child = g_mime_part_new_with_type ("text", "plain");
			g_mime_part_set_content_byte_array (plain_child, array);
		}

		/* decode the attachment and put it in a child */
		if (1)
		{
			GByteArray * buf = g_byte_array_new ();
			gchar * filename = NULL;
			gchar * pch = NULL;
			gchar * type = NULL;
			gchar * subtype = NULL;

			gint step_state = 0;
			guint32 step_save = 0;
			gchar step_uulen = '\0';
			guint decode_len;
			gulong permissions = 0;
			gint pch_len; 

			/* skip past the "begin xxx filename.gif" line */
			pch = strchr (uu_start, '\n') + 1;
			pch_len = uu_len - (pch-uu_start);
			g_byte_array_set_size (buf, (guint)(pch_len * 0.8));
			decode_len = g_mime_utils_uudecode_step (pch, pch_len, buf->data, &step_state, &step_save, &step_uulen);
			g_byte_array_set_size (buf, decode_len);

			/* build the data attachment */
			uu_get_file_info (uu_start, &filename, &permissions);
			guess_part_type_from_filename (filename, &type, &subtype);
			uu_child = g_mime_part_new_with_type (type, subtype);
			g_mime_part_set_filename (uu_child, filename);
			g_mime_part_set_content_id (uu_child, pch=g_strdup_printf("<%s>", filename));
			g_mime_part_set_content_disposition (uu_child, "inline");
			g_mime_part_set_content_byte_array (uu_child, buf);
			buf = NULL;

			/* cleanup */	
			g_free (filename);
			g_free (pch);
			g_free (type);
			g_free (subtype);
		}

		/* recast this part as a multipart/mixed instead of a text/plain */
		mime_type = g_mime_content_type_new ("multipart", "mixed");
		g_mime_part_set_content_type (part, mime_type);
		g_mime_part_set_content_byte_array (part, NULL);
		body = NULL;

		g_mime_part_add_subpart (part, uu_child);
		if (plain_child != NULL)
			g_mime_part_add_subpart (part, plain_child);
	}
}

/***
****
***/

GMimeMessage*
pan_g_mime_parser_construct_message (const gchar * data)
{
	gint force_uu_check;
	GMimeMessage * message;

	message  = g_mime_parser_construct_message (data, TRUE);

	/* HogWasher does this */
	force_uu_check = strstr (data, "x-uuencode") != NULL;

	g_mime_message_foreach_part (message,
	                             look_for_uuencoded_data,
	                             GINT_TO_POINTER(force_uu_check));

	return message;
}
