/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-mime.h>
#include <libgnome/gnome-mime-info.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-messagebox.h>
#include <libgnomeui/gnome-stock.h>
#include <gmime/gmime-utils.h>

#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include <arpa/inet.h>
#include <netdb.h>
#include <pwd.h>

#include "debug.h"
#include "globals.h"
#include "gnksa.h"
#include "util.h"
#include "prefs.h"

/***
****
****  GUI
****
***/

void
pan_error_dialog_parented (gpointer window, const gchar * format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);

	pan_lock();
	if (window != NULL)
		gnome_error_dialog_parented (str, GTK_WINDOW(window));
	else
		gnome_error_dialog (str);
	pan_unlock();

	g_free (str);
}



void
pan_error_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);
	pan_lock();
	gnome_error_dialog (str);
	pan_unlock();
	g_free (str);
}



static pthread_t has_lock_thr = 0;

void
pan_lock_unconditional_from (const gchar * file, int line)
{
	pthread_t thr = pthread_self();
	if (thr == has_lock_thr) {
		g_warning (_("ERROR thread %lu attempted double lock! %s:%d"),
			(gulong)thr, file, line);
		pan_warn_if_reached ();
	}
	gdk_threads_enter();
	has_lock_thr = thr;
	debug3 (DEBUG_LOCK,"thread %lu entered gdk_threads from %s %d", thr, file, line);
}

void
pan_lock_from (const gchar * file, int line)
{
	pthread_t thr = pthread_self();

	if (thr==Pan.main_t && !has_lock_thr)
	{
		debug3 (DEBUG_LOCK,"mainthread %lu attempted lock from %s %d", thr, file, line);
	}
	else
	{
		pan_lock_unconditional_from (file, line);
	}
}

void
pan_unlock_unconditional_from (const gchar * file, int line)
{
	has_lock_thr = (pthread_t)0;
	gdk_threads_leave();
	debug3 (DEBUG_LOCK,"thread %lu left gdk_threads from %s %d", pthread_self(), file, line);
}

void
pan_unlock_from (const gchar* file, int line)
{
	pthread_t thr = pthread_self();

	if (thr==Pan.main_t && !has_lock_thr)
	{
		debug3 (DEBUG_LOCK,"mainthread %lu attempted unlock from %s %d", thr, file, line);
	}
	else
	{
		pan_unlock_unconditional_from (file, line);
	}
}


/***
****
****  TEMP FILES
****
***/


char *
pan_make_temp (FILE ** setme_fp)
{
	char * retval = NULL;
	char * fname = g_strdup_printf ("/%s/pan%uXXXXXX", temp_dir, (guint)getpid());
	int fd = mkstemp (fname);

	if (fd == -1)
		g_free (fname);
	else {
		retval = fname;
		*setme_fp = fdopen (fd, "w+");
	}

	return retval;
}

void
pan_clean_temp_dir (void)
{
	GString *filename = g_string_new (NULL);
	char *dirpath = g_strdup_printf ("/%s", temp_dir); 
	char *prefix = g_strdup_printf ("pan%u", (unsigned int)getpid ());
	gint prefix_l = strlen (prefix);
	DIR *dir_p = opendir(dirpath);
	struct dirent *dirent_p;

	while ((dirent_p = readdir(dir_p)))
	{
		if (!strncmp (prefix, dirent_p->d_name, prefix_l))
		{
			g_string_sprintf (filename,
					  "/%s/%s", temp_dir, dirent_p->d_name);
			unlink (filename->str);
		}	
	}
	closedir (dir_p);
	g_free (prefix);
	g_free (dirpath);
	g_string_free (filename, TRUE);
}


/***
****
****  FILES
****
***/

gchar*
pan_normalize_filename (const gchar * filename)
{
	gchar * pch;
	gchar * retval;
	gchar doublesep[3] = { G_DIR_SEPARATOR, G_DIR_SEPARATOR, '\0' };

	g_return_val_if_fail (filename!=NULL, NULL);

       	retval = g_strdup (filename);

	while ((pch = strstr (retval, doublesep)))
		g_memmove (pch, pch+1, strlen(pch+1)+1);

	return retval;
}

gboolean
file_exists (const char* filename)
{
	return is_nonempty_string(filename) && g_file_exists (filename);
}

GArray*
read_file (const char* filename)
{
	GArray* array = NULL;
	FILE* fp;

	/* get length of file... */
	size_t len = get_filesize(filename);
	if (len<=0)
		return NULL;

	/* open the file... */
	fp = fopen ( filename, "r" );
	if (!fp)
		return NULL;

	/* read the file... */
	array = g_array_new (TRUE, FALSE, 1);
	g_array_set_size (array, len);
	array->len = fread (array->data, 1, len, fp);
	fclose (fp);
	return array;
}

size_t
get_filesize (const char* filename)
{
	struct stat buf;
	if ( !stat(filename, &buf) )
		return buf.st_size;

	return -1;
}

void
directory_check (const char *pathname)
{
	g_return_if_fail (is_nonempty_string(pathname));

	if (!g_file_test (pathname, G_FILE_TEST_ISDIR))
	{
		const gchar * in = pathname;
		gchar * buf = g_malloc (strlen(pathname) + 1);
		gchar * out = buf;

		for (;;)
		{
			*out++ = *in;

			if (*in=='\0' || *in==G_DIR_SEPARATOR)
			{
				if (*in==G_DIR_SEPARATOR)
					*out = '\0';

				if (!g_file_test (buf, G_FILE_TEST_ISDIR))
				{
					if (mkdir (buf, S_IRUSR | S_IWUSR | S_IXUSR))
					{
						pan_error_dialog (_("Couldn't create directory `%s': %s"),
								  buf, g_strerror (errno));
						break;
					}
				}
			}

			if (!*in)
				break;

			++in;
		}

		g_free (buf);
	}
}

/***
****
****  HOSTNAME
****
***/

char*
get_host_name (void)
{
	char *ptr;
	char hostname[256] = { '\0' };

	if ((gethostname(hostname, sizeof(hostname))) == -1) {
		if ((ptr = getenv("HOST")) != NULL)
			strncpy(hostname, ptr, MAXHOSTNAMELEN);
		else if ((ptr = getenv("HOSTNAME")) != NULL)
			strncpy(hostname, ptr, MAXHOSTNAMELEN);
		else
			hostname[0] = '\0';
	}
	hostname[255] = '\0';
	ptr = strtok (hostname, ".");

	return g_strdup (ptr);
}

/* Returns the fully qualified domain name */
char*
get_fqdn (const char * host)
{
	char name[512] = { '\0' };
	char line[1025];
	char fqdn[1024] = { '\0' };
	char * cp;
	char * domain = NULL;
	FILE * inf;
	struct hostent * hp;
	struct in_addr in;

	if (host) {
		if (strchr(host, '.'))
			return g_strdup(host);
		strncpy(name, host, MAXHOSTNAMELEN);
	}
	else {
		if (gethostname(name, MAXHOSTNAMELEN))
			return NULL;
	}

	if (isdigit((int)*name)) {
		in.s_addr = inet_addr (name);
		if ((hp = gethostbyaddr((char *) &in.s_addr, 4, AF_INET)))
			in.s_addr = (*hp->h_addr);
		return g_strdup((hp && strchr(hp->h_name, '.') ? hp->h_name : (char *) inet_ntoa(in)));
	}
	if ((hp = gethostbyname(name)) && !strchr(hp->h_name, '.'))
		if ((hp = gethostbyaddr(hp->h_addr, hp->h_length, hp->h_addrtype)))
			in.s_addr = (*hp->h_addr);

	sprintf(fqdn, "%s", hp ? strchr(hp->h_name, '.')
		? hp->h_name : (char *) inet_ntoa(in)
		: "");
	if (!*fqdn || (fqdn[strlen(fqdn) - 1] <= '9')) {
		*fqdn = '\0';
		inf = fopen("/etc/resolv.conf", "r");
		if (inf) {
			while (fgets(line, 1024, inf)) {
				line[1024] = '\0';
				g_strchug(line);
				g_strchomp(line);
				if (strncmp(line, "domain ", 7) == 0) {
					domain = line + 7;
					break;
				}
				if (strncmp(line, "search ", 7) == 0) {
					domain = line + 7;
					cp = strchr(domain, ' ');
					if (cp)
						*cp = '\0';
					break;
				}
			}
			if (domain)
				sprintf(fqdn, "%s.%s", name, domain);
			fclose(inf);
		}
	}
	return g_strdup (fqdn);
}


/***
****
****  STRINGS
****
***/

/* null-safe strncmp.  NULLs < strings. */
int
pan_strncmp (const char * a,
             const char * b,
             int          len)
{
	int retval;

	if (!a && !b)
		retval = 0;
	else if (!a)
		retval = -1;
	else if (!b)
		retval = 1;
	else
		retval = strncmp (a, b, len);

	return retval;
}

/* null-safe strcmp.  NULLs < strings. */
int 
pan_strcmp (const char * a,
            const char* b)
{
	int retval;

	if (a == b)
		retval = 0;
	else if (!a)
		retval = -1;
	else if (!b)
		retval = 1;
	else
		retval = strcmp (a, b);

	return retval;
}

gchar*
pan_stristr (const char * string, const char * pattern)
{
	size_t slen = strlen (string);
	size_t plen = strlen (pattern);
	char * start = (gchar*) string;

	for (; slen>=plen; ++start, --slen)
	{
		gchar * sptr;
		gchar * pptr;

		/* find start of pattern in string */
		while (toupper(*start) != toupper(*pattern))
		{
			++start;
			--slen;

			/* if pattern longer than string */
			if (slen < plen)
				return NULL;
		}

		sptr = start;
		pptr = (char *) pattern;

		while (toupper(*sptr) == toupper(*pptr))
		{
			++sptr;
			++pptr;

			if (!*pptr)
				return start;
		}
	}

	return NULL;
}

#if 0
void
array_shrink (gpointer   array,
              size_t     index,
              size_t     element_size,
              size_t     element_qty)
{
	g_return_if_fail (array!=NULL);
	g_return_if_fail (element_size!=0);
	g_return_if_fail (index>=0);
	g_return_if_fail (index<=element_qty);

	if (index != element_qty-1)
	{
		g_memmove (
			((char*)array) + element_size * index,
			((char*)array) + element_size * (index + 1),
			element_size * (element_qty - index -1));
	}
}
#endif

gboolean
string_ends_with (const gchar * str,
                  const gchar * end)
{
	size_t str_len;
	size_t end_len;
	gboolean retval = FALSE;
 
	g_return_val_if_fail (str!=NULL, FALSE);
	g_return_val_if_fail (end!=NULL, FALSE);
 
	end_len = strlen (end);
	str_len = strlen (str);
 
	if (end_len <= str_len)
		retval = !memcmp(end, str+str_len-end_len, end_len);
 
	return retval;
}


void
replace_gstr (char   ** target_gfree_old,
              char    * assign_from_me)
{
	char * gfree_old;

 	g_return_if_fail (target_gfree_old != NULL);

	gfree_old = *target_gfree_old;
	*target_gfree_old = assign_from_me;
	g_free (gfree_old);
}

/**
 * Replaces the search string inside original with the replace string.
 * This should be safe with overlapping elements of text, though I haven't
 * not tested it with elaborately cruel cases.
 */
char*
pan_substitute (const char * original,
                const char * search,
                const char * replace)
{
	size_t slen;		/* length of search */
	size_t rlen;		/* length of replace */
	size_t tlen;		/* length of target (predicted) */
	int i;
	const char * o;
	const char * pchar;
	char * t;
	char * retval = NULL;

	g_return_val_if_fail (original!=NULL, NULL);
	g_return_val_if_fail (*original!='\0', NULL);
	g_return_val_if_fail (search!=NULL, NULL);
	g_return_val_if_fail (*search!='\0', NULL);
	g_return_val_if_fail (replace!=NULL, NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	/* calculate the length */

	i = 0;
	tlen = 0;
	pchar = original;
	while ((pchar = strstr (pchar, search))) {
		i++;
		pchar += slen;
	}
	tlen = strlen(original) + i*(rlen - slen);

	/**
	***  Make the substitution.
	**/

	o = original;
	t = retval = g_malloc(tlen + 1);
	while ((pchar = strstr (o, search))) {
		(void) memcpy (t, o, (size_t)(pchar-o));
		t += pchar-o;
		(void) memcpy (t, replace, (size_t)rlen);
		t += rlen;
		o = pchar + slen;
	}
	(void) strcpy ( t, o );

	pan_warn_if_fail (strlen(retval)==tlen);

	return retval;
}

/***
****
****  GLIB AUGMENTATION
****
***/

void
pan_g_ptr_array_sort (GPtrArray   * a,
                      int  (*compare)(const void *, const void *))
{
	msort (a->pdata, a->len, sizeof(gpointer), compare);
}

void
pan_g_ptr_array_merge_sorted (GPtrArray * target,
                              GPtrArray * s1,
                              GPtrArray * s2,
                              int  (*compare)(const void *, const void *))
{
	guint i1, i2;

	g_return_if_fail (target!=NULL);
	g_return_if_fail (s1!=NULL);
	g_return_if_fail (s2!=NULL);

	pan_g_ptr_array_reserve (target, s1->len + s2->len);

	for (i1=0, i2=0; i1!=s1->len && i2!=s2->len; )
	{
		gpointer * addme = NULL;

		if (i1==s1->len)
			addme = g_ptr_array_index (s2, i2++);
		else if (i2==s2->len)
			addme = g_ptr_array_index (s1, i1++);
		else {
			gpointer p1 = g_ptr_array_index (s1, i1);
			gpointer p2 = g_ptr_array_index (s2, i2);
			const int result = compare (&p1, &p2);
			if (result <= 0) {
				addme = p1;
				++i1;
			} else {
				addme = p2;
				++i2;
			}
		}

		g_ptr_array_add (target, addme);
	}
}

void
pan_g_ptr_array_append (GPtrArray   * target,
                        gpointer    * source_ptr,
                        guint         source_qty)
{
	const guint old_len = target->len;
	g_ptr_array_set_size (target, old_len+source_qty);
	memcpy (target->pdata+old_len, source_ptr, sizeof(gpointer)*source_qty);
}

void
pan_g_ptr_array_assign  (GPtrArray  * target,
                         gpointer   * source_ptr,
                         guint        source_qty)
{
	g_ptr_array_set_size (target, source_qty);
	memcpy (target->pdata, source_ptr, sizeof(gpointer)*source_qty);
}

GPtrArray*
pan_g_ptr_array_dup     (GPtrArray  * source)
{
	GPtrArray * retval = g_ptr_array_new ();
	pan_g_ptr_array_assign (retval, source->pdata, source->len);
	return retval;
}



void
pan_g_ptr_array_reserve (GPtrArray  * a,
                         int          n)
{
	int len;

	g_return_if_fail (a!=NULL);

	len = a->len;
	g_ptr_array_set_size (a, MAX(len, n));
	a->len = len;
}

void
pan_g_ptr_array_insert (GPtrArray   * a,
                        gpointer      ptr,
                        int           index)
{
        g_return_if_fail (a!=NULL);

        if (index<0 || index>=a->len)
	{
                g_ptr_array_add (a, ptr);
	}
        else
        {
		pan_g_ptr_array_reserve (a, a->len+1);
                g_memmove (&a->pdata[index+1],
                           &a->pdata[index],
                           sizeof(gpointer)*(a->len-index));
                a->pdata[index] = ptr;
		++a->len;
        }
}

void
pan_g_ptr_array_foreach (GPtrArray   * a,
                         GFunc         func,
                         gpointer      user_data)
{
	guint i;

	g_return_if_fail (a!=NULL);

	for (i=0; i!=a->len; ++i)
	{
		gpointer call_data = g_ptr_array_index (a, i);
		(*func)(call_data, user_data);
	}
}

void
pan_g_string_replace (GString       * string,
                      const char    * search,
		      const char    * replace)
{
	const gchar * pch;
	size_t slen;
	size_t rlen;
	size_t offset;

	g_return_if_fail (string != NULL);
	g_return_if_fail (is_nonempty_string(search));
	g_return_if_fail (replace != NULL);

	slen = strlen (search);
	rlen = strlen (replace);

	offset = 0;
	while ((pch = strstr (string->str + offset, search)) && *pch) {
		const gint pos = pch - string->str;
		g_string_erase (string, pos, slen);
		g_string_insert (string, pos, replace);
		offset = pos + rlen;
	}
}

void
pan_g_string_strstrip (GString * string)
{
	g_return_if_fail (string != NULL);

	if (string->len != 0)
	{
		g_strstrip (string->str);
		string->len = strlen (string->str);
	}
}

/***
****
****  TOKENS
****
***/

void
skip_next_token (const char   * pch,
                 const char     delimiter,
                 const char  ** setme_next_token)
{
	if (is_nonempty_string(pch))
	{
		while (*pch && *pch!=delimiter)
			++pch;
		if (*pch==delimiter)
			++pch; /* skip past delimiter */
		if (setme_next_token)
			*setme_next_token = pch;
	}
}

glong
get_next_token_long (const char   * pch,
                     const char     delimiter,
                     const char  ** setme_next_token)
{
	glong retval = 0;

	if (is_nonempty_string(pch))
	{
		retval = atol (pch);
		skip_next_token (pch, delimiter, setme_next_token);
	}

	return retval;
}

gulong
get_next_token_ulong (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token)
{
	gulong retval = 0;

	if (is_nonempty_string(pch))
	{
		retval = strtoul (pch, NULL, 10);
		skip_next_token (pch, delimiter, setme_next_token);
	}

	return retval;
}

int
get_next_token_int (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	int retval = 0;

	if (is_nonempty_string(pch))
	{
		retval = atoi (pch);
		skip_next_token (pch, delimiter, setme_next_token);
	}

	return retval;
}

char*
get_next_token_str (const char   * pch,
                    const char     delimiter,
                    const char  ** setme_next_token)
{
	char* retval = NULL;
	register const char* end = pch;

	while (is_nonempty_string(end) && *end!=delimiter)
		++end;

	if (end != pch)
		retval = g_strndup (pch, end-pch);

	if (setme_next_token)
		*setme_next_token = *end ? end+1 : end;

	return retval;
}

gboolean
get_next_token_g_str (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token,
                      GString      * setme)
{
	register const char* end = pch;
	gboolean retval;

	while (is_nonempty_string(end) && *end!=delimiter)
		++end;

	if (end != pch)
	{
		gchar * tmp = g_strndup (pch, end-pch);
		g_string_assign (setme, tmp);
		g_free (tmp);
		retval = TRUE;
	}
	else
	{
		g_string_truncate (setme, 0);
		retval = *end==delimiter;
	}

	if (setme_next_token)
		*setme_next_token = *end ? end+1 : end;

	return retval;
}

/***
****
****  VERSION CHANGE
****
***/


#define DB_FORMAT_LAST_CHANGED_IN_VERSION "0.8.1 beta 3"
#define DB_FORMAT_VERSION 2

void
check_and_warn_if_version_change (void)
{
	int old_db_version =
		gnome_config_get_int ("/Pan/General/DBFormatVersion=0");
	gchar * username = gnome_config_get_string ("/Pan/User/Full_Name");
	const gboolean new_user = !is_nonempty_string (username);

	/* 0.8.1 beta 1 used a different key for finding db format version */
	if (old_db_version < 1)
	{
		gchar * s = gnome_config_get_string ("/Pan/General/AppVersion");
		if (!pan_strcmp(s, "0.8.1beta1"))
			old_db_version = 1;
		g_free (s);
	}

	/* do something if the versions differ... */
	if (old_db_version != DB_FORMAT_VERSION)
	{
	       	/* not a new user, let them know what's happening... */
		if (!new_user)
		{
			gchar * msg = g_strdup_printf (
_("Thanks for upgrading to Pan %s!"
"\n"
"\nThe database formats changed in version %s of Pan."
"\nYou need to delete your Pan database files (located in the directory"
"\n``.pan/data'' in your home directory, by default) before using this"
"\nversion of Pan."
"\n"
"\nFor more information, see the installation section of the"
"\nPan FAQ at <http://www.superpimp.org/faq.html>."),
				VERSION,
				DB_FORMAT_LAST_CHANGED_IN_VERSION);
			GtkWidget * dialog = gnome_message_box_new (msg,
				GNOME_MESSAGE_BOX_INFO,
				GNOME_STOCK_BUTTON_CANCEL,
				GNOME_STOCK_BUTTON_NEXT,
				NULL);
			int button = gnome_dialog_run_and_close (
				GNOME_DIALOG(dialog));
			if (button == 0)
				exit (0);
			g_free (msg);
		}

		/* update the config so that the user won't be warned
		   the next time they run Pan. */
		gnome_config_set_int (
			"/Pan/General/DBFormatVersion", DB_FORMAT_VERSION);
	}

	g_free (username);
}

/***
****
****  LINKIFY TEXT
****
***/

#if 0
static gboolean
badchar (gchar c)
{
	return c==' '
		|| c==','
		|| c=='('
		|| c==')'
		|| c=='\0'
		|| c=='\n'
		|| c=='<'
		|| c=='>';
}

static const gchar*
end_of_url (const gchar * pch)
{
	while (!badchar(*pch))
		++pch;
	if (pch[-1] == '.')
		--pch;
	return pch;
}


gchar*
linkify_text (const char * text)
{
	GString * out = g_string_new (NULL);
        const gchar * c;

	/* walk through the text */
	c = text;
	while (*c)
	{
		/* pass over already-existing anchors */
		if (!g_strncasecmp (c, "<A", 2))
		{
			for (;;)
			{
                                if (!g_strncasecmp(c, "/A>", 3))
                                        break;

				g_string_append_c (out, *c);
				++c;
                                if (!*c)
                                        break;
                        }
                }

		/* add an anchor to a url */
		else if ((!g_strncasecmp(c, "http://", 7) ||
			 (!g_strncasecmp(c, "https://", 8))))
		{
			const gchar * end = end_of_url (c);
			gchar * url = g_strndup (c, end-c);
			g_string_sprintfa (out,
				"<A HREF=\"%s\">%s</A>", url, url);
			g_free (url);
			c = end;
		}

		/* add anchors to likely web sites */
		else if (!g_strncasecmp(c, "www.", 4) &&
		          g_strncasecmp(c, "www..", 5))
		{
			const gchar * end = end_of_url (c);
			if (end-c != 4)
			{
				gchar * url = g_strndup (c, end-c);
				g_string_sprintfa (out,
					"<A HREF=\"http://%s\">%s</A>",
					url, url);
				g_free (url);
				c = end;
			}
                }

		/* add anchors to ftp links */
		else if (!g_strncasecmp(c, "ftp://", 6))
		{
			const gchar * end = end_of_url (c);
			gchar * url = g_strndup (c, end-c);
			g_string_sprintfa (out,
				"<A HREF=\"ftp://%s\">%s</A>", url, url);
			g_free (url);
			c = end;
		}

		/* more ftp anchors */
		else if (!strncasecmp(c, "ftp.", 4))
		{
			const gchar * end = end_of_url (c);
			if (end-c != 4)
			{
				gchar * url = g_strndup (c, end-c);
				g_string_sprintfa (out,
					"<A HREF=\"ftp://%s\">%s</A>",
					url, url);
				g_free (url);
				c = end;
                        }
                }

		/* anchor a mailto */
		else if (!g_strncasecmp(c, "mailto:", 7))
		{
			const gchar * end = end_of_url(c);
			gchar * url = g_strndup (c, end-c);
			g_string_sprintfa (out,
				"<a href=\"%s\">%s</a>",
				url, url);
			g_free (url);
			c = end;
                }

		/* possible mailto 
		else if (!strncasecmp(c, "@", 1))
		{
			const gchar * t = c;
			const gchar * end = end_of_url (c);
			gchar * url;
			while (!badchar(*t) && t!=text) --t;
			if (*t=='<') ++t;
			url = g_strndup (t, end-t);
			if (GNKSA_OK == gnksa_check_address (url))
			{
				g_string_sprintfa (out,
					"<a href=\"mailto:%s\">%s</a>",
					url, url);
			}
			g_free (url);
			c = end;
                } */

		/* keep appending */
		else
		{
			g_string_append_c (out, *c++);
		}
        }

	c = out->str;
	g_string_free (out, FALSE);
	return (gchar*)c;
}
#endif

/***
****
****  DATE
****
***/

/* converts struct tm to time_t, but does not take the local
   timezone into account unless ``local'' is nonzero.
   This is lifted from mutt 1.2.4, which is GPL and
   copyrighted by Michael R. Elkins <me@cs.hmc.edu> */
static time_t
mutt_mktime (struct tm *t)
{
	time_t g;

	static int AccumDaysPerMonth[12] = {
		0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
	};

	/* Compute the number of days since January 1 in the same year */
	g = AccumDaysPerMonth [t->tm_mon % 12];

	/* The leap years are 1972 and every 4. year until 2096,
	 * but this algoritm will fail after year 2099 */
	g += t->tm_mday;
	if ((t->tm_year % 4) || t->tm_mon < 2)
		g--;
	t->tm_yday = g;

	/* Compute the number of days since January 1, 1970 */
	g += (t->tm_year - 70) * 365;
	g += (t->tm_year - 69) / 4;

	/* Compute the number of hours */
	g *= 24;
	g += t->tm_hour;

	/* Compute the number of minutes */
	g *= 60;
	g += t->tm_min;

	/* Compute the number of seconds */
	g *= 60;
	g += t->tm_sec;

	return g;
}

gchar*
rfc822_date_generate (time_t secs)
{
	static const gchar *const months[] = {
		"Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec"
	};
	static const gchar *const days[] = {
		"Sun","Mon","Tue","Wed","Thu","Fri","Sat"
	};

	int diff;
	int tz_diff_hours;
	int tz_diff_minutes;
	gboolean neg;
	gchar fmt[64] = { '\0' };
	gchar buf[64] = { '\0' };
	struct tm ut;
	struct tm local;

	/* if 0 passed in, use current time */
	if (secs == 0)
		secs = time (NULL);

	/* figure out the difference between this timezone and UT */
	localtime_r (&secs, &local);
	gmtime_r (&secs, &ut);
	diff = (int) difftime (mutt_mktime(&local), mutt_mktime(&ut));
	neg = diff<0;
	diff = abs(diff);
	tz_diff_hours = diff / 3600;
	diff %= 3600;
	tz_diff_minutes = diff / 60;

	/* decide on the time string format.  see son of 1036 sec 5.1 for
	 * the rationale of the offset-from-ut notation.
         *
	 * Update: INN 1.7.2-4 (and others) don't accept the <TZ> string at
	 * the end of son-of-1036's date string.  Thanks to Valentini Jove
	 * for testing to see what worked & what didn't.
	 *
	 * Update 2: Big Thanks to Tov Jacobsen for investigating what
	 * format other programs use, for suggesting code, and for noting
	 * that we need to roll our own day-of-week and month abbreviations
	 * because son-of-1036 requires English text but strftime() localizes
	 * (Okt instead of Oct or Man instead of Mon).
	 *
	 * Update 3: Gaelyne Gasson spotted a bug that the minute tz offset
	 * should have been divided by 60 to give minutes instead of seconds.
	 * This was causing problems in that tz, which is +0930.  I'm feeling
	 * pretty dumb here... :)
	 */
	sprintf (fmt, "%s, %%d %s %%Y %%H:%%M:%%S %c%02d%02d",
		days[local.tm_wday],
		months[local.tm_mon],
		(neg?'-':'+'),
		tz_diff_hours,
		tz_diff_minutes);

	/* generate the time */
	strftime (buf, sizeof(buf), fmt, &local);
	return g_strdup (buf);
}


/***
****
****  REGRESSION TESTS
****
***/


void
util_regression_tests (void)
{
	gchar * test_type;

	/* date parsing/generation test... */
	if (1)
	{
		time_t time_1, time_2;
		gchar *pch_1, *pch_2;

		test_type = "date test";
		time_1 = time(0);
		pch_1 = rfc822_date_generate (time_1);
		time_2 = g_mime_utils_header_decode_date (pch_1, NULL);
		pch_2 = rfc822_date_generate (time_2);
		regression_test_result_va (time_1==time_2, test_type, 1, "time is %lu, generated as `%s', parsed back to %lu", time_1, pch_1, time_2);
		regression_test_result_va (!strcmp(pch_1, pch_2), test_type, 2, "dogfood compare: `%s' and `%s'", pch_1, pch_2);

		g_free (pch_1);
		g_free (pch_2);
	}
}

static gint regression_test_pass = 0;

static gint regression_test_fail = 0;

void
regression_tests_summary (void)
{
	printf ("%d Tests Failed (%d Passed)\n",
		regression_test_fail,
		regression_test_pass);
}

void
regression_test_result (gboolean pass,
                        const gchar * test_type,
                        gint test_number,
                        const gchar * msg)
{
	if (pass)
		++regression_test_pass;
	else
		++regression_test_fail;

	printf ("%s - %s #%d - %s\n",
	        (pass ? "PASS" : "FAIL"),
	        test_type, test_number,
	        msg);
}

void
regression_test_result_va (gboolean pass,
                           const gchar * test_type,
                           gint test_number,
                           const gchar * fmt,
                           ...)
{
	va_list args;
	char * str = NULL;

	/* convert vararg into string... */
	g_return_if_fail (fmt != NULL);
	va_start (args, fmt);
	str = g_strdup_vprintf (fmt, args);
	va_end (args);

	/* output the result... */
	regression_test_result (pass, test_type, test_number, str);

	/* cleanup */
	g_free (str);
}


/***
****
****  MBOX
****
***/

const gchar*
mbox_get_next_message (const gchar * buf,
                       gchar ** setme_from,
                       gchar ** setme_msg)
{
	GString * tmp = g_string_new (NULL);
	const gchar * march;
	int lines = 0;

	/* find the end of the message */
	march = buf;
	for (;;)
	{
		gchar * pch;
		gchar * line;
		gboolean is_nested_from = FALSE;
		gboolean is_from;

		/* ran out of text */
		if (march == NULL)
			break;

		/* get next line */
		pch = strchr (march, '\n');
		if (pch) {
			line = g_strndup (march, pch-march);
			march = pch + 1;
		} else { /* this new line is the last one */
			line = g_strdup (march);
			march = NULL;
		}

		/* check the line for delimiter */
		is_from = !strncmp (line, "From ", 5);
		if (is_from && lines!=0) {
			g_free (line);
			break;
		}

		/* is this a nested From_ line? */
		if (*line=='>') {
			gchar * pch = line;
			while (*pch=='>')
				++pch;
			if (!strncmp(pch,"From ", 5))
				is_nested_from = TRUE;
		}

		/* append the line to the output */
		if (is_from)
			*setme_from = g_strdup (line);
		else {
			g_string_append (tmp, is_nested_from?line+1:line);
			g_string_append_c (tmp, '\n');
		}

		/* cleanup this iteration */
		g_free (line);
		++lines;
	}

	/* retval & cleanup */
	*setme_msg = tmp->len ? tmp->str : NULL;
	g_string_free (tmp, FALSE);

	return march!=buf ? march : NULL;
}

gchar*
mbox_format_message (const gchar * message,
                     const gchar * author_address,
                     time_t date)
{
	gchar * retval;
	GString * tmp;
	const gchar * march;

	/* sanity checking */
	g_return_val_if_fail (message!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(author_address), NULL);

	/* add the From_ line.  note ctime() adds the '\n' */
       	tmp = g_string_new (NULL);
	g_string_sprintf (tmp, "From %s %s", author_address, ctime(&date));

	/* walk through the message, adding each line, prepending
	   From_ lines with '>' as specified by the mbox format */
	march = message;
	for (;;)
	{
		const gchar * line_begin;
		const gchar * line_end;
		gboolean is_from = FALSE;
		gboolean is_nested_from = FALSE;

		/* ran out of text */
		if (march == NULL)
			break;

		/* delimit the line */
		line_begin = march;
		line_end = strchr (march, '\n');
		if (line_end != NULL)
			march = line_end + 1;
		else { /* this new line is the last one */
			line_end = line_begin + strlen(line_begin);
			march = NULL;
		}

		/* check the line for delimiter */
		is_from = !strncmp (line_begin, "From ", 5);

		/* is this a nested From_ line? */
		if (*line_begin) {
			const gchar * pch = line_begin;
			while (*pch=='>')
				++pch;
			if (!strncmp(pch,"From ", 5))
				is_nested_from = TRUE;
		}

		/* append the line to the output */
		g_string_sprintfa (tmp, "%s%*.*s\n",
			(is_from||is_nested_from) ? ">" : "",
			line_end-line_begin,
			line_end-line_begin,
			line_begin);
	}

	/* mbox message ends with a blank line */
	if (tmp->len != 0)
		g_string_append_c (tmp, '\n');

	/* cleanup */
	retval = tmp->len ? tmp->str : NULL;
	g_string_free (tmp, retval==NULL);

	return retval;
}


/***
****
****  UNSORTED
****
***/


int
lower_bound (const void         * key,
             const void         * base,
             size_t               n,
             size_t               size,
             int                  (*compare)(const void *, const void *),
             gboolean           * exact_match )
{
	register int low = 0;
	register int high = n - 1;

	while (low<=high)
	{
		const int mid = (unsigned)(low+high)/2u;
		const void * checkme = (void*)(((char*)(base))+(mid*size));
		const int comp = (*compare)(key,checkme);

		if (comp>0) low = mid+1;
		else if (comp<0 ) high = mid-1;
		else {
			if ( exact_match!=NULL )
				*exact_match = TRUE;
			return mid;
		}
	}

	if (exact_match!=NULL)
		*exact_match = FALSE;

	return low;
}



void
commatize_ulong (gulong    num,
		 char    * setme)
{
	char buf[32], *src=buf;
	int len;
	int i;

	sprintf (buf, "%ld", num);
	len = strlen (buf);
	i = len % 3;
	for (;;)
	{
		while (*src && i--)
			*setme++ = *src++;
		if (!*src) {
			*setme = '\0';
			return;
		}
		else if (src!=buf)
		   *setme++ = ',';
		i = 3;
	}

	pan_warn_if_reached ();
}


gchar*
get_default_author_address (void)
{
	gchar * email = gnome_config_get_string ("/Pan/User/Email");

	if (email != NULL)
		g_strstrip (email);

	/* if no email in prefs, do the best we can */
	if (!is_nonempty_string(email)) {
		gchar * host = get_host_name();
		gchar * fqdn = get_fqdn (host);
		replace_gstr (&email,
			g_strdup_printf ("%s@%s", g_get_user_name(), fqdn));
		g_free (host);
		g_free (fqdn);
	}

	return email;
}

gchar*
get_default_author_from (void)
{
	gchar * retval;
	gchar * real = gnome_config_get_string ("/Pan/User/Full_Name");
	gchar * email = get_default_author_address ();

	g_strstrip (real);

	/* if no name give a default; if not quoted, quote it */
	if (!is_nonempty_string (real))
		replace_gstr (&real, g_strdup_printf ("Dave Rhodes"));
	if (real!=NULL && *real!='"')
		replace_gstr (&real, g_strdup_printf ("\"%s\"", real));

	retval = g_strdup_printf ("%s <%s>", real, email);

	g_free (real);
	g_free (email);

	return retval;
}

gchar*
get_date_display_string (time_t date)
{
	gchar * retval;
	struct tm tm_date;

        if (display_article_dates_in_local_time)
                localtime_r (&date, &tm_date);
        else
                gmtime_r (&date, &tm_date);

	retval = g_strdup_printf ("%d.%02d %02d:%02d",
                tm_date.tm_mon+1,
                tm_date.tm_mday,
                tm_date.tm_hour,
                tm_date.tm_min);

	return retval;
}

void
open_outside_file (const char *fname)
{
	const char * mime_type = NULL;
	const char * mime_prog = NULL;
	char **argv;
	int argc;

	if (!((mime_type = gnome_mime_type (fname))))
		return;
	if (!((mime_prog = gnome_mime_program (mime_type))))
		return;
	if (!strstr (mime_prog, "%f"))
		return;

	argv = g_strsplit (mime_prog, " ", -1);
	for (argc=0; argv[argc]; argc++)
		if (!strcmp(argv[argc], "%f"))
			replace_gstr (argv+argc, g_strdup(fname));

	gnome_execute_async (NULL, argc, argv);

	g_strfreev (argv);
}            
