/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#ifndef __UTIL_H__
#define __UTIL_H__

#include <glib.h>
#include <stdio.h> /* for FILE */
#include <time.h> /* for time_t */

/**
 * Works like bsearch, except it returns the lower bound index
 * in the array.  This is useful for finding a sorted insertion
 * point or for finding the nearest match.
 *
 * @param key the item to search for
 * @param base the beginning of the sorted array to search
 * @param n number of items in the array
 * @param size the size of each item in the array
 * @param compare standard c compare func
 * @param exact_match (optional): if non-NULL, this is set to TRUE or FALSE if
 *        the lower bound is an exact match of key
 *
 * @return the index of the lower bound of the search
 * @prerequisite base must be sorted with respect to compare
 */
int    lower_bound   (const void   * key,
                      const void   * base,
                      size_t         n,
                      size_t         size,
                      int            (*compare)(const void *, const void *),
                      gboolean     * exact_match );

void   msort         (void         * b,
                      size_t         n,
                      size_t         s,
                      int            (*cmp)(const void*, const void*));

/*---------------------------------------------------------------------------
**
**  UI LOCKING FOR THREADS
*/

/**
 * Used to lock the UI thread.
 *
 * @see pan_unlock
 * @see pan_lock_unconditional
 */
#define pan_lock() pan_lock_from(__FILE__, __LINE__)
void pan_lock_from (const gchar * file, int line);

/**
 * Used to lock the UI thread unconditionally.
 * The difference between this and pan_lock is that this will attempt
 * to lock the gui even if we're in the main thread.  This is important
 * for locking down gui calls being made from timeouts in the main thread.
 *
 * @see pan_lock
 * @see pan_unlock_unconditional
 */
#define pan_lock_unconditional() pan_lock_unconditional_from(__FILE__,__LINE__)
void pan_lock_unconditional_from (const gchar * file, int line);

/**
 * Used to unlock the UI thread after pan_lock() has been called.
 *
 * @see pan_lock
 * @see pan_unlock_unconditional
 */
#define pan_unlock() pan_unlock_from(__FILE__, __LINE__)
void pan_unlock_from (const gchar * file, int line);

/**
 * Used to unlock the UI thread after pan_lock_unconditional() has been called.
 *
 * The difference between this and pan_unlock is that this will attempt
 * to unlock the gui even if we're in the main thread.  This is important
 * for unlocking gui calls being made from timeouts in the main thread.
 *
 * @see pan_lock_unconditional
 * @see pan_unlock
 */
#define pan_unlock_unconditional() pan_unlock_unconditional_from(__FILE__,__LINE__)
void pan_unlock_unconditional_from (const gchar * file, int line);


/*---------------------------------------------------------------------------
**
**  TEMP FILES
*/

char* pan_make_temp (FILE **);

void pan_clean_temp_dir (void);

gchar* get_default_author_address (void);

gchar* get_default_author_from (void);

/**
 * Pops up a gnome error dialog with the specified message.
 * This function is threadsafe.
 *
 * @param message or printf-style format string for the message.
 */
void pan_error_dialog_parented (gpointer parent,
                                const char * fmt,
                                ...);

/**
 * Pops up a gnome error dialog with the specified message.
 * This function is threadsafe.
 *
 * @param message or printf-style format string for the message.
 */
void pan_error_dialog (const char * fmt,
                       ...);

/**
 * Returns the number of bytes in the file, or 0 on error.
 * @return the number of bytes in the file, or 0 on error.
 * @param filename
 */
size_t get_filesize (const char* filename);

/**
 * Checks to make sure the directory exists.  If it doesn't, directory_check()
 * tries to create it.  If its parent directories don't exist either,
 * directory_check() tries to create them first.
 *
 * @param pathname
 */
void directory_check (const char* pathname);

/**
 * Formats a number with commas; ie, 10321 becomes "10,321"
 * @param num the number to format
 * @param setme the string where the resulting string is written
 */
void commatize_ulong (gulong num,
                      char* setme);

/**
 * @return a new GArray of lines in the specified file.
 */
GArray* read_file (const char* filename);

char* pan_substitute (const char  * original,
                      const char  * search,
                      const char  * replace);

void pan_g_string_replace (GString    * str,
                           const char * search,
                           const char * replace);

void pan_g_string_strstrip (GString    * str);

gboolean file_exists (const char  * filename);

char* get_fqdn (const char * host);

char* get_host_name (void);

void array_shrink (gpointer    a,
                   size_t      idx,
                   size_t      ele_size,
                   size_t      ele_qty);

int pan_strcmp (const char  * a,
                const char  * b);

int pan_strncmp (const char  * a,
                 const char  * b,
                 int len);


void pan_g_ptr_array_sort   (GPtrArray   * target,
                             int  (*compare)(const void *, const void *));

void pan_g_ptr_array_merge_sorted (GPtrArray * target,
                                   GPtrArray * s1,
                                   GPtrArray * s2,
                                   int  (*compare)(const void *, const void *));

void pan_g_ptr_array_append (GPtrArray   * target,
                             gpointer    * source_ptr,
			     guint         source_qty);

void pan_g_ptr_array_insert (GPtrArray   * target,
                             gpointer      ptr,
                             int           index);

void pan_g_ptr_array_reserve (GPtrArray  * target,
                              int          n);

void pan_g_ptr_array_foreach (GPtrArray  * target,
                              GFunc        func,
                              gpointer     user_data);

void pan_g_ptr_array_assign  (GPtrArray  * target,
                              gpointer   * source_ptr,
                              guint        source_qty);

GPtrArray * pan_g_ptr_array_dup     (GPtrArray  * source);

void open_outside_file (const gchar * filename);

void msort (void *b,
            size_t n,
            size_t s,
            int(*cmp)(const void*, const void*));


void skip_next_token (const char   * pch,
                      const char     delimiter,
                      const char  ** setme_next_token);

int get_next_token_int (const char   * pch,
                        const char     delimiter,
                        const char  ** setme_next_token);

glong get_next_token_long (const char   * pch,
                           const char     delimiter,
                           const char  ** setme_next_token);

gulong get_next_token_ulong (const char   * pch,
                             const char     delimiter,
                             const char  ** setme_next_token);

char* get_next_token_str (const char   * pch,
                          const char     delimiter,
                          const char  ** setme_next_token);

gboolean get_next_token_g_str (const char   * pch,
                               const char     delimiter,
                               const char  ** setme_next_token,
                               GString      * setme);

gchar* linkify_text (const gchar * text);

void replace_gstr (char ** target_free_old, char * assign_from_me);

void check_and_warn_if_version_change (void);

gchar* rfc822_date_generate (time_t secs);

gboolean string_ends_with (const gchar* str, const gchar* end);

gchar* pan_stristr (const char * string, const char * pattern);

gchar* pan_normalize_filename (const gchar * filename);

gchar* get_date_display_string (time_t date);

#define is_nonempty_string(A) ((A) && *(A)!='\0')


/**
*** mbox
**/

const gchar*
mbox_get_next_message (const gchar * buf,
                       gchar ** setme_from,
                       gchar ** setme_msg);


gchar*
mbox_format_message (const gchar * message,
                     const gchar * author_addr,
                     time_t date);
/**
***  Regression
**/

void util_regression_tests (void);

void regression_test_result (gboolean pass, const gchar * test_type, int number, const gchar * msg);

void regression_test_result_va (gboolean pass, const gchar * test_type, int number, const gchar * fmt, ...);

void regression_tests_summary (void);





#ifdef __GNUC__
#define PRETTY_FUNCTION __PRETTY_FUNCTION__
#else
#define PRETTY_FUNCTION ""
#endif

#define pan_warn_if_reached()           G_STMT_START{           \
     g_log (G_LOG_DOMAIN,                                       \
            G_LOG_LEVEL_WARNING,                                \
            "file %s: line %d func %s: should not be reached, please report bug to pan@superpimp.org",     \
            __FILE__,                                           \
            __LINE__,                                           \
            PRETTY_FUNCTION);       }G_STMT_END

#define pan_warn_if_fail(expr)          G_STMT_START{                   \
     if (!(expr))                                                       \
       {                                                                \
         g_log (G_LOG_DOMAIN,                                           \
                G_LOG_LEVEL_CRITICAL,                                   \
                "file %s: line %d func %s: assertion `%s' failed, please report bug to pan@superpimp.org.",        \
                __FILE__,                                               \
                __LINE__,                                               \
                PRETTY_FUNCTION,                                        \
                #expr);                                                 \
       };                               }G_STMT_END


#endif /* __UTIL_H__ */
