#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <ctype.h>

#include <pan/dialogs/dialogs.h>
#include <pan/dialogs/rules_callbacks.h>
#include <pan/dialogs/rules_interface.h>
#include <pan/dialogs/support.h>

#include <pan/group.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/rule.h>
#include <pan/rule-manager.h>
#include <pan/rule-manager-p.h>
#include <pan/util.h>

/**
***
**/

static void rule2ui (GtkWidget * widget, Rule * r);

/**
***
**/

static GtkWidget * _rule_dialog = NULL;

static GtkWidget * _edit_dialog = NULL;

/**
***
**/

void
dialog_rules (void)
{
	if (_rule_dialog == NULL)
		_rule_dialog = create_rule_dialog ();

	gtk_widget_show (_rule_dialog);
}

void
dialog_edit_rule (Rule * rule)
{
	if (_edit_dialog == NULL)
		_edit_dialog = create_rule_edit_dialog ();

	gtk_object_set_user_data (GTK_OBJECT(_edit_dialog), rule);
	rule2ui (_edit_dialog, rule);
	gtk_widget_show (_edit_dialog);
}


/**
***
**/



static gboolean
is_nesting_rc (const RuleCriteria * rc)
{
	return rc->type==RULE_CRITERIA_AND || rc->type==RULE_CRITERIA_OR;
}

/****
*****
*****  Criteria CList helpers
*****
****/

static RuleCriteria*
criteria_clist_get_rule_at_line (GtkCList * clist, gint line, gint * setme_depth)
{
	gchar * str;
	gtk_clist_get_text (clist, line, 0, &str);
	return string_to_rule_criteria_single (str, setme_depth);
}

static void
criteria_clist_set_line_depth (GtkCList * clist, int line, int depth)
{
	gint old_depth = 0;
	RuleCriteria * rc = criteria_clist_get_rule_at_line (clist, line, &old_depth);
	if (rc != NULL)
	{
		gchar * pch = rule_criteria_tostring_single (rc, depth);
		gtk_clist_set_text (clist, line, 0, pch);
		g_free (pch);
		pan_object_unref (PAN_OBJECT(rc));
	}
}

static int
criteria_clist_get_line_depth (GtkCList * clist, int line)
{
	gint depth = 0;
	RuleCriteria * rc = criteria_clist_get_rule_at_line (clist, line, &depth);
	pan_object_unref (PAN_OBJECT(rc));
	return depth;
}

static void
criteria_clist_inc_line_depth (GtkCList * clist, int line, int inc)
{
	gint depth = criteria_clist_get_line_depth (clist, line);
	depth += inc;
	if (depth<0) depth=0;
	criteria_clist_set_line_depth (clist, line, depth);
}

static int
criteria_clist_get_suggested_depth (GtkCList * clist, int line)
{
	gint depth = 0;

	/* try to make the appropriate depth wrt the previous line */
	if (line>0) {
		gint above_depth = 0;
		RuleCriteria * above_rc = criteria_clist_get_rule_at_line (clist, line-1, &above_depth);
		if (above_rc != NULL) {
			depth = above_depth;
			if (is_nesting_rc(above_rc))
				++depth;
			pan_object_unref (PAN_OBJECT(above_rc));
		}
	}

	return depth;
}

static void
criteria_clist_add_at_line (GtkCList * clist, RuleCriteria * rc, int line)
{
	gchar * pch;
	gint depth = criteria_clist_get_suggested_depth (clist, line);

	/* insert the new line */
	pch = rule_criteria_tostring_single (rc, depth);
	gtk_clist_insert (clist, line, &pch);

	/* cleanup */
	g_free (pch);
}

static void
criteria_clist_add (GtkCList * clist, RuleCriteria * rc)
{
	gint cur_row = clist->selection
		? GPOINTER_TO_INT(clist->selection->data)
		: MAX(0,clist->rows-1);

	criteria_clist_add_at_line (clist, rc, cur_row+1);
}

/****
*****
*****  Main Dialog Helpers
*****
****/

static Rule*
main_list_get_selected_rule (GtkCList * clist)
{
	Rule * retval = NULL;
	GList * l = clist->selection;
	if (l != NULL)
		retval = RULE(gtk_clist_get_row_data(clist, GPOINTER_TO_INT(l->data)));
	return retval;
}

static void
main_list_update_rule_row (GtkCList * clist, const Rule * rule)
{
	int row;
	double percent;
	gchar percent_str[32];
	gchar * apply_str;

	/* sanity checks */
	g_return_if_fail (clist!=NULL);
	g_return_if_fail (rule!=NULL);

	/* figure out what we're going to update with */
	percent = 100.0*rule->hits / (double)(rule->tries ? rule->tries : 1);
	sprintf (percent_str, "%.0f%%", percent);
	apply_str = rule->apply_to_incoming ? _("Yes") : "";

	/* find the clist row */
	row = gtk_clist_find_row_from_data (clist, (gpointer)rule);

	/* update the clist row */
	gtk_clist_freeze (clist);
	gtk_clist_set_text (clist, row, 0, rule->name);
	gtk_clist_set_text (clist, row, 1, apply_str);
	gtk_clist_set_text (clist, row, 2, percent_str);
	gtk_clist_thaw (clist);
}

/****
*****
*****  Misc UI Helpers
*****
****/

static gboolean
toggle_button_is_on (GtkWidget * widget, const gchar * togglebutton_name)
{
	gboolean retval = FALSE;
	GtkWidget * w;
       
	if ((w = lookup_widget (widget, togglebutton_name)) != NULL)
		retval = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(w));

	return retval;
}

static const gchar*
get_option_menu_selection (GtkWidget * widget, const gchar * om_name)
{
	const gchar * retval = NULL;
	GtkWidget * om;
	
	om = lookup_widget (widget, om_name);
	if (om!=NULL && GTK_BIN(om)->child!=NULL)
	{
		GtkWidget * child = GTK_BIN(om)->child;
		gtk_label_get (GTK_LABEL(child), (gchar**)&retval);
	}

	return retval;
}


/****
*****
*****  Rule <-> UI
*****
****/


static void
rule2ui (GtkWidget * widget, Rule * r)
{
	GtkCList * subscribed;
	GtkCList * specific;
	GtkWidget * w;
	gboolean b;
	gboolean b2;
	Server * server = grouplist_get_server ();
	const gchar * lookup_str;
	guint i;

	g_assert (r != NULL);

	/**
	***  Update the group criteria
	**/

	/* group clists */
       	w = lookup_widget (widget,"newsgroups_specific_clist");
       	specific = GTK_CLIST(w);
	gtk_clist_freeze (specific);
	gtk_clist_clear (specific);
	gtk_clist_set_sort_column (specific, 0);
	gtk_clist_set_auto_sort (specific, TRUE);
       	w = lookup_widget (widget,"newsgroups_subscribed_clist");
       	subscribed = GTK_CLIST(w);
	gtk_clist_freeze (subscribed);
	gtk_clist_clear (subscribed);
	gtk_clist_set_sort_column (subscribed, 0);
	gtk_clist_set_auto_sort (subscribed, TRUE);
	for (i=0; i!=server->groups->len; ++i)
	{
		Group * g = GROUP(g_ptr_array_index(server->groups,i));
		if (group_is_subscribed(g))
		{
			gchar * name = (gchar*) group_get_readable_name (g);
			GtkCList * clist = (r->group_type==RULE_GROUP_LIST
				&& rule_test_group (r,g->name))
				? specific
				: subscribed;
			gint row = gtk_clist_insert (clist, -1, &name);
			gtk_clist_set_row_data (clist, row, g);
		}
	}
	gtk_clist_thaw (subscribed);
	gtk_clist_thaw (specific);


	/* group wildcard entry */
	w = lookup_widget (widget, "newsgroup_wildcard_entry");
	b = (r->group_type == RULE_GROUP_WILDCARD) != 0;
	gtk_entry_set_text (GTK_ENTRY(w), b ? r->group_wildcard : "");

	lookup_str = NULL;
	switch (r->group_type) {
		case RULE_GROUP_WILDCARD:
			lookup_str = "newsgroups_wildcard_rb"; break;
		case RULE_GROUP_LIST:
			lookup_str = "newsgroups_specific_rb"; break;
		case RULE_GROUP_ALL:
			lookup_str = "newsgroups_all_rb"; break;
		default:
			pan_warn_if_reached ();
			lookup_str = "newsgroups_all_rb"; break;
	}
	w = lookup_widget (widget, lookup_str);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), TRUE);

	/**
	***  Criteria
	**/

	w = lookup_widget (widget, "criteria_clist");
	gtk_clist_clear (GTK_CLIST(w));

	if (r->criteria != NULL)
	{
		GtkCList * clist = GTK_CLIST(w);
		GString * str = g_string_new (NULL);
		gchar ** lines;
		gint i;

		/* get the lines to add */
		rule_criteria_tostring_recursive (r->criteria, str, 0);
	       	lines = g_strsplit (str->str, "\n", -1);

		/* update the criteria clist */
		gtk_clist_clear (clist);
		for (i=0; lines[i]!=NULL; ++i)
			gtk_clist_append (clist, &lines[i]);

		/* cleanup */
		g_strfreev (lines);
		g_string_free (str, TRUE);
	}
	else
	{
		GtkCList * clist = GTK_CLIST(lookup_widget (widget, "criteria_clist"));
		RuleCriteria * rc;

		/* make a new top-level "AND" criteria */
		rc = rule_criteria_new ();
		rc->type = RULE_CRITERIA_AND;
		criteria_clist_add_at_line (clist, rc, 0);

		/* cleanup */
		pan_object_unref (PAN_OBJECT(rc));

	}

	/**
	***  Actions
	**/

	/* sound file */
	w = lookup_widget (widget, "action_play_sound_check");
	b = (r->action->flags & RULE_ACTION_PLAY_SOUND) != 0;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "sound_file_entry");
	w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w));
	gtk_entry_set_text (GTK_ENTRY(w), b ? r->action->sound_file : "");

	/* alert message */
	w = lookup_widget (widget, "action_show_alert_check");
	b = (r->action->flags & RULE_ACTION_SHOW_ALERT) != 0;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "action_show_alert_entry");
	gtk_entry_set_text (GTK_ENTRY(w), b ? r->action->alert_message : "");

#if 0
	/* append to file */
	w = lookup_widget (widget, "action_append_to_file_check");
	b = (r->action->flags & RULE_ACTION_APPEND_TO_FILE) != 0;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "append_file_entry");
	w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w));
	gtk_entry_set_text (GTK_ENTRY(w), b ? r->action->append_file : "");

	/* forward to */
	w = lookup_widget (widget, "action_forward_to_check");
	b = (r->action->flags & RULE_ACTION_FORWARD_TO) != 0;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "action_forward_to_entry");
	gtk_entry_set_text (GTK_ENTRY(w), b ? r->action->forward_to : "");
#endif

	/* mark read/unread */
	b  = (r->action->flags & RULE_ACTION_MARK_READ) != 0;
	b2 = (r->action->flags & RULE_ACTION_MARK_UNREAD) != 0;
	w = lookup_widget (widget, "action_mark_as_read_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b || b2);
	w = lookup_widget (widget, "action_read_rb");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "action_unread_rb");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b2);

#if 0
	/* important / normal */
	b = (r->action->flags & RULE_ACTION_MARK_IMPORTANT) != 0;
	b2 = (r->action->flags & RULE_ACTION_MARK_NORMAL) != 0;
	w = lookup_widget (widget, "action_mark_as_important_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b || b2);
	w = lookup_widget (widget, "action_important_rb");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "action_normal_rb");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b2);

	/* protected / deletable */
	b = (r->action->flags & RULE_ACTION_MARK_PROTECTED) != 0;
	b2 = (r->action->flags & RULE_ACTION_MARK_DELETABLE) != 0;
	w = lookup_widget (widget, "action_mark_as_protected_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b || b2);
	w = lookup_widget (widget, "action_protected_rb");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "action_deleteable_rb");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b2);
#endif

	/* decode */
	b = (r->action->flags & RULE_ACTION_DECODE) != 0;
	w = lookup_widget (widget, "action_decode_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
	w = lookup_widget (widget, "decode_file_entry");
	w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w));
	gtk_entry_set_text (GTK_ENTRY(w), b ? r->action->decode_path : "");

	/* tag for retrieval */
	b = (r->action->flags & RULE_ACTION_TAG_FOR_RETRIEVAL) != 0;
	w = lookup_widget (widget, "action_tag_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);

	/* retrieve body */
	b = (r->action->flags & RULE_ACTION_RETREIVE_BODY) != 0;
	w = lookup_widget (widget, "action_download_body_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);

	/* watch thread */
	b = (r->action->flags & RULE_ACTION_WATCH) != 0;
	w = lookup_widget (widget, "action_watch_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);

	/* bozoize */
	b = (r->action->flags & RULE_ACTION_BOZOIZE) != 0;
	w = lookup_widget (widget, "action_bozo_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);

	/* discard */
	b = (r->action->flags & RULE_ACTION_DISCARD) != 0;
	w = lookup_widget (widget, "action_discard_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);

	/* ignore */
	b = (r->action->flags & RULE_ACTION_IGNORE) != 0;
	w = lookup_widget (widget, "action_ignore_check");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), b);
}

static void
ui2rule (GtkWidget * widget, Rule * r)
{
	GtkWidget * w;

	g_return_if_fail (widget!=NULL);
	g_return_if_fail (r!=NULL);

	/**
	***  Update the group criteria
	**/

	replace_gstr (&r->group_wildcard, NULL);
	if (r->group_list != NULL) {
		pan_g_ptr_array_foreach (r->group_list, (GFunc)g_free, NULL);
		g_ptr_array_free (r->group_list, TRUE);
		r->group_list = NULL;
	}

	if (toggle_button_is_on (widget, "newsgroups_wildcard_rb"))
	{
		r->group_type = RULE_GROUP_WILDCARD;
		w = lookup_widget (widget, "newsgroup_wildcard_entry");
		r->group_wildcard = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
	}
	else if (toggle_button_is_on (widget, "newsgroups_all_rb"))
	{
		r->group_type = RULE_GROUP_ALL;
	}
	else
	{
		gint i;
		GtkCList * specific = GTK_CLIST(
			lookup_widget(widget, "newsgroups_specific_clist"));
		r->group_type = RULE_GROUP_LIST;
		r->group_list = g_ptr_array_new ();
		for (i=0; i!=specific->rows; ++i) {
			gchar * text = NULL;
			gtk_clist_get_text (specific, i, 0, &text);
			g_ptr_array_add (r->group_list, g_strdup(text));
		}
	}

	/**
	***  Update the criteria
	**/ 

	if (r->criteria != NULL) {
		pan_object_unref (PAN_OBJECT(r->criteria));
		r->criteria = NULL;
	}
	if (1)
	{
		GPtrArray * rules = g_ptr_array_new ();
		GtkCList * clist = GTK_CLIST(lookup_widget (widget, "criteria_clist"));
		gint row;

		/* build a RuleCriteria tree from the clist */
		for (row=0; row<clist->rows; ++row)
		{
			gint depth = 0;
			RuleCriteria * parent = NULL;
			RuleCriteria * rc = criteria_clist_get_rule_at_line (clist, row, &depth);

			/* link with the parent */
			g_assert ((depth <= rules->len+1) && "this rule is too deep!");
			g_assert ((rules->len ? depth>0 : TRUE) && "must be only one root node");
			if (depth != 0) {
				parent = RULE_CRITERIA (g_ptr_array_index(rules,depth-1));
				rule_criteria_add_child (parent, rc);
			}

			/* update the parent hierarchy */
			g_ptr_array_set_size (rules, depth+1);
			g_ptr_array_index (rules, depth) = rc;
		}

		/* update the Rule's criteria field */
		g_assert (rules->len>0);
		r->criteria = RULE_CRITERIA(g_ptr_array_index(rules,0));

		g_ptr_array_free (rules, TRUE);
	}

	/**
	***  Update the actions
	**/ 

	rule_action_clear (r->action);

	if (toggle_button_is_on (widget, "action_play_sound_check"))
	{
		r->action->flags |= RULE_ACTION_PLAY_SOUND;
		w = lookup_widget (widget, "sound_file_entry");
		w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w));
	       	replace_gstr (&r->action->sound_file, gtk_editable_get_chars (GTK_EDITABLE(w),0,-1));
	}
	if (toggle_button_is_on (widget, "action_show_alert_check"))
	{
		r->action->flags |= RULE_ACTION_SHOW_ALERT;
		w = lookup_widget (widget, "action_show_alert_entry");
		replace_gstr (&r->action->alert_message, gtk_editable_get_chars(GTK_EDITABLE(w),0,-1));
	}
#if 0
	if (toggle_button_is_on (widget, "action_append_to_file_check"))
	{
		r->action->flags |= RULE_ACTION_APPEND_TO_FILE;
		w = lookup_widget (widget, "append_file_entry");
		w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w));
		replace_gstr (&r->action->append_file, gtk_editable_get_chars(GTK_EDITABLE(w),0,-1));
	}
	if (toggle_button_is_on (widget, "action_forward_to_check"))
	{
		r->action->flags |= RULE_ACTION_FORWARD_TO;
		w = lookup_widget (widget, "action_forward_to_entry");
		replace_gstr (&r->action->forward_to, gtk_editable_get_chars(GTK_EDITABLE(w),0,-1));
	}
#endif
	if (toggle_button_is_on (widget, "action_mark_as_read_check"))
	{
		if (toggle_button_is_on (widget, "action_read_rb"))
			r->action->flags |= RULE_ACTION_MARK_READ;
		else
			r->action->flags |= RULE_ACTION_MARK_UNREAD;
	}
#if 0
	if (toggle_button_is_on (widget, "action_mark_as_important_check"))
	{
		if (toggle_button_is_on (widget, "action_important_rb"))
			r->action->flags |= RULE_ACTION_MARK_IMPORTANT;
		else
			r->action->flags |= RULE_ACTION_MARK_NORMAL;
	}
	if (toggle_button_is_on (widget, "action_mark_as_protected_check"))
	{
		if (toggle_button_is_on (widget, "action_protected_rb"))
			r->action->flags |= RULE_ACTION_MARK_PROTECTED;
		else
			r->action->flags |= RULE_ACTION_MARK_DELETABLE;
	}
#endif
	if (toggle_button_is_on (widget, "action_decode_check"))
	{
		r->action->flags |= RULE_ACTION_DECODE;
		w = lookup_widget (widget, "decode_file_entry");
		w = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(w));
		replace_gstr (&r->action->decode_path, gtk_editable_get_chars(GTK_EDITABLE(w),0,-1));
	}
	if (toggle_button_is_on (widget, "action_tag_check"))
	{
		r->action->flags |= RULE_ACTION_TAG_FOR_RETRIEVAL;
	}
	if (toggle_button_is_on (widget, "action_download_body_check"))
	{
		r->action->flags |= RULE_ACTION_RETREIVE_BODY;
	}
	if (toggle_button_is_on (widget, "action_watch_check"))
	{
		r->action->flags |= RULE_ACTION_WATCH;
	}
	if (toggle_button_is_on (widget, "action_bozo_check"))
	{
		r->action->flags |= RULE_ACTION_BOZOIZE;
	}
	if (toggle_button_is_on (widget, "action_discard_check"))
	{
		r->action->flags |= RULE_ACTION_DISCARD;
	}
	if (toggle_button_is_on (widget, "action_ignore_check"))
	{
		r->action->flags |= RULE_ACTION_IGNORE;
	}
}



/****
*****
*****  CALLBACKS
*****
****/


void
on_add_newsgroup_apply_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * subscribed
		= GTK_CLIST(lookup_widget (w, "newsgroups_subscribed_clist"));
	GtkCList * applies_to
		= GTK_CLIST(lookup_widget (w, "newsgroups_specific_clist"));

	gtk_clist_freeze (subscribed);
	gtk_clist_freeze (applies_to);

	/* if the subscribed list isn't in the 'applies' list, add it */
	while (subscribed->selection != NULL)
	{
		GList * l = subscribed->selection;
		gint idx = GPOINTER_TO_INT(l->data);
		Group * g = GROUP(gtk_clist_get_row_data(subscribed, idx));
		if (g != NULL)
		{
			gchar * name = (gchar*) group_get_readable_name (g);
			gchar ** text = &name;
			gint row = gtk_clist_insert (applies_to, -1, text);
			gtk_clist_set_row_data (applies_to, row, g);
			row = gtk_clist_find_row_from_data (subscribed, g);
			gtk_clist_remove (subscribed, row);
		}
	}

	gtk_clist_thaw (subscribed);
	gtk_clist_thaw (applies_to);
}


void
on_remove_newsgroup_from_apply_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * subscribed
		= GTK_CLIST(lookup_widget(w,"newsgroups_subscribed_clist"));
	GtkCList * applies_to
		= GTK_CLIST(lookup_widget(w,"newsgroups_specific_clist"));

	gtk_clist_freeze (subscribed);
	gtk_clist_freeze (applies_to);

	/* if the subscribed list isn't already in the 'applies' list, add it */
	while (applies_to->selection != NULL)
	{
		GList * l = applies_to->selection;
		gint idx = GPOINTER_TO_INT(l->data);
		Group * g = GROUP(gtk_clist_get_row_data(applies_to, idx));
		if (g != NULL)
		{
			gchar * name = (gchar*) group_get_readable_name (g);
			gchar ** text = &name;
			gint row = gtk_clist_insert (subscribed, -1, text);
			gtk_clist_set_row_data (subscribed, row, g);
			row = gtk_clist_find_row_from_data (applies_to, g);
			gtk_clist_remove (applies_to, row);
		}
	}

	gtk_clist_thaw (subscribed);
	gtk_clist_thaw (applies_to);

}

void
on_add_condition_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w;
	GtkWidget * widget = GTK_WIDGET(button);
	RuleCriteria * rc = rule_criteria_new ();
	gboolean err = FALSE;

	if (toggle_button_is_on (widget, "article_phrase_rb"))
	{
		gboolean is_regex;
		gboolean contains;
		const gchar * cpch;
		gchar * pch;

		rc->type = RULE_CRITERIA_PHRASE;

		/* phrase field */
		cpch = get_option_menu_selection (widget, "article_phrase_field_om");
		if (!strcmp(cpch, _("Subject")))
			rc->phrase_type = RULE_CRITERIA_PHRASE_SUBJECT;
		else if (!strcmp(cpch, _("Author")))
			rc->phrase_type = RULE_CRITERIA_PHRASE_AUTHOR;
		else {
			pan_warn_if_reached ();
			rc->phrase_type = RULE_CRITERIA_PHRASE_AUTHOR;
		}

		/* contains or does not contain */
		cpch = get_option_menu_selection (widget, "article_phrase_does_om");
		if (!strcmp(cpch, _("Does Not")))
			contains = FALSE;
		else if (!strcmp(cpch, _("Does")))
			contains = TRUE;
		else {
			pan_warn_if_reached ();
			contains = TRUE;
		}

		/* get the phrase and whether or not it's a regex */
		w = lookup_widget (widget, "article_phrase_entry");
		pch = gtk_editable_get_chars (GTK_EDITABLE(w), 0, -1);
		g_strstrip (pch);
		is_regex = toggle_button_is_on (widget, "article_phrase_regex_tb");
		rule_criteria_set_phrase (rc, pch, contains, is_regex);

		/* error checking */
		if (!is_nonempty_string(pch))
			err = TRUE;

		g_free (pch);
	}
#if 0
	else if (toggle_button_is_on (widget, "article_is_reply_to_my_message_rb"))
	{
		rc->type = RULE_CRITERIA_IS_REPLY_TO_MINE;
	}
#endif
	else if (toggle_button_is_on (widget, "article_lines_rb"))
	{
		rc->type = RULE_CRITERIA_EXCEEDS_N_LINES;
		w = lookup_widget (widget, "article_lines_sb");
		rc->exceeds_n = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(w));
	}
	else if (toggle_button_is_on (widget, "article_crossposted_rb"))
	{
		rc->type = RULE_CRITERIA_EXCEEDS_N_GROUPS;
		w = lookup_widget (widget, "article_crossposted_sb");
		rc->exceeds_n = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(w));
	}
	else if (toggle_button_is_on (widget, "article_days_rb"))
	{
		rc->type = RULE_CRITERIA_EXCEEDS_N_DAYS_OLD;
		w = lookup_widget (widget, "article_days_sb");
		rc->exceeds_n = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(w));
	}
	else if (toggle_button_is_on (widget, "article_thread_rb"))
	{
		const gchar * cpch;

		rc->type = RULE_CRITERIA_THREAD;

		/* get thread type */
		cpch = get_option_menu_selection (widget, "article_thread_om");
		if (!strcmp (cpch, _("Watched")))
			rc->thread_type = RULE_CRITERIA_THREAD_WATCHED;
		else if (!strcmp (cpch, _("Ignored")))
			rc->thread_type = RULE_CRITERIA_THREAD_IGNORED;
		else {
			rc->thread_type = RULE_CRITERIA_THREAD_WATCHED;
			pan_warn_if_reached ();
		}
	}
	else if (toggle_button_is_on (widget, "bozo_rb"))
	{
		rc->type = RULE_CRITERIA_BOZO;
	}
	else if (toggle_button_is_on (widget, "article_marked_rb"))
	{
		const gchar * cpch;

		rc->type = RULE_CRITERIA_MARKED_AS;

		/* get marked type */
		cpch = get_option_menu_selection (widget, "article_marked_om");
		if (!strcmp (cpch, _("Read")))
			rc->marked_type = RULE_CRITERIA_MARKED_READ;
		else if (!strcmp (cpch, _("Unread")))
			rc->marked_type = RULE_CRITERIA_MARKED_UNREAD;
		else {
			rc->marked_type = RULE_CRITERIA_MARKED_READ;
			pan_warn_if_reached ();
		}
	}
	else if (toggle_button_is_on (widget, "article_binary_rb"))
	{
		const gchar * cpch;

		rc->type = RULE_CRITERIA_BINARY;

		/* get binary type */
		cpch = get_option_menu_selection (widget, "article_binary_om");
		if (!strcmp (cpch, _("Complete Binary Attachment")))
			rc->binary_type = RULE_CRITERIA_BINARY_COMPLETE;
		else if (!strcmp (cpch, _("Incomplete Binary Attachment")))
			rc->binary_type = RULE_CRITERIA_BINARY_INCOMPLETE;
		else if (!strcmp (cpch, _("No Binary Attachments")))
			rc->binary_type = RULE_CRITERIA_BINARY_NONBINARY;
		else {
			pan_warn_if_reached ();
			rc->binary_type = RULE_CRITERIA_BINARY_NONBINARY;
		}
	}
	else
	{
		pan_warn_if_reached ();
	}

	/* if it's a valid rule, add that RuleCriteria to the UI */
	if (!err)
	{
		GtkCList * clist = GTK_CLIST(lookup_widget(widget,"criteria_clist"));
		criteria_clist_add (clist, rc);
	}

	pan_object_unref (PAN_OBJECT(rc));
}


void
on_condition_and_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkCList * clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button),"criteria_clist"));
	RuleCriteria * rc = rule_criteria_new ();
	rc->type = RULE_CRITERIA_AND;
	criteria_clist_add (clist, rc);
	pan_object_unref (PAN_OBJECT(rc));
}


void
on_condition_or_button_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkCList * clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button),"criteria_clist"));
	RuleCriteria * rc = rule_criteria_new ();
	rc->type = RULE_CRITERIA_OR;
	criteria_clist_add (clist, rc);
	pan_object_unref (PAN_OBJECT(rc));
}


void
on_condition_not_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkCList * clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button),"criteria_clist"));
	gint cur_row = clist->selection
		? GPOINTER_TO_INT(clist->selection->data)
		: clist->rows-1;

	if (cur_row != -1)
	{
		gint depth = 0;
		RuleCriteria * rc = criteria_clist_get_rule_at_line (clist, cur_row, &depth);
		if (rc != NULL)
		{
			gchar * new_line = NULL;

			/* negate the rc and write it back */
			rc->not = !rc->not;
			new_line = rule_criteria_tostring_single (rc, 0);
			gtk_clist_set_text (clist, cur_row, 0, new_line);
			criteria_clist_set_line_depth (clist, cur_row, depth);

			/* cleanup */
			g_free (new_line);
			pan_object_unref (PAN_OBJECT(rc));
		}
	}
}


void
on_line_up_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * widget = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget(widget,"criteria_clist"));
	if (clist->selection)
	{
		gint row = GPOINTER_TO_INT(clist->selection->data);
		if (row > 1)
		{
			RuleCriteria * rc;
			RuleCriteria * rc_prev;
			int depth = 0;
			int depth_prev = 0;

			/* get the rc of the current and previous line */
			rc      = criteria_clist_get_rule_at_line (clist, row,   &depth);
			rc_prev = criteria_clist_get_rule_at_line (clist, row-1, &depth_prev);

			if (depth_prev!=depth && !is_nesting_rc(rc_prev))
			{
				/* if moving into/out of a nest, just change depth */
				gint inc = depth_prev>=depth ? 1 : -1;
				criteria_clist_set_line_depth (clist, row, depth+inc);
				gtk_clist_select_row (clist, row, 0);
			}
			else
			{
				/* otherwise, really do move the line up */
				gtk_clist_remove (clist, row);
				criteria_clist_add_at_line (clist, rc, row-1);

				/* if the moved rc is an AND or OR, modify its new child */
				if (is_nesting_rc(rc))
					criteria_clist_inc_line_depth (clist, row, 1);

				gtk_clist_select_row (clist, row-1, 0);
			}

			/* cleanup */
			pan_object_unref (PAN_OBJECT(rc));
			pan_object_unref (PAN_OBJECT(rc_prev));
		}
	}
}


void
on_line_down_button_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * widget = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget(widget,"criteria_clist"));
	gint row = clist->selection
		? GPOINTER_TO_INT(clist->selection->data)
		: clist->rows-1;

	if (row>0)
	{
		if (row < clist->rows-1)
		{
			gint depth = 0;
			gint depth_next = 0;
			RuleCriteria * rc;
			RuleCriteria * rc_next;

			/* get the rc of the current and next line */
			rc      = criteria_clist_get_rule_at_line (clist, row,   &depth);
			rc_next = criteria_clist_get_rule_at_line (clist, row+1, &depth_next);

			if (depth_next!=depth)
			{
				/* moving out of a nesting -- change the depth */
				gint inc = depth_next > depth ? 1 : -1;
				criteria_clist_set_line_depth (clist, row, depth+inc);
				gtk_clist_select_row (clist, row, 0);
			}
			else
			{
				gtk_clist_remove (clist, row);

				/* if the new line is an AND or OR, modify its old child */
				if (is_nesting_rc(rc))
					criteria_clist_inc_line_depth (clist, row, -1);

				/* add the new line */
				criteria_clist_add_at_line (clist, rc, row+1);
				gtk_clist_select_row (clist, row+1, 0);
			}

			/* cleanup */
			pan_object_unref (PAN_OBJECT(rc));
			pan_object_unref (PAN_OBJECT(rc_next));
		}
		else if (row == clist->rows-1)
		{
			gint depth = criteria_clist_get_line_depth (clist, row);
			if (depth > 1)
				criteria_clist_inc_line_depth (clist, row, -1);
		}
	}
}


void
on_delete_line_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * widget = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget(widget,"criteria_clist"));
	gint cur_row = 0;

	if (clist->selection)
		cur_row = GPOINTER_TO_INT(clist->selection->data);

	if (cur_row > 0)
	{
		gint depth = 0;
		RuleCriteria * rc = criteria_clist_get_rule_at_line (clist, cur_row, &depth);
		gtk_clist_remove (clist, cur_row);

		/* promote all its children */
		if (is_nesting_rc(rc)) {
			gint row;
			for (row=cur_row; row<clist->rows; ++row) {
				int row_depth = criteria_clist_get_line_depth (clist, row);
				if (row_depth <= depth)
					break;
				criteria_clist_inc_line_depth (clist, row, -1);
			}
		}

		gtk_clist_select_row (clist, cur_row, 0);
	}
}


void
on_insert_from_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{

}


void
on_test_sound_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * widget = GTK_WIDGET(button);
	GtkWidget * file_entry = lookup_widget (widget, "sound_file_entry");
	GtkWidget * entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY(file_entry));
	open_outside_file (gtk_entry_get_text (GTK_ENTRY(entry)));
}

void
on_rule_edit_dialog_clicked            (GnomeDialog     *edit_dialog,
                                        gint             arg1,
                                        gpointer         user_data)
{
	GtkWidget * widget = GTK_WIDGET(edit_dialog);
	Rule * r = RULE(gtk_object_get_user_data(GTK_OBJECT(edit_dialog)));

	if (arg1 == 0)  /* okay */
	{
		GPtrArray * errors = g_ptr_array_new ();
		Rule * r_tmp;

		/* make a temp copy and check it for errors */
		r_tmp = rule_new ();
		ui2rule (widget, r_tmp);

		/* error check #1: check for no criteria */
		if (r_tmp->criteria->children==NULL || r_tmp->criteria->children->len==0)
			g_ptr_array_add (errors, g_strdup(_("ERROR: No conditions specified!")));

		if (errors->len != 0) /* errors; don't update */
		{
			GString * err_str = g_string_new (NULL);
			guint i;
			for (i=0; i!=errors->len; ++i)
				g_string_sprintfa (err_str, "%s\n", (gchar*)g_ptr_array_index(errors,i));
			pan_error_dialog_parented (GTK_WINDOW(edit_dialog), "%s", err_str->str);
			g_string_free (err_str, TRUE);
		}
		else /* no errors; update */
		{
			/* update the rule */
			ui2rule (widget, r);

			/* add this to the rules list, if need be */
			if (!is_nonempty_string(r->name))
			{
				gint row;
				GtkCList * clist;
				gchar * text [3] = { "larry", "moe", "curly" }; /* placeholders */
	
				/* add the new rule to the rule manager (updating its name) */
				rule_manager_add_new_rule (r);
	
				/* add the new rule to the main rule dialog list */
				pan_lock ();
				clist = GTK_CLIST(lookup_widget(_rule_dialog,"rules_dialog_clist"));
				pan_warn_if_fail (is_nonempty_string(r->name));
				row = gtk_clist_insert (clist, -1, text);
				gtk_clist_set_row_data (clist, row, r);
				pan_unlock ();

				/* update the list ui */
				main_list_update_rule_row (clist, r);
			}

			/* close the dialog */
			gnome_dialog_close_hides (edit_dialog, TRUE); // FIXME
			gnome_dialog_close (edit_dialog);
		}

		/* cleanup */
		pan_g_ptr_array_foreach (errors, (GFunc)g_free, NULL);
		g_ptr_array_free (errors, TRUE);
		pan_object_unref (PAN_OBJECT(r_tmp));
	}
	else if (arg1 == 1) /* cancel */
	{
		/* if this rule was created from the "new rule" button, free it */
		if (!is_nonempty_string(r->name))
			pan_object_unref (PAN_OBJECT(r));

		/* close the dialog */
		gnome_dialog_close_hides (edit_dialog, TRUE); // FIXME
		gnome_dialog_close (edit_dialog);
	}
}



/**
***
***
**/


static void
main_list_edit_rule (GtkCList * clist, Rule * r)
{
	dialog_edit_rule (r);
}

void
on_main_rule_add_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	main_list_edit_rule (NULL, rule_new());
}

void
on_main_rule_edit_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
	Rule * r;
	
	r = main_list_get_selected_rule (clist);
	if (r != NULL)
		main_list_edit_rule (clist, r);
}

void
on_main_rule_copy_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
	Rule * r;
	
	r = main_list_get_selected_rule (clist);
	if (r != NULL)
		main_list_edit_rule (clist, rule_dup(r));
}

void
on_main_rule_delete_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget(w,"rules_dialog_clist"));
	Rule * rule = NULL;
	int row = -1;

	/* find the current rule... */
	if (clist->selection != NULL)
	{
		row = GPOINTER_TO_INT(clist->selection->data);
		rule = RULE(gtk_clist_get_row_data(clist, row));
	}

	/* if there is one, free it... */
	if (rule != NULL)
	{
		rule_manager_destroy_rule (rule);
		rule = NULL;
		gtk_clist_remove (clist, row);
	}
}




void
on_main_rule_up_button_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{

}


void
on_main_rule_down_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{

}



void
on_apply_to_incoming_button_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
	Rule * r;
	
	r = main_list_get_selected_rule (clist);
	if (r != NULL)
	{
		r->apply_to_incoming = !r->apply_to_incoming;
		main_list_update_rule_row (clist, r);
	}
}

typedef struct
{
	GPtrArray * groups;
	Rule * rule;
}
ApplyNow;

static gchar*
apply_now_describe (const StatusItem * item)
{
	return g_strdup (_("Applying Rule to Selected Groups"));
}

static void
apply_now_thread (void * data)
{
	StatusItem * status;
	ApplyNow * now = (ApplyNow*) data;
	Rule * rule = now->rule;
	GPtrArray * groups = (GPtrArray *) now->groups;
	guint i;

	/* add a gui feedback tool */
	status = STATUS_ITEM(status_item_new(apply_now_describe));
	status_item_set_active (status, TRUE);
	status_item_emit_init_steps (status, groups->len);

	/* process each group in turn */
	for (i=0; i!=groups->len; ++i)
	{
		Group * g = GROUP(g_ptr_array_index(groups,i));
		GPtrArray * articles;

		status_item_emit_status_va (status, _("Loading group `%s'"), group_get_readable_name(g));
		group_ref_articles (g, NULL);
		group_thread_if_needed (g, NULL);

		status_item_emit_status_va (status, _("Applying rule `%s' to Group `%s'"), rule->name, group_get_readable_name(g));

		articles = group_get_article_array (g);
		rule_apply (rule, articles);
		g_ptr_array_free (articles, TRUE);

		status_item_emit_status_va (status, _("Saving group `%s'"), group_get_readable_name(g));
		group_unref_articles (g, NULL);

		status_item_emit_next_step (status);
	}

	/* cleanup ui */
	status_item_set_active (status, FALSE);
	pan_object_unref (PAN_OBJECT(status));

	/* cleanup memory */
	g_ptr_array_free (groups, TRUE);
	g_free (data);
}

static void
apply_rules_impl (GtkWidget * w, GPtrArray * groups)
{
	GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
	Rule * rule = main_list_get_selected_rule (clist);

	if (rule!=NULL && groups->len)
	{
		ApplyNow * now;
	        pthread_t thread;

		now = g_new (ApplyNow, 1);
		now->groups = groups;
		now->rule = rule;
		pthread_create (&thread, NULL, (void*)apply_now_thread, now);
		pthread_detach (thread);
	}
	else
	{
		g_ptr_array_free (groups, TRUE);
	}
}
void
on_rule_apply_selected_button_clicked (GtkButton * button, gpointer user_data)
{
	apply_rules_impl (GTK_WIDGET(button), grouplist_get_selected_groups ());
}
void
on_rule_apply_all_button_clicked (GtkButton * button, gpointer user_data)
{
	GPtrArray * a = g_ptr_array_new ();
	Server * s = grouplist_get_server ();
	pan_g_ptr_array_assign  (a, s->groups->pdata, s->groups->len);
	apply_rules_impl (GTK_WIDGET(button), a);
}
void
on_rule_apply_subscribed_button_clicked (GtkButton * button, gpointer user_data)
{
	guint i;
	Server * s = grouplist_get_server ();
	GPtrArray * a = g_ptr_array_new ();
	GPtrArray * groups = a==NULL ? NULL : s->groups;
	for (i=0; groups!=NULL && i!=s->groups->len; ++i) {
		Group * g = GROUP(g_ptr_array_index(groups,i));
		if (group_is_subscribed(g))
			g_ptr_array_add (a, g);
	}
	apply_rules_impl (GTK_WIDGET(button), a);
}


static void
rename_dialog_cb (gchar * string, gpointer data)
{
	GtkWidget * w = GTK_WIDGET(data);
	GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
	Rule * r;
	
	r = main_list_get_selected_rule (clist);
	if (r != NULL)
	{
		replace_gstr (&r->name, g_strdup(string));
		main_list_update_rule_row (clist, r);
	}
}

void
on_main_rule_rename_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(button);
	GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
	Rule * r;
	
	r = main_list_get_selected_rule (clist);
	if (r != NULL)
	{
		GtkWidget * rename_dialog = gnome_request_dialog (
			FALSE,
			_("New Rule Name"),
			r->name,
			-1,
			rename_dialog_cb,
			w,
			NULL);
		gtk_widget_show (rename_dialog);
	}
}


void
on_rule_edit_dialog_realize            (GtkWidget       *widget,
                                        gpointer         user_data)
{
}


void
on_rule_dialog_clicked                 (GnomeDialog     *gnomedialog,
                                        gint             arg1,
                                        gpointer         user_data)
{
	gnome_dialog_close_hides (gnomedialog, TRUE); // FIXME
	gnome_dialog_close (gnomedialog);
}

gint
on_rule_dialog_close                   (GnomeDialog     *object,
                                        gpointer         user_data)
{
	rule_manager_save_rules ();
	return FALSE;
}

void
on_rule_dialog_realize                 (GtkWidget       *widget,
                                        gpointer         user_data)
{
	GtkCList * clist = GTK_CLIST(lookup_widget (widget, "rules_dialog_clist"));
	guint i;
	const GPtrArray * rules = rule_manager_get_rules ();

	/* populate the rule clist */
	for (i=0; i!=rules->len; ++i)
	{
		Rule * rule = RULE(g_ptr_array_index(rules,i));
		gchar * text [3];
		int row;
		gchar percent_str[32];
		double percent
			= 100.0*rule->hits/(double)(rule->tries?rule->tries:1);

		text[0] = rule->name;
		text[1] = rule->apply_to_incoming ? _("Yes") : "";
		text[2] = percent_str;
		sprintf (percent_str, "%.0f%%", percent);

		row = gtk_clist_insert (clist, -1, text);
		gtk_clist_set_row_data (clist, row, rule);
	}
}

void
on_rules_dialog_clist_select_row       (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	GtkWidget * w = GTK_WIDGET(clist);
	GdkEventButton * bevent;
	
	bevent = (GdkEventButton*) event;
	if (bevent == NULL)
		return;

	/* double-clicking a row will edit the rule on that row */
	if (bevent->button==1 && bevent->type==GDK_2BUTTON_PRESS)
	{
		GtkCList * clist = GTK_CLIST(lookup_widget (w, "rules_dialog_clist"));
		Rule * r = main_list_get_selected_rule (clist);
		if (r != NULL)
			main_list_edit_rule (clist, r);
	}
}

