/*
 * Pan - A Newsreader for X
 * Copyright (C) 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"
#include "file-headers.h"
#include "file-grouplist.h"
#include "group.h"
#include "log.h"
#include "prefs.h"
#include "server.h"
#include "util.h"

static void
file_grouplist_destroy (const Server * server)
{
	gchar * path;
	debug_enter ("file_grouplist_destroy");

	g_return_if_fail (server != NULL);

	path = g_strdup_printf ("/%s/%s.idx", data_dir, server->name);
	if (file_exists (path)) {
		remove (path);
	}
	g_free (path);

	path = g_strdup_printf ("/%s/%s.dat", data_dir, server->name);
	if (file_exists (path)) {
		remove (path);
	}
	g_free (path);

	path = g_strdup_printf ("/%s/%s", data_dir, server->name);
	if (file_exists (path)) {
		remove (path);
	}
	g_free (path);

	debug_exit ("file_grouplist_destroy");
}

gboolean
file_grouplist_exists (const Server* server)
{
	gchar * path;
	gboolean retval;
	debug_enter ("file_grouplist_exists");

       	path = g_strdup_printf ("/%s/%s.idx", data_dir, server->name);
       	retval = file_exists (path);
	g_free (path);

	debug_exit ("file_grouplist_exists");
	return retval;
}

void
file_grouplist_load (Server * server, StatusItem * status)
{
	gchar * path;
	struct timeval start;
	struct timeval finish;
	double diff;
	GArray *dat, *idx;
	gboolean success = FALSE;
	Group ** groups = NULL;
	int group_qty = 0;
	glong version = -1;
	debug_enter ("file_grouplist_load");

	gettimeofday (&start, NULL);

	/* open the index file */
	path = g_strdup_printf ("/%s/%s.idx", data_dir, server->name);
	idx = read_file (path);
	g_free (path);

	/* open the data file */
	path = g_strdup_printf ("/%s/%s.dat", data_dir, server->name);
	dat = read_file (path);
	g_free (path);

	if (dat!=NULL && idx!=NULL)
	{
		const gchar * dat_str = dat->data;
		const gchar * march_idx = idx->data;
		version = get_next_token_int (march_idx, '\n', &march_idx);

		if (version==1 || version==2)
		{
			long i;
			glong qty = get_next_token_long (march_idx, '\n', &march_idx);

			groups = g_new (Group*, qty);

			if (status != NULL)
				status_item_emit_init_steps (status, qty);

			for (i=0; i<qty; ++i)
			{
				gchar * newsrc;
				Group * g;
				glong l;

				/* group name */
				l = get_next_token_long (march_idx, '\n', &march_idx);
				g = group_new (server, dat_str+l);

				/* readable name */
				l = get_next_token_long (march_idx, '\n', &march_idx);
				if (l != -1)
					g->readable_name = server_chunk_string (server, dat_str+l);

				/* description */
				l = get_next_token_long (march_idx, '\n', &march_idx);
				if (l != -1)
					g->description = server_chunk_string (server, dat_str+l);

				/* download dir */
				l = get_next_token_long (march_idx, '\n', &march_idx);
				if (l != -1)
					g->download_dir = server_chunk_string (server, dat_str+l);

				/* newsrc string */
				newsrc = NULL;
				l = get_next_token_long (march_idx, '\n', &march_idx);
				if (l != -1)
					newsrc = g_strdup (dat_str+l);

				g->flags            = (gint8)   get_next_token_int   (march_idx, '\n', &march_idx);
				g->state_filter     = (guint16) get_next_token_int   (march_idx, '\n', &march_idx);
				g->sort_style       = (gint8)   get_next_token_int   (march_idx, '\n', &march_idx);
				g->article_low      = (gulong)  get_next_token_ulong (march_idx, '\n', &march_idx);
				g->article_high     = (gulong)  get_next_token_ulong (march_idx, '\n', &march_idx);
				g->article_high_old = (gulong)  g->article_high;
				g->article_qty      = (gint32)  get_next_token_int   (march_idx, '\n', &march_idx);
				g->article_read_qty = (gint32)  get_next_token_int   (march_idx, '\n', &march_idx);

				if (version==1)
				{
					g->permission = 'y'; /* a reasonable default... */
				}
				else if (version==2)
				{
					gchar * foo = get_next_token_str (march_idx, '\n', &march_idx);
					g->permission = foo ? foo[0] : '\0';
					g_free (foo);
				}

				if (is_nonempty_string(newsrc))
					newsrc_init (group_get_newsrc(g), newsrc, g->article_low, g->article_high);

				groups[group_qty++] = g;

				if (status != NULL) {
					status_item_emit_next_step (status);
					if (!(group_qty % 1024))
						status_item_emit_status_va (status,
							_("Loading %u of %d groups"), group_qty, qty);
				}

				g_free (newsrc);
			}

			success = TRUE;
		}
		else
		{
			pan_error_dialog (_("Unsupported data version %d for `%s' data file"),
				version,
				server->name);
		}
	}

	if (group_qty != 0)
		server_add_groups (server, groups, group_qty, NULL, NULL);

	/* stats */	
	gettimeofday (&finish, NULL);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/1000000.0;
	log_add_va (LOG_INFO, _("Loaded %d groups for server `%s' in %.1f seconds (%.0f groups/sec), format %d"),
		server->groups->len,
		server->name,
		diff,
		server->groups->len/(fabs(diff)<0.001?0.001:diff),
		(int)version);

	/* cleanup */
	if (dat!=NULL) g_array_free (dat, TRUE);
	if (idx!=NULL) g_array_free (idx, TRUE);
	g_free (groups);

	debug_exit ("file_grouplist_load");
}

 
void
file_grouplist_save (Server * server, StatusItem * status)
{
	guint i;
	FILE * idx_fp;
	FILE * dat_fp;
	gchar * pch;
	gchar * idx_path;
	gchar * dat_path;
	long pos = 0;
	struct timeval start;
	struct timeval finish;
	double diff;
	GString * str;
	debug_enter ("file_grouplist_save");

	g_return_if_fail (server!=NULL);
	g_return_if_fail (server->groups!=NULL);
	g_return_if_fail (is_nonempty_string(data_dir));

	/* trivial case #1: no change, so no need to save */
	if (server->groups_dirty == FALSE)
		return;

	/* trivial case #2: nothing to save */
	if (server->groups->len == 0) {
		file_grouplist_destroy (server);
		return;
	}

	/**
	***  Save the Groups
	**/

	gettimeofday (&start, NULL);

	/* open index file */
	idx_path = g_strdup_printf ("/%s/%s.idx.tmp", data_dir, server->name);
	idx_fp = fopen (idx_path, "w+");
	if (idx_fp == NULL) {
		g_free (idx_path);
		return;
	}

	/* open data file */
	dat_path = g_strdup_printf ("/%s/%s.dat.tmp", data_dir, server->name);
	dat_fp = fopen (dat_path, "w+");
	if (dat_fp == NULL) {
		fclose (idx_fp);
		remove (idx_path);
		g_free (idx_path);
		g_free (dat_path);
		return;
	}

	/* Write the group information... */
	pos = 0;
	str = g_string_sized_new (1024);
	g_string_sprintfa (str, "2\n%u\n", server->groups->len);
	for (i=0; i!=server->groups->len; ++i)
	{
		Group * group = GROUP(g_ptr_array_index (server->groups, i));

		gchar * pch = group_has_newsrc(group)
			? newsrc_get_read_str(group_get_newsrc(group))
			: NULL;

		/* write the group name */
		const long name_idx = get_string_offset (NULL, NULL, group->name, dat_fp, &pos);
		const long read_idx = get_string_offset (NULL, NULL, group->readable_name, dat_fp, &pos);
		const long desc_idx = get_string_offset (NULL, NULL, group->description, dat_fp, &pos);
		const long path_idx = get_string_offset (NULL, NULL, group->download_dir, dat_fp, &pos);
		const long news_idx = get_string_offset (NULL, NULL, pch, dat_fp, &pos);

		/* write the non-string fields. */
		g_string_sprintfa (str,
			"%ld\n" "%ld\n" "%ld\n" "%ld\n" "%ld\n"
			"%u\n" "%u\n" "%d\n" "%lu\n" "%lu\n" "%d\n" "%d\n" "%c\n",
			name_idx, read_idx, desc_idx, path_idx, news_idx,
			(unsigned)(group->flags & ~GROUP_NEW),
			(unsigned)group->state_filter,
			(int)group->sort_style,
			(unsigned long)group->article_low,
			(unsigned long)group->article_high,
			(int)group->article_qty,
			(int)group->article_read_qty,
			(char)(group->permission!='\0' ? group->permission : '?'));

		/* flush */
		if (!(i%64)) {
			fwrite (str->str, sizeof(char), str->len, idx_fp);
			g_string_truncate (str, 0);
		}

		/* let the user know what's going on */
		if (status != NULL) {
			status_item_emit_next_step (status);
			if (!(i % 512))
				status_item_emit_status_va (status,
					_("Saved %d of %u groups"),
					i, server->groups->len);
		}

		/* cleanup */
		g_free (pch);
	}

	fwrite (str->str, sizeof(char), str->len, idx_fp);

	/* the write went okay; update the idx file */
	fclose (idx_fp);
	pch = g_strdup_printf ("/%s/%s.idx", data_dir, server->name);
	remove (pch);
	rename (idx_path, pch);
	g_free (idx_path);
	g_free (pch);

	/* the write went okay; update the dat file */
	fclose (dat_fp);
	pch = g_strdup_printf ("/%s/%s.dat", data_dir, server->name);
	remove (pch);
	rename (dat_path, pch);
	g_free (dat_path);
	g_free (pch);

	/* cleanup */
	server->groups_dirty = FALSE;

	/* finish timing stats */
	gettimeofday (&finish, NULL);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/1000000.0;
	log_add_va (LOG_INFO, _("Saved %d groups in \"%s\" in %.1f seconds (%.0f groups/sec)"),
		server->groups->len,
		server->name,
		diff,
		server->groups->len/(fabs(diff)<0.001?0.001:diff));

	debug_exit ("file_grouplist_save");
}
