/*
 * Pan - A Newsreader for X
 * Copyright (C) 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "log.h"
#include "rule.h"
#include "file-rules.h"
#include "status-item.h"
#include "prefs.h"
#include "util.h"

/***
****
****
****
***/

void
file_rules_load (GPtrArray * setme, StatusItem * status)
{
	gboolean success = FALSE;
        gchar * path;
	struct timeval start;
	struct timeval finish;
	double diff;
	GArray * dat;

	gettimeofday (&start, NULL);

	/* open the rule file */
	path = g_strdup_printf ("/%s/rules.dat", data_dir);
	dat = read_file (path);
	g_free (path);

	if (dat!=NULL)
	{
		const gchar * march = dat->data;
		const glong version = get_next_token_int (march, '\n', &march);

		if (version == 1)
		{
			int i;
			int qty = get_next_token_int (march, '\n', &march);

			pan_g_ptr_array_reserve (setme, qty);

			for (i=0; i!=qty; ++i)
			{
				int j;
				gchar * pch;
				Rule * r = rule_new ();

				r->name  = get_next_token_str (march, '\n', &march);
				r->hits  = get_next_token_int (march, '\n', &march);
				r->tries = get_next_token_int (march, '\n', &march);
				r->apply_to_incoming = (gboolean) get_next_token_int (march, '\n', &march);

				/* newsgroup criteria */
				r->group_type = get_next_token_int (march, '\n', &march);
				switch (r->group_type)
				{
					case RULE_GROUP_ALL:
						break;

					case RULE_GROUP_WILDCARD:
						r->group_wildcard = get_next_token_str (march, '\n', &march);
						break;

					case RULE_GROUP_LIST:
					{
						int group_qty = get_next_token_int (march, '\n', &march);
						r->group_list = g_ptr_array_new ();
						pan_g_ptr_array_reserve (r->group_list, group_qty);
						for (j=0; j<group_qty; ++j) {
							gchar * name = get_next_token_str (march, '\n', &march);
							g_ptr_array_add (r->group_list, name);
						}
						break;
					}
				}

				/* more complex criteria */
				pch = get_next_token_str (march, '\n', &march);
				g_assert (!strcmp(pch, "<criteria>"));
				g_free (pch);
				if (TRUE)
				{
					gchar * endkey = "\n</criteria>\n";
					gchar * endptr = strstr (march, endkey);
					*endptr = '\0';
					r->criteria = string_to_rule_criteria_recursive (march);
					march = endptr + strlen(endkey);
				}

				/* actions */
				if (TRUE)
				{
					RuleAction * a   = r->action;
					a->flags         = get_next_token_ulong (march, '\n', &march);
					a->sound_file    = get_next_token_str (march, '\n', &march);
					a->alert_message = get_next_token_str (march, '\n', &march);
					if (is_nonempty_string (a->alert_message))
						g_strdelimit (a->alert_message, "\r", '\n');
					a->append_file   = get_next_token_str (march, '\n', &march);
					a->forward_to    = get_next_token_str (march, '\n', &march);
					a->decode_path   = get_next_token_str (march, '\n', &march);
					a->is_read       = (gboolean) get_next_token_int (march, '\n', &march);
					a->is_important  = (gboolean) get_next_token_int (march, '\n', &march);
					a->is_protected  = (gboolean) get_next_token_int (march, '\n', &march);
				}

				/* let the user know what we're doing */
				if (status != NULL) {
					status_item_emit_next_step (status);
					status_item_emit_status_va (status,
						_("Loaded %u of %d rules"), setme->len, qty);
				}

				/* add the article to the group */
				g_ptr_array_add (setme, r);
			}

			success = TRUE;
		}
		else
		{
			pan_error_dialog (_("Unsupported data version: %d"), version);
		}
	}

	/* timing stats */
	gettimeofday (&finish, NULL);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/1000000.0;
	log_add_va (LOG_INFO, _("Got %d rules in %.1f seconds (%.0f rules/sec)"),
		setme->len,
		diff,
		setme->len/(fabs(diff)<0.001?0.001:diff));

	/* cleanup */
	if (dat != NULL) g_array_free (dat, TRUE);
}

void
file_rules_save (GPtrArray * rules, StatusItem * status)
{
	struct timeval start;
	struct timeval finish;
	double diff;
	guint i;
	FILE * fp;
	gchar * pch;
	gchar * path;
	long pos = 0;

	g_return_if_fail (rules!=NULL);

	gettimeofday (&start, NULL);

	/**
	***  If nothing to Save
	**/

	if (rules->len == 0) {
		file_rules_destroy ();
		return;
	}

	/**
	***  Save the Rules
	**/

	/* open data file */
	path = g_strdup_printf ("/%s/rules.dat.tmp", data_dir);
	fp = fopen (path, "w+");
	if (fp == NULL) {
		g_free (path);
		return;
	}

	/* Write DATBASE_VERSION */
	fprintf (fp, "1\n%u\n", rules->len);

	/* Write the rules... */
	pos = 0;
	for (i=0; i!=rules->len; ++i)
	{
		Rule * r = RULE(g_ptr_array_index(rules,i));

		/* name, hits, tries, apply */
		fprintf (fp,
			"%s\n"
			"%d\n" "%d\n"
			"%d\n",
			r->name,
			r->hits, r->tries,
			(int)r->apply_to_incoming);

		/* newsgroup criteria */
		fprintf (fp, "%d\n", (int)r->group_type);
		switch (r->group_type)
		{
			case RULE_GROUP_ALL:
				break;

			case RULE_GROUP_WILDCARD:
				fprintf (fp, "%s\n", r->group_wildcard);
				break;

			case RULE_GROUP_LIST:
			{
				guint j;
				fprintf (fp, "%u\n", r->group_list->len);
				for (j=0; j!=r->group_list->len; ++j)
					fprintf (fp, "%s\n", (gchar*)g_ptr_array_index(r->group_list,j));
				break;
			}
		}

		/* other criteria */
		if (r->criteria != NULL)
		{
			GString * s = g_string_new (NULL);
			rule_criteria_tostring_recursive (r->criteria, s, 0);

			fprintf (fp, "<criteria>\n");
			fprintf (fp, s->str);
			fprintf (fp, "</criteria>\n");

			g_string_free (s, TRUE);
		}

		/* actions */
		g_assert (r->action != NULL);
		if (TRUE)
		{
			RuleAction * a = r->action;
			gchar * tmp_message = g_strdup (a->alert_message ? a->alert_message : "");
			g_strdelimit (tmp_message, "\n", '\r');
			fprintf (fp, "%lu\n"
				"%s\n" "%s\n" "%s\n"
				"%s\n" "%s\n"
				"%d\n" "%d\n" "%d\n",
				a->flags,
				(a->sound_file!=NULL ? a->sound_file : ""),
				tmp_message,
				(a->append_file!=NULL ? a->append_file : ""),
				(a->forward_to!=NULL ? a->forward_to : ""),
				(a->decode_path!=NULL ? a->decode_path : ""),
				(a->is_read ? 1 : 0),
				(a->is_important ? 1 : 0),
				(a->is_protected ? 1 : 0));
			g_free (tmp_message);
		}
	}

	/* the write went okay; move the file over */
	fclose (fp);
	pch = g_strdup_printf ("/%s/rules.dat", data_dir);
	remove (pch);
	rename (path, pch);
	g_free (path);
	g_free (pch);

	/* timing stats */
	gettimeofday (&finish, NULL);
	diff = finish.tv_sec - start.tv_sec;
	diff += (finish.tv_usec - start.tv_usec)/1000000.0;
	log_add_va (LOG_INFO, _("Saved %d rules in %.1f seconds (%.0f rules/sec)"),
		rules->len,
		diff,
		rules->len/(fabs(diff)<0.001?0.001:diff));
}

void
file_rules_destroy (void)
{
	gchar * path;

	path = g_strdup_printf ("/%s/rules.dat", data_dir);
	remove (path);
	g_free (path);
}
