/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <gmime/gmime-utils.h>

#include "debug.h"
#include "globals.h"
#include "gnksa.h"
#include "grouplist.h"
#include "log.h"
#include "message-check.h"
#include "message-send.h"
#include "nntp.h"
#include "prefs.h"
#include "queue.h"
#include "task-func-ptr.h"
#include "smtp.h"
#include "util.h"

enum MessageSendResult
{
	SMTP_FAILED     = (1<<0),
	NNTP_FAILED     = (1<<1)
};

typedef struct
{
	Group * original_group;
	Article * article;
	gboolean do_post;
	gboolean do_mail;
}
SendStruct;

static gint send_article_now (StatusItem*, PanSocket*, Server*, SendStruct*);

/***
****  PRIVATE UTILITIES
***/

static GString *
build_nntp_header (GString        * postme,
                   Article        * article,
                   Server         * server,
                   const gchar    * subject_append,
		   gboolean         force_new_message_id)
{
	gchar * tmp;
	const gchar * pch;
	debug_enter ("build_nntp_header");

	/* From: */
	tmp = article_get_author_str (article);
	pan_warn_if_fail (is_nonempty_string (tmp));
	replace_gstr (&tmp, g_mime_utils_8bit_header_encode(tmp));
	g_string_sprintfa (postme, "From: %s\r\n", tmp);
	g_free (tmp);

	/* Subject: */
	pan_warn_if_fail (is_nonempty_string (article->subject));
	tmp = g_mime_utils_8bit_header_encode (article->subject);
	if (subject_append)
		g_string_sprintfa (postme, "Subject: %s %s\r\n", tmp,
				subject_append);
	else
		g_string_sprintfa (postme, "Subject: %s\r\n", tmp);
	g_free (tmp);

	/* Date: */
	tmp = rfc822_date_generate (0);
	replace_gstr (&tmp, g_mime_utils_8bit_header_encode (tmp));
	g_string_sprintfa (postme, "Date: %s\r\n", tmp);
	g_free (tmp);

	/* Newsgroups: */
	pch = article_get_header (article, HEADER_NEWSGROUPS);
	tmp = g_mime_utils_8bit_header_encode (pch);
	pan_warn_if_fail (is_nonempty_string (tmp));
	g_string_sprintfa (postme, "Newsgroups: %s\r\n", tmp);
	g_free (tmp);
	pch = NULL;

	/* Path: */
	g_string_sprintfa (postme, "Path: not-for-mail\r\n");

	/* Followup-To: */
	pch = article_get_header (article, HEADER_FOLLOWUP_TO);
	tmp = g_mime_utils_8bit_header_encode (pch);
	if (is_nonempty_string (tmp))
		g_string_sprintfa (postme, "Followup-To: %s\r\n", tmp);
	g_free (tmp);

	/* Reply-To: */
	pch = article_get_header (article, HEADER_REPLY_TO);
	tmp = g_mime_utils_8bit_header_encode (pch);
	if (is_nonempty_string (tmp))
		g_string_sprintfa (postme, "Reply-To: %s\r\n", tmp);
	g_free (tmp);

	/* Organization: */
	pch = article_get_header (article, HEADER_ORGANIZATION);
	tmp = g_mime_utils_8bit_header_encode (pch);
	if (is_nonempty_string (tmp))
		g_string_sprintfa (postme, "Organization: %s\r\n", tmp);
	g_free (tmp);

	/* Message ID: */
	if (server->gen_msgid) {
		tmp = g_strdup (article_get_header (article, HEADER_MESSAGE_ID));
		if (force_new_message_id || !is_nonempty_string(tmp))
			replace_gstr (&tmp, gnksa_generate_message_id(article->author_addr));
		g_string_sprintfa (postme, "Message-ID: %s\r\n", tmp);
		g_free (tmp);
	}

	/* References: */
	pch = article_get_header (article, HEADER_REFERENCES);
	if (is_nonempty_string (pch))
		g_string_sprintfa (postme, "References: %s\r\n", pch);

	/* User Agent: */
	g_string_append (postme, "User-Agent: Pan/" VERSION " (Unix)\r\n");

	/* Mime -- this hardcoding will go away soon; this is a stopgap */
	g_string_append (postme, "Mime-Version: 1.0\r\n");
	g_string_append (postme, "Content-Type: text/plain; charset=ISO-8859-1\r\n");
	g_string_append (postme, "Content-Transfer-Encoding: 8bit\r\n");

	/* Extra headers: */
	if (1) {
		GPtrArray * a = article_get_all_headers (article);
		guint i;
		for (i=0; i!=a->len; i+=2)
		{
			const gchar * key = (const gchar*) g_ptr_array_index (a, i);
			const gchar * val = (const gchar*) g_ptr_array_index (a, i+1);
			if (article_header_is_extra (key))
			{
				gchar * enkey = g_mime_utils_8bit_header_encode (key);
				gchar * enval = g_mime_utils_8bit_header_encode (val);
				g_string_sprintfa (postme, "%s: %s\r\n", enkey, enval);
				g_free (enkey);
				g_free (enval);
			}
		}
		g_ptr_array_free (a, TRUE);
	}

	/* separator that goes between headers and body */
	g_string_sprintfa (postme, "\r\n");

	debug_exit ("build_nntp_header");
	return (postme);
}

/**
 * Build the message that will be passed to the news server.
 * Error-checking is done elsewhere; by the time we reach this
 * point we assume it's in the clear and only work on building
 * the message.
 */
static GPtrArray*
build_nntp_message (Server * server, Article * article)
{
	GPtrArray * bodies;
	gchar * tmp; /* *append; */
	GString * body;
	GString * postme;
	debug_enter ("build_nntp_message");

	/* sanity checks... */
	g_return_val_if_fail (article!=NULL, NULL);
	g_return_val_if_fail (article_get_header(article,HEADER_NEWSGROUPS), 0);

       	bodies = g_ptr_array_new();
#if 0
	if (article_has_attachment (article))
	{
		const gchar * filename = article_get_header (article, PAN_ATTACH_FILE);
		const gint lines_per_part = strtol (article_get_header (article, PAN_LINES_PER_PART), NULL, 10);
		GPtrArray *parts = encode_file (filename, lines_per_part); 
		int i;

		if (!parts)
		{
			g_message ("Error encoding");
		}

		/* Handle part 0 if necessary*/
		if ((tmp = article_get_body (article)))
		{
			postme = g_string_sized_new (1024);
			body = g_string_sized_new (1024);
			append = g_strdup_printf ("(%d/%d)", 0, parts->len-2); 
			build_nntp_header (postme, article, server, append, TRUE);
			g_free (append);
			g_string_assign (body, tmp);
			g_free (tmp);
			pan_g_string_replace (body, "\n.", "\n.."); /* RFC977 3.10.1 ... */
			pan_g_string_replace (body, "\n", "\r\n"); /* ..make sure that each.. */
			pan_g_string_replace (body, "\r\r\n", "\r\n"); /* ..line ends in \r\n */
			g_string_append (postme, body->str);
			g_ptr_array_add (bodies, postme->str);
			g_string_free (postme, FALSE);
			g_string_free (body, TRUE);
		}

		/* Handle part 1 */
		postme = g_string_sized_new (1024);
		append = g_strdup_printf ("(%d/%d)", 1, parts->len-2);
		build_nntp_header (postme, article, server, append, TRUE);
		g_free (append);
		g_string_append (postme, g_ptr_array_index (parts, 0));
		g_string_append (postme, g_ptr_array_index (parts, 1));
		if (parts->len == 3) {
			g_string_append (postme, g_ptr_array_index (parts, 2));
		}
		g_ptr_array_add (bodies, postme->str);
		g_string_free (postme, FALSE);
		
		/* Handle parts 2 to (N-1) if necessary */
		if (parts->len > 3)
		{
			for (i=2; i<parts->len-2; ++i)
			{
				postme = g_string_sized_new (1024);
				append = g_strdup_printf ("(%d/%d)", i, parts->len-2);
				build_nntp_header (postme, article, server, append, TRUE);
				g_free (append);
				g_string_append (postme, g_ptr_array_index (parts, i));
				g_ptr_array_add (bodies, postme->str);
				g_string_free (postme, FALSE);
			}
			/* Handle last part */
			postme = g_string_sized_new (1024);
			append = g_strdup_printf ("(%d/%d)", parts->len-2, parts->len-2);
			build_nntp_header (postme, article, server, append, FALSE);
			g_free (append);
			g_string_append (postme, g_ptr_array_index (parts, parts->len-2));
			g_string_append (postme, g_ptr_array_index (parts, parts->len-1));
			g_ptr_array_add (bodies, postme->str);
			g_string_free (postme, FALSE);
		}
	} 
	else 
	{
#endif
		body = g_string_sized_new (1024);
		postme = g_string_sized_new (1024);

		/**
		***  Build the Article
		**/

		build_nntp_header (postme, article, server, NULL, FALSE);

		/* body */
		tmp = article_get_body (article);
		g_string_assign (body, tmp);
		g_free (tmp);
		pan_g_string_replace (body, "\n.", "\n.."); /* RFC977 3.10.1 ... */
		pan_g_string_replace (body, "\n", "\r\n"); /* ..make sure that each.. */
		pan_g_string_replace (body, "\r\r\n", "\r\n"); /* ..line ends in \r\n */
		g_string_append (postme, body->str);

		/* cleanup */
		tmp = postme->str;
		g_string_free (postme, FALSE);
		g_string_free (body, TRUE);

		g_ptr_array_add (bodies, tmp);
#if 0
	}
#endif
	debug_exit ("build_nntp_message");
	return bodies;
}




/****
*****  A FuncPtr task for sending the message.
*****  A Task is necessary because it lets us go through the proper
*****  channels for allocating a socket.
****/

static gchar*
task_send_describe (const StatusItem * item)
{
	TaskFuncPtr * task = TASK_FUNC_PTR(item);
	SendStruct * data = (SendStruct*) task->user_data;
	Article * article = data->article;
	const gboolean posting = data->do_post;
	const gboolean mailing = data->do_mail;
	const gchar * fmt;
	if (posting && mailing)
		fmt = _("Posting and Mailing Article ``%s''");
	else if (posting)
		fmt = _("Posting Article ``%s''");
	else if (mailing)
		fmt = _("Mailing Article ``%s''");
	else 
		fmt = _("I have no idea what I'm doing with Article ``%s''");
	return g_strdup_printf (fmt, article->subject);
}

static void
task_send_dtor (gpointer data)
{
	TaskFuncPtr * task;
	SendStruct * send;
	debug_enter ("task_send_dtor");

       	task = (TaskFuncPtr*) data;
       	send = (SendStruct*) task->user_data;
	group_unref_articles (send->original_group, NULL);
	g_free (send);

	debug_exit ("task_send_dtor");
}

static int
task_send_run (TaskFuncPtr* task, gpointer user_data)
{
	SendStruct * data;
	Article * article;
	gint status;
	debug_enter ("task_send_run");

       	data = (SendStruct*) user_data;
       	article = data->article;

	/* try to send the message */
	status = send_article_now (STATUS_ITEM(task),
	                           TASK(task)->sock,
	                           TASK(task)->server,
	                           data);

	if (status != 0)
	{
		GString * text = g_string_new (NULL);

		/*FIXME these should give a subject, too */
		/* figure out what worked and what didn't. */
		if (status & SMTP_FAILED)
			g_string_append (text, _("E-Mail send failed.\n"));
		if (status & NNTP_FAILED)
			g_string_append (text, _("Usenet posting failed.\n"));

		/* let the user know that something didn't work. */
		if (text->len != 0) {
			g_string_append (text, _("Your message was saved in the folder ``pan.sendlater''"));
			pan_error_dialog (text->str);
			log_add (LOG_ERROR, text->str);
		}

		g_string_free (text, TRUE);
	}
	else /* successful; dispose of the article object. */
	{
		Group * sent;

		/* save to "Sent".  Note that the call to group_add_article
		   will also remove the article from pan.sendlater */
		sent = folder_get_by_name (PAN_SENT);
		group_add_article (sent, article);
	}

	/* if we failed, we make it hopeless because we've saved
	 * the message in pan.sendlater anyway.  This way the user
	 * can try to correct the problem before resending.
	 */
	debug_exit ("task_send_run");
	return status==0 ? TASK_SUCCESS : TASK_FAIL_HOPELESS;
}

/**
 * At this point we've already gotten a task and socket from the queue
 * and have done all the error-checking with the headers/body.  We're
 * ready to post the article and/or send the mail.
 */
static gint
send_article_now (StatusItem           * item,
                  PanSocket            * sock,
                  Server               * server,
                  SendStruct           * data)
{
	gint retval = 0;
	Article * article;
	const gchar * subject;
	const gchar * mail_to;
	const gchar * post_to;
	debug_enter ("send_article_now");

	/* initialize variables */
       	article = data->article;
       	subject = article->subject;
       	mail_to = data->do_mail ? article_get_header (article, PAN_MAIL_TO) : NULL;
       	post_to = data->do_post ? article_get_header (article, HEADER_NEWSGROUPS) : NULL;

	/* mail the message */
	if (is_nonempty_string(mail_to))
	{
		gchar * leader = NULL;
		int val;

		if (is_nonempty_string(post_to))
		{
			leader = g_strdup_printf (
			_("[This is an email copy of a Usenet post to \"%s\"]"),
			post_to);
		}

		val = smtp_send (item, article, leader);

		/* mail the message */
		if (!val)
		{
			log_add_va (LOG_IMPORTANT, _("E-mail Message \"%s\" sent."), subject);
			data->do_mail = FALSE;
		}
		else
		{
			log_add_va (LOG_ERROR, _("E-mail Message \"%s\" not sent."), subject);
			retval |= SMTP_FAILED;
		}

		/* cleanup */
		g_free (leader);
	}

	/* post the message */
	if (is_nonempty_string(post_to))
	{
		GPtrArray * post_me = build_nntp_message (server, article);
		guint i;

		for (i=0; i!=post_me->len; ++i)
		{
			gchar * a = g_ptr_array_index (post_me, i);

			/* let the user know what we're doing */
			status_item_emit_status_va (STATUS_ITEM(item),
				_("Posting article \"%s\" part %d of %d."),
				article->subject, i+1, post_me->len);

			/* post the part */
			if (nntp_post(item,sock,a) == TASK_SUCCESS) {
				log_add_va (LOG_IMPORTANT, _("NNTP Article \"%s\" (%d of %d) posted."),
					subject, i+1, post_me->len);
				data->do_post = FALSE;
			} else {
				log_add_va (LOG_ERROR, _("NNTP Article \"%s\" (%d of %d) not posted."),
					subject, i+1, post_me->len);
				retval |= NNTP_FAILED;
			}
		}
		
		/* cleanup */
		pan_g_ptr_array_foreach (post_me, (GFunc)g_free, NULL);
		g_ptr_array_free (post_me, TRUE);
	}

	debug_exit ("send_article_now");
	return retval;
}


/****
*****
*****
*****
****/

gint
queue_article_for_posting (Server     * server,
                           Article    * article,
                           gboolean     check_article_first)
{
	GPtrArray * errors;
	GoodnessLevel goodness = OKAY;
	debug_enter ("queue_article_for_posting");

	/* sanity check */
	g_return_val_if_fail (server!=NULL, 1);
	g_return_val_if_fail (article!=NULL, 1);

	/* if desired, make sure the article is okay. */
       	errors = g_ptr_array_new ();
	if (check_article_first)
		check_article_and_prompt (article, server, errors, &goodness);

	/* if the goodness is okay, queue a send task */
	if (goodness == OKAY)
	{
		Task * task;
		SendStruct * data = g_new0 (SendStruct, 1);

		/* ref the articles to ensure that this Article* will still
		   be valid when the task runs.  The accompanying unref is
		   in task_send_dtor... */
		group_ref_articles (article->group, NULL);

		data->article = article;
		data->original_group = article->group;
		data->do_post = article_get_header(article,HEADER_NEWSGROUPS) != NULL;
		data->do_mail = article_get_header(article,PAN_MAIL_TO) != NULL;
		task = TASK(task_func_ptr_new (server,
		                               task_send_describe,
		                               task_send_run,
		                               data,
		                               task_send_dtor));
		task->needs_socket = is_nonempty_string(
			article_get_header(article,HEADER_NEWSGROUPS));
		queue_add (task);
	}

	/* cleanup */
	pan_g_ptr_array_foreach (errors, (GFunc)g_free, NULL);
	g_ptr_array_free (errors, TRUE);
	debug_exit ("queue_article_for_posting");
	return goodness==OKAY ? 0 : 1;
}


void
flush_sendlater_articles (void)
{
	guint i;
        Group * folder;
	GPtrArray * articles;
	debug_enter ("flush_sendlater_articles");

	/* find the sendlater folder */
       	folder = folder_get_by_name (PAN_SENDLATER);
	g_return_if_fail (folder!=NULL);

	/* load the articles for that group */
	group_ref_articles (folder, NULL);
	articles = group_get_article_array (folder);

	for (i=0; i!=articles->len; ++i)
	{
		Article * a;
		Server * s;
	       
		a = ARTICLE(g_ptr_array_index(articles,i));
		s = server_get_by_name (article_get_header (a, PAN_SERVER));
		if (s == NULL) {
			g_message (_("Article \"%s\" not flagged for any particular server; using current"), a->subject);
			s = grouplist_get_server ();
			if (s == NULL)
				continue;
		}

		queue_article_for_posting (s, a, TRUE);
	}

	/* cleanup */
	g_ptr_array_free (articles, TRUE);
	group_unref_articles (folder, NULL);
	debug_exit ("flush_sendlater_articles");
}
