/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include "pan-callback.h"

typedef struct
{
	PanCallbackFunc callback;
	gpointer user_data;
}
CallbackInfo;

PanCallback*
pan_callback_new (void)
{
	PanCallback *pcl = g_new0 (PanCallback, 1);
	pcl->mutex = g_mutex_new ();
	pcl->list = NULL;
	return pcl;
}

void
pan_callback_free (PanCallback* pcl)
{
	g_mutex_free (pcl->mutex);
	g_slist_foreach (pcl->list, (GFunc)g_free, NULL);
	g_slist_free (pcl->list);

	pcl->mutex = NULL;
	pcl->list = NULL;

	g_slist_foreach (pcl->removed, (GFunc)g_free, NULL);
	g_slist_free (pcl->removed);

	g_free (pcl);
}

static gint
cbi_compare (const CallbackInfo* a, const CallbackInfo* b)
{
	return !(a->callback==b->callback && a->user_data == b->user_data);
}
		 
void
pan_callback_call (PanCallback* pcl, gpointer call_object, gpointer call_arg)
{
	GSList * l;
	GSList * tmp;

	/* make a copy of the recipients */
	g_mutex_lock (pcl->mutex);
	tmp = g_slist_copy (pcl->list);
	g_mutex_unlock (pcl->mutex);

	/* invoke the recipients */
	if (tmp != NULL)
	{
		for (l=tmp; l!=NULL; l=l->next)
		{
			CallbackInfo *cbi = (CallbackInfo*) l->data;
			(*cbi->callback)(call_object, call_arg, cbi->user_data);
		}
	}
	g_slist_free (tmp);

	/* free any unlinked nodes */
	g_mutex_lock (pcl->mutex);
	tmp = pcl->removed;
	pcl->removed = NULL;
	g_mutex_unlock (pcl->mutex);


	/* cleanup */ 
	if (tmp != NULL) {
		g_slist_foreach (tmp, (GFunc)g_free, NULL);
		g_slist_free (tmp);
	}
}

void
pan_callback_add (PanCallback* pcl, PanCallbackFunc cb, gpointer user)
{
	CallbackInfo *info = g_new0 (CallbackInfo, 1);
	info->callback = cb;
	info->user_data = user;

	/* add the callback */
	g_mutex_lock (pcl->mutex);
	pcl->list = g_slist_append (pcl->list, info);
	g_mutex_unlock (pcl->mutex);
}

void
pan_callback_remove (PanCallback* pcl, PanCallbackFunc cb, gpointer user)
{
	CallbackInfo cbi;
	GSList *l = NULL;
	cbi.callback = cb;
	cbi.user_data = user;

	g_mutex_lock (pcl->mutex);
	l = g_slist_find_custom (pcl->list, &cbi, (GCompareFunc)cbi_compare);
	if (l==NULL)
		g_warning ("Couldn't find callback to remove");
	else {
		pcl->removed = g_slist_prepend (pcl->removed, l->data);
		pcl->list = g_slist_remove_link (pcl->list, l);
		g_slist_free_1 (l);
	}
	g_mutex_unlock (pcl->mutex);
}
