/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include "acache.h"
#include "article.h"
#include "articlelist.h"
#include "bozo.h"
#include "debug.h"
#include "globals.h"
#include "gnksa.h"
#include "gui.h"
#include "log.h"
#include "pan.h"
#include "prefs.h"
#include "queue.h"
#include "rule-manager.h"
#include "text.h"
#include "thread-watch.h"
#include "util.h"
#include "util-wrap.h"
#include "dialogs/dialogs.h"

#ifdef GTKHTML_HAVE_GCONF
#include <gconf/gconf-client.h>
#endif

static gint save_session (GnomeClient        * client,
			  gint                 phase,
			  GnomeSaveStyle       save_style,
			  gint                 is_shutdown,
			  GnomeInteractStyle   interact_style,
			  gint                 is_fast,
			  gpointer             client_data);

static void session_die  (GnomeClient        * client,
                          gpointer             client_data);

const gchar* PAN_SENT;
const gchar* PAN_SENDLATER;
const gchar* sockread_err_msg = NULL;
const gchar* sockwrite_err_msg = NULL;

PanApp Pan;

static gchar     *geometry = NULL;
static gint       debug_acache = 0;
static gint       debug_socket = 0;
static gint       debug_socket_input = 0;
static gint       debug_socket_output = 0;
static gint       debug_queue = 0;
static gint       debug_gnksa = 0;
static gint       debug_newsrc = 0;
static gint       debug_trace = 0;
static gint       debug_decode = 0;
static gint       debug_gui_locks = 0;
static gint       debug_pan_object = 0;
static gint       debug_defaults = 0;
static gint       mute = 0;
static gint       regression_tests = 0;

struct poptOption options[] = {
	{ "geometry", '\0', POPT_ARG_STRING, &geometry, 0,
	  N_("Specify the geometry of the window."),
	  N_("GEOMETRY") },
	{ "mute", '\0', 0, &mute, 0,
	  N_("Send outgoing messages to stdout instead of smtp/nntp servers for debugging purposes."), NULL },
	{ "debug-acache", '\0', 0, &debug_acache, 0,
	  N_("Turn on article cache debugging messages."), NULL },
	{ "debug-socket-input", '\0', 0, &debug_socket_input, 0,
	  N_("Turn on socket input debugging messages."), NULL },
	{ "debug-socket-output", '\0', 0, &debug_socket_output, 0,
	  N_("Turn on socket output debugging messages."), NULL },
	{ "debug-socket", '\0', 0, &debug_socket, 0,
	  N_("Turn on socket input/output debugging messages."), NULL },
	{ "debug-sockets", '\0', 0, &debug_socket, 0,
	  N_("Same as --debug-socket"), NULL },
	{ "debug-trace", '\0', 0, &debug_trace, 0,
	  N_("Trace function calls"), NULL },
	{ "debug-queue", '\0', 0, &debug_queue, 0,
	  N_("Turn on task scheduler debugging messages."), NULL },
	{ "debug-decode", '\0', 0, &debug_decode, 0,
	  N_("Turn on attachment decoding debugging messages."), NULL },
	{ "debug-newsrc", '\0', 0, &debug_newsrc, 0,
	  N_("Turn on newsrc decoding debugging messages."), NULL },
	{ "debug-gui-locks", '\0', 0, &debug_gui_locks, 0,
	  N_("Turn on gui thread lock debugging messages."), NULL },
	{ "debug-gnksa", '\0', 0, &debug_gnksa, 0,
	  N_("Turn on debugging of GNKSA code."), NULL },
	{ "debug-pan-object", '\0', 0, &debug_pan_object, 0,
	  N_("Turn on pan object system debugging messages."), NULL },
	{ "debug", '\0', 0, &debug_defaults, 0,
	  N_("Turn on most of the generally-useful debug flags.  A good first choice when preparing a bug report."), NULL },
	{ "regression-tests", '\0', 0, &regression_tests, 0,
	  N_("Run regression tests for the Pan programmers."), NULL },

	{ NULL, '\0', 0, NULL, 0 }
};


static int
pan_post_gtkmain_init (gpointer null)
{
	server_init ();
	return 0;
}

/**
 * pan_init:
 *
 * For first time users it is called after the druid finishes.  For
 * second time users it is called right before gtk_main()
 **/
void
pan_init (void)
{
	gtk_idle_add (pan_post_gtkmain_init, NULL);
}

/*---[ main ]---------------------------------------------------------
 * _the_ main function!
 *--------------------------------------------------------------------*/
int
main (int argc, char *argv[])
{
	GnomeClient * client;
	gulong debug_flags = 0;
        poptContext pctx;

	gtk_set_locale ();

	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	sockread_err_msg = _("Error reading from socket.");
	sockwrite_err_msg = _("Error writing to socket.");
	PAN_SENT = _("pan.sent");
	PAN_SENDLATER = _("pan.sendlater");

	g_thread_init (NULL);
	gnome_init_with_popt_table ("Pan", VERSION, argc, argv, options, 0, &pctx);
	poptFreeContext (pctx);

#if 0 /* for rpm */
#if GNOME_LIBS_VERSION>=60
	gnome_window_icon_set_default_from_file (GNOME_ICONDIR "/pan.png");
#endif
#endif
#ifdef GTKHTML_HAVE_GCONF
	gconf_init (argc, argv, NULL);
#endif

	debug_flags = (debug_acache ? DEBUG_ACACHE : 0)
		| (debug_socket||debug_socket_input ? DEBUG_SOCKET_INPUT : 0)
		| (debug_socket||debug_socket_output ? DEBUG_SOCKET_OUTPUT : 0)
		| (debug_queue ? DEBUG_QUEUE : 0)
		| (debug_decode ? DEBUG_DECODE : 0)
		| (debug_newsrc ? DEBUG_NEWSRC : 0)
		| (debug_gnksa ? DEBUG_GNKSA : 0)
		| (debug_trace ? DEBUG_TRACE : 0)
		| (debug_gui_locks ? DEBUG_LOCK : 0)
		| (debug_pan_object ? DEBUG_PAN_OBJECT : 0)
		| (debug_defaults ? (DEBUG_TRACE|DEBUG_DECODE|DEBUG_ACACHE|DEBUG_SOCKET_INPUT|DEBUG_SOCKET_OUTPUT): 0);
	set_debug_level (debug_flags);

	default_incoming_name_real = _("Unknown");
	default_incoming_name_addr = _("unknown@spammesenseless.net");


	log_init ();
	log_add_va (LOG_INFO, _("Pan %s Started"), VERSION);

	if (regression_tests)
	{
		util_regression_tests ();
		gnksa_regression_tests ();
		wrap_regression_tests ();
		regression_tests_summary ();
		return 0;
	}

	prefs_init ();
	acache_init ();

	check_and_warn_if_version_change ();

	/* Connect up to the session manager */
	client = gnome_master_client ();
	gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
			    GTK_SIGNAL_FUNC (save_session), argv[0]);
	gtk_signal_connect (GTK_OBJECT (client), "die",
			    GTK_SIGNAL_FUNC (session_die), NULL);

	/* initialize variables in the global app structure */
	Pan.main_t = pthread_self ();
	pan_mute = mute;
	if (pan_mute)
		g_message ("Running in mute mode: messages will be sent to "
		           "stdout instead of smtp/nntp server");

	queue_init (
		gnome_config_get_int("/Pan/General/Max_Connections_Total=4"),
		gnome_config_get_int("/Pan/General/Max_Task_Tries_Total=4"));

	/* initialize the UI */
	gui_construct (geometry);

	/* Check if a Preferences file exists. */
	if (!gnome_config_has_section ("/Pan/User"))
	{
		dialog_newuser ();
	}
	else
	{
		pan_init ();
		gtk_widget_show_all (Pan.window);
	}

	gtk_main ();

	return 0;
}

/**
 * pan_shutdown:
 *
 * Shutdown Pan, closing all TCP connections and saving all preferences
 **/
void
pan_shutdown (void)
{
	/* FIXME do we save tasks? */
	queue_shutdown (FALSE);

	g_free (message_body_font);
	g_free (articlelist_font);
	g_free (grouplist_font);

	server_shutdown ();
	acache_shutdown ();
	bozo_shutdown_module ();
	thread_shutdown_module ();
	rule_manager_shutdown_module ();
	log_shutdown_module ();

	pan_object_bookkeeping ();

	gtk_main_quit ();
	pan_clean_temp_dir ();
}


/***
****   Session management
***/

/*
** save_session
** saves the current session for later revival
*/

static gint
save_session (GnomeClient         * client,
              gint                  phase,
              GnomeSaveStyle        save_style,
              gint                  is_shutdown,
              GnomeInteractStyle    interact_style,
              gint                  is_fast,
              gpointer              client_data)
{
	gchar **argv;
	guint argc;

	argv = g_new0 (gchar*, 4);
	argc = 1;

	argv[0] = client_data;
	/* add any addtional state info here. */
	gnome_client_set_clone_command (client, argc, argv);
	gnome_client_set_restart_command (client, argc, argv);
	return TRUE;
}

/**
*** session_die: Gracefullly end the session
**/

static void
session_die (GnomeClient *    client,
             gpointer         client_data)
{
	pan_shutdown ();
}
