/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-sound.h>
#include <libgnomeui/gnome-messagebox.h>
#include <libgnomeui/gnome-stock.h>

#include <ctype.h>
#include <stdlib.h>

#include <pan/bozo.h>
#include <pan/debug.h>
#include <pan/decode.h>
#include <pan/globals.h>
#include <pan/group.h>
#include <pan/gui.h>
#include <pan/log.h>
#include <pan/queue.h>
#include <pan/rule-action.h>
#include <pan/task-bodies.h>
#include <pan/task-decode.h>
#include <pan/thread-watch.h>
#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

/*****
******
*****/

/*****
******
*****/

/*****
******
*****/

/************
*************  PROTECTED
************/

void
rule_action_constructor (RuleAction * r)
{
	g_return_if_fail (r!=NULL);

	pan_object_constructor (PAN_OBJECT(r), rule_action_destructor);
	debug1 (DEBUG_PAN_OBJECT, "rule_action_ctor: %p", r);

	r->flags = 0;
	r->sound_file = NULL;
	r->alert_message = NULL;
	r->append_file = NULL;
	r->forward_to = NULL;
	r->decode_path = NULL;
	r->is_read = FALSE;
	r->is_important = FALSE;
	r->is_protected = FALSE;
}

void
rule_action_destructor (PanObject * obj)
{
	RuleAction * r = RULE_ACTION(obj);

	debug1 (DEBUG_PAN_OBJECT, "rule_action_dtor: %p", r);
	g_free (r->sound_file);
	g_free (r->alert_message);
	g_free (r->append_file);
	g_free (r->forward_to);
	g_free (r->decode_path);

	/* free the superclass */
	pan_object_destructor (obj);
}

/************
*************  PUBLIC
************/

RuleAction*
rule_action_new (void)
{
	RuleAction * r = g_new0 (RuleAction, 1);
	rule_action_constructor (r);
	return r;
}

/*****
******
*****/

void
rule_action_clear (RuleAction * a)
{
	g_return_if_fail (a != NULL);

	a->flags = 0;
	replace_gstr (&a->sound_file, NULL);
	replace_gstr (&a->alert_message, NULL);
	replace_gstr (&a->append_file, NULL);
	replace_gstr (&a->forward_to, NULL);
	replace_gstr (&a->decode_path, NULL);
	a->is_read = FALSE;
	a->is_important = FALSE;
	a->is_protected = FALSE;
}

/*****
******
*****/

RuleAction*
rule_action_dup (const RuleAction * src)
{
	RuleAction * tgt;

	g_return_val_if_fail (src!=NULL, NULL);

	tgt = rule_action_new ();

	/* duplicate this object's bits */
	tgt->flags          = src->flags;
	tgt->sound_file     = g_strdup (src->sound_file);
	tgt->alert_message  = g_strdup (src->alert_message);
	tgt->append_file    = g_strdup (src->append_file);
	tgt->forward_to     = g_strdup (src->forward_to);
	tgt->decode_path    = g_strdup (src->decode_path);
	tgt->is_read        = src->is_read;
	tgt->is_important   = src->is_important;
	tgt->is_protected   = src->is_protected;

	return tgt;
}

/*****
******
*****/

void
rule_action_apply (const RuleAction * action,
                   GPtrArray * articles,
		   gboolean * deleted,
		   const gchar * rule_name)
{
	guint i;

	/* sanity checks */
	g_return_if_fail (action!=NULL);
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (deleted!=NULL);
	for (i=0; i!=articles->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(articles,i));
		g_return_if_fail (a != NULL);
		g_return_if_fail (a->group != NULL);
	}

	/* if there are no articles, there's nothing to do. */
	if (articles->len < 1)
		return;

	*deleted = FALSE;

	if (action->flags & RULE_ACTION_DISCARD)
	{
		Group * g = ARTICLE(g_ptr_array_index(articles,0))->group;
		group_remove_articles (g, articles);
		*deleted = TRUE;
		return;
	}
	if (action->flags & RULE_ACTION_PLAY_SOUND)
	{
		pan_warn_if_fail (is_nonempty_string(action->sound_file));
		if (is_nonempty_string (action->sound_file))
			gnome_sound_play (action->sound_file);
	}
	if (action->flags & RULE_ACTION_SHOW_ALERT)
	{
		const Group * g = ARTICLE(g_ptr_array_index(articles,0))->group;
		gchar * msg = is_nonempty_string(action->alert_message)
			? action->alert_message
			: _("HEY you forgot to put a message here! :)");
		gchar * s = g_strdup_printf (
			_("%s\n\nMatches: %u\nGroup: %s\nRule: %s"),
			msg,
			articles->len,
			group_get_readable_name(g),
			rule_name);
#if 1
		GtkWidget * w;

		pan_lock ();
		w = gnome_message_box_new (s,
			GNOME_MESSAGE_BOX_INFO,
			GNOME_STOCK_BUTTON_OK, NULL);
		gnome_dialog_set_parent (GNOME_DIALOG(w),
		                         GTK_WINDOW(Pan.window));
		gtk_widget_show_all (w);
		pan_unlock ();
#else
		g_message ("FIXME Charles, Remember to add the dialog back");
#endif
		log_add (LOG_INFO, s);
		g_free (s);
	}
	if (action->flags & RULE_ACTION_MARK_READ)
	{
		articles_set_read ((Article**)articles->pdata, articles->len, TRUE);
	}
	if (action->flags & RULE_ACTION_MARK_UNREAD)
	{
		articles_set_read ((Article**)articles->pdata, articles->len, FALSE);
	}
	if (action->flags & RULE_ACTION_DECODE)
	{
		guint i;
		for (i=0; i!=articles->len; ++i)
		{
			Article * a = ARTICLE(g_ptr_array_index(articles,i));
			GSList * slist;
			PanObject * task;

			/* make a list of this article and its children */
			slist = g_slist_copy (a->threads);
			slist = g_slist_prepend (slist, a);

			/* create a decode task */
			task = task_decode_new (a->group,
			                        a->subject,
			                        action->decode_path,
			                        NULL,
			                        slist,
			                        FALSE);

			queue_add (TASK(task));
		}
	}
	if (action->flags & RULE_ACTION_TAG_FOR_RETRIEVAL)
	{
		articles_add_flag ((Article**)articles->pdata, articles->len, STATE_DOWNLOAD_FLAGGED);
	}
	if (action->flags & RULE_ACTION_RETREIVE_BODY)
	{
		Task * task;
		GPtrArray * tmp;
		Article * article;

		tmp = g_ptr_array_new ();
		article = ARTICLE(g_ptr_array_index(articles,0));
		pan_g_ptr_array_assign (tmp, articles->pdata, articles->len);
		task = TASK(task_bodies_new (article->group, tmp));
		if (task != NULL)
			queue_add (task);
		g_ptr_array_free (tmp, TRUE);
	}
	if (action->flags & RULE_ACTION_BOZOIZE)
	{
		guint i;
		for (i=0; i!=articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(articles,i));
			g_assert (a != NULL);
			bozo_add (a->author_addr, a->author_real);
		}
	}
	if (action->flags & RULE_ACTION_WATCH)
	{
		guint i;
		for (i=0; i!=articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(articles,i));
			thread_set_state (a, THREAD_WATCHED);
		}
	}
	if (action->flags & RULE_ACTION_IGNORE)
	{
		guint i;
		for (i=0; i!=articles->len; ++i) {
			Article * a = ARTICLE(g_ptr_array_index(articles,i));
			thread_set_state (a, THREAD_IGNORED);
		}
	}
}
