/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>

#include "article.h"
#include "debug.h"
#include "log.h"
#include "nntp.h"
#include "queue.h"
#include "task-bodies.h"
#include "util.h"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static int task_bodies_run (Task* item);

static gchar* task_bodies_describe (const StatusItem* item);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
task_bodies_destructor (PanObject *o)
{
	TaskBodies * task;
	debug_enter ("task_bodies_destructor");

	/* task-bodies dtor: unref articles + group */
 	task = TASK_BODIES(o);
	pan_g_ptr_array_foreach (task->articles, (GFunc)pan_object_unref, NULL);
	group_unref_articles (task->group, STATUS_ITEM(task));

	/* task-bodies dtor: cleanup fields */
	g_ptr_array_free (task->articles, TRUE);
	task->articles = NULL;
	task->working_index = 0;

	/* destroy parent class */
	task_destructor (o);

	debug_exit ("task_bodies_destructor");
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_bodies_new (Group             * group,
                 const GPtrArray   * articles)
{
	TaskBodies* bodies = NULL;
	debug_enter ("task_bodies_new");

	/* sanity clause */
	g_return_val_if_fail (group!=NULL, NULL);
	g_return_val_if_fail (articles!=NULL, NULL);
	g_return_val_if_fail (articles->len!=0, NULL);

	/* create the object */
	group_ref_articles (group, NULL);
	pan_g_ptr_array_foreach ((GPtrArray*)articles, (GFunc)pan_object_ref, NULL);
       	bodies = g_new0 (TaskBodies, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_bodies_new: %p", bodies);

	/* initialize the parent class */
	task_constructor (TASK(bodies), task_bodies_destructor,
	                                task_bodies_describe,
	                                task_bodies_run,
	                                group->server, TRUE, TRUE);
	TASK(bodies)->gets_bodies = TRUE;

	/* initialize the task-bodies */
	bodies->group = group;
	bodies->articles = pan_g_ptr_array_dup ((GPtrArray*)articles);
	bodies->working_index = 0;

	debug_exit ("task_bodies_new");
	return PAN_OBJECT(bodies);
}

/*****
******
*****/

static gchar*
task_bodies_describe (const StatusItem* si)
{
	TaskBodies *item = TASK_BODIES(si);

	return g_strdup_printf (_("Downloading %d articles from group %s"),
		item->articles->len,
		group_get_readable_name(item->group));
}

/*****
******
*****/

static int
task_bodies_run (Task* task)
{
	gint retval;
	gint steps = (int)TASK_BODIES(task)->articles->len;
	debug_enter ("task_bodies_run");

	status_item_emit_init_steps (STATUS_ITEM(task), steps);

	retval = nntp_download_bodies (STATUS_ITEM(task),
	                               TASK_BODIES(task)->group,
                                       task->sock,
                                       &task->hint_abort,
                                       TASK_BODIES(task)->articles,
                                       &TASK_BODIES(task)->working_index,
                                       FALSE, FALSE);

	if (retval == TASK_SUCCESS) {
		log_add_va (LOG_INFO, ("Fetched %u bodies from `%s'"),
			steps,
			group_get_readable_name(TASK_BODIES(task)->group));
	}

	debug_exit ("task_bodies_run");
	return retval;
}
