/* Pango2
 * testhbfont.c: Test program for Pango2HbFont etc
 *
 * Copyright (C) 2021 Matthias Clasen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <glib.h>
#include <gio/gio.h>
#include <pango2/pango.h>

#include <hb-ot.h>


/* Verify that the variable and monospace properties work as expected
 * for Pango2HbFamily
 */
static void
test_hbfont_monospace (void)
{
  Pango2FontMap *map;
  Pango2FontFamily *family;
  char *path;

  map = pango2_font_map_new ();

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);
  pango2_font_map_add_file (map, path);

  family = pango2_font_map_get_family (PANGO2_FONT_MAP (map), "Cantarell");

  g_assert_nonnull (family);

  pango2_font_map_add_face (map, PANGO2_FONT_FACE (pango2_hb_face_new_from_file (path, 0, -2, NULL, NULL)));

  g_free (path);

  path = g_test_build_filename (G_TEST_DIST, "fonts", "DejaVuSansMono.ttf", NULL);
  pango2_font_map_add_file (map, path);
  g_free (path);

  family = pango2_font_map_get_family (PANGO2_FONT_MAP (map), "DejaVu Sans Mono");

  g_assert_nonnull (family);

  g_object_unref (map);
}

/* Verify that a description -> face -> description roundtrip works for
 * Pango2HbFaces created with pango2_hb_face_new_synthetic or pango2_hb_face_new_instance
 */
static void
test_hbface_roundtrip (void)
{
  char *path;
  Pango2HbFace *face;
  Pango2HbFace *face2;
  Pango2FontDescription *desc;
  const int NO_FACEID = ~PANGO2_FONT_MASK_FACEID;
  hb_variation_t v;
  Pango2HbFaceBuilder *builder;

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);

  face = pango2_hb_face_new_from_file (path, 0, -1, NULL, NULL);
  g_assert_true (PANGO2_IS_HB_FACE (face));
  g_assert_cmpstr (pango2_font_face_get_name (PANGO2_FONT_FACE (face)), ==, "Regular");
  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face));
  g_assert_cmpint (pango2_font_description_get_set_fields (desc) & NO_FACEID, ==, PANGO2_FONT_MASK_FAMILY |
                                                                                 PANGO2_FONT_MASK_STYLE |
                                                                                 PANGO2_FONT_MASK_VARIANT |
                                                                                 PANGO2_FONT_MASK_WEIGHT |
                                                                                 PANGO2_FONT_MASK_STRETCH);
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarell");
  g_assert_cmpint (pango2_font_description_get_style (desc), ==, PANGO2_STYLE_NORMAL);
  g_assert_cmpint (pango2_font_description_get_weight (desc), ==, PANGO2_WEIGHT_NORMAL);
  g_assert_cmpint (pango2_font_description_get_stretch (desc), ==, PANGO2_STRETCH_NORMAL);
  pango2_font_description_free (desc);

  desc = pango2_font_description_new ();
  pango2_font_description_set_style (desc, PANGO2_STYLE_OBLIQUE);
  builder = pango2_hb_face_builder_new (face);
  pango2_hb_face_builder_set_transform (builder, &(Pango2Matrix){ 1, 0.2, 0, 1, 0, 0 });
  pango2_hb_face_builder_set_description (builder, desc);
  pango2_hb_face_builder_set_name (builder, "Oblique");
  face2 = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  pango2_font_description_free (desc);

  g_assert_true (PANGO2_IS_HB_FACE (face2));
  g_assert_cmpstr (pango2_font_face_get_name (PANGO2_FONT_FACE (face2)), ==, "Oblique");
  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face2));
  g_assert_cmpint (pango2_font_description_get_set_fields (desc) & NO_FACEID, ==, PANGO2_FONT_MASK_FAMILY |
                                                                                 PANGO2_FONT_MASK_STYLE |
                                                                                 PANGO2_FONT_MASK_VARIANT |
                                                                                 PANGO2_FONT_MASK_WEIGHT |
                                                                                 PANGO2_FONT_MASK_STRETCH);
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarell");
  g_assert_cmpint (pango2_font_description_get_style (desc), ==, PANGO2_STYLE_OBLIQUE);
  g_assert_cmpint (pango2_font_description_get_weight (desc), ==, PANGO2_WEIGHT_NORMAL);
  g_assert_cmpint (pango2_font_description_get_stretch (desc), ==, PANGO2_STRETCH_NORMAL);
  pango2_font_description_free (desc);
  g_object_unref (face2);

  desc = pango2_font_description_new ();
  pango2_font_description_set_weight (desc, PANGO2_WEIGHT_BOLD);
  builder = pango2_hb_face_builder_new (face);
  pango2_hb_face_builder_set_embolden (builder, TRUE);
  pango2_hb_face_builder_set_description (builder, desc);
  pango2_hb_face_builder_set_name (builder, "Bold");
  face2 = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  pango2_font_description_free (desc);

  g_assert_true (PANGO2_IS_HB_FACE (face2));
  g_assert_cmpstr (pango2_font_face_get_name (PANGO2_FONT_FACE (face2)), ==, "Bold");
  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face2));
  g_assert_cmpint (pango2_font_description_get_set_fields (desc) & NO_FACEID, ==, PANGO2_FONT_MASK_FAMILY |
                                                                                 PANGO2_FONT_MASK_STYLE |
                                                                                 PANGO2_FONT_MASK_VARIANT |
                                                                                 PANGO2_FONT_MASK_WEIGHT |
                                                                                 PANGO2_FONT_MASK_STRETCH);
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarell");
  g_assert_cmpint (pango2_font_description_get_style (desc), ==, PANGO2_STYLE_NORMAL);
  g_assert_cmpint (pango2_font_description_get_weight (desc), ==, PANGO2_WEIGHT_BOLD);
  g_assert_cmpint (pango2_font_description_get_stretch (desc), ==, PANGO2_STRETCH_NORMAL);
  pango2_font_description_free (desc);
  g_object_unref (face2);

  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cantarellagain");
  builder = pango2_hb_face_builder_new (face);
  pango2_hb_face_builder_set_description (builder, desc);
  face2 = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  pango2_font_description_free (desc);

  g_assert_true (PANGO2_IS_HB_FACE (face2));
  g_assert_cmpstr (pango2_font_face_get_name (PANGO2_FONT_FACE (face2)), ==, "Regular");
  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face2));
  g_assert_cmpint (pango2_font_description_get_set_fields (desc) & NO_FACEID, ==, PANGO2_FONT_MASK_FAMILY |
                                                                                 PANGO2_FONT_MASK_STYLE |
                                                                                 PANGO2_FONT_MASK_VARIANT |
                                                                                 PANGO2_FONT_MASK_WEIGHT |
                                                                                 PANGO2_FONT_MASK_STRETCH);
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarellagain");
  g_assert_cmpint (pango2_font_description_get_style (desc), ==, PANGO2_STYLE_NORMAL);
  g_assert_cmpint (pango2_font_description_get_weight (desc), ==, PANGO2_WEIGHT_NORMAL);
  g_assert_cmpint (pango2_font_description_get_stretch (desc), ==, PANGO2_STRETCH_NORMAL);
  pango2_font_description_free (desc);
  g_object_unref (face2);

  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cat");
  pango2_font_description_set_weight (desc, PANGO2_WEIGHT_ULTRABOLD);

  v.tag = HB_OT_TAG_VAR_AXIS_WEIGHT;
  v.value = 768.;

  builder = pango2_hb_face_builder_new (face);
  pango2_hb_face_builder_set_variations (builder, &v, 1);
  pango2_hb_face_builder_set_name (builder, "Fat");
  pango2_hb_face_builder_set_description (builder, desc);
  face2 = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  pango2_font_description_free (desc);

  g_assert_true (PANGO2_IS_HB_FACE (face2));
  g_assert_cmpstr (pango2_font_face_get_name (PANGO2_FONT_FACE (face2)), ==, "Fat");
  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face2));
  g_assert_cmpint (pango2_font_description_get_set_fields (desc) & NO_FACEID, ==, PANGO2_FONT_MASK_FAMILY |
                                                                                 PANGO2_FONT_MASK_STYLE |
                                                                                 PANGO2_FONT_MASK_WEIGHT |
                                                                                 PANGO2_FONT_MASK_VARIANT |
                                                                                 PANGO2_FONT_MASK_VARIATIONS |
                                                                                 PANGO2_FONT_MASK_STRETCH);
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cat");
  g_assert_cmpint (pango2_font_description_get_style (desc), ==, PANGO2_STYLE_NORMAL);
  g_assert_cmpint (pango2_font_description_get_weight (desc), ==, PANGO2_WEIGHT_ULTRABOLD);
  g_assert_cmpint (pango2_font_description_get_stretch (desc), ==, PANGO2_STRETCH_NORMAL);
  g_assert_cmpstr (pango2_font_description_get_variations (desc), ==, "wght=768");
  pango2_font_description_free (desc);
  g_object_unref (face2);

  g_object_unref (face);
  g_free (path);
}

/* Verify that face -> font -> description works as expected for Pango2HbFont */
static void
test_hbfont_roundtrip (void)
{
  char *path;
  Pango2HbFace *face;
  Pango2HbFont *font;
  Pango2FontDescription *desc;
  unsigned int n_features;

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);

  face = pango2_hb_face_new_from_file (path, 0, -1, NULL, NULL);
  g_assert_true (PANGO2_IS_HB_FACE (face));

  font = pango2_hb_font_new (face, 11 * PANGO2_SCALE, NULL, 0, NULL, 0, PANGO2_GRAVITY_AUTO, 96., NULL);
  g_assert_true (PANGO2_IS_HB_FONT (font));
  g_assert_true (pango2_font_get_face (PANGO2_FONT (font)) == PANGO2_FONT_FACE (face));
  pango2_hb_font_get_features (PANGO2_HB_FONT (font), &n_features);
  g_assert_cmpint (n_features, ==, 0);

  desc = pango2_font_describe (PANGO2_FONT (font));
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarell");
  g_assert_cmpint (pango2_font_description_get_style (desc), ==, PANGO2_STYLE_NORMAL);
  g_assert_cmpint (pango2_font_description_get_weight (desc), ==, PANGO2_WEIGHT_NORMAL);
  g_assert_cmpint (pango2_font_description_get_stretch (desc), ==, PANGO2_STRETCH_NORMAL);
  g_assert_cmpint (pango2_font_description_get_size (desc), ==, 11 * PANGO2_SCALE);
  pango2_font_description_free (desc);

  g_object_unref (font);
  g_object_unref (face);
  g_free (path);
}

/* Verify that pango2_font_describe and pango2_font_describe_with_absolute_size
 * work as expected with Pango2HbFont
 */
static void
test_hbfont_describe (void)
{
  char *path;
  Pango2HbFace *face;
  Pango2HbFont *font;
  Pango2FontDescription *desc;

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);

  face = pango2_hb_face_new_from_file (path, 0, -1, NULL, NULL);
  g_assert_true (PANGO2_IS_HB_FACE (face));

  font = pango2_hb_font_new (face, 11 * PANGO2_SCALE, NULL, 0, NULL, 0, PANGO2_GRAVITY_AUTO, 96., NULL);
  g_assert_true (PANGO2_IS_HB_FONT (font));

  desc = pango2_font_describe (PANGO2_FONT (font));
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarell");
  g_assert_cmpint (pango2_font_description_get_size (desc), ==, 11 * PANGO2_SCALE);
  g_assert_true (!pango2_font_description_get_size_is_absolute (desc));
  pango2_font_description_free (desc);

  desc = pango2_font_describe_with_absolute_size (PANGO2_FONT (font));
  g_assert_cmpstr (pango2_font_description_get_family (desc), ==, "Cantarell");
  g_assert_cmpint (pango2_font_description_get_size (desc), ==, 11 * PANGO2_SCALE * 96./72.);
  g_assert_true (pango2_font_description_get_size_is_absolute (desc));
  pango2_font_description_free (desc);

  g_object_unref (font);
  g_object_unref (face);
  g_free (path);
}

/* Test that describing fonts and faces works with variations */
static void
test_hbfont_describe_variation (void)
{
  char *path;
  Pango2HbFace *face, *face2;
  Pango2HbFont *font;
  Pango2FontDescription *desc;
  hb_variation_t v;
  Pango2HbFaceBuilder *builder;

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);

  face = pango2_hb_face_new_from_file (path, 0, -1, NULL, NULL);
  g_assert_true (PANGO2_IS_HB_FACE (face));

  font = pango2_hb_font_new (face, 11 * PANGO2_SCALE, NULL, 0, NULL, 0, PANGO2_GRAVITY_AUTO, 96., NULL);
  g_assert_true (PANGO2_IS_HB_FONT (font));

  desc = pango2_font_describe (PANGO2_FONT (font));
  g_assert_true ((pango2_font_description_get_set_fields (desc) & PANGO2_FONT_MASK_VARIATIONS) == 0);
  pango2_font_description_free (desc);
  g_object_unref (font);

  v.tag = HB_OT_TAG_VAR_AXIS_WEIGHT;
  v.value = 768.;
  font = pango2_hb_font_new (face, 11 * PANGO2_SCALE, NULL, 0, &v, 1, PANGO2_GRAVITY_AUTO, 96., NULL);

  desc = pango2_font_describe (PANGO2_FONT (font));
  g_assert_true ((pango2_font_description_get_set_fields (desc) & PANGO2_FONT_MASK_VARIATIONS) != 0);
  g_assert_cmpstr (pango2_font_description_get_variations (desc), ==, "wght=768");
  pango2_font_description_free (desc);
  g_object_unref (font);

  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cantarellagain");

  v.tag = HB_OT_TAG_VAR_AXIS_WEIGHT;
  v.value = 512.;
  builder = pango2_hb_face_builder_new (face);
  pango2_hb_face_builder_set_variations (builder, &v, 1);
  pango2_hb_face_builder_set_name (builder, "Medium");
  pango2_hb_face_builder_set_description (builder, desc);
  face2 = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  g_assert_true (PANGO2_IS_HB_FACE (face));
  pango2_font_description_free (desc);

  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face2));
  g_assert_true ((pango2_font_description_get_set_fields (desc) & PANGO2_FONT_MASK_VARIATIONS) != 0);
  g_assert_cmpstr (pango2_font_description_get_variations (desc), ==, "wght=512");
  pango2_font_description_free (desc);

  font = pango2_hb_font_new (face2, 11 * PANGO2_SCALE, NULL, 0, NULL, 0, PANGO2_GRAVITY_AUTO, 96., NULL);
  g_assert_true (PANGO2_IS_HB_FONT (font));

  desc = pango2_font_describe (PANGO2_FONT (font));
  g_assert_true ((pango2_font_description_get_set_fields (desc) & PANGO2_FONT_MASK_VARIATIONS) != 0);
  g_assert_cmpstr (pango2_font_description_get_variations (desc), ==, "wght=512");
  pango2_font_description_free (desc);
  g_object_unref (font);

  v.tag = HB_OT_TAG_VAR_AXIS_WEIGHT;
  v.value = 768.;
  font = pango2_hb_font_new (face2, 11 * PANGO2_SCALE, NULL, 0, &v, 1, PANGO2_GRAVITY_AUTO, 96., NULL);

  desc = pango2_font_describe (PANGO2_FONT (font));
  g_assert_true ((pango2_font_description_get_set_fields (desc) & PANGO2_FONT_MASK_VARIATIONS) != 0);
  g_assert_cmpstr (pango2_font_description_get_variations (desc), ==, "wght=768");
  pango2_font_description_free (desc);
  g_object_unref (font);

  g_object_unref (face2);
  g_object_unref (face);
  g_free (path);
}

/* Test that we get different faceids for the different named instances
 * or variants of Cantarell.
 */
static void
test_hbfont_faceid (void)
{
  char *path;
  Pango2HbFace *face, *face2, *face3;
  Pango2FontDescription *desc, *desc2, *desc3;
  Pango2HbFaceBuilder *builder;

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);

  face = pango2_hb_face_new_from_file (path, 0, -1, NULL, NULL);
  face2 = pango2_hb_face_new_from_file (path, 0, 2, NULL, NULL);
  desc = pango2_font_description_new ();
  pango2_font_description_set_weight (desc, PANGO2_WEIGHT_BOLD);
  builder = pango2_hb_face_builder_new (face);
  pango2_hb_face_builder_set_embolden (builder, TRUE);
  pango2_hb_face_builder_set_description (builder, desc);
  face3 = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  pango2_font_description_free (desc);

  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face));
  g_assert_true (pango2_font_description_get_set_fields (desc) & PANGO2_FONT_MASK_FACEID);

  desc2 = pango2_font_face_describe (PANGO2_FONT_FACE (face2));
  g_assert_true (pango2_font_description_get_set_fields (desc2) & PANGO2_FONT_MASK_FACEID);

  desc3 = pango2_font_face_describe (PANGO2_FONT_FACE (face3));
  g_assert_true (pango2_font_description_get_set_fields (desc3) & PANGO2_FONT_MASK_FACEID);

  g_assert_cmpstr (pango2_font_description_get_faceid (desc), !=, pango2_font_description_get_faceid (desc2));
  g_assert_cmpstr (pango2_font_description_get_faceid (desc), !=, pango2_font_description_get_faceid (desc3));
  g_assert_cmpstr (pango2_font_description_get_faceid (desc2), !=, pango2_font_description_get_faceid (desc3));

  pango2_font_description_free (desc);
  pango2_font_description_free (desc2);
  pango2_font_description_free (desc3);

  g_object_unref (face);
  g_object_unref (face2);
  g_object_unref (face3);

  g_free (path);
}

/* Test font -> description -> font roundtrips with a difficult family */
static void
test_hbfont_load (void)
{
  Pango2FontMap *map;
  Pango2Context *context;
  char *path;
  Pango2FontDescription *desc;
  Pango2HbFace *face_fat, *face_wild;
  char *s;
  Pango2Font *font;

  /* Make a Cat family, with the two faces Fat and Wild */
  map = pango2_font_map_new ();
  context = pango2_context_new_with_font_map (PANGO2_FONT_MAP (map));

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);
  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cat");
  face_fat = pango2_hb_face_new_from_file (path, 0, -1, "Fat", desc);
  pango2_font_description_free (desc);
  g_free (path);

  pango2_font_map_add_face (map, PANGO2_FONT_FACE (face_fat));

  path = g_test_build_filename (G_TEST_DIST, "fonts", "DejaVuSans.ttf", NULL);
  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cat");
  face_wild = pango2_hb_face_new_from_file (path, 0, -1, "Wild", desc);
  pango2_font_description_free (desc);

  pango2_font_map_add_face (map, PANGO2_FONT_FACE (face_wild));

  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face_wild));
  pango2_font_description_set_size (desc, 12 * PANGO2_SCALE);

  s = pango2_font_description_to_string (desc);
  g_assert_cmpstr (s, ==, "Cat 12 @faceid=hb:DejaVuSans:0:-1:0:1:1:0");
  g_free (s);

  /* loading with faceid set works as expected */
  font = pango2_font_map_load_font (PANGO2_FONT_MAP (map), context, desc);
  g_assert_true (pango2_font_get_face (font) == PANGO2_FONT_FACE (face_wild));
  g_object_unref (font);

  pango2_font_description_unset_fields (desc, PANGO2_FONT_MASK_FACEID);

  /* ...and without doesn't */
  s = pango2_font_description_to_string (desc);
  g_assert_cmpstr (s, ==, "Cat 12");
  g_free (s);

  font = pango2_font_map_load_font (PANGO2_FONT_MAP (map), context, desc);
  g_assert_true (pango2_font_get_face (font) == PANGO2_FONT_FACE (face_fat));
  g_object_unref (font);

  pango2_font_description_free (desc);

  g_object_unref (context);
  g_object_unref (map);
}

/* Test font -> description -> font roundtrips with variations */
static void
test_hbfont_load_variation (void)
{
  Pango2FontMap *map;
  Pango2Context *context;
  char *path;
  Pango2FontDescription *desc;
  Pango2HbFace *face_fat, *face_wild;
  char *s;
  Pango2Font *font;
  hb_variation_t v;
  hb_font_t *hb_font;
  const float *coords;
  unsigned int length;
  Pango2HbFaceBuilder *builder;

  /* Make a Cat family, with the two faces Fat and Wild */
  map = pango2_font_map_new ();
  context = pango2_context_new_with_font_map (PANGO2_FONT_MAP (map));

  path = g_test_build_filename (G_TEST_DIST, "fonts", "Cantarell-VF.otf", NULL);
  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cat");
  face_fat = pango2_hb_face_new_from_file (path, 0, -1, "Fat", desc);
  pango2_font_description_free (desc);
  g_free (path);

  pango2_font_map_add_face (map, PANGO2_FONT_FACE (face_fat));

  desc = pango2_font_description_new ();
  pango2_font_description_set_family (desc, "Cat");
  v.tag = HB_OT_TAG_VAR_AXIS_WEIGHT;
  v.value = 624.;
  builder = pango2_hb_face_builder_new (face_fat);
  pango2_hb_face_builder_set_variations (builder, &v, 1);
  pango2_hb_face_builder_set_name (builder,  "Wild");
  pango2_hb_face_builder_set_description (builder,  desc);
  face_wild = pango2_hb_face_builder_get_face (builder);
  pango2_hb_face_builder_free (builder);
  pango2_font_description_free (desc);

  pango2_font_map_add_face (map, PANGO2_FONT_FACE (face_wild));

  desc = pango2_font_face_describe (PANGO2_FONT_FACE (face_wild));

  g_assert_cmpstr (pango2_font_description_get_variations (desc), ==, "wght=624");

  pango2_font_description_set_size (desc, 12 * PANGO2_SCALE);

  s = pango2_font_description_to_string (desc);
  g_assert_cmpstr (s, ==, "Cat 12 @faceid=hb:Cantarell-Regular:0:-1:0:1:1:0:wght_624 @wght=624");
  g_free (s);

  font = pango2_font_map_load_font (PANGO2_FONT_MAP (map), context, desc);
  g_assert_true (pango2_font_get_face (font) == PANGO2_FONT_FACE (face_wild));

  hb_font = pango2_font_get_hb_font (font);
  coords = hb_font_get_var_coords_design (hb_font, &length);
  g_assert_cmpint (length, ==, 1);
  g_assert_cmphex (coords[0], ==, 624.);

  g_object_unref (font);

  pango2_font_description_free (desc);

  g_object_unref (context);
  g_object_unref (map);
}

/* Verify that pango2_fontmap_load_fontset produces a non-empty result
 * even if the language isn't covered - our itemization code relies
 * on this.
 */
static gboolean
get_font (Pango2Fontset *fontset,
          Pango2Font    *font,
          gpointer      data)
{
  gboolean *found = data;

  *found = TRUE;

  return TRUE;
}

static void
test_fontmap_language (void)
{
  Pango2FontMap *map;
  Pango2Context *context;
  Pango2FontDescription *desc;
  Pango2Fontset *fonts;
  gboolean found;

  map = PANGO2_FONT_MAP (pango2_font_map_new_default ());
  context = pango2_context_new_with_font_map (map);
  desc = pango2_font_description_from_string ("serif 11");

  /* zz isn't assigned, so there should not be any fonts claiming to support
   * this language. We are expecting to get a nonempty fontset regardless.
   */
  fonts = pango2_font_map_load_fontset (map, context, desc, pango2_language_from_string ("zz"));
  g_assert_true (PANGO2_IS_FONTSET (fonts));

  found = FALSE;
  pango2_fontset_foreach (fonts, get_font, &found);
  g_assert_true (found);

  g_object_unref (fonts);
  pango2_font_description_free (desc);
}

int
main (int argc, char *argv[])
{
  g_test_init (&argc, &argv, NULL);

  g_test_add_func ("/hbfont/monospace", test_hbfont_monospace);
  g_test_add_func ("/hbface/roundtrip", test_hbface_roundtrip);
  g_test_add_func ("/hbfont/roundtrip", test_hbfont_roundtrip);
  g_test_add_func ("/hbfont/describe", test_hbfont_describe);
  g_test_add_func ("/hbfont/describe/variation", test_hbfont_describe_variation);
  g_test_add_func ("/hbfont/faceid", test_hbfont_faceid);
  g_test_add_func ("/hbfont/load", test_hbfont_load);
  g_test_add_func ("/hbfont/load/variation", test_hbfont_load_variation);
  g_test_add_func ("/fontmap/language", test_fontmap_language);

  return g_test_run ();
}
