// -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*-

/*
 *  PaperBox - main-window.cc
 *
 *  Copyright (C) 2007 Marko Anastasov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>
#include <iostream>
#include <list>
#include <map>
#include <vector>
#include <glib/gi18n.h>
#include <glibmm/markup.h>
#include <glibmm-utils/ustring.h>
#include <gtkmm-utils/tile.h>
#include "browser.hh"
#include "category-editor.hh"
#include "category-factory.hh"
#include "file-utils.hh"
#include "document-tag-cloud-model.hh"
#include "document-tile.hh"
#include "i18n-utils.hh"
#include "main-window.hh"
#include "paths.hh"

namespace paperbox {

    using std::list;
    using std::map;
    using std::vector;
    using boost::shared_ptr;

    ///

    class CategoryModel
    {
    public:
        explicit CategoryModel() {}
        ~CategoryModel() {}

        std::list<shared_ptr<Category> > load_data();

        shared_ptr<Category> get_category(const Glib::ustring& name);

    protected:
        std::map<Glib::ustring, shared_ptr<Category> > data_;
    };

    ///

    list<shared_ptr<Category> >
    CategoryModel::load_data()
    {
        list<shared_ptr<Category> > categories(
            CategoryFactory::load_categories());

        list<shared_ptr<Category> >::iterator it(categories.begin());
        list<shared_ptr<Category> >::iterator end(categories.end());

        for ( ; it != end; ++it)
            data_[(*it)->get_name()] = *it;

        return categories;
    }

    shared_ptr<Category>
    CategoryModel::get_category(const Glib::ustring& name)
    {
        shared_ptr<Category> ptr;
        map<Glib::ustring, shared_ptr<Category> >::iterator it = 
            data_.find(name);

        if (it != data_.end())
            ptr = it->second;

        return ptr;
    }

    ///

    const int DOCS_ALL = 1;
    const int DOCS_RECENT = 2;
    const int DOCS_UNTAGGED = 3;

    const int MAX_RECENT_DOCS = 7;

    MainWindow::MainWindow(GtkWindow* cobject,
                           const Glib::RefPtr<Gnome::Glade::Xml>& glade)
        :
        Gtk::Window(cobject),
        glade_(glade),
        hpane_(0),
        left_top_vbox_(0),
        tile_view_(0),
        right_top_vbox_(0),
        right_vpane_(0),
        category_vbox_(0),
        button_edit_category_(Gtk::Stock::EDIT),
        tag_cloud_vbox_(0),
        tag_box_(false, 4)
    {
        init_gui();
        set_default_size(800, 690);
        setup_pane_pos();

        browser_ = Browser::instance();
        tiles_.reset(new TileSet());
        category_model_.reset(new CategoryModel());
        reload_category_view();

        connect_signals();
    }

    MainWindow::~MainWindow()
    {
    }

    MainWindow*
    MainWindow::create()
    {
        Glib::RefPtr<Gnome::Glade::Xml> glade_xml =
            Gnome::Glade::Xml::create(glade_window_main);

        MainWindow* p = 0;
        glade_xml->get_widget_derived("MainWindow", p);
        return p;
    }

    void
    MainWindow::init_gui()
    {
        get_widgets_from_ui_file();
        setup_tiles();

        label_tags_.set_markup(boldify(_("Tags")));
        tag_cloud_vbox_->pack_start(label_tags_, false, false);

        model_.reset(new DocumentTagCloudModel(10, 18));
        tag_cloud_.set_model(model_);
        tag_cloud_vbox_->pack_start(tag_box_);

        tag_box_.pack_start(tag_cloud_, true, true);

        setup_categories();

        right_vpane_->set_position(280);

        set_title(Glib::Util::uprintf("%s %s", PACKAGE_NAME, PACKAGE_VERSION));

        show_all_children();
    }

    void
    MainWindow::get_widgets_from_ui_file()
    {
        // make sure the glade object has been initialized properly in the ctor
        g_assert(glade_);

        glade_->get_widget("hpane", hpane_);
        g_assert(hpane_);

        glade_->get_widget("left_top_vbox", left_top_vbox_);
        g_assert(left_top_vbox_);

        glade_->get_widget("right_top_vbox", right_top_vbox_);
        g_assert(right_top_vbox_);

        glade_->get_widget("right_vpane", right_vpane_);
        g_assert(right_vpane_);

        glade_->get_widget("category_vbox", category_vbox_);
        g_assert(category_vbox_);

        glade_->get_widget("tag_cloud_vbox", tag_cloud_vbox_);
        g_assert(tag_cloud_vbox_);
    }

    void
    MainWindow::setup_tiles()
    {
        tile_view_ = Gtk::manage(new DocumentTileView());
        tile_view_->set_navigator_title(_("Showing documents"));
        tile_view_->set_tiles_per_page(7);

        left_top_vbox_->pack_start(*tile_view_);
    }

    void
    MainWindow::setup_pane_pos()
    {
        int width, height;
        get_size(width, height);
        hpane_->set_position(3 * (width / 4));
    }

    void
    MainWindow::setup_categories()
    {
        category_view_.reset(new CategoryView(category_vbox_));
        category_vbox_->pack_start(button_edit_category_, false, false);

        category_view_->selection->set_select_function(
            sigc::mem_fun(*this, &MainWindow::on_category_selected));
    }

    bool
    MainWindow::on_category_selected(const Glib::RefPtr<Gtk::TreeModel>& /*m*/,
                                     const Gtk::TreeModel::Path& path,
                                     bool path_selected)
    {
        Gtk::TreeModel::Row row = *(category_view_->treemodel->get_iter(path));
        int id = row[category_view_->columns.col_id];
        Glib::ustring cat_name = row[category_view_->columns.col_name];

        if (! ((cat_name != selected_cat_name_) && (! path_selected)))
            return true; // selection hasn't changed

        selected_cat_name_ = cat_name;

        vector<shared_ptr<Document> > docs;

        // decide which set of documents to display
        if (id == DOCS_ALL)
            browser_->get_all_documents(docs);
        else if (id == DOCS_RECENT)
            browser_->get_recent_documents(docs, MAX_RECENT_DOCS);
        else if (id == DOCS_UNTAGGED)
            browser_->get_untagged_documents(docs);
        else {
            shared_ptr<Category> cat =
                category_model_->get_category(selected_cat_name_);

            if (! cat.get())
                g_warning("Invalid selection for category %s",
                          selected_cat_name_.c_str());
            else {
                vector<Glib::ustring> tags = cat->get_tags();
                browser_->get_documents_for_tag_bundle(tags, docs);
            }
        }

        render_documents(docs);

        return true;
    }

    void
    MainWindow::reload_category_view()
    {
        category_view_->treemodel->clear();

        Gtk::TreeModel::Row row;

        row = *(category_view_->treemodel->append());
        row[category_view_->columns.col_id] = DOCS_ALL;
        //TRANSLATORS: 'all' means 'all categories'
        row[category_view_->columns.col_name] = _("All");

        row = *(category_view_->treemodel->append());
        row[category_view_->columns.col_id] = DOCS_RECENT;
        //'recent' means 'recent categories'
        row[category_view_->columns.col_name] = _("Recent");

        row = *(category_view_->treemodel->append());
        row[category_view_->columns.col_id] = DOCS_UNTAGGED;
        //'untagged categories'
        row[category_view_->columns.col_name] = _("Untagged");


        list<shared_ptr<Category> > categories(category_model_->load_data());
        
        list<shared_ptr<Category> >::iterator it(categories.begin());
        list<shared_ptr<Category> >::iterator end(categories.end());

        for ( ; it != end; ++it) {
            row = *(category_view_->treemodel->append());
            row[category_view_->columns.col_name] = (*it)->get_name();
        }

        category_view_->select_first();
    }

    void
    MainWindow::connect_signals()
    {
        browser_->signal_new_document().connect(
            sigc::mem_fun(*this, &MainWindow::on_new_document));

        tile_view_->signal_tile_activated().connect(
            sigc::mem_fun(*this, &MainWindow::on_document_tile_selected));

        button_edit_category_.signal_clicked().connect(
            sigc::mem_fun(*this, &MainWindow::on_edit_category));

        tile_view_->signal_tag_clicked().connect(
            sigc::mem_fun(*this, &MainWindow::on_tag_clicked));

        tag_cloud_.signal_tag_clicked().connect(
            sigc::mem_fun(*this, &MainWindow::on_tag_clicked));
    }

    // Invoked from Browser during idle time.
    // All this should be re-thought when we get xesam dbus api in tracker.
    void
    MainWindow::on_new_document(const boost::shared_ptr<Document>& doc)
    {
        shared_ptr<DocumentTile> tile(new DocumentTile(thumbnailer_, doc));

        bool ok = tiles_->add(tile);
        if (! ok) return; // maybe not?

        tile_view_->add_tile(*tile);

        std::vector<Glib::ustring> tags = doc->get_tags();
        std::vector<Glib::ustring>::iterator it(tags.begin());
        std::vector<Glib::ustring>::iterator end(tags.end());

        for ( ; it != end; ++it)
            model_->add_tag(*it);

        show_all_children();
    }

    void
    MainWindow::on_document_tile_selected(/*Document*/Gtk::Util::Tile& t)
    {
        DocumentTile* dt = dynamic_cast<DocumentTile*>(&t);
        open_file_with_xdg(dt->get_document_uri());
    }

    void
    MainWindow::render_new_tile_set(const vector<shared_ptr<Document> >& docs)
    {
        tile_view_->clear();

        vector<shared_ptr<Document> >::const_iterator it(docs.begin());
        vector<shared_ptr<Document> >::const_iterator end(docs.end());

        for ( ; it != end; ++it) {
            std::string uri = (*it)->get_uri().raw();
            shared_ptr<DocumentTile> tile = tiles_->get_tile(uri);
            if (tile) tile_view_->add_tile(*tile);
        }
    }

    void
    MainWindow::on_tag_clicked(const Glib::ustring& tag)
    {
        vector<shared_ptr<Document> > docs;
        browser_->get_documents_for_tag(tag, docs);

        tile_view_->reset_selection();
        render_new_tile_set(docs);

        category_view_->selection->unselect_all();
        selected_cat_name_ = "";

        // "manually" put a marker on the tag in the cloud
        tag_cloud_.select_tag(tag);
    }

    void
    MainWindow::on_edit_category()
    {
        shared_ptr<CategoryEditor> dialog(CategoryEditor::create());
        dialog->set_default_response(Gtk::RESPONSE_OK);

        dialog->run();

        reload_category_view();
    }

    void
    MainWindow::render_documents(const vector<shared_ptr<Document> > docs)
    {
        render_new_tile_set(docs);
        tag_cloud_.reset_selection();
    }

} // namespace paperbox
