#region PDFsharp - A .NET library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005-2008 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
// http://sourceforge.net/projects/pdfsharp
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
// DEALINGS IN THE SOFTWARE.
#endregion

using System;
using System.Diagnostics;
using System.Collections;
using PdfSharp.Pdf.Internal;

namespace PdfSharp.Pdf.AcroForms
{
  /// <summary>
  /// Represents the signature field.
  /// </summary>
  public sealed class PdfSignatureField : PdfAcroField
  {
    /// <summary>
    /// Initializes a new instance of PdfSignatureField.
    /// </summary>
    internal PdfSignatureField(PdfDocument document) : base(document)
    {
    }

    internal PdfSignatureField(PdfDictionary dict) : base(dict)
    {
    }

    /// <summary>
    /// Predefined keys of this dictionary. 
    /// The description comes from PDF 1.4 Reference.
    /// </summary>
    public new class Keys : PdfAcroField.Keys
    {
      /// <summary>
      /// (Optional) The type of PDF object that this dictionary describes; if present,
      /// must be Sig for a signature dictionary.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Optional)]
      public const string Type = "/Type";

      /// <summary>
      /// (Required; inheritable) The name of the signature handler to be used for
      /// authenticating the fields contents, such as Adobe.PPKLite, Entrust.PPKEF,
      /// CICI.SignIt, or VeriSign.PPKVS.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Required)]
      public const string Filter = "/Filter";

      /// <summary>
      /// (Optional) The name of a specific submethod of the specified handler.
      /// </summary>
      [KeyInfo(KeyType.Name | KeyType.Optional)]
      public const string SubFilter = "/SubFilter";

      /// <summary>
      /// (Required) An array of pairs of integers (starting byte offset, length in bytes)
      /// describing the exact byte range for the digest calculation. Multiple discontiguous
      /// byte ranges may be used to describe a digest that does not include the
      /// signature token itself.
      /// </summary>
      [KeyInfo(KeyType.Array | KeyType.Required)]
      public const string ByteRange = "/ByteRange";

      /// <summary>
      /// (Required) The encrypted signature token.
      /// </summary>
      [KeyInfo(KeyType.String | KeyType.Required)]
      public const string Contents = "/Contents";

      /// <summary>
      /// (Optional) The name of the person or authority signing the document.
      /// </summary>
      [KeyInfo(KeyType.TextString | KeyType.Optional)]
      public const string Name = "/Name";

      /// <summary>
      /// (Optional) The time of signing. Depending on the signature handler, this
      /// may be a normal unverified computer time or a time generated in a verifiable
      /// way from a secure time server.
      /// </summary>
      [KeyInfo(KeyType.Date | KeyType.Optional)]
      public const string M = "/M";

      /// <summary>
      /// (Optional) The CPU host name or physical location of the signing.
      /// </summary>
      [KeyInfo(KeyType.TextString| KeyType.Optional)]
      public const string Location = "/Location";

      /// <summary>
      /// (Optional) The reason for the signing, such as (I agree).
      /// </summary>
      [KeyInfo(KeyType.TextString| KeyType.Optional)]
      public const string Reason = "/Reason";

      /// <summary>
      /// Gets the KeysMeta for these keys.
      /// </summary>
      internal static DictionaryMeta Meta
      {
        get
        {
          if (Keys.meta == null)
            Keys.meta = CreateMeta(typeof(Keys));
          return Keys.meta;
        }
      }
      static DictionaryMeta meta;
    }

    /// <summary>
    /// Gets the KeysMeta of this dictionary type.
    /// </summary>
    internal override DictionaryMeta Meta
    {
      get {return Keys.Meta;}
    }
  }
}
