/* Procman - callbacks
 * Copyright (C) 2001 Kevin Vandersloot
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <glibtop/xmalloc.h>
#include <glibtop/mountlist.h>
#include <glibtop/fsusage.h>
#include <signal.h>
#include "callbacks.h"
#include "interface.h"
#include "proctable.h"
#include "infoview.h"
#include "procdialogs.h"
#include "memmaps.h"
#include "favorites.h"
#include "procactions.h"
#include "load-graph.h"

void
cb_properties_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}

void
cb_preferences_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	ProcData *procdata = user_data;
	
	procdialog_create_preferences_dialog (procdata);
}

void
cb_renice (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	procdialog_create_renice_dialog (procdata);
	
}

void
cb_end_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	if (!procdata->selected_node)
		return;
	
	if (procdata->config.show_kill_warning)
		procdialog_create_kill_dialog (procdata, SIGTERM);
	else
		kill_process (procdata, SIGTERM);
	
}

void
cb_kill_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	if (!procdata->selected_node)
		return;
	
	if (procdata->config.show_kill_warning)
		procdialog_create_kill_dialog (procdata, SIGKILL);
	else
		kill_process (procdata, SIGKILL);
	
}

void
cb_show_memory_maps (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	create_memmaps_dialog (procdata);
}

void
cb_add_to_favorites (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	ProcInfo *info;
	
	if (!procdata->selected_node)
		return;
	
	info = e_tree_memory_node_get_data (procdata->memory, procdata->selected_node);
	add_to_favorites (procdata, info->cmd);
	
}

void		
cb_show_hidden_processes (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	create_blacklist_dialog (procdata);
}

void
cb_hide_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	if (!procdata->selected_node)
		return;
	
	if (procdata->config.show_hide_message)
	{	
		GtkWidget *dialog;
		dialog = procdialog_create_hide_dialog (procdata);
		gtk_widget_show (dialog);
	}
	else
	{
		add_selected_to_blacklist (procdata);
		proctable_update_all (procdata);
	}
	
}


void
cb_about_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

	GtkWidget *about;
	const gchar *authors[] = {
				 _("Kevin Vandersloot (kfv101@psu.edu)"),
				 _("Erik Johnsson (zaphod@linux.nu) - icon support"),
				 NULL
				 };
				 
	about = gnome_about_new (_("Process Manager"), VERSION,
				 _("(C) 2001 Kevin Vandersloot"),
				 authors,
				 _("Simple process viewer using libgtop"),
				 NULL);
				 
	gtk_widget_show (about);  
}


void
cb_app_exit (GtkObject *object, gpointer user_data)
{
	ProcData *procdata = user_data;
	
	cb_app_delete (NULL, NULL, procdata);
	
}

void		
cb_app_delete (GtkWidget *window, GdkEventAny *ev, gpointer data)
{
	ProcData *procdata = data;
	
	if (procdata)
	{
		procman_save_config (procdata);
	}
	if (procdata->timeout != -1)
		gtk_timeout_remove (procdata->timeout);
	if (procdata->cpu_graph)
		gtk_timeout_remove (procdata->cpu_graph->timer_index);
	if (procdata->mem_graph)
		gtk_timeout_remove (procdata->mem_graph->timer_index);
	if (procdata->disk_timeout != -1)
		gtk_timeout_remove (procdata->disk_timeout);
	
	gtk_main_quit ();
	
}

gboolean	
cb_close_simple_dialog (GnomeDialog *dialog, gpointer data)
{
	ProcData *procdata = data;
	
	if (procdata->timeout != -1)
		gtk_timeout_remove (procdata->timeout);
	
	gtk_main_quit ();
		
	return FALSE;

}

void
cb_all_process_menu_clicked 		(GtkWidget	*widget,
					 gpointer	data)
{
	ProcData *procdata = data;
	
	procdata->config.whose_process = ALL_PROCESSES;
	proctable_clear_tree (procdata);
	proctable_update_all (procdata);
}


void
cb_my_process_menu_clicked		(GtkWidget	*widget,
					 gpointer	data)
{
	ProcData *procdata = data;
	
	procdata->config.whose_process = MY_PROCESSES;
	proctable_clear_tree (procdata);
	proctable_update_all (procdata);
}

void
cb_running_process_menu_clicked		(GtkWidget	*widget,
					 gpointer	data)
{
	ProcData *procdata = data;
	
	procdata->config.whose_process = RUNNING_PROCESSES;
	proctable_clear_tree (procdata);
	proctable_update_all (procdata);
}				


void
cb_favorites_menu_clicked (GtkWidget *widget, gpointer data)
{
	ProcData *procdata = data;
	
	procdata->config.whose_process = FAVORITE_PROCESSES;
	proctable_clear_tree (procdata);
	proctable_update_all (procdata);
}

void
popup_menu_renice (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	procdialog_create_renice_dialog (procdata);
}

void
popup_menu_show_memory_maps (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	create_memmaps_dialog (procdata);
}

void
popup_menu_hide_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	
	if (!procdata->selected_node)
		return;
	
	if (procdata->config.show_hide_message)
	{	
		GtkWidget *dialog;
		dialog = procdialog_create_hide_dialog (procdata);
		gtk_widget_show (dialog);
	}
	else
	{
		add_selected_to_blacklist (procdata);
		proctable_update_all (procdata);
	}
	
}

void 
popup_menu_end_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;

        if (!procdata->selected_node)
		return;
	
	if (procdata->config.show_kill_warning)
		procdialog_create_kill_dialog (procdata, SIGTERM);
	else
		kill_process (procdata, SIGTERM);	
}

void 
popup_menu_kill_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;

        if (!procdata->selected_node)
		return;
		
	if (procdata->config.show_kill_warning)
		procdialog_create_kill_dialog (procdata, SIGKILL);
	else	
		kill_process (procdata, SIGKILL);
			
}

void 
popup_menu_about_process (GtkMenuItem *menuitem, gpointer data)
{
	ProcData *procdata = data;
	ProcInfo *info = NULL;
	gchar *name;
	
	if (!procdata->selected_node)
		return;
		
	info = e_tree_memory_node_get_data (procdata->memory, 
					    procdata->selected_node);
	g_return_if_fail (info != NULL);
	
	/* FIXME: this is lame. GNOME help browser sucks balls. There should be a way
	to first check man pages, then info pages and give a nice error message if nothing
	exists */			    
	name = g_strjoin (NULL, "man:", info->cmd, NULL);
	gnome_url_show (name);
	g_free (name);
					    
}

gint
cb_right_click (ETree *tree, int row, ETreePath path, int col,
                 GdkEvent *event, gpointer data)
{
        ProcData *procdata = data;

        do_popup_menu (procdata, event);

        return TRUE;

}

void
cb_end_process_button_pressed          (GtkButton       *button,
                                        gpointer         data)
{

	ProcData *procdata = data;

	if (!procdata->selected_node)
		return;
		
	if (procdata->config.show_kill_warning)
		procdialog_create_kill_dialog (procdata, SIGTERM);
	else
		kill_process (procdata, SIGTERM);
	
}

void
cb_logout (GtkButton *button, gpointer data)
{
	ProcData *procdata = data;
	
	g_print ("logout \n");
	
}

void
cb_info_button_pressed			(GtkButton	*button,
					 gpointer	user_data)
{
	ProcData *procdata = user_data;
	
	toggle_infoview (procdata);
		
}	

void		
cb_search (GtkEditable *editable, gpointer data)
{
	ProcData *procdata = data;
	gchar *text;
	
	text = gtk_editable_get_chars (editable, 0, -1);
	
	proctable_search_table (procdata, text);
	
	g_free (text);
}

void		
cb_cpu_color_changed (GnomeColorPicker *cp, guint r, guint g, guint b,
		      guint a, gpointer data)
{
	ProcData *procdata = data;
	
	procdata->cpu_graph->colors[2].red = r;
	procdata->cpu_graph->colors[2].green = g;
	procdata->cpu_graph->colors[2].blue = b;
	procdata->config.cpu_color.red = r;
	procdata->config.cpu_color.green = g;
	procdata->config.cpu_color.blue = b;	
	procdata->cpu_graph->colors_allocated = FALSE;

}

void		
cb_mem_color_changed (GnomeColorPicker *cp, guint r, guint g, guint b,
		      guint a, gpointer data)
{
	ProcData *procdata = data;
	
	procdata->mem_graph->colors[2].red = r;
	procdata->mem_graph->colors[2].green = g;
	procdata->mem_graph->colors[2].blue = b;
	procdata->config.mem_color.red = r;
	procdata->config.mem_color.green = g;
	procdata->config.mem_color.blue = b;	
	procdata->mem_graph->colors_allocated = FALSE;

}

void		
cb_swap_color_changed (GnomeColorPicker *cp, guint r, guint g, guint b,
		       guint a, gpointer data)
{
	ProcData *procdata = data;
	
	procdata->mem_graph->colors[3].red = r;
	procdata->mem_graph->colors[3].green = g;
	procdata->mem_graph->colors[3].blue = b;
	procdata->config.swap_color.red = r;
	procdata->config.swap_color.green = g;
	procdata->config.swap_color.blue = b;	
	procdata->mem_graph->colors_allocated = FALSE;

}

void
cb_table_selected (ETree *tree, int row, ETreePath path, gpointer data)
{

	ProcData *procdata = data;
	ProcInfo *info;
	
	if (!tree)
		return;
	if (row == -1)
	{
		update_sensitivity (procdata, FALSE);
		return;
	}
		
	
	info = e_tree_memory_node_get_data (procdata->memory, path);
	procdata->selected_pid = info->pid;
	procdata->selected_node = path;
	
		
	update_sensitivity (procdata, TRUE);
		
	if (procdata->config.show_more_info == TRUE)
		infoview_update (procdata);
	
	update_memmaps_dialog (procdata);
		 


}

gint
cb_tree_key_press (ETree *tree, int row, ETreePath path, int col,
		   GdkEvent *event, gpointer data)
{
	ProcData *procdata = data;
		
	switch (event->key.keyval) {
	case GDK_Return:
	case GDK_space:
		toggle_infoview (procdata);
		break;
	default:
	}
		
	return FALSE;
		
}

void
cb_double_click (ETree *tree, int row, ETreePath path, int col, 
		 GdkEvent *event, gpointer data)
{
	ProcData *procdata = data;
	
	toggle_infoview (procdata);

}

void		
cb_switch_page (GtkNotebook *nb, GtkNotebookPage *page, 
		gint num, gpointer data)
{
	ProcData *procdata = data;
		
	procdata->config.current_tab = num;
	
	if (num == 0) {
		if (procdata->timeout == -1) 
			procdata->timeout = gtk_timeout_add (procdata->config.update_interval,
			 			     	     cb_timeout, procdata);
		load_graph_stop (procdata->cpu_graph);
		load_graph_stop (procdata->mem_graph);
		if (procdata->selected_node)
			update_sensitivity (procdata, TRUE);
	}
	else {
		if (procdata->timeout != -1 ) {
			gtk_timeout_remove (procdata->timeout);
			procdata->timeout = -1;
		}
		load_graph_start (procdata->cpu_graph);
		load_graph_start (procdata->mem_graph);
		if (procdata->selected_node)
			update_sensitivity (procdata, FALSE);
	}

}

static gchar *
get_size_string (gint size)
{
	gfloat fsize;

	fsize = (gfloat) size;
	if (fsize < 1024.0) 
		return g_strdup_printf (_("%d K"), (int)fsize);
		
	fsize /= 1024.0;
	if (fsize < 1024.0)
		return g_strdup_printf (_("%.0f MB"), fsize);
	
	fsize /= 1024.0;
	return g_strdup_printf (_("%.1f GB"), fsize);

}

gint
cb_update_disks (gpointer data)
{
	ProcData *procdata = data;
	GtkWidget *clist = procdata->disk_clist;
	glibtop_mountentry *entry;
	glibtop_mountlist mountlist;
	float old_adj_hval = GTK_CLIST (clist)->hadjustment->value;
        float old_adj_vval = GTK_CLIST (clist)->vadjustment->value;
        gint old_hoffset = GTK_CLIST (clist)->hoffset;
        gint old_voffset = GTK_CLIST (clist)->voffset;
	gint i;
	
	gtk_clist_freeze (GTK_CLIST (clist));
	gtk_clist_clear (GTK_CLIST (clist));
	
	entry = glibtop_get_mountlist (&mountlist, 0);
	for (i=0; i < mountlist.number; i++) {
		glibtop_fsusage usage;
		gchar *text[4];
		
		glibtop_get_fsusage (&usage, entry[i].mountdir);
		text[0] = g_strdup (entry[i].devname);
		text[2] = get_size_string ((usage.blocks - usage.bfree) / 2);
		//text[3] = get_size_string (usage.bfree / 2);
		text[3] = get_size_string (usage.blocks / 2);
		text[1] = g_strdup (entry[i].mountdir);
		//text[4] = g_strdup_printf ("%s / %s", text[1], text[3]);
		/* Hmm, usage.blocks == 0 seems to get rid of /proc and all
		** the other useless entries */
		if (usage.blocks != 0)
			gtk_clist_append (GTK_CLIST (clist), text);
		
		g_free (text[0]);
		g_free (text[1]);
		g_free (text[2]);
		g_free (text[3]);
		//g_free (text[4]);
	}
	
	GTK_CLIST (clist)->hadjustment->value = old_adj_hval;
        GTK_CLIST (clist)->vadjustment->value = old_adj_vval;

        GTK_CLIST (clist)->hoffset = old_hoffset;
        GTK_CLIST (clist)->voffset = old_voffset;
	
	gtk_clist_thaw (GTK_CLIST (clist));
	
	glibtop_free (entry);
	
	return TRUE;
}
	
gint
cb_timeout (gpointer data)
{
	ProcData *procdata = data;
	
	proctable_update_all (procdata);
	
	return TRUE;
}
