/* Procman
 * Copyright (C) 2001 Kevin Vandersloot
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <glibtop.h>
#include <gal/widgets/e-cursors.h>
#include <pthread.h>
#include "procman.h"
#include "proctable.h"
#include "interface.h"
#include "favorites.h"
#include "prettytable.h"

GtkWidget *app;
static int simple_view = FALSE;
pthread_t thread;

struct poptOption options[] = {
  {
    "simple",
    's',
    POPT_ARG_NONE,
    &simple_view,
    0,
    N_("show simple dialog to end processes and logout"),
    NULL,
  },
  {
    NULL,
    '\0',
    0,
    NULL,
    0,
    NULL,
    NULL
  }
};

static void
procman_get_save_files (ProcData *procdata)
{
	gchar *homedir;
	
	homedir = g_get_home_dir ();
	procdata->config.tree_state_file = g_strconcat (homedir, "/.gnome/", 
							"procman_header_state", NULL);
	procdata->config.memmaps_state_file = g_strconcat (homedir, "/.gnome/",
							   "procman_memmaps_state", NULL);

}

static gint
idle_func (gpointer data)
{
	ProcData *procdata = data;
	
	if (!procdata->pretty_table)
	{
		if (pthread_create (&thread, NULL, (void*) prettytable_load_async,
				   (void*) procdata)) 
			g_print ("error creating new thread \n");
	}
			
	return FALSE;
	
}

static gint
icon_load_finished (gpointer data)
{
	ProcData *procdata = data;
	PrettyTable *table = NULL;
	
	if (procdata->desktop_load_finished)
	{
		if (pthread_join (thread, (void *)&table))
			g_print ("error joining thread \n");
		procdata->pretty_table = (PrettyTable *)table;
		proctable_update_all (procdata);
		return FALSE;
	}
	
	return TRUE;
}

static void
load_desktop_files (ProcData *pd)
{
	/* delay load the .desktop files. Procman will display icons and app names
	** if the pd->pretty_table structure is not NULL. The timeout will monitor
	** whether or not that becomes the case and update the table if so. There is 
	** undoubtedly a better solution, but I am not too knowlegabe about threads.
	** Basically what needs to be done is to update the table when the thread is
	** finished 
	*/
	if (pd->config.load_desktop_files && pd->config.delay_load)
	{
		pd->pretty_table = NULL;
		pd->desktop_load_finished = FALSE;
		gtk_idle_add_priority (800, idle_func, pd);
		/*idle_func (pd);*/
		gtk_timeout_add (500, icon_load_finished, pd);
	}
	else if (pd->config.load_desktop_files && !pd->config.delay_load) 
		pd->pretty_table = pretty_table_new ();
	else
		pd->pretty_table = NULL;
}


static ProcData *
procman_data_new (void)
{

	ProcData *pd;

	pd = g_new0 (ProcData, 1);

	pd->tree = NULL;
	pd->model = NULL;
	pd->memory = NULL;
	pd->infobox = NULL;
	pd->info = NULL;
	pd->proc_num = 0;
	pd->selected_pid = -1;
	pd->selected_node = NULL;
	pd->timeout = -1;
	pd->favorites = NULL;
	pd->blacklist = NULL;
	pd->cpu_graph = NULL;
	pd->mem_graph = NULL;
	pd->disk_timeout = -1;
	
	pd->config.width = 
		gnome_config_get_int ("procman/Config/width=440");
	pd->config.height = 
		gnome_config_get_int ("procman/Config/height=495");
	pd->config.show_more_info = 
		gnome_config_get_bool ("procman/Config/more_info=FALSE");
	pd->config.show_tree = 
		gnome_config_get_bool ("procman/Config/show_tree=TRUE");
	pd->config.show_kill_warning = 
		gnome_config_get_bool ("procman/Config/kill_dialog=TRUE");
	pd->config.show_hide_message = 
		gnome_config_get_bool ("procman/Config/hide_message=TRUE");
	pd->config.delay_load = 
		gnome_config_get_bool ("procman/Config/delay_load=TRUE");
	pd->config.load_desktop_files = 
		gnome_config_get_bool ("procman/Config/load_desktop_files=TRUE");
	pd->config.show_pretty_names = 
		gnome_config_get_bool ("procman/Config/show_pretty_names=FALSE");
	pd->config.show_threads = 
		gnome_config_get_bool ("procman/Config/show_threads=FALSE");
	pd->config.update_interval = 
		gnome_config_get_int ("procman/Config/update_interval=3000");
	pd->config.graph_update_interval = 
		gnome_config_get_int ("procman/Config/graph_update_interval=500");
	pd->config.disks_update_interval = 
		gnome_config_get_int ("procman/Config/disks_update_interval=5000");
	pd->config.whose_process = gnome_config_get_int ("procman/Config/view_as=1");
	pd->config.current_tab = gnome_config_get_int ("procman/Config/current_tab=0");
	pd->config.pane_pos = gnome_config_get_int ("procman/Config/pane_pos=300");
	pd->config.disk_width1 = gnome_config_get_int ("procman/Config/disk1=60");
	pd->config.disk_width2 = gnome_config_get_int ("procman/Config/disk2=120");
	pd->config.disk_width3 = gnome_config_get_int ("procman/Config/disk3=80");
	pd->config.disk_width4 = gnome_config_get_int ("procman/Config/disk4=10");
	pd->config.bg_color.red = gnome_config_get_int
		("procman/Config/bg_red=0");
	pd->config.bg_color.green = gnome_config_get_int
		("procman/Config/bg_green=0");
	pd->config.bg_color.blue= gnome_config_get_int
		("procman/Config/bg_blue=0");
	pd->config.frame_color.red = gnome_config_get_int
		("procman/Config/frame_red=20409");
	pd->config.frame_color.green = gnome_config_get_int
		("procman/Config/frame_green=32271");
	pd->config.frame_color.blue = gnome_config_get_int
		("procman/Config/frame_blue=17781");
	pd->config.cpu_color.red = gnome_config_get_int
		("procman/Config/cpu_red=65535");
	pd->config.cpu_color.green = gnome_config_get_int
		("procman/Config/cpu_green=591");
	pd->config.cpu_color.blue = gnome_config_get_int
		("procman/Config/cpu_blue=0");
	pd->config.mem_color.red = gnome_config_get_int
		("procman/Config/mem_red=65535");
	pd->config.mem_color.green = gnome_config_get_int
		("procman/Config/mem_green=591");
	pd->config.mem_color.blue = gnome_config_get_int
		("procman/Config/mem_blue=0");
	pd->config.swap_color.red = gnome_config_get_int
		("procman/Config/swap_red=1363");
	pd->config.swap_color.green = gnome_config_get_int
		("procman/Config/swap_green=52130");
	pd->config.swap_color.blue = gnome_config_get_int
		("procman/Config/swap_blue=18595");
	
	procman_get_save_files (pd);

	get_blacklist (pd);
	
	pd->config.simple_view = simple_view;	
	if (pd->config.simple_view) {
		pd->config.width = 325;
		pd->config.height = 400;
		pd->config.whose_process = 1;
		pd->config.show_more_info = FALSE;
		pd->config.show_tree = FALSE;
		pd->config.show_kill_warning = TRUE;
		pd->config.show_pretty_names = TRUE;
		pd->config.show_threads = FALSE;
		pd->config.current_tab = 0;
	}	

	return pd;

}

static void
procman_free_data (ProcData *procdata)
{

	proctable_free_table (procdata);
	
	pretty_table_free (procdata->pretty_table);
	
	g_free (procdata);
	
}

void
procman_save_config (ProcData *data)
{
	gint width, height, pane_pos;

	if (!data)
		return;
		
	if (data->config.simple_view)
		return;
		
	proctable_save_state (data);
		
	gdk_window_get_size (app->window, &width, &height);
	data->config.width = width;
	data->config.height = height;
	
	pane_pos = get_sys_pane_pos ();
	data->config.pane_pos = pane_pos;		
		
	gnome_config_set_int ("procman/Config/width",data->config.width);
	gnome_config_set_int ("procman/Config/height",data->config.height);	
	gnome_config_set_int ("procman/Config/view_as",data->config.whose_process);
	gnome_config_set_bool ("procman/Config/more_info", data->config.show_more_info);
	gnome_config_set_bool ("procman/Config/kill_dialog", data->config.show_kill_warning);
	gnome_config_set_bool ("procman/Config/hide_message", data->config.show_hide_message);
	gnome_config_set_bool ("procman/Config/show_tree", data->config.show_tree);
	gnome_config_set_bool ("procman/Config/delay_load", data->config.delay_load);
	gnome_config_set_bool ("procman/Config/load_desktop_files", 
			       data->config.load_desktop_files);
	gnome_config_set_bool ("procman/Config/show_pretty_names", 
			       data->config.show_pretty_names);
	gnome_config_set_bool ("procman/Config/show_threads", data->config.show_threads);
	gnome_config_set_int ("procman/Config/update_interval", data->config.update_interval);
	gnome_config_set_int ("procman/Config/graph_update_interval", 
			      data->config.graph_update_interval);
	gnome_config_set_int ("procman/Config/disks_update_interval", 
			      data->config.disks_update_interval);
	gnome_config_set_int ("procman/Config/current_tab", data->config.current_tab);
	gnome_config_set_int ("procman/Config/pane_pos", data->config.pane_pos);
	gnome_config_set_int ("procman/Config/disk1", data->config.disk_width1);
	gnome_config_set_int ("procman/Config/disk2", data->config.disk_width2);
	gnome_config_set_int ("procman/Config/disk3", data->config.disk_width3);
	gnome_config_set_int ("procman/Config/disk4", data->config.disk_width4);
	gnome_config_set_int ("procman/Config/bg_red", data->config.bg_color.red);
	gnome_config_set_int ("procman/Config/bg_green", data->config.bg_color.green);
	gnome_config_set_int ("procman/Config/bg_blue", data->config.bg_color.blue);
	gnome_config_set_int ("procman/Config/frame_red", data->config.frame_color.red);
	gnome_config_set_int ("procman/Config/frame_green", data->config.frame_color.green);
	gnome_config_set_int ("procman/Config/frame_blue", data->config.frame_color.blue);
	gnome_config_set_int ("procman/Config/cpu_red", data->config.cpu_color.red);
	gnome_config_set_int ("procman/Config/cpu_green", data->config.cpu_color.green);
	gnome_config_set_int ("procman/Config/cpu_blue", data->config.cpu_color.blue);
	gnome_config_set_int ("procman/Config/mem_red", data->config.mem_color.red);
	gnome_config_set_int ("procman/Config/mem_green", data->config.mem_color.green);
	gnome_config_set_int ("procman/Config/mem_blue", data->config.mem_color.blue);
	gnome_config_set_int ("procman/Config/swap_red", data->config.swap_color.red);
	gnome_config_set_int ("procman/Config/swap_green", data->config.swap_color.green);
	gnome_config_set_int ("procman/Config/swap_blue", data->config.swap_color.blue);
	
	save_blacklist (data);
	gnome_config_sync ();

}

int
main (int argc, char *argv[])
{
	ProcData *procdata;
	poptContext pctx;
	char **args;
	
#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);
#endif

	gnome_init_with_popt_table ("procman", VERSION, argc, argv, options, 0, &pctx);
	gnome_window_icon_set_default_from_file (GNOME_ICONDIR"/procman.png");
	
	args = (char**) poptGetArgs (pctx);
	poptFreeContext(pctx);
	
	e_cursors_init ();

	glibtop_init ();

	procdata = procman_data_new ();

	if (procdata->config.simple_view) 
		app = create_simple_view_dialog (procdata);
	else 
		app = create_main_window (procdata);
		
	load_desktop_files (procdata);
	
	proctable_update_all (procdata);
	
	if (!app)
		return 0;  
			
 	gtk_widget_show (app);	
 	
	gtk_main ();
	
	e_cursors_shutdown ();
	
	procman_free_data (procdata);
	
	return 0;
}

