%{

/*
 This file is part of pybliographer
 
 Copyright (C) 1998-1999 Frederic GOBRY
 Email : gobry@idiap.ch
 	   
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2 
 of the License, or (at your option) any later version.
   
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 
 $Id: bibparse.y,v 1.24 1999/07/30 12:34:45 gobry Exp $
*/

#include "parsername.h"
#include "bibtex.h"

extern void bibtex_parser_initialize (BibtexFile *);
extern void bibtex_parser_continue (BibtexFile *);
extern void bibtex_parser_finish (BibtexFile *);

extern gboolean bibtex_parser_is_content;

extern int yydebug;

static BibtexEntry *	entry	= NULL;
static int		start_line;
static BibtexFile *	current_file;
static GString *	error_string = NULL;

static void 
nop (void) { 
    return ;
}

void 
bibtex_next_line (void) { 
    entry->length ++; 
}

void 
bibtex_analyzer_initialize (BibtexFile * file)  {
    bibtex_parser_initialize (file);
}

void 
bibtex_analyzer_finish (BibtexFile * file)  {
    g_return_if_fail (file != NULL);

    bibtex_parser_finish (file);
    
    if (error_string)
	g_string_free (error_string, TRUE);
    
    error_string = NULL;
    current_file = NULL;
}
 
BibtexEntry * 
parse_bibtex (BibtexFile * file) {
  int ret;

  g_return_val_if_fail (file != NULL, NULL);

  if (! error_string) {
      error_string = g_string_new (NULL);
  }

  current_file = file;

  yydebug = file->debug;
  start_line = file->line;

  entry = bibtex_entry_new ();

  bibtex_parser_continue (file);
  bibtex_parser_is_content = FALSE;

  ret = yyparse ();

  bibtex_tmp_string_free ();

  if (ret != 0) {
      file->line += entry->length;
      bibtex_entry_destroy (entry, TRUE);
      entry = NULL;
  }

  return entry;
}

void 
yyerror (char * s) {
    if (current_file) {
	bibtex_error ("%s:%d: %s", current_file->filename,
		      start_line + entry->length, s);
    }
    else {
	bibtex_error ("%d: %s", 
		      start_line + entry->length, s);

    }
}

static void 
yywarning (char * s) {
    if (current_file) {
	bibtex_warning ("%s:%d: %s", current_file->filename,
			start_line + entry->length, s);
    }
    else {
	bibtex_warning ("%d: %s", 
			start_line + entry->length, s);

    }
}

%}	

%union{
    gchar * text;
    BibtexStruct * body;
}

%token end_of_file
%token <text> L_NAME
%token <text> L_DIGIT
%token <text> L_COMMAND
%token <text> L_BODY
%token <text> L_SPACE
%token <text> L_UBSPACE

%type <entry> entry
%type <entry> values
%type <entry> value

%type <body> content
%type <body> simple_content
%type <body> content_brace
%type <body> content_quote
%type <body> text_quote
%type <body> text_brace
%type <body> text_part

%% 

/* Les deux types d'entrees */
/* ================================================== */
entry:	  '@' L_NAME '{' values '}' 
/* -------------------------------------------------- */
{
    entry->type = g_strdup ($2);
    g_strdown (entry->type);

    YYACCEPT; 
}
/* -------------------------------------------------- */
        | '@' L_NAME '(' values ')' 
/* -------------------------------------------------- */
{ 
    entry->type = g_strdup ($2);
    g_strdown (entry->type);

    YYACCEPT; 	
}
/* -------------------------------------------------- */
	| end_of_file		    
/* -------------------------------------------------- */
{ 
    current_file->eof = TRUE; 
    YYABORT; 
}
/* -------------------------------------------------- */
	| '@' L_NAME '(' error ')'
/* -------------------------------------------------- */
{
    yyerror ("perhaps there is a missing coma");
    YYABORT;
}
/* -------------------------------------------------- */
	| '@' L_NAME '{' error '}'
/* -------------------------------------------------- */
{
    yyerror ("perhaps there is a missing coma");
    YYABORT;
}
/* -------------------------------------------------- */
	;



/* ================================================== */
/* La liste des valeurs */
/* -------------------------------------------------- */
values:	  value ',' values
/* -------------------------------------------------- */
{
    nop ();
}
/* -------------------------------------------------- */
	| value	','
/* -------------------------------------------------- */
{
    nop ();
}
/* -------------------------------------------------- */
	| value
/* -------------------------------------------------- */
{
    nop ();
}
	;




/* Une valeur */
/* ================================================== */
value:	  L_NAME '=' content 
/* -------------------------------------------------- */
{ 
    char * name;
    BibtexField * field;
    BibtexFieldType type = BIBTEX_OTHER;

    g_strdown ($1);
    field = g_hash_table_lookup (entry->table, $1);

    /* Get a new instance of a field name */
    if (field) {
	g_string_sprintf (error_string, "field `%s' is already defined", $1); 
	yywarning (error_string->str);

	bibtex_field_destroy (field, TRUE);
	name = $1;
    }
    else {
	name = g_strdup ($1);
    }

    /* Search its type */
    do {
	if (strcmp (name, "author") == 0) {
	    type = BIBTEX_AUTHOR;
	    break;
	}

	if (strcmp (name, "title") == 0) {
	    type = BIBTEX_TITLE;
	    break;
	}

	if (strcmp (name, "year") == 0) {
	    type = BIBTEX_DATE;
	    break;
	}
    } 
    while (0);

    /* Convert into the right field */
    field = bibtex_struct_as_field (bibtex_struct_flatten ($3),
				    type);

    g_hash_table_insert (entry->table, name, field);
}
/* -------------------------------------------------- */
	| content
/* -------------------------------------------------- */
{ 
    if (entry->preamble) {
	yyerror ("entry already contains a preamble");
	YYABORT;
    }

    entry->preamble = $1;
}
/* -------------------------------------------------- */
	;



/* ================================================== */
content:    simple_content '#' content	
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_append ($1, $3);
} 
/* -------------------------------------------------- */
	  | simple_content	
/* -------------------------------------------------- */
{
    $$ = $1;
}
/* -------------------------------------------------- */
	  ;

/* 
   Definition du contenu d'un champ, encadre par des accolades
*/

/* ================================================== */
content_brace: '{' { bibtex_parser_is_content = TRUE; }
		text_brace '}'
/* -------------------------------------------------- */
{ 
    bibtex_parser_is_content = FALSE; 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_SUB);

    $$->value.sub->encloser = BIBTEX_ENCLOSER_BRACE;
    $$->value.sub->content  = $3;
} 
;

/*
  Definition du contenu d'un champ encadre par des guillemets
*/
/* ================================================== */
content_quote: '"' { bibtex_parser_is_content = TRUE; }
		text_quote '"'
/* -------------------------------------------------- */
{ 
    bibtex_parser_is_content = FALSE; 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_SUB);

    $$->value.sub->encloser = BIBTEX_ENCLOSER_QUOTE;
    $$->value.sub->content  = $3;
} 
;

/* ================================================== */
simple_content:   L_DIGIT 
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_TEXT);
    $$->value.text = g_strdup ($1);
}
/* -------------------------------------------------- */
	       | L_NAME 
/* -------------------------------------------------- */
{
    $$ = bibtex_struct_new (BIBTEX_STRUCT_REF);
    $$->value.ref = g_strdup ($1);
}
/* -------------------------------------------------- */
	       | content_brace
/* -------------------------------------------------- */
{ 
    $$ = $1;
}
/* -------------------------------------------------- */
	       | content_quote
/* -------------------------------------------------- */
{ 
    $$ = $1;
}
/* -------------------------------------------------- */
	       ;


/* ================================================== */
text_part: L_COMMAND 
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_COMMAND);
    $$->value.com = g_strdup ($1 + 1);
}
/* -------------------------------------------------- */
	   | '{' text_brace '}'		
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_SUB);
    $$->value.sub->encloser = BIBTEX_ENCLOSER_BRACE;
    $$->value.sub->content  = $2;
}
/* -------------------------------------------------- */
	       | L_SPACE
/* -------------------------------------------------- */
{
    $$ = bibtex_struct_new (BIBTEX_STRUCT_SPACE);
}
/* -------------------------------------------------- */
	       | L_UBSPACE
/* -------------------------------------------------- */
{
    $$ = bibtex_struct_new (BIBTEX_STRUCT_SPACE);
    $$->value.unbreakable = TRUE;
}
/* -------------------------------------------------- */
	   | L_BODY
/* -------------------------------------------------- */
{
    $$ = bibtex_struct_new (BIBTEX_STRUCT_TEXT);
    $$->value.text = g_strdup ($1);
}
/* -------------------------------------------------- */
	   ;


/* ================================================== */
text_brace:				
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_TEXT);
    $$->value.text = g_strdup ("");
}
/* -------------------------------------------------- */
       | '"'  text_brace		
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_TEXT);
    $$->value.text = g_strdup ("\"");

    $$ = bibtex_struct_append ($$, $2);
}
/* -------------------------------------------------- */
       | text_part text_brace 	
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_append ($1, $2);
}
/* -------------------------------------------------- */
       ;


/* ================================================== */
text_quote:				
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_new (BIBTEX_STRUCT_TEXT);
    $$->value.text = g_strdup ("");
}
/* -------------------------------------------------- */
       | text_part text_quote 	
/* -------------------------------------------------- */
{ 
    $$ = bibtex_struct_append ($1, $2);
}
/* -------------------------------------------------- */
;


%%     
	
