/*
 This file is part of pybliographer
 
 Copyright (C) 1998-1999 Frederic GOBRY
 Email : gobry@idiap.ch
 	   
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2 
 of the License, or (at your option) any later version.
   
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 
 $Id: bibtexparser.c,v 1.6 1999/07/22 07:10:25 gobry Exp $
*/

#include "Python.h"
#include "bibtex.h"

/* Destructor of BibtexFile */

static void bibtex_py_close (void * pointer)
{
    bibtex_close ((BibtexFile *) pointer);
}

/* Destructor of BibtexEntry */

static void destroy_field (void * pointer)
{
    bibtex_field_destroy ((BibtexField *) pointer, TRUE);
}

static void 
py_message_handler (const gchar *log_domain G_GNUC_UNUSED,
		    GLogLevelFlags log_level,
		    const gchar *message,
		    gpointer user_data G_GNUC_UNUSED)
{
    PyErr_SetString (PyExc_IOError, message);
}

static PyObject *
bib_open (PyObject * self, PyObject * args)
{
    char * name;
    BibtexFile * file;

    if (! PyArg_ParseTuple(args, "s", & name))
	return NULL;
 
    file = bibtex_open (name);

    if (file == NULL) {
	return NULL;
    }
  
    return PyCObject_FromVoidPtr (file, bibtex_py_close);
}

static PyObject *
bib_expand (PyObject * self, PyObject * args) {
    PyObject * liste, * tmp;
    BibtexFieldType type;
    BibtexField * field;
    BibtexFile * file;
    BibtexAuthor * author;
    int i;

    if (! PyArg_ParseTuple(args, "OOi", &file, & field, & type))
	return NULL;

    file = (BibtexFile *) PyCObject_AsVoidPtr ((gpointer) file);
    field = (BibtexField *) PyCObject_AsVoidPtr ((gpointer) field);

    if (! field->converted) {
	if (type != -1) {
	    field->type = type;
	}

	bibtex_field_parse (field, file->table);
    }

    switch (field->type) {
    case BIBTEX_TITLE:
    case BIBTEX_OTHER:
	tmp = Py_BuildValue ("iis", field->type, field->loss,
			     field->text);
	break;
    case BIBTEX_DATE:
	tmp = Py_BuildValue ("iisiii", 
			     field->type,
			     field->loss,
			     field->text,
			     field->field.date.year,
			     field->field.date.month,
			     field->field.date.day);
	break;

    case BIBTEX_AUTHOR:
	liste = PyList_New (field->field.author->len);
	for (i = 0; i < field->field.author->len; i++) {
	    author = & g_array_index (field->field.author, 
				      BibtexAuthor, i);
	    PyList_SetItem (liste, i,
			    Py_BuildValue ("ssss", author->honorific,
					   author->first,
					   author->last,
					   author->lineage));
	}
	tmp = Py_BuildValue ("iisO", 
			     field->type, 
			     field->loss, 
			     field->text, liste);
	Py_DECREF (liste);
	break;

    default:
	tmp = Py_None;
	Py_INCREF (Py_None);
    }

    return tmp;
}

static PyObject *
bib_native (PyObject * self, PyObject * args) {
    PyObject * liste, * tmp;
    BibtexFieldType type;
    BibtexField * field;
    BibtexFile * file;
    BibtexAuthor * author;
    gchar * text;
    int i;

    if (! PyArg_ParseTuple(args, "O", & field))
	return NULL;

    field = (BibtexField *) PyCObject_AsVoidPtr ((gpointer) field);

    text = bibtex_struct_as_bibtex (field->structure);
    tmp = Py_BuildValue("s", text); 
    g_free (text);

    return tmp;
}


static void 
fill_dico (gpointer key, gpointer value, gpointer user)
{
    PyObject * dico = (PyObject *) user;
    PyObject * tmp1, * tmp2;

    tmp1 = PyString_FromString ((char *) key);
    tmp2 = PyCObject_FromVoidPtr (value, destroy_field);

    PyDict_SetItem (dico, tmp1, tmp2);

    Py_DECREF (tmp1);
    Py_DECREF (tmp2);
}


static PyObject *
bib_next (PyObject * self, PyObject * args)
{
    BibtexEntry * ent;
    BibtexFile * file;

    PyObject * dico, * tmp, * tmp2;

    if (! PyArg_ParseTuple(args, "O", & file))
	return NULL;

    file = (BibtexFile *) PyCObject_AsVoidPtr ((gpointer) file);

    ent = bibtex_next_entry (file);

    if (ent == NULL) {
	if (file->eof) {
	    Py_INCREF(Py_None);
	    return Py_None;
	}
	return NULL;
    }

    /* Retour de la fonction */
    dico = PyDict_New (); 
    g_hash_table_foreach (ent->table, fill_dico, dico);

    tmp = Py_BuildValue ("ssiiO", ent->name, ent->type, 
			 ent->offset, file->line,
			 dico);
    Py_DECREF (dico);

    bibtex_entry_destroy (ent, FALSE);

    return tmp;
}


static PyObject *
bib_first (PyObject * self, PyObject * args)
{
    BibtexEntry * ent;
    BibtexFile * file;

    if (! PyArg_ParseTuple(args, "O", & file))
	return NULL;

    file = (BibtexFile *) PyCObject_AsVoidPtr ((gpointer) file);
    bibtex_first_entry (file);

    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *
bib_set_offset (PyObject * self, PyObject * args)
{
    BibtexEntry * ent;
    BibtexFile * file;
    gint offset;

    if (! PyArg_ParseTuple(args, "Oi", & file, & offset))
	return NULL;

    file = (BibtexFile *) PyCObject_AsVoidPtr ((gpointer) file);
    bibtex_offset_entry (file, offset);

    if (file->error) {
	return NULL;
    }
  
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *
bib_get_offset (PyObject * self, PyObject * args)
{
    BibtexFile * file;
    gint offset;
    PyObject * tmp;

    if (! PyArg_ParseTuple(args, "O", & file))
	return NULL;

    file = (BibtexFile *) PyCObject_AsVoidPtr ((gpointer) file);
    offset = bibtex_current_offset (file);
    
    tmp = PyInt_FromLong ((long) offset);
    return tmp;
}

static PyObject *
bib_text_as_field (PyObject * self, PyObject * args) 
{
    BibtexFieldType type;
    gchar * string;
    BibtexField * field;

    if (! PyArg_ParseTuple(args, "si", & string, & type))
	return NULL;
    
    field = bibtex_string_as_field (string, type);
    if (! field) return NULL;
    
    return PyCObject_FromVoidPtr (field, destroy_field);
}


static PyMethodDef bibtexMeth [] = {
    { "open", bib_open, METH_VARARGS },
    { "next", bib_next, METH_VARARGS },
    { "first", bib_first, METH_VARARGS },
    { "set_offset", bib_set_offset, METH_VARARGS },
    { "get_offset", bib_get_offset, METH_VARARGS },
    { "expand", bib_expand, METH_VARARGS },
    { "native", bib_native, METH_VARARGS },
    { "text_as_field", bib_text_as_field, METH_VARARGS},
    {NULL, NULL, 0},
};


void initbibtex ()
{
    bibtex_set_default_handler ();
    g_log_set_handler ("BibTeX", BIB_LEVEL_ERROR,   
		       py_message_handler, NULL);

    (void) Py_InitModule("bibtex", bibtexMeth);
}

