# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: Entry.py,v 1.6 1999/08/18 14:05:32 gobry Exp $

from Pyblio.Base import *
import string, re
from gnome.ui import *
from gtk import *

from Pyblio import Fields, Config, Base, Types

from Pyblio.GnomeUI import FieldsInfo

def popup_add (menu, item, action = None, argument = None):
    """ Helper to add a new menu entry """
    tmp = GtkMenuItem (item)
    if action:
        tmp.connect ("activate", action, argument)
    tmp.show ()
    menu.append (tmp)
    
    return tmp

regular_field = re.compile ("^\w+$")
regular_id    = re.compile ("^\w[\d\w:_-]+$")

class BasicEditor (GtkHBox):
    def __init__ (self, entry, field):
        GtkHBox.__init__ (self, FALSE, 5)

        self.entry = entry
        self.field = string.lower (field)
        self.name  = field
        
        if entry.has_key (self.field):
            self.value, loss = entry.text (self.field)
        else:
            self.value, loss = None, 0

        if loss:
            icon = GnomeStock (STOCK_BUTTON_CANCEL)
        else:
            icon = GnomeStock (STOCK_BUTTON_APPLY)
            
        button = GtkButton ()
        button.add (icon)
        button.connect ("clicked", self.edit_native)
        
        self.pack_end (button, FALSE, FALSE)
        self.show_all ()
        return


    def edit_native (self, * arg):
        return


    def fill (self):
        if self.value:
            self.editor.set_text (str (self.value))
        return

    
    def update (self):
        new = self.get ()
        if new != self.value:
            
            if new:
                self.entry [self.field] = new
            else:
                # remove empty entries
                del self.entry [self.field]

            return 1
        
        return 0


    def get (self):
        text = string.strip (self.editor.get_text ())
        
        if text == '': return None
        
        return Fields.Text (text)

    
class TextEditor (BasicEditor):
    
    def __init__ (self, entry, field):
        BasicEditor.__init__ (self, entry, field)
        
        scrolled = GtkScrolledWindow ()
        scrolled.set_policy (POLICY_AUTOMATIC, POLICY_AUTOMATIC)
        
        self.editor = GtkText ()
        self.editor.set_editable (TRUE)
        self.editor.set_usize (300,100)
        self.editor.set_word_wrap (1)

        scrolled.add (self.editor)
        self.pack_start (scrolled, TRUE, TRUE)
        self.fill ()
        return

    def fill (self):
        if self.value:
            self.editor.insert_defaults (str (self.value))
        return

    def get (self):
        text = self.editor.get_chars (0,-1)
        if text == '': return None

        return Fields.Text (text)
    

class AuthorEditor (BasicEditor):
    
    def __init__ (self, entry, field):
        BasicEditor.__init__ (self, entry, field)

        scrolled = GtkScrolledWindow ()
        scrolled.set_policy (POLICY_AUTOMATIC, POLICY_AUTOMATIC)
        
        self.editor = GtkText ()
        self.editor.set_editable (TRUE) 
        self.editor.set_word_wrap (1)
        self.editor.set_usize (300,100)
        
        scrolled.add (self.editor)
        self.pack_start (scrolled, TRUE, TRUE)
        self.fill ()
        return

    def fill (self):
        def str_auth (auth):
            if auth.first:
                ret = auth.last + ', ' + auth.first
            else:
                ret = auth.last

            if auth.lineage: ret = ret + ' ' + auth.lineage

            return ret

        if self.value:
            self.editor.insert_defaults (join (map (str_auth, self.value), "\n"))
        return

    def get (self):
        text = string.strip (self.editor.get_chars (0,-1))
        
        autg = Fields.AuthorGroup ()
        for line in string.split (text, '\n'):
            line = string.strip (line)
            if line == '': continue
            
            autg.append (Fields.Author (line))

        if len (autg) == 0: return None
        return autg
    
class FieldEditor (BasicEditor):
    
    def __init__ (self, entry, field):
        BasicEditor.__init__ (self, entry, field)
        self.editor = GtkEntry ()
        self.editor.set_editable (TRUE)
        self.pack_start (self.editor, TRUE, TRUE)
        self.fill ()
        return



class DateEditor (BasicEditor):
    
    def __init__ (self, entry, field):
        BasicEditor.__init__ (self, entry, field)
        self.editor = GtkEntry ()
        self.editor.set_editable (TRUE)
        self.pack_start (self.editor, TRUE, TRUE)
        self.fill ()
        return

    def get (self):
        text = string.strip (self.editor.get_text ())
        if text == '': return None

        return Fields.Date (text)
    
class FieldGroup:
    def __init__ (self, order = []):
        self.dict  = {}
        self.order = order
        return

    def __getitem__ (self, key):
        return self.dict [key]

    def __setitem__ (self, key, value):
        self.dict [key] = value
        return

    def __len__ (self):
        return len (self.dict.keys ())

    def has_key (self, key):
        return self.dict.has_key (key)

    def update (self):
        changed = 0
        
        for k in self.dict.keys ():
            changed = self [k].update () or changed
            
        return changed
    
    def as_table (self):
        table = GtkTable (len (self), 2)
        
        table.set_border_width (5)
        table.set_row_spacings (5)
        table.set_col_spacings (5)
            
        kt = self.dict.keys ()
        
        row = 0
        for k in self.order:
            
            lc = string.lower (k.name)
            if not self.has_key (lc): continue

            kt.remove (lc)
            entry = self [lc]

            tmp = GtkLabel (k.name)
            table.attach (tmp, 0, 1, row, row + 1)
            tmp.show ()

            table.attach (entry, 1, 2, row, row + 1)
            entry.show ()

            row = row + 1

        for k in kt:

            entry = self [k]

            tmp = GtkLabel (k)
            table.attach (tmp, 0, 1, row, row + 1, 0)
            tmp.show ()

            table.attach (entry, 1, 2, row, row + 1)
            entry.show ()

            row = row + 1

        table.show_all ()
        return table
    
    
class Entry (GtkVBox):
    """ Entry displayer """
    
    def __init__ (self, entry, properties):
        GtkVBox.__init__ (self, FALSE, 5)
        
        self.entry      = entry
        self.properties = properties
        self.changed    = 0
        
        ent = entry.type

        table = GtkTable (2, 2)
        table.set_border_width (5)
        table.set_row_spacings (5)
        table.set_col_spacings (5)
        
        self.pack_start (table, FALSE, FALSE)
        
        table.attach (GtkLabel ('Type'), 0, 1, 0, 1)
        table.attach (GtkLabel ('Identifier'), 1, 2, 0, 1)

        self.name = GtkEntry ()

        if self.properties ['change_type']:
            self.type = GtkOptionMenu ()
            available = map (lambda x: x.name, Config.get ("base/entries").data.values ())
            
            menu = GtkMenu ()
            self.type.set_menu (menu)
            
            for avail in available:
                popup_add (menu, avail, self.type_changed, avail)

            self.type.set_history (available.index (ent.name))
        else:
            self.type = GtkEntry ()
            self.type.set_text (ent.name)
            self.type.set_editable (FALSE)

        
        self.name.set_text (entry.name)

        if not self.properties ['change_id']:
            self.name.set_editable (FALSE)


        table.attach (self.type, 0, 1, 1, 2)
        table.attach (self.name, 1, 2, 1, 2)
            
        table.show_all ()

        if self.properties ['has_extra']:
            # extra field addition
            hbox   = GtkHBox (FALSE, 5)
            hbox.set_border_width (5)
            select = GnomeEntry ("extra-entry")
            hbox.pack_start (select)
            add_button = GtkButton ("Add Extra Field")
            hbox.pack_start (add_button)
            self.pack_start (hbox, fill=FALSE, expand=FALSE)

            add_button.connect ("clicked", self.extra_field, select)

        self.create_notebook (ent)
        return
    
    def create_notebook (self, ent):
        
        # Notebook holding Mandatory/Optional fields
        self.notebook = GtkNotebook ()
        
        self.mandatory = FieldGroup (ent.mandatory)
        self.optional  = FieldGroup (ent.optional)
        self.extra     = FieldGroup ()
        
        def get_field (entry, field):
            type = FieldsInfo.fieldinfo (field).type
            
            if type == FieldsInfo.WidgetText: 
                tmp = TextEditor (entry, field)

            elif type == FieldsInfo.WidgetEntry:
                tmp = FieldEditor (entry, field)

            elif type == FieldsInfo.WidgetAuthor:
                tmp = AuthorEditor (entry, field)

            elif type == FieldsInfo.WidgetDate:
                tmp = DateEditor (entry, field)

            else:
                tmp = TextEditor (entry, field)

            tmp.show ()
            return tmp
        
        dico = self.entry.keys ()

        for f in ent.mandatory:
            lc = string.lower (f.name)
            self.mandatory [lc] = get_field (self.entry, f.name)
            if self.entry.has_key (lc): dico.remove (lc)
            
        for f in ent.optional:
            lc = string.lower (f.name)
            self.optional [lc] = get_field (self.entry, f.name)
            if self.entry.has_key (lc): dico.remove (lc)

        for lc in dico:
            self.extra [lc] = get_field (self.entry, lc)

        table = self.mandatory.as_table ()
        self.notebook.append_page (table, GtkLabel ("Mandatory"))

        table = self.optional.as_table ()
        self.notebook.append_page (table, GtkLabel ("Optional"))

        table = self.extra.as_table ()
        self.notebook.append_page (table, GtkLabel ("Extra"))

        self.notebook.show ()
        self.pack_start (self.notebook)
        return

    def type_changed (self, menu, type):
        type = Types.getentry (type)
        self.entry.type = type

        # Refresh the whole stuff
        self.remove (self.notebook)
        self.create_notebook (type)

        self.changed = 1
        return

    def extra_field (self, button, widget):
        text = string.strip (widget.gtk_entry ().get_text ())
        
        if not regular_field.search (text):
            d = GnomeErrorDialog ("Invalid field name `%s'" % text)
            d.run_and_close ()
            return

        self.entry [text] = 'new entry'
        
        # Refresh the whole stuff
        self.remove (self.notebook)
        self.create_notebook (self.entry.type)
        self.changed = 1
        self.notebook.set_page (2)
        return
    
class EntryDialog (GtkDialog):
    
    def __init__ (self, entry, database, callback = None, user = None):

        GtkDialog.__init__ (self)
        
        self.set_title ('Edit')
        self.connect("delete_event", self.close)

        self.database = database
        self.callback = callback
        self.user     = user

        self.entry = Entry (entry, database.properties)
        
        apply_b = GnomeStockButton (STOCK_BUTTON_APPLY)
        apply_b.connect ("clicked", self.apply)

        close_b = GnomeStockButton (STOCK_BUTTON_CANCEL)
        close_b.connect ("clicked", self.close)
        
        self.action_area.add (apply_b)
        self.action_area.add (close_b)

        self.vbox.pack_start (self.entry)
        self.entry.show_all ()
        return

    def apply (self, widget):
        # ask every entry to apply its changes
        changed = self.entry.changed
        
        changed = self.entry.mandatory.update () or changed
        changed = self.entry.optional.update () or changed
        changed = self.entry.extra.update () or changed

        # check the id
        id = string.strip (self.entry.name.get_text ())
        
        if not regular_id.search (id):
            d = GnomeErrorDialog ("Invalid identifier `%s'" % id)
            d.run_and_close ()
            return
            
        elif id != self.entry.entry.name:
            
            # check unicity
            if not self.database.has_key (Base.Key (self.database, id)):
                self.entry.entry.name = id
                changed = 1
            else:
                d = GnomeErrorDialog ("Identifier `%s' already exists" % id)
                d.run_and_close ()
                return
        
        if changed and self.callback:
            self.callback (self.entry.entry, self.user)
        
        self.destroy ()
        return
    
    def close (self, *arg):
        self.destroy ()
        return
